#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/fs.h>
#include <sys/ioctl.h>
#include <asm/errno.h>
#include "mpdev.h"

typedef enum mpdev_cmds_s {
	FAIL,
	RECOVER,
	UNKNOWN=99
} mpdev_cmd;


char        filename[128];
mpdev_cmd   cmd_arg=UNKNOWN;
int         fd=-1;


void useage(void)
{
	printf("mp: Tells the Multipath ramdisk device driver (mpdev.o)\n");
	printf("    which i/o paths are to be failed or recovered.\n");
	printf("\n");
	printf("Syntax:\n");
	printf("\tmp [cmd|?|help] device_name\n");
	printf("\n");
	printf("Examples:\n");
	printf("\tmp fail /dev/mpa              ... will fail any i/o to the device\n");
	printf("\tmp recover /dev/mpa           ... will allow i/o to the device\n");
	printf("\n");
	printf("Notes on installing the ramdisk:\n");
	printf("\tCreate new device nodes       ... mknod /dev/mpa b 252 0\n");
	printf("\t                                     mknod /dev/mpb b 252 1\n");
	printf("\tAdd alias to /etc/modules.conf... alias block-major-252 mpdev\n");
	printf("\tCopy mpdev to /lib/modules    ... mkdir /lib/modules/2.4.18/misc/\n");
	printf("\t                                  cp mpdev.o /lib/modules/2.4.18/misc/\n\n");
}


int parse_args(int argc, char **argv)
{
	int  rc=0;

	if (argc != 3) return EINVAL;

	if ( (strncmp(argv[1],"fail",4)==0) ||
	     (strncmp(argv[1],"FAIL",4)==0)) {
		cmd_arg = FAIL;            
	} else if ( (strncmp(argv[1],"recover",7)==0) ||
		    (strncmp(argv[1],"RECOVER",7)==0)) {
		cmd_arg = RECOVER;
	}

	switch (cmd_arg) {
	case FAIL:
	case RECOVER:
		strncpy(filename, argv[2], 127 );

		if (  strlen(filename) == 0 ) {
			rc = EINVAL;
		} else {
			rc = 0;
		}

		break;

	default:
		rc= EINVAL;
		break;
	}

	return rc;
}


int open_device(int *fd)
{    
	*fd = open(filename, O_RDONLY | O_NONBLOCK);
	return(errno);
}


int close_device(int fd)
{       
	if ( close (fd) != 0 ) {
		fprintf (stderr, "mpdev device close failed: %s\n", strerror (errno));
	}
	return(errno);
}


int fail(int fd)
{
	mpdev_cmd_t  cmd = {1,0};
	int          status;
	int          rc=0;

	status = ioctl (fd, MPDEV_CMD, &cmd);
	if (status) {
		fprintf(stderr, "fail cmd failed rc= %s\n", strerror(errno));
		rc = errno;
	} else if ( cmd.status ) {
		fprintf(stderr, "fail cmd failed rc= %d\n", cmd.status );
		rc = cmd.status;
	}

	return rc;
}

int recover(int fd)
{
	mpdev_cmd_t  cmd = {0,0};
	int          status;
	int          rc=0;

	status = ioctl (fd, MPDEV_CMD, &cmd);
	if (status) {
		fprintf(stderr, "recover cmd failed rc= %s\n", strerror(errno));
		rc = errno;
	} else if ( cmd.status ) {
		fprintf(stderr, "recover cmd failed rc= %d\n", cmd.status );
		rc = cmd.status;
	}

	return rc;
}

int main (int argc, char **argv)
{
	int fd;                 
	int rc=0;    

	if ( argc == 2 ) {
		if ( (strcmp(argv[1], "?")==0) ||
		     (strcmp(argv[1], "-?")==0) ||
		     (strcmp(argv[1], "help")==0) ||
		     (strcmp(argv[1], "HELP")==0) ||
		     (strcmp(argv[1], "-help")==0) ||
		     (strcmp(argv[1], "-HELP")==0)) {
			useage();
			return 0;
		}
	} else if (argc==1) {
		useage();
		return 0;
	}

	if ( parse_args(argc, argv) ) {
		useage();
		return 0;
	}

	if ( open_device(&fd) == 0 ) {

		switch (cmd_arg) {
		case FAIL:
			rc = fail(fd);
			break;
		case RECOVER:
			rc = recover(fd);
			break;
		default:
			rc = EINVAL;
			break;
		}

		if (!rc) fsync(fd);

		close_device(fd);

	} else {
		fprintf(stderr, "device open failed\n" );
	}

	if (!rc) {
		printf("Success\n");
	}

	return(rc);
}
