/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: rsct_internal.c
 */
#include "rsct_internal.h"
#include "eceinternal.h"
#include <pthread.h>


/***************external global variables***********************/
extern ha_gs_token_t            gl_provider_token;
extern ha_gs_token_t            gl_subscriber_token;

extern GSList*                  gl_cb_list;
extern engine_functions_t *     gl_ece_engine_funcs;
extern pthread_mutex_t          gl_ack_lock;
extern pthread_cond_t           gl_ack_cond;
extern unsigned                 gl_ack_arrived;
extern pthread_t                gl_rsct_thread;
extern pthread_t                gl_appcb_thread;
extern int                      gl_proposal_id;
extern pthread_mutex_t          gl_proposal_id_lock;
extern pthread_mutex_t          gl_send_lock;
/*************** global variables *****************************/
int                             gl_transid=0;
static ha_gs_proposal_info_t    proposal_info;

/************** internal data structure***************/
typedef  struct{
  int node_number;
  int instance_number;
  ece_nodeid_t node_id;
  int proposal_id;
}state_value_t;


/***************internal static functions*************************/


/* BEGIN OF FUNCTION THAT HANDLES AGGREGATION OF MESSAGE FRAGMENTS */
static unsigned char *
bmap_alloc(const unsigned int no_bits)
{
	unsigned int no_bytes = ((no_bits-1)/CHAR_BIT)+1;
	unsigned char *bitmap = (unsigned char *)
		g_malloc0(no_bytes*sizeof(char));
	LOG_ENTRY();
	RETURN(bitmap);
}

static void
bmap_set(unsigned char *bmp, const unsigned int pos)
{
	unsigned int byteno = (pos)/CHAR_BIT;
	unsigned int bitno = pos%CHAR_BIT;
	
	LOG_ENTRY();
	
	bmp[byteno] |= 1<<bitno;
	
	LOG_EXIT_VOID();
	return;
}

static gboolean
bmap_all_set(const unsigned char *bmp, const unsigned int no_bits)
{
	unsigned int num,shifts,i;
	unsigned int no_bytes = ((no_bits-1)/CHAR_BIT);
	gboolean ret;

	LOG_ENTRY();
	
	for (i=0; i< no_bytes; i++) {
		if(bmp[i]!=UCHAR_MAX) {
			RETURN(FALSE);
		}
	}
	shifts = (no_bits-1)%CHAR_BIT+1;
	num = ((1<<shifts)-1);
	ret = ((bmp[no_bytes]&num)==num);
	
	LOG_EXIT_BOOL(ret);
	return(ret);
}

static inline void
bmap_free(char *bmp)
{
	LOG_ENTRY();
	g_free(bmp);
	LOG_EXIT_VOID();
	return;
}


frag_t *gl_frag=NULL;

static void
frag_init(int max_node_number)
{
	LOG_ENTRY();
	gl_frag = (frag_t *)g_malloc0(max_node_number*sizeof(frag_t));
	LOG_EXIT_VOID();
	return;
}

static void
frag_clean(int indx)
{
	LOG_ENTRY();
	if(!gl_frag) {
		LOG_EXIT_VOID();
		return;
	}
	if(gl_frag[indx].datastr)
		g_free(gl_frag[indx].datastr);
	if(gl_frag[indx].seq_bmap)
		bmap_free(gl_frag[indx].seq_bmap);
	gl_frag[indx].datastr = NULL;
	gl_frag[indx].seq_bmap = NULL;
	LOG_EXIT_VOID();
	return;
}


static void
frag_cleanup()
{
	uint i, n;
	
	LOG_ENTRY();
	if(!gl_frag) {
		LOG_EXIT_VOID();
		return;
	}
	n = sizeof(gl_frag)/sizeof(frag_t);
	for ( i = 0 ; i < n ; i++ ) {
		if(gl_frag[i].datastr){
			g_free(gl_frag[i].datastr);
			gl_frag[i].datastr=NULL;
		}
		if(gl_frag[i].seq_bmap){
			bmap_free(gl_frag[i].seq_bmap);
			gl_frag[i].seq_bmap=NULL;
		}
	}
	g_free(gl_frag);
	gl_frag=NULL;
	LOG_EXIT_VOID();
	return;
}


static int
frag_assemble(const rsct_msg_t *rsct_msg, char **data_str)
{
	const char *d_str;
	int indx, seqno, tlen, flen, no_frag;
	frag_t *fr=NULL;
	int send_node_number;
	
	
	LOG_ENTRY();
	if(!gl_frag) {
		LOG_ERROR("frag_assemble: recevied "
			     "message before initialization\n");

		LOG_EXIT_BOOL(FALSE);
		return(FALSE);
	}
	
	//ha_gs_get_node_number(&local_node_number);
	send_node_number=rsct_msg->send_node_number;
	
	indx=send_node_number-1;
	assert( indx  >=0);
	fr = gl_frag+indx;
	
	seqno = rsct_msg->seqno;
  
	flen=rsct_msg->flen;
	tlen=rsct_msg->tlen;
	
	if(fr->datastr) {
		//if(rsct_msg->seqno !=0){
		assert(flen == fr->frag_size);
		assert(tlen == fr->tot_size);
		assert(fr->seq_bmap);
		d_str=rsct_msg->ecemsg.msg;
		
  } else {
		fr->tot_size = tlen;
		fr->frag_size = flen;
		fr->seq_bmap = bmap_alloc(((tlen-1)/flen)+1);
		fr->msg_count = rsct_msg->msg_count;
		fr->datastr = (char *)g_malloc0(tlen+1);
		d_str= rsct_msg->ecemsg.msg;
		
	}
	if(rsct_msg->data_length)
		memcpy(fr->datastr+seqno*flen, d_str, rsct_msg->data_length);
	bmap_set(fr->seq_bmap, seqno);
	
	no_frag = (tlen-1)/flen+1;
	if(bmap_all_set(fr->seq_bmap,no_frag)){
		*data_str = fr->datastr;
		memset(fr,0,sizeof(frag_t));
		bmap_free(fr->seq_bmap);
		fr->seq_bmap = NULL;

		LOG_EXIT_BOOL(TRUE);
		return(TRUE);
	}
	
	LOG_EXIT_BOOL(FALSE);
	return(FALSE);
}





/* END OF FUNCTION THAT HANDLES AGGREGATION OF MESSAGE FRAGMENTS */

static void 
send_back_ack(ece_nodeid_t* sender_id,int msg_count)
{
	
	rsct_msg_t ack_rsctmsg;
	ha_gs_proposal_info_t    proposal_info;
	ha_gs_rc_t               gs_rc;
	
	LOG_ENTRY();
	
	ack_rsctmsg.type=RSCT_ACK;
	ack_rsctmsg.msg_count =msg_count;
	memcpy(&(ack_rsctmsg.ecemsg.node),sender_id,sizeof(ece_nodeid_t));
	proposal_info.gs_message_request.gs_num_phases         = HA_GS_1_PHASE;
	proposal_info.gs_message_request.gs_time_limit         = TIME_LIMIT;
	proposal_info.gs_message_request.gs_message.gs_length  = sizeof(rsct_msg_t);
	proposal_info.gs_message_request.gs_message.gs_message = (char*)&ack_rsctmsg;
	
	
	pthread_mutex_lock(&gl_send_lock);
	
	gs_rc = ha_gs_send_message(gl_provider_token,
				   &proposal_info);
	
	pthread_mutex_unlock(&gl_send_lock);

	if(gs_rc != HA_GS_OK) {
		
		if(gs_rc == HA_GS_COLLIDE) 
			LOG_ERROR("ERROR: ha_gs_send_message() for ACK HA_GS_COLLIDE");
		else 
			LOG_ERROR("ERROR: ha_gs_send_message() for ACK failed rc=%d\n", gs_rc);
		
	}
	
	LOG_EXIT_VOID();
	return;
}


static void 
deliver_msg (gpointer data,
	     gpointer user_data)
{ 
	
	registercb_t* rcb=(registercb_t*) data;
	ece_msg_t* ecemsg=(ece_msg_t*)user_data;
	
	LOG_ENTRY();
	//rcb->reg_callback(CALLBACK_MESSAGE,sizeof(ece_msg_t)+ecemsg->size, user_data);
	RSCT_appcb_add(CALLBACK_MESSAGE,
		       rcb,
		       sizeof(ece_msg_t)+ecemsg->size,
		       user_data);
	
	LOG_EXIT_VOID();
	return;
	
}

static void 
deliver_mem( gpointer data,
	     gpointer user_data)
{
	
	registercb_t* rcb=(registercb_t*) data;
	int num_entries;
	ece_event_t* event=(ece_event_t*)user_data;
	int size;
	void* buf;
	
	LOG_ENTRY();
	
	if((rcb->reg_type == DELTAS && 
	    (event->type == DELTA_JOIN || event->type== DELTA_LEAVE))||
	   (rcb->reg_type == FULL_MEMBERSHIP && event->type == MEMBERSHIP)){
		
		num_entries=event->num_entries;
		/*rcb->reg_callback(CALLBACK_MEMBERSHIP,
		  sizeof(ece_event_t)+
		  sizeof(ece_nodeid_t)*(num_entries-1),
		  user_data);
		*/
		size= sizeof(ece_event_t)+sizeof(ece_nodeid_t)*(num_entries-1);
		
		buf=malloc(size);
		memcpy(buf,user_data,size);    
		RSCT_appcb_add(CALLBACK_MEMBERSHIP,
			       rcb,
			       size,
			       buf);
	}
	
	LOG_EXIT_VOID();
	return;
}


static void 
process_join(const ha_gs_approved_notification_t *block)
{
  
	int count= block->gs_proposal->gs_changing_providers->gs_count;
	int i;
	int local_node_number;
	int local_instance_number;
	state_value_t            sv;
	ha_gs_rc_t               gs_rc;
	
	LOG_ENTRY();
	
	if(gl_ece_engine_funcs->get_engine_mode() == ENGINE_DAEMON)
		local_instance_number=MASTER_INSTANCE_NUM;
	else
		local_instance_number=SLAVE_INSTANCE_NUM;
	
	
	ha_gs_get_node_number(&local_node_number);
	for(i=0;i<count;i++){
		
		//broadcast a message 
		sv.node_number=local_node_number;
		sv.instance_number=local_instance_number;
		sv.proposal_id=RSCT_get_proposal_id();
		RSCT_mem_get_mynode(&sv.node_id);
    
		proposal_info.gs_state_change_request.gs_num_phases         
			= HA_GS_1_PHASE;
		proposal_info.gs_state_change_request.gs_time_limit
			= TIME_LIMIT;
		proposal_info.gs_state_change_request.gs_new_state.gs_length
			= sizeof(state_value_t);
		proposal_info.gs_state_change_request.gs_new_state.gs_state =(char*)&sv;
		
		usleep(100);
		gs_rc = ha_gs_change_state_value( gl_provider_token,
						  &proposal_info);
		if(gs_rc != HA_GS_OK) {
			LOG_ERROR("ha_gs_change_state_value failed rc=%d ***\n", 
				     gs_rc);
		}
	}
	
	LOG_EXIT_VOID(); 
	return;
}

static void 
process_leave_or_failure(const ha_gs_approved_notification_t *block)
{
	
	int count= block->gs_proposal->gs_changing_providers->gs_count;
	int leave_node_number;
	int leave_instance_number;
	ece_nodeid_t leave_nodeid;
	ece_nodeid_t* temp;
	char* leave_node_mode;
	int i;
	
	LOG_ENTRY();
	
	for(i=0;i<count;i++){
		leave_node_number=
			block->gs_proposal->gs_changing_providers->gs_providers->gs_node_number;
		leave_instance_number=
			block->gs_proposal->gs_changing_providers->gs_providers->gs_instance_number;
		
		
		if(leave_instance_number ==MASTER_INSTANCE_NUM)
			leave_node_mode="master";
		else
			leave_node_mode="slave";
		
		//delete the entry 
		temp=RSCT_mem_map_table_lookup(leave_node_number);
		assert(temp &&"the to be deleted node not in the table?");
		memcpy(&leave_nodeid,temp,sizeof(ece_nodeid_t));
		
		
		//remove the plugin now
		RSCT_plugin_remove(leave_node_number,leave_instance_number);
	}
	
	//print out all nodes online
	//RSCT_plugin_print_all_nodes();
	
	LOG_EXIT_VOID();
	return;
	
}

static void process_msg(const ha_gs_approved_notification_t *block)
{
	
	//the received messages
	ha_gs_provider_message_t* prov_msg=NULL;
	rsct_msg_t* rsctmsg=NULL;
	ece_msg_t *ecemsg=NULL;
	
	// all node constant
	ece_nodeid_t all_nodes;
	
	//sender node information
	int sender_node_num;
	int sender_instance_num;
	ece_nodeid_t* sender_id;
	char* sender_mode;
	
	//local plugin info
	int local_node_num;
	int local_instance_num;
	ece_nodeid_t* local_nodeid=NULL;
	
	//some information about the message
	boolean msgForMe;
	boolean broadcast;
	boolean isAck;
	
	
	LOG_ENTRY();
	
	//initialize all_nodes  variable
	all_nodes=(ece_nodeid_t)ECE_ALL_NODES;
	
	
	//get local plugin information
	ha_gs_get_node_number(&local_node_num);
	if(gl_ece_engine_funcs->get_engine_mode() == ENGINE_DAEMON)
		local_instance_num=MASTER_INSTANCE_NUM;
	else
		local_instance_num=SLAVE_INSTANCE_NUM;
	local_nodeid=RSCT_mem_map_table_lookup(local_node_num);
	assert(local_nodeid &&"local node is not in the map table?");
	
	//get sender plugin information
	sender_instance_num =
		block->gs_proposal->gs_proposed_by.gs_instance_number;
	sender_node_num=block->gs_proposal->gs_proposed_by.gs_node_number;
	sender_id=RSCT_mem_map_table_lookup(sender_node_num);
  
	//get the actual rsctmsg and ecemsg
	prov_msg=block->gs_proposal->gs_provider_message;
	assert(prov_msg &&"provider_message is NULL");
	assert(prov_msg->gs_length && prov_msg->gs_message &&"provider_message empty?");
	rsctmsg=(rsct_msg_t*)(prov_msg->gs_message);
	ecemsg=&(rsctmsg->ecemsg);
	ecemsg->msg = prov_msg->gs_message + sizeof(rsct_msg_t);
	
  
	//this msg is specifically for me?
	msgForMe=(strncmp((char*)&(ecemsg->node),
			  (char*)local_nodeid, 
			  sizeof(ece_nodeid_t)) ==0);
	
	//this msg is a broadcast msg?
	broadcast=(strncmp((char*)&(ecemsg->node),
			   (char*) &all_nodes, 
			   sizeof(ece_nodeid_t))==0);
	
	//is this message an ACK?
	isAck= rsctmsg->type == RSCT_ACK;
	
	
	if(sender_instance_num == MASTER_INSTANCE_NUM)
		sender_mode="master";
	else 
		sender_mode="slave";
	
	LOG_DEBUG(" ... get a message from %s:%s:isack=%d:%d,seqno=%d\n",
		     sender_id,
		     sender_mode,
		     rsctmsg->type == RSCT_ACK,
		     rsctmsg->msg_count,
		     rsctmsg->seqno);
	
	if((local_instance_num != sender_instance_num)  //different mode of plugins
	   //&& (local_node_num != sender_node_num)           //in different nodes
	   && (msgForMe || broadcast)){                      //either the msg is sent to me or it's a broadcast
		
		
		//I am eligible to receive this msg
		
		
		//if this msg is an ACK, signal the condition
		if(isAck){
			assert(msgForMe &&!broadcast &&" it should be  the Ack for me??");
			
			LOG_DEBUG("get an ACK msg from %s\n", 
				     sender_id);
			usleep(1000);
			pthread_mutex_lock(&gl_ack_lock);
			gl_ack_arrived=1;
			pthread_cond_signal(&gl_ack_cond);
			pthread_mutex_unlock(&gl_ack_lock);
		}else{ 
			char* data_str;
			
			//a real msg sent for me, 
			//send back an ACK if necessary 
			//and call client-registered callbacks
			
			//assemble the message first
			if(frag_assemble(rsctmsg,&data_str)){
				ece_msg_t* asm_msg;
				
				asm_msg=(ece_msg_t*)malloc(sizeof(ece_msg_t)+ecemsg->size);
				memcpy(asm_msg,ecemsg,sizeof(ece_msg_t));
				asm_msg->msg = ((void*)asm_msg) + sizeof(ece_msg_t);
				memcpy(asm_msg->msg, data_str,asm_msg->size);
				frag_clean(sender_node_num - 1);
				
				
				if(sender_id )
					memcpy(&(asm_msg->node),sender_id, sizeof(ece_nodeid_t));
				assert(prov_msg->gs_length == 
				       sizeof(rsct_msg_t)+rsctmsg->data_length &&"Wrong messge size!");
				
				//send back an ACK if msg is specifically for me
				if(msgForMe){
					assert( !broadcast &&"broadcast msg?");
					
					LOG_DEBUG("sendback to %s ACK for msg_count=%d \n",sender_id,rsctmsg->msg_count);	    
					send_back_ack(sender_id,rsctmsg->msg_count);
				}      
				
				//call the client-registered callbacks
				g_slist_foreach(gl_cb_list, deliver_msg,(gpointer)asm_msg);
			}
		}
	}
	
	LOG_DEBUG("process_msg done\n");
	LOG_EXIT_VOID();
	return;
	
}

static void 
process_state_value_change(const ha_gs_approved_notification_t 
			   *block)
{
	state_value_t* sv;
	int local_node_number;
	int local_instance_number;
	boolean callNow;
	static boolean selfAlreadyCalled=FALSE;
	
	
	LOG_ENTRY();
	
	//get local node number
	ha_gs_get_node_number(&local_node_number);
	
	sv=(state_value_t*)malloc(sizeof(state_value_t));
	memcpy(sv,
	       block->gs_proposal->gs_proposed_state_value->gs_state,
	       sizeof(state_value_t));
	
	if(gl_ece_engine_funcs->get_engine_mode() == ENGINE_DAEMON)
		local_instance_number=MASTER_INSTANCE_NUM;
	else
		local_instance_number=SLAVE_INSTANCE_NUM;
	
	callNow=FALSE;
	if(sv->node_number ==local_node_number 
	   && sv->instance_number== local_instance_number
	   && local_instance_number== MASTER_INSTANCE_NUM
	   && (!selfAlreadyCalled)){
		callNow=TRUE;
		selfAlreadyCalled=TRUE;
	}
	
	RSCT_plugin_add(sv->node_number,sv->instance_number, &(sv->node_id));        
	free(sv);
	
	//print out node_map 
	//RSCT_plugin_print_all_nodes();  
	LOG_EXIT_VOID();
	return;
	
}


/***************************** callback functions *************************/

void 
RSCT_INTERNAL_cb_init()
{
	int max_node_number;
	
	LOG_ENTRY();
	
	max_node_number=RSCT_mem_get_max_node_number();
	frag_init(max_node_number);
	
	LOG_EXIT_VOID();
	return;
}

void 
RSCT_INTERNAL_cb_cleanup()
{
	
	LOG_ENTRY();
	frag_cleanup();
	LOG_EXIT_VOID();
	return;
}


void 
RSCT_INTERNAL_delayed_error_cb(const ha_gs_delayed_error_notification_t 
			       *block)
{
	
	ha_gs_rc_t               gs_rc;
	
	LOG_ENTRY();
	
	switch(block->gs_protocol_type) { 
	case HA_GS_JOIN:
		if(block->gs_delayed_return_code == HA_GS_DUPLICATE_INSTANCE_NUMBER) {
			LOG_ERROR("Error: delayed error(HA_GS_JOIN)\n");
			LOG_EXIT_VOID();
			return;
		}
		break;
	case HA_GS_LEAVE:
		if(block->gs_delayed_return_code == HA_GS_COLLIDE) {
			LOG_ERROR("Error: delayed error(HA_GS_LEAVE)\n");
			LOG_EXIT_VOID();
			return;
		}
		
		break;
	case HA_GS_STATE_VALUE_CHANGE:
		{
			if(block->gs_delayed_return_code == HA_GS_COLLIDE) {
				LOG_ERROR("warning: delayed error(HA_GS_STATE_VALUE_CHANGE),length=%d \n",
					     block->gs_failing_request->gs_state_change_request.gs_new_state.gs_length);
			}
			
			do{
				usleep(1000);
				gs_rc = ha_gs_change_state_value( gl_provider_token,
								  block->gs_failing_request);
				
				
				if(gs_rc != HA_GS_OK) {
					if(gs_rc != HA_GS_COLLIDE)
						assert(0&&"*** ha_gs_change_state_value failed ***\n");
				}
			}while(gs_rc !=HA_GS_OK);
		}
		break;
	case HA_GS_PROVIDER_MESSAGE:{
		rsct_msg_t* rsct_msg;
		int length;
		
		rsct_msg =(rsct_msg_t*)block->gs_failing_request->gs_message_request.gs_message.gs_message;
		length =block->gs_failing_request->gs_message_request.gs_message.gs_length;
		assert(length >= sizeof(rsct_msg_t));
		if(block->gs_delayed_return_code == HA_GS_COLLIDE) {
			LOG_ERROR("warning: delayed error(HA_GS_PROVIDER_MESSAGE) msg_count=%d,"
				     "seqno=%d,length=%d\n",rsct_msg->msg_count,rsct_msg->seqno,length);
		}else
			{
				LOG_ERROR("ERROR: sending message delayed error!\n");
				exit(1);
			}
		
		do{
			
			usleep(100000); 
			pthread_mutex_lock(&gl_send_lock);
			gs_rc = ha_gs_send_message(gl_provider_token,
						   block->gs_failing_request);
			pthread_mutex_unlock(&gl_send_lock);
			
			if(gs_rc == HA_GS_COLLIDE) {
				LOG_ERROR("warning:sending delaed message"
					     "collides again! msg_count=%d, seqno=%d,length=%d\n",
					     rsct_msg->msg_count,rsct_msg->seqno,length);
				
			}else if(gs_rc != HA_GS_OK){
				LOG_ERROR("Error: send message failed\n");
				break;
			}
		}while(gs_rc != HA_GS_OK);
		break;
	}
	default:
		assert(0&& "*** delayed error notification is not expected ***\n");
		break;
	}   
	
	LOG_EXIT_VOID();
	return;
}


void
RSCT_INTERNAL_n_phase_cb(const ha_gs_n_phase_notification_t *block) 
{
	
	LOG_ENTRY();
	
	LOG_EXIT_VOID();
	return;
}

void
RSCT_INTERNAL_approved_cb(const ha_gs_approved_notification_t *block) 
{
	
	
	LOG_ENTRY();
	
	switch(block->gs_protocol_type){
	case HA_GS_JOIN:
		process_join(block);
		break;
		
	case HA_GS_FAILURE_LEAVE:
	case HA_GS_LEAVE:
		process_leave_or_failure(block);
		break;
		
	case HA_GS_STATE_VALUE_CHANGE:
		process_state_value_change(block);
		break;
		
	case HA_GS_PROVIDER_MESSAGE:
		process_msg(block);
		break;
		
	case HA_GS_RESPONSIVENESS:
	case HA_GS_EXPEL:
	case HA_GS_CAST_OUT:
	case HA_GS_SOURCE_STATE_REFLECTION:
	case HA_GS_MERGE:
	case HA_GS_SUBSCRIPTION:
	case HA_GS_GROUP_ATTRIBUTE_CHANGE:
	default:
		exit(1);
		break;
	}
	
	LOG_EXIT_VOID();
	return;
}

void
RSCT_INTERNAL_rejected_cb(const ha_gs_rejected_notification_t *block)
{
	
	LOG_ENTRY();
	
	LOG_EXIT_VOID();
	return;
}

void 
RSCT_INTERNAL_announcement_cb(const ha_gs_announcement_notification_t *block) 
{
	
	LOG_ENTRY();
	
	switch(block->gs_summary_code) {
	case HA_GS_RESPONSIVENESS_NO_RESPONSE:
	case HA_GS_RESPONSIVENESS_RESPONSE:
	case HA_GS_TIME_LIMIT_EXCEEDED:
	default:
		break;
	}
	LOG_EXIT_VOID();
	return;
}


static void
membershipDeliver( ece_event_type_t type,
		   ha_gs_membership_t* membership_list )
{
	
	ece_event_t* event;
	int i;
	ha_gs_provider_t* member;
	
	LOG_ENTRY();
	
	if(membership_list){
		event =(ece_event_t*) malloc(sizeof(ece_event_t) 
					     + (membership_list->gs_count-1)*sizeof(ece_nodeid_t));
		
		event->num_entries =membership_list->gs_count;
		
		member=membership_list->gs_providers;
		for(i=0;i < membership_list->gs_count; i++,member++)
			RSCT_mem_nodenum_to_nodeid(event->node+i,member->gs_node_number);  
		
	}
	else{
		event=(ece_event_t*)malloc(sizeof(ece_event_t));
		
		event->num_entries = 0;
		
	}
	event->quorum_flag=RSCT_mem_has_quorum();
	event->type = type;

	g_slist_foreach(gl_cb_list, deliver_mem,(gpointer)event);
	
	free(event); 
	
	LOG_EXIT_VOID();
	return;
} 


void 
RSCT_INTERNAL_subscriber_cb(const ha_gs_subscription_notification_t* block)
{
	
	LOG_ENTRY();
	
	
	if(block->gs_subscription_type & HA_GS_SUBSCRIPTION_DISSOLVED) {
		LOG_ERROR("*** group dissolved ***\n");
		exit(1);
	} else {
		int type;
		boolean full;
		boolean join;
		boolean leave;
		
		type=block->gs_subscription_type;
		join=type & HA_GS_SUBSCRIPTION_DELTA_JOIN;
		leave=type & HA_GS_SUBSCRIPTION_DELTA_LEAVE;
		full=type & HA_GS_SUBSCRIPTION_MEMBERSHIP;
		
		if( !join && !leave && full){
			//first time, return after update online-offline info
			//usleep(100000);
			//membershipDeliver(DELTA_JOIN,block->gs_full_membership);  
			
			RSCT_mem_online_offline_update(block->gs_full_membership,TRUE);
			LOG_EXIT_VOID();
			return;
		}
		
		//join
		if(join){
			membershipDeliver(DELTA_JOIN,block->gs_changing_membership);  
			membershipDeliver(DELTA_LEAVE,NULL);  
      
			RSCT_mem_online_offline_update(block->gs_full_membership,TRUE);
		}
		//leave
		if(leave){
			membershipDeliver(DELTA_JOIN,NULL);   
			membershipDeliver(DELTA_LEAVE,block->gs_changing_membership);   
			RSCT_mem_online_offline_update(block->gs_full_membership,FALSE);
			
		}
		//full membership
		if(full){
			//print_nodes(block->gs_full_membership);
			membershipDeliver(MEMBERSHIP,block->gs_full_membership); 
		}
	}
	
	LOG_EXIT_VOID();
	return;
}




/**************************************************************/


int 
RSCT_get_proposal_id(){
	int temp;
	
	LOG_ENTRY();
	pthread_mutex_lock(&gl_proposal_id_lock);
	temp=gl_proposal_id++;
	pthread_mutex_unlock(&gl_proposal_id_lock);
	
	LOG_EXIT_INT(temp);
	return(temp);
}


