/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Multipath Plugin
 * File: evms2/engine/plugins/multipath/mp_segments.c
 *
 * Common routines for the multipath plugin.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "multipath.h"

/**
 * multipath_allocate
 *
 * Allocate and initialize a multipath segment.
 **/
storage_object_t *multipath_allocate(char *name, sector_count_t size,
				     multipath_type_t type)
{
	storage_object_t *segment = NULL;
	multipath_t *mp = NULL;
	int rc;

	LOG_ENTRY();

	rc = EngFncs->allocate_segment(name, &segment);
	if (rc) {
		goto out;
	}

	mp = EngFncs->engine_alloc(sizeof(*mp));
	if (!mp) {
		rc = ENOMEM;
		goto out;
	}

	mp->type = type;
	segment->data_type = DATA_TYPE;
	segment->plugin = my_plugin_record;
	segment->size = size;
	segment->private_data = mp;

	rc = mp_modules[type].allocate(segment);

out:
	if (rc) {
		EngFncs->engine_free(mp);
		EngFncs->free_segment(segment);
		segment = NULL;
	}
	LOG_EXIT_PTR(segment);
	return segment;
}

/**
 * make_parent_and_child
 *
 * Make "parent" a parent of "child", and make "child" a child of "parent".
 **/
void make_parent_and_child(storage_object_t *parent, storage_object_t *child)
{
	LOG_ENTRY();

	EngFncs->insert_thing(parent->child_objects, child, INSERT_AFTER, NULL);
	EngFncs->insert_thing(child->parent_objects, parent, INSERT_AFTER, NULL);

	LOG_EXIT_VOID();
}

/**
 * unmake_parent_and_child
 *
 * Opposite of make_parent_and_child().
 **/
void unmake_parent_and_child(storage_object_t *parent, storage_object_t *child)
{
	LOG_ENTRY();

	if (parent && child) {
		EngFncs->remove_thing(parent->child_objects, child);
		EngFncs->remove_thing(child->parent_objects, parent);
	}

	LOG_EXIT_VOID();
}

/**
 * activate_segment
 **/
int activate_segment(storage_object_t *object)
{
	multipath_t *mp = object->private_data;
	dm_target_t *targets;
	int rc = 0;

	LOG_ENTRY();

	if (!(mp->flags & MP_FLAG_ACTIVATE_SEGMENT)) {
		goto out;
	}

	rc = mp_modules[mp->type].build_targets(object, &targets);
	if (!rc) {
		rc = EngFncs->dm_activate(object, targets);
		if (!rc) {
			mp->flags &= ~MP_FLAG_ACTIVATE_SEGMENT;
		}
		EngFncs->dm_deallocate_targets(targets);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}
