/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: fsimext2.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <plugin.h>
#include "fsimext2.h"

int fsim_rw_diskblocks( logical_volume_t *, int, int64_t, int32_t, void *, int );
void set_mkfs_options( option_array_t *, char **, logical_volume_t *, char * );
void set_fsck_options( option_array_t *, char **, logical_volume_t * );

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                   Common Routines                                    +
+                                                                                      +
+--------------------------------------------------------------------------------------*/


/*
 * Get the size limits for this volume.
 */
int fsim_get_volume_limits( struct ext2_super_block * sb,
			    sector_count_t   * fs_min_size,
			    sector_count_t   * fs_max_size,
			    sector_count_t   * vol_max_size)
{
	int rc = 0;
	int            blk_to_sect;

	blk_to_sect = (1 + sb->s_log_block_size);
	*fs_min_size = (sb->s_blocks_count - sb->s_free_blocks_count) << blk_to_sect;
	*fs_max_size = (sector_count_t) 1 << (32+blk_to_sect);
	*vol_max_size = 0xffffffffffULL;

	return rc;
}


/*
 * Un-Format the volume.
 */
int fsim_unmkfs( logical_volume_t * volume )
{
	int  fd;
	int  rc = 0;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) return -fd;

	if (volume->private_data) {
		/* clear private data */
		memset( (void *) volume->private_data, 0, SIZE_OF_SUPER );
		/* zero primary superblock */
		rc =  fsim_rw_diskblocks( volume, fd, EXT2_SUPER_LOC, SIZE_OF_SUPER,
					  volume->private_data, PUT );
	} else {
		rc = EINVAL;
	}

	EngFncs->close_volume(volume,fd);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Format the volume.
 */
int fsim_mkfs(logical_volume_t * volume, option_array_t * options )
{
	int     rc;
	char   *argv[MKFS_EXT2_OPTIONS_COUNT + 6] = {NULL};
	char    logsize[sizeof(unsigned int) + 1];
	pid_t   pidm;
	int     status;
	int     fds2[2];

	LOG_ENTRY();

	set_mkfs_options( options, argv, volume, logsize );

	fds2[0] = 0;
	fds2[1] = open("/dev/null", O_WRONLY);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		waitpid( pidm, &status, 0 );
		if (WIFEXITED(status)) {
			/* get mke2fs exit code */
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = errno;
	}

	close(fds2[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: set_mkfs_options
 *
 * FUNCTION: Build options array (argv) for mkfs.ext2
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume name on which program will be executed
 *
 */
void set_mkfs_options( option_array_t * options,
		       char ** argv,
		       logical_volume_t * volume,
		       char * logsize )
{
	int i, bufsize, opt = 0;
	char *buf;

	LOG_ENTRY();

	argv[opt++] = "mke2fs";

	/* 'quiet' and 'Force' options */
	argv[opt++] = "-qF";

	/* the following is a big hack to make sure that we don't use a block */
	/* size smaller than hardsector size since this does not work. */
	/* would be nice if the ext2/3 utilities (mkfs) handled this themselves */
	/* also, eventually we will implement this as a user option to manually */
	/* set block size */
	if (volume->object->geometry.bytes_per_sector != EVMS_VSECTOR_SIZE) {
		switch (volume->object->geometry.bytes_per_sector) {
		case 1024:
			argv[opt++] = "-b1024";
			break;
		case 2048:
			argv[opt++] = "-b2048";
			break;
		case 4096:
			argv[opt++] = "-b4096";
			break;
		default:
			/* not one we expect, just skip it */
			break;
		}
	}

	for (i=0; i<options->count; i++) {

		if (!options->option[i].is_number_based) {

			if (!strcmp(options->option[i].name, "badblocks")) {
				options->option[i].number = MKFS_CHECKBB_INDEX;
			} else if (!strcmp(options->option[i].name, "badblocks_rw")) {
				options->option[i].number = MKFS_CHECKRW_INDEX;
			} else if (!strcmp(options->option[i].name, "journal")) {
				options->option[i].number = MKFS_JOURNAL_INDEX;
			} else if (!strcmp(options->option[i].name, "vollabel")) {
				options->option[i].number = MKFS_SETVOL_INDEX;
			} else {
				/* unknown. ignore. */
				continue;
			}
		}

		switch (options->option[i].number) {
			
		case MKFS_CHECKBB_INDEX:
			/* 'check for bad blocks' option */
			if (options->option[i].value.b == TRUE) {
				argv[opt++] = "-c";
			}
			break;

		case MKFS_CHECKRW_INDEX:
			/* 'check for r/w bad blocks' option */
			if (options->option[i].value.b == TRUE) {
				argv[opt++] = "-cc";
			}
			break;

		case MKFS_JOURNAL_INDEX:
			/* 'create ext3 journal' option */
			if (options->option[i].value.b == TRUE) {
				argv[opt++] = "-j";
			}
			break;

		case MKFS_SETVOL_INDEX:
			/* 'set volume name' option */
			if (strlen(options->option[i].value.s) > 0) {
				argv[opt++] = "-L";
				argv[opt++] = options->option[i].value.s;
			}
			break;

		default:
			break;
		}
	}

	argv[opt++] = EVMS_GET_DEVNAME(volume);
	argv[opt++] = NULL;

	bufsize = 0;
	for (i = 0; argv[i]; i++)
		bufsize += strlen(argv[i]) + 5;
	buf = malloc(bufsize+1);
	if (!buf)
		return;
	buf[0] = 0;
	for (i=0; argv[i]; i++) {
		strcat(buf, argv[i]);
		strcat(buf, " ");
	}
	LOG_DEBUG("mke2fs command: %s\n", buf);
	free(buf);

	LOG_EXIT_VOID();
	return;
}


/*
 * Run fsck on the volume.
 */
int fsim_fsck(logical_volume_t * volume, option_array_t * options,
	      int *ret_status)
{
	int     rc;
	char   *argv[FSCK_EXT2_OPTIONS_COUNT + 3];
	pid_t	pidf;
	int     status, bytes_read;
	char    *buffer = NULL;
	int     fds2[2];
	int     banner = 0;

	LOG_ENTRY();

	/* open pipe, alloc buffer for collecting fsck.ext2 output */
	rc = pipe(fds2);
	if (rc) {
		return(errno);
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		close(fds2[0]);
		close(fds2[1]);
		return(ENOMEM);
	}

	set_fsck_options( options, argv, volume );
	pidf = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidf != -1) {
		/* wait for child to complete */
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			/* read e2fsck output */
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				/* display e2fsck output */
				if (!banner)
					MESSAGE(_("e2fsck output:"));
				banner = 1;
				MESSAGE("%s",buffer);
				memset(buffer,0,bytes_read); /* clear out message  */
			}
			usleep(10000); /* don't hog all the cpu */
		}

		/* do final read, just in case we missed some */
		bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
		if (bytes_read > 0) {
			if (!banner)
				MESSAGE(_("e2fsck output:"));
			MESSAGE("%s",buffer);
		}
		if (WIFEXITED(status)) {
			/* get e2fsck exit code */
			*ret_status = WEXITSTATUS(status);
			LOG_DEFAULT("e2fsck completed with exit code %d \n",
				    *ret_status);
			rc = 0;
		} else {
			rc = EINTR;
		}
	} else {
		rc = errno;
	}

	EngFncs->engine_free(buffer);

	close(fds2[0]);
	close(fds2[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: set_fsck_options
 *
 * FUNCTION: Build options array (argv) for e2fsck
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - fsck options array
 *      volume    - volume on which program will be executed
 *
 */
void set_fsck_options( option_array_t * options, char ** argv, logical_volume_t * volume )
{
	int i, bufsize, num_opts, opt_count = 1;
	int do_preen = 1;
	char *buf;

	LOG_ENTRY();

	argv[0] = "e2fsck";

	if (options)
		num_opts = options->count;
	else {
		/* No options, assume force (for resizing) */
		argv[opt_count++] = "-f";
		num_opts = 0;
	}

	for (i=0; i < num_opts; i++) {

		if (!options->option[i].is_number_based) {
			/* 'force check' option selected */
			if (!strcmp(options->option[i].name, "force")) {
  				options->option[i].number = FSCK_FORCE_INDEX;
			} else if (!strcmp(options->option[i].name, "readonly")) {
				/* 'check read only' option selected */
				options->option[i].number = FSCK_READONLY_INDEX;
			} else if (!strcmp(options->option[i].name, "badblocks")) {
				/* 'check badblocks' option selected */
				options->option[i].number = FSCK_CHECKBB_INDEX;
			} else if (!strcmp(options->option[i].name, "badblocks_rw")) {
				/* 'check r/w badblocks' option selected */
				options->option[i].number = FSCK_CHECKRW_INDEX;
			} else if (!strcmp(options->option[i].name, "timing")) {
				/* 'timing' option selected */
				options->option[i].number = FSCK_TIMING_INDEX;
			} else {
 				/* unknown. ignore. */
				continue;
			}
		}

		/* 'force check' option */
		if ((options->option[i].number == FSCK_FORCE_INDEX) &&
		    (options->option[i].value.b == TRUE)) {
			argv[opt_count++] = "-f";

			/* 'check read only' option or mounted */
		} else if ((options->option[i].number == FSCK_READONLY_INDEX) &&
			    ((options->option[i].value.b == TRUE) ||
			    EVMS_IS_MOUNTED(volume))) {
				argv[opt_count++] = "-n";
				do_preen = 0;
			/* 'bad blocks check' option and NOT mounted */

		} else if ((options->option[i].number == FSCK_CHECKBB_INDEX) &&
			    (options->option[i].value.b == TRUE)         &&
			    !EVMS_IS_MOUNTED(volume)) {
				argv[opt_count++] = "-c";
				do_preen = 1;

			/* 'bad blocks check' option and NOT mounted */
		} else if ((options->option[i].number == FSCK_CHECKRW_INDEX) &&
			    (options->option[i].value.b == TRUE)         &&
			    !EVMS_IS_MOUNTED(volume)) {
				argv[opt_count++] = "-cc";
				do_preen = 1;

			/* timing option */
		} else if ((options->option[i].number == FSCK_TIMING_INDEX) &&
			    (options->option[i].value.b == TRUE)) {
				argv[opt_count++] = "-tt";
		}
	}

	if (do_preen)
		argv[opt_count++] = "-p";
	argv[opt_count++] = EVMS_GET_DEVNAME(volume);
	argv[opt_count]   = NULL;

	bufsize = 0;
	for (i=0; argv[i]; i++)
		bufsize += strlen(argv[i]) + 5;
	buf = malloc(bufsize+1);
	if (!buf)
		return;
	buf[0] = 0;
	for (i=0; argv[i]; i++) {
		strcat(buf, argv[i]);
		strcat(buf, " ");
	}
	LOG_DEBUG("fsck command: %s\n", buf);
	free(buf);

	LOG_EXIT_VOID();
	return;
}
/*
 * NAME:ext2fs_swap_super
 *
 * FUNCTION: Swap all fields in the super block to CPU format.
 *
 * PARAMETERS:
 *      sb   - pointer to superblock
 *
 * RETURNS:
 *        void
 */
static void ext2fs_swap_super(struct ext2_super_block * sb)
{
	LOG_ENTRY();
	sb->s_inodes_count = DISK_TO_CPU32(sb->s_inodes_count);
	sb->s_blocks_count = DISK_TO_CPU32(sb->s_blocks_count);
	sb->s_r_blocks_count = DISK_TO_CPU32(sb->s_r_blocks_count);
	sb->s_free_blocks_count = DISK_TO_CPU32(sb->s_free_blocks_count);
	sb->s_free_inodes_count = DISK_TO_CPU32(sb->s_free_inodes_count);
	sb->s_first_data_block = DISK_TO_CPU32(sb->s_first_data_block);
	sb->s_log_block_size = DISK_TO_CPU32(sb->s_log_block_size);
	sb->s_log_frag_size = DISK_TO_CPU32(sb->s_log_frag_size);
	sb->s_blocks_per_group = DISK_TO_CPU32(sb->s_blocks_per_group);
	sb->s_frags_per_group = DISK_TO_CPU32(sb->s_frags_per_group);
	sb->s_inodes_per_group = DISK_TO_CPU32(sb->s_inodes_per_group);
	sb->s_mtime = DISK_TO_CPU32(sb->s_mtime);
	sb->s_wtime = DISK_TO_CPU32(sb->s_wtime);
	sb->s_mnt_count = DISK_TO_CPU16(sb->s_mnt_count);
	sb->s_max_mnt_count = DISK_TO_CPU16(sb->s_max_mnt_count);
	sb->s_magic = DISK_TO_CPU16(sb->s_magic);
	sb->s_state = DISK_TO_CPU16(sb->s_state);
	sb->s_errors = DISK_TO_CPU16(sb->s_errors);
	sb->s_minor_rev_level = DISK_TO_CPU16(sb->s_minor_rev_level);
	sb->s_lastcheck = DISK_TO_CPU32(sb->s_lastcheck);
	sb->s_checkinterval = DISK_TO_CPU32(sb->s_checkinterval);
	sb->s_creator_os = DISK_TO_CPU32(sb->s_creator_os);
	sb->s_rev_level = DISK_TO_CPU32(sb->s_rev_level);
	sb->s_def_resuid = DISK_TO_CPU16(sb->s_def_resuid);
	sb->s_def_resgid = DISK_TO_CPU16(sb->s_def_resgid);
	sb->s_first_ino = DISK_TO_CPU32(sb->s_first_ino);
	sb->s_inode_size = DISK_TO_CPU16(sb->s_inode_size);
	sb->s_block_group_nr = DISK_TO_CPU16(sb->s_block_group_nr);
	sb->s_feature_compat = DISK_TO_CPU32(sb->s_feature_compat);
	sb->s_feature_incompat = DISK_TO_CPU32(sb->s_feature_incompat);
	sb->s_feature_ro_compat = DISK_TO_CPU32(sb->s_feature_ro_compat);
	sb->s_algorithm_usage_bitmap = DISK_TO_CPU32(sb->s_algorithm_usage_bitmap);
	sb->s_journal_inum = DISK_TO_CPU32(sb->s_journal_inum);
	sb->s_journal_dev = DISK_TO_CPU32(sb->s_journal_dev);
	sb->s_last_orphan = DISK_TO_CPU32(sb->s_last_orphan);
	LOG_EXIT_VOID();
}


/*
 * NAME: fsim_get_ext2_superblock
 *
 * FUNCTION: Get and validate a ext2/3 superblock
 *
 * PARAMETERS:
 *      volume   - pointer to volume from which to get the superblock
 *      sb_ptr   - pointer to superblock
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 */
int fsim_get_ext2_superblock( logical_volume_t *volume, struct ext2_super_block *sb_ptr )
{
	int  fd;
	int  rc = 0;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = -rc;
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* get primary superblock */
	rc = fsim_rw_diskblocks( volume, fd, EXT2_SUPER_LOC, SIZE_OF_SUPER, sb_ptr, GET );

	if (rc == 0) {
		ext2fs_swap_super(sb_ptr);
		/* see if superblock is ext2/3 */
		if (( sb_ptr->s_magic != EXT2_SUPER_MAGIC ) ||
		    ( sb_ptr->s_rev_level > 1 ))
			rc = ENOENT;
	}

	EngFncs->close_volume(volume, fd);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * NAME: fsim_rw_diskblocks
 *
 * FUNCTION: Read or write specific number of bytes for an opened device.
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of disk
 *                        block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET (read) or PUT (write)
 *
 * RETURNS:
 *      FSIM_SUCCESS (0) for success
 *      EINVAL
 *      EIO
 *
 */
int fsim_rw_diskblocks( logical_volume_t * vol,
			int      dev_ptr,
			int64_t  disk_offset,
			int32_t  disk_count,
			void     *data_buffer,
			int      mode )
{
	int32_t  Bytes_Transferred;

	LOG_ENTRY();

	switch (mode) {
	case GET:
		Bytes_Transferred = EngFncs->read_volume(vol,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	case PUT:
		Bytes_Transferred = EngFncs->write_volume(vol,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	default:
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (Bytes_Transferred < 0) {
		LOG_EXIT_INT(-Bytes_Transferred);
		return -Bytes_Transferred;

	} else if (Bytes_Transferred != disk_count) {
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	LOG_EXIT_INT(FSIM_SUCCESS);
	return FSIM_SUCCESS;
}


/*
 * Test e2fsprogs version.
 *
 * We don't bother since we don't need any special functionality that
 * hasn't been around for *years*
 */
int fsim_test_version( )
{
	return 0;
}

