/*
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Local Disk Manager plugin.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <wait.h>

#include <plugin.h>
#include "localdskmgr.h"
#include "info.h"

/**
 * get_basic_info
 *
 * Returns basic info for the specified disk.
 **/
int get_basic_info(storage_object_t *disk,
		   extended_info_array_t **info)
{
	local_disk_t *ld = disk->private_data;
	extended_info_array_t *Info;
	int i = 0;

	LOG_ENTRY();

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     10 * sizeof(extended_info_t));
	if (!Info) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
		
	Info->info[i].name = EngFncs->engine_strdup("Name");
	Info->info[i].title = EngFncs->engine_strdup(_("Name"));
	Info->info[i].desc = EngFncs->engine_strdup(_("EVMS name for the DISK storage object"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(disk->name);
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Size");
	Info->info[i].title = EngFncs->engine_strdup(_("Size"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Size of the disk in sectors"));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].unit = EVMS_Unit_Sectors;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;
	Info->info[i].value.ui64 = disk->size;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Cyl");
	Info->info[i].title = EngFncs->engine_strdup(_("Cylinders"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Drive geometry -- number of cylinders"));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].value.ui64 = disk->geometry.cylinders;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Heads");
	Info->info[i].title = EngFncs->engine_strdup(_("Heads"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Drive geometry -- number of heads"));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].value.ui32 = disk->geometry.heads;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Sectors");
	Info->info[i].title = EngFncs->engine_strdup(_("Sectors"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Drive geometry -- sectors per track"));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].value.ui32 = disk->geometry.sectors_per_track;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("SectorSize");
	Info->info[i].title = EngFncs->engine_strdup(_("Sector Size"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Number of bytes per sector"));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].unit = EVMS_Unit_Bytes;
	Info->info[i].value.ui32 = disk->geometry.bytes_per_sector;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("BlockSize");
	Info->info[i].title = EngFncs->engine_strdup(_("Block Size"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Number of bytes per block"));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].unit = EVMS_Unit_Bytes;
	Info->info[i].value.ui64 = disk->geometry.block_size;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("BootLimit");
	Info->info[i].title = EngFncs->engine_strdup(_("Boot Cylinder Limit"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Logical block address of the first sector above the boot cylinder limit for this drive"));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].value.ui64 = disk->geometry.boot_cylinder_limit;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Segments");
	Info->info[i].title = EngFncs->engine_strdup(_("Segments"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Number of segments discovered on the drive (metadata, data, freespace)"));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].value.ui32 = EngFncs->list_count(disk->parent_objects);
	i++;

	Info->info[i].name = EngFncs->engine_strdup("Type");
	Info->info[i].title = EngFncs->engine_strdup(_("Type"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	Info->info[i].value.s = EngFncs->engine_strdup((ld->flags & LD_FLAG_IDE) ? "IDE" :
							   (ld->flags & LD_FLAG_SCSI) ? "SCSI" : "Unknown");
	i++;

	Info->count = i;
	*info = Info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * parse_hdparm_buffer
 *
 * Parse through the output buffer from the hdparm command and extract the
 * desired info fields and values.
 **/
static void parse_hdparm_buffer(char *buffer, hdparm_info_t *hdparm)
{
	char state[25] = {0};
	char *ptr;

	LOG_ENTRY();

	memset(hdparm, 0, sizeof(*hdparm));

	ptr = strstr(buffer, "readahead");
	if (ptr) {
		sscanf(ptr, "readahead = %d", &hdparm->read_ahead);
	}

	ptr = strstr(buffer, "busstate");
	if (ptr) {
		sscanf(ptr, "busstate = %d", &hdparm->bus_state);
	}

	ptr = strstr(buffer, "IO_support");
	if (ptr) {
		sscanf(ptr, "IO_support = %d", &hdparm->io_support);
	}

	ptr = strstr(buffer, "drive state is");
	if (ptr) {
		sscanf(ptr, "drive state is: %25s", state);
		if (!strcmp(state, "unknown")) {
			hdparm->drive_state = 0;
		} else if (!strcmp(state, "active/idle")) {
			hdparm->drive_state = 1;
		} else if (!strcmp(state, "standby")) {
			hdparm->drive_state = 2;
		} else if (!strcmp(state, "sleeping")) {
			hdparm->drive_state = 3;
		}
	}

	ptr = strstr(buffer, "using_dma");
	if (ptr) {
		sscanf(ptr, "using_dma = %d", &hdparm->using_dma);
	}

	ptr = strstr(buffer, "keepsettings");
	if (ptr) {
		sscanf(ptr, "keepsettings = %d", &hdparm->keep_settings);
	}

	ptr = strstr(buffer, "acoustic");
	if (ptr) {
		sscanf(ptr, "acoustic = %d", &hdparm->acoustic);
	}

	ptr = strstr(buffer, "multcount");
	if (ptr) {
		sscanf(ptr, "multcount = %d", &hdparm->multi_count);
	}

	ptr = strstr(buffer, "nowerr");
	if (ptr) {
		sscanf(ptr, "nowerr = %d", &hdparm->ignore_errors);
	}

	ptr = strstr(buffer, "unmaskirq");
	if (ptr) {
		sscanf(ptr, "unmaskirq = %d", &hdparm->unmask_irq);
	}

	LOG_EXIT_VOID();
}

/**
 * get_hdparm_info
 *
 * Run hdparm with the appropriate flags and save the output buffer. Search
 * through the buffer for each desired field and fill in the info structure.
 **/
static int get_hdparm_info(storage_object_t *disk,
			   hdparm_info_t *hdparm)
{
	char devname[EVMS_NAME_SIZE], *argv[4];
	int rc, status, bytes_read;
	int output_fd[2] = {0};
	char *buffer, *buf;
	pid_t pid;

	LOG_ENTRY();

	snprintf(devname, EVMS_NAME_SIZE, "%s/%s",
		 EVMS_OBJECT_NODE_DIR, disk->name);

	argv[0] = "hdparm";
	argv[1] = "-abcCdkmMnu";
	argv[2] = devname;
	argv[3] = NULL;

	/* Create a pipe and buffer to capture the output. */
	buffer = EngFncs->engine_alloc(4096);
	if (!buffer) {
		rc = errno;
		goto out;
	}

	rc = pipe(output_fd);
	if (rc) {
		rc = errno;
		goto out;
	}

	/* Run the hdparm command. */
	pid = EngFncs->fork_and_execvp(NULL, argv, NULL, output_fd, output_fd);
	if (pid == -1) {
		rc = EINVAL;
		goto out;
	}

	/* Wait for the command to finish. Read output from the pipe until
	 * the command is done.
	 */
	fcntl(output_fd[0], F_SETFL,
	      fcntl(output_fd[0], F_GETFL, 0) | O_NONBLOCK);

	buf = buffer;
	while (!(waitpid(pid, &status, WNOHANG))) {
		bytes_read = read(output_fd[0], buf, 4096 - (int)(buf - buffer));
		if (bytes_read > 0) {
			buf += bytes_read;
		}
		usleep(10000);
	}

	/* One more read in case we missed something. */
	bytes_read = read(output_fd[0], buf, 4096 - (int)(buf - buffer));

	rc = WIFEXITED(status) ? WEXITSTATUS(status) : EINTR;
	if (rc) {
		goto out;
	}

	parse_hdparm_buffer(buffer, hdparm);

out:
	EngFncs->engine_free(buffer);
	close(output_fd[0]);
	close(output_fd[1]);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * get_ide_info
 *
 * Return IDE-specific information about this disk.
 **/
int get_ide_info(storage_object_t *disk,
		 extended_info_array_t **info)
{
	hdparm_info_t hdparm;
	extended_info_array_t *Info;
	int rc, i = 0;

	LOG_ENTRY();

	rc = get_hdparm_info(disk, &hdparm);
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     10 * sizeof(extended_info_t));
	if (!Info) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	Info->info[i].name = EngFncs->engine_strdup("read_ahead");
	Info->info[i].title = EngFncs->engine_strdup(_("Read-Ahead"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Number of read-ahead sectors for the filesystem."));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].unit = EVMS_Unit_Sectors;
	Info->info[i].value.ui32 = hdparm.read_ahead;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("bus_state");
	Info->info[i].title = EngFncs->engine_strdup(_("Bus State"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup((hdparm.bus_state == 2) ? "Tri-State" :
						       (hdparm.bus_state == 1) ? "On" : "Off");
	i++;

	Info->info[i].name = EngFncs->engine_strdup("io_support");
	Info->info[i].title = EngFncs->engine_strdup(_("I/O Support"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Size of data transfers"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup((hdparm.io_support == 3) ? "32-bit w/ Sync" :
						       (hdparm.io_support == 1) ? "32-bit" : "16-bit");
	i++;

	Info->info[i].name = EngFncs->engine_strdup("drive_state");
	Info->info[i].title = EngFncs->engine_strdup(_("Drive State"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The drive's activity state."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup((hdparm.drive_state == 3) ? "Sleeping" :
						       (hdparm.drive_state == 2) ? "Standby" :
						       (hdparm.drive_state == 1) ? "Active/Idle" : "Unknown");
	i++;

	Info->info[i].name = EngFncs->engine_strdup("using_dma");
	Info->info[i].title = EngFncs->engine_strdup(_("Using DMA"));
	Info->info[i].type = EVMS_Type_Boolean;
	Info->info[i].value.b = hdparm.using_dma;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("keep_settings");
	Info->info[i].title = EngFncs->engine_strdup(_("Keep Settings"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Keep drive settings across a reset."));
	Info->info[i].type = EVMS_Type_Boolean;
	Info->info[i].value.b = hdparm.keep_settings;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("acoustic");
	Info->info[i].title = EngFncs->engine_strdup(_("Acoustic"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Automatic Acoustic Management (AAM) setting."));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].value.ui32 = hdparm.acoustic;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("multi_count");
	Info->info[i].title = EngFncs->engine_strdup(_("Multi-sector Count"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Count of multiple sector I/O"));
	Info->info[i].type = EVMS_Type_Unsigned_Int32;
	Info->info[i].unit = EVMS_Unit_Sectors;
	Info->info[i].value.ui32 = hdparm.multi_count;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("ignore_errors");
	Info->info[i].title = EngFncs->engine_strdup(_("Ignore Write Errors"));
	Info->info[i].type = EVMS_Type_Boolean;
	Info->info[i].value.b = hdparm.ignore_errors;
	i++;

	Info->info[i].name = EngFncs->engine_strdup("unmask_irq");
	Info->info[i].title = EngFncs->engine_strdup(_("Unmask IRQs"));
	Info->info[i].desc = EngFncs->engine_strdup(_("Unmask IRQs while processing interrupts for this drive."));
	Info->info[i].type = EVMS_Type_Boolean;
	Info->info[i].value.b = hdparm.unmask_irq;
	i++;

	Info->count = i;
	*info = Info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * get_scsi_info
 *
 * Return SCSI-specific information about this disk.
 **/
int get_scsi_info(storage_object_t *disk,
		  extended_info_array_t **info)
{
	LOG_ENTRY();

	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

