/*
 * (C) Copyright IBM Corp. 2001, 2003
 *
 * This program is free software;  you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY;  without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program;  if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: plugins/bbr_seg/options.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "bbr_seg.h"
#include "options.h"

/**
 * get_acceptable_create_objects
 *
 * All disks and segments are acceptable for creating a new BBR segment
 * (except for other BBR segments).
 **/
int get_acceptable_create_objects(task_context_t *context)
{
	storage_object_t *object;
	list_element_t itr1, itr2;
	int rc;

	LOG_ENTRY();

	rc = EngFncs->get_object_list(SEGMENT | DISK, DATA_TYPE, NULL,
				      NULL, VALID_INPUT_OBJECT,
				      &context->acceptable_objects);
	if (!rc) {
		LIST_FOR_EACH_SAFE(context->acceptable_objects, itr1, itr2, object) {
			if (object->plugin == my_plugin_record) {
				EngFncs->delete_element(itr1);
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * bbr_get_info
 *
 * Return information about a BBR segment.
 **/
int bbr_get_info(storage_object_t *segment, char *name,
		 extended_info_array_t **info)
{
	bbr_private_data_t *pdata = segment->private_data;
	extended_info_array_t *Info;
	int i, rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Getting info for segment %s.\n", segment->name);

	*info = NULL;

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     BBR_INFO_COUNT * sizeof(extended_info_t));
	if (!Info) {
		LOG_ERROR("Error allocating memory for extended info array.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	i = BBR_INFO_NAME_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Name");
	Info->info[i].title = EngFncs->engine_strdup(_("Name"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The unique name the system gave the segment."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(segment->name);

	i = BBR_INFO_UUID_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("UUID");
	Info->info[i].title = EngFncs->engine_strdup(_("UUID"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The unique id the system gave the segment."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(segment->uuid);

	i = BBR_INFO_SIZE_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Size");
	Info->info[i].title = EngFncs->engine_strdup(_("Size"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The size in sectors of the storage object after reserving space for metadata."));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].unit = EVMS_Unit_Sectors;
	Info->info[i].value.ui64 = segment->size;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

	i = BBR_INFO_BLKS_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Reserved Blocks");
	Info->info[i].title = EngFncs->engine_strdup(_("Reserved Blocks"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The number of replacement blocks reserved for this storage object."));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].value.ui64 = pdata->replacement_blocks_needed;

	i = BBR_INFO_BLKSZ_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Block Size");
	Info->info[i].title = EngFncs->engine_strdup(_("Block Size"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The size of each replacement block."));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].value.ui64 = pdata->block_size;

	i = BBR_INFO_TBLSZ_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Table Size");
	Info->info[i].title = EngFncs->engine_strdup(_("Size of BBR table"));
	Info->info[i].desc = EngFncs->engine_strdup(_("The number of sectors reserved for the BBR remapping table."));
	Info->info[i].type = EVMS_Type_Unsigned_Int64;
	Info->info[i].unit = EVMS_Unit_Sectors;
	Info->info[i].value.ui64 = pdata->bbr_table_size_in_sectors;
	Info->info[i].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

	Info->count = BBR_INFO_COUNT;
	*info = Info;

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * bbr_get_plugin_info
 **/
int bbr_get_plugin_info(char *descriptor_name, extended_info_array_t **info)
{
	extended_info_array_t *Info;
	char version_string[64];
	char required_engine_api_version_string[64];
	char required_plugin_api_version_string[64];
	int i, rc = 0;

	LOG_ENTRY();

	*info = NULL;

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     BBR_PLUGIN_INFO_COUNT * sizeof(extended_info_t));
	if (!Info) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	sprintf(version_string, "%d.%d.%d",
		MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);

	sprintf(required_engine_api_version_string, "%d.%d.%d",
		my_plugin_record->required_engine_api_version.major,
		my_plugin_record->required_engine_api_version.minor,
		my_plugin_record->required_engine_api_version.patchlevel);

	sprintf(required_plugin_api_version_string, "%d.%d.%d",
		my_plugin_record->required_plugin_api_version.plugin.major,
		my_plugin_record->required_plugin_api_version.plugin.minor,
		my_plugin_record->required_plugin_api_version.plugin.patchlevel);

	i = BBR_PLUGIN_INFO_SNAME_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Short Name");
	Info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
	Info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);

	i = BBR_PLUGIN_INFO_LNAME_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Long Name");
	Info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
	Info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);

	i = BBR_PLUGIN_INFO_TYPE_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Type");
	Info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
	Info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(_("Segment Manager"));

	i = BBR_PLUGIN_INFO_VERSION_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Version");
	Info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
	Info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(version_string);

	i = BBR_PLUGIN_INFO_REQENGSVCVERSION_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Required Engine Services Version");
	Info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
	Info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
						      "It will not run on older versions of the Engine services."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(required_engine_api_version_string);

	i = BBR_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX;
	Info->info[i].name = EngFncs->engine_strdup("Required Engine Plug-in API Version");
	Info->info[i].title = EngFncs->engine_strdup(_("Required Engine Plug-in API Version"));
	Info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires. "
						      "It will not run on older versions of the Engine plug-in API."));
	Info->info[i].type = EVMS_Type_String;
	Info->info[i].value.s = EngFncs->engine_strdup(required_plugin_api_version_string);

	Info->count = BBR_PLUGIN_INFO_COUNT;
	*info = Info;

	LOG_EXIT_INT(rc);
	return rc;
}

