/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: message.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>

#include "fullengine.h"
#include "memman.h"
#include "message.h"
#include "engine.h"
#include "handlemgr.h"

/*
 * Global data
 */
ui_callbacks_t * ui_callbacks = NULL;

/*
 * message_buffer has some added fudge space in the buffer to allow for the
 * header we prepend and cover any possible overrun from expanding the format
 * string.
 */
char message_buffer[MAX_USER_MESSAGE_LEN + 1024];

int engine_user_message(int    * answer,
			char * * choice_text,
			char   * message_fmt,
			...) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_message != NULL) {
			va_list args;

			if (engine_mode & ENGINE_DAEMON) {
				strcpy(message_buffer, "Daemon: ");
			} else {
				strcpy(message_buffer, "Engine: ");
			}

			va_start(args, message_fmt);
			vsprintf(message_buffer + strlen(message_buffer), message_fmt, args);
			va_end(args);

			/*
			 * Put the message (and later, any answer) into the log.  Use the
			 * CRITICAL level so that the message will always go to the log,
			 * even though the message itself may not be CRITICAL.
			 */
			LOG_CRITICAL("Message is: %s\n", message_buffer);
			rc = ui_callbacks->user_message(message_buffer, answer, choice_text);

			if (rc == 0) {
				if ((answer != NULL) && (choice_text != NULL)) {
					LOG_CRITICAL("Answer is: \"%s\"\n", choice_text[*answer]);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int plugin_user_message(plugin_record_t * plugin,
			int             * answer,
			char          * * choice_text,
			char            * message_fmt,
			...) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_message != NULL) {
			va_list args;

			strcpy(message_buffer, plugin->short_name);
			strcat(message_buffer, ": ");

			va_start(args, message_fmt);
			vsprintf(message_buffer + strlen(message_buffer), message_fmt, args);
			va_end(args);

			/*
			 * Put the message (and later, any answer) into the log.  Use the
			 * CRITICAL level so that the message will always go to the log,
			 * even though the message itself may not be CRITICAL.
			 */
			LOG_CRITICAL("Message is: %s\n", message_buffer);
			rc = ui_callbacks->user_message(message_buffer, answer, choice_text);

			if (rc == 0) {
				if ((answer != NULL) && (choice_text != NULL)) {
					LOG_CRITICAL("Answer is: \"%s\"\n", choice_text[*answer]);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int plugin_user_communication(void                * object_instance,
			      char                * message_text,
			      option_desc_array_t * options){
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_communication != NULL) {

			task_context_t * task = engine_alloc(sizeof(task_context_t));

			if (task != NULL) {
				task_handle_t task_handle;

				/*
				 * Fill in the appropriate task fields.  The rest are
				 * left 0 (NULL).
				 */

				task->plugin = NULL;
				task->object = object_instance;
				task->action = EVMS_Task_Message;

				task->option_descriptors = options;

				/* Make a handle for the task. */
				rc = create_handle(task,
						   TASK,
						   &task_handle);

				if (rc == 0) {

					/* Call the UI callback. */
					rc = ui_callbacks->user_communication(message_text, task_handle);

				} else {
					LOG_WARNING("create_handle() returned error %d.\n", rc);
				}

				engine_free(task);

			} else {
				LOG_CRITICAL("Memory allocation of task_context_t failed.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


#define NUM_TIMERS	1024
typedef struct timer_data_s {
	u_int64_t	timers[NUM_TIMERS];
	u_int64_t	counts[NUM_TIMERS];
	u_int64_t	rates[NUM_TIMERS];
	int		oldest;
	int		newest;
} timer_data_t;

static void calculate_time_estimate(progress_t * progress) {

	timer_data_t * timer_data;
	int newest;
	int oldest;
	u_int64_t block_time;
	u_int64_t block_count;
	u_int64_t ms_per_chunk;
	struct timeval time;
	struct timezone tz;

	LOG_PROC_ENTRY();

	if (progress->timer_data == NULL) {
		LOG_DEBUG("progress has no plug-in private data.  Can't calculate a time estimate without timer data.\n");
                LOG_PROC_EXIT_VOID();
		return;
	}

	timer_data = progress->timer_data;
	newest = timer_data->newest;
	oldest = timer_data->oldest;

	if (newest < NUM_TIMERS - 1) {
		newest++;
	} else {
		newest = 0;
	}
	timer_data->newest = newest;

	if (newest == oldest) {
		if (oldest < NUM_TIMERS - 1) {
			oldest++;
		} else {
			oldest = 0;
		}
	}

	gettimeofday(&time, &tz);
	timer_data->timers[newest] = (u_int64_t) time.tv_sec * (u_int64_t) 1000000 + (u_int64_t) time.tv_usec;
	timer_data->counts[newest] = progress->count;

	/*
	 * Don't use more than the last thirty seconds of data
	 * for making an estimate.
	 */
	while (timer_data->timers[newest] - timer_data->timers[oldest] > 30000000) {
		int next_oldest = oldest;

		if (next_oldest < NUM_TIMERS - 1) {
			next_oldest++;
		} else {
			next_oldest = 0;
		}
		if (next_oldest == newest) {
			break;
		} else {
			oldest = next_oldest;
		}
	}
	timer_data->oldest = oldest;

	block_time  = timer_data->timers[newest] - timer_data->timers[oldest];
	block_count = timer_data->counts[newest] - timer_data->counts[oldest];
	if (block_count != 0) {
		ms_per_chunk = block_time / block_count;
	} else {
		ms_per_chunk = UINT64_MAX;
	}
	timer_data->rates[newest] = ms_per_chunk;

	/*
	 * Wait until we have at least five seconds of data before making an
	 * estimate of the remaining time.
	 */
	if (timer_data->timers[newest] - timer_data->timers[oldest] > 5000000) {

		/*
		 * Can't give an estimate if the count hasn't incremented at all
		 * across the array of timers.
		 */
		if (block_count != 0) {
			u_int64_t rem_chunks = progress->total_count - progress->count;
			u_int64_t average_ms_per_chunk = 0;
			u_int64_t rem_time_ms;
			uint rem_secs;
			int i = 0;
			int j = oldest;

			while (j != newest) {
				if (timer_data->rates[j] != -1) {
					average_ms_per_chunk += timer_data->rates[j];
					i++;
				}
				if (j < NUM_TIMERS - 1) {
					j++;
				} else {
					j = 0;
				}
			}

			average_ms_per_chunk /= i;

			/*
			 * The calculation:
			 *   rem_time_ms = rem_chunks * ms_per_chunk;
			 * can cause rem_time_ms to drift wildly between
			 * progress reports.  ms_per_chunk fluctuates.  It also
			 * loses precision from the division since it is an
			 * integer.  For large values of rem_chunks, the
			 * fluctuations and loss of precision get amplified,
			 * resulting in drifting time estimates.
			 *
			 * So, use the time and count for the whole block when
			 * calculating rem_time_ms.  It helps preserve some
			 * precision which keeps the times more consistent
			 * between the progress reports.
			 */
			rem_time_ms = (rem_chunks / block_count) * block_time +
				      (rem_chunks % block_count) * average_ms_per_chunk;

			/*
			 * Add a half second before rounding to seconds.
			 * Without it, the calculation will round down to zero
			 * as soon as rem_time_ms is under 1000000.  That leaves
			 * the time estimate at zero for the last second.
			 * Depending on how the UI handles it, the time estimate
			 * either reads 00:00 or it disappears for the last
			 * second before progress is complete.  Either one looks
			 * funny.  The extra half second makes the time estimate
			 * appear to hit zero when progress is complete.
			 */
			rem_secs = (rem_time_ms + 500000) / 1000000;

			/*
			 * This check helps dampen the bouncing of the estimate
			 * due to fractional noise from the calculations above.
			 */
			if (abs(rem_secs - progress->remaining_seconds) > 3) {
				progress->remaining_seconds += (int) (rem_secs - progress->remaining_seconds) / 2;
			} else {
				if (rem_secs < progress->remaining_seconds) {
					progress->remaining_seconds = rem_secs;
				}
			}
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


int plugin_progress(progress_t * progress) {

	int rc;

	LOG_PROC_ENTRY();

	LOG_DEBUG("    id:                %d\n", progress->id);
	LOG_DEBUG("    title:             %s\n", progress->title);
	LOG_DEBUG("    description:       %s\n", progress->description);
	LOG_DEBUG("    type:              %s\n", (progress->type == DISPLAY_PERCENT) ? "DISPLAY_PERCENT" :
						 (progress->type == DISPLAY_COUNT)   ? "DISPLAY_COUNT" :
						 (progress->type == INDETERMINATE)   ? "INDETERMINATE" :
			       							       "(unknown)");
	LOG_DEBUG("    count:             %"PRIu64"\n", progress->count);
	LOG_DEBUG("    total_count:       %"PRIu64"\n", progress->total_count);

	if (ui_callbacks == NULL) {
		LOG_DEBUG("There are no UI callbacks.\n");
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (ui_callbacks->progress == NULL) {
		LOG_DEBUG("The UI did not provide a progress callback.\n");
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (progress->timer_data != NULL) {
		calculate_time_estimate(progress);
	}

	LOG_DEBUG("    remaining_seconds: %u\n", progress->remaining_seconds);
	
	/*
	 * Allocate an initilaize the timer data on the first progress call
	 * if the progress is not indeterminate and the caller did not forbid
	 * time estimates.
	 */
	if ((progress->id == 0) &&
	    (progress->type != INDETERMINATE) &&
	    !(progress->flags & PROGRESS_NO_TIME_ESTIMATE)) {

		progress->timer_data = engine_alloc(sizeof(timer_data_t));
		if (progress->timer_data != NULL) {
			struct timeval  time;
			struct timezone tz;

			/* Fill in the first entry */
			gettimeofday(&time, &tz);
			progress->timer_data->timers[0] = (u_int64_t) time.tv_sec * (u_int64_t) 1000000 + (u_int64_t) time.tv_usec;
			progress->timer_data->oldest = 0;
		}
	}

	rc = ui_callbacks->progress(progress);

	if (progress->count >= progress->total_count) {
		/* engine_free() handles NULL pointers. */
		engine_free(progress->timer_data);
		progress->timer_data = NULL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
