/*
 * This file is licensed under the terms of the GNU General Public License,
 * version 2. See the file COPYING in the main directory for details.
 * 
 *  Copyright (C) 2002,2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 */

#include "delo.h"
#include "stringops.h"

/*
 * A list of extents.
 */

static struct extentlist *extentlist_head = NULL;
static struct extentlist *extentlist_last;

void extentlist_clear(void)
{
	while (extentlist_head) {
		struct extentlist *tmp = extentlist_head;
		extentlist_head = tmp->next;
		free(tmp);
	}
}

int extentlist_add_tail(unsigned int extent, unsigned int count)
{
	struct extentlist *tmp;

	tmp = (struct extentlist *)malloc(sizeof(struct extentlist));
	if (!tmp)
		return 1;

	tmp->next = NULL;
	tmp->extent = extent;
	tmp->count = count;

	if (extentlist_head)
		extentlist_last->next = tmp;
	else
		extentlist_head = tmp;

	extentlist_last = tmp;
	
	return 0;
}

unsigned int extentlist_size(void)
{
	struct extentlist *tmp = extentlist_head;
	int count = 0;

	while (tmp) {
		count += tmp->count;
		tmp = tmp->next;
	}
	return count * SECTOR_SIZE;
}

void extentlist_dump(void)
{
	struct extentlist *tmp = extentlist_head;

	printf("Extents: ");
	while (tmp) {
		printf("%d/%d ", tmp->extent, tmp->count);
		tmp = tmp->next;
	}
	printf("\n");
}

/*
 * File I/O.
 */

int delo_open(char *partition, char *file)
{
	/* Check for iso9660 fs, which has no partitions. */
	if (!readisoblocks(file))
		return 0;

	dprintf("%s not in iso9660 filesystem\n", file);

	/* Check for ext2/ext3 fs, this checks also the partition table. */
	if(!readext2blocks(partition, file))
		return 0;

	dprintf("%s not in ext2 filesystem on partition %s\n", file,
		partition);

	/* Error */
	return 1;
}

int delo_read(void *dest, unsigned int offset, unsigned int len, int progress)
{
	struct extentlist *list = extentlist_head;
	int l;
	int rl;
	int bl;

	dprintf("read %x-%x->%p-%p: ", offset, offset + len, dest,
		dest + len);

	/*
	 * The REX firmware only pretends to read bytes, in fact it clobbers
	 * some bytes beyond the end, too. Thus we read always full blocks.
	 */

	if (list && offset) {
		unsigned int count = offset / SECTOR_SIZE;
		unsigned int rest = offset % SECTOR_SIZE;

		/* Find first block. */
		while (list) {
			if (list->count > count)
				break;
			count -= list->count;
			list = list->next;
		}

		/* Read first partial block. */
		l = MIN(SECTOR_SIZE, rest + len);
		if (rest && l) {
			char block[SECTOR_SIZE];

			dprintf("\n[%d]+%x-%x->%p-%p ", list->extent + count,
				rest, l - rest, dest, dest + l - rest);

			if (bootread(list->extent + count, block, l) != l)
				return 1;
			memcpy(dest, block + rest, l - rest);
			dest += l - rest;
			len -= l - rest;
			count++;
		}

		/* Read rest of first extent. */
		l = MIN((list->count - count) * SECTOR_SIZE, len);
		rl = l % SECTOR_SIZE;
		bl = l - rl;

		if (l)
			dprintf("\n[%d]-%x->%p-%p ", list->extent + count, l,
				dest, dest + l);

		if (bl && (bootread(list->extent + count, dest, bl) != bl))
			return 1;
		if (rl) {
			char block[SECTOR_SIZE];
			unsigned int re = list->extent + count + bl / SECTOR_SIZE;

			if ((bootread(re, block, rl) != rl))
				return 1;
			memcpy(dest + bl, block, rl);
		}
		dest += l;
		len -= l;
		list = list->next;
		if (progress)
			printf(".");
	}

	/* Read subsequent extents. */
	while (list && len) {
		l = MIN(list->count * SECTOR_SIZE, len);
		rl = l % SECTOR_SIZE;
		bl = l - rl;

		dprintf("\n[%d]-%x->%p-%p ", list->extent, l, dest, dest + l);

		if (bl && (bootread(list->extent, dest, bl) != bl))
			return 1;
		if (rl) {
			char block[SECTOR_SIZE];
			unsigned int re = list->extent + bl / SECTOR_SIZE;

			if ((bootread(re, block, rl) != rl))
				return 1;
			memcpy(dest + bl, block, rl);
		}
		dest += l;
		len -= l;
		list = list->next;
		if (progress)
			printf(".");
	}

	dprintf("\n");

	return len ? 1 : 0;
}

int readfile(void **dest, char *partition, char *file, size_t align)
{
	unsigned int size;

	/* Begin of progress indicator. */
	printf("Loading %s ", file);

	if (delo_open(partition, file))
		return 0;

	size = extentlist_size();

	if (!(*dest = malloc_aligned(size, align)))
		return 0;

	if (delo_read(*dest, 0, size, 1))
		return 0;

	/* End of progress indicator. */
	printf(" ok\n");

	return size;
}
