/* Authors:  Jens Peter Secher (jpsecher@diku.dk)
 *           Arne Glenstrup (panic@diku.dk)
 *           Henning Makholm (makholm@diku.dk)
 * Content:  C-Mix gegen subsystem: Fitting it all together
 * History:  Derived from theory by Peter Holst Andersen and Lars Ole Andersen.
 *
 * Copyright  1998. The TOPPS group at DIKU, U of Copenhagen.
 * Redistribution and modification are allowed under certain
 * terms; see the file COPYING.cmix for details.
 */

#include <cmixconf.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "gegen.h"
#include "directives.h"

GegenStream &
operator <<(GegenStream &ost,GegenEnv::multichar mc)
{
    while(mc.count-- > 0 ) ost << mc.ch ;
    return ost ;
}

static void
banner(GegenStream &ost, char const *text)
{
  unsigned spaces = 68-strlen(text);
  ost << "/" << GegenEnv::multichar('*',70) << "\n *"
      << GegenEnv::multichar(' ',68) << "*\n *"
      << GegenEnv::multichar(' ',spaces/2) << text
      << GegenEnv::multichar(' ',(spaces+1)/2) << "*\n *"
      << GegenEnv::multichar(' ',68) << "*\n "
      << GegenEnv::multichar('*',70) << "/\n\n" ;
}

GegenStream &
GegenEnv::oksection(char const *newsection)
{
  ost << '\n' ;
  if ( newsection != cursection && strcmp(newsection,cursection) == 0 ) {
    cursection = newsection ;
    banner(ost,newsection);
  }
  return ost ;
}

bool
GegenEnv::spectime(C_Type *t)
{
  return bt.Static(t->array_contents_type());
}

bool
GegenEnv::spectime(C_Decl *d)
{
  return spectime(d->type);
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////// 

static char * date_as_string() {
  time_t timer ;
  time(&timer);
  return ctime(&timer);
}

void
GegenEnv::emit_header()
{
  oksection("A GENERATING EXTENSION CREATED BY C-MIX/II");
  ost << "/* This is AUTOMATICALLY GENERATED code!\n"
    " * It would be foolish to try editing it\n"
    " * (unless you're debugging C-Mix/II and "
    "know what you're doing, that is)\n *\n"
    " * Generation date: " << date_as_string() << " */\n\n" ;
  
  foreach(i,directives.headers,Plist<const char>) {
    ost << "#include " << *i << '\n' ;
  }
  const char *speclib_h = getenv("CMIX_SPECLIB_H");
  if ( speclib_h == NULL )
    speclib_h = "<cmix/speclib.h>" ;
  ost << "#include " << speclib_h << "\n"
    "#if cmixSPECLIB_VERSION != 2011\n"
    "#error Wrong speclib version\n"
    "#endif\n" ;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

void
GegenEnv::emit_prototypes()
{
  oksection("Function prototypes");
  
  foreach(i,pgm.functions,Plist<C_Decl>)
    specializable_heading(*i,true);
  
  SymbolTable<const char> externs_made ;
  foreach(ii,pgm.exfuns,Plist<C_Decl>)
    if ( bt.Static(ii->type) ) {
      const char *name = ii->get_name() ;
      if ( directives.is_wellknown(name) || externs_made.lookup(name) )
        continue ;
      externs_made.insert(name,name);
      ost << Unqualified(ii->type->ptr_next(),name) << ";\n";
    }
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////.

void
GegenEnv::define_initfun()
{
  oksection("Initialization function") ;
  ost << "void\ncmixGenInit()\n{\n"
    "  cmixSpeclibInit();\n" ;
  
  init_struct();
  foreach(d,pgm.globals,Plist<C_Decl>)
    NameDynamicObject(*d,cmixGlobal);
  memo.emit_globalmemocode() ;
  
  
  // find all the external functions
  if ( 1 ) { // local scope for the symbol table
    bool first_extfun = true ;
    SymbolTable<const char> exfunsmet ;
    foreach(e,pgm.exfuns,Plist<C_Decl>) {
      if ( bt.Static(e->type) )
        continue ;
      char const*const name = e->get_name() ;
      if ( directives.is_wellknown(name) )
        continue ;
      if ( !exfunsmet.lookup(name) ) {
        if ( first_extfun )
          ost << "  cmixDeclare(cmixGlobal,0," << enter ;
        else
          ost << ";\\n\"\n\t\"" ;
        first_extfun = false ;
        ost << Unqualified(e->type->ptr_next(),e->get_name()) ;
        exfunsmet.insert(name,name);
        res_taboos.push_back(e->get_name());
      }
    }
    if ( !first_extfun )
      ost << "\"\n\t\"" << exit << ");\n" ;
  }
  ost << "  return;\n"
    "}\n" ;
}

void
GegenEnv::define_exitfun()
{
  oksection("Clean-up and write-out function") ;
  
  ost << "void\ncmixGenExit(FILE *fp)\n{\n" 
    "  fprintf(fp,\"/* This program was generated by C-Mix/II\\n\"\n"
    "\t\" *\\n\"\n"
    "\t\" * THIS IS AUTOMATICALLY GENERATED CODE\\n\"\n"
    "\t\" */\\n\"\n" ;
  foreach(i,directives.headers,Plist<const char>) {
    ost << "\t\"#include " ;
    for ( const char *pc = *i ; *pc ; pc++ )
      switch(*pc) {
      case '\"': case '\\':
        ost << '\\' ;
      default:
        ost << *pc ;
      }
    ost << "\\n\"\n" ;
  }
  ost << "\t);\n" ;
  exit_struct();
  ost << "  cmixUnparse(cmixNametable,cmixNametableS,"
      << resnames.get_size() << ",fp);\n}\n" ;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

GegenEnv::GegenEnv(ostream &o,C_Pgm const &PGM,
                   BTresult const &BT, PAresult const &PA,
                   SAresult const &SA, DFresult const &DF,
                   NameMap const &NAMES, Pset<C_UserDef> &RESSTRUCTS)
  : ost(o,BT,NAMES,RESSTRUCTS), cursection("foo!"), pgm(PGM),
    bt(BT), pa(PA), names(NAMES), residual_structs(RESSTRUCTS), PutName(0),
    memo(*this,DF), sa(SA), max_residual_versions(NULL)
{
}

extern const char *PresTaboos[] ; // in taboos.cc, from taboos.

void
emitGeneratingExtension(C_Pgm const&corepgm,
                        BTresult const&bt,
                        PAresult const&pa,
                        SAresult const&sa,
                        DFresult const&df,
                        ostream &ost)
{
  NameMap names ;
  Pset<C_UserDef> residual_structs ;
  MakeUniqueNames(corepgm,bt,names,&residual_structs) ;
  
  GegenEnv env(ost,corepgm,bt,pa,sa,df,names,residual_structs);
  
  foreach(i,directives.debug_flags,Plist<DebugDirective>) {
    if ( i->subject_def == NULL )
      Diagnostic(WARNING,i->pos)
        << "ignoring debugging request for unknown function "
        << i->subject_name ;
    else
      env.max_residual_versions[i->subject_def->translated] = i->limit ;
  }           
  
  env.emit_header();
  env.pgen_usertype_fwds();
  env.pgen_usertype_decls();
  env.define_cmixMember();
  env.define_cmixPutName();
  env.memo.follow_functions();
  env.define_pgen_globals();
  env.memo.emit_globalmemodata();
  env.emit_prototypes();
  env.define_functions();
  env.define_initfun();
  
  foreach(t,directives.taboos,Plist<const char>)
    env.resnames.add_taboo(*t);
  foreach(tt,env.res_taboos,Plist<const char>)
    env.resnames.add_taboo(*tt);
  env.resnames.add_taboos(PresTaboos);
  env.resnames.emittable(ost) ;
  
  env.define_exitfun();
}
