#!/usr/bin/perl -w

package Bloksi ;

sub sayif ;
sub bloksi_datestr ;

use strict "vars";

use Bloksi::Parse ;

use Exporter   ;
@Bloksi::ISA = qw(Exporter) ;
@Bloksi::EXPORT = qw(&new         &load
         	     &may_move    &move
	             &move_clean  &unmake &remake &register
	             &piece_mask  &piece_list
                     &iden_shape
	             @dirs 
	             &sayif
	            );


@Bloksi::dirs = ("left","right","down","up");

=head1 NAME

Bloksi - Basic string-based puzzle object.

=head1 DESCRIPTION

=head1 SYNOPSIS 

=head1 new Bloksi $string

A Bloksi is a blessed ref to hash whith keys

  s       => $string,         # Represents the puzzle
  w       => $width
  size    => [$width,$heigth]
  unmake  => [ ... ]          # Previous moves (can be undone)
  remake  => [ ... ]          # Moves that can be redone
  nmoves  => $number          # Number of moves until now
  pieces  => { $p => $m,... } # Masks as returned by piece_mask()
  iden    => { $p => $q,... } # Identify identical pieces.

Elements in 'unmake' and 'remake' lists have keys :

  s       => $string          # Represents the puzzle
  moved   => [$piece1,...]    # Pieces that have moved
  nmoves  => $curr_num_moves  # Number of moves
  shape   => $shape           # Puzzle, as returned by iden_shape()
  dir     => $dir             # Direction of move


After the graphical object has been built by C<canvas_root()>, each
piece has a corresponding Gnome::CanvasGroup, accessible through
C<$blok-E<gt>{canvasp}-E<gt>{$p}>.

=cut

sub new
{
    my ($class,$s) = @_ ;

    ## my $sss = $s; $sss =~ s/\n/|/g; 
    ## print "New bloksi : >$sss<\n";

    $s =~ /(.*)/; 

    my $g = bless {
	s       => $s,
	initial => $s,
	w       => length($1),	# Width
	remake  => [],		# Redo moves
	nmoves  => 0,
	file    => "noname.lev", # HERE : Suppress this field?
    } ;
				# Compute identifiers table
    my %iden = ();
    foreach (split "",$s){
	next if /[\n\s]/ ;
	next if defined $g->{pieces}->{$_} ;
	$g->{pieces}->{$_} = piece_mask( $g, $_ );
	my $id =  $g->{pieces}->{$_}->{iden} ;
	$iden{$id} = $_ unless defined $iden{$id} ;
	$g->{iden}->{$_} = $iden{$id} ;
	sayif 0, "Piece '$_' == '$iden{$id}' => '$id'\n";
    }

    $g->{height} =  $g->{s} =~ s/(.)$/$1/mg ; # number of rows

    $g->{unmake} = [ {s      => $s,
		      moved  => [],
		      shape  => iden_shape($g),
		      nmoves => 0,
		      dir    => "none",
		       } ] ;
    sayif 0,  "shape = \n$g->{unmake}->[0]->{shape}\n";
    return $g ;
}

# Returns a string that does not vary when the names of pieces of same
# shape are varied.
##
## Uses $g->{s,w,iden}
sub iden_shape {
    ## return "";
    my $g = shift ;
    my $s = $g->{s};
    my @w = (" ") x (2*$g->{w});
    my $res = "";
    my $l;
    foreach $l (split "\n",$s){
	my $p = "";
	my ($q,@u,@v) ;
	@u = map { $q = /$p/ ? $p:" "; $p=$_; ($q,$_) } $l =~ /(.)/g ;
	@v = map { shift(@w) eq $_ ? $_ : " " } @u ;
	$res .= "\n" . join("",@v) . "\n" . join("",@u) ;
	@w = @u ;
    }
    $res =~ s/\n.*\n//m ;	# Remove superfluous line
				# Identify identical pieces
    $res =~ s/(\w)/$g->{iden}->{$1}/g ;
    return $res ;
}

=head2 C<@res = may_move( $g, $piece, $dir )>

Checks if it is possible to move $piece in direction $dir, eventually
by pushing some other pieces, in $dir. If the move is possible, return
the list of pieces that are moved. Returns () otherwise.

Uses keys of C<$g> 's' and 'w'.

=cut

sub may_move
{
    my ($g,$piece,$dir,$exclude) = @_ ;

    $exclude = "" unless defined $exclude;

    return () if $piece !~ /\w/ || ($dir eq "none") ;

    my $regex ;

    die "Bloksi::may_move : attempt to may_move in direction '$dir'\n"
	unless defined ( $regex =
			 {"left" =>"(.)(?=$piece)",
			  "right"=>"$piece(?=(.))",
			  "down" =>"$piece(?=.{$g->{w}}(.))",
			  "up"   =>"(.)(?=.{$g->{w}}$piece)",
		      }->{$dir} ) ;

    ## sayif 1,  "\$regex is '$regex'\n" ;

    my @a = $g->{s} =~ /$regex/sg; 
    my @b = $g->{s} =~ /($piece)/g; 
    my (@res,@r2) ;
    ## sayif 1,  "\@a has length ",0+@a," \@b has length ",0+@b,"\n";

				# There should be one neighbor per
				# piece. Otherwise, I'm pushing
				# against edge?
    ## print "may_move : ",0+@a," ",0+@b,"\n" unless @a+0 == @b; # Check
    return () unless @a+0 == @b;
				# poor man's "uniq"
    ## print "--", join ",",@a,"\n";
    my $tmp="";
    @a = map { $tmp ne $_ ? ($tmp=$_) : ()} sort @a;
    ## print "++", join ",",@a,"\n";
    @a = grep { ! /[$exclude]/ } @a if $exclude;
    ## print "**", join ",",@a,"\n";
    ## print "%% $exclude\n";
    my $ex2 = join("", @a) . $exclude;
    foreach (@a)
    {
	## sayif 1,  "found '$_'\n";
	next if $_ eq " " or $_ eq $piece ;
	## sayif 1,  "trying to push $_ in dir $dir\n";
	return () unless @r2 = may_move($g,$_,$dir,$ex2);
	## sayif 1,  "push $_ in dir $dir worked!\n";
	push @res, @r2 ;
    }
    push @res, $piece;
    # sayif 1,  "Must move '",join("','",@res),"' in order to move $piece\n"
    #   if 0+@res ;
    # Do a "uniq" on the list of pieces
    ## @res = @{uniq \@res} ;
    ## my $tmp = '' ;
    $tmp = '';
    @res = grep { ($_ ne $tmp) && (($tmp=$_),1) } sort @res ;
    ## print "### " unless $exclude;
    ## print "move $piece, $dir --", join (",",@res),"\n"; ## if $exclude;
    return @res ;
}

=head2 C<move_dumb( $g, [$piece1,$piece2,..], $dir )>

Move pieces $piece1,$piece2,... in direction $dir. Does not check if
the move is valid. Does not return anything. Only C<$g->{s}> is
changed.

Uses keys of C<$g> 's' and 'w'.

=cut

sub move_dumb
{
    my ($g,$pieces,$dir) = @_ ;

    return unless @$pieces;
    # sayif 1,  "move_dumb '",join("','",@$pieces),"' in direction '$dir'\n";

    my $off ;				# Offset to new position
    unless ( defined( $off =
		      {"left" => -1,
		       "right"=>  1,
		       "down" =>  $g->{w}+1,
		       "up"   => -$g->{w}-1,
		      }->{$dir} ) )
      {
	die "Bloksi::move : attempt to move in direction '$dir'\n" ;
      }


    my @where = () ;
    my @what = () ;
				# Find current position of moved pieces
    foreach( @$pieces ){	
	my $tmp = -1 ;		# If only @a = index ... would work ...
	while ( ($tmp = index($g->{s},$_,$tmp+1)) > $[ - 1 ) {
	    push @where, $tmp ;
	    push @what , $_ ;
	}
    }
				# Take pieces out
    my $ps = "[". join("",@$pieces) . "]" ;
    $g->{s} =~ s/$ps/ /g;

				# Put piecs back in new location
    foreach( @where ){
	substr $g->{s}, $_ + $off, 1, shift @what ;
    }
}

=head2 C<@res = move( $g, $piece, $dir )>

Move C<$piece> in direction C<$dir>, if the move is valid. Returns the
list of the pieces that were effectively moved. C<$g> is changed to
reflect that move.

Uses keys of C<$g> 's' and 'w'.

=cut

sub move
{
    my ($g,$piece,$dir) = @_ ;
    my @r = may_move( $g, $piece, $dir );

    # sayif 1,  @r ? "" : "can't " , "move '$piece' in direction '$dir'\n" ;
    # sayif 1,  "  this moves '",join("','",@r),"'\n" if @r ;
    move_dumb($g,\@r,$dir) if 0+@r ;
    ## sayif 0,  "After move : \n$g->{s}\n";
    ## sayif 0,  "Shape       :\n",iden_shape($g),"\n";
    return @r ;
}

sub piece_list			# Return all distinct 'pieces'.
{
    my $pieces = shift ;	
    $pieces =~ s/[^0-9A-Za-z]//g ;
    $pieces = join "",sort split "",$pieces ;
    $pieces=~s/(\w)\1+/$1/g;
    ## print "Pieces are : '$pieces'\n";
    split "",$pieces ;
}

sub print_hist			# Pretty printer
{
    my @a = map { [split "\n", $_] } splice @_, 0, 5 ;
    DONE: while (1){
	foreach (@a){
	    last DONE unless @$_ ;
	    print shift @$_, "|" ;
	}
	print "\n" ;
    }
}

=head2 C<register($g,$moved,$dir)>

Registers a move, supposing it has already been performed with
C<$g-E<gt>move()> or C<$g-E<gt>move_dumb()>. All that is left to do is
book-keeping in C<$g-E<gt>{(un|re)make}>. 

If C<$g> is back to a previous position, the intermediate moves are
taken out of history and move count is reduced accordingly.

Also, checks if the target has been reached.

=cut

sub register
{
    my ($g,$moved,$dir) = @_ ;
    
    my $mv1 = join "",sort @$moved ;
    my $mv2 = join "",sort @{$g->{unmake}->[0]->{moved}} ;
    my $common1 = $mv1 ne "" ? $mv2 =~ s/([$mv1])/$1/g : 0 ;

    my $tmp = join "", sort @$moved,@{$g->{unmake}->[0]->{moved}} ;
				# common pieces with last move
    
    my $common = 0+($tmp =~ s/(.)\1/$1$1/g) ;
    warn "\$common1=$common1 != \$common=$common (\$tmp=$tmp)\n" 
	unless $common1==$common;
				# New moved pieces
    my $count = 0+ @$moved - $common ;
    my $ocnt = 0+ @{$g->{unmake}->[0]->{moved}} - $common ;
				# Check for goal
    my $target = $g->{target} ;
    $target =~ s/ /./g ;
    if( $g->{s} =~ /$target/ ){
	sayif 0,  "GOAL!!!\n";
	if( ! defined $g->{solved} ){
	    $g->{solved} = 1 ;
	    ## Should be like &$g->{callbacks}->{target_reached}
	    &main::congratulations() ;
	}
    }

				# Check for looping    
    my $shape = iden_shape( $g );
    my $hcnt = 0 ;

    foreach (@{$g->{unmake}}){
	$hcnt++ ;
	sayif 0, "no shape for $hcnt\n" unless defined $_->{shape} ;
	next unless defined $_->{shape} ;
				# Loop detected!
	if( $_->{shape} eq $shape ){ 
	    sayif 0,  "Whoa !! Looping ($hcnt moves) \n";

	    # print_hist $shape,map {$_->{shape}} @{$g->{unmake}} ;

	    splice @{$g->{unmake}}, 0, $hcnt-1 ;

				# This will rename some pieces. I
				# should rename elements in @$moved.
	    &main::loop_detected( $g, $moved );
	    
				# update 
	    @{$g}{qw/nmoves s/} = @{$g->{unmake}->[0]}{qw/nmoves s/};
	    sayif 0,  "Back to \n$g->{s}\n " ;
	    return ;
	} else {
	}
    }

    unshift @{$g->{unmake}}, { s      => $g->{s}, 
			       moved  => $moved,
			       nmoves => $g->{nmoves}+$count,
			       shape  => $shape,
			       dir    => [],
			   } ;
    $g->{remake} = [] ;
    sayif 0,  "Registered '$dir', \$count=$count\n" ;

    $g->{nmoves} += $count ;
    # sayif 1,  "This move costs $count elementary moves\n" ;
    ## sayif 1,  "Length of unmake : ",0+@{$g->{unmake}},"\n";
    sayif 0,  "registered on $g\n";
    push @{$g->{unmake}->[0]->{dir}}, $dir ;

}				# End register()

=head2 C<unmake($g,$piece,$dir), remake($g,$piece,$dir)>

Undo last move and redo last undone move. Do not return anything, but
modify C<$g>.

=cut

sub unmake
{
    my ($g) = @_ ;
    my $tmp = 0 ;
    if ( @{$g->{unmake}} > 1 ){
      unshift  @{$g->{remake}},  $tmp = shift @{$g->{unmake}} ;
      if (exists $g->{unmake}->[0]->{s}) {
	  $g->{s} = $g->{unmake}->[0]->{s} ; 
	} else {
	  my %gg;
          @gg{"s","w"} = @$g{"s","w"};
          my @ok = move (\%gg, , );
	  $g->{s} = $gg{s};
	}
	$g->{nmoves} = $g->{unmake}->[0]->{nmoves};
	sayif 0,  "New pos is \n------\n$g->{s}\n------\n" ;
    } else {
				# HERE FIXME : Should go back to end
	sayif 0,  "Can't unmake further\n";
    }
    $tmp;
}

sub remake
{
    my ($g) = @_ ;
    my $tmp = 0 ;
    if ( @{$g->{remake}} ){
      if (exists $g->{remake}->[0]->{s}) {
	$g->{s} = $g->{remake}->[0]->{s};
      } else {
	print "Redo without 's' : Not yet implemetned\n";
      }
      sayif 0,  "New pos is \n------\n$g->{s}\n------\n";
      unshift  @{$g->{unmake}}, $tmp = shift @{$g->{remake}} ;
    } else {
				# HERE FIXME : Should go back to start
	sayif 0,  "Can't remake further\n";
    }
    $g->{nmoves} = $g->{unmake}->[0]->{nmoves} ;
    $tmp ;
}

=head2 C<$m = piece_mask($g,$piece)>

Returns a hash C<$m> containing some information about $piece. For
example, if $piece == 'X', and has a 'L' shape, one gets :

	          $m = { size => [$width,$height],
    X	  -->            mask => "++ +",
    XX                   iden => "$width,$height,$mask",
	                 pos  => [$posx,$posy],
                       }

To produce the mask, the smallest rectangle containing the piece is
scanned from top-left, top to bottom and left to right ("matlab"-like
ordering). The mask string characters are either "+" (occupied) or " "
(un-occupied). $pos(x|y) start from 0.

=cut

sub piece_mask
{
    my ($g,$p,$str) = @_ ;
    $str = $g->{s} unless defined $str ;

    my $sss = $str;
    $sss =~ s/\n/|/g;

    unless ($str =~ /$p/){
	warn "piece_mask : Can't find piece '$p'\n" ;
	return {"size"=>[undef,undef],"mask"=>undef,"iden"=>undef};
    }
    my @xy = ();
    my ($x,$y,$xx,$yy,$XX,$YY) ;	# tmp, min and MAX

    my $tmp = -1 ;		# If only @a = index ... would work ...
    while ( ($tmp = index($str,$p,$tmp+1)) > $[ - 1 ) {

				# Position of cell belonging to piece
	($x,$y) = ($tmp % ($g->{w}+1),int( $tmp / ($g->{w}+1)) ) ;
	## sayif 1,  "full at $tmp // $g->{w} -> ($x,$y)\n";
	push @xy, [$x,$y] ;
	$xx = $x if !defined($xx) || $xx > $x ;	# smallest col
	$XX = $x if !defined($XX) || $XX < $x ;	# greatest col
	$yy = $y if !defined($yy) || $yy > $y ;	# smallest row
	$YY = $y if !defined($YY) || $YY < $y ;	# greatest row
    }
    ## sayif 1,  "Bounds are $xx-$XX and $yy-$YY\n";
				# Width
    ($XX,$YY) = (1+$XX-$xx,1+$YY-$yy) ;

    my $mask = " " x ($XX*$YY) ;
    foreach (@xy){
	substr $mask,($_->[0]-$xx)*$YY + $_->[1]-$yy,1,"+" ;
    }
    ## print ">$p< >$sss< >$XX,$YY,$mask,$xx,$yy<\n";
    return {"size"=>[$XX,$YY],
	    "mask"=>$mask,
	    "iden"=>"$XX,$YY,$mask" ,
	    "pos" =>[$xx,$yy],
	} ;
}

sub sayif
{ return unless @_ && shift ;
  print @_ ;
}


1;
