/*
 * Resource families
 *
 * Copyright (C) 2002, Olaf Kirch <okir@lst.de>
 */

#include <errno.h>
#include "resmgrd.h"

static res_family_t *	families[] = {
	&res_family_file,
	&res_family_scsi,
	&res_family_usb,
	&res_family_paride,
	NULL
};

/*
 * Find resource family given its name
 */
res_family_t *
res_family_get(const char *name)
{
	res_family_t	*f;
	unsigned int	n;

	for (n = 0; (f = families[n]) != NULL; n++) {
		if (!strcasecmp(f->name, name))
			return f;
	}
	return NULL;
}

/*
 * Handle resource names
 */
res_name_t *
res_name_parse(const char *name)
{
	char		temp_name[128], *colon;
	res_family_t	*f;

	if (name[0] == '/') {
		f = &res_family_file;
	} else {
		if (strlen(name) >= sizeof(temp_name))
			return NULL;
		strcpy(temp_name, name);
		if (!(colon = strchr(temp_name, ':')))
			return NULL;
		*colon++ = '\0';
		if (!(f = res_family_get(temp_name)))
			return NULL;
		name = colon;
	}

	if (name[0] == '/' && !res_device_sane(name))
		return NULL;

	return f->parse_name(name);
}

const char *
res_name_print(res_name_t *np)
{
	if (np && np->ops && np->ops->print_name)
		return np->ops->print_name(np);
	return "<unspec>";
}

void
res_name_free(res_name_t *np)
{
	if (np && np->ops && np->ops->free_name)
		np->ops->free_name(np);
}

int
res_name_match(res_name_t *np, res_device_t *dev)
{
	if (!np || !np->ops || !np->ops->match)
		return 0;

	/* Check access flags. This means that you cannot
	 * access e.g. scsi:/dev/cdrom unless /dev/cdrom
	 * has the DEV_FLAGS_SCSI flag set */
	if (np->ops->access_flags & ~dev->flags)
		return 0;

	return np->ops->match(np, dev);
}

int
res_name_open(res_name_t *np, int flags)
{
	if (!np || !np->ops || !np->ops->open) {
		errno = EACCES;
		return -1;
	}

	return np->ops->open(np, flags);
}
