/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      GTK 1.2                                                        *
 * Modul:     callback.c                                                     *
 *            Callback functions for gtkpaul.c                               *
 * Author:    Andreas Tille                                                  *
 * Date:      14.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; Gnu Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <gdk/gdkkeysyms.h>
#include "paul.h"
#include "callback.h"

/*****************************************************************************************
 * General callbacks                                                                     *
 *****************************************************************************************/

void DoPaulExit(GtkWidget *w, PAUL *p)
/* wrapper to PaulExit
 */
{
  PaulExit(p);
}

void PaulExit(PAUL *p)
/* clean all neccessary things
 */
{
  if ( !IS_PAUL(p) ) {
    gtk_exit(-1);
    return;
  }
  if ( p->opt && SaveBild(p->opt->f) ) SavePictures(p->piclist, p->opt->f);
  FreePaul(p);
  gtk_exit(0);
}

void ConfigureCallback(GtkWidget *show, GdkEventConfigure *event, PAUL *p)
{
  static   int w, h;
  PICTURE *bild;

  g_return_if_fail ( GTK_IS_WINDOW(show) );
  g_return_if_fail ( IS_PAUL(p) );
  g_return_if_fail ( event && event->type == GDK_CONFIGURE ) ;
  g_return_if_fail ( (bild = BILD(p->activ)) );

  if ( UPDATE_IMAGE || event->width != bild->W || event->height != bild->H ) {
    w = event->width;
    h = event->height;
    gdk_imlib_apply_image(bild->im, p->show->window);
    UPDATE_IMAGE = 0;
  }
}

void PictureKeyPressed(GtkWidget *show, GdkEvent *event, PAUL *p)
{
  GList     *pl = NULL;
  int        select = 0;
  

  g_return_if_fail ( GTK_IS_WINDOW(show) );
  g_return_if_fail ( IS_PAUL(p) );
  g_return_if_fail ( event && event->type == GDK_KEY_PRESS ) ;

  if ( !(pl = p->activ) ) pl = p->piclist;
  if ( !CheckPicList(pl) ) {
     if ( show ) gtk_widget_destroy(show);
     return;
  }

  /*   mod = event->key.state;   */
  switch ( event->key.keyval ) {
    case GDK_Page_Up:
      if ( !pl->prev ) {
        gdk_beep();
        return;
      }
      select = 1;
      p->activ = pl->prev;
      break;

    case GDK_Page_Down: 
      if ( !pl->next ) {
	gdk_beep();
        return;
      }
      select = 1;
      p->activ = pl->next;
      break;

    case GDK_KP_Add:
    case GDK_plus:      
      GrowPic(show->window, BILD(pl));
      return;

    case GDK_KP_Subtract:
    case GDK_minus:     
      ShrinkPic(show->window, BILD(pl));
      return;

    case GDK_KP_Left:
    case GDK_Left:
      break;

    case GDK_KP_Right:
    case GDK_Right:
      break;

    case GDK_KP_Up:
    case GDK_Up:
      break;

    case GDK_KP_Down:
    case GDK_Down:
      break;

    case GDK_0:
      break;

    case GDK_space:
      break;

    case GDK_s :        /* exit and save all images */
    case GDK_S :        
      SetPNG(p->opt->f);

    case GDK_q :
    case GDK_Q :        /* exit and save, if program was started with -o option */
    case GDK_Escape:
      PaulExit(p);
      return;

    case GDK_x :        /* quit without save */
    case GDK_X :        
      DontSaveBild(p->opt->f);
      PaulExit(p);
      return;

    case GDK_v :        /* Mirror vertical */
      MirrorCallback(p, THIS | MIRROR_V);
      return;

    case GDK_w :        /* Mirror horizontal */
      MirrorCallback(p, THIS | MIRROR_H);
      return;

    case GDK_Shift_L:
    case GDK_Shift_R:
    case GDK_Control_L:
    case GDK_Control_R:
    case GDK_Caps_Lock: 
    case GDK_Alt_L:
    case GDK_Alt_R:
      return;
    case GDK_KP_2:      /* If <Shift> is pressed the arrow keys return these values. */
    case GDK_KP_4:      
    case GDK_KP_6:      
    case GDK_KP_8:      
      return;
    default:
      gdk_beep();
      return;
  }
  if ( select == -1 ) gdk_beep();
  else if ( select == 1 ) 
    SelectAndShowActiv(p);
  else
    ApplyPicture(show->window, BILD(p->activ));

  return;
}

/*****************************************************************************************
 * Callback to change contrast or brightness of images                                   *
 * related to: this/mark/all images                                                      *
 *****************************************************************************************/

void ScaleCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)SkaliereBilder, PAUL_FUNC_TYPE, which, 0);
}

/*****************************************************************************************
 * Callback to analyse brightness of images                                              *
 * related to: this/mark/all images                                                      *
 *****************************************************************************************/

void BrightnessCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)AnalyseBrightness, PAUL_FUNC_TYPE, which, 0);
}

/*****************************************************************************************
 * Callback to cut image                                                                 *
 * related to: this/mark/all images                                                      *
 *****************************************************************************************/

void CutCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)CutPaul, PAUL_FUNC_TYPE, which, 0);
}

/*****************************************************************************************
 * Callbacks to cut white/black borders of images                                        *
 * related to: this/mark/all images                                                      *
 *****************************************************************************************/

void DelBorderCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)SelectPictureWithoutBorder, PAUL_FUNC_TYPE, which, 0);
}

void DelScanBorderCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)DeleteScannerBorder, PAUL_FUNC_TYPE, which, 0);
}


/*****************************************************************************************
 * Callback to filter images                                                             *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void FilterCallback(PAUL *p, int which)
{
  p->opt->filter = 'h';
  ApplyPaulFunc(p, (void *)FilterBilder, PAUL_FUNC_TYPE, which, 1);
}

/*****************************************************************************************
 * Callback for displaying histogram of an image                                         *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void HistogramCallback(PAUL *p, int which)
/* display histograms of images
 */
{
  ApplyPaulFunc(p, (void *)ShowHistograms, PICLIST_FUNC_TYPE, which, 0);
}
   
/*****************************************************************************************
 * Callbacks to move images                                                              *
 * related to: all images                                                                *
 *****************************************************************************************/

void MoveImageCallback(PAUL *p, int which)
/* Call gtkimmov to get matching images
 * --- Parameter: ---
 * PAUL      *p     : image structure
 */
{
  ApplyPaulFunc(p, (void *)MakeMove, PAUL_FUNC_TYPE, which, 0);
}

/*****************************************************************************************
 * Make difference of images                                                             *
 * related to: all images for queue and to this/mark/all for normal difference           *
 *****************************************************************************************/

void DifferenceCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)MakeDifference, PAUL_FUNC_TYPE, which, 0);
}

void QueueDifferenceCallback(PAUL *p)
/* build differences from images following each other
 * (number of images will be reduced by one)
 */
{
  g_return_if_fail ( IS_PAUL(p) );

  if ( NBILDER(p->piclist) < 2 ) 
    g_warning(_("At least two images are neccessary to build differences"));

  if ( DifferenceQueue(p) == RET_OK ) {
    p->activ = p->piclist;
    MakeNewFileList(p);
  }
}


/*****************************************************************************************
 * Make FFT of images                                                                    *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void FFTCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)FastFourierTransform, PAUL_FUNC_TYPE, which, 1);
}

/*****************************************************************************************
 * Make monochrome image                                                                 *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void MonochromeCallback(PAUL *p, int which)
{
  long    f;
   
  g_return_if_fail ( IS_PAUL(p) );

  f          = p->opt->f;
  p->opt->f |= MAKEGRAY;
  if ( p->opt->greps == 3 ) p->opt->greps = 255;
  ApplyPaulFunc(p, (void *)MakeGray, PAUL_FUNC_TYPE, which, 0);
  p->opt->f  = f;
}

/*****************************************************************************************
 * Make negativ of image                                                                 *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void NegativCallback(PAUL *p, int which)
{
  ApplyPaulFunc(p, (void *)MakeNegative, PICLIST_FUNC_TYPE, which, 0);
}


/*****************************************************************************************
 * Mirror image                                                                          *
 * related to: this/all images                                                           *
 *****************************************************************************************/

void MirrorCallback(PAUL *p, int which)
/* mirror activ image horizontally (at a vertical symmetry axis)
 */
{
  long    f;
  int     what;
   
  g_return_if_fail ( IS_PAUL(p) );

  f          = p->opt->f;
  what       = which & (THIS | MARK | ALL);
  p->opt->f |= which & (~what);
  g_return_if_fail ( p->opt->f & (MIRROR_H | MIRROR_V) );
  ApplyPaulFunc(p, (void *)MakeMirror, PAUL_FUNC_TYPE, what, 0);
  p->opt->f  = f;
}

/*****************************************************************************************
 * Rotate image                                                                          *
 * related to: this/all images                                                           *
 *****************************************************************************************/

void RotateCallback(PAUL *p, int which)
{
  long    f;
  int     what;
   
  g_return_if_fail ( IS_PAUL(p) );

  f          = p->opt->f;
  what       = which & (THIS | MARK | ALL);
  p->opt->f |= which & (~what);
  g_return_if_fail ( p->opt->f & ROT_270 );
  ApplyPaulFunc(p, (void *)MakeRotate, PAUL_FUNC_TYPE, what, 0);
  p->opt->f  = f;
}

void RotateVariableCallback(PAUL *p, int which)
{
  long    f;
  int     what;
   
  g_return_if_fail ( IS_PAUL(p) );

  f          = p->opt->f;
  what       = which & (THIS | MARK | ALL);
  p->opt->f |= which & (~what);
  ApplyPaulFunc(p, (void *)MakeRotate, PAUL_FUNC_TYPE, what, 0);
  p->opt->f  = f;
}

/*****************************************************************************************
 * Shrink image by width/2 and height/2                                                  *
 * related to: this/marked/all images                                                    *
 *****************************************************************************************/

void ShrinkCallback(PAUL *p, int which)
{
  g_return_if_fail ( IS_PAUL(p) );

  p->opt->f  |= SHRINK;
  if ( !p->opt->shr ) p->opt->shr = 1;
  ApplyPaulFunc(p, (void *)ShrinkPictures, PAUL_FUNC_TYPE, which, 0);
}

