/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: IPv4Dialog.cpp,v 1.23 2004/09/12 23:55:16 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include "IPv4Dialog.h"
#include "ObjectManipulator.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/FWException.h"

#include <qlineedit.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qdns.h>
#include <qapplication.h>
#include <qcursor.h>

#include <iostream>

using namespace std;
using namespace libfwbuilder;

void IPv4Dialog::loadFWObject(FWObject *o)
{
    obj=o;
    IPv4 *s = dynamic_cast<IPv4*>(obj);
    assert(s!=NULL);
    
    dnsBusy=false;
    init=true;

    fillLibraries(libs,obj);

    obj_name->setText( QString::fromUtf8(s->getName().c_str()) );
    comment->setText( QString::fromUtf8(s->getComment().c_str()) );

/*
 * if this is an address that belongs to an interface, we can't move
 * it from library to library just like that. Only IPv4 objects that
 * belong to the standard group "Addresses" can be moved.
 */
    if ( Interface::isA( obj->getParent() ) )
    {
        showNetmask=true;
        libs->setEnabled( false );
        netmaskLabel->show();
        netmask->show();
    } else
    {
        showNetmask=false;
        libs->setEnabled( true );
        netmaskLabel->hide();
        netmask->hide();
    }

/* catch exceptions separately so even if we have a bad address, we
 * still can show netmask */
    try
    {
        address->setText( s->getAddress().toString().c_str() );
    } catch (FWException &ex) {}

    try
    {
        if ( Interface::isA( obj->getParent() ) )
            netmask->setText( s->getNetmask().toString().c_str() );
    } catch (FWException &ex) {}

    apply->setEnabled( false );

    obj_name->setEnabled(!o->isReadOnly());
    setDisabledPalette(obj_name);

    libs->setEnabled(!o->isReadOnly());
    setDisabledPalette(libs);

    address->setEnabled(!o->isReadOnly());
    setDisabledPalette(address);

    netmask->setEnabled(!o->isReadOnly());
    setDisabledPalette(netmask);

    comment->setReadOnly(o->isReadOnly());
    setDisabledPalette(comment);



    init=false;
}
    
void IPv4Dialog::changed()
{
    apply->setEnabled( true );
}

void IPv4Dialog::validate(bool *res)
{
    *res=true;
    IPv4 *s = dynamic_cast<IPv4*>(obj);
    assert(s!=NULL);
    try
    {
        IPAddress( address->text().latin1() );
    } catch (FWException &ex)
    {
        *res=false;
        QMessageBox::critical(this, "Firewall Builder",
                              tr("Illegal IP address '%1'").arg(address->text()),
                              tr("&Continue"), 0, 0,
                              0 );
    }

    if ( showNetmask )
    {
        try
        {
            Netmask( netmask->text().latin1() );
        } catch (FWException &ex)
        {
            *res=false;
            QMessageBox::critical(this, "Firewall Builder",
                                  tr("Illegal netmask '%1'").arg(netmask->text()),
                                  tr("&Continue"), 0, 0,
                                  0 );
        }
    }
}

void IPv4Dialog::isChanged(bool *res)
{
    *res=(!init && apply->isEnabled());
}

void IPv4Dialog::libChanged()
{
    changed();
}

void IPv4Dialog::applyChanges()
{
    if (!isTreeReadWrite(this,obj)) return;
    if (!validateName(this,obj,obj_name->text())) return;

    bool res=true;
    validate( &res );
    if (!res) return;

    IPv4 *s = dynamic_cast<IPv4*>(obj);
    assert(s!=NULL);

    string oldname=obj->getName();
    obj->setName( string(obj_name->text().utf8()) );
    obj->setComment( string(comment->text().utf8()) );

    try
    {
        s->setAddress( address->text().latin1() );
    } catch (FWException &ex) { }

    if ( showNetmask )
    {
        try
        {
            s->setNetmask( netmask->text().latin1() );
        } catch (FWException &ex) { }
    } else
        s->setNetmask( "255.255.255.255" );

    om->updateObjName(obj,QString::fromUtf8(oldname.c_str()));

    init=true;

/* move to another lib if we have to */
    if ( ! Interface::isA( obj->getParent() ) &&
         libs->currentText() != QString(obj->getLibrary()->getName().c_str()))
        om->moveObject(libs->currentText(), obj);

    init=false;

    apply->setEnabled( false );
}

void IPv4Dialog::discardChanges()
{
    loadFWObject(obj);
}

void IPv4Dialog::DNSlookup()
{
    if (fwbdebug)
        qDebug("IPv4Dialog::DNSlookup()  dnsBusy=%d", dnsBusy);

    if (!dnsBusy)
    {
        QString name = obj_name->text();
        if (fwbdebug) qDebug("IPv4Dialog::DNSlookup()  name=%s", name.ascii());
        dnsBusy=true;
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );
        QString addr = getAddrByName(name);
        QApplication::restoreOverrideCursor();
        dnsBusy=false;
        if (fwbdebug) qDebug("IPv4Dialog::DNSlookup()  done");

        if (! addr.isEmpty())
        {
            address->setText( addr );
            return;
        }

        if ( Interface::isA(obj->getParent()) )
        {
            FWObject *host = obj->getParent()->getParent();
            assert(host!=NULL);
            name = host->getName().c_str();

            if (fwbdebug) qDebug("IPv4Dialog::DNSlookup()  name=%s",
                                 name.ascii());
            dnsBusy=true;
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );
            QString addr = getAddrByName(name);
            QApplication::restoreOverrideCursor();
            dnsBusy=false;
            if (fwbdebug) qDebug("IPv4Dialog::DNSlookup()  done");
            if ( ! addr.isEmpty())
            {
                address->setText( addr );
                return;
            }
            QMessageBox::warning(
                this,"Firewall Builder", 
                tr("DNS lookup failed for both names of the address object '%1' and the name of the host '%2'.")
                .arg(obj_name->text()).arg(name),
                "&Continue", QString::null,QString::null, 0, 1 );
            return;
        } 
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("DNS lookup failed for name of the address object '%1'.")
            .arg(name),
            "&Continue", QString::null,QString::null, 0, 1 );
        return;
    }
}

