/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: fwbedit.cpp,v 1.11 2004/04/11 07:18:18 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "fwbuilder/libfwbuilder-config.h"

#include <qsettings.h>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <fstream>
#include <iostream>
#include <algorithm>
#include <functional>

#ifndef _WIN32
#  include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <getopt.h>

#include "fwbuilder/Resources.h"

#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Group.h"

#include "../common/init.cpp"


typedef enum { NONE, ADDGRP, REMGRP, DELETE } command;

static char      *filename        = NULL;
static char      *object          = NULL;
static char      *group           = NULL;
static command    cmd             = NONE;

using namespace libfwbuilder;
using namespace std;


class UpgradePredicate: public XMLTools::UpgradePredicate
{
    public:
    virtual bool operator()(const string &msg) const 
    { 
	cout << _("Data file has been created in the old version of Firewall Builder. Use fwbuilder GUI to convert it.") << endl;
	return false;
    }
};
    
void usage(const char *name)
{
    cout << _("Firewall Builder:  general purpose object tree editing tool") << endl;
    cout << _("Version ") << VERSION << "-" << RELEASE_NUM << endl;
    cout << _("Usage: ") << name
         << _(" [-V] -f filename.xml [-a obj,grp] [-r obj,grp] [-d obj]") << endl;
    cout << endl;
    cout << "      " << _("-a obj,grp :  create reference to object 'obj' in the group 'grp'") << endl;
    cout << "      " << _("-r obj,grp :  remove reference to object 'obj' from the group 'grp'") << endl;
    cout << "      " << _("-d obj     :  delete object 'obj' and remove references to it from") << endl;
    cout << "                    " << _("all groups and rules") << endl;
    cout << endl;
    cout << _("An object and a group can be defined by their ID or ") <<  endl;
    cout << _("by the full path and name in the XML tree") << endl;
}

FWObject *find_object(const string &obj_path,
                      FWObject *root=FWObjectDatabase::db)
{
    string path=obj_path;
    string::size_type n=path.find("/",0);
    string tree_node=path.substr(0,n);

    FWObject::iterator j=std::find_if(root->begin(), root->end(), 
                                      FWObjectNameEQPredicate(tree_node));
    if (j==root->end()) return NULL;
    if ((*j)->getName()==obj_path) return (*j);
    else    return find_object( path.substr(n+1) , (*j) );
}


FWObject *getObject(const char *objstr) throw (FWException)
{
    char errstr[128];
    FWObject *obj=NULL;
    if (strchr(objstr,'/')!=NULL)
    {
/* got full path to the object */
        string path=objstr;
        string::size_type n=path.find("/",0);
        if (n==0 || 
            path.substr(0,n)=="FWObjectDatabase" || 
            path.substr(0,n)=="User")
            obj= find_object( path.substr(n+1) );
        else
            obj= find_object( path );
        if (obj==NULL)
        {
            SNPRINTF(errstr,sizeof(errstr),_("Object %s not found"),objstr );
            throw FWException(errstr);
        }
    } else {
/* got object ID */
        obj=FWObjectDatabase::db->getById(objstr,true);
        if (obj==NULL) 
        {
            SNPRINTF(errstr,sizeof(errstr),_("Object with ID='%s' not found"),objstr );
            throw FWException(errstr);
        }
    }
    return obj;
}

int main(int argc, char * const *argv)
{   

#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");

    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#else
#  ifdef HAVE_SETLOCALE
    setlocale (LC_ALL, "");
#  endif
#endif
    
    if (argc<=1)
    {
        usage(argv[0]);
        exit(1);
    }

    int   opt;

    while( (opt=getopt(argc,argv,"f:a:r:d:")) != EOF )
    {
        switch(opt)
        {
        case 'f':     filename=strdup(optarg); break;
        case 'a':     cmd=ADDGRP;      break;
        case 'r':     cmd=REMGRP;      break;
        case 'd':     cmd=DELETE;      break;
        case 'V':     usage(argv[0]);  exit(1);
        }
        if (cmd==ADDGRP || cmd==REMGRP)
        {
            char *cptr=strchr(optarg,',');
            if (cptr==NULL) 
            {
                usage(argv[0]);
                exit(1);
            }
            object=(char*)malloc(cptr-optarg+1);
            strncpy(object,optarg,cptr-optarg);
            cptr++;
            group=strdup(cptr);
            break;
        }
        if (cmd==DELETE)
        {
            object=strdup(optarg);
            break;
        }
    }


    if(cmd==NONE || filename==NULL)
    {
        usage(argv[0]);
        exit(1);
    }

    init(argv);

    try 
    {
        new Resources(respath+FS_SEPARATOR+"resources.xml");

        new Resources(respath);   // creates singleton Resources::global_res

        /* create database */
        new FWObjectDatabase();

        /* load the data file */
        UpgradePredicate upgrade_predicate; 

        FWObjectDatabase::db->load(filename,  &upgrade_predicate, librespath);
        FWObject *obj=getObject(object);
        if (cmd==DELETE)
        {
            cout << _("Removing object '") << obj->getName()
                 << _("' from the tree.") << endl;
            FWObjectDatabase::db->removeAllInstances(obj);
        }
        if (cmd==ADDGRP)
        {
            Group *grp=Group::cast(getObject(group));
            cout << _("Adding object '") << obj->getName()
                 << _("' to the group '") << grp->getName()
                 << "'" << endl;
            grp->addRef(obj);
        }
        if (cmd==REMGRP)
        {
            Group *grp=Group::cast(getObject(group));
            cout << _("Removing object '") << obj->getName()
                 << _("' from the group '") << grp->getName()
                 << "'" << endl;
            grp->removeRef(obj);
        }

        char* bakfile=(char*)malloc(strlen(filename)+5);
        strcpy(bakfile,filename);
        strcat(bakfile,".bak");

        rename(filename,bakfile);
	FWObjectDatabase::db->saveFile(filename);

    } catch(FWException &ex)  {
	cerr << ex.toString() << endl;
        exit(1);
    } catch (std::string s) {
	cerr << s;
        exit(1);
    } catch (std::exception ex) {
	cerr << ex.what();
        exit(1);
    } catch (...) {
	cerr << _("Unsupported exception");
        exit(1);
    }

    return(0);
}

