# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::self::doc::commands;
use base 'AXP::Command::self::doc';

use Arch::Util qw(save_file);

my $name_tag    = "<{NAME}>";
my $prefix_tag  = "<{PREFIX}>";
my $rootdir_tag = "<{ROOTDIR}>";
my $content_tag = "<{CONTENT}>";

sub infoline {
	"create html doc pages for all axp subcommands"
}

sub helptext {
	qq{
		For each axp command creates a sub-directory in docdir.
		The actual documentation is placed in index.html files.

		If the template option is given, the $content_tag in the
		template is replaced with the actual html content. Other
		tags are $name_tag, $prefix_tag and $rootdir_tag.
	}
}

sub create_doc ($$) {
	my $self = shift;
	my $command = shift;
	my $prefix = $command->{prefix};
	my $name = $command->{name};

	my $infoline = $self->escape_html($command->infoline);
	my $optusage = $self->escape_html($command->optusage);
	my $helptext = $self->escape_html($command->helptext);
	$helptext =~ s/^\n//;
	$helptext =~ s/\s*$//;
	$helptext =~ s/^\t\t//mg;

	my $opttable = "";
	my @option_entries = $command->all_options;
	while (@option_entries) {
		my ($option, $entry) = splice(@option_entries, 0, 2);
		my $short_name = $entry->{sh}? "-$entry->{sh}, ": "";
		my $long_name = $command->long_option($option);
		my $arg = $entry->{arg}? " $entry->{arg}": $entry->{type}? " ARG": "";
		my $desc = $self->escape_html($entry->{desc});
		$opttable .= qq(\n<tr><td>$short_name$long_name$arg</td><td>$desc</td></tr>);
	}

	my $subcmds = "";
	if ($command->has_subcommands) {
		$subcmds .= qq(<h3 class="cmd_subcmds">Subcommands:</h3>);
		$subcmds .= qq(<table class="cmd_subcmds">);
		foreach my $subcommand ($command->all_subcommands) {
			my $name = $subcommand->{name};
			my $infoline = $self->escape_html($subcommand->infoline);
			my $subprefix = "$prefix $name";
			$subcmds .= qq(\n<tr><td><a href="$name/">$subprefix</a></td><td>$infoline</td></tr>);
		}
		$subcmds .= qq(</table>\n);
	}

	my $links = "";
	my @names = split(/ /, $prefix);
	pop @names;
	my $updir = "";
	foreach my $name ((reverse @names), "[home]") {
		$updir .= "../";
		$links = qq(<a href="$updir">$name</a> $links);
	}
	$links = qq(<div class="cmd_links">Return to: $links</div>) if $links;

	my $content = qq(
		<div class="main1">
		<h3 class="cmd_usage">Usage:</h3>
		<div class="cmd_usage">
			<span class="cmd_prefix">$prefix</span>
			<span class="cmd_opts">$optusage</span>
		</div>
		<h3 class="cmd_name">Infoline:</h3>
		<div class="cmd_infoline">$infoline</div>
		<h3 class="cmd_options">Options:</h3>
		<table class="cmd_options">$opttable
		</table>
		$subcmds
		</div>
		<div class="main2">
		<pre class="cmd_helptext">$helptext</pre>
		</div>
		$links
	);

	$content =~ s/\t$//;
	$content =~ s/^\t\t//mg;

	my $subdir = $prefix;
	$subdir =~ s! !/!g;
	$subdir =~ s!^[^/]+(/|$)!!;
	my $dir = $self->{docdir};
	$dir .= "/$subdir" if $subdir;
	unless (-d $dir) {
		mkdir($dir, 0777) || die "Can't create $dir: $!\n";
	}

	my $rootdir = $prefix;
	$rootdir =~ s! !/!g;
	$rootdir =~ s![^/]+!..!g;

	my $html = $self->{template};
	$html =~ s/\Q$name_tag\E/$name/g;
	$html =~ s/\Q$prefix_tag\E/$prefix/g;
	$html =~ s/\Q$rootdir_tag\E/$rootdir/g;
	$html =~ s/\Q$content_tag\E/$content/g;

	save_file("$dir/index.html", $html);
}

sub default_template {
	"<html>\n<head><title>$prefix_tag</title></head>\n"
	. "<body>\n$content_tag\n</body></html>\n"
}

sub execute {
	my $self = shift;
	$self->_execute;

	$self->{root}->foreach_subcommand(sub { $self->create_doc(@_) });
}

1;
