/*
 * LibrePCB - Professional EDA for everyone!
 * Copyright (C) 2013 LibrePCB Developers, see AUTHORS.md for contributors.
 * https://librepcb.org/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*******************************************************************************
 *  Includes
 ******************************************************************************/
#include "newprojectwizardpage_initialization.h"

#include "ui_newprojectwizardpage_initialization.h"

#include <librepcb/core/fileio/filepath.h>

#include <QtCore>
#include <QtWidgets>

/*******************************************************************************
 *  Namespace
 ******************************************************************************/
namespace librepcb {
namespace editor {

/*******************************************************************************
 *  Constructors / Destructor
 ******************************************************************************/

NewProjectWizardPage_Initialization::NewProjectWizardPage_Initialization(
    QWidget* parent) noexcept
  : QWizardPage(parent), mUi(new Ui::NewProjectWizardPage_Initialization) {
  mUi->setupUi(this);
  setPixmap(QWizard::LogoPixmap, QPixmap(":/img/actions/plus_2.png"));
  setPixmap(QWizard::WatermarkPixmap, QPixmap(":/img/wizards/watermark.jpg"));

  // signal/slot connections
  connect(mUi->cbxAddSchematic, &QGroupBox::toggled, this,
          &NewProjectWizardPage_Initialization::completeChanged);
  connect(mUi->cbxAddBoard, &QGroupBox::toggled, this,
          &NewProjectWizardPage_Initialization::completeChanged);
  connect(mUi->edtSchematicName, &QLineEdit::textChanged, this,
          &NewProjectWizardPage_Initialization::schematicNameChanged);
  connect(mUi->edtBoardName, &QLineEdit::textChanged, this,
          &NewProjectWizardPage_Initialization::boardNameChanged);

  // insert values
  mUi->edtSchematicName->setText(
      "Main");  // do not translate this into other languages!
  mUi->edtBoardName->setText(
      "default");  // do not translate this into other languages!
}

NewProjectWizardPage_Initialization::
    ~NewProjectWizardPage_Initialization() noexcept {
}

/*******************************************************************************
 *  Getters
 ******************************************************************************/

bool NewProjectWizardPage_Initialization::getCreateSchematic() const noexcept {
  return mUi->cbxAddSchematic->isChecked();
}

QString NewProjectWizardPage_Initialization::getSchematicName() const noexcept {
  return mUi->edtSchematicName->text();
}

bool NewProjectWizardPage_Initialization::getCreateBoard() const noexcept {
  return mUi->cbxAddBoard->isChecked();
}

QString NewProjectWizardPage_Initialization::getBoardName() const noexcept {
  return mUi->edtBoardName->text();
}

/*******************************************************************************
 *  GUI Action Handlers
 ******************************************************************************/

void NewProjectWizardPage_Initialization::schematicNameChanged(
    const QString& name) noexcept {
  mSchematicDirName = FilePath::cleanFileName(
      name, FilePath::ReplaceSpaces | FilePath::ToLowerCase);
  if (mSchematicDirName.isEmpty()) {
    mUi->lblSchematicDirectory->setText(tr("Invalid name!"));
  } else {
    mUi->lblSchematicDirectory->setText("schematics/" % mSchematicDirName %
                                        "/");
  }
  emit completeChanged();
}

void NewProjectWizardPage_Initialization::boardNameChanged(
    const QString& name) noexcept {
  mBoardDirName = FilePath::cleanFileName(
      name, FilePath::ReplaceSpaces | FilePath::ToLowerCase);
  if (mBoardDirName.isEmpty()) {
    mUi->lblBoardDirectory->setText(tr("Invalid name!"));
  } else {
    mUi->lblBoardDirectory->setText("boards/" % mBoardDirName % "/");
  }
  emit completeChanged();
}

/*******************************************************************************
 *  Private Methods
 ******************************************************************************/

bool NewProjectWizardPage_Initialization::isComplete() const noexcept {
  // check base class
  if (!QWizardPage::isComplete()) return false;

  // check schematic filename
  if (mUi->cbxAddSchematic->isChecked() && mSchematicDirName.isEmpty()) {
    return false;
  }

  // check board filename
  if (mUi->cbxAddBoard->isChecked() && mBoardDirName.isEmpty()) {
    return false;
  }

  return true;
}

/*******************************************************************************
 *  End of File
 ******************************************************************************/

}  // namespace editor
}  // namespace librepcb
