//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Descriptor/AxisProperty.h
//! @brief     Defines class AxisProperty
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2022
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_DESCRIPTOR_AXISPROPERTY_H
#define BORNAGAIN_GUI_MODEL_DESCRIPTOR_AXISPROPERTY_H

#include "Base/Axis/MakeScale.h"
#include "GUI/Model/Descriptor/DoubleProperty.h"
#include <memory>

//! Holds values which can be used to describe a EquiDivision.
//!
//! Use this as a member in your class to
//! * reduce dependency to axis items and axis classes
//! * simplify serialization
//! * easily have a UI supporting solution.
//!
//! This class is designed to work seamlessly with the serialization methods; just call it's
//! serialize for both reading and writing.
//! For UI integration just use AxisPropertyForm.
//!
//! Do not forget to call all two init functions from within the containing class's constructor!

class AxisProperty {
public:
    explicit AxisProperty();

    DoubleProperty& min() { return m_min; }
    const DoubleProperty& min() const { return m_min; }
    void setMin(double v) { m_min.setValue(v); }

    DoubleProperty& max() { return m_max; }
    const DoubleProperty& max() const { return m_max; }
    void setMax(double v) { m_max.setValue(v); }

    void initMin(const QString& label, const QString& tooltip, double value,
                 const std::variant<QString, Unit>& unit,
                 const RealLimits& limits = RealLimits::nonnegative(), uint decimals = 3);
    void initMax(const QString& label, const QString& tooltip, double value,
                 const std::variant<QString, Unit>& unit,
                 const RealLimits& limits = RealLimits::nonnegative(), uint decimals = 3);

    //! Creates an axis with empty title
    std::unique_ptr<Scale> createAxis(double scaleFactor) const;

    uint nbins() const { return m_nbins; }
    void setNbins(uint v) { m_nbins = v; }

    bool isExpandGroupBox() const { return m_expandGroupBox; }
    void setExpandGroupBox(bool b) { m_expandGroupBox = b; }

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

private:
    uint m_nbins = 100;
    DoubleProperty m_min;
    DoubleProperty m_max;
    bool m_expandGroupBox = true;
};

#endif // BORNAGAIN_GUI_MODEL_DESCRIPTOR_AXISPROPERTY_H
