﻿' Licensed to the .NET Foundation under one or more agreements.
' The .NET Foundation licenses this file to you under the MIT license.
' See the LICENSE file in the project root for more information.

Imports Microsoft.CodeAnalysis
Imports Microsoft.CodeAnalysis.Test.Utilities
Imports Microsoft.CodeAnalysis.VisualBasic
Imports Microsoft.CodeAnalysis.VisualBasic.Syntax
Imports Roslyn.Test.Utilities
Imports Roslyn.Test.Utilities.TestMetadata

Namespace Microsoft.CodeAnalysis.VisualBasic.UnitTests

    ' this place is dedicated to binding related error tests
    Public Class BindingErrorTests
        Inherits BasicTestBase

#Region "Targeted Error Tests - please arrange tests in the order of error code"

        <Fact()>
        Public Sub BC0ERR_None_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Imports System

        Class TimerState
            Public Delegate Sub MyEventHandler(ByVal sender As Object, ByVal e As System.EventArgs)
            Private m_MyEvent As MyEventHandler
            Public Custom Event MyEvent As MyEventHandler
                RaiseEvent(ByVal sender As Object, ByVal e As System.EventArgs)
                    m_MyEvent.Invoke(sender, e)
                End RaiseEvent
                AddHandler(ByVal value As MyEventHandler)
                    m_MyEvent = DirectCast ( _
                    [Delegate].Combine(m_MyEvent, value), _
                    MyEventHandler) : End addHandler
                RemoveHandler(ByVal value As MyEventHandler)
                    m_MyEvent = DirectCast ( _
                    [Delegate].RemoveAll(m_MyEvent, value), _
                    MyEventHandler)
                End RemoveHandler
            End Event
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Class AAA
            Private _name As String
            Public ReadOnly Property Name() As String
                Get
                    Return _name : End Get
            End Property
            Private _age As String
            Public ReadOnly Property Age() As String
                Get
                    Return _age
        : End Get
            End Property
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="None">
    <file name="a.vb">
        Module M1
            Function B As string
                Dim x = 1: End Function
            Function C As string
                Dim x = 2
            :End Function
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
       Public Class S1
            Public Sub New()
                Dim cond = True
                GoTo l1
                If False Then
        l1:
                End If
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
       Public Structure S1
            Function FOO As String
                Return "h"
            End Function
            Sub Main()
                FOO
                FOO()
            End Sub
        End Structure
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
       Public Structure S1
            Sub Main()
                dim a?()(,) as integer
                dim b(2) as integer
                dim c as integer(,)
            End Sub
        End Structure
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class D
            Public Class Foo
                Public x As Integer
            End Class
            Public Class FooD
                Inherits Foo
                Public Sub Baz()
                End Sub
            End Class
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_8()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class D
            Public Shared Sub Main()
                Dim x As Integer = 1
                Dim b As Boolean = x = 1
                System.Console.Write(b )
                Dim l As Long = 5
                System.Console.Write(l &gt; 6 )
                Dim f As Single = 25
                System.Console.Write(f &gt;= 25 )
                Dim d As Double = 3
                System.Console.Write(d &lt;= f )
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_9()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class C
            Public Shared Sub Main()
                System.Console.Write("{0}{1}{2}{3}{4}{5}", "a", "b", "c", "d", "e", "f" )
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_10()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class D
            Public Class Moo(Of T)
                Public Sub Boo(x As T)
                    Dim local As T = x
                End Sub
            End Class
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_11()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class C
            public class Moo
                public shared S as integer
            end class
            Public Shared Sub Main()
                System.Console.Write(Moo.S )
                Moo.S =  42
                System.Console.Write(Moo.S )
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_12()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class C
            Private Shared Function M(ByVal x As Integer, ByVal y As Integer, ByVal z As Integer) As Integer
                Return y
            End Function
            Public Shared Sub Main()
                System.Console.Write(M(0, 42, 1))
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_13()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Public Class C
            Private Shared Function M() As System.AppDomain
                dim y as object = System.AppDomain.CreateDomain("qq")
                dim z as System.AppDomain = ctype(y,System.AppDomain)
                return  z
            End Function
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_14()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Imports System
            Class Program
                Public Class C
                    Public Function Foo(ByVal p1 As Short, ByVal p2 As UShort) As UInteger
                        Return CUShort (p1 + p2)
                    End Function
                    Public Function Foo(ByVal p1 As Short, ByVal p2 As String) As UInteger
                        Return CUInt (p1)
                    End Function
                    Public Function Foo(ByVal p1 As Short, ByVal ParamArray p2 As UShort()) As UInteger
                        Return CByte (p2(0) + p2(1))
                    End Function
                End Class
            End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_15()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Class C
            Private Property P As Integer
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_16()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC0ERR_None_16">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
        Me.New()
    End Sub
    Public Sub New(s As String)
        :
        Call Me.New(1)
    End Sub
    Public Sub New(s As Double)
        ' comment
        Call Me.New(1)
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_17()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC0ERR_None_17">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
#Const a = 1
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_18()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC0ERR_None_18">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
#Const a = 1
#If a = 0 Then
        Me.New()
#End If
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <Fact()>
        Public Sub BC0ERR_None_19()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC0ERR_None_19">
    <file name="a.vb">
Class b
    Public Sub New(ParamArray t() As Integer)
    End Sub
End Class

Class d
    Inherits b
    Public Sub New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertNoErrors(compilation)
        End Sub

        <WorkItem(540629, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/540629")>
        <Fact()>
        Public Sub BC30002ERR_UndefinedType1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InvalidModuleAttribute1">
        <file name="a.vb"><![CDATA[
Imports System
Class Outer
    <AttributeUsage(AttributeTargets.All)> Class MyAttribute
        Inherits Attribute

    End Class
End Class

<MyAttribute()>
Class Test
End Class
        ]]></file>
    </compilation>).
            VerifyDiagnostics(Diagnostic(ERRID.ERR_UndefinedType1, "MyAttribute").WithArguments("MyAttribute"))

        End Sub

        <Fact()>
        Public Sub BC30020ERR_IsOperatorRequiresReferenceTypes1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Public Structure zzz
    Shared Sub Main()
        Dim a As New yyy
        Dim b As New yyy
        System.Console.WriteLine(a Is b)
        b = a
        System.Console.WriteLine(a Is b)
    End Sub
End Structure
Public Structure yyy
    Public i As Integer
    Public Sub abc()
        System.Console.WriteLine(i)
    End Sub
End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30020: 'Is' operator does not accept operands of type 'yyy'. Operands must be reference or nullable types.
        System.Console.WriteLine(a Is b)
                                 ~
BC30020: 'Is' operator does not accept operands of type 'yyy'. Operands must be reference or nullable types.
        System.Console.WriteLine(a Is b)
                                      ~
BC30020: 'Is' operator does not accept operands of type 'yyy'. Operands must be reference or nullable types.
        System.Console.WriteLine(a Is b)
                                 ~
BC30020: 'Is' operator does not accept operands of type 'yyy'. Operands must be reference or nullable types.
        System.Console.WriteLine(a Is b)
                                      ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30020ERR_IsOperatorRequiresReferenceTypes1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Interface I
End Interface
Class A
End Class
Class B(Of T As Structure)
End Class
Class C
    Shared Sub M1(Of T1)(_1 As T1)
        If _1 Is Nothing Then
        End If
        If Nothing Is _1 Then
        End If
    End Sub
    Shared Sub M2(Of T2 As Class)(_2 As T2)
        If _2 Is Nothing Then
        End If
        If Nothing Is _2 Then
        End If
    End Sub
    Shared Sub M3(Of T3 As Structure)(_3 As T3)
        If _3 Is Nothing Then
        End If
        If Nothing Is _3 Then
        End If
    End Sub
    Shared Sub M4(Of T4 As New)(_4 As T4)
        If _4 Is Nothing Then
        End If
        If Nothing Is _4 Then
        End If
    End Sub
    Shared Sub M5(Of T5 As I)(_5 As T5)
        If _5 Is Nothing Then
        End If
        If Nothing Is _5 Then
        End If
    End Sub
    Shared Sub M6(Of T6 As A)(_6 As T6)
        If _6 Is Nothing Then
        End If
        If Nothing Is _6 Then
        End If
    End Sub
    Shared Sub M7(Of T7 As U, U)(_7 As T7)
        If _7 Is Nothing Then
        End If
        If Nothing Is _7 Then
        End If
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30020: 'Is' operator does not accept operands of type 'T3'. Operands must be reference or nullable types.
        If _3 Is Nothing Then
           ~~
BC30020: 'Is' operator does not accept operands of type 'T3'. Operands must be reference or nullable types.
        If Nothing Is _3 Then
                      ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30029ERR_CantRaiseBaseEvent()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="CantRaiseBaseEvent">
    <file name="a.vb">
    Option Explicit On
    Class class1
        Public Event MyEvent()
    End Class
    Class class2
        Inherits class1
        Sub RaiseEvt()
            'COMPILEERROR: BC30029,"MyEvent"
            RaiseEvent MyEvent()
        End Sub
    End Class
    </file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_CantRaiseBaseEvent, "MyEvent"))

        End Sub

        <Fact()>
        Public Sub BC30030ERR_TryWithoutCatchOrFinally()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TryWithoutCatchOrFinally">
    <file name="a.vb">
    Module M1
        Sub Scen1()
            'COMPILEERROR: BC30030, "Try"
            Try
            End Try
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30030: Try must have at least one 'Catch' or a 'Finally'.
            Try
            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30038ERR_StrictDisallowsObjectOperand1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="StrictDisallowsObjectOperand1">
    <file name="a.vb">
Imports System

    Structure myStruct1
        shared result = New Guid() And New Guid()
    End structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30452: Operator 'And' is not defined for types 'Guid' and 'Guid'.
        shared result = New Guid() And New Guid()
                        ~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30038ERR_StrictDisallowsObjectOperand1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="StrictDisallowsObjectOperand1">
    <file name="a.vb">
    Option Infer Off
    option Strict on
        Structure myStruct1
            sub foo()
                Dim x1$ = "hi"
                Dim [dim]  = "hi" &amp; "hello"
                Dim x31 As integer = x1 &amp; [dim]
            end sub
        End structure
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30209: Option Strict On requires all variable declarations to have an 'As' clause.
                Dim [dim]  = "hi" &amp; "hello"
                    ~~~~~
BC30038: Option Strict On prohibits operands of type Object for operator '&amp;'.
                Dim x31 As integer = x1 &amp; [dim]
                                          ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30038ERR_StrictDisallowsObjectOperand1_1a()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="StrictDisallowsObjectOperand1">
    <file name="a.vb">
        <![CDATA[
option Strict on
Structure myStruct1
    sub foo()
        Dim x1$ = 33 & 2.34 'No inference here
    end sub
End structure
        ]]>
    </file>
</compilation>)
            AssertTheseEmitDiagnostics(compilation,
<expected><![CDATA[
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.Conversions.ToString' is not defined.
        Dim x1$ = 33 & 2.34 'No inference here
                  ~~
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.Conversions.ToString' is not defined.
        Dim x1$ = 33 & 2.34 'No inference here
                       ~~~~
]]></expected>)
        End Sub

        <Fact()>
        Public Sub BC30039ERR_LoopControlMustNotBeProperty()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Module M
    Sub M()
        For <x/>.@a = "" To ""
        Next
    End Sub
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC30039: Loop control variable cannot be a property or a late-bound indexed array.
        For <x/>.@a = "" To ""
            ~~~~~~~
BC30337: 'For' loop control variable cannot be of type 'String' because the type does not support the required operators.
        For <x/>.@a = "" To ""
            ~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC30039ERR_LoopControlMustNotBeProperty_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System
Class C
    Property P() As Integer
    Dim F As Integer
    Sub Method1(A As Integer, ByRef B As Integer)
        ' error
        For Each P In {1}
        Next
        ' warning
        For Each F In {2}
        Next
        For Each Me.F In {3}
        Next
        For Each A In {4}
        Next
        For Each B In {5}
        Next
    End Sub
    Shared Sub Main()
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30039: Loop control variable cannot be a property or a late-bound indexed array.
        For Each P In {1}
                 ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30039ERR_LoopControlMustNotBeProperty_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict Off
Option Infer On
Public Class MyClass1
    Public Property z As Integer
    Public Shared Sub Main()
    End Sub
    Public Sub Foo()
        For z = 1 To 10
        Next
        For x = 1 To z Step z
        Next
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30039: Loop control variable cannot be a property or a late-bound indexed array.
        For z = 1 To 10
            ~
</expected>)
        End Sub

        <Fact(), WorkItem(545641, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/545641")>
        Public Sub MissingLatebindingHelpers()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class Program
    Shared Sub Main()
        Dim Result As Object
        For Result = 1 To 2
        Next
    End Sub
End Class
    </file>
</compilation>)
            AssertTheseEmitDiagnostics(compilation,
<expected>
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.ObjectFlowControl+ForLoopControl.ForLoopInitObj' is not defined.
        For Result = 1 To 2
        ~~~~~~~~~~~~~~~~~~~~
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.ObjectFlowControl+ForLoopControl.ForNextCheckObj' is not defined.
        For Result = 1 To 2
        ~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub MissingLatebindingHelpersObjectFor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict Off

Imports System

Class Program
    Shared Sub Main()
        Dim obj As Object = New cls1
        obj.P1 = 42                         ' assignment    (Set)
        obj.P1()                            ' side-effect   (Call)
        Console.WriteLine(obj.P1)           ' value         (Get)
    End Sub

    Class cls1
        Private _p1 As Integer
        Public Property p1 As Integer
            Get
                Console.Write("Get")
                Return _p1
            End Get
            Set(value As Integer)
                Console.Write("Set")
                _p1 = value
            End Set
        End Property
    End Class
End Class
    </file>
</compilation>)
            AssertTheseEmitDiagnostics(compilation,
<expected>
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.NewLateBinding.LateSet' is not defined.
        obj.P1 = 42                         ' assignment    (Set)
        ~~~~~~~~~~~
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.NewLateBinding.LateCall' is not defined.
        obj.P1()                            ' side-effect   (Call)
        ~~~~~~
BC35000: Requested operation is not available because the runtime library function 'Microsoft.VisualBasic.CompilerServices.NewLateBinding.LateGet' is not defined.
        Console.WriteLine(obj.P1)           ' value         (Get)
                          ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="UseOfKeywordNotInInstanceMethod1">
    <file name="a.vb">
        Class [ident1]
            Public k As Short
            Public Shared Function foo2() As String
                'COMPILEERROR: BC30043, "Me"
                Me.k = 333
                Return Nothing
            End Function
        End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'Me' is valid only within an instance method.
                Me.k = 333
                ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_WithinFieldInitializers()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="FieldsUsingMe">
    <file name="a.vb">
Option strict on
imports system

Class C1
    private f1 as integer = 21
    private shared f2 as integer = Me.f1 + 1
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'Me' is valid only within an instance method.
    private shared f2 as integer = Me.f1 + 1
                                   ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MeInAttribute()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MeInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Reflection
Imports System.Runtime.InteropServices

&lt;Assembly: AssemblyCulture(Me.AAA)&gt;

&lt;StructLayout(Me.AAA)&gt;
Structure S1
End Structure
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'Me' is valid only within an instance method.
&lt;Assembly: AssemblyCulture(Me.AAA)&gt;
                           ~~
BC30043: 'Me' is valid only within an instance method.
&lt;StructLayout(Me.AAA)&gt;
              ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyBaseInAttribute()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyBaseInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Reflection
Imports System.Runtime.InteropServices

&lt;Assembly: AssemblyCulture(MyBase.AAA)&gt;

&lt;StructLayout(MyBase.AAA)&gt;
Structure S1
End Structure
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyBase' is valid only within an instance method.
&lt;Assembly: AssemblyCulture(MyBase.AAA)&gt;
                           ~~~~~~
BC30043: 'MyBase' is valid only within an instance method.
&lt;StructLayout(MyBase.AAA)&gt;
              ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyClassInAttribute()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyClassInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Reflection
Imports System.Runtime.InteropServices

&lt;Assembly: AssemblyCulture(MyClass.AAA)&gt;

&lt;StructLayout(MyClass.AAA)&gt;
Structure S1
End Structure
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyClass' is valid only within an instance method.
&lt;Assembly: AssemblyCulture(MyClass.AAA)&gt;
                           ~~~~~~~
BC30043: 'MyClass' is valid only within an instance method.
&lt;StructLayout(MyClass.AAA)&gt;
              ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_WithinFieldInitializers_MyBase()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_WithinFieldInitializers_MyBase">
    <file name="a.vb">
Option strict on
imports system

Class Base
    Shared Function GetBar(i As Integer) As Integer
        Return Nothing
    End Function
End Class
Class C2
    Inherits Base
    Public Shared f As Func(Of Func(Of Integer, Integer)) =
            Function() New Func(Of Integer, Integer)(AddressOf MyBase.GetBar)
    Public Shared Property P As Integer = MyBase.GetBar(1)
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyBase' is valid only within an instance method.
            Function() New Func(Of Integer, Integer)(AddressOf MyBase.GetBar)
                                                               ~~~~~~
BC30043: 'MyBase' is valid only within an instance method.
    Public Shared Property P As Integer = MyBase.GetBar(1)
                                          ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_WithinFieldInitializers_MyClass()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_WithinFieldInitializers_MyClass">
    <file name="a.vb">
Option strict on
imports system

Class C2
    Shared Function GetBar(i As Integer) As Integer
        Return Nothing
    End Function
    Public Shared f As Func(Of Func(Of Integer, Integer)) =
            Function() New Func(Of Integer, Integer)(AddressOf MyClass.GetBar)
    Public Shared Property P As Integer = MyClass.GetBar(1)
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyClass' is valid only within an instance method.
            Function() New Func(Of Integer, Integer)(AddressOf MyClass.GetBar)
                                                               ~~~~~~~
BC30043: 'MyClass' is valid only within an instance method.
    Public Shared Property P As Integer = MyClass.GetBar(1)
                                          ~~~~~~~
</expected>)
        End Sub

        <WorkItem(542958, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542958")>
        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyClassInAttribute_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyClassInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Runtime.InteropServices
Public Class S1
    Const str As String = ""
    &lt;MyAttribute(MyClass.color.blue)&gt;
    Sub foo()
    End Sub
    Shared Sub main()
    End Sub
    Enum color
        blue
    End Enum
End Class
Class MyAttribute
    Inherits Attribute
    Sub New(str As S1.color)
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyClass' is valid only within an instance method.
    &lt;MyAttribute(MyClass.color.blue)&gt;
                 ~~~~~~~
</expected>)
        End Sub

        <WorkItem(542958, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542958")>
        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MeInAttribute_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MeInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Runtime.InteropServices
Public Class S1
    Const str As String = ""
    &lt;MyAttribute(Me.color.blue)&gt;
    Sub foo()
    End Sub
    Shared Sub main()
    End Sub
    Enum color
        blue
    End Enum
End Class
Class MyAttribute
    Inherits Attribute
    Sub New(str As S1.color)
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'Me' is valid only within an instance method.
    &lt;MyAttribute(Me.color.blue)&gt;
                 ~~
</expected>)
        End Sub

        <WorkItem(542958, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542958")>
        <Fact()>
        Public Sub BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyBaseInAttribute_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30043ERR_UseOfKeywordNotInInstanceMethod1_MyBaseInAttribute">
    <file name="a.vb">
Option Strict On
Imports System
Imports System.Runtime.InteropServices
Public Class BaseClass
    Enum color
        blue
    End Enum
End Class
Public Class S1
    Inherits BaseClass
    &lt;MyAttribute(MyBase.color.blue)&gt;
    Sub foo()
    End Sub
    Shared Sub main()
    End Sub
End Class
Class MyAttribute
    Inherits Attribute
    Sub New(x As S1.color)
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30043: 'MyBase' is valid only within an instance method.
    &lt;MyAttribute(MyBase.color.blue)&gt;
                 ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30044ERR_UseOfKeywordFromStructure1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="UseOfKeywordFromStructure1">
    <file name="a.vb">
        Module M1
            Structure S
                Public Overrides Function ToString() As String
                    Return MyBase.ToString()
                End Function
            End Structure
        End Module
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30044: 'MyBase' is not valid within a structure.
                    Return MyBase.ToString()
                           ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30045ERR_BadAttributeConstructor1()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BadAttributeConstructor1">
    <file name="a.vb"><![CDATA[
Imports System
            Module M1
                Class myattr1
                    Inherits Attribute
                    Sub New(ByVal o() As c1)
                        Me.o = o
                    End Sub
                    Public o() As c1
                End Class
                Public Class c1
                End Class
                <myattr1(Nothing)>
                Class Scen18
                End Class
                Class myattr2
                    Inherits Attribute
                    Sub New(ByVal o() As delegate1)
                        Me.o = o
                    End Sub
                    Public o() As delegate1
                End Class
                Delegate Sub delegate1()
                <myattr2(Nothing)>
                Class Scen20
                End Class
            End Module
    ]]></file>
</compilation>).
            VerifyDiagnostics(Diagnostic(ERRID.ERR_BadAttributeConstructor1, "myattr1").WithArguments("M1.c1()"),
                              Diagnostic(ERRID.ERR_BadAttributeConstructor1, "myattr2").WithArguments("M1.delegate1()"))

            Dim scen18 = compilation.GlobalNamespace.GetTypeMember("M1").GetTypeMember("Scen18")
            Dim attribute = scen18.GetAttributes().Single()
            Assert.Equal("M1.myattr1(Nothing)", attribute.ToString())
            Dim argument = attribute.CommonConstructorArguments(0)
            Assert.Null(argument.Type)
        End Sub

        <Fact, WorkItem(3380, "DevDiv_Projects/Roslyn")>
        Public Sub BC30046ERR_ParamArrayWithOptArgs()
            CreateCompilationWithMscorlib40(<compilation name="ERR_ParamArrayWithOptArgs">
                                                <file name="a.vb"><![CDATA[
                Class C1
                    Shared Sub Main()
                    End Sub
                    sub abc( optional k as string = "hi", paramarray s() as integer )
                    End Sub
                End Class
            ]]></file>
                                            </compilation>).VerifyDiagnostics(
                                          Diagnostic(ERRID.ERR_ParamArrayWithOptArgs, "s"))

        End Sub

        <Fact()>
        Public Sub BC30049ERR_ExpectedArray1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExpectedArray1">
    <file name="a.vb">
    Module M1
        Sub Main()
            Dim boolVar_12 As Boolean
            'COMPILEERROR: BC30049, "boolVar_12"
            ReDim boolVar_12(120)

            'COMPILEERROR: BC30049, "boolVar_12", BC30811, "as"
            ReDim boolVar_12(120, 130)
        End Sub
    End Module
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30049: 'Redim' statement requires an array.
            ReDim boolVar_12(120)
                  ~~~~~~~~~~
BC30049: 'Redim' statement requires an array.
            ReDim boolVar_12(120, 130)
                  ~~~~~~~~~~
</expected>)
        End Sub

        <WorkItem(542209, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542209")>
        <Fact()>
        Public Sub BC30052ERR_ArrayRankLimit()
            CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="ArrayRankLimit">
    <file name="a.vb">
    Public Class C1
        Dim S1(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32) As Byte
        Dim S2(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33) As Byte
    End Class
    </file>
</compilation>).VerifyDiagnostics(
            Diagnostic(ERRID.ERR_ArrayRankLimit, "(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33)"))
        End Sub

        <Fact, WorkItem(2424, "https://github.com/dotnet/roslyn/issues/2424")>
        Public Sub BC30053ERR_AsNewArray_01()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AsNewArray">
        <file name="a.vb">
        Module M1
            Sub Foo()
                Dim c() As New System.Exception
                Dim d(), e() As New System.Exception
                Dim f(), g As New System.Exception
                Dim h, i() As New System.Exception
            End Sub

            Dim x() As New System.Exception
            Dim y(), z() As New System.Exception
            Dim u(), v As New System.Exception
            Dim w, q() As New System.Exception
        End Module
        </file>
    </compilation>)

            Dim expectedErrors1 = <errors>
BC30053: Arrays cannot be declared with 'New'.
                Dim c() As New System.Exception
                           ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim d(), e() As New System.Exception
                                ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim d(), e() As New System.Exception
                                ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim f(), g As New System.Exception
                              ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim h, i() As New System.Exception
                              ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim x() As New System.Exception
                ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim y(), z() As New System.Exception
                ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim y(), z() As New System.Exception
                     ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim u(), v As New System.Exception
                ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim w, q() As New System.Exception
                   ~~~
</errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact, WorkItem(2424, "https://github.com/dotnet/roslyn/issues/2424")>
        Public Sub BC30053ERR_AsNewArray_02()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AsNewArray">
        <file name="a.vb">
        Module M1
            Sub Foo()
                Dim c(1) As New System.Exception
                Dim d(1), e(1) As New System.Exception
                Dim f(1), g As New System.Exception
                Dim h, i(1) As New System.Exception
            End Sub

            Dim x(1) As New System.Exception
            Dim y(1), z(1) As New System.Exception
            Dim u(1), v As New System.Exception
            Dim w, q(1) As New System.Exception
        End Module
        </file>
    </compilation>)

            Dim expectedErrors1 = <errors>
BC30053: Arrays cannot be declared with 'New'.
                Dim c(1) As New System.Exception
                            ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim d(1), e(1) As New System.Exception
                                  ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim d(1), e(1) As New System.Exception
                                  ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim f(1), g As New System.Exception
                               ~~~
BC30053: Arrays cannot be declared with 'New'.
                Dim h, i(1) As New System.Exception
                               ~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim x(1) As New System.Exception
                ~~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim y(1), z(1) As New System.Exception
                ~~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim y(1), z(1) As New System.Exception
                      ~~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim u(1), v As New System.Exception
                ~~~~
BC30053: Arrays cannot be declared with 'New'.
            Dim w, q(1) As New System.Exception
                   ~~~~
</errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30057ERR_TooManyArgs1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TooManyArgs1">
    <file name="a.vb">
    Module M1
        Sub Main()
            test("CC", 15, 45)
        End Sub
        Sub test(ByVal name As String, ByVal age As Integer)
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30057: Too many arguments to 'Public Sub test(name As String, age As Integer)'.
            test("CC", 15, 45)
                           ~~
</expected>)
        End Sub

        ' 30057 is better here
        <WorkItem(528720, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528720")>
        <Fact()>
        Public Sub BC30057ERR_TooManyArgs1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConstructorNotFound1">
    <file name="a.vb">
        Module M1
            Sub FOO()
                Dim DynamicArray_2() As Byte
                Dim DynamicArray_3() As Long
                'COMPILEERROR: BC30251, "New Byte(1, 2, 3, 4)"
                DynamicArray_2 = New Byte(1, 2, 3, 4)
                'COMPILEERROR: BC30251, "New Byte(1)"
                DynamicArray_3 = New Long(1)
                Exit Sub
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30057: Too many arguments to 'Public Sub New()'.
                DynamicArray_2 = New Byte(1, 2, 3, 4)
                                          ~
BC30057: Too many arguments to 'Public Sub New()'.
                DynamicArray_3 = New Long(1)
                                          ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30057ERR_TooManyArgs1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConstructorNotFound1">
    <file name="a.vb">
        Option Infer On
        Imports System
        Module Module1
            Sub Main()
                Dim arr16 As New Integer(2, 3) { {1, 2}, {2, 1} }' Invalid
            End Sub
        End Module
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30057: Too many arguments to 'Public Sub New()'.
                Dim arr16 As New Integer(2, 3) { {1, 2}, {2, 1} }' Invalid
                                         ~
BC30205: End of statement expected.
                Dim arr16 As New Integer(2, 3) { {1, 2}, {2, 1} }' Invalid
                                               ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30057ERR_TooManyArgs1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConstructorNotFound1">
    <file name="a.vb">
        Option Strict On
        Imports System
        Module Module1
            Sub Main()
                Dim myArray8 As Integer(,) = New Integer(,) 1,2,3,4,5
            End Sub
        End Module
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30057: Too many arguments to 'Public Sub New()'.
                Dim myArray8 As Integer(,) = New Integer(,) 1,2,3,4,5
                                                         ~
BC30205: End of statement expected.
                Dim myArray8 As Integer(,) = New Integer(,) 1,2,3,4,5
                                                            ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_fields()
            Dim source =
<compilation>
    <file name="a.vb">
Option Strict On
option Infer On

imports system
Imports microsoft.visualbasic.strings

        Class A
            Public Const X As Integer = 1
        End Class

        Class B
            Sub New(x As Action)
            End Sub

            Sub New(x As Integer)
            End Sub

            Public Const X As Integer = 2
        End Class

        Class C
            Sub New(x As Integer)
            End Sub

            Public Const X As Integer = 3
        End Class

        Class D
            Sub New(x As Func(Of Integer))
            End Sub

            Public Const X As Integer = 4
        End Class

Class C1
    Public Delegate Sub SubDel(p as integer)
    Public Shared Sub foo(p as Integer)
        Console.WriteLine("DelegateField works :) " + p.ToString())
    End Sub

    public shared function f() as integer
        return 23
    end function

    ' should work because of const propagation
    public const f1 as integer = 1 + 1

    '' should not work
    Public const f2 as SubDel = AddressOf C1.foo
    public const f3 as integer = C1.f()
    public const f4,f5 as integer = C1.f()

    '' should also give a BC30059 for inferred types
    public const f6 as object = new C1()
    public const f7 = new C1()

    public const f8 as integer = Asc(chrW(255)) ' > 127 are not const

    public const f9() as integer = new integer() {1, 2}
    public const f10 = new integer() {1, 2}

    public const f11 = GetType(Integer)
    public const f12 as system.type = GetType(Integer)

    public const f13 as integer = cint(cint(cbyte("1")))
    public const f14 as integer = cint(cint(cbyte(1))) ' works

    public const f15 as long = clng(cint(cbyte("1")))
    public const f16 as long = clng(cint(cbyte(1))) ' works


    public const ValueWorks1 as Integer = new C(23).X
    public const ValueWorks2 as Integer = new A().X
    public const ValueWorks3 as Integer = 23 + new A().X
    public const ValueWorks4 as Integer = if(nothing, 23)
    public const ValueWorks5 as Integer = if(23 = 42, 23, 42)
    public const ValueWorks6 as Integer = if(new A().X = 0, 23, 42)
    public const ValueWorks7 as Integer = if(new A(), nothing).X
    public const ValueWorks8 as Integer = if(23 = 42, 23, new A().X)
    public const ValueWorks9 as Integer = if(23 = 42, new A().X, 42)
    public const ValueWorks10 as Integer = CType("12", Integer).MaxValue ' needs option strict off ...
    public const ValueWorks11 as Integer = New B(Sub() Exit Sub).X
    public const ValueWorks12 = New D(Function() 23).X

    public const ValueDoesntWork1 as Integer = f()                       
    public const ValueDoesntWork2 as Integer = 1 + f()
    public const ValueDoesntWork3 as Integer = f() + 1

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    Public const f2 as SubDel = AddressOf C1.foo
                       ~~~~~~
BC30059: Constant expression is required.
    public const f3 as integer = C1.f()
                                 ~~~~~~
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
    public const f4,f5 as integer = C1.f()
                 ~~~~~~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    public const f4,f5 as integer = C1.f()
                                    ~~~~~~
BC30059: Constant expression is required.
    public const f6 as object = new C1()
                                ~~~~~~~~
BC30059: Constant expression is required.
    public const f7 = new C1()
                      ~~~~~~~~
BC30059: Constant expression is required.
    public const f8 as integer = Asc(chrW(255)) ' > 127 are not const
                                 ~~~~~~~~~~~~~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f9() as integer = new integer() {1, 2}
                 ~~
BC30059: Constant expression is required.
    public const f10 = new integer() {1, 2}
                       ~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    public const f11 = GetType(Integer)
                       ~~~~~~~~~~~~~~~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f12 as system.type = GetType(Integer)
                        ~~~~~~~~~~~
BC30060: Conversion from 'String' to 'Byte' cannot occur in a constant expression.
    public const f13 as integer = cint(cint(cbyte("1")))
                                                  ~~~
BC30060: Conversion from 'String' to 'Byte' cannot occur in a constant expression.
    public const f15 as long = clng(cint(cbyte("1")))
                                               ~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks1 as Integer = new C(23).X
                                          ~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks2 as Integer = new A().X
                                          ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks3 as Integer = 23 + new A().X
                                               ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks6 as Integer = if(new A().X = 0, 23, 42)
                                             ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks7 as Integer = if(new A(), nothing).X
                                          ~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks8 as Integer = if(23 = 42, 23, new A().X)
                                                          ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks9 as Integer = if(23 = 42, new A().X, 42)
                                                      ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks10 as Integer = CType("12", Integer).MaxValue ' needs option strict off ...
                                           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks11 as Integer = New B(Sub() Exit Sub).X
                                           ~~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
    public const ValueWorks12 = New D(Function() 23).X
                                ~~~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    public const ValueDoesntWork1 as Integer = f()                       
                                               ~~~
BC30059: Constant expression is required.
    public const ValueDoesntWork2 as Integer = 1 + f()
                                                   ~~~
BC30059: Constant expression is required.
    public const ValueDoesntWork3 as Integer = f() + 1
                                               ~~~
</expected>)
        End Sub

        ' The non-constant initializer should result in
        ' a single error, even if declaring multiple fields.
        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_2()
            Dim source =
<compilation>
    <file name="a.vb">
Option Strict On
Module M
    Const A, B As Integer = F()
    Function F() As Integer
        Return 0
    End Function
End Module
    </file>
</compilation>
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)
            compilation.AssertTheseDiagnostics(
<expected>
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
    Const A, B As Integer = F()
          ~~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    Const A, B As Integer = F()
                            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_locals()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict off
imports system
Imports microsoft.visualbasic.strings

        Class A
            Public Const X As Integer = 1
        End Class

        Class B
            Sub New(x As Action)
            End Sub

            Sub New(x As Integer)
            End Sub

            Public Const X As Integer = 2
        End Class

        Class C
            Sub New(x As Integer)
            End Sub

            Public Const X As Integer = 3
        End Class

        Class D
            Sub New(x As Func(Of Integer))
            End Sub

            Public Const X As Integer = 4
        End Class

Class C1
    Public Delegate Sub SubDel(p as integer)
    Public Shared Sub foo(p as Integer)
        Console.WriteLine("DelegateField works :) " + p.ToString())
    End Sub

    public shared function f() as integer
        return 23
    end function

    Public Sub Main()
        ' should work because of const propagation
        const f1 as integer = 1 + 1

        ' should not work
        const f2 as SubDel = AddressOf C1.foo
        const f3 as integer = C1.f()
        const f4,f5 as integer = C1.f()

        ' should also give a BC30059 for inferred types
        const f6 as object = new C1()
        const f7 = new C1()

        const f8 as integer = Asc(chrW(255)) ' > 127 are not const

        const f9() as integer = new integer() {1, 2}
        const f10 = new integer() {1, 2}

        const f11 = GetType(Integer)
        const f12 as system.type = GetType(Integer)

        const f13 as integer = cint(cint(cbyte("1")))
        const f14 as integer = cint(cint(cbyte(1))) ' works

        const f15 as long = clng(cint(cbyte("1")))
        const f16 as long = clng(cint(cbyte(1))) ' works

        const ValueWorks1 as Integer = new C(23).X
        const ValueWorks2 as Integer = new A().X
        const ValueWorks3 as Integer = 23 + new A().X
        const ValueWorks4 as Integer = if(nothing, 23)
        const ValueWorks5 as Integer = if(23 = 42, 23, 42)
        const ValueWorks6 as Integer = if(new A().X = 0, 23, 42)
        const ValueWorks7 as Integer = if(new A(), nothing).X
        const ValueWorks8 as Integer = if(23 = 42, 23, new A().X)
        const ValueWorks9 as Integer = if(23 = 42, new A().X, 42)
        const ValueWorks10 as Integer = CType("12", Integer).MaxValue ' needs option strict off ...
        const ValueWorks11 as Integer = New B(Sub() Exit Sub).X
        const ValueWorks12 as Integer = New D(Function() 23).X

        const ValueDoesntWork1 as Integer = f()                       

        Dim makeThemUsed as long = f1 + f3 + f4 + f5 + f8 + f13 + f14 + f15 + f16 + 
                                   ValueWorks1 + ValueWorks2 + ValueWorks3 + ValueWorks4 + 
                                   ValueWorks5 + ValueWorks6 + ValueWorks7 + ValueWorks8 + ValueWorks9 +
                                   ValueWorks10 + ValueWorks11 + ValueWorks12
    End Sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
        const f2 as SubDel = AddressOf C1.foo
                    ~~~~~~
BC30059: Constant expression is required.
        const f3 as integer = C1.f()
                              ~~~~~~
BC30438: Constants must have a value.
        const f4,f5 as integer = C1.f()
              ~~
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
        const f4,f5 as integer = C1.f()
              ~~~~~~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
        const f4,f5 as integer = C1.f()
                                 ~~~~~~
BC30059: Constant expression is required.
        const f6 as object = new C1()
                             ~~~~~~~~
BC30059: Constant expression is required.
        const f7 = new C1()
                   ~~~~~~~~
BC30059: Constant expression is required.
        const f8 as integer = Asc(chrW(255)) ' > 127 are not const
                              ~~~~~~~~~~~~~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
        const f9() as integer = new integer() {1, 2}
              ~~~~
BC30059: Constant expression is required.
        const f10 = new integer() {1, 2}
                    ~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
        const f11 = GetType(Integer)
                    ~~~~~~~~~~~~~~~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
        const f12 as system.type = GetType(Integer)
                     ~~~~~~~~~~~
BC30060: Conversion from 'String' to 'Byte' cannot occur in a constant expression.
        const f13 as integer = cint(cint(cbyte("1")))
                                               ~~~
BC30060: Conversion from 'String' to 'Byte' cannot occur in a constant expression.
        const f15 as long = clng(cint(cbyte("1")))
                                            ~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks1 as Integer = new C(23).X
                                       ~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks2 as Integer = new A().X
                                       ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks3 as Integer = 23 + new A().X
                                            ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks6 as Integer = if(new A().X = 0, 23, 42)
                                          ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks7 as Integer = if(new A(), nothing).X
                                       ~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks8 as Integer = if(23 = 42, 23, new A().X)
                                                       ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks9 as Integer = if(23 = 42, new A().X, 42)
                                                   ~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks10 as Integer = CType("12", Integer).MaxValue ' needs option strict off ...
                                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks11 as Integer = New B(Sub() Exit Sub).X
                                        ~~~~~~~~~~~~~~~~~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        const ValueWorks12 as Integer = New D(Function() 23).X
                                        ~~~~~~~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
        const ValueDoesntWork1 as Integer = f()                       
                                            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_3()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
Option Infer On
imports system

Class C1
    Public Delegate Sub SubDel(p as integer)
    Public Shared Sub foo(p as Integer)
        Console.WriteLine("DelegateField works :) " + p.ToString())
    End Sub

    public shared function f() as integer
        return 23
    end function

    public shared function g(p as integer) as integer
        return 23
    end function

    ' should not work
    Public const f2 = AddressOf C1.foo
    Public const f3 as object = AddressOf C1.foo
    public const f4 as integer = 1 + 2 + 3 + f() 
    public const f5 as boolean = not (f() = 23)
    public const f6 as integer = f() + f() + f()
    public const f7 as integer = g(1 + 2 + f())

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30059: Constant expression is required.
    Public const f2 = AddressOf C1.foo
                      ~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    Public const f3 as object = AddressOf C1.foo
                                ~~~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    public const f4 as integer = 1 + 2 + 3 + f() 
                                             ~~~
BC30059: Constant expression is required.
    public const f5 as boolean = not (f() = 23)
                                      ~~~
BC30059: Constant expression is required.
    public const f6 as integer = f() + f() + f()
                                 ~~~
BC30059: Constant expression is required.
    public const f6 as integer = f() + f() + f()
                                       ~~~
BC30059: Constant expression is required.
    public const f6 as integer = f() + f() + f()
                                             ~~~
BC30059: Constant expression is required.
    public const f7 as integer = g(1 + 2 + f())
                                 ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact>
        Public Sub TestArrayLocalConst()

            Dim source =
<compilation>
    <file name="a.vb">
Imports System
Module C
    Sub Main()
        Const A As Integer() = Nothing
        Console.Write(A)
    End Sub
End Module
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
        Const A As Integer() = Nothing
              ~
</expected>)

        End Sub

        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_Attr()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ERR_RequiredConstExpr_Attr">
        <file name="at30059.vb"><![CDATA[
Imports System

<AttributeUsage(AttributeTargets.All)>
Public Class MyAttribute
    Inherits Attribute
    Public Sub New(p As ULong)
    End Sub
End Class

<My(Foo.FG)>
Public Class Foo
    Public Shared FG As ULong = 12345
    Public Function F() As Byte
        Dim x As Byte = 1
        Return x
    End Function
End Class
    ]]></file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_RequiredConstExpr, "Foo.FG"))

        End Sub

        <WorkItem(542967, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542967")>
        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_QueryInAttr()

            CompilationUtils.CreateCompilationWithMscorlib40AndReferences(
    <compilation name="ERR_RequiredConstExpr_Attr">
        <file name="at30059.vb"><![CDATA[
Imports System
Imports System.Linq
Class Program
    Const q As String = ""
    Sub Main(args As String())
    End Sub
    <My((From x In q Select x).Count())>
    Shared Sub sum()
    End Sub
End Class
Class MyAttribute
    Inherits Attribute
    Sub New(s As Integer)
    End Sub
End Class
    ]]></file>
    </compilation>, references:={Net40.SystemCore}).VerifyDiagnostics(Diagnostic(ERRID.ERR_RequiredConstExpr, "(From x In q Select x).Count()"))
        End Sub

        <WorkItem(542967, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542967")>
        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_QueryInAttr_2()

            CompilationUtils.CreateCompilationWithMscorlib40AndReferences(
    <compilation name="ERR_RequiredConstExpr_Attr">
        <file name="at30059.vb"><![CDATA[
Imports System
Imports System.Linq
Class Program
    Const q As String = ""
    Sub Main(args As String())
    End Sub
    Public F1 As Object
    <My((From x In q Select F1).Count())>
    Shared Sub sum()
    End Sub
End Class
Class MyAttribute
    Inherits Attribute
    Sub New(s As Integer)
    End Sub
End Class
    ]]></file>
    </compilation>, references:={Net40.SystemCore}).VerifyDiagnostics(Diagnostic(ERRID.ERR_BadInstanceMemberAccess, "F1"))
        End Sub

        <WorkItem(542967, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542967")>
        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_QueryInAttr_3()

            CompilationUtils.CreateCompilationWithMscorlib40AndReferences(
    <compilation name="ERR_RequiredConstExpr_Attr">
        <file name="at30059.vb"><![CDATA[
Imports System
Imports System.Linq

<My((From x In "s" Select x).Count())>
Class Program
    Public F1 As Integer
End Class

<My((From x In "s" Select Program.F1).Count())>
Class Program2
End Class

Class MyAttribute
    Inherits Attribute
    Sub New(s As Integer)
    End Sub
End Class
    ]]></file>
    </compilation>, references:={Net40.SystemCore}).VerifyDiagnostics({Diagnostic(ERRID.ERR_RequiredConstExpr, "(From x In ""s"" Select x).Count()"),
                                                                    Diagnostic(ERRID.ERR_ObjectReferenceNotSupplied, "Program.F1")})
        End Sub

        <Fact()>
        Public Sub BC30059ERR_RequiredConstExpr_XmlEmbeddedExpression()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Option Strict On
Module M
    Private Const F1 = Nothing
    Private Const F2 As String = "v2"
    Private Const F3 = <%= Nothing %>
    Private Const F4 = <%= "v4" %>
    Private Const F5 As String = <%= "v5" %>
    Private F6 As Object = <x a0=<%= "v0" %> a1=<%= F1 %> a2=<%= F2 %> a3=<%= F3 %> a4=<%= F4 %> a5=<%= F5 %>/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC30059: Constant expression is required.
    Private Const F3 = <%= Nothing %>
                       ~~~~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private Const F3 = <%= Nothing %>
                       ~~~~~~~~~~~~~~
BC30059: Constant expression is required.
    Private Const F4 = <%= "v4" %>
                       ~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private Const F4 = <%= "v4" %>
                       ~~~~~~~~~~~
BC30059: Constant expression is required.
    Private Const F5 As String = <%= "v5" %>
                                 ~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private Const F5 As String = <%= "v5" %>
                                 ~~~~~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC30060ERR_RequiredConstConversion2()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
imports system

Class C1
    ' should show issues
    public const f1 as integer = CInt("23")
    public const f2 as integer = CType("23", integer)
    public const f3 as byte = CType(300, byte)
    public const f4 as byte = CType(300, BORG)
    public const f5 as byte = 300
    public const f6 as string = 23
    public const f10 as date = CDate("November 04, 2008")
    public const f13 as decimal = Ctype("20100607",decimal)


    ' should not show issues
    public const f7 as integer = CInt(23)
    public const f8 as integer = CType(23, integer)
    public const f9 as byte = CType(254, byte)
    public const f11 as date = Ctype(#06/07/2010#,date)    
    public const f12 as decimal = Ctype(20100607,decimal)

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30060: Conversion from 'String' to 'Integer' cannot occur in a constant expression.
    public const f1 as integer = CInt("23")
                                      ~~~~
BC30060: Conversion from 'String' to 'Integer' cannot occur in a constant expression.
    public const f2 as integer = CType("23", integer)
                                       ~~~~
BC30439: Constant expression not representable in type 'Byte'.
    public const f3 as byte = CType(300, byte)
                                    ~~~
BC30002: Type 'BORG' is not defined.
    public const f4 as byte = CType(300, BORG)
                                         ~~~~
BC30439: Constant expression not representable in type 'Byte'.
    public const f5 as byte = 300
                              ~~~
BC30512: Option Strict On disallows implicit conversions from 'Integer' to 'String'.
    public const f6 as string = 23
                                ~~
BC30060: Conversion from 'String' to 'Date' cannot occur in a constant expression.
    public const f10 as date = CDate("November 04, 2008")
                                     ~~~~~~~~~~~~~~~~~~~
BC30060: Conversion from 'String' to 'Decimal' cannot occur in a constant expression.
    public const f13 as decimal = Ctype("20100607",decimal)
                                        ~~~~~~~~~~

</expected>)
        End Sub

        <Fact()>
        Public Sub BC30060ERR_RequiredConstConversion2_StrictOff()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict off
imports system

Class C1
    public const f6 as string = 23
    public const f7 as string = CType(23,string)

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30060: Conversion from 'Integer' to 'String' cannot occur in a constant expression.
    public const f6 as string = 23
                                ~~
BC30060: Conversion from 'Integer' to 'String' cannot occur in a constant expression.
    public const f7 as string = CType(23,string)
                                      ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30064ERR_ReadOnlyAssignment()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ReadOnlyAssignment">
    <file name="a.vb">
    Imports System

    Module M1

    Class ReferenceType
        Implements IDisposable
        Dim dummy As Integer
        Public Sub Dispose() Implements System.IDisposable.Dispose
        End Sub
    End Class

    Class TestClass
        ReadOnly Name As String = "Cici"
        Sub test()
            Name = "string"

            ' variables declared in a using statement are considered read only as well
            Using a As New ReferenceType(), b As New ReferenceType()
                a = New ReferenceType()
                b = New ReferenceType()
            End Using
        End Sub
    End Class
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30064: 'ReadOnly' variable cannot be the target of an assignment.
            Name = "string"
            ~~~~
BC30064: 'ReadOnly' variable cannot be the target of an assignment.
                a = New ReferenceType()
                ~
BC30064: 'ReadOnly' variable cannot be the target of an assignment.
                b = New ReferenceType()
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30065ERR_ExitSubOfFunc_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ExitSubOfFunc">
        <file name="a.vb">
        Public Class C1
                    Function FOO()
                        If (True)
                            Exit Sub
                        End If
                        Return Nothing
                    End Function
                End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30065: 'Exit Sub' is not valid in a Function or Property.
                            Exit Sub
                            ~~~~~~~~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30067ERR_ExitFuncOfSub_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ExitFuncOfSub">
        <file name="a.vb">
        Public Class C1
            Sub FOO()
                If (True)
        lb1:        Exit Function
                End If
            End Sub
        End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30067: 'Exit Function' is not valid in a Sub or Property.
        lb1:        Exit Function
                    ~~~~~~~~~~~~~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30068ERR_LValueRequired()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="LValueRequired">
    <file name="a.vb">
    Module M1
        Class TestClass
            ReadOnly Name As String = "Cici"
            Sub test()
                Dim obj As Cls1
                obj.Test = 1
                obj.Test() = 1
            End Sub
        End Class
        Class Cls1
            Public Overridable Sub Test()
            End Sub
        End Class
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42104: Variable 'obj' is used before it has been assigned a value. A null reference exception could result at runtime.
                obj.Test = 1
                ~~~
BC30068: Expression is a value and therefore cannot be the target of an assignment.
                obj.Test = 1
                ~~~~~~~~
BC30068: Expression is a value and therefore cannot be the target of an assignment.
                obj.Test() = 1
                ~~~~~~~~~~    
</expected>)
        End Sub


        <WorkItem(575055, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/575055")>
        <Fact>
        Public Sub BC30068ERR_IdentifierWithSameNameDifferentScope()
            Dim source = <compilation>
                             <file name="DuplicateID.vb"><![CDATA[
Module Module1    
	Dim list2 As Integer() = {1}
	Dim ddd = From i In list2 Where i > Foo(Function(m1) If(True, Sub(m2) Call Function(m3)
                                                                                   Return Sub() If True Then For Each i In list2 : m1 = i : Exit Sub : Exit For : Next
                                                                               End Function(m2), Sub(m2) Call Function(m3)
                                                                                                                  Return Sub() If True Then For Each i In list2 : m1 = i : Exit Sub : Exit For : Next
                                                                                                              End Function(m2)))
	Sub Main()    
	End Sub    

	Function Foo(ByVal x)        
		Return x.Invoke(1)
	End Function
End Module
    ]]></file></compilation>

            CreateCompilationWithMscorlib40AndVBRuntime(source).VerifyDiagnostics(Diagnostic(ERRID.ERR_ExpectedQueryableSource, "list2").WithArguments("Integer()"),
                                                                                                 Diagnostic(ERRID.ERR_LValueRequired, "i"),
                                                                                                 Diagnostic(ERRID.ERR_LValueRequired, "i"))
        End Sub


        ' change error 30098 to 30068
        <WorkItem(538107, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538107")>
        <Fact()>
        Public Sub BC30068ERR_LValueRequired_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ReadOnlyProperty1">
    <file name="a.vb">
    Module SB008mod
        Public ReadOnly Name As String
        Public ReadOnly Name1 As Struct1
        Public ReadOnly Name2 As Class1

        Sub SB008()
            Name = "15"
            Name1.Field = "15"
            Name2.Field = "15"
            System.TypeCode.Boolean=0
            A().Field="15"
        End Sub

        Function A() As Struct1 
            Return Nothing
        End Function
    End Module

    Structure Struct1
        Public Field As String
    End Structure

    Class Class1
        Public Field As String
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30064: 'ReadOnly' variable cannot be the target of an assignment.
            Name = "15"
            ~~~~
BC30064: 'ReadOnly' variable cannot be the target of an assignment.
            Name1.Field = "15"
            ~~~~~~~~~~~
BC30074: Constant cannot be the target of an assignment.
            System.TypeCode.Boolean=0
            ~~~~~~~~~~~~~~~~~~~~~~~
BC30068: Expression is a value and therefore cannot be the target of an assignment.
            A().Field="15"
            ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30068ERR_LValueRequired_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="LValueRequired">
    <file name="a.vb">
    Class A
        Property P
        Shared Property Q
    End Class
    Structure B
        Property P
        Shared Property Q
    End Structure
    Class C
        Property P As A
        Property Q As B
        Sub M()
            P.P = Nothing ' no error
            Q.P = Nothing ' BC30068
            A.Q = Nothing ' no error
            B.Q = Nothing ' no error
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30068: Expression is a value and therefore cannot be the target of an assignment.
            Q.P = Nothing ' BC30068
            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30069ERR_ForIndexInUse1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ForIndexInUse1">
    <file name="a.vb">
    Module A
        Sub TEST()
            Dim n(3) As Integer
            Dim u As Integer
            For u = n(0) To n(3) Step n(0)
                ' BC30069: For loop control variable 'u' already in use by an enclosing For loop.
                For u = 1 To 9
                Next
            Next
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30069: For loop control variable 'u' already in use by an enclosing For loop.
                For u = 1 To 9
                    ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30070ERR_NextForMismatch1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NextForMismatch1">
    <file name="a.vb">
    Module A
        Sub TEST()
            Dim n(3) As Integer
            Dim u As Integer
            Dim k As Integer
            For u = n(0) To n(3) Step n(0)
                For k = 1 To 9
                Next u
            Next k
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30070: Next control variable does not match For loop control variable 'k'.
                Next u
                     ~
BC30070: Next control variable does not match For loop control variable 'u'.
            Next k
                 ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30070ERR_NextForMismatch1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="NextForMismatch1">
    <file name="a.vb">
Class C
    Public Shared Sub Main()
        Dim S As String = "ABC"
        Dim T As String = "XYZ"
        For Each x As Char In S
            For Each y As Char In T
        Next y, x

        For Each x As Char In S
            For Each y As Char In T
        Next x, y
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30070: Next control variable does not match For loop control variable 'y'.
        Next x, y
             ~
BC30451: 'y' is not declared. It may be inaccessible due to its protection level.
        Next x, y
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30074ERR_CantAssignToConst()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="CantAssignToConst">
    <file name="a.vb">
    Class c1_0
        Public foo As Byte
    End Class
    Class c2_1
        Inherits c1_0
        Public Shadows Const foo As Short = 15
        Sub test()
            'COMPILEERROR: BC30074, "foo"
            foo = 1
        End Sub
    End Class
    Class c3_1
        Inherits c1_0
        Public Shadows Const foo As Short = 15
    End Class
    Class c2_2
        Sub test()
            Dim obj As c3_1
            obj.foo = 10
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30074: Constant cannot be the target of an assignment.
            foo = 1
            ~~~
BC30074: Constant cannot be the target of an assignment.
            obj.foo = 10
            ~~~~~~~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
            obj.foo = 10
            ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30075ERR_NamedSubscript()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="NamedSubscript">
    <file name="a.vb">
    Class c1
        Sub test()
            Dim Array As Integer() = new integer(){1}
            Array(Index:=10) = 1
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30075: Named arguments are not valid as array subscripts.
            Array(Index:=10) = 1
            ~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30089ERR_ExitDoNotWithinDo()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ERR_ExitDoNotWithinDo">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            Select Case s
                Case "userID"
                    Exit do
            End Select
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30089: 'Exit Do' can only appear inside a 'Do' statement.
                    Exit do
                    ~~~~~~~    
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30089ERR_ExitDoNotWithinDo_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ERR_ExitDoNotWithinDo">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            If s Then
                Exit Do
            Else
            End If
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30089: 'Exit Do' can only appear inside a 'Do' statement.
                Exit Do
                ~~~~~~~    
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30094ERR_MultiplyDefined1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="MultiplyDefined1">
    <file name="a.vb">
    Module M1
        Sub Main()
            SB008()
        End Sub
        Sub SB008()
    [cc]:
            'COMPILEERROR: BC30094, "cc"
    cc:
            Exit Sub
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30094: Label 'cc' is already defined in the current method.
    cc:
    ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub Bug585223_notMultiplyDefined()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="MultiplyDefined1">
    <file name="a.vb">
        <![CDATA[

Module Program
    Sub Main()
&H100000000:
&H000000000:
    End Sub
End Module

]]>
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>

</expected>)
        End Sub

        <Fact()>
        Public Sub Bug585223_notMultiplyDefined_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="MultiplyDefined1">
    <file name="a.vb">
        <![CDATA[

Module Program
    Sub Main()
&HF:
&HFF:
&HFFF:
&HFFFF:
&HFFFFF:
&HFFFFFF:
&HFFFFFFF:
&HFFFFFFFF:
&HFFFFFFFFF:
&HFFFFFFFFFF:
&HFFFFFFFFFFF:
&HFFFFFFFFFFFF:
&HFFFFFFFFFFFFF:
&HFFFFFFFFFFFFFF:
&HFFFFFFFFFFFFFFF:
&HFFFFFFFFFFFFFFFF:
    End Sub
End Module


]]>
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>

</expected>)
        End Sub

        <Fact()>
        Public Sub BC30096ERR_ExitForNotWithinFor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExitForNotWithinFor">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            Select Case s
                Case "userID"
                    Exit For
            End Select
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30096: 'Exit For' can only appear inside a 'For' statement.
                    Exit For
                    ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30096ERR_ExitForNotWithinFor_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExitForNotWithinFor">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            If s Then
                Exit For
            Else
            End If
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30096: 'Exit For' can only appear inside a 'For' statement.
                Exit For
                ~~~~~~~~   
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30097ERR_ExitWhileNotWithinWhile()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExitWhileNotWithinWhile">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            Select Case s
                Case "userID"
                    Exit While
            End Select
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30097: 'Exit While' can only appear inside a 'While' statement.
                    Exit While
                    ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30097ERR_ExitWhileNotWithinWhile_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExitWhileNotWithinWhile">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            If s Then
                Exit While
            Else
            End If
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30097: 'Exit While' can only appear inside a 'While' statement.
                Exit While
                ~~~~~~~~~~   
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30099ERR_ExitSelectNotWithinSelect()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ExitSelectNotWithinSelect">
    <file name="a.vb">
    Structure myStruct1
        Public Sub m(ByVal s As String)
            If s Then
                Exit Select
            Else
            End If
        End Sub
    End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30099: 'Exit Select' can only appear inside a 'Select' statement.
                Exit Select
                ~~~~~~~~~~~   
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30101ERR_BranchOutOfFinally()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BranchOutOfFinally">
    <file name="a.vb">
    Imports System    
    Module M1
        Sub Foo()
            Try
                Try
    Label1:         
                Catch

                Finally
                    GoTo Label1
                End Try
            Catch
            Finally
            End Try
            Exit Sub
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30101: Branching out of a 'Finally' is not valid.
                    GoTo Label1
                         ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30101ERR_BranchOutOfFinally2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BC30101ERR_BranchOutOfFinally2">
    <file name="a.vb">
    Imports System    
    Module M1
        Sub Foo()
            Try
                Try
                Catch
                Finally
Label2:
                    GoTo Label2
                End Try
            Catch
            Finally
            End Try
            Exit Sub
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation, <expected></expected>)
        End Sub

        <Fact()>
        Public Sub BC30101ERR_BranchOutOfFinally3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BC30101ERR_BranchOutOfFinally3">
    <file name="a.vb">
    Imports System    
    Module M1
        Sub Foo()
            Try
            Catch
            Finally
Label2:
                GoTo Label2
            End Try
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation, <expected></expected>)
        End Sub

        <Fact()>
        Public Sub BC30101ERR_BranchOutOfFinally4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BC30101ERR_BranchOutOfFinally4">
    <file name="a.vb">
    Imports System    
    Module M1
        Sub Foo()
            Try
            Catch
            Finally
                GoTo L2
L2:
                GoTo L2
            End Try
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation, <expected></expected>)
        End Sub

        <Fact()>
        Public Sub BC30103ERR_QualNotObjectRecord1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    Imports System
    Enum E
        A
    End Enum
    Class C
        Shared Sub M(Of T)()
            Dim [object] As Object = Nothing
            M([object]!P)
            Dim [enum] As E = E.A
            M([enum]!P)
            Dim [boolean] As Boolean = False
            M([boolean]!P)
            Dim [char] As Char = Nothing
            M([char]!P)
            Dim [sbyte] As SByte = Nothing
            M([sbyte]!P)
            Dim [byte] As Byte = Nothing
            M([byte]!P)
            Dim [int16] As Int16 = Nothing
            M([int16]!P)
            Dim [uint16] As UInt16 = Nothing
            M([uint16]!P)
            Dim [int32] As Int32 = Nothing
            M([int32]!P)
            Dim [uint32] As UInt32 = Nothing
            M([uint32]!P)
            Dim [int64] As Int64 = Nothing
            M([int64]!P)
            Dim [uint64] As UInt64 = Nothing
            M([uint64]!P)
            Dim [decimal] As Decimal = Nothing
            M([decimal]!P)
            Dim [single] As Single = Nothing
            M([single]!P)
            Dim [double] As Double = Nothing
            M([double]!P)
            Dim [type] As Type = Nothing
            M([type]!P)
            Dim [array] As Integer() = Nothing
            M([array]!P)
            Dim [nullable] As Nullable(Of Integer) = Nothing
            M([nullable]!P)
            Dim [datetime] As DateTime = Nothing
            M([datetime]!P)
            Dim [action] As Action = Nothing
            M([action]!P)
            Dim tp As T = Nothing
            M(tp!P)
        End Sub
        Shared Sub M(o)
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'E'.
            M([enum]!P)
              ~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Boolean'.
            M([boolean]!P)
              ~~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Char'.
            M([char]!P)
              ~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'SByte'.
            M([sbyte]!P)
              ~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Byte'.
            M([byte]!P)
              ~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Short'.
            M([int16]!P)
              ~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'UShort'.
            M([uint16]!P)
              ~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Integer'.
            M([int32]!P)
              ~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'UInteger'.
            M([uint32]!P)
              ~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Long'.
            M([int64]!P)
              ~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'ULong'.
            M([uint64]!P)
              ~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Decimal'.
            M([decimal]!P)
              ~~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Single'.
            M([single]!P)
              ~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Double'.
            M([double]!P)
              ~~~~~~~~
BC30367: Class 'Type' cannot be indexed because it has no default property.
            M([type]!P)
              ~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Integer()'.
            M([array]!P)
              ~~~~~~~
BC30690: Structure 'Integer?' cannot be indexed because it has no default property.
            M([nullable]!P)
              ~~~~~~~~~~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Date'.
            M([datetime]!P)
              ~~~~~~~~~~
BC30555: Default member of 'Action' is not a property.
            M([action]!P)
              ~~~~~~~~
BC30547: 'T' cannot be indexed because it has no default property.
            M(tp!P)
              ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30103ERR_QualNotObjectRecord1a()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    Enum E
        A
    End Enum
    Class C
        Shared Sub M(Of T)()
            Dim [string] As String = Nothing
            M([string]!P)
        End Sub
        Shared Sub M(o)
        End Sub
    End Class
    </file>
</compilation>)
            compilation.VerifyEmitDiagnostics(
                Diagnostic(ERRID.ERR_MissingRuntimeHelper, "P").WithArguments("Microsoft.VisualBasic.CompilerServices.Conversions.ToInteger"))
        End Sub

        <Fact()>
        Public Sub BC30103ERR_QualNotObjectRecord2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="QualNotObjectRecord1">
    <file name="a.vb">
    Imports System
    Module BitOp001mod
        Sub BitOp001()
            Dim b As Byte = 2
            Dim c As Byte = 3

            Dim s As Short = 2
            Dim t As Short = 3

            Dim i As Integer = 2
            Dim j As Integer = 3

            Dim l As Long = 2
            Dim m As Long = 3
            b = b!c
            s = s!t
            i = i!j
            l = l!m
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Byte'.
            b = b!c
                ~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Short'.
            s = s!t
                ~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Integer'.
            i = i!j
                ~
BC30103: '!' requires its left operand to have a type parameter, class or interface type, but this operand has the type 'Long'.
            l = l!m
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30105ERR_TooFewIndices()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TooFewIndices">
    <file name="a.vb">
    Imports System
    Module M1
        Sub AryChg001()
            Dim a() As Integer = New Integer() {9, 10}
            ReDim a(10)
            Dim a8() As Integer = New Integer() {1, 2}
            Dim b8() As Integer = New Integer() {3, 4}
            Dim c8 As Integer
            a8() = b8
            b8 = a()
            a8() = c8
            c8 = a8()
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30105: Number of indices is less than the number of dimensions of the indexed array.
            a8() = b8
              ~~
BC30105: Number of indices is less than the number of dimensions of the indexed array.
            b8 = a()
                  ~~
BC30105: Number of indices is less than the number of dimensions of the indexed array.
            a8() = c8
              ~~
BC30105: Number of indices is less than the number of dimensions of the indexed array.
            c8 = a8()
                   ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30106ERR_TooManyIndices()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TooManyIndices">
    <file name="a.vb">
    Imports System
    Module M1
        Sub AryChg001()
            Dim a() As Integer = New Integer() {9, 10}
            ReDim a(10)
            Dim a8() As Integer = New Integer() {1, 2}
            Dim b8() As Integer = New Integer() {3, 4}
            Dim c8 As Integer
            a8(1, 2) = b8(1)
            b8 = a(0, 4)
            a8(4, 5, 6) = c8
            c8 = a8(1, 2)
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30106: Number of indices exceeds the number of dimensions of the indexed array.
            a8(1, 2) = b8(1)
              ~~~~~~
BC30106: Number of indices exceeds the number of dimensions of the indexed array.
            b8 = a(0, 4)
                  ~~~~~~
BC30106: Number of indices exceeds the number of dimensions of the indexed array.
            a8(4, 5, 6) = c8
              ~~~~~~~~~
BC30106: Number of indices exceeds the number of dimensions of the indexed array.
            c8 = a8(1, 2)
                   ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30107ERR_EnumNotExpression1()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="EnumNotExpression1">
    <file name="a.vb">
    Option Strict On
    Module BitOp001mod1
        Sub BitOp001()
            Dim b As Byte = 2
            Dim c As Byte = 3
            Dim s As Short = 2
            Dim t As Short = 3
            Dim i As Integer = 2
            Dim j As Integer = 3
            Dim l As Long = 2
            Dim m As Long = 3
            b = b &amp; c
            b = b ^ c
            s = s &amp; t
            s = s ^ t
            i = i &amp; j
            i = i ^ j
            l = l &amp; m
            l = l ^ m
            Exit Sub
        End Sub
    End Module
    </file>
</compilation>).VerifyDiagnostics(
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "b & c").WithArguments("String", "Byte"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "b ^ c").WithArguments("Double", "Byte"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "s & t").WithArguments("String", "Short"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "s ^ t").WithArguments("Double", "Short"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "i & j").WithArguments("String", "Integer"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "i ^ j").WithArguments("Double", "Integer"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "l & m").WithArguments("String", "Long"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "l ^ m").WithArguments("Double", "Long"))

        End Sub

        <Fact()>
        Public Sub BC30108ERR_TypeNotExpression1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TypeNotExpression1">
    <file name="a.vb">
    Module Module1
        Sub Main()
            Module1
        End Sub
    End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30108: 'Module1' is a type and cannot be used as an expression.
            Module1
            ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30108ERR_TypeNotExpression1_1()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TypeNotExpression1">
    <file name="a.vb">
Imports System.Collections.Generic
Module Program
    Sub Main()
        Dim lst As New List(Of String) From {Program, "abc", "def", "ghi"}
    End Sub
End Module
    </file>
</compilation>).VerifyDiagnostics(
            Diagnostic(ERRID.ERR_TypeNotExpression1, "Program").WithArguments("Program"))
        End Sub

        <WorkItem(545166, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/545166")>
        <Fact()>
        Public Sub BC30109ERR_ClassNotExpression1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ClassNotExpression1">
    <file name="a.vb">
        Imports System
        Module M1
            Sub FOO()
                Dim c As Object
                c = String(3, "Hai123")
                c = String
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30109: 'String' is a class type and cannot be used as an expression.
                c = String(3, "Hai123")
                    ~~~~~~
BC30109: 'String' is a class type and cannot be used as an expression.
                c = String
                    ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30110ERR_StructureNotExpression1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="StructureNotExpression1">
    <file name="a.vb">
        Imports System
        Structure S1
        End Structure
        Module M1
            Sub FOO()
                Dim c As Object
                c = S1(3, "Hai123")
                c = S1
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30110: 'S1' is a structure type and cannot be used as an expression.
                c = S1(3, "Hai123")
                    ~~
BC30110: 'S1' is a structure type and cannot be used as an expression.
                c = S1
                    ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30111ERR_InterfaceNotExpression1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="StructureNotExpression1">
    <file name="a.vb">
        Imports System
        Interface S1
        End Interface
        Module M1
            Sub FOO()
                Dim c As Object
                c = S1(3, "Hai123")
                c = S1
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30111: 'S1' is an interface type and cannot be used as an expression.
                c = S1(3, "Hai123")
                    ~~
BC30111: 'S1' is an interface type and cannot be used as an expression.
                c = S1
                    ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30112ERR_NamespaceNotExpression1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NamespaceNotExpression1">
    <file name="a.vb">
        Imports System
        Module M1
            Sub Foo()
                'COMPILEERROR: BC30112, "Text$"
                Text$
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30112: 'System.Text' is a namespace and cannot be used as an expression.
                Text$
                ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30112ERR_NamespaceNotExpression2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NamespaceNotExpression1">
    <file name="a.vb">
        Option Infer On

        Namespace X
            Class Program
                Sub Main()
                    'COMPILEERROR: BC30112, "x"
                    For Each x In ""
                    Next
                End Sub
            End Class
        End Namespace
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30112: 'X' is a namespace and cannot be used as an expression.
                    For Each x In ""
                             ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30114ERR_XmlPrefixNotExpression()
            Dim options = TestOptions.ReleaseDll.WithGlobalImports(GlobalImport.Parse({"<xmlns:p1=""..."">"}))
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System
Imports <xmlns:p2="...">
Imports <xmlns:p4="...">
Module M
    Private F1 As Object = p1
    Private F2 As Object = P1
    Private F3 As Object = xml
    Private F4 As Object = XML
    Private F5 As Object = xmlns
    Private F6 As Object = XMLNS
    Private F7 As Object = <x xmlns:p3="...">
                               <%= p2 %>
                               <%= p3 %>
                               <%= xmlns %>
                           </x>
    Private Function F8(p1 As Object) As Object
        Return p1
    End Function
    Private Function F9(xmlns As Object) As Object
        Return p2
    End Function
    Private F10 As Object = p4
End Module
    ]]></file>
    <file name="b.vb"><![CDATA[
Class p4
End Class
    ]]></file>
</compilation>, references:=XmlReferences, options:=options)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC30114: 'p1' is an XML prefix and cannot be used as an expression.  Use the GetXmlNamespace operator to create a namespace object.
    Private F1 As Object = p1
                           ~~
BC30451: 'P1' is not declared. It may be inaccessible due to its protection level.
    Private F2 As Object = P1
                           ~~
BC30112: 'System.Xml' is a namespace and cannot be used as an expression.
    Private F3 As Object = xml
                           ~~~
BC30112: 'System.Xml' is a namespace and cannot be used as an expression.
    Private F4 As Object = XML
                           ~~~
BC30114: 'xmlns' is an XML prefix and cannot be used as an expression.  Use the GetXmlNamespace operator to create a namespace object.
    Private F5 As Object = xmlns
                           ~~~~~
BC30451: 'XMLNS' is not declared. It may be inaccessible due to its protection level.
    Private F6 As Object = XMLNS
                           ~~~~~
BC30114: 'p2' is an XML prefix and cannot be used as an expression.  Use the GetXmlNamespace operator to create a namespace object.
                               <%= p2 %>
                                   ~~
BC30451: 'p3' is not declared. It may be inaccessible due to its protection level.
                               <%= p3 %>
                                   ~~
BC30114: 'xmlns' is an XML prefix and cannot be used as an expression.  Use the GetXmlNamespace operator to create a namespace object.
                               <%= xmlns %>
                                   ~~~~~
BC30114: 'p2' is an XML prefix and cannot be used as an expression.  Use the GetXmlNamespace operator to create a namespace object.
        Return p2
               ~~
BC30109: 'p4' is a class type and cannot be used as an expression.
    Private F10 As Object = p4
                            ~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC30131ERR_ModuleSecurityAttributeNotAllowed1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BC30131ERR_ModuleSecurityAttributeNotAllowed1">
    <file name="a.vb">
        <![CDATA[
Imports System
Imports System.Security.Permissions
Imports System.Security.Principal

<Module: MySecurity(Security.Permissions.SecurityAction.Assert)>

<AttributeUsage(AttributeTargets.Module)>
Class MySecurityAttribute
    Inherits SecurityAttribute

    Public Sub New(action As SecurityAction)
        MyBase.New(action)
    End Sub

    Public Overrides Function CreatePermission() As Security.IPermission
        Return Nothing
    End Function
End Class

Module Foo
    Public Sub main()

    End Sub
End Module
]]>
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected><![CDATA[
BC36979: Security attribute 'MySecurityAttribute' is not valid on this declaration type. Security attributes are only valid on assembly, type and method declarations.
<Module: MySecurity(Security.Permissions.SecurityAction.Assert)>
         ~~~~~~~~~~
]]>
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30132ERR_LabelNotDefined1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="LabelNotDefined1">
    <file name="a.vb">
        Module Implicitmod
            Sub Implicit()
                'COMPILEERROR: BC30132, "ns1"
                GoTo ns1
            End Sub
            Sub Test()
        ns1:
            End Sub
        End Module
        Namespace NS2
            Module Implicitmod
                Sub Implicit()
        ns1:
                End Sub
            End Module
        End Namespace
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30132: Label 'ns1' is not defined.
                GoTo ns1
                     ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30148ERR_RequiredNewCall2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="RequiredNewCall2">
    <file name="a.vb">
        Imports System
        Module Test
            Class clsTest0
                Sub New(ByVal strTest As String)
                End Sub
            End Class
            Class clsTest1
                Inherits clsTest0
                Private strTest As String = "Hello"
                Sub New(ByVal ArgX As String)
                    'COMPILEERROR: BC30148, "Console.WriteLine(ArgX)"
                    Console.WriteLine(ArgX)
                End Sub
            End Class
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30148: First statement of this 'Sub New' must be a call to 'MyBase.New' or 'MyClass.New' because base class 'Test.clsTest0' of 'Test.clsTest1' does not have an accessible 'Sub New' that can be called with no arguments.
                Sub New(ByVal ArgX As String)
                    ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30157ERR_BadWithRef()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports System        
Module M1
    Sub Main()
        .xxx = 3
    End Sub
End Module
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        .xxx = 3
        ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30157ERR_BadWithRef_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Default Property P(x As String)
        Get
            Return Nothing
        End Get
        Set(value)
        End Set
    End Property
    Sub M()
        !A = Me!B
        Me!A = !B
    End Sub
End Class
    </file>
</compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        !A = Me!B
        ~~
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        Me!A = !B
               ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30157ERR_BadWithRef_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Option Strict On
Module M
    Sub M()
        Dim o As Object
        o = .<x>
        o = ...<x>
        .@a = .@<a>
    End Sub
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        o = .<x>
            ~~~~
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        o = ...<x>
            ~~~~~~
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        .@a = .@<a>
        ~~~
BC30157: Leading '.' or '!' can only appear inside a 'With' statement.
        .@a = .@<a>
              ~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC30182_ERR_UnrecognizedType_2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="UnrecognizedType">
        <file name="a.vb">
        Namespace NS
            Class C1
                Sub FOO()
                    Dim v = 1
                    Dim s = CType(v, NS)
                End Sub
            End Class
        End Namespace
        </file>
    </compilation>)

            Dim expectedErrors1 = <errors>
BC30182: Type expected.
                    Dim s = CType(v, NS)
                                     ~~
</errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30203ERR_ExpectedIdentifier()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30203ERR_ExpectedIdentifier">
        <file name="a.vb">
        Option Strict On
        Class C1
            Public Property 
            Public Property _ as Integer
            Shared Public
        End Class
                </file>
    </compilation>)
            Dim expectedErrors1 =
<errors>
BC30203: Identifier expected.
            Public Property 
                            ~
BC30301: 'Public Property  As Object' and 'Public Property  As Integer' cannot overload each other because they differ only by return types.
            Public Property 
                            ~
BC30203: Identifier expected.
            Public Property _ as Integer
                            ~
BC30203: Identifier expected.
            Shared Public
                         ~
</errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30209ERR_StrictDisallowImplicitObject()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="StrictDisallowImplicitObject">
        <file name="a.vb">
        Option Strict On
        Structure myStruct
            Dim s
        End Structure
                </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
        BC30209: Option Strict On requires all variable declarations to have an 'As' clause.
            Dim s
                ~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30209ERR_StrictDisallowImplicitObject_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="StrictDisallowImplicitObject">
        <file name="a.vb">
        Option Strict On
        Structure myStruct
            Sub Scen1()
                'COMPILEERROR: BC30209, "i"
                Dim i
            End Sub
        End Structure
                </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
        BC30209: Option Strict On requires all variable declarations to have an 'As' clause.
                Dim i
                    ~
BC42024: Unused local variable: 'i'.
                Dim i
                    ~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact(), WorkItem(528749, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528749")>
        Public Sub BC30209ERR_StrictDisallowImplicitObject_3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="StrictDisallowImplicitObject">
        <file name="a.vb">
Option strict on
option infer off

imports system

        Class C1
            Public Const f1 = "foo"
            Public Const f2 As Object = "foo"
            Public Const f3 = 23
            Public Const f4 As Object = 42

            Public Shared Sub Main(args() As String)
                console.writeline(f1)
                console.writeline(f2)
                console.writeline(f3)
                console.writeline(f4)
            End Sub
        End Class
                </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30209: Option Strict On requires all variable declarations to have an 'As' clause.
            Public Const f1 = "foo"
                         ~~
BC30209: Option Strict On requires all variable declarations to have an 'As' clause.
            Public Const f3 = 23
                         ~~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30239ERR_ExpectedRelational_SelectCase()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="StrictDisallowImplicitObject">
        <file name="a.vb">
            <file name="a.vb"><![CDATA[
Imports System        
Module M1
    Sub Main()
        Select Case 0
            Case Is << 1
        End Select
    End Sub
End Module
        ]]></file>
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected><![CDATA[
BC30239: Relational operator expected.
            Case Is << 1
                    ~
]]></expected>)
        End Sub

        <Fact()>
        Public Sub BC30272ERR_NamedParamNotFound2()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NamedParamNotFound2">
    <file name="a.vb">
        Module Module1
            Class C0(Of T)
                Public whichOne As String
                Sub Foo(ByVal t1 As T)
                    whichOne = "T"
                End Sub
            End Class
            Class C1(Of T, Y)
                Inherits C0(Of T)
                Overloads Sub Foo(ByVal y1 As Y)
                    whichOne = "Y"
                End Sub
            End Class
            Sub GenUnif0060()
                Dim tc1 As New C1(Of Integer, Integer)
                ' BC30272: 't1' is not a parameter of 'Public Overloads Sub Foo(y1 As Y)'.
                Call tc1.Foo(t1:=1000)
            End Sub
        End Module
    </file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_NamedParamNotFound2, "t1").WithArguments("t1", "Public Overloads Sub Foo(y1 As Integer)"),
    Diagnostic(ERRID.ERR_OmittedArgument2, "Foo").WithArguments("y1", "Public Overloads Sub Foo(y1 As Integer)"))

        End Sub

        <Fact()>
        Public Sub BC30274ERR_NamedArgUsedTwice2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NamedArgUsedTwice2">
    <file name="a.vb">
        Module Module1
            Class C0
                Public whichOne As String
                Sub Foo(ByVal t1 As String)
                    whichOne = "T"
                End Sub
            End Class
            Class C1
                Inherits C0
                Overloads Sub Foo(ByVal y1 As String)
                    whichOne = "Y"
                End Sub
            End Class
            Sub test()
                Dim [ident1] As C0 = New C0()
                Dim clsNarg2get As C1 = New C1()
                Dim str1 As String = "Visual Basic"
                'COMPILEERROR: BC30274, "y"
                [ident1].Foo(1, t1:=2) = str1
                'COMPILEERROR: BC30274, "x"
                [ident1].Foo(t1:=1, t1:=1) = str1
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30274: Parameter 't1' of 'Public Sub Foo(t1 As String)' already has a matching argument.
                [ident1].Foo(1, t1:=2) = str1
                                ~~
BC30274: Parameter 't1' of 'Public Sub Foo(t1 As String)' already has a matching argument.
                [ident1].Foo(t1:=1, t1:=1) = str1
                                    ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30277ERR_TypecharNoMatch2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="TypecharNoMatch2">
    <file name="a.vb">
        Imports Microsoft.VisualBasic.Information
        Namespace NS30277
            Public Class genClass1(Of CT)
                Public Function genFun7(Of T)(ByVal x As T) As T()
                    Dim t1(2) As T
                    Return t1
                End Function
            End Class
            Module MD30277
                Sub GenMethod9102()
                    Const uiConst As UInteger = 1000
                    Dim o As New genClass1(Of Object)
                    Dim objTmp As Object = CShort(10)
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(1&amp;)
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(True)
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%((True And False))
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(CDbl(1))
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(Fun1)
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(TypeName(o))
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(uiConst)
                    ' BC30277: type character does not match declared data type.
                    o.genFun7%(objTmp)
                End Sub
                Function Fun1() As Byte
                    Return 1
                End Function
            End Module
        End Namespace
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30277: Type character '%' does not match declared data type 'Long'.
                    o.genFun7%(1&amp;)
                    ~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'Boolean'.
                    o.genFun7%(True)
                    ~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'Boolean'.
                    o.genFun7%((True And False))
                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'Double'.
                    o.genFun7%(CDbl(1))
                    ~~~~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'Byte'.
                    o.genFun7%(Fun1)
                    ~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'String'.
                    o.genFun7%(TypeName(o))
                    ~~~~~~~~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'UInteger'.
                    o.genFun7%(uiConst)
                    ~~~~~~~~~~~~~~~~~~~
BC30277: Type character '%' does not match declared data type 'Object'.
                    o.genFun7%(objTmp)
                    ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30277ERR_TypecharNoMatch2_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="TypecharNoMatch2">
    <file name="a.vb">
Class C
    Public Shared Sub Main()
        'declare with explicit type, use in next with a type char")
        For Each x As Integer In New Integer() {1, 1, 1}
            'COMPILEERROR: BC30277, "x#"
        Next x#
        For Each [me] As Integer In New Integer() {1, 1, 1}
        Next me%
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30277: Type character '#' does not match declared data type 'Integer'.
        Next x#
             ~~
</expected>)
        End Sub

        <WorkItem(528681, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528681")>
        <Fact()>
        Public Sub BC30277ERR_TypecharNoMatch2_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="TypecharNoMatch2">
    <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        For ivar% As Long = 1 To 10
        Next
        For dvar# As Single = 1 To 10
        Next
        For cvar@ As Decimal = 1 To 10
        Next
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30302: Type character '%' cannot be used in a declaration with an explicit type.
        For ivar% As Long = 1 To 10
            ~~~~~
BC30302: Type character '#' cannot be used in a declaration with an explicit type.
        For dvar# As Single = 1 To 10
            ~~~~~
BC30302: Type character '@' cannot be used in a declaration with an explicit type.
        For cvar@ As Decimal = 1 To 10
            ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_InvalidConstructorCall1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="InvalidConstructorCall1">
    <file name="a.vb">
        Module Error30282
            Class Class1
                Sub New()
                End Sub
            End Class
            Class Class2
                Inherits Class1
                Sub New()
                    'COMPILEERROR: BC30282, "Class1.New"
                    Class1.New()
                End Sub
            End Class
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
                    Class1.New()
                    ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_InvalidConstructorCall2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="InvalidConstructorCall2">
    <file name="a.vb">
Imports System
Class C
    Sub New(x As Integer)
        Me.New(Of Integer)
    End Sub

    Sub New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New(Of Integer)
        ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_InvalidConstructorCall3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="InvalidConstructorCall3">
    <file name="a.vb">
Imports System
Class C
    Sub New(x As Integer)
        Me.New
        Me.New(Of Integer)
    End Sub

    Sub New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New(Of Integer)
        ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_InvalidConstructorCall4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="InvalidConstructorCall4">
    <file name="a.vb">
Imports System
Class C
    Sub New(x As Integer)
        Me.New(Of Integer)(1.ToString(123, 2, 3, 4))
    End Sub

    Sub New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New(Of Integer)(1.ToString(123, 2, 3, 4))
        ~~~~~~~~~~~~~~~~~~
BC30516: Overload resolution failed because no accessible 'ToString' accepts this number of arguments.
        Me.New(Of Integer)(1.ToString(123, 2, 3, 4))
                             ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_InvalidConstructorCall5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="InvalidConstructorCall5">
    <file name="a.vb">
Imports System
Class C
    Sub New(x As Integer)
        Dim a = 1 + Me.New(Of Integer)
    End Sub

    Sub New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Dim a = 1 + Me.New(Of Integer)
                    ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_1">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
        Me.New()
    End Sub
    Public Sub New(t As Tests)
        t.New(1)
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        t.New(1)
        ~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_2">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
        Me.New()
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New()
        ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_3">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
l1:
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New()
        ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_4">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New2()
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New()
        ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_5">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
#Const a = 1
#If a = 1 Then
        Me.New()
#End If
        Me.New()
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Me.New()
        ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_6">
    <file name="a.vb">
Class Tests
    Public Sub New()
    End Sub
    Public Sub New(i As Integer)
        Dim a As Integer = 1 + Me.New() + Me.New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Dim a As Integer = 1 + Me.New() + Me.New
                               ~~~~~~
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Dim a As Integer = 1 + Me.New() + Me.New
                                          ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_7">
    <file name="a.vb">
Class Tests
    Public Sub New(i As String)
    End Sub
    Public Sub New(i As Integer)
        Dim a As Integer = 1 + Me.New(1, 2) + Me.New
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Dim a As Integer = 1 + Me.New(1, 2) + Me.New
                               ~~~~~~
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Dim a As Integer = 1 + Me.New(1, 2) + Me.New
                                              ~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_8()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30282ERR_ConstructorCallIsValidOnlyAsTheFirstStatementInAnInstanceConstructor_8">
    <file name="a.vb">
Class Tests
    Public Sub New(i As String)
    End Sub
    Public Sub New(i As Integer)
        Tests.New(1, 2)
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30282: Constructor call is valid only as the first statement in an instance constructor.
        Tests.New(1, 2)
        ~~~~~~~~~
</errors>)
        End Sub

        <WorkItem(541012, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541012")>
        <Fact()>
        Public Sub BC30283ERR_CantOverrideConstructor()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="CantOverrideConstructor">
    <file name="a.vb">
        Module Error30283
            Class Class1
                mustoverride Sub New()
                End Sub
            End Class
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
                                               <expected>
BC30364: 'Sub New' cannot be declared 'mustoverride'.
                mustoverride Sub New()
                ~~~~~~~~~~~~
BC30429: 'End Sub' must be preceded by a matching 'Sub'.
                End Sub
                ~~~~~~~                                                   
                                               </expected>)
        End Sub

        <Fact()>
        Public Sub BC30288ERR_DuplicateLocals1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DuplicateLocals1">
        <file name="a.vb">
        Public Class Class1
            Public Sub foo(ByVal val As Short)
                Dim i As Integer
                Dim i As String
            End Sub
        End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC42024: Unused local variable: 'i'.
                Dim i As Integer
                    ~
BC30288: Local variable 'i' is already declared in the current block.
                Dim i As String
                    ~
BC42024: Unused local variable: 'i'.
                Dim i As String
                    ~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact(), WorkItem(531346, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/531346")>
        Public Sub UnicodeCaseInsensitiveLocals()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="UnicodeCaseInsensitiveLocals">
        <file name="a.vb">
        Public Class Class1
            Public Sub foo()
                Dim X&#x130;
                'COMPILEERROR:BC30288, "xi"
                Dim xi

                Dim &#x130;
                'COMPILEERROR:BC30288, "i"
                Dim i
            End Sub
        End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC42024: Unused local variable: 'X&#x130;'.
                Dim X&#x130;
                    ~~
BC30288: Local variable 'xi' is already declared in the current block.
                Dim xi
                    ~~
BC42024: Unused local variable: 'xi'.
                Dim xi
                    ~~
BC42024: Unused local variable: '&#x130;'.
                Dim &#x130;
                    ~
BC30288: Local variable 'i' is already declared in the current block.
                Dim i
                    ~
BC42024: Unused local variable: 'i'.
                Dim i
                    ~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30290ERR_LocalSameAsFunc()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="LocalSameAsFunc">
    <file name="a.vb">
        Module Error30290
            Class Class1
                Function Foo(ByVal Name As String)
                    'COMPILEERROR : BC30290, "Foo" 
                    Dim Foo As Date
                    Return Name
                End Function
            End Class
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30290: Local variable cannot have the same name as the function containing it.
                    Dim Foo As Date
                        ~~~
BC42024: Unused local variable: 'Foo'.
                    Dim Foo As Date
                        ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30290ERR_LocalSameAsFunc_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="LocalSameAsFunc">
    <file name="a.vb">
Class C
    Shared Sub Main()
    End Sub
    Function foo() as Object
        'COMPILEERROR: BC30290, 
        For Each foo As Integer In New Integer() {1, 2, 3}
        Next
        return nothing
    End Function
    Sub foo1()
        For Each foo1 As Integer In New Integer() {1, 2, 3}
        Next
    End SUB
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30290: Local variable cannot have the same name as the function containing it.
        For Each foo As Integer In New Integer() {1, 2, 3}
                 ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30297ERR_ConstructorCannotCallItself_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30297ERR_ConstructorCannotCallItself_1">
    <file name="a.vb">
Imports System
Class Tests
    Public Sub New(i As Integer)
        Me.New("")
    End Sub
    Public Sub New(i As String)
        Me.New(1)
    End Sub
    Public Sub New(i As DateTime)
        Me.New(1)
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30298: Constructor 'Public Sub New(i As Integer)' cannot call itself: 
    'Public Sub New(i As Integer)' calls 'Public Sub New(i As String)'.
    'Public Sub New(i As String)' calls 'Public Sub New(i As Integer)'.
    Public Sub New(i As Integer)
               ~~~
BC30298: Constructor 'Public Sub New(i As String)' cannot call itself: 
    'Public Sub New(i As String)' calls 'Public Sub New(i As Integer)'.
    'Public Sub New(i As Integer)' calls 'Public Sub New(i As String)'.
    Public Sub New(i As String)
               ~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30297ERR_ConstructorCannotCallItself_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30297ERR_ConstructorCannotCallItself_2">
    <file name="a.vb">
Imports System
Class Tests
    Public Sub New(i As Byte)
        Me.New(CType(i, Int16))
    End Sub
    Public Sub New(i As Int16)
        Me.New(DateTime.Now)
    End Sub
    Public Sub New(i As DateTime)
        Me.New(DateTime.Now)
    End Sub
    Public Sub New(i As Int64)
        Me.New("")
    End Sub
    Public Sub New(i As Int32)
        Me.New(ctype(1, UInt32))
    End Sub
    Public Sub New(i As UInt32)
        Me.New("")
    End Sub
    Public Sub New(i As String)
        Me.New(cint(1))
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30298: Constructor 'Public Sub New(i As Date)' cannot call itself: 
    'Public Sub New(i As Date)' calls 'Public Sub New(i As Date)'.
    Public Sub New(i As DateTime)
               ~~~
BC30298: Constructor 'Public Sub New(i As Integer)' cannot call itself: 
    'Public Sub New(i As Integer)' calls 'Public Sub New(i As UInteger)'.
    'Public Sub New(i As UInteger)' calls 'Public Sub New(i As String)'.
    'Public Sub New(i As String)' calls 'Public Sub New(i As Integer)'.
    Public Sub New(i As Int32)
               ~~~
BC30298: Constructor 'Public Sub New(i As UInteger)' cannot call itself: 
    'Public Sub New(i As UInteger)' calls 'Public Sub New(i As String)'.
    'Public Sub New(i As String)' calls 'Public Sub New(i As Integer)'.
    'Public Sub New(i As Integer)' calls 'Public Sub New(i As UInteger)'.
    Public Sub New(i As UInt32)
               ~~~
BC30298: Constructor 'Public Sub New(i As String)' cannot call itself: 
    'Public Sub New(i As String)' calls 'Public Sub New(i As Integer)'.
    'Public Sub New(i As Integer)' calls 'Public Sub New(i As UInteger)'.
    'Public Sub New(i As UInteger)' calls 'Public Sub New(i As String)'.
    Public Sub New(i As String)
               ~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30297ERR_ConstructorCannotCallItself_3()
            ' NOTE: Test case ensures that the error in calling the 
            '       constructor suppresses the cycle detection
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC30297ERR_ConstructorCannotCallItself_3">
    <file name="a.vb">
Imports System
Class Tests
    Public Sub New(i As Integer)
        Me.New("")
    End Sub
    Public Sub New(i As String)
        Me.New(qqq)
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<errors>
BC30451: 'qqq' is not declared. It may be inaccessible due to its protection level.
        Me.New(qqq)
               ~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC30306ERR_MissingSubscript()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="MissingSubscript">
    <file name="a.vb">
        Module Mod30306
            Sub ArExtFrErr002()
                Dim scen1(,) As Integer
                ReDim scen1(2, 2)
                Dim scen4() As Integer
                'COMPILEERROR: BC30306, "(", BC30306, ","
                ReDim scen4(, )
                Dim scen8a(,,) As Integer
                'COMPILEERROR: BC30306, "(", BC30306, ",", BC30306, ","
                ReDim scen8a(, , )
                Dim Scen8b(,,) As Integer
                'COMPILEERROR: BC30306, ",", BC30306, ","
                ReDim Scen8b(5, , )
                Dim scen8c(,,) As Integer
                'COMPILEERROR: BC30306, "(", BC30306, ","
                ReDim scen8c(, , 5)
                Exit Sub
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30306: Array subscript expression missing.
                ReDim scen4(, )
                            ~
BC30306: Array subscript expression missing.
                ReDim scen4(, )
                              ~
BC30306: Array subscript expression missing.
                ReDim scen8a(, , )
                             ~
BC30306: Array subscript expression missing.
                ReDim scen8a(, , )
                               ~
BC30306: Array subscript expression missing.
                ReDim scen8a(, , )
                                 ~
BC30306: Array subscript expression missing.
                ReDim Scen8b(5, , )
                                ~
BC30306: Array subscript expression missing.
                ReDim Scen8b(5, , )
                                  ~
BC30306: Array subscript expression missing.
                ReDim scen8c(, , 5)
                             ~
BC30306: Array subscript expression missing.
                ReDim scen8c(, , 5)
                               ~
</expected>)
        End Sub

        '        <Fact()>
        '        Public Sub BC30310ERR_FieldOfValueFieldOfMarshalByRef3()
        '            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib(
        '<compilation name="FieldOfValueFieldOfMarshalByRef3">
        '    <file name="a.vb">

        '    </file>
        '</compilation>)
        '            CompilationUtils.AssertTheseErrors(compilation,
        '<expected>
        'BC30310: Local variable cannot have the same name as the function containing it.
        '           Dim Foo As Date
        '            ~~~~
        'BC30310: Local variable cannot have the same name as the function containing it.
        '           Dim scen16 = 3
        '            ~~~~
        '</expected>)
        '        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Imports System
        Public Class C
            Sub FOO()
                Dim a As S1?
                Dim b As E1?
                Dim c As System.Exception
                Dim d As I1
                Dim e As C1.C2
                Dim f As C1.C2(,)
                Dim z = DirectCast (b, S1?)
                z = DirectCast (b, System.Nullable)
                z = DirectCast (a, E1?)
                z = DirectCast (a, System.Nullable)
                z = DirectCast (c, S1?)
                z = DirectCast (c, E1?)
                z = DirectCast (c, System.Nullable)
                z = DirectCast (d, S1?)
                z = DirectCast (d, E1?)
                z = DirectCast (d, System.Nullable)
                z = DirectCast (d, System.ValueType)
                z = DirectCast (e, S1?)
                z = DirectCast (e, E1?)
                z = DirectCast (e, System.Nullable)
                z = DirectCast (e, System.ValueType)
                z = DirectCast (f, S1?)
                z = DirectCast (f, E1?)
                z = DirectCast (f, System.Nullable)
                z = DirectCast (f, System.ValueType)
            End Sub
        End Class
        Structure S1
        End Structure
        Enum E1
            one
        End Enum
        Interface I1
        End Interface
        Class C1
            Public Class C2
            End Class
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'E1?' cannot be converted to 'S1?'.
                Dim z = DirectCast (b, S1?)
                                    ~
BC30311: Value of type 'E1?' cannot be converted to 'Nullable'.
                z = DirectCast (b, System.Nullable)
                                ~
BC30311: Value of type 'S1?' cannot be converted to 'E1?'.
                z = DirectCast (a, E1?)
                                ~
BC30311: Value of type 'S1?' cannot be converted to 'Nullable'.
                z = DirectCast (a, System.Nullable)
                                ~
BC30311: Value of type 'Exception' cannot be converted to 'S1?'.
                z = DirectCast (c, S1?)
                                ~
BC42104: Variable 'c' is used before it has been assigned a value. A null reference exception could result at runtime.
                z = DirectCast (c, S1?)
                                ~
BC30311: Value of type 'Exception' cannot be converted to 'E1?'.
                z = DirectCast (c, E1?)
                                ~
BC30311: Value of type 'Exception' cannot be converted to 'Nullable'.
                z = DirectCast (c, System.Nullable)
                                ~
BC30311: Value of type 'I1' cannot be converted to 'S1?'.
                z = DirectCast (d, S1?)
                                ~
BC42104: Variable 'd' is used before it has been assigned a value. A null reference exception could result at runtime.
                z = DirectCast (d, S1?)
                                ~
BC30311: Value of type 'I1' cannot be converted to 'E1?'.
                z = DirectCast (d, E1?)
                                ~
BC30311: Value of type 'Nullable' cannot be converted to 'S1?'.
                z = DirectCast (d, System.Nullable)
                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC42322: Runtime errors might occur when converting 'I1' to 'Nullable'.
                z = DirectCast (d, System.Nullable)
                                ~
BC30311: Value of type 'C1.C2' cannot be converted to 'S1?'.
                z = DirectCast (e, S1?)
                                ~
BC42104: Variable 'e' is used before it has been assigned a value. A null reference exception could result at runtime.
                z = DirectCast (e, S1?)
                                ~
BC30311: Value of type 'C1.C2' cannot be converted to 'E1?'.
                z = DirectCast (e, E1?)
                                ~
BC30311: Value of type 'C1.C2' cannot be converted to 'Nullable'.
                z = DirectCast (e, System.Nullable)
                                ~
BC30311: Value of type 'C1.C2' cannot be converted to 'ValueType'.
                z = DirectCast (e, System.ValueType)
                                ~
BC30311: Value of type 'C1.C2(*,*)' cannot be converted to 'S1?'.
                z = DirectCast (f, S1?)
                                ~
BC42104: Variable 'f' is used before it has been assigned a value. A null reference exception could result at runtime.
                z = DirectCast (f, S1?)
                                ~
BC30311: Value of type 'C1.C2(*,*)' cannot be converted to 'E1?'.
                z = DirectCast (f, E1?)
                                ~
BC30311: Value of type 'C1.C2(*,*)' cannot be converted to 'Nullable'.
                z = DirectCast (f, System.Nullable)
                                ~
BC30311: Value of type 'C1.C2(*,*)' cannot be converted to 'ValueType'.
                z = DirectCast (f, System.ValueType)
                                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Imports System

Class C
    Shared Sub Main()
        For Each x As Integer In New Exception() {Nothing, Nothing}
        Next
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Exception' cannot be converted to 'Integer'.
        For Each x As Integer In New Exception() {Nothing, Nothing}
                                 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Shared Sub Main()
        Dim numbers2D As Integer()() = New Integer()() {New Integer() {1, 2}, New Integer() {1, 2}}
        For Each i As Integer In numbers2D
            System.Console.Write("{0} ", i)
        Next
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Integer()' cannot be converted to 'Integer'.
        For Each i As Integer In numbers2D
                                 ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports System
Module Program
    Sub Main(args As String())
    End Sub
    Private Function fun(Of T)(Parm1 As T) As T
        Dim temp As T
        Return If(temp, temp, 1)
    End Function
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'T' cannot be converted to 'Boolean'.
        Return If(temp, temp, 1)
                  ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_4()
            CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Public Class Test

    Public Sub Test()
        Dim at1 = New With {.f1 = Nothing, .f2 = String.Empty}
        Dim at2 = New With {.f2 = String.Empty, .f1 = Nothing}
        at1 = at2
    End Sub

End Class
    </file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_TypeMismatch2, "at2").WithArguments("<anonymous type: f2 As String, f1 As Object>", "<anonymous type: f1 As Object, f2 As String>"))

        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Imports System
Module Program
    Public Sub Main()
        Dim arr1 As Integer(,) = New Integer(2, 1) {{1, 2}, {3, 4}, {5, 6}}
        arr1 = 0        ' Invalid
    End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Integer' cannot be converted to 'Integer(*,*)'.
        arr1 = 0        ' Invalid
               ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module Program
    Public Sub Main()
        Dim arr As Integer(,) = New Integer(2, 1) {{6, 7}, {5, 8}, {8, 10}}
        Dim x As Integer
        x = arr        'Invalid
        arr = x        'Invalid
    End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Integer(*,*)' cannot be converted to 'Integer'.
        x = arr        'Invalid
            ~~~
BC30311: Value of type 'Integer' cannot be converted to 'Integer(*,*)'.
        arr = x        'Invalid
              ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30311ERR_TypeMismatch2_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict Off
Option Infer On
Class B
    Public Shared Sub Main()
    End Sub
    Private Sub Foo(Of T)()
        Dim x As T(,) = New T(1, 2) {}
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
    End Sub
End Class

Public Class Class1(Of T)
    Private x As T(,) = New T(1, 2) {}
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
    Private Sub Foo()
        Dim x As T(,) = New T(1, 2) {}
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                      ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                         ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                            ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                 ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                    ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                       ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                      ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                         ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                            ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                                 ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                                    ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
    Private Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}    ' invalid
                                                       ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                      ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                         ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                            ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                 ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                    ~
BC30311: Value of type 'Integer' cannot be converted to 'T'.
        Dim Y As T(,) = New T(1, 2) {{1, 2, 3}, {1, 2, 3}}        ' invalid
                                                       ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30332ERR_ConvertArrayMismatch4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConvertArrayMismatch4">
    <file name="a.vb">
        Module M1
            Class Cls2_1
            End Class
            Class Cls2_2
            End Class
            Sub Main()
                Dim Ary1_1() As Integer
                Dim Ary1_2() As Long = New Long() {1, 2}
                'COMPILEERROR: BC30332, "Ary1_2"
                Ary1_1 = CType(Ary1_2, Integer())
                Dim Ary2_1() As Cls2_1 = New Cls2_1() {}
                Dim Ary2_2(,) As Cls2_2
                'COMPILEERROR: BC30332, "Ary2_1"
                Ary2_2 = CType(Ary2_1, Cls2_2())
                Dim Ary3_1(,) As Double = New Double(,) {}
                Dim Ary3_2(,) As Cls2_2
                'COMPILEERROR: BC30332, "Ary3_1"
                Ary3_2 = CType(Ary3_1, Cls2_2(,))
                'COMPILEERROR: BC30332, "Ary3_2"
                Ary3_1 = CType(Ary3_2, Double(,))
                Dim Ary4_1() As Integer
                Dim Ary4_2() As Object = New Object() {}
                Ary4_1 = CType(Ary4_2, Integer())
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30332: Value of type 'Long()' cannot be converted to 'Integer()' because 'Long' is not derived from 'Integer'.
                Ary1_1 = CType(Ary1_2, Integer())
                               ~~~~~~
BC30332: Value of type 'M1.Cls2_1()' cannot be converted to 'M1.Cls2_2()' because 'M1.Cls2_1' is not derived from 'M1.Cls2_2'.
                Ary2_2 = CType(Ary2_1, Cls2_2())
                               ~~~~~~
BC30332: Value of type 'Double(*,*)' cannot be converted to 'M1.Cls2_2(*,*)' because 'Double' is not derived from 'M1.Cls2_2'.
                Ary3_2 = CType(Ary3_1, Cls2_2(,))
                               ~~~~~~
BC30332: Value of type 'M1.Cls2_2(*,*)' cannot be converted to 'Double(*,*)' because 'M1.Cls2_2' is not derived from 'Double'.
                Ary3_1 = CType(Ary3_2, Double(,))
                               ~~~~~~
BC30332: Value of type 'Object()' cannot be converted to 'Integer()' because 'Object' is not derived from 'Integer'.
                Ary4_1 = CType(Ary4_2, Integer())
                               ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30332ERR_ConvertArrayMismatch4_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConvertArrayMismatch4">
    <file name="a.vb">
        Option Strict On
Imports System
Module Module1
    Sub Main()
        Dim arrString$(,) = New Decimal(1, 2) {} ' Invalid
        Dim arrInteger%(,) = New Decimal(1, 2) {} ' Invalid
        Dim arrLong&amp;(,) = New Decimal(1, 2) {} ' Invalid
        Dim arrSingle!(,) = New Decimal(1, 2) {} ' Invalid
    End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30332: Value of type 'Decimal(*,*)' cannot be converted to 'String(*,*)' because 'Decimal' is not derived from 'String'.
        Dim arrString$(,) = New Decimal(1, 2) {} ' Invalid
                            ~~~~~~~~~~~~~~~~~~~~
BC30332: Value of type 'Decimal(*,*)' cannot be converted to 'Integer(*,*)' because 'Decimal' is not derived from 'Integer'.
        Dim arrInteger%(,) = New Decimal(1, 2) {} ' Invalid
                             ~~~~~~~~~~~~~~~~~~~~
BC30332: Value of type 'Decimal(*,*)' cannot be converted to 'Long(*,*)' because 'Decimal' is not derived from 'Long'.
        Dim arrLong&amp;(,) = New Decimal(1, 2) {} ' Invalid
                          ~~~~~~~~~~~~~~~~~~~~
BC30332: Value of type 'Decimal(*,*)' cannot be converted to 'Single(*,*)' because 'Decimal' is not derived from 'Single'.
        Dim arrSingle!(,) = New Decimal(1, 2) {} ' Invalid
                            ~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30333ERR_ConvertObjectArrayMismatch3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ConvertObjectArrayMismatch3">
    <file name="a.vb">
       Module M1
            Sub Main()
                Dim Ary1() As Integer = New Integer() {}
                Dim Ary2() As Object
                Ary2 = CType(Ary1, Object())
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30333: Value of type 'Integer()' cannot be converted to 'Object()' because 'Integer' is not a reference type.
                Ary2 = CType(Ary1, Object())
                             ~~~~
</expected>)
        End Sub

        <WorkItem(579764, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/579764")>
        <Fact()>
        Public Sub BC30311ERR_WithArray_ParseAndDeclarationErrors()
            'This test is because previously in native command line compiler we would produce errors for both parsing and binding errors,  now 
            ' we won't produce the binding if parsing was not successful from the command line.  However, the diagnostics will display both messages and 
            ' hence the need for two tests to verify this behavior.

            Dim source =
<compilation>
    <file name="ParseErrorOnly.vb">
Module M
	Dim x As Integer() {1, 2, 3}
    	Dim y = CType({1, 2, 3}, System.Collections.Generic.List(Of Integer))

	Sub main
	End Sub
End Module        </file>
</compilation>
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source).VerifyDiagnostics(Diagnostic(ERRID.ERR_ExpectedEOS, "{"),
                                                                                                 Diagnostic(ERRID.ERR_TypeMismatch2, "{1, 2, 3}").WithArguments("Integer()", "System.Collections.Generic.List(Of Integer)"))


            ' This 2nd scenario will produce 1 error because it passed the parsing stage and now 
            ' fails in the binding
            source =
<compilation>
    <file name="ParseOK.vb">
Module M
	Dim x As Integer() = {1, 2, 3}
    	Dim y = CType({1, 2, 3}, System.Collections.Generic.List(Of Integer))

	Sub main
	End Sub
End Module        </file>
</compilation>
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source).VerifyDiagnostics(Diagnostic(ERRID.ERR_TypeMismatch2, "{1, 2, 3}").WithArguments("Integer()", "System.Collections.Generic.List(Of Integer)"))
        End Sub


        <Fact(), WorkItem(542069, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542069")>
        Public Sub BC30337ERR_ForLoopType1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ForLoopType1">
    <file name="a.vb">
       Module M1
            Sub Test()
                'COMPILEERROR : BC30337, "i" 
                For i = New base To New first()
                Next

                For j = New base To New first() step new second()
                Next
            End Sub
        End Module
        Class base
        End Class
        Class first
            Inherits base
            Overloads Shared Widening Operator CType(ByVal d As first) As second
                Return New second()
            End Operator
        End Class
        Class second
            Inherits base
            Overloads Shared Widening Operator CType(ByVal d As second) As first
                Return New first()
            End Operator
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30311: Value of type 'Integer' cannot be converted to 'base'.
                For i = New base To New first()
                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC33038: Type 'base' must define operator '+' to be used in a 'For' statement.
                For j = New base To New first() step new second()
                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC33038: Type 'base' must define operator '-' to be used in a 'For' statement.
                For j = New base To New first() step new second()
                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC33038: Type 'base' must define operator '&lt;=' to be used in a 'For' statement.
                For j = New base To New first() step new second()
                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC33038: Type 'base' must define operator '>=' to be used in a 'For' statement.
                For j = New base To New first() step new second()
                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact(), WorkItem(542069, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542069")>
        Public Sub BC30337ERR_ForLoopType1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ForLoopType1">
    <file name="a.vb">
Option Strict On
Option Infer Off
Module Program
    Sub Main(args As String())
        For x As Date = #1/2/0003# To 10
        Next
    End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30337: 'For' loop control variable cannot be of type 'Date' because the type does not support the required operators.
        For x As Date = #1/2/0003# To 10
            ~~~~~~~~~
</expected>)
        End Sub

        <Fact(), WorkItem(542069, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542069"), WorkItem(544464, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/544464")>
        Public Sub BC30337ERR_ForLoopType1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ForLoopType1">
    <file name="a.vb">
Option Strict On
Option Infer Off

Interface IFoo
End Interface

Module Program
    Sub Main(args As String())
        For x As Boolean = False To True
        Next

        Dim foo as Boolean
        For foo = False To True
        Next

        for z as IFoo = nothing to nothing
        next        
    End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30337: 'For' loop control variable cannot be of type 'Boolean' because the type does not support the required operators.
        For x As Boolean = False To True
            ~~~~~~~~~~~~
BC30337: 'For' loop control variable cannot be of type 'Boolean' because the type does not support the required operators.
        For foo = False To True
            ~~~
BC30337: 'For' loop control variable cannot be of type 'IFoo' because the type does not support the required operators.
        for z as IFoo = nothing to nothing
            ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30367ERR_NoDefaultNotExtend1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Class C
            Shared Sub M(Of T)(x As C, y As T)
                N(x(0))
                N(x!P)
                x(0)
                N(y(1))
                N(y!Q)
                y(1)
            End Sub
            Shared Sub N(o As Object)
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30367: Class 'C' cannot be indexed because it has no default property.
                N(x(0))
                  ~
BC30367: Class 'C' cannot be indexed because it has no default property.
                N(x!P)
                  ~
BC30454: Expression is not a method.
                x(0)
                ~
BC30547: 'T' cannot be indexed because it has no default property.
                N(y(1))
                  ~
BC30547: 'T' cannot be indexed because it has no default property.
                N(y!Q)
                  ~
BC30454: Expression is not a method.
                y(1)
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
       Structure S1
            Dim b3 As Integer()
            Public Shared Sub Scenario_6()
                dim b4 = b3
            End Sub
            shared Function foo() As Integer()
                Return b3
            End Function
        End Structure
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                dim b4 = b3
                         ~~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                Return b3
                       ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Class C
            Property P
                Get
                    Return Nothing
                End Get
                Set
                End Set
            End Property
            Shared Sub M()
                Dim o = P
                P = o
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                Dim o = P
                        ~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                P = o
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Class C
            Shared Sub Main()
                For Each x As Integer In F(x)
                Next
            End Sub
            Private Sub F(x As Integer)
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                For Each x As Integer In F(x)
                                         ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Class C
            Shared Sub Main()
                For Each x As Integer In F(x)
                Next
            End Sub
            Private Function F(x As Integer) As Object
                Return New Object()
            End Function
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
                For Each x As Integer In F(x)
                                         ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Property P1(ByVal x As Integer) As integer
        Get
            Return x +5
        End Get
        Set(ByVal Value As integer)
        End Set
    End Property
    Public Shared Sub Main()
        For Each x As integer In New integer() {P1(x), P1(x), P1(x)}
        Next
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As integer In New integer() {P1(x), P1(x), P1(x)}
                                                ~~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As integer In New integer() {P1(x), P1(x), P1(x)}
                                                       ~~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As integer In New integer() {P1(x), P1(x), P1(x)}
                                                              ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Public Shared Sub Main()
        For Each x As Integer In New Integer() {foo(x), foo(x), foo(x)}
        Next
    End Sub
    Function foo(ByRef x As Integer) As Integer
        x = 10
        Return x + 10
    End Function
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As Integer In New Integer() {foo(x), foo(x), foo(x)}
                                                ~~~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As Integer In New Integer() {foo(x), foo(x), foo(x)}
                                                        ~~~
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For Each x As Integer In New Integer() {foo(x), foo(x), foo(x)}
                                                                ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30369ERR_BadInstanceMemberAccess_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Option Strict Off
Option Infer On
Public Class MyClass1
    Dim global_x As Integer = 10
    Const global_y As Long = 20
    Public Shared Sub Main()
        For global_x = global_y To 10
        Next
    End Sub
End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30369: Cannot refer to an instance member of a class from within a shared method or shared member initializer without an explicit instance of the class.
        For global_x = global_y To 10
            ~~~~~~~~
</expected>)
        End Sub

        <Fact, WorkItem(529193, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/529193")>
        Public Sub BC30369ERR_BadInstanceMemberAccess_8()
            CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class derive
    Shared Sub main()
        TestEvents
    End Sub
    Shared Sub TestEvents()
        Dim Obj As New Class1
        RemoveHandler Obj.MyEvent, AddressOf EventHandler
    End Sub
    Function EventHandler()
        Return Nothing
    End Function
    Public Class Class1
        Public Event MyEvent(ByRef x As Decimal)
        Sub CauseSomeEvent()
            RaiseEvent MyEvent(x:=1)
        End Sub
    End Class
End Class
    </file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_BadInstanceMemberAccess, "AddressOf EventHandler"))
        End Sub

        <Fact()>
        Public Sub BC30375ERR_NewIfNullOnNonClass()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NewIfNullOnNonClass">
    <file name="a.vb">
       Module M1
            Sub Foo()
                Dim interf1 As New Interface1()
                Dim interf2 = New Interface1()
            End Sub
        End Module
        Interface Interface1
        End Interface
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30375: 'New' cannot be used on an interface.
                Dim interf1 As New Interface1()
                               ~~~~~~~~~~~~~~~~
BC30375: 'New' cannot be used on an interface.
                Dim interf2 = New Interface1()
                              ~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        ''' We decided to not implement this for Roslyn as BC30569 and BC31411 cover the scenarios that BC30376 addresses.
        <Fact()>
        Public Sub BC30376ERR_NewIfNullOnAbstractClass1()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NewIfNullOnAbstractClass1">
    <file name="a.vb">
        Module M1
            Sub Foo()
                Throw (New C1)
            End Sub
        End Module
        MustInherit Class C1
            MustOverride Sub foo()
        End Class
    </file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_NewOnAbstractClass, "New C1"),
    Diagnostic(ERRID.ERR_CantThrowNonException, "Throw (New C1)").WithArguments("C1"))

        End Sub

        <Fact(), WorkItem(999399, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/999399")>
        Public Sub BC30387ERR_NoConstructorOnBase2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NoConstructorOnBase2">
    <file name="a.vb">
        Module M1
            Class Base
                Sub New(ByVal x As Integer)
                End Sub
            End Class
            Class c1
                Inherits Base
            End Class
        End Module
    </file>
</compilation>)

            Dim expected =
<expected>
BC30387: Class 'M1.c1' must declare a 'Sub New' because its base class 'M1.Base' does not have an accessible 'Sub New' that can be called with no arguments.
            Class c1
                  ~~
</expected>

            CompilationUtils.AssertTheseDiagnostics(compilation, expected)

            CompilationUtils.AssertTheseDiagnostics(compilation.GetDiagnosticsForSyntaxTree(CompilationStage.Compile, compilation.SyntaxTrees.Single()), expected)
        End Sub

        <Fact()>
        Public Sub BC30389ERR_InaccessibleSymbol2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
            Module mod30389
                Sub foo()
                    'COMPILEERROR: BC30389, "Namespace1.Module1.Class1.Struct1"
                    Dim Scen3 As Namespace1.Module1.Class1.Struct1
                    Exit Sub
                End Sub
            End Module
            Namespace Namespace1
                Module Module1
                    Private Class Class1
                        Public Structure Struct1
                            Public Int As Integer
                        End Structure
                    End Class
                End Module
            End Namespace
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC42024: Unused local variable: 'Scen3'.
                    Dim Scen3 As Namespace1.Module1.Class1.Struct1
                        ~~~~~
BC30389: 'Namespace1.Module1.Class1' is not accessible in this context because it is 'Private'.
                    Dim Scen3 As Namespace1.Module1.Class1.Struct1
                                 ~~~~~~~~~~~~~~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30389ERR_InaccessibleSymbol2_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            Protected Structure S
            End Structure
            Private F As Integer
            Private Property P As Integer
            Protected Shared ReadOnly Property Q(o)
                Get
                    Return Nothing
                End Get
            End Property
        End Class
        Class D
            Shared Sub M(o)
                Dim x As C = Nothing
                M(New C.S())
                M(x.F)
                M(x.P)
                M(C.Q(Nothing))
            End Sub
        End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30389: 'C.S' is not accessible in this context because it is 'Protected'.
                M(New C.S())
                      ~~~
BC30389: 'C.F' is not accessible in this context because it is 'Private'.
                M(x.F)
                  ~~~
BC30389: 'C.P' is not accessible in this context because it is 'Private'.
                M(x.P)
                  ~~~
BC30389: 'C.Q(o As Object)' is not accessible in this context because it is 'Protected'.
                M(C.Q(Nothing))
                  ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30390ERR_InaccessibleMember3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            Protected Shared Function F()
                Return Nothing
            End Function
            Private Sub M(o)
            End Sub
        End Class
        Class D
            Shared Sub M(x As C)
                x.M(C.F())
            End Sub
        End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30390: 'C.Private Sub M(o As Object)' is not accessible in this context because it is 'Private'.
                x.M(C.F())
                ~~~
BC30390: 'C.Protected Shared Function F() As Object' is not accessible in this context because it is 'Protected'.
                x.M(C.F())
                    ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30390ERR_InaccessibleMember3_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
        Class C1
            Public Delegate Sub myDelegate()
            private shared sub mySub()
            end sub
        End Class
        Module M1
            Sub Main()
                Dim d1 As C1.myDelegate
                d1 = New C1.myDelegate(addressof C1.mySub)
                d1 = addressof C1.mysub
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30390: 'C1.Private Shared Sub mySub()' is not accessible in this context because it is 'Private'.
                d1 = New C1.myDelegate(addressof C1.mySub)
                                                 ~~~~~~~~
BC30390: 'C1.Private Shared Sub mySub()' is not accessible in this context because it is 'Private'.
                d1 = addressof C1.mysub
                               ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30390ERR_InaccessibleMember3_2a()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC30390ERR_InaccessibleMember3_2a">
        <file name="a.vb">
Imports System
Module M1
    Class B
        Private Sub M()
            Console.WriteLine("B.M()")
        End Sub
    End Class

    Class D
        Inherits B

        Public Sub M()
            Console.WriteLine("D.M()")
        End Sub

        Public Sub Test()
            MyBase.M()
            Me.M()
        End Sub
    End Class

    Public Sub Main()
        Call (New D()).Test()
    End Sub
End Module
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30390: 'B.Private Sub M()' is not accessible in this context because it is 'Private'.
            MyBase.M()
            ~~~~~~~~
</expected>)
        End Sub

        <WorkItem(540640, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/540640")>
        <Fact()>
        Public Sub BC30390ERR_InaccessibleMember3_3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30390ERR_InaccessibleMember3_3">
        <file name="a.vb"><![CDATA[
Imports System
Namespace AttrRegress001
    Public Class Attr
        Inherits Attribute

        Public Property PriSet() As Short
            Get
                Return 1
            End Get
            Private Set(ByVal value As Short)
            End Set
        End Property

        Public Property ProSet() As Short
            Get
                Return 2
            End Get
            Protected Set(ByVal value As Short)
            End Set
        End Property

    End Class

    'COMPILEERROR: BC30390, "foo1"
    <Attr(PriSet:=1)> Class Scen2
    End Class

    'COMPILEERROR: BC30390, "foo2"
    <Attr(ProSet:=1)> Class Scen3
    End Class

End Namespace
        ]]></file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_InaccessibleMember3, "PriSet").WithArguments("AttrRegress001.Attr", "Public Property PriSet As Short", "Private"),
                                      Diagnostic(ERRID.ERR_InaccessibleMember3, "ProSet").WithArguments("AttrRegress001.Attr", "Public Property ProSet As Short", "Protected"))

        End Sub

        <Fact()>
        Public Sub BC30392ERR_CatchNotException1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="CatchNotException1">
        <file name="a.vb">
            Module M1
                Sub Foo()
                    Try
                    Catch o As Object
                        Throw
                    End Try
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30392: 'Catch' cannot catch type 'Object' because it is not 'System.Exception' or a class that inherits from 'System.Exception'.
                    Catch o As Object
                               ~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30393ERR_ExitTryNotWithinTry()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="ExitTryNotWithinTry">
    <file name="a.vb">
     Class S1
         sub FOO()
             if (true)
                 exit try
             End If
         End sub
    End CLASS
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30393: 'Exit Try' can only appear inside a 'Try' statement.
                 exit try
                 ~~~~~~~~
</expected>)
        End Sub

        <WorkItem(542801, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542801")>
        <Fact()>
        Public Sub BC30393ERR_ExitTryNotWithinTry_ExitFromFinally()

            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndReferences(
<compilation name="ExitTryNotWithinTry">
    <file name="a.vb">
Imports System
Imports System.Linq
Class BaseClass
    Function Method() As String
        Dim x = New Integer() {}
        x.Where(Function(y)
                    Try
                        Exit Try
                    Catch ex1 As Exception When True
                        Exit Try
                    Finally
                        Exit Try
                    End Try
                    Return y = ""
                End Function)
        Return "x"
    End Function
End Class
Class DerivedClass
    Inherits BaseClass
    Shared Sub Main()
    End Sub
End Class
    </file>
</compilation>, {Net40.SystemCore})
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30393: 'Exit Try' can only appear inside a 'Try' statement.
                        Exit Try
                        ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    MustInherit Class Base
        MustOverride Sub Pearl()
    End Class
    MustInherit Class C2
        Inherits Base
        Sub foo()
            Call MyBase.Pearl()
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Sub Pearl()' because it is declared 'MustOverride'.
            Call MyBase.Pearl()
                 ~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    MustInherit Class A
        MustOverride Property P
    End Class
    Class B
        Inherits A
        Overrides Property P
            Get
                Return Nothing
            End Get
            Set(ByVal value As Object)
            End Set
        End Property
        Sub M()
            MyBase.P = MyBase.P
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Property P As Object' because it is declared 'MustOverride'.
            MyBase.P = MyBase.P
            ~~~~~~~~
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Property P As Object' because it is declared 'MustOverride'.
            MyBase.P = MyBase.P
                       ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    Imports System
    MustInherit Class Base
        MustOverride Sub Pearl()
    End Class
    MustInherit Class C2
        Inherits Base
        Sub foo()
            Dim _action As Action = AddressOf MyBase.Pearl
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Sub Pearl()' because it is declared 'MustOverride'.
            Dim _action As Action = AddressOf MyBase.Pearl
                                              ~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    Imports System
    MustInherit Class Base
        MustOverride Function GetBar(i As Integer) As Integer
    End Class
    MustInherit Class C2
        Inherits Base
        Sub foo()
            Dim f As Func(Of Func(Of Integer, String)) = Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Function GetBar(i As Integer) As Integer' because it is declared 'MustOverride'.
            Dim f As Func(Of Func(Of Integer, String)) = Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
                                                                                                           ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
    Imports System
    MustInherit Class Base
        MustOverride Function GetBar(i As Integer) As Integer
    End Class
    MustInherit Class C2
        Inherits Base

        Public FLD As Func(Of Func(Of Integer, String)) =
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)

        Public Property PROP As Func(Of Func(Of Integer, String)) =
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)

        Public Overrides Function GetBar(i As Integer) As Integer
            Return Nothing
        End Function
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Function GetBar(i As Integer) As Integer' because it is declared 'MustOverride'.
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
                                                              ~~~~~~~~~~~~~
BC30399: 'MyBase' cannot be used with method 'Public MustOverride Function GetBar(i As Integer) As Integer' because it is declared 'MustOverride'.
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
                                                              ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
    Imports System
    Class Base
        Function GetBar(i As Integer) As Integer
            Return Nothing
        End Function
    End Class
    Class C2
        Inherits Base
        Sub foo()
            Dim f As Func(Of Func(Of Integer, String)) = 
                Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
        End Sub
    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation, <errors></errors>)
        End Sub

        <Fact()>
        Public Sub BC30399ERR_MyBaseAbstractCall1_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
    Imports System
    Class Base
        Shared Function GetBar(i As Integer) As Integer
            Return 0
        End Function
        Function GetFoo(i As Integer) As Integer
            Return 0
        End Function
    End Class
    MustInherit Class C2
        Inherits Base

        Public FLD As Func(Of Func(Of Integer, String)) =
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetBar)
        Public Property PROP As Func(Of Func(Of Integer, String)) =
            Function() New Func(Of Integer, String)(AddressOf MyBase.GetFoo)

    End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation, <errors></errors>)
        End Sub

        <Fact>
        Public Sub BC30399ERR_MyBaseAbstractCall1_8()
            Dim source =
<compilation>
    <file name="a.vb"><![CDATA[
Imports System
Imports System.Runtime.InteropServices
Imports System.Runtime.CompilerServices

MustInherit Class B
    Sub New()
    End Sub

    <DllImport("doo")>
    Shared Function DllImp(i As Integer) As Integer
    End Function

    Declare Function DeclareFtn Lib "foo" (i As Integer) As Integer
End Class

MustInherit Class C
    Inherits B

    Public FLD1 As Func(Of Func(Of Integer, String)) =
        Function() New Func(Of Integer, String)(AddressOf MyBase.DllImp)

    Public FLD2 As Func(Of Func(Of Integer, String)) =
        Function() New Func(Of Integer, String)(AddressOf MyBase.DeclareFtn)
End Class
]]>
    </file>
</compilation>
            CompileAndVerify(source)
        End Sub

        <Fact()>
        Public Sub BC30414ERR_ConvertArrayRankMismatch2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ConvertArrayRankMismatch2">
        <file name="a.vb">
       Module M1
            Sub Main()
                Dim Ary1() As Integer = New Integer() {1}
                Dim Ary2 As Integer(,) = CType(Ary1, Integer(,))
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30414: Value of type 'Integer()' cannot be converted to 'Integer(*,*)' because the array types have different numbers of dimensions.
                Dim Ary2 As Integer(,) = CType(Ary1, Integer(,))
                                               ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30414ERR_ConvertArrayRankMismatch2_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ConvertArrayRankMismatch2">
        <file name="a.vb">
        Option Strict On
        Imports System
        Module Module1
            Sub Main()
                Dim arr1 As Integer(,,) = New Integer(9, 5) {} ' Invalid
                Dim arr2 As Integer() = New Integer(9, 5) {} ' Invalid
                Dim arr3() As Integer = New Integer(2, 3) {} ' Invalid
                Dim arr4(,) As Integer = New Integer(2, 3, 1) {} ' Invalid
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30414: Value of type 'Integer(*,*)' cannot be converted to 'Integer(*,*,*)' because the array types have different numbers of dimensions.
                Dim arr1 As Integer(,,) = New Integer(9, 5) {} ' Invalid
                                          ~~~~~~~~~~~~~~~~~~~~
BC30414: Value of type 'Integer(*,*)' cannot be converted to 'Integer()' because the array types have different numbers of dimensions.
                Dim arr2 As Integer() = New Integer(9, 5) {} ' Invalid
                                        ~~~~~~~~~~~~~~~~~~~~
BC30414: Value of type 'Integer(*,*)' cannot be converted to 'Integer()' because the array types have different numbers of dimensions.
                Dim arr3() As Integer = New Integer(2, 3) {} ' Invalid
                                        ~~~~~~~~~~~~~~~~~~~~
BC30414: Value of type 'Integer(*,*,*)' cannot be converted to 'Integer(*,*)' because the array types have different numbers of dimensions.
                Dim arr4(,) As Integer = New Integer(2, 3, 1) {} ' Invalid
                                         ~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30414ERR_ConvertArrayRankMismatch2_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ConvertArrayRankMismatch2">
        <file name="a.vb">
        Option Strict On
        Imports System
        Module Module1
            Sub Main()
                Dim myArray10 As Integer(,) = {1, 2}        ' Invalid
                Dim myArray11 As Integer(,) = {{{1, 2}}}    ' Invalid
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30414: Value of type 'Integer()' cannot be converted to 'Integer(*,*)' because the array types have different numbers of dimensions.
                Dim myArray10 As Integer(,) = {1, 2}        ' Invalid
                                              ~~~~~~
BC30414: Value of type 'Integer(*,*,*)' cannot be converted to 'Integer(*,*)' because the array types have different numbers of dimensions.
                Dim myArray11 As Integer(,) = {{{1, 2}}}    ' Invalid
                                              ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30415ERR_RedimRankMismatch()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="RedimRankMismatch">
        <file name="a.vb">
            Class C1
                Sub foo(ByVal Ary() As Date)
                    ReDim Ary(4, 4)
                End Sub
            End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30415: 'ReDim' cannot change the number of dimensions of an array.
                    ReDim Ary(4, 4)
                          ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30424ERR_ConstAsNonConstant()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ConstAsNonConstant">
        <file name="a.vb">
            Class C1(Of T)
                Const f As T = Nothing
                Const c As C1(Of T) = Nothing
            End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
                Const f As T = Nothing
                           ~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
                Const c As C1(Of T) = Nothing
                           ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30424ERR_ConstAsNonConstant02()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
imports system

Enum E
    foo
End Enum

structure S1
end structure

Class C1
    ' should work
    public const f1 as E = E.foo
    public const f2 as object = nothing
    public const f3 as boolean = True

    ' should not work
    public const f4 as C1 = nothing
    public const f5 as S1 = nothing
    public const f6() as integer = {1,2,3}
    public const f7() as S1 = {new S1(), new S1(), new S1()}

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f4 as C1 = nothing
                       ~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f5 as S1 = nothing
                       ~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f6() as integer = {1,2,3}
                 ~~
BC30424: Constants must be of an intrinsic or enumerated type, not a class, structure, type parameter, or array type.
    public const f7() as S1 = {new S1(), new S1(), new S1()}
                 ~~
</expected>)
            ' todo: the last two errors need to be removed once collection initialization is supported
        End Sub

        <Fact()>
        Public Sub BC30438ERR_ConstantWithNoValue()
            Dim source =
<compilation>
    <file name="a.vb">
Option Strict On
Option Infer On

imports system

Class C1
    public const f1
    public const f2 as object 
    public const f3 as boolean

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30438: Constants must have a value.
    public const f1
                 ~~
BC30438: Constants must have a value.
    public const f2 as object 
                 ~~
BC30438: Constants must have a value.
    public const f3 as boolean
                 ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30439ERR_ExpressionOverflow1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ConvertArrayRankMismatch2">
        <file name="a.vb">
                Public Class C1
                    Shared Sub Main()
                        Dim i As Integer
                        i = 10000000000000
                        System.Console.WriteLine(i)
                    End Sub
                End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30439: Constant expression not representable in type 'Integer'.
                        i = 10000000000000
                            ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30439ERR_ExpressionOverflow1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ConvertArrayRankMismatch2">
        <file name="a.vb">
                Public Class C1
                    Shared Sub Main()
                        Dim FIRST As UInteger = (0UI - 860UI)
                    End Sub
                End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30439: Constant expression not representable in type 'UInteger'.
                        Dim FIRST As UInteger = (0UI - 860UI)
                                                 ~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
       Class C1
            function foo() as integer
                return 1 
            End function
        End Class
        Class C2
            function foo1() as integer
                dim s = foo()
                return 1
            End function
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30451: 'foo' is not declared. It may be inaccessible due to its protection level.
                dim s = foo()
                        ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            Shared ReadOnly Property P
                Get
                    Return Nothing
                End Get
            End Property
            ReadOnly Property Q
                Get
                    Return Nothing
                End Get
            End Property
            Property R
            Sub M()
                set_P(get_P)
                set_Q(get_Q)
                set_R(get_R)
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30451: 'set_P' is not declared. It may be inaccessible due to its protection level.
                set_P(get_P)
                ~~~~~
BC30451: 'get_P' is not declared. It may be inaccessible due to its protection level.
                set_P(get_P)
                      ~~~~~
BC30451: 'set_Q' is not declared. It may be inaccessible due to its protection level.
                set_Q(get_Q)
                ~~~~~
BC30451: 'get_Q' is not declared. It may be inaccessible due to its protection level.
                set_Q(get_Q)
                      ~~~~~
BC30451: 'set_R' is not declared. It may be inaccessible due to its protection level.
                set_R(get_R)
                ~~~~~
BC30451: 'get_R' is not declared. It may be inaccessible due to its protection level.
                set_R(get_R)
                      ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoDirectDelegateConstruction1">
        <file name="a.vb">
        Class C1
            Public Delegate Sub myDelegate()
            private shared sub mySub()
            end sub
        End Class
        Module M1
            Sub Main()
                Dim d1 As C1.myDelegate
                d1 = New C1.myDelegate(addressof BORG)
                d1 = addressof BORG
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30451: 'BORG' is not declared. It may be inaccessible due to its protection level.
                d1 = New C1.myDelegate(addressof BORG)
                                                 ~~~~
BC30451: 'BORG' is not declared. It may be inaccessible due to its protection level.
                d1 = addressof BORG
                               ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoDirectDelegateConstruction1">
        <file name="a.vb">
Class C
    Public Shared Sub Main()
        For Each x As Integer In New Integer() {1, 1, 1}
            'COMPILEERROR: BC30451, "y"
        Next y
        'escaped vs. nonescaped (should work)"
        For Each [x] As Integer In New Integer() {1, 1, 1}
        Next x
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30451: 'y' is not declared. It may be inaccessible due to its protection level.
        Next y
             ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoDirectDelegateConstruction1">
        <file name="a.vb">
Option Infer Off
Option Strict On
Public Class MyClass1
    Public Shared Sub Main()
        For n = 0 To 2
            For m = 1 To 2
            Next n
        Next m
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30451: 'n' is not declared. It may be inaccessible due to its protection level.
        For n = 0 To 2
            ~
BC30451: 'm' is not declared. It may be inaccessible due to its protection level.
            For m = 1 To 2
                ~
BC30451: 'n' is not declared. It may be inaccessible due to its protection level.
            Next n
                 ~
BC30451: 'm' is not declared. It may be inaccessible due to its protection level.
        Next m
             ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_NoErrorDuplicationForObjectInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC30451ERR_NameNotDeclared1_NoErrorDuplicationForObjectInitializer">
        <file name="a.vb">
Imports System
Imports System.Collections.Generic

Class S
    Public Y As Object
End Class

Public Module Program
    Public Sub Main(args() As String)
        Dim a, b, c As New S() With {.Y = aaa}
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30451: 'aaa' is not declared. It may be inaccessible due to its protection level.
        Dim a, b, c As New S() With {.Y = aaa}
                                          ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30451ERR_NameNotDeclared1_NoWarningDuplicationForObjectInitializer()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC30451ERR_NameNotDeclared1_NoWarningDuplicationForObjectInitializer">
        <file name="a.vb">
Option Strict On

Imports System

Class S
    Public Y As Byte
End Class

Public Module Program
    Public Sub Main(args() As String)
        Dim x As Integer = 1
        Dim a, b, c As New S() With {.Y = x}
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30512: Option Strict On disallows implicit conversions from 'Integer' to 'Byte'.
        Dim a, b, c As New S() With {.Y = x}
                                          ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BinaryOperands3">
        <file name="a.vb">
option infer on            
Class C1
    dim d = new c1() + new c1.c2()
    function foo() as integer
        return 1
    End function
    Class C2
        function foo1() as integer
            dim d1 = new c1()
            dim d2 = new c2()
            dim d3 = d1 + d2
            return 1
        End function
    End Class
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30452: Operator '+' is not defined for types 'C1' and 'C1.C2'.
    dim d = new c1() + new c1.c2()
            ~~~~~~~~~~~~~~~~~~~~~~
BC30452: Operator '+' is not defined for types 'C1' and 'C1.C2'.
            dim d3 = d1 + d2
                     ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BinaryOperands3">
        <file name="a.vb">
        Class C1
                Class C2
                    function foo1(byval d1 as c1, byval d2 as c2 )as integer
                        dim d3 as object = d1 + d2
                        return 1
                    End function
                End Class
            End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30452: Operator '+' is not defined for types 'C1' and 'C1.C2'.
                        dim d3 as object = d1 + d2
                                           ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="None">
    <file name="a.vb">
        Imports System
        class myClass1
        shared result = New Guid() And New Guid()
        End class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30452: Operator 'And' is not defined for types 'Guid' and 'Guid'.
        shared result = New Guid() And New Guid()
                        ~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="None">
    <file name="a.vb">
        Imports System
        Module Program
            Sub Main(args As String())
                Dim f1 As New Foo(), f2 As New Foo(), f3 As New Foo()
                Dim b As Boolean = True
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
                b = False
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
            End Sub
        End Module
        Class Foo
            Public i As Integer
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30452: Operator '=' is not defined for types 'Foo' and 'Foo'.
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
                           ~~~~~~~~~~~~~~
BC30452: Operator '=' is not defined for types 'Foo' and 'Foo'.
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
                                           ~~~~~~~~~~~~~~
BC30452: Operator '=' is not defined for types 'Foo' and 'Foo'.
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
                           ~~~~~~~~~~~~~~
BC30452: Operator '=' is not defined for types 'Foo' and 'Foo'.
                f3 = If(b, f1 = New Foo(), f2 = New Foo())
                                           ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3_4()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="None">
    <file name="a.vb"><![CDATA[
Module Program
    Sub Main()
        Dim First = New With {.a = 1, .b = 2}
        Dim Second = New With {.a = 1, .b = 2}
    'COMPILEERROR: BC30452, "first = second"
        If first = second Then
        ElseIf second <> first Then
        End If
    End Sub
End Module
    ]]></file>
</compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_BinaryOperands3, "first = second").WithArguments("=", "<anonymous type: a As Integer, b As Integer>", "<anonymous type: a As Integer, b As Integer>"),
    Diagnostic(ERRID.ERR_BinaryOperands3, "second <> first").WithArguments("<>", "<anonymous type: a As Integer, b As Integer>", "<anonymous type: a As Integer, b As Integer>"))
        End Sub

        <Fact()>
        Public Sub BC30452ERR_BinaryOperands3_4a()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="BC30452ERR_BinaryOperands3_4a">
    <file name="a.vb"><![CDATA[
Imports System
Module Program
    Sub Main()
    'COMPILEERROR: BC30452, "first = second"
        If New With {.a = 1, .b = 2} = New With {.a = 1, .b = 2} Then
        ElseIf New With {.a = 1, .b = 2} <> New With {.a = 1, .b = 2} Then
        End If
    End Sub
End Module
    ]]></file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected><![CDATA[
BC30452: Operator '=' is not defined for types '<anonymous type: a As Integer, b As Integer>' and '<anonymous type: a As Integer, b As Integer>'.
        If New With {.a = 1, .b = 2} = New With {.a = 1, .b = 2} Then
           ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC30452: Operator '<>' is not defined for types '<anonymous type: a As Integer, b As Integer>' and '<anonymous type: a As Integer, b As Integer>'.
        ElseIf New With {.a = 1, .b = 2} <> New With {.a = 1, .b = 2} Then
               ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]]></expected>)
        End Sub

        <Fact()>
        Public Sub BC30454ERR_ExpectedProcedure()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ExpectedProcedure">
        <file name="a.vb">
    Module IsNotError001mod
        Sub foo(byval value as integer())
            value()
        exit sub
        End Sub
    End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30454: Expression is not a method.
            value()
            ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30454ERR_ExpectedProcedure_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ExpectedProcedure">
        <file name="a.vb">
        Class C
            Private s As String
            Shared Sub M(x As C, y() As Integer)
                Dim o As Object
                o = x.s(3)
                N(x.s(3))
                x.s(3) ' BC30454
                o = y(3)
                N(y(3))
                y(3) ' BC30454
            End Sub
            Shared Sub N(o)
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30454: Expression is not a method.
                x.s(3) ' BC30454
                ~~~
BC30454: Expression is not a method.
                y(3) ' BC30454
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30455ERR_OmittedArgument2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
            Class C
                Default ReadOnly Property P(x, y)
                    Get
                        Return Nothing
                    End Get
                End Property
                Shared Sub M(x As C)
                    N(x!Q)
                End Sub
                Shared Sub N(o As Object)
                End Sub
            End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30455: Argument not specified for parameter 'y' of 'Public ReadOnly Default Property P(x As Object, y As Object) As Object'.
                    N(x!Q)
                      ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30455ERR_OmittedArgument2_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
            Structure C1
               &lt;System.Runtime.InteropServices.FieldOffset()&gt;
               Dim i As Integer
            End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30455: Argument not specified for parameter 'offset' of 'Public Overloads Sub New(offset As Integer)'.
               &lt;System.Runtime.InteropServices.FieldOffset()&gt;
                                               ~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30456ERR_NameNotMember2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
    Class Class1
    Private Shared foo As S1
    Class Class2
        Sub Test()
                foo.bar1()
        End Sub
    End Class
    End Class
    Structure S1
        Public shared bar As String = "Hello"
    End Structure
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30456: 'bar1' is not a member of 'S1'.
                foo.bar1()
                ~~~~~~~~
</expected>)
        End Sub

        <WorkItem(538903, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538903")>
        <Fact()>
        Public Sub BC30456ERR_NameNotMember2_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
    Module M1
        Sub FOO()
                My.Application.Exit()
        End Sub
    End Module
        </file>
    </compilation>)

            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30451: 'My' is not declared. It may be inaccessible due to its protection level.
                My.Application.Exit()
                ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30456ERR_NameNotMember2_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
    Module M1
        Sub FOO()
            Dim blnReturn As Boolean = False
            Dim x As System.Nullable(Of Integer)
            blnReturn = system.nullable.hasvalue(x)
        End Sub
    End Module
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30456: 'hasvalue' is not a member of 'Nullable'.
            blnReturn = system.nullable.hasvalue(x)
                        ~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30456ERR_NameNotMember2_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            Shared ReadOnly Property P
                Get
                    Return Nothing
                End Get
            End Property
            ReadOnly Property Q
                Get
                    Return Nothing
                End Get
            End Property
            Sub M()
                C.set_P(C.get_P)
                Me.set_Q(Me.get_Q)
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30456: 'set_P' is not a member of 'C'.
                C.set_P(C.get_P)
                ~~~~~~~
BC30456: 'get_P' is not a member of 'C'.
                C.set_P(C.get_P)
                        ~~~~~~~
BC30456: 'set_Q' is not a member of 'C'.
                Me.set_Q(Me.get_Q)
                ~~~~~~~~
BC30456: 'get_Q' is not a member of 'C'.
                Me.set_Q(Me.get_Q)
                         ~~~~~~~~
</expected>)
        End Sub

        <WorkItem(10046, "DevDiv_Projects/Roslyn")>
        <Fact()>
        Public Sub BC30456ERR_NameNotMember2_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Option Infer On
Imports System
Class Program
    Dim x As New Product With {.Name = "paperclips", .price1 = 1.29}
End Class
Class Product
    Property price As Double
    Property Name As String
End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30456: 'price1' is not a member of 'Product'.
    Dim x As New Product With {.Name = "paperclips", .price1 = 1.29}
                                                      ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30456ERR_NameNotMember2_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC30456ERR_NameNotMember2_4">
        <file name="a.vb">
Module M1
    Class B
    End Class

    Class D
        Inherits B

        Public Shadows Sub M()
        End Sub

        Public Sub Test()
            MyBase.M()
            Me.M()
        End Sub
    End Class
End Module
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30456: 'M' is not a member of 'M1.B'.
            MyBase.M()
            ~~~~~~~~
</expected>)
        End Sub

        <WorkItem(529710, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/529710")>
        <Fact()>
        Public Sub BC30456ERR_NameNotMember3_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Namespace N
    Module X
        Sub Main()
            N.Equals("", "")
        End Sub
    End Module
End Namespace
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30456: 'Equals' is not a member of 'N'.
            N.Equals("", "")
            ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
       Class C1
            function foo() as integer
                return 1 
            End function
            Class C2
                function foo1() as integer
                    dim s = foo()
                    return 1
                End function
            End Class
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30469: Reference to a non-shared member requires an object reference.
                    dim s = foo()
                            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ObjectReferenceNotSupplied">
        <file name="a.vb">
       Class P(Of T)
            Public ReadOnly son As T
            Class P1
                Sub New1(ByVal tval As T)
                    son = tval
                End Sub
            End Class
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30469: Reference to a non-shared member requires an object reference.
                    son = tval
                    ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            Property P
                Get
                    Return Nothing
                End Get
                Set(ByVal value)
                End Set
            End Property
            Shared Sub M()
                Dim o = C.P
                C.P = o
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30469: Reference to a non-shared member requires an object reference.
                Dim o = C.P
                        ~~~
BC30469: Reference to a non-shared member requires an object reference.
                C.P = o
                ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class A
            Property P
                Get
                    Return Nothing
                End Get
                Set(ByVal value)
                End Set
            End Property
            Class B
                Sub M(ByVal value)
                    P = value
                    value = P
                End Sub
            End Class
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30469: Reference to a non-shared member requires an object reference.
                    P = value
                    ~
BC30469: Reference to a non-shared member requires an object reference.
                    value = P
                            ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_FieldInitializers()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
       Class C1
            public f1 as integer
            public shared f2 as integer = C1.f1
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30469: Reference to a non-shared member requires an object reference.
            public shared f2 as integer = C1.f1
                                          ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30469ERR_ObjectReferenceNotSupplied_DelegateCreation()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoDirectDelegateConstruction1">
        <file name="a.vb">
        Class C1
            Public Delegate Sub myDelegate()
            public sub mySub()
            end sub
        End Class
        Module M1
            Sub foo()
                Dim d1 As C1.myDelegate
                d1 = New C1.myDelegate(addressof C1.mySub)
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30469: Reference to a non-shared member requires an object reference.
                d1 = New C1.myDelegate(addressof C1.mySub)
                                       ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30470ERR_MyClassNotInClass()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="MyClassNotInClass">
        <file name="a.vb">
        Module M1
            Sub FOO()
                MyClass.New()
            End Sub
            Sub New()
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30470: 'MyClass' cannot be used outside of a class.
                MyClass.New()
                ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30487ERR_UnaryOperand2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="UnaryOperand2">
        <file name="a.vb">
        Class C1
            Shared Sub FOO()
                Dim expr As c2 = new c2()
                expr = -expr
            End Sub
        End Class
        Class C2
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30487: Operator '-' is not defined for type 'C2'.
                expr = -expr
                       ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30491ERR_VoidValue()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="VoidValue">
        <file name="a.vb">
        Structure C1
            Sub FOO()
                'Dim a1 = If (True, New Object, TestMethod)
                'Dim a2 = If (True, {TestMethod()}, {TestMethod()})
                Dim a3 = TestMethod() = TestMethod()
            End Sub
            Sub TestMethod()
            End Sub
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30491: Expression does not produce a value.
                Dim a3 = TestMethod() = TestMethod()
                         ~~~~~~~~~~~~
BC30491: Expression does not produce a value.
                Dim a3 = TestMethod() = TestMethod()
                                        ~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30491ERR_VoidValue_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="VoidValue">
        <file name="a.vb">
        Imports System
            Module Program
                Sub Main(args As String())
                    Dim x = If(True, Console.WriteLine(0), Console.WriteLine(1))
                    Dim y = If(True, fun_void(), fun_int(1))
                    Dim z = If(True, fun_Exception(1), fun_int(1))
                    Dim r = If(True, fun_long(0), fun_int(1))
                    Dim s = If(False, fun_long(0), fun_int(1))
                End Sub
                Private Sub fun_void()
                    Return
                End Sub
                Private Function fun_int(x As Integer) As Integer
                    Return x
                End Function
                Private Function fun_long(x As Integer) As Long
                    Return CLng(x)
                End Function
                Private Function fun_Exception(x As Integer) As Exception
                    Return New Exception()
                End Function
            End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30491: Expression does not produce a value.
                    Dim x = If(True, Console.WriteLine(0), Console.WriteLine(1))
                                     ~~~~~~~~~~~~~~~~~~~~
BC30491: Expression does not produce a value.
                    Dim x = If(True, Console.WriteLine(0), Console.WriteLine(1))
                                                           ~~~~~~~~~~~~~~~~~~~~
BC30491: Expression does not produce a value.
                    Dim y = If(True, fun_void(), fun_int(1))
                                     ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30491ERR_VoidValue_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="VoidValue">
        <file name="a.vb">
Imports System
Module Program
    Sub Main(args As String())
        Dim x As Integer = 1
        Dim y As Object = 0
        Dim s = If(True, fun(x), y)
        Dim s1 = If(False, sub1(x), y)
    End Sub
    Private Function fun(Of T)(Parm1 As T) As T
        Return Parm1
    End Function
    Private Sub sub1(Of T)(Parm1 As T)
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30491: Expression does not produce a value.
        Dim s1 = If(False, sub1(x), y)
                           ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30491ERR_VoidValue_SelectCase()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="VoidValue">
        <file name="a.vb">
        Structure C1
            Sub Foo()
                Select Case TestMethod()
                End Select
            End Sub
            Sub TestMethod()
            End Sub
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30491: Expression does not produce a value.
                Select Case TestMethod()
                            ~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30500ERR_CircularEvaluation1()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
imports system

Class C1
    Enum E
        A = A
    End Enum

    public const f1 as integer = f2
    public const f2 as integer = f1

    Public shared Sub Main(args() as string)
        Console.WriteLine(E.A)    
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30500: Constant 'A' cannot depend on its own value.
        A = A
        ~
BC30500: Constant 'f1' cannot depend on its own value.
    public const f1 as integer = f2
                 ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30500ERR_CircularEvaluation1_02()
            Dim source =
<compilation>
    <file name="a.vb">
Option Strict On
Option Infer On

imports system

Class C1
    private const f1 = f2
    private const f2 = f1

    Public shared Sub Main(args() as string)
        console.writeline(f1)
        console.writeline(f2)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30500: Constant 'f1' cannot depend on its own value.
    private const f1 = f2
                  ~~
</expected>)

        End Sub

        <Fact(), WorkItem(528728, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528728")>
        Public Sub BC30500ERR_CircularEvaluation1_03()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="CircularEvaluation1">
        <file name="a.vb">
        Module M1
            Sub New()
                Const Val As Integer = Val
            End Sub
        End Module
    </file>
    </compilation>)
            compilation.AssertTheseDiagnostics(
<expected>
BC30500: Constant 'Val' cannot depend on its own value.
                Const Val As Integer = Val
                      ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30500ERR_CircularEvaluation1_04()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
Class C
    Const c0, c1 = c2 + 1
    Const c2, c3 = c0 + 1
End Class
    </file>
</compilation>)
            compilation.AssertTheseDiagnostics(
<expected>
BC30500: Constant 'c0' cannot depend on its own value.
    Const c0, c1 = c2 + 1
          ~~
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
    Const c0, c1 = c2 + 1
          ~~~~~~~~~~~~~~~
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
    Const c2, c3 = c0 + 1
          ~~~~~~~~~~~~~~~
BC30060: Conversion from 'Integer' to 'Object' cannot occur in a constant expression.
    Const c2, c3 = c0 + 1
                        ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NarrowingConversionDisallowed2">
        <file name="a.vb">
    Option Strict On
    Imports System
    Module M1
        Sub Foo()
            Dim b As Byte = 2
            Dim c As Byte = 3
            Dim s As Short = 2
            Dim t As Short = 3
            Dim i As Integer = 2
            Dim j As Integer = 3
            Dim l As Long = 2
            Dim m As Long = 3
            b = b &lt; c
            b = b ^ c
            s = s &lt; t
            s = s ^ t
            i = i &gt; j
            i = i ^ j
            l = l &gt; m
            l = l ^ m
        End Sub
    End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Byte'.
            b = b &lt; c
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Byte'.
            b = b ^ c
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Short'.
            s = s &lt; t
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Short'.
            s = s ^ t
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Integer'.
            i = i &gt; j
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
            i = i ^ j
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Long'.
            l = l &gt; m
                ~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Long'.
            l = l ^ m
                ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
            Option Strict On
            Class C
                Default ReadOnly Property P(i As Integer) As Object
                    Get
                        Return Nothing
                    End Get
                End Property
                Shared Sub M(x As C)
                    N(x("Q"))
                    N(x!Q)
                End Sub
                Shared Sub N(o As Object)
                End Sub
            End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Integer'.
                    N(x("Q"))
                        ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Integer'.
                    N(x!Q)
                        ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_2()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Module Program
    Sub Main(args As String())
        Dim S1 As String = "3"
        Dim S1_b As Object = If(S1, 3, 2)
        Dim S2 As SByte = 4
        Dim S2_b As Object = If(S2, 4, 2)
        Dim S3 As Byte = 5
        Dim S3_b As Object = If(S3, 5, 2)
        Dim S4 As Short = 6
        Dim S4_b As Object = If(S4, 6, 2)
        Dim S5 As UShort = 7
        Dim S5_b As Object = If(S5, 7, 2)
        Dim S6 As Integer = 8
        Dim S6_b As Object = If(S6, 8, 2)
        Dim S7 As UInteger = 9
        Dim S7_b As Object = If(S7, 9, 2)
        Dim S8 As Long = 10
        Dim S8_b As Object = If(S8, 10, 2)
        Dim S9 As Short? = 5
        Dim S9_b As Object = If(S9, 3, 2)
        Dim S10 As Integer? = 51
        Dim S10_b As Object = If(S10, 3, 2)
        Dim S11 As Boolean? = Nothing
        Dim S11_b As Object = If(S11, 3, 2)
    End Sub
End Module
    </file>
    </compilation>).VerifyDiagnostics(
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S1").WithArguments("String", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S2").WithArguments("SByte", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S3").WithArguments("Byte", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S4").WithArguments("Short", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S5").WithArguments("UShort", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S6").WithArguments("Integer", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S7").WithArguments("UInteger", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S8").WithArguments("Long", "Boolean"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S9").WithArguments("Short?", "Boolean?"),
    Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "S10").WithArguments("Integer?", "Boolean?"))

        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Imports System
Module Program
    Sub Main(args As String())
        Dim S1_a As Short? = 5
        Dim S1_b As Integer? = 51
        Dim S1_c As Short? = If(True, S1_a, S1_b)
        Dim S1_d As Boolean? = If(True, S1_a, S1_b)
        Dim S2_a As Char
        Dim S2_b As String = "31"
        Dim S2_c As String = If(True, S2_a, S2_b)
        Dim S2_d As Char = If(False, S2_a, S2_b)
        Dim S2_e As Short = If(False, S2_a, S2_b)
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'Integer?' to 'Short?'.
        Dim S1_c As Short? = If(True, S1_a, S1_b)
                             ~~~~~~~~~~~~~~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Integer?' to 'Boolean?'.
        Dim S1_d As Boolean? = If(True, S1_a, S1_b)
                               ~~~~~~~~~~~~~~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim S2_d As Char = If(False, S2_a, S2_b)
                           ~~~~~~~~~~~~~~~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Short'.
        Dim S2_e As Short = If(False, S2_a, S2_b)
                            ~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
Option Strict On
Public Class MyClass1
    Public Shared Sub Main()
        For ivar As Integer = 0.1 To 10
        Next
        For dvar As Double = #12:00:00 AM# To 10
        Next
        For dvar As Double = True To 10
        Next
        For dvar1 As Double = 123&amp; To 10
            'ok
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
        For ivar As Integer = 0.1 To 10
                              ~~~
BC30532: Conversion from 'Date' to 'Double' requires calling the 'Date.ToOADate' method.
        For dvar As Double = #12:00:00 AM# To 10
                             ~~~~~~~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Double'.
        For dvar As Double = True To 10
                             ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Imports System
Module Module1
    Sub Main()
        Dim arr As Integer(,) = New Integer(1.0, 4) {} ' Invalid
        Dim arr1 As Integer(,) = New Integer(CDbl(5), 4) {} ' Invalid
        Dim arr2 As Integer(,) = New Integer(CDec(5), 4) {} ' Invalid
        Dim arr3 As Integer(,) = New Integer(CSng(5), 4) {} ' Invalid
        Dim x As Double = 5
        Dim arr4 As Integer(,) = New Integer(x, 4) {} ' Invalid
        Dim y As Single = 5
        Dim arr5 As Integer(,) = New Integer(y, 4) {} ' Invalid
        Dim z As Decimal = 5
        Dim arr6 As Integer(,) = New Integer(z, 4) {} ' Invalid
        Dim m As Boolean = True
        Dim arr7 As Integer(,) = New Integer(m, 4) {} ' Invalid
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
        Dim arr As Integer(,) = New Integer(1.0, 4) {} ' Invalid
                                            ~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
        Dim arr1 As Integer(,) = New Integer(CDbl(5), 4) {} ' Invalid
                                             ~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Decimal' to 'Integer'.
        Dim arr2 As Integer(,) = New Integer(CDec(5), 4) {} ' Invalid
                                             ~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Single' to 'Integer'.
        Dim arr3 As Integer(,) = New Integer(CSng(5), 4) {} ' Invalid
                                             ~~~~~~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
        Dim arr4 As Integer(,) = New Integer(x, 4) {} ' Invalid
                                             ~
BC30512: Option Strict On disallows implicit conversions from 'Single' to 'Integer'.
        Dim arr5 As Integer(,) = New Integer(y, 4) {} ' Invalid
                                             ~
BC30512: Option Strict On disallows implicit conversions from 'Decimal' to 'Integer'.
        Dim arr6 As Integer(,) = New Integer(z, 4) {} ' Invalid
                                             ~
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Integer'.
        Dim arr7 As Integer(,) = New Integer(m, 4) {} ' Invalid
                                             ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Imports System
Module Module1
    Sub Main()
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                   ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                        ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                               ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                                    ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                                           ~~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Char'.
        Dim myArray9 As Char(,) = New Char(2, 1) {{"a", "b"}, {"c", "d"}, {"e", "f"}}
                                                                                ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Infer On
Option Strict On
Imports System
Imports Microsoft.VisualBasic.Strings
Module Module1
    Sub Main(args As String())
        Dim arr As Integer(,) = New Integer(4, 4) {}
        Dim x As Integer = 0
        Dim idx As Double = 2.0
        Dim idx1 As ULong = 0
        Dim idx2 As Char = ChrW(3)
        arr(idx, 3) = 100      ' Invalid
        arr(idx1, x) = 100     ' Invalid
        arr(idx2, 3) = 100      'Invalid
        arr(" "c, 32) = 100     'Invalid
        Dim arr1 As Integer(,,) = {{{1, 2}, {1, 2}}, {{1, 2}, {1, 2}}}
        Dim i1 As ULong = 0
        Dim i2 As UInteger = 0
        Dim i3 As Integer = 0
        arr1(i1, i2, i3) = 9        ' Invalid
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
        arr(idx, 3) = 100      ' Invalid
            ~~~
BC30512: Option Strict On disallows implicit conversions from 'ULong' to 'Integer'.
        arr(idx1, x) = 100     ' Invalid
            ~~~~
BC32006: 'Char' values cannot be converted to 'Integer'. Use 'Microsoft.VisualBasic.AscW' to interpret a character as a Unicode value or 'Microsoft.VisualBasic.Val' to interpret it as a digit.
        arr(idx2, 3) = 100      'Invalid
            ~~~~
BC32006: 'Char' values cannot be converted to 'Integer'. Use 'Microsoft.VisualBasic.AscW' to interpret a character as a Unicode value or 'Microsoft.VisualBasic.Val' to interpret it as a digit.
        arr(" "c, 32) = 100     'Invalid
            ~~~~
BC30512: Option Strict On disallows implicit conversions from 'ULong' to 'Integer'.
        arr1(i1, i2, i3) = 9        ' Invalid
             ~~
BC30512: Option Strict On disallows implicit conversions from 'UInteger' to 'Integer'.
        arr1(i1, i2, i3) = 9        ' Invalid
                 ~~
</expected>)
        End Sub

        <WorkItem(528762, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528762")>
        <Fact>
        Public Sub BC30512ERR_NarrowingConversionDisallowed2_8()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Option Infer On
Public Class Class2(Of T As Res)
    Private x As T(,) = New T(1, 1) {{New Res(), New Res()}, {New Res(), New Res()}}    ' invalid
    Public Sub Foo()
        Dim x As T(,) = New T(1, 2) {}
    End Sub
End Class
Public Class Res
End Class
    </file>
    </compilation>).VerifyDiagnostics(
            Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "New Res()").WithArguments("Res", "T"),
            Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "New Res()").WithArguments("Res", "T"),
            Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "New Res()").WithArguments("Res", "T"),
            Diagnostic(ERRID.ERR_NarrowingConversionDisallowed2, "New Res()").WithArguments("Res", "T"))
        End Sub

        <Fact()>
        Public Sub BC30512ERR_SelectCaseNarrowingConversionErrors()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb"><![CDATA[
Option Strict On
Imports System        
Module M1
    Sub Main()
        Dim success As Boolean = True
        For count = 0 To 13
            Test(count, success)
        Next

        If success Then
            Console.Write("Success")
        Else
            Console.Write("Fail")
        End If
    End Sub

    Sub Test(count As Integer, ByRef success As Boolean)
        Dim Bo As Boolean
        Dim Ob As Object
        Dim SB As SByte
        Dim By As Byte
        Dim Sh As Short
        Dim US As UShort
        Dim [In] As Integer
        Dim UI As UInteger
        Dim Lo As Long
        Dim UL As ULong
        Dim De As Decimal
        Dim Si As Single
        Dim [Do] As Double
        Dim St As String

        Bo = False
        Ob = 1
        SB = 2
        By = 3
        Sh = 4
        US = 5
        [In] = 6
        UI = 7
        Lo = 8
        UL = 9
        Si = 10
        [Do] = 11
        De = 12D
        St = "13"

        Select Case count
            Case Bo
                success = success AndAlso If(count = 0, True, False)
            Case Is < Ob
                success = success AndAlso If(count = 1, True, False)
            Case SB
                success = success AndAlso If(count = 2, True, False)
            Case By
                success = success AndAlso If(count = 3, True, False)
            Case Sh
                success = success AndAlso If(count = 4, True, False)
            Case US
                success = success AndAlso If(count = 5, True, False)
            Case [In]
                success = success AndAlso If(count = 6, True, False)
            Case UI To Lo
                success = success AndAlso If(count = 7, True, False)
            Case Lo
                success = success AndAlso If(count = 8, True, False)
            Case UL
                success = success AndAlso If(count = 9, True, False)
            Case Si
                success = success AndAlso If(count = 10, True, False)
            Case [Do]
                success = success AndAlso If(count = 11, True, False)
            Case De
                success = success AndAlso If(count = 12, True, False)
            Case St
                success = success AndAlso If(count = 13, True, False)
            Case Else
                success = False
        End Select
    End Sub
End Module
    ]]></file>
    </compilation>)
            Dim expectedErrors = <errors><![CDATA[
BC30512: Option Strict On disallows implicit conversions from 'Boolean' to 'Integer'.
            Case Bo
                 ~~
BC30038: Option Strict On prohibits operands of type Object for operator '<'.
            Case Is < Ob
                      ~~
BC30512: Option Strict On disallows implicit conversions from 'UInteger' to 'Integer'.
            Case UI To Lo
                 ~~
BC30512: Option Strict On disallows implicit conversions from 'Long' to 'Integer'.
            Case UI To Lo
                       ~~
BC30512: Option Strict On disallows implicit conversions from 'Long' to 'Integer'.
            Case Lo
                 ~~
BC30512: Option Strict On disallows implicit conversions from 'ULong' to 'Integer'.
            Case UL
                 ~~
BC30512: Option Strict On disallows implicit conversions from 'Single' to 'Integer'.
            Case Si
                 ~~
BC30512: Option Strict On disallows implicit conversions from 'Double' to 'Integer'.
            Case [Do]
                 ~~~~
BC30512: Option Strict On disallows implicit conversions from 'Decimal' to 'Integer'.
            Case De
                 ~~
BC30512: Option Strict On disallows implicit conversions from 'String' to 'Integer'.
            Case St
                 ~~
]]></errors>

            CompilationUtils.AssertTheseDiagnostics(compilation, expectedErrors)
        End Sub

        <Fact()>
        Public Sub BC30516ERR_NoArgumentCountOverloadCandidates1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoArgumentCountOverloadCandidates1">
        <file name="a.vb">
        Module Module1
            Class C0
                Public whichOne As String
                Sub Foo(ByVal t1 As String)
                    whichOne = "T"
                End Sub
            End Class
            Class C1
                Inherits C0
                Overloads Sub Foo(ByVal y1 As String)
                    whichOne = "Y"
                End Sub
            End Class
            Sub test()
                Dim [ident1] As C0 = New C0()
                Dim clsNarg2get As C1 = New C1()
                Dim str1 As String = "Visual Basic"
                'COMPILEERROR: BC30516, "y"
                clsNarg2get.Foo(1, y1:=2)
                'COMPILEERROR: BC30516, "x"
                clsNarg2get.Foo(y1:=1, y1:=1)
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30516: Overload resolution failed because no accessible 'Foo' accepts this number of arguments.
                clsNarg2get.Foo(1, y1:=2)
                            ~~~
BC30516: Overload resolution failed because no accessible 'Foo' accepts this number of arguments.
                clsNarg2get.Foo(y1:=1, y1:=1)
                            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30517ERR_NoViableOverloadCandidates1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoViableOverloadCandidates1">
        <file name="a.vb">
        Imports System
        &lt;AttributeUsage(AttributeTargets.All)&gt; Class attr2
            Inherits Attribute
            Private Sub New(ByVal i As Integer)
            End Sub
            Protected Sub New(ByVal i As Char)
            End Sub
        End Class
        &lt;attr2(1)&gt; Class target2
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30517: Overload resolution failed because no 'New' is accessible.
        &lt;attr2(1)&gt; Class target2
         ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30518ERR_NoCallableOverloadCandidates2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoCallableOverloadCandidates2">
        <file name="a.vb">
        class M1
            Sub sub1(Of U, V) (ByVal p1 As U, ByVal p2 As V)
            End Sub
            Sub sub1(Of U, V) (ByVal p1() As V, ByVal p2() As U)
            End Sub
            Sub GenMethod6210()
                sub1(Of String, Integer) (17@, #3/3/2003#)
                sub1(Of Integer, String) (New Integer() {1, 2, 3}, New String() {"a", "b"})
            End Sub
        End class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30518: Overload resolution failed because no accessible 'sub1' can be called with these arguments:
    'Public Sub sub1(Of String, Integer)(p1 As String, p2 As Integer)': Value of type 'Date' cannot be converted to 'Integer'.
    'Public Sub sub1(Of String, Integer)(p1 As Integer(), p2 As String())': Value of type 'Decimal' cannot be converted to 'Integer()'.
    'Public Sub sub1(Of String, Integer)(p1 As Integer(), p2 As String())': Value of type 'Date' cannot be converted to 'String()'.
                sub1(Of String, Integer) (17@, #3/3/2003#)
                ~~~~~~~~~~~~~~~~~~~~~~~~
BC30518: Overload resolution failed because no accessible 'sub1' can be called with these arguments:
    'Public Sub sub1(Of Integer, String)(p1 As Integer, p2 As String)': Value of type 'Integer()' cannot be converted to 'Integer'.
    'Public Sub sub1(Of Integer, String)(p1 As Integer, p2 As String)': Value of type 'String()' cannot be converted to 'String'.
    'Public Sub sub1(Of Integer, String)(p1 As String(), p2 As Integer())': Value of type 'Integer()' cannot be converted to 'String()' because 'Integer' is not derived from 'String'.
    'Public Sub sub1(Of Integer, String)(p1 As String(), p2 As Integer())': Value of type 'String()' cannot be converted to 'Integer()' because 'String' is not derived from 'Integer'.
                sub1(Of Integer, String) (New Integer() {1, 2, 3}, New String() {"a", "b"})
                ~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <WorkItem(546763, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/546763")>
        <Fact()>
        Public Sub BC30518ERR_NoCallableOverloadCandidates_LateBindingDisabled()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
    <compilation>
        <file name="a.vb">
Option Strict On
Option Infer On
Imports System
Imports System.Threading.Tasks

Public Module Program
    Sub Main()
        Dim a As Object = Nothing
        Parallel.ForEach(a, Sub(x As Object) Console.WriteLine(x))
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30518: Overload resolution failed because no accessible 'ForEach' can be called with these arguments:
    'Public Shared Overloads Function ForEach(Of TSource)(source As IEnumerable(Of TSource), body As Action(Of TSource)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
    'Public Shared Overloads Function ForEach(Of TSource)(source As IEnumerable(Of TSource), body As Action(Of TSource, ParallelLoopState)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
    'Public Shared Overloads Function ForEach(Of TSource)(source As IEnumerable(Of TSource), body As Action(Of TSource, ParallelLoopState, Long)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
    'Public Shared Overloads Function ForEach(Of TSource)(source As Partitioner(Of TSource), body As Action(Of TSource)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
    'Public Shared Overloads Function ForEach(Of TSource)(source As Partitioner(Of TSource), body As Action(Of TSource, ParallelLoopState)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
    'Public Shared Overloads Function ForEach(Of TSource)(source As OrderablePartitioner(Of TSource), body As Action(Of TSource, ParallelLoopState, Long)) As ParallelLoopResult': Data type(s) of the type parameter(s) cannot be inferred from these arguments. Specifying the data type(s) explicitly might correct this error.
        Parallel.ForEach(a, Sub(x As Object) Console.WriteLine(x))
                 ~~~~~~~
</expected>)
        End Sub

        <Fact(), WorkItem(542956, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542956")>
        Public Sub BC30518ERR_NoCallableOverloadCandidates2_trycatch()

            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndReferences(
<compilation name="ExitTryNotWithinTry">
    <file name="a.vb">
Imports System
Imports System.Linq
Class BaseClass
    Function Method() As String
        Dim x = New Integer() {}
        x.Where(Function(y)
                    Try
                        Exit Try
                    Catch ex1 As Exception When True
                        Exit Try
                    Finally
                        Exit Function
                    End Try
                    Return y = ""
                End Function)
        Return "x"
    End Function
End Class
Class DerivedClass
    Inherits BaseClass
    Shared Sub Main()
    End Sub
End Class
    </file>
</compilation>, {Net40.SystemCore})
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30101: Branching out of a 'Finally' is not valid.
                        Exit Function
                        ~~~~~~~~~~~~~
BC42353: Function '&lt;anonymous method>' doesn't return a value on all code paths. Are you missing a 'Return' statement?
                End Function)
                ~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30519ERR_NoNonNarrowingOverloadCandidates2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoNonNarrowingOverloadCandidates2">
        <file name="a.vb">
    Module Module1
        Class C0(Of T)
            Public whichOne As String
            Sub Foo(ByVal t1 As T)
                whichOne = "T"
            End Sub
            Default Property Prop1(ByVal t1 As T) As Integer
                Get
                    Return 100
                End Get
                Set(ByVal Value As Integer)
                    whichOne = "T"
                End Set
            End Property
        End Class
        Class C1(Of T, Y)
            Inherits C0(Of T)
            Overloads Sub Foo(ByVal y1 As Y)
                whichOne = "Y"
            End Sub
            Default Overloads Property Prop1(ByVal y1 As Y) As Integer
                Get
                    Return 200
                End Get
                Set(ByVal Value As Integer)
                    whichOne = "Y"
                End Set
            End Property
        End Class
        Structure S1
            Dim i As Integer
        End Structure
        Class Scenario11
            Public Shared Narrowing Operator CType(ByVal Arg As Scenario11) As C1(Of Integer, Integer)
                Return New C1(Of Integer, Integer)
            End Operator
            Public Shared Narrowing Operator CType(ByVal Arg As Scenario11) As S1
                Return New S1
            End Operator
        End Class
        Sub GenUnif0060()
            Dim iTmp As Integer = 2000
            Dim dTmp As Decimal = CDec(2000000)
            Dim tc2 As New C1(Of S1, C1(Of Integer, Integer))
            Dim tc3 As New C1(Of Short, Long)
            Dim sc11 As New Scenario11
            ' COMPILEERROR: BC30519,"Call tc2.Foo (New Scenario11)"
            Call tc2.Foo(New Scenario11)
            ' COMPILEERROR: BC30519,"Call tc2.Foo (sc11)"
            Call tc2.Foo(sc11)
            ' COMPILEERROR: BC30519,"Call tc3.Foo (dTmp)"
            Call tc3.Foo(dTmp)
            ' COMPILEERROR: BC30519,"tc2 (New Scenario11) = 1000"
            tc2(New Scenario11) = 1000
            ' COMPILEERROR: BC30519,"tc2 (New Scenario11)"
            iTmp = tc2(New Scenario11)
            ' COMPILEERROR: BC30519,"tc3 (dTmp) = 2000"
            tc3(dTmp) = 2000
            ' COMPILEERROR: BC30519,"tc3 (dTmp)"
            iTmp = tc3(dTmp)
        End Sub
    End Module
    </file>
    </compilation>)

            compilation.VerifyDiagnostics(
                    Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Overloads Sub Foo(y1 As Module1.C1(Of Integer, Integer))': Argument matching parameter 'y1' narrows from 'Module1.Scenario11' to 'Module1.C1(Of Integer, Integer)'.
    'Public Sub Foo(t1 As Module1.S1)': Argument matching parameter 't1' narrows from 'Module1.Scenario11' to 'Module1.S1'.]]>.Value.Replace(vbLf, Environment.NewLine)),
                    Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Overloads Sub Foo(y1 As Module1.C1(Of Integer, Integer))': Argument matching parameter 'y1' narrows from 'Module1.Scenario11' to 'Module1.C1(Of Integer, Integer)'.
    'Public Sub Foo(t1 As Module1.S1)': Argument matching parameter 't1' narrows from 'Module1.Scenario11' to 'Module1.S1'.]]>.Value.Replace(vbLf, Environment.NewLine)),
                    Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Overloads Sub Foo(y1 As Long)': Argument matching parameter 'y1' narrows from 'Decimal' to 'Long'.
    'Public Sub Foo(t1 As Short)': Argument matching parameter 't1' narrows from 'Decimal' to 'Short'.]]>.Value.Replace(vbLf, Environment.NewLine)),
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "tc2").WithArguments("Prop1", <![CDATA[
    'Public Overloads Default Property Prop1(y1 As Module1.C1(Of Integer, Integer)) As Integer': Argument matching parameter 'y1' narrows from 'Module1.Scenario11' to 'Module1.C1(Of Integer, Integer)'.
    'Public Default Property Prop1(t1 As Module1.S1) As Integer': Argument matching parameter 't1' narrows from 'Module1.Scenario11' to 'Module1.S1'.]]>.Value.Replace(vbLf, Environment.NewLine)),
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "tc2").WithArguments("Prop1", <![CDATA[
    'Public Overloads Default Property Prop1(y1 As Module1.C1(Of Integer, Integer)) As Integer': Argument matching parameter 'y1' narrows from 'Module1.Scenario11' to 'Module1.C1(Of Integer, Integer)'.
    'Public Default Property Prop1(t1 As Module1.S1) As Integer': Argument matching parameter 't1' narrows from 'Module1.Scenario11' to 'Module1.S1'.]]>.Value.Replace(vbLf, Environment.NewLine)),
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "tc3").WithArguments("Prop1", <![CDATA[
    'Public Overloads Default Property Prop1(y1 As Long) As Integer': Argument matching parameter 'y1' narrows from 'Decimal' to 'Long'.
    'Public Default Property Prop1(t1 As Short) As Integer': Argument matching parameter 't1' narrows from 'Decimal' to 'Short'.]]>.Value.Replace(vbLf, Environment.NewLine)),
            Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "tc3").WithArguments("Prop1", <![CDATA[
    'Public Overloads Default Property Prop1(y1 As Long) As Integer': Argument matching parameter 'y1' narrows from 'Decimal' to 'Long'.
    'Public Default Property Prop1(t1 As Short) As Integer': Argument matching parameter 't1' narrows from 'Decimal' to 'Short'.]]>.Value.Replace(vbLf, Environment.NewLine))
                )
        End Sub

        <Fact()>
        Public Sub BC30520ERR_ArgumentNarrowing3_RoslynBC30519()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ArgumentNarrowing3">
        <file name="a.vb">
    Option Strict Off
    Module Module1
        Class sample7C1(Of X)
            Enum E
                e1
                e2
                e3
            End Enum
        End Class
        Class sample7C2(Of T, Y)
            Public whichOne As String
            Sub Foo(ByVal p1 As sample7C1(Of T).E)
                whichOne = "1"
            End Sub
            Sub Foo(ByVal p1 As sample7C1(Of Y).E)
                whichOne = "2"
            End Sub
            Sub Scenario8(ByVal p1 As sample7C1(Of T).E)
                Call Me.Foo(p1)
            End Sub
        End Class
        Sub test()
            Dim tc7 As New sample7C2(Of Integer, Integer)
            Dim sc7 As New sample7C1(Of Byte)
             'COMPILEERROR: BC30520, "sample7C1(Of Long).E.e1"
            Call tc7.Foo (sample7C1(Of Long).E.e1)
            'COMPILEERROR: BC30520, "sample7C1(Of Short).E.e2"
            Call tc7.Foo (sample7C1(Of Short).E.e2)
            'COMPILEERROR: BC30520, "sc7.E.e3"
            Call tc7.Foo (sc7.E.e3)
        End Sub
    End Module
    </file>
    </compilation>)
            ' BC0000 - Test bug

            ' Roslyn BC30519 - Dev11 BC30520
            compilation.VerifyDiagnostics(
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Long).E' to 'Module1.sample7C1(Of Integer).E'.
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Long).E' to 'Module1.sample7C1(Of Integer).E'.]]>.Value.Replace(vbLf, Environment.NewLine)),
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Short).E' to 'Module1.sample7C1(Of Integer).E'.
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Short).E' to 'Module1.sample7C1(Of Integer).E'.]]>.Value.Replace(vbLf, Environment.NewLine)),
             Diagnostic(ERRID.WRN_SharedMemberThroughInstance, "sc7.E"),
             Diagnostic(ERRID.ERR_NoNonNarrowingOverloadCandidates2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Byte).E' to 'Module1.sample7C1(Of Integer).E'.
    'Public Sub Foo(p1 As Module1.sample7C1(Of Integer).E)': Argument matching parameter 'p1' narrows from 'Module1.sample7C1(Of Byte).E' to 'Module1.sample7C1(Of Integer).E'.]]>.Value.Replace(vbLf, Environment.NewLine))
                    )

            'CompilationUtils.AssertTheseErrors(compilation,
            '    <expected>
            'BC30520: Argument matching parameter 'p1' narrows from 'ConsoleApplication10.Module1.sample7C1(Of Long).E' to 'ConsoleApplication10.Module1.sample7C1(Of Integer).E'.	
            '            Call tc7.Foo (sample7C1(Of Long).E.e1)
            '                ~~~~
            'BC30520: Argument matching parameter 'p1' narrows from 'ConsoleApplication10.Module1.sample7C1(Of Short).E' to 'ConsoleApplication10.Module1.sample7C1(Of Integer).E'.	
            '            Call tc7.Foo (sample7C1(Of Short).E.e2)
            '                ~~~~
            'BC30520: Argument matching parameter 'p1' narrows from 'ConsoleApplication10.Module1.sample7C1(Of Byte).E' to 'ConsoleApplication10.Module1.sample7C1(Of Integer).E'.	
            '                Call tc7.Foo (sc7.E.e3)
            '                ~~~~
            '</expected>)
        End Sub

        <Fact()>
        Public Sub BC30521ERR_NoMostSpecificOverload2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoMostSpecificOverload2">
        <file name="a.vb">
            Module Module1
                Class C0(Of T)
                    Sub Foo(ByVal t1 As T)
                    End Sub
                End Class
                Class C1(Of T, Y)
                    Inherits C0(Of T)
                    Overloads Sub Foo(ByVal y1 As Y)
                    End Sub
                End Class
                Structure S1
                    Dim i As Integer
                End Structure
                Class C2
                    Public Shared Widening Operator CType(ByVal Arg As C2) As C1(Of Integer, Integer)
                        Return New C1(Of Integer, Integer)
                    End Operator
                    Public Shared Widening Operator CType(ByVal Arg As C2) As S1
                        Return New S1
                    End Operator
                End Class
                Sub test()
                    Dim C As New C1(Of S1, C1(Of Integer, Integer))
                    Call C.Foo(New C2)
                End Sub
            End Module
    </file>
    </compilation>)

            compilation.VerifyDiagnostics(Diagnostic(ERRID.ERR_NoMostSpecificOverload2, "Foo").WithArguments("Foo", <![CDATA[
    'Public Overloads Sub Module1.C1(Of Module1.S1, Module1.C1(Of Integer, Integer)).Foo(y1 As Module1.C1(Of Integer, Integer))': Not most specific.
    'Public Sub Module1.C0(Of Module1.S1).Foo(t1 As Module1.S1)': Not most specific.]]>.Value.Replace(vbLf, Environment.NewLine))
                    )

        End Sub

        <Fact()>
        Public Sub BC30524ERR_NoGetProperty1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            WriteOnly Property P
                Set
                End Set
            End Property
            Shared WriteOnly Property Q
                Set
                End Set
            End Property
            Sub M()
                Dim o
                o = P
                o = Me.P
                o = Q
                o = C.Q
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30524: Property 'P' is 'WriteOnly'.
                o = P
                    ~
BC30524: Property 'P' is 'WriteOnly'.
                o = Me.P
                    ~~~~
BC30524: Property 'Q' is 'WriteOnly'.
                o = Q
                    ~
BC30524: Property 'Q' is 'WriteOnly'.
                o = C.Q
                    ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30524ERR_NoGetProperty1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Interface I
            Default WriteOnly Property P(s As String)
        End Interface
        Structure S
            Default WriteOnly Property Q(s As String)
                Set(value)
                End Set
            End Property
        End Structure
        Class C
            Default WriteOnly Property R(s As String)
                Set(value)
                End Set
            End Property
            Shared Sub M(x As I, y As S, z As C)
                x!Q = x!R
                y!Q = y!R
                z!Q = z!R
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30524: Property 'P' is 'WriteOnly'.
                x!Q = x!R
                      ~~~
BC30524: Property 'Q' is 'WriteOnly'.
                y!Q = y!R
                      ~~~
BC30524: Property 'R' is 'WriteOnly'.
                z!Q = z!R
                      ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30524ERR_NoGetProperty1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
        Module M1
            Sub Main()
                foo(p)
            End Sub
            WriteOnly Property p() As Single
                Set(ByVal Value As Single)
                End Set
            End Property
            Public Sub foo(ByRef x As Single)
            End Sub
        End Module
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30524: Property 'p' is 'WriteOnly'.
                foo(p)
                    ~
</expected>)
        End Sub

        <Fact(), WorkItem(6810, "DevDiv_Projects/Roslyn")>
        Public Sub BC30524ERR_NoGetProperty1_3()
            CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            WriteOnly Property P As Object()
                Set(value As Object())
                End Set
            End Property
            WriteOnly Property Q As System.Action(Of Object)
                Set(value As System.Action(Of Object))
                End Set
            End Property
            WriteOnly Property R As C
                Set(value As C)
                End Set
            End Property
            Default ReadOnly Property S(i As Integer)
                Get
                    Return Nothing
                End Get
            End Property
            Sub M()
                Dim o
                o = P()(1)
                o = Q()(2)
                o = R()(3)
                o = P(1)
                o = Q(2)
                o = R(3)
            End Sub
        End Class
        </file>
    </compilation>).VerifyDiagnostics(
        Diagnostic(ERRID.ERR_NoGetProperty1, "P()").WithArguments("P"),
        Diagnostic(ERRID.ERR_NoGetProperty1, "Q()").WithArguments("Q"),
        Diagnostic(ERRID.ERR_NoGetProperty1, "R()").WithArguments("R"),
        Diagnostic(ERRID.ERR_NoGetProperty1, "P").WithArguments("P"),
        Diagnostic(ERRID.ERR_NoGetProperty1, "Q").WithArguments("Q"),
        Diagnostic(ERRID.ERR_NoGetProperty1, "R").WithArguments("R"))

        End Sub

        ''' <summary>
        ''' Report BC30524 even in cases when the
        ''' expression will be ignored.
        ''' </summary>
        ''' <remarks></remarks>
        <Fact()>
        Public Sub BC30524ERR_NoGetProperty1_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Class A
    Class B
        Friend Const F As Object = Nothing
    End Class
    Shared WriteOnly Property P As A
        Set(value As A)
        End Set
    End Property
    Shared ReadOnly Property Q As A
        Get
            Return Nothing
        End Get
    End Property
    Shared Sub M()
        Dim o As Object
        o = P.B.F
        o = Q.B.F
    End Sub
End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30524: Property 'P' is 'WriteOnly'.
        o = P.B.F
            ~
BC42025: Access of shared member, constant member, enum member or nested type through an instance; qualifying expression will not be evaluated.
        o = Q.B.F
            ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30526ERR_NoSetProperty1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Class C
            ReadOnly Property P
                Get
                    Return Nothing
                End Get
            End Property
            Shared ReadOnly Property Q
                Get
                    Return Nothing
                End Get
            End Property
            Sub M()
                P = Nothing
                Me.P = Nothing
                Q = Nothing
                C.Q = Nothing
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30526: Property 'P' is 'ReadOnly'.
                P = Nothing
                ~~~~~~~~~~~
BC30526: Property 'P' is 'ReadOnly'.
                Me.P = Nothing
                ~~~~~~~~~~~~~~
BC30526: Property 'Q' is 'ReadOnly'.
                Q = Nothing
                ~~~~~~~~~~~
BC30526: Property 'Q' is 'ReadOnly'.
                C.Q = Nothing
                ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30526ERR_NoSetProperty1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
        Interface I
            Default ReadOnly Property P(s As String)
        End Interface
        Structure S
            Default ReadOnly Property Q(s As String)
                Get
                    Return Nothing
                End Get
            End Property
        End Structure
        Class C
            Default ReadOnly Property R(s As String)
                Get
                    Return Nothing
                End Get
            End Property
            Shared Sub M(x As I, y As S, z As C)
                x!Q = x!R
                y!Q = y!R
                z!Q = z!R
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30526: Property 'P' is 'ReadOnly'.
                x!Q = x!R
                ~~~~~~~~~
BC30526: Property 'Q' is 'ReadOnly'.
                y!Q = y!R
                ~~~~~~~~~
BC30526: Property 'R' is 'ReadOnly'.
                z!Q = z!R
                ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30532ERR_DateToDoubleConversion()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DateToDoubleConversion">
        <file name="a.vb">
        Structure s1
            function foo() as double
                return #1/1/2000#
            End function
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30532: Conversion from 'Date' to 'Double' requires calling the 'Date.ToOADate' method.
                return #1/1/2000#
                       ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30532ERR_DateToDoubleConversion_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DateToDoubleConversion">
        <file name="a.vb">
Imports System
Class C
    Shared Sub Main()
        For Each x As Double In New Date() {#12:00:00 AM#}
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30532: Conversion from 'Date' to 'Double' requires calling the 'Date.ToOADate' method.
        For Each x As Double In New Date() {#12:00:00 AM#}
                                ~~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30533ERR_DoubleToDateConversion()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DoubleToDateConversion">
        <file name="a.vb">
        Structure s1
            Function foo() As Date
                Dim a As Double = 12
                Return a
            End Function
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30533: Conversion from 'Double' to 'Date' requires calling the 'Date.FromOADate' method.
                Return a
                       ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30542ERR_ZeroDivide()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="DateToDoubleConversion">
        <file name="a.vb">
                Module M1
                    'Const z = 0
                    Sub foo()
                        Dim s = 1 \ Nothing
                        Dim m = 1 \ 0
                        'Dim n = 1 \ z
                        If (1 \ 0 = 1) Then
                        End If
                    End Sub
                End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30542: Division by zero occurred while evaluating this expression.
                        Dim s = 1 \ Nothing
                                ~~~~~~~~~~~
BC30542: Division by zero occurred while evaluating this expression.
                        Dim m = 1 \ 0
                                ~~~~~
BC30542: Division by zero occurred while evaluating this expression.
                        If (1 \ 0 = 1) Then
                            ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30545ERR_PropertyAccessIgnored()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="PropertyAccessIgnored">
        <file name="a.vb">
        Class C
            Shared Property P
            ReadOnly Property Q
                Get
                    Return Nothing
                End Get
            End Property
            Property R(o)
                Get
                    Return Nothing
                End Get
                Set(value)
                End Set
            End Property
            Sub M(o)
                P
                M(P)
                C.P
                C.P = Nothing
                Q
                M(Q)
                Me.Q
                M(Me.Q)
                R(Nothing)
                R(Nothing) = Nothing
                Me.R(Nothing)
                M(Me.R(Nothing))
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30545: Property access must assign to the property or use its value.
                P
                ~
BC30545: Property access must assign to the property or use its value.
                C.P
                ~~~
BC30545: Property access must assign to the property or use its value.
                Q
                ~
BC30545: Property access must assign to the property or use its value.
                Me.Q
                ~~~~
BC30545: Property access must assign to the property or use its value.
                R(Nothing)
                ~~~~~~~~~~
BC30545: Property access must assign to the property or use its value.
                Me.R(Nothing)
                ~~~~~~~~~~~~~
</expected>)
        End Sub

        <WorkItem(531311, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/531311")>
        <Fact()>
        Public Sub BC30545ERR_PropertyAccessIgnored_Latebound()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="PropertyAccessIgnored">
        <file name="a.vb">
Structure s1
    Dim z As Integer
    Property foo(ByVal i As Integer)
        Get
            Return Nothing
        End Get
        Set(ByVal Value)
        End Set
    End Property

    Property foo(ByVal i As Double)
        Get
            Return Nothing
        End Get
        Set(ByVal Value)
        End Set
    End Property

    Sub goo()
        Dim o As Object = 1
        'COMPILEERROR: BC30545, "foo(o)"
        foo(o)
    End Sub
End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30545: Property access must assign to the property or use its value.
        foo(o)
        ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30547ERR_InterfaceNoDefault1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Namespace N
            Interface I
            End Interface
            Class C
                Shared Sub M(x As I)
                    N(x(0))
                    N(x!P)
                End Sub
                Shared Sub N(o As Object)
                End Sub
            End Class
        End Namespace
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30547: 'I' cannot be indexed because it has no default property.
                    N(x(0))
                      ~
BC30547: 'I' cannot be indexed because it has no default property.
                    N(x!P)
                      ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30554ERR_AmbiguousInUnnamedNamespace1()
            Dim Lib1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="App1">
        <file name="a.vb">
            Public Class C1
            End Class
        </file>
    </compilation>)
            Dim Lib2 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="App2">
        <file name="a.vb">
            Public Class C1
            End Class
        </file>
    </compilation>)
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="APP">
        <file name="a.vb">
            Imports System
            Module Module1
                Sub Main()
                    Dim obj = New C1()
                End Sub
            End Module
        </file>
    </compilation>)
            Dim ref1 = New VisualBasicCompilationReference(Lib1)
            Dim ref2 = New VisualBasicCompilationReference(Lib2)
            compilation1 = compilation1.AddReferences(ref1)
            compilation1 = compilation1.AddReferences(ref2)
            Dim expectedErrors1 = <errors>
BC30554: 'C1' is ambiguous.
                    Dim obj = New C1()
                                  ~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30555ERR_DefaultMemberNotProperty1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
Imports System
Imports System.Collections.Generic
Class C
    Shared Sub M(a As Array, f As Func(Of Dictionary(Of Object, Object)))
        Dim o As Object
        o = Function()
                Return New Dictionary(Of String, String)
            End Function!a
        o = a!b
        o = f!c
        o = f()!d
    End Sub
End Class
        </file>
    </compilation>)
            ' For now, lambdas result in BC30491 which differs from Dev10.
            ' This should change once lambda support is complete.
            Dim expectedErrors1 = <errors>
BC30555: Default member of 'Function &lt;generated method&gt;() As Dictionary(Of String, String)' is not a property.
        o = Function()
            ~~~~~~~~~~~
BC30555: Default member of 'Array' is not a property.
        o = a!b
            ~
BC30555: Default member of 'Func(Of Dictionary(Of Object, Object))' is not a property.
        o = f!c
            ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30565ERR_ArrayInitializerTooFewDimensions()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ArrayInitializerTooFewDimensions">
        <file name="a.vb">
        Module Module1
            Sub test()
                Dim FixedRankArray_1(,) As Double
                'COMPILEERROR: BC30565, "(0", BC30198, ","
                FixedRankArray_1 = New Double(,) {(0.1), {2.4, 4.6}}
                Exit Sub
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30565: Array initializer has too few dimensions.
                FixedRankArray_1 = New Double(,) {(0.1), {2.4, 4.6}}
                                                  ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30566ERR_ArrayInitializerTooManyDimensions()
            CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ArrayInitializerTooManyDimensions">
        <file name="a.vb">
        Module Module1
            Structure S1
                Public x As Long
                Public s As String
            End Structure
            Sub foo()
                Dim obj = New S1() {{1, "one"}}
                Exit Sub
            End Sub
        End Module
    </file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_ArrayInitializerTooManyDimensions, "{1, ""one""}"))

        End Sub

        ' Roslyn too many extra errors (last 4)
        <Fact()>
        Public Sub BC30566ERR_ArrayInitializerTooManyDimensions_1()
            CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ArrayInitializerTooManyDimensions">
        <file name="a.vb">
        Module Module1
            Sub foo()
                Dim myArray As Integer(,) = New Integer(2, 1) {{{1, 2}, {3, 4}, {5, 6}}}
            End Sub
        End Module
    </file>
    </compilation>).VerifyDiagnostics(
    Diagnostic(ERRID.ERR_ArrayInitializerTooManyDimensions, "{1, 2}"),
    Diagnostic(ERRID.ERR_ArrayInitializerTooManyDimensions, "{3, 4}"),
    Diagnostic(ERRID.ERR_ArrayInitializerTooManyDimensions, "{5, 6}"),
    Diagnostic(ERRID.ERR_InitializerTooManyElements1, "{{1, 2}, {3, 4}, {5, 6}}").WithArguments("1"),
    Diagnostic(ERRID.ERR_InitializerTooFewElements1, "{{{1, 2}, {3, 4}, {5, 6}}}").WithArguments("2"))

        End Sub

        <Fact()>
        Public Sub BC30567ERR_InitializerTooFewElements1()
            CreateCompilationWithMscorlib40(
    <compilation name="InitializerTooFewElements1">
        <file name="a.vb">
        Class A
            Sub foo()
                Dim x = {{1, 2, 3}, {4, 5}}
            End Sub
        End Class
    </file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_InitializerTooFewElements1, "{4, 5}").WithArguments("1"))

        End Sub

        <Fact()>
        Public Sub BC30567ERR_InitializerTooFewElements1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InitializerTooFewElements1">
        <file name="a.vb">
        Class A
            Sub foo()
                Dim x127 As Object(,) = New System.Exception(,) {{New System.AccessViolationException, New System.ArgumentException}, {New System.Exception}}
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30567: Array initializer is missing 1 elements.
                Dim x127 As Object(,) = New System.Exception(,) {{New System.AccessViolationException, New System.ArgumentException}, {New System.Exception}}
                                                                                                                                      ~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30567ERR_InitializerTooFewElements1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation>
        <file name="a.vb">
Class C
    Private A() As Object = New Object(0) {}
    Private B As Object() = New Object(2) {}
    Private C As Object(,) = New Object(1, 0) {}
    Private D As Object(,) = New Object(1, 0) {{}, {2}}
    Private E As Object(,) = New Object(0, 2) {}
    Private F()() As Object = New Object(0)() {}
    Private G()() As Object = New Object(0)() {New Object(0) {}}
End Class
    </file>
    </compilation>)
            compilation.AssertTheseDiagnostics(
    <expected>
BC30567: Array initializer is missing 1 elements.
    Private D As Object(,) = New Object(1, 0) {{}, {2}}
                                               ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30568ERR_InitializerTooManyElements1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InitializerTooManyElements1">
        <file name="a.vb">
        Class A
            Sub foo()
                Dim x127 As Object(,) = New System.Exception(,) {{New System.AccessViolationException}, {New System.Exception, New System.ArgumentException}}
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30568: Array initializer has 1 too many elements.
                Dim x127 As Object(,) = New System.Exception(,) {{New System.AccessViolationException}, {New System.Exception, New System.ArgumentException}}
                                                                                                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30569ERR_NewOnAbstractClass()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NewOnAbstractClass">
        <file name="a.vb">
        Class C1
            MustInherit Class C2
                Public foo As New C2()
            End Class
            Public foo As New C2()
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30569: 'New' cannot be used on a class that is declared 'MustInherit'.
                Public foo As New C2()
                              ~~~~~~~~
BC30569: 'New' cannot be used on a class that is declared 'MustInherit'.
            Public foo As New C2()
                          ~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30574ERR_StrictDisallowsLateBinding()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="StrictDisallowsLateBinding">
        <file name="a.vb">
            Module Module1
                Dim bol As Boolean
                Class C1
                    Property Prop As Long
                End Class
                Sub foo()
                    Dim Obj As Object = New C1()
                    bol = Obj(1)
                    bol = Obj!P
                End Sub
            End Module
    </file>
    </compilation>, TestOptions.ReleaseDll.WithOptionStrict(OptionStrict.On))
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30574: Option Strict On disallows late binding.
                    bol = Obj(1)
                          ~~~
BC30574: Option Strict On disallows late binding.
                    bol = Obj!P
                          ~~~~~
</expected>)

            compilation = compilation.WithOptions(compilation.Options.WithOptionStrict(OptionStrict.Custom))

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42017: Late bound resolution; runtime errors could occur.
                    bol = Obj(1)
                          ~~~
BC42016: Implicit conversion from 'Object' to 'Boolean'.
                    bol = Obj(1)
                          ~~~~~~
BC42016: Implicit conversion from 'Object' to 'Boolean'.
                    bol = Obj!P
                          ~~~~~
BC42017: Late bound resolution; runtime errors could occur.
                    bol = Obj!P
                          ~~~~~
</expected>)
        End Sub

        <WorkItem(546763, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/546763")>
        <Fact()>
        Public Sub BC30574ERR_StrictDisallowsLateBinding_16745()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Option Strict On
Option Infer On

Public Module Program
    Sub Main()
        Dim a As Object = Nothing
        a.DoSomething()
    End Sub
End Module
    </file>
    </compilation>, TestOptions.ReleaseDll.WithOptionStrict(OptionStrict.On))
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30574: Option Strict On disallows late binding.
        a.DoSomething()
        ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30574ERR_StrictDisallowsLateBinding1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="StrictDisallowsLateBinding">
        <file name="a.vb">
Imports System

Module Program

    Delegate Sub d1(ByRef x As Integer, y As Integer)

    Sub Main()
        Dim obj As Object = New cls1

        Dim o As d1 = AddressOf obj.foo

        Dim l As Integer = 0
        o(l, 2)

        Console.WriteLine(l)
    End Sub

    Class cls1
        Shared Sub foo(ByRef x As Integer, y As Integer)
            x = 42
            Console.WriteLine(x + y)
        End Sub
    End Class
End Module

    </file>
    </compilation>, TestOptions.ReleaseExe.WithOptionStrict(OptionStrict.On))

            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30574: Option Strict On disallows late binding.
        Dim o As d1 = AddressOf obj.foo
                                ~~~~~~~
</expected>)

            compilation = compilation.WithOptions(compilation.Options.WithOptionStrict(OptionStrict.Custom))

            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42017: Late bound resolution; runtime errors could occur.
        Dim o As d1 = AddressOf obj.foo
                                ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30577ERR_AddressOfOperandNotMethod()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AddressOfOperandNotMethod">
        <file name="a.vb">
            Delegate Function MyDelegate()
            Module M1
                Enum MyEnum
                    One
                End Enum
                Sub Main()
                    Dim x As MyDelegate
                    Dim oEnum As MyEnum
                    x = AddressOf oEnum
                    x.Invoke()
                End Sub
            End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30577: 'AddressOf' operand must be the name of a method (without parentheses).
                    x = AddressOf oEnum
                                  ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30581ERR_AddressOfNotDelegate1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AddressOfNotDelegate1">
        <file name="a.vb">
Module M
    Sub Main()
        Dim x = New Object()
        Dim f = AddressOf x.GetType
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30581: 'AddressOf' expression cannot be converted to 'Object' because 'Object' is not a delegate type.
        Dim f = AddressOf x.GetType
                ~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30582ERR_SyncLockRequiresReferenceType1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="SyncLockRequiresReferenceType1">
        <file name="a.vb">
                Imports System
                Module M1
                    Class C
                        Private Shared count = 0
                        Sub IncrementCount()
                            Dim i As Integer
                            SyncLock i = 0
                                count = count + 1
                            End SyncLock
                        End Sub
                    End Class
                End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30582: 'SyncLock' operand cannot be of type 'Boolean' because 'Boolean' is not a reference type.
                            SyncLock i = 0
                                     ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30582ERR_SyncLockRequiresReferenceType1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="SyncLockRequiresReferenceType1">
        <file name="a.vb">
Imports System
Module Program
    Sub Main(args As String())
        Dim S1_a As New Object()
        Dim S1_b As Integer? = 4
        Dim S1_c As Integer? = 41
        SyncLock If(False, S1_a, S1_a)
        End SyncLock
        SyncLock If(True, S1_b, S1_b)
        End SyncLock
        SyncLock If(False, S1_b, 1)
        End SyncLock
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30582: 'SyncLock' operand cannot be of type 'Integer?' because 'Integer?' is not a reference type.
        SyncLock If(True, S1_b, S1_b)
                 ~~~~~~~~~~~~~~~~~~~~
BC30582: 'SyncLock' operand cannot be of type 'Integer?' because 'Integer?' is not a reference type.
        SyncLock If(False, S1_b, 1)
                 ~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30582ERR_SyncLockRequiresReferenceType1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="SyncLockRequiresReferenceType1">
        <file name="a.vb">
Interface I
End Interface
Class A
End Class
Class C
    Shared Sub M(Of T1, T2 As Class, T3 As Structure, T4 As New, T5 As I, T6 As A, T7 As U, U)(_1 As T1, _2 As T2, _3 As T3, _4 As T4, _5 As T5, _6 As T6, _7 As T7)
        SyncLock _1
        End SyncLock
        SyncLock _2
        End SyncLock
        SyncLock _3
        End SyncLock
        SyncLock _4
        End SyncLock
        SyncLock _5
        End SyncLock
        SyncLock _6
        End SyncLock
        SyncLock _7
        End SyncLock
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30582: 'SyncLock' operand cannot be of type 'T1' because 'T1' is not a reference type.
        SyncLock _1
                 ~~
BC30582: 'SyncLock' operand cannot be of type 'T3' because 'T3' is not a reference type.
        SyncLock _3
                 ~~
BC30582: 'SyncLock' operand cannot be of type 'T4' because 'T4' is not a reference type.
        SyncLock _4
                 ~~
BC30582: 'SyncLock' operand cannot be of type 'T5' because 'T5' is not a reference type.
        SyncLock _5
                 ~~
BC30582: 'SyncLock' operand cannot be of type 'T7' because 'T7' is not a reference type.
        SyncLock _7
                 ~~
</expected>)
        End Sub

        <Fact>
        Public Sub BC30587ERR_NamedParamArrayArgument()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NamedParamArrayArgument">
        <file name="a.vb">
                Class C1
                    Shared Sub Main()
                        Dim a As New C1
                        a.abc(s:=10)
                    End Sub
                    Sub abc(ByVal ParamArray s() As Integer)
                    End Sub
                End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30587: Named argument cannot match a ParamArray parameter.
                        a.abc(s:=10)
                              ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30588ERR_OmittedParamArrayArgument()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="OmittedParamArrayArgument">
        <file name="a.vb">
Imports System
Imports C1(Of String, Integer)
Class C1(Of T As {Class}, U As Structure)
    Public Shared Property Overloaded(ByVal ParamArray y() As Exception) As C2
        Get
            Return New C2
        End Get
        Set(ByVal value As C2)
        End Set
    End Property
End Class
Class C2
End Class
Module M1
    Sub FOO()
        Overloaded(, , ) = Nothing
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30588: Omitted argument cannot match a ParamArray parameter.
        Overloaded(, , ) = Nothing
                   ~
BC30588: Omitted argument cannot match a ParamArray parameter.
        Overloaded(, , ) = Nothing
                     ~
BC30588: Omitted argument cannot match a ParamArray parameter.
        Overloaded(, , ) = Nothing
                       ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30611ERR_NegativeArraySize()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NegativeArraySize">
        <file name="a.vb">
        Class C1
            Sub foo()
                Dim x8(-2) As String
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30611: Array dimensions cannot have a negative size.
                Dim x8(-2) As String
                       ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30611ERR_NegativeArraySize_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NegativeArraySize">
        <file name="a.vb">
        Class C1
            Sub foo()
                Dim arr11 As Integer(,) = New Integer(-2, -2) {} ' Invalid
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30611: Array dimensions cannot have a negative size.
                Dim arr11 As Integer(,) = New Integer(-2, -2) {} ' Invalid
                                                      ~~
BC30611: Array dimensions cannot have a negative size.
                Dim arr11 As Integer(,) = New Integer(-2, -2) {} ' Invalid
                                                          ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30611ERR_NegativeArraySize_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NegativeArraySize">
        <file name="a.vb">
        Class C1
            Sub foo()
                Dim arr(0 To 0, 0 To -2) As Integer 'Invalid
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30611: Array dimensions cannot have a negative size.
                Dim arr(0 To 0, 0 To -2) As Integer 'Invalid
                                ~~~~~~~

</expected>)
        End Sub

        <Fact()>
        Public Sub BC30614ERR_MyClassAbstractCall1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30614ERR_MyClassAbstractCall1_1">
        <file name="a.vb">
        MustInherit Class C1
            Public Sub UseMyClass()
                MyClass.foo()
            End Sub
            MustOverride Sub foo()
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30614: 'MustOverride' method 'Public MustOverride Sub foo()' cannot be called with 'MyClass'.
                MyClass.foo()
                ~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30614ERR_MyClassAbstractCall1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30614ERR_MyClassAbstractCall1_2">
        <file name="a.vb">
        Public MustInherit Class Base1
            Public MustOverride Property P1()

            Public Sub M2()
                MyClass.P1 = MyClass.P1
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30614: 'MustOverride' method 'Public MustOverride Property P1 As Object' cannot be called with 'MyClass'.
                MyClass.P1 = MyClass.P1
                ~~~~~~~~~~
BC30614: 'MustOverride' method 'Public MustOverride Property P1 As Object' cannot be called with 'MyClass'.
                MyClass.P1 = MyClass.P1
                             ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30614ERR_MyClassAbstractCall1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30614ERR_MyClassAbstractCall1_3">
        <file name="a.vb">
Imports System
Public MustInherit Class Base1
    Public MustOverride Function F1() As Integer

    Public Sub M2()
        Dim _func As Func(Of Integer) = AddressOf MyClass.F1
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30614: 'MustOverride' method 'Public MustOverride Function F1() As Integer' cannot be called with 'MyClass'.
        Dim _func As Func(Of Integer) = AddressOf MyClass.F1
                                                  ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30614ERR_MyClassAbstractCall1_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30614ERR_MyClassAbstractCall1_4">
        <file name="a.vb">
Imports System
Public MustInherit Class Base1
    Public MustOverride Function F1() As Integer
    Public Function F2() As Integer
        Return 1
    End Function

    Public Sub M2()
        Dim _func As Func(Of Func(Of Integer)) = Function() New Func(Of Integer)(AddressOf MyClass.F1)
        Dim _func2 As Func(Of Func(Of Integer)) = Function() New Func(Of Integer)(AddressOf MyClass.F2)
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30614: 'MustOverride' method 'Public MustOverride Function F1() As Integer' cannot be called with 'MyClass'.
        Dim _func As Func(Of Func(Of Integer)) = Function() New Func(Of Integer)(AddressOf MyClass.F1)
                                                                                           ~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30614ERR_MyClassAbstractCall1_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BC30614ERR_MyClassAbstractCall1_5">
        <file name="a.vb">
Imports System
Public MustInherit Class Base1
    Public MustOverride Function F1() As Integer
    Public Function F2() As Integer
        Return 1
    End Function

    Public FLD As Func(Of Func(Of Integer)) =
        Function() New Func(Of Integer)(AddressOf MyClass.F1)

    Public Property PROP As Func(Of Func(Of Integer)) =
        Function() New Func(Of Integer)(AddressOf MyClass.F1)

    Public FLD2 As Func(Of Func(Of Integer)) =
        Function() New Func(Of Integer)(AddressOf MyClass.F2)

    Public Property PROP2 As Func(Of Func(Of Integer)) =
        Function() New Func(Of Integer)(AddressOf MyClass.F2)

End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30614: 'MustOverride' method 'Public MustOverride Function F1() As Integer' cannot be called with 'MyClass'.
        Function() New Func(Of Integer)(AddressOf MyClass.F1)
                                                  ~~~~~~~~~~
BC30614: 'MustOverride' method 'Public MustOverride Function F1() As Integer' cannot be called with 'MyClass'.
        Function() New Func(Of Integer)(AddressOf MyClass.F1)
                                                  ~~~~~~~~~~
</expected>)
        End Sub

        ' Different error
        <Fact()>
        Public Sub BC30615ERR_EndDisallowedInDllProjects()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="EndDisallowedInDllProjects">
        <file name="a.vb">
        Class C1
            Function foo()
                End
        End Class
    </file>
    </compilation>, options:=TestOptions.ReleaseDll)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30027: 'End Function' expected.
            Function foo()
            ~~~~~~~~~~~~~~
BC30615: 'End' statement cannot be used in class library projects.
                End
                ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
        Class C1
            Sub foo()
                dim s = 10
                if s>5
                    dim s = 5
                    if s > 7
                        dim s = 7
                    End If
                End If
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30616: Variable 's' hides a variable in an enclosing block.
                    dim s = 5
                        ~
BC30616: Variable 's' hides a variable in an enclosing block.
                        dim s = 7
                            ~
</expected>)
        End Sub

        ' spec changes in Roslyn
        <WorkItem(528680, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528680")>
        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Class C
    Private field As Integer = 0
    Private Property [property]() As Integer
        Get
            Return m_property
        End Get
        Set(value As Integer)
            m_property = value
        End Set
    End Property
    Property prop() As Integer
        Get
            Return 1
        End Get
        Set(ByVal Value As Integer)
            ' Was Dev10: COMPILEERROR: BC30616, "value"
            ' now Dev10: BC30734: 'value' is already declared as a parameter of this method.
            For Each value As Byte In New Byte() {1, 2, 3}
            Next
        End Set
    End Property
    Private m_property As Integer

    Shared Sub Main()
        Dim ints As Integer() = New Integer() {1, 2, 3}
        Dim strings As String() = New String() {"1", "2", "3"}
        Dim conflict As Integer = 1
        For Each field As Integer In ints
        Next
        For Each [property] As String In strings
        Next
        For Each conflict As String In strings
        Next
        Dim [qq] As Integer = 23
        'COMPILEERROR: BC30616, "qq"
        For Each qq As Integer In New Integer() {1, 2, 3}
        Next
        Dim ww As Integer = 23
        'COMPILEERROR: BC30616, "[ww]"
        For Each [ww] As Integer In New Integer() {1, 2, 3}
        Next
        For Each z As Integer In New Integer() {1, 2, 3}
            'COMPILEERROR: BC30616, "z"
            For Each z As Decimal In New Decimal() {1, 2, 3}
            Next
        Next
        For Each t As Long In New Long() {1, 2, 3}
            For Each u As Boolean In New Boolean() {False, True}
                'COMPILEERROR: BC30616, "t"
                Dim t As Integer
            Next
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30734: 'value' is already declared as a parameter of this method.
            For Each value As Byte In New Byte() {1, 2, 3}
                     ~~~~~
BC30616: Variable 'conflict' hides a variable in an enclosing block.
        For Each conflict As String In strings
                 ~~~~~~~~
BC30616: Variable 'qq' hides a variable in an enclosing block.
        For Each qq As Integer In New Integer() {1, 2, 3}
                 ~~
BC30616: Variable 'ww' hides a variable in an enclosing block.
        For Each [ww] As Integer In New Integer() {1, 2, 3}
                 ~~~~
BC30616: Variable 'z' hides a variable in an enclosing block.
            For Each z As Decimal In New Decimal() {1, 2, 3}
                     ~
BC30616: Variable 't' hides a variable in an enclosing block.
                Dim t As Integer
                    ~
BC42024: Unused local variable: 't'.
                Dim t As Integer
                    ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Option Strict Off
Option Infer On
Public Class MyClass1
    Public Shared Sub Main()
        Dim var1 As Integer
        For var1 As Integer = 1 To 10
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42024: Unused local variable: 'var1'.
        Dim var1 As Integer
            ~~~~
BC30616: Variable 'var1' hides a variable in an enclosing block.
        For var1 As Integer = 1 To 10
            ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        Static var2 As Long
        For var2 As Short = 0 To 10
        Next

       var2 = 0
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30616: Variable 'var2' hides a variable in an enclosing block.
        For var2 As Short = 0 To 10
            ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_5()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        For varo As Integer = 0 To 10
            For varo As Integer = 0 To 10
            Next
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30616: Variable 'varo' hides a variable in an enclosing block.
            For varo As Integer = 0 To 10
                ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_6()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        For varo As Integer = 0 To 10
            Dim [qqq] As Integer
            For qqq As Integer = 0 To 10
            Next
            Dim www As Integer
            For [www] As Integer = 0 To 10
            Next
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42024: Unused local variable: 'qqq'.
            Dim [qqq] As Integer
                ~~~~~
BC30616: Variable 'qqq' hides a variable in an enclosing block.
            For qqq As Integer = 0 To 10
                ~~~
BC42024: Unused local variable: 'www'.
            Dim www As Integer
                ~~~
BC30616: Variable 'www' hides a variable in an enclosing block.
            For [www] As Integer = 0 To 10
                ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_7()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        For x As Integer = 0 To 10
            Dim x As Integer
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30616: Variable 'x' hides a variable in an enclosing block.
            Dim x As Integer
                ~
BC42024: Unused local variable: 'x'.
            Dim x As Integer
                ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_8()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
        For var1 As Integer = 0 To 10
            For var2 As Integer = 0 To 10
                Dim var1 As Integer
            Next
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30616: Variable 'var1' hides a variable in an enclosing block.
                Dim var1 As Integer
                    ~~~~
BC42024: Unused local variable: 'var1'.
                Dim var1 As Integer
                    ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30616ERR_BlockLocalShadowing1_9()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DuplicateLocals1">
        <file name="a.vb">
Class C
    Shared Sub Main()
        For Each r As Integer In New Integer() {1, 2, 3}
            'Was COMPILEERROR: BC30288, "r" in Dev10
            'Now BC30616
            Dim r As Integer
        Next
    End Sub
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30616: Variable 'r' hides a variable in an enclosing block.
            Dim r As Integer
                ~
BC42024: Unused local variable: 'r'.
            Dim r As Integer
                ~
     </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30647ERR_ReturnFromNonFunction()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ReturnFromNonFunction">
        <file name="a.vb">
        Structure S1
            shared sub foo()
                return  1
            end sub
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30647: 'Return' statement in a Sub or a Set cannot return a value.
                return  1
                ~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30654ERR_ReturnWithoutValue()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ReturnWithoutValue">
        <file name="a.vb">
        Structure S1
            shared Function foo 
                return  
            end Function
        End Structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30654: 'Return' statement in a Function, Get, or Operator must return a value.
                return  
                ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30657ERR_UnsupportedMethod1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="UnsupportedMethod1">
        <file name="a.vb">
        Class C1
            Dim x As System.Threading.IOCompletionCallback
            Sub Sub1()
            End Sub
            Sub New()
                x = New System.Threading.IOCompletionCallback(AddressOf Sub1)
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30657: 'IOCompletionCallback' has a return type that is not supported or parameter types that are not supported.
                x = New System.Threading.IOCompletionCallback(AddressOf Sub1)
                                                              ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30658ERR_NoNonIndexProperty1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoNonIndexProperty1">
        <file name="a.vb">
        Option Explicit On
        Imports System
        Module M1
            Class MyAttr
                Inherits Attribute
                Public Property Prop(ByVal i As Integer) As Integer
                    Get
                        Return Nothing
                    End Get
                    Set(ByVal Value As Integer)
                    End Set
                End Property
            End Class
            &lt;MyAttr(Prop:=1)&gt;'BIND:"Prop"
            Class C1
            End Class
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30658: Property 'Prop' with no parameters cannot be found.
            &lt;MyAttr(Prop:=1)&gt;'BIND:"Prop"
                    ~~~~
</expected>)

            VerifyOperationTreeForTest(Of IdentifierNameSyntax)(compilation, "a.vb", <![CDATA[
IPropertyReferenceOperation: Property M1.MyAttr.Prop(i As System.Int32) As System.Int32 (OperationKind.PropertyReference, Type: System.Int32, IsInvalid) (Syntax: 'Prop')
  Instance Receiver: 
    null]]>.Value)
        End Sub

        <Fact()>
        Public Sub BC30659ERR_BadAttributePropertyType1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BadAttributePropertyType1">
        <file name="a.vb"><![CDATA[
Imports System

<AttributeUsage(AttributeTargets.Class, AllowMultiple:=True, Inherited:=True)>
Class MultiUseAttribute
    Inherits System.Attribute
    Public Sub New(ByVal Value As Integer)
    End Sub
End Class

<AttributeUsage(AttributeTargets.Class, Inherited:=True)>
Class SingleUseAttribute
    Inherits Attribute
    Property A() As Date
        Get
            Return Nothing
        End Get
        Set(value As Date)
        End Set
    End Property
    Public Sub New(ByVal Value As Integer)
    End Sub
End Class
<SingleUse(1, A:=1.1), MultiUse(1)>
Class Base
End Class
<SingleUse(0, A:=1.1), MultiUse(0)>
Class Derived
    Inherits Base
End Class
    ]]></file>
    </compilation>).VerifyDiagnostics(
            Diagnostic(ERRID.ERR_BadAttributePropertyType1, "A").WithArguments("A"),
            Diagnostic(ERRID.ERR_DoubleToDateConversion, "1.1"),
            Diagnostic(ERRID.ERR_BadAttributePropertyType1, "A").WithArguments("A"),
            Diagnostic(ERRID.ERR_DoubleToDateConversion, "1.1")) ' BC30533: Dev10 NOT report

        End Sub

        <Fact()>
        Public Sub BC30661ERR_PropertyOrFieldNotDefined1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="PropertyOrFieldNotDefined1">
        <file name="a.vb"><![CDATA[
        Imports System
        <AttributeUsage(AttributeTargets.All)>
        Public Class GeneralAttribute
            Inherits Attribute
        End Class
        <General(NotExist:=10)>
        Class C1
        End Class
    ]]></file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_PropertyOrFieldNotDefined1, "NotExist").WithArguments("NotExist"))

        End Sub

        <Fact()>
        Public Sub BC30665ERR_CantThrowNonException()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
        Module M1
            Sub Foo()
                Throw (Nothing)
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30665: 'Throw' operand must derive from 'System.Exception'.
                Throw (Nothing)
                ~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30665ERR_CantThrowNonException_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
Class A
End Class
Class C
    Shared Sub M1(Of T As System.Exception)(e As T)
        Throw e
    End Sub
    Shared Sub M2(Of T As {System.Exception, New})()
        Throw New T()
    End Sub
    Shared Sub M3(Of T As A)(a As T)
        Throw a
    End Sub
    Shared Sub M4(Of U As New)()
        Throw New U()
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30665: 'Throw' operand must derive from 'System.Exception'.
        Throw a
        ~~~~~~~
BC30665: 'Throw' operand must derive from 'System.Exception'.
        Throw New U()
        ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30666ERR_MustBeInCatchToRethrow()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="MustBeInCatchToRethrow">
        <file name="a.vb">
        Imports System
        Class C1
            Sub foo()
                Try
                    Throw
                Catch ex As Exception
                End Try
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30666: 'Throw' statement cannot omit operand outside a 'Catch' statement or inside a 'Finally' statement.
                    Throw
                    ~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30671ERR_InitWithMultipleDeclarators()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InitWithMultipleDeclarators">
        <file name="a.vb">
            Imports System

            Public Structure Class1
                implements IDisposable

                Public Sub Dispose() implements Idisposable.Dispose
                End Sub
            End Structure

            Public Class Class2
                Sub foo()
                    Dim a, b As Class1 = New Class1
                    a = nothing
                    b = nothing

                    Using c, d as Class1 = nothing
                    End Using
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
                    Dim a, b As Class1 = New Class1
                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC36011: 'Using' resource variable must have an explicit initialization.
                    Using c, d as Class1 = nothing
                          ~
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
                    Using c, d as Class1 = nothing
                          ~~~~~~~~~~~~~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30671ERR_InitWithMultipleDeclarators02()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
imports system

Class C1
    ' not so ok
    public i, j as integer = 23

    ' ok enough :)
    public k as integer,l as integer = 23


    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            CompilationUtils.AssertTheseDiagnostics(c1,
<expected>
BC30671: Explicit initialization is not permitted with multiple variables declared with a single type specifier.
    public i, j as integer = 23
           ~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30672ERR_InitWithExplicitArraySizes()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InitWithExplicitArraySizes">
        <file name="a.vb">
        Structure myStruct1
            sub foo()
                Dim a6(,) As Integer
                Dim b6(5, 5) As Integer = a6
            end Sub
        End structure
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30672: Explicit initialization is not permitted for arrays declared with explicit bounds.
                Dim b6(5, 5) As Integer = a6
                    ~~~~~~~~
BC42104: Variable 'a6' is used before it has been assigned a value. A null reference exception could result at runtime.
                Dim b6(5, 5) As Integer = a6
                                          ~~
</expected>)
        End Sub

        <WorkItem(542258, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542258")>
        <Fact()>
        Public Sub BC30672ERR_InitWithExplicitArraySizes_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="InitWithExplicitArraySizes">
        <file name="a.vb">
        Class Cls1
            Public Arr(3) As Cls1 = New Cls1() {New Cls1}

            Sub foo
                Dim Arr(3) As Cls1 = New Cls1() {New Cls1}
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30672: Explicit initialization is not permitted for arrays declared with explicit bounds.
            Public Arr(3) As Cls1 = New Cls1() {New Cls1}
                   ~~~~~~
BC30672: Explicit initialization is not permitted for arrays declared with explicit bounds.
                Dim Arr(3) As Cls1 = New Cls1() {New Cls1}
                    ~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30672ERR_InitWithExplicitArraySizes_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="InitWithExplicitArraySizes">
        <file name="a.vb">
        Option Infer On
        Imports System
        Module Module1
            Sub Main()
                Dim arr14(1, 2) = New Double(1, 2) {} ' Invalid
            End Sub
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30672: Explicit initialization is not permitted for arrays declared with explicit bounds.
                Dim arr14(1, 2) = New Double(1, 2) {} ' Invalid
                    ~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30676ERR_NameNotEvent2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NameNotEvent2">
        <file name="a.vb">
Option Strict Off
Module M
    Sub Foo()
        Dim x As C1 = New C1
        AddHandler x.E, Sub() Console.WriteLine()
    End Sub
End Module
Class C1
    Public Dim E As String
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30676: 'E' is not an event of 'C1'.
        AddHandler x.E, Sub() Console.WriteLine()
                     ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30677ERR_AddOrRemoveHandlerEvent()
            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AddOrRemoveHandlerEvent">
        <file name="a.vb">
        Module M
            Sub Main()
                AddHandler Nothing, Nothing
            End Sub
        End Module
    </file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_AddOrRemoveHandlerEvent, "Nothing"))

        End Sub

        <Fact(), WorkItem(918579, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/918579"), WorkItem(34, "CodePlex")>
        Public Sub BC30685ERR_AmbiguousAcrossInterfaces3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="AmbiguousAcrossInterfaces3">
        <file name="a.vb">
        Interface A
            Sub fun(ByVal i As Integer)
        End Interface
        Interface AB
            Inherits A
            Shadows Sub fun(ByVal i As Integer)
        End Interface
        Interface AC
            Inherits A
            Shadows Sub fun(ByVal i As Integer)
        End Interface
        Interface ABS
            Inherits AB, AC
        End Interface
        Class Test
            Sub D(ByVal d As ABS)
                d.fun(2)
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC30521: Overload resolution failed because no accessible 'fun' is most specific for these arguments:
    'Sub AB.fun(i As Integer)': Not most specific.
    'Sub AC.fun(i As Integer)': Not most specific.
                d.fun(2)
                  ~~~
    </expected>)
        End Sub

        <Fact()>
        Public Sub BC30686ERR_DefaultPropertyAmbiguousAcrossInterfaces4()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="DefaultPropertyAmbiguousAcrossInterfaces4">
        <file name="a.vb">
Option Strict On
Interface IA(Of T)
    Default ReadOnly Property P(o As T) As Object
End Interface
Interface IB1(Of T)
    Inherits IA(Of T)
End Interface
Interface IB2(Of T)
    Inherits IA(Of T)
    Default Overloads ReadOnly Property P(x As T, y As T) As Object
End Interface
Interface IB3(Of T)
    Inherits IA(Of T)
    Default Overloads ReadOnly Property Q(x As Integer, y As Integer, z As Integer) As Object
End Interface
Interface IC1
    Inherits IA(Of String), IB1(Of String)
End Interface
Interface IC2
    Inherits IA(Of String), IB1(Of Object)
End Interface
Interface IC3
    Inherits IA(Of String), IB2(Of String)
End Interface
Interface IC4
    Inherits IA(Of String), IB3(Of String)
End Interface
Module M
    Sub M(c1 As IC1, c2 As IC2, c3 As IC3, c4 As IC4)
        Dim value As Object
        value = c1("")
        value = c2("")
        value = c3("")
        value = c4("")
    End Sub
End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC40007: Default property 'Q' conflicts with the default property 'P' in the base interface 'IA'. 'Q' will be the default property. 'Q' should be declared 'Shadows'.
    Default Overloads ReadOnly Property Q(x As Integer, y As Integer, z As Integer) As Object
                                        ~
BC30686: Default property access is ambiguous between the inherited interface members 'ReadOnly Default Property P(o As String) As Object' of interface 'IA(Of String)' and 'ReadOnly Default Property P(o As Object) As Object' of interface 'IA(Of Object)'.
        value = c2("")
                ~~
BC30686: Default property access is ambiguous between the inherited interface members 'ReadOnly Default Property P(o As String) As Object' of interface 'IA(Of String)' and 'ReadOnly Default Property P(x As String, y As String) As Object' of interface 'IB2(Of String)'.
        value = c3("")
                ~~
BC30686: Default property access is ambiguous between the inherited interface members 'ReadOnly Default Property P(o As String) As Object' of interface 'IA(Of String)' and 'ReadOnly Default Property Q(x As Integer, y As Integer, z As Integer) As Object' of interface 'IB3(Of String)'.
        value = c4("")
                ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30690ERR_StructureNoDefault1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation>
    <file name="a.vb">
        Namespace N
            Structure S
            End Structure
        End Namespace
        Namespace M
            Class C
                Shared Sub M(x As N.S)
                    N(x(0))
                    N(x!P)
                End Sub
                Shared Sub N(o As Object)
                End Sub
            End Class
        End Namespace
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30690: Structure 'S' cannot be indexed because it has no default property.
                    N(x(0))
                      ~
BC30690: Structure 'S' cannot be indexed because it has no default property.
                    N(x!P)
                      ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30734ERR_LocalNamedSameAsParam1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="LocalNamedSameAsParam1">
        <file name="a.vb">
           Class cls0(Of G)
                Sub foo(Of T) (ByVal x As T)
                    Dim x As T
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30734: 'x' is already declared as a parameter of this method.
                    Dim x As T
                        ~
BC42024: Unused local variable: 'x'.
                    Dim x As T
                        ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30734ERR_LocalNamedSameAsParam1_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC30734ERR_LocalNamedSameAsParam1_2">
        <file name="a.vb">
Class C
    Private field As Integer = 0
    Private Property [property]() As Integer
        Get
            Return m_property
        End Get
        Set(value As Integer)
            m_property = value
        End Set
    End Property
    Property prop() As Integer
        Get
            Return 1
        End Get
        Set(ByVal Value As Integer)
            ' Was Dev10: COMPILEERROR: BC30616, "value"
            ' Now: BC30734
            For Each value As Byte In New Byte() {1, 2, 3}
            Next
        End Set
    End Property
    Private m_property As Integer

    Shared Sub Main()
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>    
BC30734: 'value' is already declared as a parameter of this method.
            For Each value As Byte In New Byte() {1, 2, 3}
                     ~~~~~
</expected>)
        End Sub

        <WorkItem(528680, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/528680")>
        <Fact()>
        Public Sub BC30734ERR_LocalNamedSameAsParam1_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BlockLocalShadowing1">
        <file name="a.vb">
Public Class MyClass1
    Public Shared Sub Main()
    End Sub
    Sub foo(ByVal p1 As Integer)
        For p1 As Integer = 1 To 10
        Next
    End Sub
End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30734: 'p1' is already declared as a parameter of this method.
        For p1 As Integer = 1 To 10
            ~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30742ERR_CannotConvertValue2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation>
        <file name="a.vb">
           Imports microsoft.visualbasic.strings
            Module M1
                    Sub foo()
                        Dim i As Integer
                        Dim c As Char
                        I% = Asc("")
                        i% = Asc("" + "" + "" + "" + "" + "" + "" + "" + "")
                        c = ChrW(65536)
                        c = ChrW(-68888)
                    End Sub
                End module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30742: Value '' cannot be converted to 'Integer'.
                        I% = Asc("")
                             ~~~~~~~
BC30742: Value '' cannot be converted to 'Integer'.
                        i% = Asc("" + "" + "" + "" + "" + "" + "" + "" + "")
                             ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC30742: Value '65536' cannot be converted to 'Char'.
                        c = ChrW(65536)
                            ~~~~~~~~~~~
BC30742: Value '-68888' cannot be converted to 'Char'.
                        c = ChrW(-68888)
                            ~~~~~~~~~~~~
                                  </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30742ERR_CannotConvertValue2_2()
            Dim source =
<compilation>
    <file name="a.vb">
Option strict on
imports system
imports microsoft.visualbasic.strings
Imports System.Text

Class C1
    private const f1 as integer = Asc("") ' empty string
    private const f2 as integer = AscW("") ' empty string
    private const f3 as integer = Asc(CStr(nothing)) ' nothing string
    private const f4 as integer = AscW(CStr(nothing)) ' nothing string
    private const f5 as Char = ChrW(65536)

    Public shared Sub Main(args() as string)
    End sub
End Class
    </file>
</compilation>

            Dim c1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source)

            AssertTheseDiagnostics(c1,
<expected>
BC30742: Value '' cannot be converted to 'Integer'.
    private const f1 as integer = Asc("") ' empty string
                                  ~~~~~~~
BC30742: Value '' cannot be converted to 'Integer'.
    private const f2 as integer = AscW("") ' empty string
                                  ~~~~~~~~
BC30742: Value '' cannot be converted to 'Integer'.
    private const f3 as integer = Asc(CStr(nothing)) ' nothing string
                                  ~~~~~~~~~~~~~~~~~~
BC30742: Value '' cannot be converted to 'Integer'.
    private const f4 as integer = AscW(CStr(nothing)) ' nothing string
                                  ~~~~~~~~~~~~~~~~~~~
BC30742: Value '65536' cannot be converted to 'Char'.
    private const f5 as Char = ChrW(65536)
                               ~~~~~~~~~~~
</expected>)
        End Sub

        <WorkItem(574290, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/574290")>
        <Fact()>
        Public Sub BC30742ERR_PassVBNullToAsc()
            Dim source =
<compilation name="ExpressionContext">
    <file name="a.vb">
        Imports Microsoft.VisualBasic
Module Module1
Sub Main
Asc(vbnullstring)
End Sub
End MOdule
    </file>
</compilation>

            CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(source).VerifyDiagnostics(Diagnostic(ERRID.ERR_CannotConvertValue2, "Asc(vbnullstring)").WithArguments("", "Integer"))

        End Sub

        <Fact()>
        Public Sub BC30752ERR_OnErrorInSyncLock()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="OnErrorInSyncLock">
        <file name="a.vb">
            Imports System
            Class C
                Sub IncrementCount()
                    SyncLock GetType(C)
                        On Error GoTo 0
                    End SyncLock
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30752: 'On Error' statements are not valid within 'SyncLock' statements.
                        On Error GoTo 0
                        ~~~~~~~~~~~~~~~ 
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30753ERR_NarrowingConversionCollection2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NarrowingConversionCollection2">
        <file name="a.vb">
            option strict on
            Imports System
            class C1
                Function Main() As Microsoft.VisualBasic.Collection
                    'Dim collection As Microsoft.VisualBasic. = Nothing
                    Dim _collection As _Collection = Nothing
                    return _collection
                End function
            End Class
            Interface _Collection
            End Interface
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30753: Option Strict On disallows implicit conversions from '_Collection' to 'Collection'; the Visual Basic 6.0 collection type is not compatible with the .NET Framework collection type.
                    return _collection
                           ~~~~~~~~~~~
BC42322: Runtime errors might occur when converting '_Collection' to 'Collection'.
                    return _collection
                           ~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30754ERR_GotoIntoTryHandler()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoTryHandler">
        <file name="a.vb">
            Imports System
            Class C1
                Sub Main()
                    Do While (True)
                        GoTo LB1
                    Loop
                    Try
                    Catch ex As Exception
                    Finally
            LB1:
                    End Try
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30754: 'GoTo LB1' is not valid because 'LB1' is inside a 'Try', 'Catch' or 'Finally' statement that does not contain this statement.
                        GoTo LB1
                             ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(543055, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/543055")>
        <Fact()>
        Public Sub BC30754ERR_GotoIntoTryHandler_2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoTryHandler">
        <file name="a.vb">
Imports System
Module Program
    Sub Main(args As String())
        Try
            GoTo label
            GoTo label5
        Catch ex As Exception
label:
        Finally
label5:
        End Try
    End Sub
End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30754: 'GoTo label' is not valid because 'label' is inside a 'Try', 'Catch' or 'Finally' statement that does not contain this statement.
            GoTo label
                 ~~~~~
BC30754: 'GoTo label5' is not valid because 'label5' is inside a 'Try', 'Catch' or 'Finally' statement that does not contain this statement.
            GoTo label5
                 ~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30755ERR_GotoIntoSyncLock()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoSyncLock">
        <file name="a.vb">
            Imports System
            Class C
                Sub IncrementCount()
                    SyncLock GetType(C)
            label:
                    End SyncLock
                    GoTo label
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30755: 'GoTo label' is not valid because 'label' is inside a 'SyncLock' statement that does not contain this statement.
                    GoTo label
                         ~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30756ERR_GotoIntoWith()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoWith">
        <file name="a.vb">
            Class C1
                Sub Main()
                    Dim s = New Type1()
                    With s
                        .x = 1
                        GoTo lab1
                    End With
                    With s
            lab1:
                        .x = 1
                    End With
                End Sub
            End Class
            Class Type1
                Public x As Short
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30756: 'GoTo lab1' is not valid because 'lab1' is inside a 'With' statement that does not contain this statement.
                        GoTo lab1
                             ~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30756ERR_GotoIntoWith_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoWith">
        <file name="a.vb">
            Class C1
                Function Main()
                    Dim s = New Type1()
                    With s
            lab1:
                        .x = 1
                    End With
                    GoTo lab1
                End Function
            End Class
            Class Type1
                Public x As Short
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30756: 'GoTo lab1' is not valid because 'lab1' is inside a 'With' statement that does not contain this statement.
                    GoTo lab1
                         ~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30757ERR_GotoIntoFor()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoFor">
        <file name="a.vb">
            Class C1
                Sub Main()
                    Dim s = New Type1()
                    With s
                        .x = 1
                        GoTo label1
                    End With
                    For i = 0 To 5
                        GoTo label1
label1:
                        Continue For
                    Next
                End Sub
            End Class
            Class Type1
                Public x As Short
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30757: 'GoTo label1' is not valid because 'label1' is inside a 'For' or 'For Each' statement that does not contain this statement.
                        GoTo label1
                             ~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30757ERR_GotoIntoFor_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoFor">
        <file name="a.vb">
            Class C1
                Function Main()
                    if (true)
                        GoTo label1
                    End If
                    For i as Integer = 0 To 5
                        GoTo label1
            label1:
                        Continue For
                    Next
                    return 1
                End Function
            End Class
            Class Type1
                Public x As Short
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30757: 'GoTo label1' is not valid because 'label1' is inside a 'For' or 'For Each' statement that does not contain this statement.
                        GoTo label1
                             ~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30757ERR_GotoIntoFor_2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="GotoIntoFor">
        <file name="a.vb">
Option Infer On
Option Strict Off
Class C1
    Function Main()
        Dim s As Type1 = New Type1()
        If (True)
            GoTo label1
        End If
        For Each i In s
            GoTo label1
label1:
            Continue For
        Next
        Return 1
    End Function
End Class
Class Type1
    Public x As Short
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30757: 'GoTo label1' is not valid because 'label1' is inside a 'For' or 'For Each' statement that does not contain this statement.
            GoTo label1
                 ~~~~~~
BC32023: Expression is of type 'Type1', which is not a collection type.
        For Each i In s
                      ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(540627, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/540627")>
        <Fact()>
        Public Sub BC30758ERR_BadAttributeNonPublicConstructor()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="BadAttributeNonPublicConstructor">
        <file name="at30758.vb"><![CDATA[
Imports System
<AttributeUsage(AttributeTargets.All)>
Public MustInherit Class MyAttribute
    Inherits Attribute
    Friend Sub New()

    End Sub
End Class

<My()>
Class Foo
End Class
    ]]></file>
    </compilation>).VerifyDiagnostics(
    Diagnostic(ERRID.ERR_AttributeCannotBeAbstract, "My").WithArguments("MyAttribute"),
    Diagnostic(ERRID.ERR_BadAttributeNonPublicConstructor, "My"))
        End Sub

        <Fact()>
        Public Sub BC30782ERR_ContinueDoNotWithinDo()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ContinueDoNotWithinDo">
        <file name="a.vb">
            Class C1
                Sub Main()
                    While True
                        Continue Do
                    End While
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30782: 'Continue Do' can only appear inside a 'Do' statement.
                        Continue Do
                        ~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30783ERR_ContinueForNotWithinFor()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ContinueForNotWithinFor">
        <file name="a.vb">
            Class C1
                Sub Main()
                        Continue for
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30783: 'Continue For' can only appear inside a 'For' statement.
                        Continue for
                        ~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30784ERR_ContinueWhileNotWithinWhile()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ContinueWhileNotWithinWhile">
        <file name="a.vb">
            Class C1
                Sub Main()
                        Continue while
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30784: 'Continue While' can only appear inside a 'While' statement.
                        Continue while
                        ~~~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30793ERR_TryCastOfUnconstrainedTypeParam1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="TryCastOfUnconstrainedTypeParam1">
        <file name="a.vb">
            Module M1
                Sub Foo(Of T) (ByVal x As T)
                    Dim o As Object = TryCast(x, T)
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30793: 'TryCast' operands must be class-constrained type parameter, but 'T' has no class constraint.
                    Dim o As Object = TryCast(x, T)
                                                 ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30794ERR_AmbiguousDelegateBinding2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AmbiguousDelegateBinding2">
        <file name="a.vb">
            Module Module1
                Public Delegate Sub Bar(ByVal x As Integer, ByVal y As Integer)
                Public Sub Foo(Of T, R)(ByVal x As T, ByVal y As R)
                End Sub
                Public Sub Foo(Of T)(ByVal x As T, ByVal y As T)
                End Sub
            End Module
            Class C1
                Sub FOO()
                    Dim x1 As Module1.Bar = AddressOf Module1.Foo
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30794: No accessible 'Foo' is most specific: 
    Public Sub Foo(Of Integer, Integer)(x As Integer, y As Integer)
    Public Sub Foo(Of Integer)(x As Integer, y As Integer)
                    Dim x1 As Module1.Bar = AddressOf Module1.Foo
                                                      ~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact>
        Public Sub BC30917ERR_NoNonObsoleteConstructorOnBase3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoNonObsoleteConstructorOnBase3">
        <file name="a.vb">
            Imports System
            Class C1
                &lt;Obsolete(Nothing, True)&gt;
                Sub New()
                End Sub
            End Class
            Class C2
                Inherits C1
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30917: Class 'C2' must declare a 'Sub New' because the 'Public Sub New()' in its base class 'C1' is marked obsolete.
            Class C2
                  ~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact>
        Public Sub BC30918ERR_NoNonObsoleteConstructorOnBase4()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoNonObsoleteConstructorOnBase4">
        <file name="a.vb">
            Imports System
            Class C1
                &lt;Obsolete("hello", True)&gt;
                Sub New()
                End Sub
            End Class
            Class C2
                Inherits C1
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30918: Class 'C2' must declare a 'Sub New' because the 'Public Sub New()' in its base class 'C1' is marked obsolete: 'hello'.
            Class C2
                  ~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact>
        Public Sub BC30919ERR_RequiredNonObsoleteNewCall3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="RequiredNonObsoleteNewCall3">
        <file name="a.vb">
            Imports System
            Class C1
                &lt;Obsolete(Nothing, True)&gt;
                Sub New()
                End Sub
            End Class
            Class C2
                Inherits C1
                Sub New()
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30919: First statement of this 'Sub New' must be an explicit call to 'MyBase.New' or 'MyClass.New' because the 'Public Sub New()' in the base class 'C1' of 'C2' is marked obsolete.
                Sub New()
                    ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact>
        Public Sub BC30920ERR_RequiredNonObsoleteNewCall4()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="RequiredNonObsoleteNewCall4">
        <file name="a.vb">
            Imports System
            Class C1
                &lt;Obsolete("hello", True)&gt;
                Sub New()
                End Sub
            End Class
            Class C2
                Inherits C1
                Sub New()
                End Sub
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30920: First statement of this 'Sub New' must be an explicit call to 'MyBase.New' or 'MyClass.New' because the 'Public Sub New()' in the base class 'C1' of 'C2' is marked obsolete: 'hello'.
                Sub New()
                    ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(531309, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/531309")>
        <Fact()>
        Public Sub BC30933ERR_LateBoundOverloadInterfaceCall1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="LateBoundOverloadInterfaceCall1">
        <file name="a.vb">
            Module m1
                Interface i1
                    Sub s1(ByVal p1 As Integer)
                    Sub s1(ByVal p1 As Double)
                End Interface
                Class c1
                    Implements i1
                    Public Overloads Sub s1(ByVal p1 As Integer) Implements i1.s1
                    End Sub
                    Public Overloads Sub s2(ByVal p1 As Double) Implements i1.s1
                    End Sub
                End Class
                Sub Main()
                    Dim refer As i1 = New c1
                    Dim o1 As Object = 3.1415
                    refer.s1(o1)
                End Sub
            End Module

            Module m2
                Interface i1
                    Property s1(ByVal p1 As Integer)
                    Property s1(ByVal p1 As Double)
                End Interface
                Class c1
                    Implements i1

                    Public Property s1(p1 As Double) As Object Implements i1.s1
                        Get
                            Return Nothing
                        End Get
                        Set(value As Object)

                        End Set
                    End Property

                    Public Property s1(p1 As Integer) As Object Implements i1.s1
                        Get
                            Return Nothing
                        End Get
                        Set(value As Object)

                        End Set
                    End Property
                End Class
                Sub Main()
                    Dim refer As i1 = New c1
                    Dim o1 As Object = 3.1415
                    refer.s1(o1) = 1
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30933: Late bound overload resolution cannot be applied to 's1' because the accessing instance is an interface type.
                    refer.s1(o1)
                          ~~
BC30933: Late bound overload resolution cannot be applied to 's1' because the accessing instance is an interface type.
                    refer.s1(o1) = 1
                          ~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30934ERR_RequiredAttributeConstConversion2()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
    <compilation>
        <file name="a.vb"><![CDATA[
Imports System
<A(<a/>)>
Class A
    Inherits Attribute
    Public Sub New(o As Object)
    End Sub
End Class
        ]]></file>
    </compilation>, references:=XmlReferences)
            compilation.VerifyDiagnostics(Diagnostic(ERRID.ERR_RequiredAttributeConstConversion2, "<a/>").WithArguments("System.Xml.Linq.XElement", "Object"))
        End Sub

        <Fact()>
        Public Sub BC30939ERR_AddressOfNotCreatableDelegate1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="AddressOfNotCreatableDelegate1">
        <file name="a.vb">
            Imports System
            Module M1
                Sub foo()
                    Dim x As [Delegate] = AddressOf main
                End Sub
                Sub main()
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30939: 'AddressOf' expression cannot be converted to '[Delegate]' because type '[Delegate]' is declared 'MustInherit' and cannot be created.
                    Dim x As [Delegate] = AddressOf main
                                          ~~~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact(), WorkItem(529157, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/529157")>
        Public Sub BC30940ERR_ReturnFromEventMethod() 'diag behavior change by design- not worth investing in.
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ReturnFromEventMethod">
        <file name="a.vb">
            Class C1
                Delegate Sub EH()
                Custom Event e1 As EH
                    AddHandler(ByVal value As EH)
                        Return value
                    End AddHandler
                    RemoveHandler(ByVal value As EH)
                        Return value
                    End RemoveHandler
                    RaiseEvent()
                        Return value
                    End RaiseEvent
                End Event
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30647: 'Return' statement in a Sub or a Set cannot return a value.
                        Return value
                        ~~~~~~~~~~~~
BC30647: 'Return' statement in a Sub or a Set cannot return a value.
                        Return value
                        ~~~~~~~~~~~~
BC30647: 'Return' statement in a Sub or a Set cannot return a value.
                        Return value
                        ~~~~~~~~~~~~
BC30451: 'value' is not declared. It may be inaccessible due to its protection level.
                        Return value
                               ~~~~~
                                  </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(542270, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542270")>
        <Fact()>
        Public Sub BC30949ERR_ArrayInitializerForNonConstDim()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="ArrayInitializerForNonConstDim">
        <file name="a.vb">
            Option Strict On
            Module M
                Sub Main()
                    Dim x as integer = 1
                    Dim y as Object = New Integer(x) {1, 2}
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30949: Array initializer cannot be specified for a non constant dimension; use the empty initializer '{}'.
                    Dim y as Object = New Integer(x) {1, 2}
                                                     ~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(542270, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542270")>
        <Fact()>
        Public Sub BC30949ERR_ArrayInitializerForNonConstDim_1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="ArrayInitializerForNonConstDim">
        <file name="a.vb">
            Option Strict On
            Imports System
            Class M1
                Public Shared Sub Main()
                    Dim myLength As Integer = 2
                    Dim arr As Integer(,) = New Integer(myLength - 1, 1) {{1, 2}, {3, 4}, {5, 6}}
                End Sub
                Private Class A
                    Private x As Integer = 1
                    Private arr As Integer(,) = New Integer(x, 1) {{1, 2}, {3, 4}, {5, 6}}
                End Class
            End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30949: Array initializer cannot be specified for a non constant dimension; use the empty initializer '{}'.
                    Dim arr As Integer(,) = New Integer(myLength - 1, 1) {{1, 2}, {3, 4}, {5, 6}}
                                                                         ~~~~~~~~~~~~~~~~~~~~~~~~
BC30949: Array initializer cannot be specified for a non constant dimension; use the empty initializer '{}'.
                    Private arr As Integer(,) = New Integer(x, 1) {{1, 2}, {3, 4}, {5, 6}}
                                                                  ~~~~~~~~~~~~~~~~~~~~~~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30950ERR_DelegateBindingFailure3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="DelegateBindingFailure3">
        <file name="a.vb">
            Option Strict On
            Imports System

            Module M
                Dim f As Action(Of Object) = CType(AddressOf 1.ToString, Action(Of Object))
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
                                  </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30978ERR_IterationVariableShadowLocal1()

            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
    <compilation name="IterationVariableShadowLocal1">
        <file name="a.vb">
Option Infer On
Option Strict Off
Imports System.Linq
Module M
    Sub Bar()
        Dim x = From bar In ""
    End Sub
    Function Foo()
        Dim x = From foo In ""
        Return Nothing
    End Function
End Module
        </file>
    </compilation>, {Net40.SystemCore})

            Dim expectedErrors1 = <errors>
BC30978: Range variable 'foo' hides a variable in an enclosing block or a range variable previously defined in the query expression.
        Dim x = From foo In ""
                     ~~~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30978ERR_IterationVariableShadowLocal1_1()

            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
    <compilation name="IterationVariableShadowLocal1">
        <file name="a.vb">
            Option Infer Off
            Imports System
            Imports System.Linq
            Module Program
                Sub Main(args As String())
                    Dim arr As String() = New String() {"aaa", "bbb", "ccc"}
                    Dim arr_int As Integer() = New Integer() {111, 222, 333}
                    Dim x = 1
                    Dim s = If(True, (From x In arr Select x).ToList(), From y As Integer In arr_int Select y)
                End Sub
            End Module
        </file>
    </compilation>, {Net40.SystemCore}, options:=TestOptions.ReleaseExe)

            Dim expectedErrors1 = <errors>
BC30978: Range variable 'x' hides a variable in an enclosing block or a range variable previously defined in the query expression.
                    Dim s = If(True, (From x In arr Select x).ToList(), From y As Integer In arr_int Select y)
                                           ~
BC30978: Range variable 'x' hides a variable in an enclosing block or a range variable previously defined in the query expression.
                    Dim s = If(True, (From x In arr Select x).ToList(), From y As Integer In arr_int Select y)
                                                           ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30980ERR_CircularInference1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="CircularInference2">
        <file name="a.vb">
            Module M
                Const x = 1
                Sub Main()
                    Dim x = Function() x
                End Sub
            End Module
        </file>
    </compilation>)
            ' Extra Warning in Roslyn
            compilation1.VerifyDiagnostics(
                Diagnostic(ERRID.ERR_CircularInference1, "x").WithArguments("x")
                )

        End Sub

        <Fact()>
        Public Sub BC30980ERR_CircularInference1_2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="CircularInference2">
        <file name="a.vb">
Option Infer On

Imports System

Class C
    Shared Sub Main()
        Dim f As Func(Of Integer) = Function()
                                        For Each x In x + 1
                                            Return x
                                        Next
                                        return 0
                                    End Function
    End Sub
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30980: Type of 'x' cannot be inferred from an expression containing 'x'.
                                        For Each x In x + 1
                                                      ~
BC42104: Variable 'x' is used before it has been assigned a value. A null reference exception could result at runtime.
                                        For Each x In x + 1
                                                      ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30980ERR_CircularInference2_2()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="CircularInference2_2">
        <file name="a.vb">
Class C
    Shared Sub Main()
        For Each y In New Object() {New With {Key .y = y}}
        Next
    End Sub
End Class
        </file>
    </compilation>)

            Dim expectedErrors1 = <errors>
BC30980: Type of 'y' cannot be inferred from an expression containing 'y'.
        For Each y In New Object() {New With {Key .y = y}}
                                                       ~
BC42104: Variable 'y' is used before it has been assigned a value. A null reference exception could result at runtime.
        For Each y In New Object() {New With {Key .y = y}}
                                                       ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact()>
        Public Sub BC30980ERR_CircularInference2_2a()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="CircularInference2_2a">
        <file name="a.vb">
Class C
    Shared Sub Main()
        'For Each y As Object In New Object() {y}
        For Each y As Object In New Object() {New With {Key .y = y}}
        Next
    End Sub
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC42104: Variable 'y' is used before it has been assigned a value. A null reference exception could result at runtime.
        For Each y As Object In New Object() {New With {Key .y = y}}
                                                                 ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(542268, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542268")>
        <Fact()>
        Public Sub BC30980ERR_CircularInference2_3()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="CircularInference2_3">
        <file name="a.vb">
Option Infer On
Class C
    Shared Sub Main()
        Dim a = a.b
    End Sub
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30980: Type of 'a' cannot be inferred from an expression containing 'a'.
        Dim a = a.b
                ~
BC42104: Variable 'a' is used before it has been assigned a value. A null reference exception could result at runtime.
        Dim a = a.b
                ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <WorkItem(542268, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542268")>
        <Fact()>
        Public Sub BC30980ERR_CircularInference2_3a()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="CircularInference2_3a">
        <file name="a.vb">
Option Infer Off
Class C
    Shared Sub Main()
        Dim a = a.ToString()
    End Sub
End Class
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC42104: Variable 'a' is used before it has been assigned a value. A null reference exception could result at runtime.
        Dim a = a.ToString()
                ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact(), WorkItem(542191, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/542191")>
        Public Sub BC30982ERR_NoSuitableWidestType1()
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="NoSuitableWidestType1">
        <file name="a.vb">
            Option Infer On
            Module M
                Sub Main()
                    Dim stepVar = "1"c
                    For i = 1 To 10 Step stepVar
                    Next
                End Sub
            End Module
        </file>
    </compilation>)
            Dim expectedErrors1 = <errors>
BC30982: Type of 'i' cannot be inferred because the loop bounds and the step clause do not convert to the same type.
                    For i = 1 To 10 Step stepVar
                        ~
                 </errors>
            CompilationUtils.AssertTheseDiagnostics(compilation1, expectedErrors1)
        End Sub

        <Fact(), WorkItem(12261, "DevDiv_Projects/Roslyn")>
        Public Sub BC30983ERR_AmbiguousWidestType3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="AmbiguousWidestType3">
    <file name="a.vb">
    Module modErr30983
        Sub Test()
            'COMPILEERROR : BC30983, "i" 
            For i = New first(1) To New second(2) Step New third(3)
            Next
        End Sub
    End Module
    Class base
    End Class
    Class first
        Inherits base
        Dim m_count As ULong
        Sub New(ByVal d As ULong)
            m_count = d
        End Sub
        Overloads Shared Widening Operator CType(ByVal d As first) As second
            Return New second(d.m_count)
        End Operator
    End Class
    Class second
        Inherits base
        Dim m_count As ULong
            Sub New(ByVal d As ULong)
            m_count = d
        End Sub
        Overloads Shared Widening Operator CType(ByVal d As second) As first
            Return New first(d.m_count)
        End Operator
    End Class
    Class third
        Inherits first
        Sub New(ByVal d As ULong)
            MyBase.New(d)
        End Sub
        Overloads Shared Widening Operator CType(ByVal d As third) As Integer
            Return 1
        End Operator
    End Class
    </file>
</compilation>)

            'BC30983: Type of 'i' is ambiguous because the loop bounds and the step clause do not convert to the same type.
            'For i = New first(1) To New second(2) Step New third(3)
            '    ~
            compilation.VerifyDiagnostics(Diagnostic(ERRID.ERR_AmbiguousWidestType3, "i").WithArguments("i"))

        End Sub

        <Fact()>
        Public Sub BC30989ERR_DuplicateAggrMemberInit1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="DuplicateAggrMemberInit1">
    <file name="a.vb">
            Module M
                Sub Main()
                     Dim cust = New Customer() With {.Name = "Bob", .Name = "Robert"}
                End Sub
            End Module
            Class Customer
                Property Name As String
            End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30989: Multiple initializations of 'Name'.  Fields and properties can be initialized only once in an object initializer expression.
                     Dim cust = New Customer() With {.Name = "Bob", .Name = "Robert"}
                                                                     ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30990ERR_NonFieldPropertyAggrMemberInit1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NonFieldPropertyAggrMemberInit1">
    <file name="a.vb">
            Module M1
                Class WithSubX
                    Public Sub x()
                    End Sub
                End Class
                Sub foo()
                    Dim z As WithSubX = New WithSubX With {.x = 5}
                End Sub
            End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30990: Member 'x' cannot be initialized in an object initializer expression because it is not a field or property.
                    Dim z As WithSubX = New WithSubX With {.x = 5}
                                                            ~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30991ERR_SharedMemberAggrMemberInit1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="SharedMemberAggrMemberInit1">
    <file name="a.vb">
            Module M
                Sub Main()
                    Dim cust As New Customer With {.totalCustomers = 21}
                End Sub
            End Module
            Public Class Customer
                Public Shared totalCustomers As Integer
            End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30991: Member 'totalCustomers' cannot be initialized in an object initializer expression because it is shared.
                    Dim cust As New Customer With {.totalCustomers = 21}
                                                    ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30992ERR_ParameterizedPropertyInAggrInit1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ParameterizedPropertyInAggrInit1">
    <file name="a.vb">
            Module M
                Sub Main()
                    Dim strs As New C1() With {.defaultProp = "One"}
                End Sub
            End Module
            Public Class C1
                Private myStrings() As String
                Default Property defaultProp(ByVal index As Integer) As String
                    Get
                        Return myStrings(index)
                    End Get
                    Set(ByVal Value As String)
                        myStrings(index) = Value
                    End Set
                End Property
            End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30992: Property 'defaultProp' cannot be initialized in an object initializer expression because it requires arguments.
                    Dim strs As New C1() With {.defaultProp = "One"}
                                                ~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30993ERR_NoZeroCountArgumentInitCandidates1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="NoZeroCountArgumentInitCandidates1">
    <file name="a.vb">
            Module M
                Sub Main()
                    Dim aCoinObject = nothing
                    Dim coinCollection As New C1 With {.Item = aCoinObject}
                End Sub
            End Module
            Class C1
                WriteOnly Property Item(ByVal Key As String) As Object
                    Set(ByVal value As Object)
                    End Set
                End Property
                WriteOnly Property Item(ByVal Index As Integer) As Object
                    Set(ByVal value As Object)
                    End Set
                End Property
                private WriteOnly Property Item(ByVal Index As Long) As Object
                    Set(ByVal value As Object)
                    End Set
                End Property
            End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30993: Property 'Item' cannot be initialized in an object initializer expression because all accessible overloads require arguments.
                    Dim coinCollection As New C1 With {.Item = aCoinObject}
                                                        ~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC30994ERR_AggrInitInvalidForObject()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="AggrInitInvalidForObject">
    <file name="a.vb">
            Module M
                Sub Main()
                    Dim obj = New Object With {.ToString = "hello"}
                End Sub
            End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30994: Object initializer syntax cannot be used to initialize an instance of 'System.Object'.
                    Dim obj = New Object With {.ToString = "hello"}
                                         ~~~~~~~~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31080ERR_ReferenceComparison3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="ReferenceComparison3">
    <file name="a.vb">
       Interface I1
            Interface I2
            End Interface
        End Interface
        Class C1
            Sub FOO()
                Dim scenario1 As I1.I2
                If scenario1 = Nothing Then
                End If
            End Sub
        End Class    
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC42104: Variable 'scenario1' is used before it has been assigned a value. A null reference exception could result at runtime.
                If scenario1 = Nothing Then
                   ~~~~~~~~~
BC31080: Operator '=' is not defined for types 'I1.I2' and 'I1.I2'. Use 'Is' operator to compare two reference types.
                If scenario1 = Nothing Then
                   ~~~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31082ERR_CatchVariableNotLocal1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="CatchVariableNotLocal1">
    <file name="a.vb">
        Imports System
        Module M
            Dim ex As Exception
            Sub Main()
                Try
                Catch ex
                End Try
            End Sub
        End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31082: 'ex' is not a local variable or parameter, and so cannot be used as a 'Catch' variable.
                Catch ex
                      ~~
</expected>)
        End Sub

        <WorkItem(538613, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538613")>
        <Fact()>
        Public Sub BC30251_ModuleConstructorCall()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module M
  Sub [New]()
    M.New()
  End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30251: Type 'M' has no constructors.
    M.New()
    ~~~~~
</expected>)
        End Sub

        <WorkItem(538613, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/538613")>
        <Fact()>
        Public Sub BC30251_ModuleGenericConstructorCall()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module M
  Sub [New](Of T)()
    M.New(Of Integer)
  End Sub
End Module
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC30251: Type 'M' has no constructors.
    M.New(Of Integer)
    ~~~~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact(), WorkItem(570936, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/570936")>
        Public Sub BC31092ERR_ParamArrayWrongType()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ParamArrayWrongType">
    <file name="a.vb">
        Module M1
            Sub Foo()
                    Dim x As New C1
                    Dim sResult As String = x.Foo(1, 2, 3, 4)
                End Sub
        End Module
        Class C1
            Function Foo(&lt;System.[ParamArray]()&gt; ByVal x As Integer) As String
                Return "He"
            End Function
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31092: ParamArray parameters must have an array type.
                    Dim sResult As String = x.Foo(1, 2, 3, 4)
                                              ~~~
</expected>)
        End Sub

        <Fact(), WorkItem(570936, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/570936")>
        Public Sub BC31092ERR_ParamArrayWrongType_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation name="ParamArrayWrongType">
    <file name="a.vb">
        Module M1
            Sub Foo()
                    Dim x As New C1
                    Dim sResult As String = x.Foo(1)
                End Sub
        End Module
        Class C1
            Function Foo(&lt;System.[ParamArray]()&gt; ByVal x As Integer) As String
                Return "He"
            End Function
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31092: ParamArray parameters must have an array type.
                    Dim sResult As String = x.Foo(1)
                                              ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31095ERR_InvalidMyClassReference()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC31095ERR_InvalidMyClassReference">
    <file name="a.vb">
        Class cls0
            Public s2 As String
        End Class
        Class Cls1
            Inherits cls0
            Sub New(ByVal x As Short)
            End Sub
            Sub New()
                'COMPILEERROR: BC31095, "MyClass"
                MyClass.New(MyClass.s2)
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31095: Reference to object under construction is not valid when calling another constructor.
                MyClass.New(MyClass.s2)
                            ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31095ERR_InvalidMyBaseReference()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="BC31095ERR_InvalidMyBaseReference">
    <file name="a.vb">
        Class cls0
            Public s2 As String
        End Class
        Class Cls1
            Inherits cls0
            Sub New(ByVal x As Short)
            End Sub
            Sub New()
                'COMPILEERROR: BC31095, "MyBase"
                MyClass.New(MyBase.s2)
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31095: Reference to object under construction is not valid when calling another constructor.
                MyClass.New(MyBase.s2)
                            ~~~~~~
</expected>)
        End Sub

        <WorkItem(541798, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541798")>
        <Fact()>
        Public Sub BC31095ERR_InvalidMeReference()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
<compilation name="InvalidMeReference">
    <file name="a.vb">
        Class cls0
            Public s2 As String
        End Class
        Class Cls1
            Inherits cls0

            Sub New(ByVal x As String)
            End Sub

            Sub New(ByVal x As Short)
                Me.New(Me.s2) 'COMPILEERROR: BC31095, "Me"
            End Sub
        End Class
    </file>
</compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31095: Reference to object under construction is not valid when calling another constructor.
                Me.New(Me.s2) 'COMPILEERROR: BC31095, "Me"
                       ~~
</expected>)
        End Sub

        <WorkItem(541799, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/541799")>
        <Fact()>
        Public Sub BC31096ERR_InvalidImplicitMeReference()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="InvalidImplicitMeReference">
        <file name="a.vb">
        Imports System
        Module M1
            Class clsTest1
                Private strTest As String = "Hello"
                Sub New()
                    'COMPILEERROR: BC31096, "strTest"
                    Me.New(strTest)
                End Sub
                Sub New(ByVal ArgX As String)
                End Sub
            End Class
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC31096: Implicit reference to object under construction is not valid when calling another constructor.
                    Me.New(strTest)
                           ~~~~~~~
    </expected>)
        End Sub

        <Fact()>
        Public Sub BC31096ERR_InvalidImplicitMeReference_MyClass()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC31096ERR_InvalidImplicitMeReference_MyClass">
        <file name="a.vb">
        Imports System
        Module M1
            Class clsTest1
                Private strTest As String = "Hello"
                Sub New()
                    'COMPILEERROR: BC31096, "strTest"
                    MyClass.New(strTest)
                End Sub
                Sub New(ByVal ArgX As String)
                End Sub
            End Class
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC31096: Implicit reference to object under construction is not valid when calling another constructor.
                    MyClass.New(strTest)
                                ~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31096ERR_InvalidImplicitMeReference_MyBase()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="BC31096ERR_InvalidImplicitMeReference_MyBase">
        <file name="a.vb">
        Imports System
        Module M1
            Class clsTest0
                Public Sub New(ByVal strTest As String)
                End Sub
            End Class
            Class clsTest1
                Inherits clsTest0
                Private strTest As String = "Hello"
                Sub New()
                    'COMPILEERROR: BC31096, "strTest"
                    MyBase.New(strTest)
                End Sub
            End Class
        End Module
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
<expected>
BC31096: Implicit reference to object under construction is not valid when calling another constructor.
                    MyBase.New(strTest)
                               ~~~~~~~
</expected>)
        End Sub

        ' Different error
        <Fact()>
        Public Sub BC31109ERR_InAccessibleCoClass3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="C">
        <file name="a.vb">
        Imports System.Runtime.InteropServices
        &lt;Assembly: ImportedFromTypeLib("NoPIANew1-PIA2.dll")&gt;
        &lt;Assembly: Guid("f9c2d51d-4f44-45f0-9eda-c9d599b58257")&gt;
        Public Class Class1
            &lt;Guid("bd60d4b3-f50b-478b-8ef2-e777df99d810")&gt; _
            &lt;ComImport()&gt; _
            &lt;InterfaceTypeAttribute(ComInterfaceType.InterfaceIsIUnknown)&gt; _
            &lt;CoClass(GetType(FooImpl))&gt; _
            Public Interface IFoo
            End Interface
            &lt;Guid("c9dcf748-b634-4504-a7ce-348cf7c61891")&gt; _
            Friend Class FooImpl
            End Class
        End Class
    </file>
    </compilation>)
            Dim compilation1 = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
            <compilation name="InAccessibleCoClass3">
                <file name="a.vb">
        Public Module Module1
            Public Sub Main()
                Dim i1 As New Class1.IFoo(1)
                Dim i2 = New Class1.IFoo(Nothing)
            End Sub
        End Module
    </file>
            </compilation>)
            Dim compRef = New VisualBasicCompilationReference(compilation)
            compilation1 = compilation1.AddReferences(compRef)
            compilation1.VerifyDiagnostics(Diagnostic(ERRID.ERR_InAccessibleCoClass3, "New Class1.IFoo(1)").WithArguments("Class1.FooImpl", "Class1.IFoo", "Friend"),
                                            Diagnostic(ERRID.ERR_InAccessibleCoClass3, "New Class1.IFoo(Nothing)").WithArguments("Class1.FooImpl", "Class1.IFoo", "Friend"))
        End Sub

        <WorkItem(6977, "DevDiv_Projects/Roslyn")>
        <Fact()>
        Public Sub BC31110ERR_MissingValuesForArraysInApplAttrs()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
    <compilation name="MissingValuesForArraysInApplAttrs">
        <file name="a.vb"><![CDATA[
Imports System

<AttributeUsage(AttributeTargets.All)>
Public Class MyAttribute
    Inherits Attribute
    Public Sub New(ByVal o As Object)
    End Sub
End Class

Namespace AttributeRegress003
    Friend Module AttributeRegress003mod
        'COMPILEERROR: BC31110, "{}"
        <My(New Integer(3) {})> Class Test
        End Class
    End Module
End Namespace
    ]]></file>
    </compilation>).VerifyDiagnostics(Diagnostic(ERRID.ERR_MissingValuesForArraysInApplAttrs, "{}"))

        End Sub

        <Fact()>
        Public Sub BC31102ERR_NoAccessibleSet()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoAccessibleSet">
        <file name="a.vb">
        Class A
            Shared Property P
                Get
                    Return Nothing
                End Get
                Private Set
                End Set
            End Property
            Property Q
                Get
                    Return Nothing
                End Get
                Private Set
                End Set
            End Property
        End Class
        Class B
            Sub M(ByVal x As A)
                A.P = Nothing
                x.Q = Nothing
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC31102: 'Set' accessor of property 'P' is not accessible.
                A.P = Nothing
                ~~~~~~~~~~~~~
BC31102: 'Set' accessor of property 'Q' is not accessible.
                x.Q = Nothing
                ~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31103ERR_NoAccessibleGet()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="NoAccessibleGet">
        <file name="a.vb">
        Class A
            Shared Property P
                Private Get
                    Return Nothing
                End Get
                Set
                End Set
            End Property
            Property Q
                Private Get
                    Return Nothing
                End Get
                Set
                End Set
            End Property
        End Class
        Class B
            Sub M(ByVal x As A)
                N(A.P)
                N(x.Q)
            End Sub
            Sub N(ByVal o As Object)
            End Sub
        End Class
        </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC31103: 'Get' accessor of property 'P' is not accessible.
                N(A.P)
                  ~~~
BC31103: 'Get' accessor of property 'Q' is not accessible.
                N(x.Q)
                  ~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31143ERR_DelegateBindingIncompatible2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40(
    <compilation name="DelegateBindingIncompatible2">
        <file name="a.vb">
        Public Class C1
            Delegate Function FunDel(ByVal i As Integer, ByVal d As Double) As Integer
            Function ExampleMethod1(ByVal m As Integer, ByVal aDate As Date) As Integer
                Return 1
            End Function
            Sub Main()
                Dim d1 As FunDel = AddressOf ExampleMethod1
            End Sub
        End Class
    </file>
    </compilation>)
            CompilationUtils.AssertTheseDiagnostics(compilation,
    <expected>
BC31143: Method 'Public Function ExampleMethod1(m As Integer, aDate As Date) As Integer' does not have a signature compatible with delegate 'Delegate Function C1.FunDel(i As Integer, d As Double) As Integer'.
                Dim d1 As FunDel = AddressOf ExampleMethod1
                                             ~~~~~~~~~~~~~~
</expected>)
        End Sub

        <Fact()>
        Public Sub BC31148ERR_UndefinedXmlPrefix()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns:p0="http://roslyn/">
Module M
    Private F1 = GetXmlNamespace(p1)
    Private F2 = <%= GetXmlNamespace(p0) %>
    Private F3 = <%= GetXmlNamespace(p3) %>
    Private F4 = <p4:x xmlns:p4="http://roslyn/"><%= GetXmlNamespace(p4) %></p4:x>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31148: XML namespace prefix 'p1' is not defined.
    Private F1 = GetXmlNamespace(p1)
                                 ~~
BC31172: An embedded expression cannot be used here.
    Private F2 = <%= GetXmlNamespace(p0) %>
                 ~~~~~~~~~~~~~~~~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private F3 = <%= GetXmlNamespace(p3) %>
                 ~~~~~~~~~~~~~~~~~~~~~~~~~~
BC31148: XML namespace prefix 'p3' is not defined.
    Private F3 = <%= GetXmlNamespace(p3) %>
                                     ~~
BC31148: XML namespace prefix 'p4' is not defined.
    Private F4 = <p4:x xmlns:p4="http://roslyn/"><%= GetXmlNamespace(p4) %></p4:x>
                                                                     ~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31148ERR_UndefinedXmlPrefix_1()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports System.Xml.Linq
Class C
    Private F1 As XElement = <p1:a q1:b="c" xmlns:p1="..." xmlns:q1="..."/>
    Private F2 As XElement = <p2:a q2:b="c"><b xmlns:p2="..." xmlns:q2="..."/></p2:a>
    Private F3 As String = <p3:a q3:b="c" xmlns:p3="..." xmlns:q3="..."/>.<p3:a>.@q3:b
End Class
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31148: XML namespace prefix 'p2' is not defined.
    Private F2 As XElement = <p2:a q2:b="c"><b xmlns:p2="..." xmlns:q2="..."/></p2:a>
                              ~~
BC31148: XML namespace prefix 'q2' is not defined.
    Private F2 As XElement = <p2:a q2:b="c"><b xmlns:p2="..." xmlns:q2="..."/></p2:a>
                                   ~~
BC31148: XML namespace prefix 'p3' is not defined.
    Private F3 As String = <p3:a q3:b="c" xmlns:p3="..." xmlns:q3="..."/>.<p3:a>.@q3:b
                                                                           ~~
BC31148: XML namespace prefix 'q3' is not defined.
    Private F3 As String = <p3:a q3:b="c" xmlns:p3="..." xmlns:q3="..."/>.<p3:a>.@q3:b
                                                                                  ~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31148ERR_UndefinedXmlPrefix_2()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns:p0="...">
Module M
    Private F = <x1 xmlns:p1="...">
                    <x2 xmlns:p2="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                        <%=
                            <x3>
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                    <%=
                                        <x5>
                                            <x6 xmlns:p6="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                <%=
                                                    <x7 xmlns:p7="...">
                                                        <x8 xmlns:p8="...">
                                                            <x9 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="..."/>
                                                        </x8>
                                                    </x7>
                                                %>
                                            </x6>
                                        </x5>
                                    %>
                                </x4>
                            </x3>
                        %>
                    </x2>
                </x1>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31148: XML namespace prefix 'p6' is not defined.
                    <x2 xmlns:p2="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                           ~~
BC31148: XML namespace prefix 'p7' is not defined.
                    <x2 xmlns:p2="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                       ~~
BC31148: XML namespace prefix 'p8' is not defined.
                    <x2 xmlns:p2="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                                   ~~
BC31148: XML namespace prefix 'p1' is not defined.
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                ~~
BC31148: XML namespace prefix 'p2' is not defined.
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                            ~~
BC31148: XML namespace prefix 'p6' is not defined.
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                        ~~
BC31148: XML namespace prefix 'p7' is not defined.
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                    ~~
BC31148: XML namespace prefix 'p8' is not defined.
                                <x4 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                                ~~
BC31148: XML namespace prefix 'p1' is not defined.
                                            <x6 xmlns:p6="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                           ~~
BC31148: XML namespace prefix 'p2' is not defined.
                                            <x6 xmlns:p6="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                       ~~
BC31148: XML namespace prefix 'p7' is not defined.
                                            <x6 xmlns:p6="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                                               ~~
BC31148: XML namespace prefix 'p8' is not defined.
                                            <x6 xmlns:p6="..." p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="...">
                                                                                                                           ~~
BC31148: XML namespace prefix 'p1' is not defined.
                                                            <x9 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="..."/>
                                                                            ~~
BC31148: XML namespace prefix 'p2' is not defined.
                                                            <x9 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="..."/>
                                                                                        ~~
BC31148: XML namespace prefix 'p6' is not defined.
                                                            <x9 p0:a0="..." p1:a1="..." p2:a2="..." p6:a6="..." p7:a7="..." p8:a8="..."/>
                                                                                                    ~~
]]></errors>)
        End Sub

        <WorkItem(531633, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/531633")>
        <Fact()>
        Public Sub BC31148ERR_UndefinedXmlPrefix_3()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Module M
    Sub Main()
        Dim x As Object
        x = <x/>.@Return:a
    End Sub
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31148: XML namespace prefix 'Return' is not defined.
        x = <x/>.@Return:a
                  ~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31149ERR_DuplicateXmlAttribute()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Module M
    Private F1 = <x a="b" a="c" A="d" A="e"/>
    Private F2 = <x a="b" a="c" a="d" xmlns="http://roslyn"/>
    Private F3 = <x p:a="b" p:a="c" xmlns:p="http://roslyn"/>
    Private F4 = <x xmlns:a="b" xmlns:a="c"/>
    Private F5 = <x p:a="b" q:a="c" xmlns:p="http://roslyn" xmlns:q="http://roslyn"/>
    Private F6 = <x p:a="b" P:a="c" xmlns:p="http://roslyn/p" xmlns:P="http://roslyn/P"/>
    Private F7 = <x a="b" <%= "a" %>="c" <%= "a" %>="d"/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31149: Duplicate XML attribute 'a'.
    Private F1 = <x a="b" a="c" A="d" A="e"/>
                          ~
BC31149: Duplicate XML attribute 'A'.
    Private F1 = <x a="b" a="c" A="d" A="e"/>
                                      ~
BC31149: Duplicate XML attribute 'a'.
    Private F2 = <x a="b" a="c" a="d" xmlns="http://roslyn"/>
                          ~
BC31149: Duplicate XML attribute 'a'.
    Private F2 = <x a="b" a="c" a="d" xmlns="http://roslyn"/>
                                ~
BC31149: Duplicate XML attribute 'p:a'.
    Private F3 = <x p:a="b" p:a="c" xmlns:p="http://roslyn"/>
                            ~~~
BC31149: Duplicate XML attribute 'xmlns:a'.
    Private F4 = <x xmlns:a="b" xmlns:a="c"/>
                                ~~~~~~~
BC31149: Duplicate XML attribute 'q:a'.
    Private F5 = <x p:a="b" q:a="c" xmlns:p="http://roslyn" xmlns:q="http://roslyn"/>
                            ~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31149ERR_DuplicateXmlAttribute_1()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Option Strict On
Imports <xmlns="http://roslyn/">
Imports <xmlns:p="http://roslyn/">
Imports <xmlns:q="">
Class C
    Private Shared F1 As Object = <x a="b" a="c"/>
    Private Shared F2 As Object = <x p:a="b" a="c"/>
    Private Shared F3 As Object = <x q:a="b" a="c"/>
End Class
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31149: Duplicate XML attribute 'a'.
    Private Shared F1 As Object = <x a="b" a="c"/>
                                           ~
BC31149: Duplicate XML attribute 'a'.
    Private Shared F3 As Object = <x q:a="b" a="c"/>
                                             ~
]]></errors>)
        End Sub

        ' Should report duplicate xmlns attributes, even for xmlns
        ' attributes that match Imports since those have special handling.
        <Fact()>
        Public Sub BC31149ERR_DuplicateXmlAttribute_2()
            Dim options = TestOptions.ReleaseDll.WithGlobalImports(GlobalImport.Parse({"<xmlns:p=""http://roslyn/p"">"}))
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Option Strict On
Imports <xmlns:q="http://roslyn/q">
Module M
    Private F1 As Object = <x xmlns:p="http://roslyn/p" xmlns:p="http://roslyn/other"/>
    Private F2 As Object = <x xmlns:q="http://roslyn/other" xmlns:q="http://roslyn/q"/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences, options:=options)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31149: Duplicate XML attribute 'xmlns:p'.
    Private F1 As Object = <x xmlns:p="http://roslyn/p" xmlns:p="http://roslyn/other"/>
                                                        ~~~~~~~
BC31149: Duplicate XML attribute 'xmlns:q'.
    Private F2 As Object = <x xmlns:q="http://roslyn/other" xmlns:q="http://roslyn/q"/>
                                                            ~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31152ERR_ReservedXmlPrefix()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns="http://roslyn/">
Imports <xmlns:xml="http://roslyn/xml">
Imports <xmlns:xmlns="http://roslyn/xmlns">
Imports <xmlns:Xml="http://roslyn/xml">
Imports <xmlns:Xmlns="http://roslyn/xmlns">
Module M
    Private F1 As Object = <x
                               xmlns="http://roslyn/"
                               xmlns:xml="http://roslyn/xml"
                               xmlns:xmlns="http://roslyn/xmlns"/>
    Private F2 As Object = <x
                               xmlns:XML="http://roslyn/xml"
                               xmlns:XMLNS="http://roslyn/xmlns"/>
    Private F3 As Object = <x
                               xmlns=""
                               xmlns:xml=""
                               xmlns:xmlns=""/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31152: XML namespace prefix 'xml' is reserved for use by XML and the namespace URI cannot be changed.
Imports <xmlns:xml="http://roslyn/xml">
               ~~~
BC31152: XML namespace prefix 'xmlns' is reserved for use by XML and the namespace URI cannot be changed.
Imports <xmlns:xmlns="http://roslyn/xmlns">
               ~~~~~
BC31152: XML namespace prefix 'xml' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xml="http://roslyn/xml"
                                     ~~~
BC31152: XML namespace prefix 'xmlns' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xmlns="http://roslyn/xmlns"/>
                                     ~~~~~
BC31152: XML namespace prefix 'xml' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xml=""
                                     ~~~
BC31152: XML namespace prefix 'xmlns' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xmlns=""/>
                                     ~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31152ERR_ReservedXmlPrefix_1()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns:xml="http://www.w3.org/XML/1998/namespace">
Imports <xmlns:xmlns="http://www.w3.org/XML/1998/namespace">
Module M
    Private F1 As Object = <x
                               xmlns:xml="http://www.w3.org/2000/xmlns/"
                               xmlns:xmlns="http://www.w3.org/2000/xmlns/"/>
    Private F2 As Object = <x
                               xmlns:xml="http://www.w3.org/XML/1998/NAMESPACE"/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31152: XML namespace prefix 'xmlns' is reserved for use by XML and the namespace URI cannot be changed.
Imports <xmlns:xmlns="http://www.w3.org/XML/1998/namespace">
               ~~~~~
BC31152: XML namespace prefix 'xml' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xml="http://www.w3.org/2000/xmlns/"
                                     ~~~
BC31152: XML namespace prefix 'xmlns' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xmlns="http://www.w3.org/2000/xmlns/"/>
                                     ~~~~~
BC31152: XML namespace prefix 'xml' is reserved for use by XML and the namespace URI cannot be changed.
                               xmlns:xml="http://www.w3.org/XML/1998/NAMESPACE"/>
                                     ~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31168ERR_NoXmlAxesLateBinding()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="a.vb"><![CDATA[
Module M1
    Sub M()
        Dim a = Nothing
        Dim b As Object = Nothing
        Dim c As Object
        c = a.<x>
        c = b.@a
    End Sub
End Module
    ]]></file>
    <file name="b.vb"><![CDATA[
Option Strict On
Module M2
    Sub M()
        Dim a As Object = Nothing
        Dim b = Nothing
        Dim c As Object
        c = a...<x>
        c = b.@<a>
    End Sub
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31168: XML axis properties do not support late binding.
        c = a.<x>
            ~~~~~
BC31168: XML axis properties do not support late binding.
        c = b.@a
            ~~~~
BC31168: XML axis properties do not support late binding.
        c = a...<x>
            ~~~~~~~
BC31168: XML axis properties do not support late binding.
        c = b.@<a>
            ~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31172ERR_EmbeddedExpression()
            Dim options = TestOptions.ReleaseDll.WithGlobalImports(GlobalImport.Parse({"<xmlns:p=<%= M1.F %>>"}))
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns:q=<%= M1.F %>>
Module M1
    Private F = "..."
End Module
Module M2
    Public F = <x xmlns=<%= M1.F %>/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences, options:=options)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31172: Error in project-level import '<xmlns:p=<%= M1.F %>>' at '<%= M1.F %>' : An embedded expression cannot be used here.
BC31172: An embedded expression cannot be used here.
Imports <xmlns:q=<%= M1.F %>>
                 ~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Public F = <x xmlns=<%= M1.F %>/>
                        ~~~~~~~~~~~
BC30389: 'M1.F' is not accessible in this context because it is 'Private'.
    Public F = <x xmlns=<%= M1.F %>/>
                            ~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31183ERR_ReservedXmlNamespace()
            Dim options = TestOptions.ReleaseDll.WithGlobalImports(GlobalImport.Parse({"<xmlns=""http://www.w3.org/XML/1998/namespace"">", "<xmlns:p1=""http://www.w3.org/2000/xmlns/"">"}))
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns="http://www.w3.org/XML/1998/namespace">
Imports <xmlns:p2="http://www.w3.org/2000/xmlns/">
Module M
    Private F1 As Object = <x
                               xmlns="http://www.w3.org/2000/xmlns/"
                               xmlns:p3="http://www.w3.org/XML/1998/namespace"/>
    Private F2 As Object = <x
                               xmlns="http://www.w3.org/2000/XMLNS/"
                               xmlns:p4="http://www.w3.org/XML/1998/NAMESPACE"/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences, options:=options)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31183: Error in project-level import '<xmlns:p1="http://www.w3.org/2000/xmlns/">' at '"http://www.w3.org/2000/xmlns/"' : Prefix 'p1' cannot be bound to namespace name reserved for 'xmlns'.
BC31183: Error in project-level import '<xmlns="http://www.w3.org/XML/1998/namespace">' at '"http://www.w3.org/XML/1998/namespace"' : Prefix '' cannot be bound to namespace name reserved for 'xml'.
BC31183: Prefix '' cannot be bound to namespace name reserved for 'xml'.
Imports <xmlns="http://www.w3.org/XML/1998/namespace">
               ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC31183: Prefix 'p2' cannot be bound to namespace name reserved for 'xmlns'.
Imports <xmlns:p2="http://www.w3.org/2000/xmlns/">
                  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC31183: Prefix '' cannot be bound to namespace name reserved for 'xmlns'.
                               xmlns="http://www.w3.org/2000/xmlns/"
                                     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
BC31183: Prefix 'p3' cannot be bound to namespace name reserved for 'xml'.
                               xmlns:p3="http://www.w3.org/XML/1998/namespace"/>
                                        ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31184ERR_IllegalDefaultNamespace()
            Dim options = TestOptions.ReleaseDll.WithGlobalImports(GlobalImport.Parse({"<xmlns="""">", "<xmlns:p="""">"}))
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns="">
Imports <xmlns:q="">
Module M
    Private F As Object = <x xmlns="" xmlns:r=""/>
End Module
    ]]></file>
</compilation>, references:=XmlReferences, options:=options)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31184: Namespace declaration with prefix cannot have an empty value inside an XML literal.
    Private F As Object = <x xmlns="" xmlns:r=""/>
                                            ~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31189ERR_IllegalXmlnsPrefix()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="c.vb"><![CDATA[
Imports <xmlns:p="">
Imports <xmlns:XMLNS="">
Module M
    Private F1 As Object = <xmlns/>
    Private F2 As Object = <xmlns:x/>
    Private F3 As Object = <p:xmlns/>
    Private F4 As Object = <XMLNS:x/>
    Private F5 As Object = <x/>.<xmlns>
    Private F6 As Object = <x/>.<xmlns:y>
    Private F7 As Object = <x/>.<p:xmlns>
    Private F8 As Object = <x/>.<XMLNS:y>
    Private F9 As Object = <x/>.@xmlns
    Private F10 As Object = <x/>.@xmlns:z
End Module
    ]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31189: Element names cannot use the 'xmlns' prefix.
    Private F2 As Object = <xmlns:x/>
                            ~~~~~
BC31189: Element names cannot use the 'xmlns' prefix.
    Private F6 As Object = <x/>.<xmlns:y>
                                 ~~~~~
]]></errors>)
        End Sub

        ' No ref to system.xml.dll	
        <Fact()>
        Public Sub BC31190ERR_XmlFeaturesNotAvailable()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
        Module M1
            Sub Foo()
                Dim x = Function() <aoeu>
                <%= 5 %>
                <%= (Function() "five")() %>
            </aoeu>
                Dim y = Function() <aoeu val=<%= (Function() <htns></htns>)().ToString() %>/>
            End Sub
        End Module
]]></file>
</compilation>)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
                Dim x = Function() <aoeu>
                                   ~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
                Dim y = Function() <aoeu val=<%= (Function() <htns></htns>)().ToString() %>/>
                                   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31190ERR_XmlFeaturesNotAvailable_1()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Imports <xmlns:p="...">
Module M
    Private A = <a><b><%= <c/> %></b></a>
    Private B = <a b=<%= <c/> %>/>
    Private C = <a/>.<b>.<c>
    Private D = <%= A %>
    Private E = <%= <x><%= A %></x> %>
    Private F = <a/>.<a>.<b>
    Private G = <a b="c"/>.<a>.@b
    Private H = <a/>...<b>
    Private J = <!-- comment -->
    Private K = <?xml version="1.0"?><x/>
End Module
]]></file>
</compilation>)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private A = <a><b><%= <c/> %></b></a>
                ~~~~~~~~~~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private B = <a b=<%= <c/> %>/>
                ~~~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private C = <a/>.<b>.<c>
                ~~~~~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private D = <%= A %>
                ~~~~~~~~
BC31172: An embedded expression cannot be used here.
    Private E = <%= <x><%= A %></x> %>
                ~~~~~~~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private E = <%= <x><%= A %></x> %>
                    ~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private F = <a/>.<a>.<b>
                ~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private G = <a b="c"/>.<a>.@b
                ~~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private H = <a/>...<b>
                ~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private J = <!-- comment -->
                ~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private K = <?xml version="1.0"?><x/>
                ~~~~~~~~~~~~~~~~~~~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC31190ERR_XmlFeaturesNotAvailable_2()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb">
Module M
    Private F1 = &lt;x&gt;&lt;![CDATA[str]]&gt;&lt;/&gt;
    Private F2 = &lt;![CDATA[str]]&gt;
End Module
</file>
</compilation>)
            compilation.AssertTheseDiagnostics(<errors>
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private F1 = &lt;x&gt;&lt;![CDATA[str]]&gt;&lt;/&gt;
                 ~~~~~~~~~~~~~~~~~~~~~
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private F2 = &lt;![CDATA[str]]&gt;
                 ~~~~~~~~~~~~~~~
</errors>)
        End Sub

        <Fact()>
        Public Sub BC31190ERR_XmlFeaturesNotAvailable_3()
            Dim compilation = CompilationUtils.CreateCompilationWithMscorlib40AndVBRuntime(
<compilation>
    <file name="a.vb"><![CDATA[
Module M
    Private F = GetXmlNamespace()
End Module
]]></file>
</compilation>)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC31190: XML literals and XML axis properties are not available. Add references to System.Xml, System.Xml.Linq, and System.Core or other assemblies declaring System.Linq.Enumerable, System.Xml.Linq.XElement, System.Xml.Linq.XName, System.Xml.Linq.XAttribute and System.Xml.Linq.XNamespace types.
    Private F = GetXmlNamespace()
                ~~~~~~~~~~~~~~~~~
]]></errors>)
        End Sub

        <Fact()>
        Public Sub BC42361WRN_UseValueForXmlExpression3()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict Off
Imports System.Collections.Generic
Imports System.Xml.Linq
Class A
End Class
NotInheritable Class B
End Class
Module M
    Sub M(Of T As Class)()
        Dim _o As Object
        Dim _s As String
        Dim _a As A
        Dim _b As B
        Dim _t As T
        _o = <x/>.<y>
        _o = CType(<x/>.<y>, Object)
        _o = DirectCast(<x/>.<y>, Object)
        _o = TryCast(<x/>.<y>, Object)
        _s = <x/>.<y>
        _s = CType(<x/>.<y>, String)
        _s = DirectCast(<x/>.<y>, String)
        _s = TryCast(<x/>.<y>, String)
        _a = <x/>.<y>
        _a = CType(<x/>.<y>, A)
        _a = DirectCast(<x/>.<y>, A)
        _a = TryCast(<x/>.<y>, A)
        _b = <x/>.<y>
        _b = CType(<x/>.<y>, B)
        _b = DirectCast(<x/>.<y>, B)
        _b = TryCast(<x/>.<y>, B)
        _t = <x/>.<y>
        _t = CType(<x/>.<y>, T)
        _t = DirectCast(<x/>.<y>, T)
        _t = TryCast(<x/>.<y>, T)
    End Sub
End Module
]]></file>
</compilation>, references:=XmlReferences)
            compilation.AssertTheseDiagnostics(<errors><![CDATA[
BC42361: Cannot convert 'IEnumerable(Of XElement)' to 'String'. You can use the 'Value' property to get the string value of the first element of 'IEnumerable(Of XElement)'.
        _s = <x/>.<y>
             ~~~~~~~~
BC42361: Cannot convert 'IEnumerable(Of XElement)' to 'String'. You can use the 'Value' property to get the string value of the first element of 'IEnumerable(Of XElement)'.
        _s = CType(<x/>.<y>, String)
                   ~~~~~~~~
BC42361: Cannot convert 'IEnumerable(Of XElement)' to 'String'. You can use the 'Value' property to get the string value of the first element of 'IEnumerable(Of XElement)'.
        _s = DirectCast(<x/>.<y>, String)
                        ~~~~~~~~
BC42361: Cannot convert 'IEnumerable(Of XElement)' to 'String'. You can use the 'Value' property to get the string value of the first element of 'IEnumerable(Of XElement)'.
        _s = TryCast(<x/>.<y>, String)
                     ~~~~~~~~
BC42322: Runtime errors might occur when converting 'IEnumerable(Of XElement)' to 'B'.
        _b = <x/>.<y>
             ~~~~~~~~
BC42322: Runtime errors might occur when converting 'IEnumerable(Of XElement)' to 'B'.
        _b = CType(<x/>.<y>, B)
                   ~~~~~~~~
BC42322: Runtime errors might occur when converting 'IEnumerable(Of XElement)' to 'B'.
        _b = DirectCast(<x/>.<y>, B)
                        ~~~~~~~~
BC42322: Runtime errors might occur when converting 'IEnumerable(Of XElement)' to 'B'.
        _b = TryCast(<x/>.<y>, B)
                     ~~~~~~~~
]]></errors>)
        End Sub

        ' Same as above but with "Option Strict On".
        <Fact()>
        Public Sub BC42361WRN_UseValueForXmlExpression3_1()
            Dim compilation = CreateCompilationWithMscorlib40AndVBRuntimeAndReferences(
<compilation>
    <file name="a.vb"><![CDATA[
Option Strict On
Imports System.Collections.Generic
Imports System.Xml.Linq
Class A
End Class
NotInheritable Class B
End Class
Module M
    Sub M(Of T As Class)()
        Dim _o As Object
        Dim _s As String
        Dim _a As A
        Dim _b As B
        Dim _t As T
        _o = <x/>.<y>
        _o = CType(<x/>.<y>, Object)
        _o = DirectCast(<x/>.<y>, Object)
        _o = TryCast(<x/>.<y>, Object)
        _s = <x/>.<y>
        _s = CType(<x/>.<y>, String)
        _s = DirectCast(<x/>.<y>, String)
        _s = TryCast(<x/>.<y>, String)
        _a = <x/>.<y>
        _a = CType(<x/>.<y>, A)
        _a = DirectCast(<x/>.<y>, A)
        _a = TryCast(<x/>.<y>, A)
        _b = <x/>.<y>
        _b = CType(<x/>.<y>, B)
        _b = DirectCast(<x/>.<y>, B)
        _b = TryCast(<x/>.<y>, B)
        _t = <x/>.<y>
        _t = CType(<x/>.<y>, T)
        _t = DirectCast(<x/>.<y>, T)
        _t = TryCast(<x/>.<y>, T)
    End 