/**
 ******************************************************************************
 * @file    shci.h
 * @author  MCD Application Team
 * @brief   HCI command for the system channel
 ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2019 STMicroelectronics. 
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the 
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __SHCI_H
#define __SHCI_H

#ifdef __cplusplus
extern "C" {
#endif

  /* Includes ------------------------------------------------------------------*/
#include "mbox_def.h" /* Requested to expose the MB_WirelessFwInfoTable_t structure */

  /* Exported types ------------------------------------------------------------*/

  /* SYSTEM EVENT */
  typedef enum
  {
    WIRELESS_FW_RUNNING = 0x00,
    RSS_FW_RUNNING = 0x01,
  } SHCI_SysEvt_Ready_Rsp_t;

  /* ERROR CODES
   *
   * These error codes are detected on M0 side and are send back to the M4 via a system
   * notification message. It is up to the application running on M4 to manage these errors
   *
   * These errors can be generated by all layers (low level driver, stack, framework infrastructure, etc..)
   */
   typedef enum
   {
     ERR_BLE_INIT = 0,
     ERR_THREAD_LLD_FATAL_ERROR = 125, /* The LLD driver used on 802_15_4 detected a fatal error            */
     ERR_THREAD_UNKNOWN_CMD = 126,     /* The command send by the M4 to control the Thread stack is unknown */
   } SCHI_SystemErrCode_t;

#define SHCI_EVTCODE                    ( 0xFF )
#define SHCI_SUB_EVT_CODE_BASE          ( 0x9200 )

  /**
   * THE ORDER SHALL NOT BE CHANGED TO GUARANTEE COMPATIBILITY WITH THE CPU1 DEFINITION
   */
  typedef enum
  {
    SHCI_SUB_EVT_CODE_READY =  SHCI_SUB_EVT_CODE_BASE,
    SHCI_SUB_EVT_ERROR_NOTIF,
  } SHCI_SUB_EVT_CODE_t;

  typedef PACKED_STRUCT{
    SHCI_SysEvt_Ready_Rsp_t sysevt_ready_rsp;
  } SHCI_C2_Ready_Evt_t;

  typedef PACKED_STRUCT{
    SCHI_SystemErrCode_t errorCode;
  } SHCI_C2_ErrorNotif_Evt_t;

  /* SYSTEM COMMAND */
  typedef PACKED_STRUCT
  {
    uint32_t MetaData[3];
  } SHCI_Header_t;

  typedef enum
  {
    SHCI_Success = 0x00,
    SHCI_UNKNOWN_CMD = 0x01,
    SHCI_ERR_UNSUPPORTED_FEATURE = 0x11,
    SHCI_ERR_INVALID_HCI_CMD_PARAMS = 0x12,
    SHCI_FUS_CMD_NOT_SUPPORTED = 0xFF,
  } SHCI_CmdStatus_t;

  typedef enum
  {
    SHCI_8BITS =  0x01,
    SHCI_16BITS = 0x02,
    SHCI_32BITS = 0x04,
  } SHCI_Busw_t;

#define SHCI_OGF                        ( 0x3F )
#define SHCI_OCF_BASE                   ( 0x50 )

  /**
   * THE ORDER SHALL NOT BE CHANGED TO GUARANTEE COMPATIBILITY WITH THE CPU2 DEFINITION
   */
  typedef enum
  {
    SHCI_OCF_C2_RESERVED1 =  SHCI_OCF_BASE,
    SHCI_OCF_C2_RESERVED2,
    SHCI_OCF_C2_FUS_GET_STATE,
    SHCI_OCF_C2_FUS_RESERVED1,
    SHCI_OCF_C2_FUS_FW_UPGRADE,
    SHCI_OCF_C2_FUS_FW_DELETE,
    SHCI_OCF_C2_FUS_UPDATE_AUTH_KEY,
    SHCI_OCF_C2_FUS_LOCK_AUTH_KEY,
    SHCI_OCF_C2_FUS_STORE_USR_KEY,
    SHCI_OCF_C2_FUS_LOAD_USR_KEY,
    SHCI_OCF_C2_FUS_START_WS,
    SHCI_OCF_C2_FUS_RESERVED2,
    SHCI_OCF_C2_FUS_RESERVED3,
    SHCI_OCF_C2_FUS_LOCK_USR_KEY,
    SHCI_OCF_C2_FUS_RESERVED5,
    SHCI_OCF_C2_FUS_RESERVED6,
    SHCI_OCF_C2_FUS_RESERVED7,
    SHCI_OCF_C2_FUS_RESERVED8,
    SHCI_OCF_C2_FUS_RESERVED9,
    SHCI_OCF_C2_FUS_RESERVED10,
    SHCI_OCF_C2_FUS_RESERVED11,
    SHCI_OCF_C2_FUS_RESERVED12,
    SHCI_OCF_C2_BLE_INIT,
    SHCI_OCF_C2_THREAD_INIT,
    SHCI_OCF_C2_DEBUG_INIT,
    SHCI_OCF_C2_FLASH_ERASE_ACTIVITY,
    SHCI_OCF_C2_CONCURRENT_SET_MODE,
    SHCI_OCF_C2_FLASH_STORE_DATA,
    SHCI_OCF_C2_FLASH_ERASE_DATA,
    SHCI_OCF_C2_RADIO_ALLOW_LOW_POWER,
    SHCI_OCF_C2_MAC_802_15_4_INIT,
    SHCI_OCF_C2_REINIT,
    SHCI_OCF_C2_ZIGBEE_INIT,
    SHCI_OCF_C2_LLD_TESTS_INIT
  } SHCI_OCF_t;

#define SHCI_OPCODE_C2_FUS_GET_STATE         (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_GET_STATE)
/** No command parameters */
/** Response parameters*/
  typedef enum
  {
    FUS_STATE_NO_ERROR =  0x00,
    FUS_STATE_IMG_NOT_FOUND = 0x01,
    FUS_STATE_IMG_CORRUPT = 0x02,
    FUS_STATE_IMG_NOT_AUTHENTIC = 0x03,
    FUS_STATE_IMG_NOT_ENOUGH_SPACE = 0x04,
    FUS_STATE_ERR_UNKNOWN = 0xFF,
  } SHCI_FUS_GetState_ErrorCode_t;

#define SHCI_OPCODE_C2_FUS_RESERVED1         (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED1)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_FW_UPGRADE   (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_FW_UPGRADE)
  /** No structure for command parameters */
  /** No response parameters*/

#define SHCI_OPCODE_C2_FUS_FW_DELETE   (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_FW_DELETE)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_UPDATE_AUTH_KEY    (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_UPDATE_AUTH_KEY)
  typedef PACKED_STRUCT{
  uint8_t KeySize;
  uint8_t KeyData[64];
  } SHCI_C2_FUS_UpdateAuthKey_Cmd_Param_t;

  /** No response parameters*/

#define SHCI_OPCODE_C2_FUS_LOCK_AUTH_KEY    (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_LOCK_AUTH_KEY)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_STORE_USR_KEY    (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_STORE_USR_KEY)
  /** Command parameters */
  /* List of supported key type */
  enum
  {
    KEYTYPE_NONE =  0x00,
    KEYTYPE_SIMPLE = 0x01,
    KEYTYPE_MASTER = 0x02,
    KEYTYPE_ENCRYPTED = 0x03,
  };

  /* List of supported key size */
  enum
  {
    KEYSIZE_16 =  16,
    KEYSIZE_32 = 32,
  };

  typedef PACKED_STRUCT{
  uint8_t KeyType;
  uint8_t KeySize;
  uint8_t KeyData[32 + 12];
  } SHCI_C2_FUS_StoreUsrKey_Cmd_Param_t;

  /** Response parameters*/
  /** It responds a 1 byte value holding the index given for the stored key */

#define SHCI_OPCODE_C2_FUS_LOAD_USR_KEY         (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_LOAD_USR_KEY)
  /** Command parameters */
  /** 1 byte holding the key index value */

  /** No response parameters*/

#define SHCI_OPCODE_C2_FUS_START_WS             (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_START_WS)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED2            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED2)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED3            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED3)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_LOCK_USR_KEY         (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_LOCK_USR_KEY)
  /** Command parameters */
  /** 1 byte holding the key index value */

  /** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED5            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED5)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED6            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED6)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED7            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED7)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED8            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED8)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED9            (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED9)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED10           (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED10)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED11           (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED11)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_FUS_RESERVED12           (( SHCI_OGF << 10) + SHCI_OCF_C2_FUS_RESERVED12)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_BLE_INIT                 (( SHCI_OGF << 10) + SHCI_OCF_C2_BLE_INIT)
  /** THE ORDER SHALL NOT BE CHANGED    */
  typedef PACKED_STRUCT{
  uint8_t* pBleBufferAddress;   /**< NOT USED CURRENTLY */
  uint32_t BleBufferSize;       /**< Size of the Buffer allocated in pBleBufferAddress  */
  uint16_t NumAttrRecord;
  uint16_t NumAttrServ;
  uint16_t AttrValueArrSize;
  uint8_t NumOfLinks;
  uint8_t ExtendedPacketLengthEnable;
  uint8_t PrWriteListSize;
  uint8_t MblockCount;
  uint16_t AttMtu;
  uint16_t SlaveSca;
  uint8_t MasterSca;
  uint8_t LsSource;
  uint32_t MaxConnEventLength;
  uint16_t HsStartupTime;
  uint8_t ViterbiEnable;
  uint8_t LlOnly;
  uint8_t HwVersion;
  } SHCI_C2_Ble_Init_Cmd_Param_t;

  typedef PACKED_STRUCT{
    SHCI_Header_t Header;       /** Does not need to be initialized by the user */
    SHCI_C2_Ble_Init_Cmd_Param_t Param;
  } SHCI_C2_Ble_Init_Cmd_Packet_t;

  /** No response parameters*/

#define SHCI_OPCODE_C2_THREAD_INIT              (( SHCI_OGF << 10) + SHCI_OCF_C2_THREAD_INIT)
/** No command parameters */
/** No response parameters*/

#define SHCI_OPCODE_C2_DEBUG_INIT              (( SHCI_OGF << 10) + SHCI_OCF_C2_DEBUG_INIT)
  /** Command parameters */
    typedef PACKED_STRUCT{
      uint8_t *pGpioConfig;
      uint8_t *pTracesConfig;
      uint8_t *pGeneralConfig;
      uint8_t GpioConfigSize;
      uint8_t TracesConfigSize;
      uint8_t GeneralConfigSize;
    } SHCI_C2_DEBUG_init_Cmd_Param_t;

    typedef PACKED_STRUCT{
      SHCI_Header_t Header;       /** Does not need to be initialized by the user */
      SHCI_C2_DEBUG_init_Cmd_Param_t Param;
    } SHCI_C2_DEBUG_Init_Cmd_Packet_t;
    /** No response parameters*/

#define SHCI_OPCODE_C2_FLASH_ERASE_ACTIVITY     (( SHCI_OGF << 10) + SHCI_OCF_C2_FLASH_ERASE_ACTIVITY)
  /** Command parameters */
    typedef enum
    {
      ERASE_ACTIVITY_OFF =  0x00,
      ERASE_ACTIVITY_ON = 0x01,
    } SHCI_EraseActivity_t;

    /** No response parameters*/

#define SHCI_OPCODE_C2_CONCURRENT_SET_MODE          (( SHCI_OGF << 10) + SHCI_OCF_C2_CONCURRENT_SET_MODE)
/** command parameters */
    typedef enum
    {
      BLE_ENABLE,
      THREAD_ENABLE,
    } SHCI_C2_CONCURRENT_Mode_Param_t;
      /** No response parameters*/

#define SHCI_OPCODE_C2_FLASH_STORE_DATA          (( SHCI_OGF << 10) + SHCI_OCF_C2_FLASH_STORE_DATA)
#define SHCI_OPCODE_C2_FLASH_ERASE_DATA          (( SHCI_OGF << 10) + SHCI_OCF_C2_FLASH_ERASE_DATA)
/** command parameters */
    typedef enum
    {
      BLE_IP,
      THREAD_IP,
    } SHCI_C2_FLASH_Ip_t;
      /** No response parameters*/

#define SHCI_OPCODE_C2_RADIO_ALLOW_LOW_POWER      (( SHCI_OGF << 10) + SHCI_OCF_C2_RADIO_ALLOW_LOW_POWER)

#define SHCI_OPCODE_C2_MAC_802_15_4_INIT        (( SHCI_OGF << 10) + SHCI_OCF_C2_MAC_802_15_4_INIT)

#define SHCI_OPCODE_C2_REINIT                   (( SHCI_OGF << 10) + SHCI_OCF_C2_REINIT)

#define SHCI_OPCODE_C2_ZIGBEE_INIT                (( SHCI_OGF << 10) + SHCI_OCF_C2_ZIGBEE_INIT)

#define SHCI_OPCODE_C2_LLD_TESTS_INIT                (( SHCI_OGF << 10) + SHCI_OCF_C2_LLD_TESTS_INIT)

 /* Exported type --------------------------------------------------------*/

typedef  MB_WirelessFwInfoTable_t SHCI_WirelessFwInfoTable_t;

/*
  *   At startup, the informations relative to the wireless binary are stored in RAM trough a structure defined by
  *   SHCI_WirelessFwInfoTable_t.This structure contains 4 fields (Version,MemorySize, Stack_info and a reserved part)
  *   each of those coded on 32 bits as shown on the table below:
  *
  *
  *               |7 |6 |5 |4 |3 |2 |1 |0 |7 |6 |5 |4 |3 |2 |1 |0 |7 |6 |5 |4 |3 |2 |1 |0 |7 |6 |5 |4 |3 |2 |1 |0 |
  *               -------------------------------------------------------------------------------------------------
  * Version       |   Major version       |    Minor version      |    Sub version        | Branch    |Releas Type|
  *               -------------------------------------------------------------------------------------------------
  * MemorySize    |   SRAM2B (kB)         |    SRAM2A (kB)        |    SRAM1 (kB)         | FLASH (4kb)           |
  *               -------------------------------------------------------------------------------------------------
  * Info stack    |   Reserved            |    Reserved           |    Reserved           | Type (MAC,Thread,BLE) |
  *               -------------------------------------------------------------------------------------------------
  * Reserved      |   Reserved            |    Reserved           |    Reserved           | Reserved              |
  *               -------------------------------------------------------------------------------------------------
  *
  */

/* Field Version */
#define INFO_VERSION_MAJOR_OFFSET                   24
#define INFO_VERSION_MAJOR_MASK                     0xff000000
#define INFO_VERSION_MINOR_OFFSET                   16
#define INFO_VERSION_MINOR_MASK                     0x00ff0000
#define INFO_VERSION_SUB_OFFSET                     8
#define INFO_VERSION_SUB_MASK                       0x0000ff00
#define INFO_VERSION_BRANCH_OFFSET                  4
#define INFO_VERSION_BRANCH_MASK                    0x0000000f0
#define INFO_VERSION_TYPE_OFFSET                    0
#define INFO_VERSION_TYPE_MASK                      0x00000000f

#define INFO_VERSION_TYPE_RELEASE                   1

/* Field Memory */
#define INFO_SIZE_SRAM2B_OFFSET                     24
#define INFO_SIZE_SRAM2B_MASK                       0xff000000
#define INFO_SIZE_SRAM2A_OFFSET                     16
#define INFO_SIZE_SRAM2A_MASK                       0x00ff0000
#define INFO_SIZE_SRAM1_OFFSET                      8
#define INFO_SIZE_SRAM1_MASK                        0x0000ff00
#define INFO_SIZE_FLASH_OFFSET                      0
#define INFO_SIZE_FLASH_MASK                        0x000000ff

/* Field stack information */
#define INFO_STACK_TYPE_OFFSET                      0
#define INFO_STACK_TYPE_MASK                        0x000000ff
#define INFO_STACK_TYPE_NONE                        0

#define INFO_STACK_TYPE_BLE_STANDARD                0x1
#define INFO_STACK_TYPE_BLE_HCI                     0x2
#define INFO_STACK_TYPE_THREAD_FTD                  0x10
#define INFO_STACK_TYPE_THREAD_MTD                  0x11
#define INFO_STACK_TYPE_ZIGBEE                      0x30
#define INFO_STACK_TYPE_MAC                         0x40
#define INFO_STACK_TYPE_BLE_THREAD_FTD_STATIC       0x50
#define INFO_STACK_TYPE_802154_LLD_TESTS            0x60

typedef struct {
/**
 * Wireless Info
 */
  uint8_t VersionMajor;
  uint8_t VersionMinor;
  uint8_t VersionSub;
  uint8_t VersionBranch;
  uint8_t VersionReleaseType;
  uint8_t MemorySizeSram2B;     /*< Multiple of 1K */
  uint8_t MemorySizeSram2A;     /*< Multiple of 1K */
  uint8_t MemorySizeSram1;      /*< Multiple of 1K */
  uint8_t MemorySizeFlash;      /*< Multiple of 4K */
  uint8_t StackType;
/**
 * Fus Info
 */
  uint8_t FusVersionMajor;
  uint8_t FusVersionMinor;
  uint8_t FusVersionSub;
  uint8_t FusMemorySizeSram2B;  /*< Multiple of 1K */
  uint8_t FusMemorySizeSram2A;  /*< Multiple of 1K */
  uint8_t FusMemorySizeFlash;   /*< Multiple of 4K */
}WirelessFwInfo_t;


/* Exported functions ------------------------------------------------------- */

/**
 * For all SHCI_C2_FUS_xxx() command:
 * When the wireless FW is running on the CPU2, the command returns SHCI_FUS_CMD_NOT_SUPPORTED
 * When any FUS command is sent after the SHCI_FUS_CMD_NOT_SUPPORTED has been received,
 * the CPU2 switches on the RSS ( This reboots automatically the device )
 */
  /**
  * SHCI_C2_FUS_GetState
  * @brief Read the FUS State
  *        If the user is not interested by the Error code response, a null value may
  *        be passed as parameter
  *
  * @param  p_rsp : return the error code when the FUS State Value = 0xFF
  * @retval FUS State Values
  */
  uint8_t SHCI_C2_FUS_GetState( SHCI_FUS_GetState_ErrorCode_t *p_rsp );

  /**
  * SHCI_C2_FUS_FwUpgrade
  * @brief Request the FUS to install the CPU2 firmware update
  *
  * @param  fw_src_add: Address of the firmware image location
  * @param  fw_dest_add: Address of the firmware destination
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_FwUpgrade( uint32_t fw_src_add,  uint32_t fw_dest_add );

  /**
  * SHCI_C2_FUS_FwDelete
  * @brief Delete the wireless stack on CPU2
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_FwDelete( void );

  /**
  * SHCI_C2_FUS_UpdateAuthKey
  * @brief Request the FUS to update the authentication key
  *
  * @param  pCmdPacket
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_UpdateAuthKey( SHCI_C2_FUS_UpdateAuthKey_Cmd_Param_t *pParam );

  /**
  * SHCI_C2_FUS_LockAuthKey
  * @brief Request the FUS to prevent any future update of the authentication key
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_LockAuthKey( void );

  /**
  * SHCI_C2_FUS_StoreUsrKey
  * @brief Request the FUS to store the user key
  *
  * @param  pParam : command parameter
  * @param  p_key_index : Index allocated by the FUS to the stored key
  *
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_StoreUsrKey( SHCI_C2_FUS_StoreUsrKey_Cmd_Param_t *pParam, uint8_t *p_key_index );

  /**
  * SHCI_C2_FUS_LoadUsrKey
  * @brief Request the FUS to load the user key into the AES
  *
  * @param  key_index : index of the user key to load in AES1
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_LoadUsrKey( uint8_t key_index );

  /**
  * SHCI_C2_FUS_StartWs
  * @brief Request the FUS to reboot on the wireless stack
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_StartWs( void );

  /**
  * SHCI_C2_FUS_LockUsrKey
  * @brief Request the FUS to lock the user key so that it cannot be updated later on
  *
  * @param  key_index : index of the user key to lock
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FUS_LockUsrKey( uint8_t key_index );

  /**
  * SHCI_C2_BLE_Init
  * @brief Provides parameters and starts the BLE Stack
  *
  * @param  pCmdPacket : Parameters to be provided to the BLE Stack
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_BLE_Init( SHCI_C2_Ble_Init_Cmd_Packet_t *pCmdPacket );

  /**
  * SHCI_C2_THREAD_Init
  * @brief Starts the THREAD Stack
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_THREAD_Init( void );

  /**
  * SHCI_C2_LLDTESTS_Init
  * @brief Starts the LLD tests CLI
  *
  * @param  param_size : Nb of bytes
  * @param  p_param : pointeur with data to give from M4 to M0
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_LLDTESTS_Init( uint8_t param_size, uint8_t * p_param );

    /**
  * SHCI_C2_ZIGBEE_Init
  * @brief Starts the Zigbee Stack
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_ZIGBEE_Init( void );
  
  /**
  * SHCI_C2_DEBUG_Init
  * @brief Starts the Traces
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_DEBUG_Init( SHCI_C2_DEBUG_Init_Cmd_Packet_t *pCmdPacket );

  /**
  * SHCI_C2_FLASH_EraseActivity
  * @brief Provides the information of the start and the end of a flash erase window on the CPU1
  *
  * @param  erase_activity: Start/End of erase activity
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FLASH_EraseActivity( SHCI_EraseActivity_t erase_activity );

  /**
  * SHCI_C2_CONCURRENT_SetMode
  * @brief Enable/Disable Thread on CPU2 (M0+)
  *
  * @param  Mode: BLE or Thread enable flag
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_CONCURRENT_SetMode( SHCI_C2_CONCURRENT_Mode_Param_t Mode );

  /**
  * SHCI_C2_FLASH_StoreData
  * @brief Store Data in Flash
  *
  * @param  Ip: BLE or THREAD
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FLASH_StoreData( SHCI_C2_FLASH_Ip_t Ip );

  /**
  * SHCI_C2_FLASH_EraseData
  * @brief Erase Data in Flash
  *
  * @param  Ip: BLE or THREAD
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_FLASH_EraseData( SHCI_C2_FLASH_Ip_t Ip );

  /**
  * SHCI_C2_RADIO_AllowLowPower
  * @brief Allow or forbid IP_radio (802_15_4 or BLE) to enter in low power mode.
  *
  * @param  Ip: BLE or 802_15_5
  * @param  FlagRadioLowPowerOn: True or false
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_RADIO_AllowLowPower( SHCI_C2_FLASH_Ip_t Ip,uint8_t  FlagRadioLowPowerOn);


  /**
  * SHCI_C2_MAC_802_15_4_Init
  * @brief Starts the MAC 802.15.4 on M0
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_MAC_802_15_4_Init( void );

  /**
   * SHCI_GetWirelessFwInfo
   * @brief This function read back the informations relative to the wireless binary loaded.
   *         Refer yourself to SHCI_WirelessFwInfoTable_t structure to get the significance
   *         of the different parameters returned.
   * @param  pWirelessInfo : Pointer to WirelessFwInfo_t.
   *
   * @retval SHCI_Success
   */
  SHCI_CmdStatus_t SHCI_GetWirelessFwInfo( WirelessFwInfo_t* pWirelessInfo );

  /**
  * SHCI_C2_Reinit
  * @brief This is required to allow the CPU1 to fake a set C2BOOT when it has already been set.
  *        In order to fake a C2BOOT, the CPU1 shall :
  *        - Send SHCI_C2_Reinit()
  *        - call SEV instruction
  *        WARNING:
  *        This function is intended to be used by the SBSFU
  *
  * @param  None
  * @retval Status
  */
  SHCI_CmdStatus_t SHCI_C2_Reinit( void );

  #ifdef __cplusplus
}
#endif

#endif /*__SHCI_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
