/*
 * Test program for creating reservations
 */
#define _GNU_SOURCE	/* required for getsid */
#include "basil_alps.h"
#include "cmdline.h"

extern int my_system(const char *fmt, ...);

/* location of the 'apmgr-cancel' program or script */
static const char cancel_prog[] = "../cancel_reservation/apmgr-cancel";

struct basil_accel_param *parse_accelerator_args(const char *args)
{
	struct basil_accel_param *accel = calloc(1, sizeof(*accel));
	char *copy = strdup(args);
	char *tok = strtok(copy, ",");

	if (tok == NULL)
		fatal("invalid accelerator args '%s'", args);

	for ( ; tok; tok = strtok(NULL, ",")) {
		char *key = tok, *val = strchr(tok, '=');

		/*
		 * Parse the following key/value pairs
	     	 *  type := accelerator type, currently only "GPU" is supported
  		 *  mem  := accelerator memory in mbytes (number without units)
	  	 *  fam  := make/model string specifying the accelerator
		 */
		if (val == NULL)
			fatal("invalid key/value token '%s'", tok);

		*val++ = '\0';

		accel->type = BA_GPU;	/* fallback value if nothing is specified */

		if (!strcmp(key, "type")) {
			for (accel->type = BA_NONE; accel->type < BA_MAX; accel->type++)
				if (!strcmp(val, nam_acceltype[accel->type]))
					break;
			if (accel->type != BA_GPU)
				fatal("unknown Accelerator type '%s'", val);
		} else if (!strncmp(key, "fam", 3)) {
			strncpy(accel->family, val, sizeof(accel->family));
		} else if (!strncmp(key, "mem", 3)) {
			accel->memory_mb = strtoul(val, NULL, 0);
		} else {
			fatal("unsupported key '%s'", key);
		}
	}
	free(copy);
	return accel;
}

int main(int argc, char **argv)
{
	struct gengetopt_args_info args;
	struct nodespec *ns_head = NULL;
	struct basil_accel_param *accel_head = NULL;
	long rsvn_id, rc;

	/*
	 * 0. Sanity checks
	 */
	if (cmdline_parser(argc, argv, &args) != 0)
		exit(EXIT_FAILURE);

	if (access(cancel_prog, X_OK) != 0)
		err(1, "can not use '%s' to cancel reservation", cancel_prog);

	if (args.nodes_given) {
		ns_head = parse_node_array(args.nodes_given, args.nodes_arg);
		if (ns_head == NULL)
			errx(1, "invalid mppnodes list");
	}

	if (args.accel_given)
		accel_head = parse_accelerator_args(args.accel_arg);

	/*
	 * 1. Create the reservation
	 */
	rc = basil_reserve(args.user_arg ? : cuserid(NULL), args.batch_arg,
			   args.width_arg, args.depth_arg, args.nppn_arg,
			   args.mem_arg, ns_head, accel_head);
	if (rc < 0)
		errx(1, "RESERVE request failed with %s error",
		     is_transient_error(rc) ? "transient" : "permanent");
	rsvn_id = rc;
	printf("STEP 1: RESERVEd reservation #%ld\n", rsvn_id);

	/* ALPS takes some time to update its state */
	my_system("sleep 1; apstat -r -R %ld", rsvn_id);

	/*
	 * 2. Confirm it using the session-leader PID as admin_cookie
	 */
	rc = basil_confirm(rsvn_id, -1, getsid(0));
	if (rc < 0)
		errx(1, "CONFIRM request failed with %s error",
		     is_transient_error(rc) ? "transient" : "permanent");
	printf("\nSTEP 2: CONFIRMed reservation #%ld\n", rsvn_id);

	my_system("sleep 1; apstat -rvv -R %ld", rsvn_id);

	/*
	 * 3. Cancel it (requires sudo)
	 */
	if (my_system("sudo %s %ld", cancel_prog, rsvn_id) < 0)
		errx(1, "Failed to cancel reservation %ld", rsvn_id);
	printf("\nSTEP 3: RELEASEd reservation #%ld\n", rsvn_id);

	return EXIT_SUCCESS;
}

