// Copyright 2018 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License. See the AUTHORS file
// for names of contributors.

package main

import "strings"

type blacklist map[string]string

var hibernateBlacklists = []struct {
	versionPrefix string
	name          string
	list          blacklist
}{
	{"v2.0", "hibernateBlackList2_0", hibernateBlackList2_0},
	{"v2.1", "hibernateBlackList2_1", hibernateBlackList2_1},
	{"v2.2", "hibernateBlackList19_1", hibernateBlackList19_1},
	{"v19.1", "hibernateBlackList19_1", hibernateBlackList19_1},
}

// getHibernateBlacklistForVersion returns the appropriate hibernate blacklist
// based on the cockroach version. This check only looks to ensure that the
// prefix from hibernateBlacklists matches.
func getHibernateBlacklistForVersion(version string) (string, blacklist) {
	for _, info := range hibernateBlacklists {
		if strings.HasPrefix(version, info.versionPrefix) {
			return info.name, info.list
		}
	}
	return "", nil
}

// These are lists of known hibernate test failures.
// When the hibernate test suite is run, the results are compared to this list.
// Any passed test that is not on this list is reported as PASS - expected
// Any passed test that is on this list is reported as PASS - unexpected
// Any failed test that is on this list is reported as FAIL - expected
// Any failed test that is not on this list is reported as FAIL - unexpected
// Any test on this list that is not run is reported as FAIL - not run
//
// Please keep these lists alphabetized for easy diffing.
// After a failed run, an updated version of this blacklist should be available
// in the test log.
var hibernateBlackList19_1 = blacklist{
	"org.hibernate.id.QuotedIdentifierTest.testDirectIdPropertyAccess":                                                                                                               "24062",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testDateTimeFunctions":                                                                                                         "31708",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigDecimals":                                                                                               "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigIntegers":                                                                                               "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfDoubles":                                                                                                   "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfFloats":                                                                                                    "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfIntegers":                                                                                                  "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfLongs":                                                                                                     "5807",
	"org.hibernate.jpa.test.criteria.basic.CastTest.testCastToString":                                                                                                                "5807",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testQuotientConversion":                                                                                                     "26732",
	"org.hibernate.jpa.test.emops.RemoveTest.testUpdatedAndRemove":                                                                                                                   "6583",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                           "6583",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                          "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                            "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                           "6583",
	"org.hibernate.jpa.test.lob.BlobTest.testBlobSerialization":                                                                                                                      "26725",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticLock":                                                                                                              "6583",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticWriteLockNoWait":                                                                                                   "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithPessimisticWriteLockTimeoutException":                                                                                          "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithTimeoutHint":                                                                                                                   "6583",
	"org.hibernate.jpa.test.lock.LockTest.testLockWriteOnUnversioned":                                                                                                                "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockSkipLocked":                                                                                               "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockWithoutNoWait":                                                                                            "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testEntityLockModeStateAfterQueryLocking":                                                                                          "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementOverall":                                                                                             "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementSpecific":                                                                                            "6583",
	"org.hibernate.jpa.test.lock.StatementIsClosedAfterALockExceptionTest.testStatementIsClosed":                                                                                     "6583",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuery":                                                                                 "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuerySpecifyingType":                                                                   "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testMultipleConstructorResultNativeQuery":                                                                         "unknown",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testAlwaysTransactionalOperations":                                                                                   "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testMergeWhenExtended":                                                                                               "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackClearPC":                                                                                                 "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackExceptionOnOptimisticLockException":                                                                      "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testSetRollbackOnlyAndFlush":                                                                                         "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionAndContains":                                                                                          "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionCommitDoesNotFlush":                                                                                   "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionalOperationsWhenExtended":                                                                             "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.IdMapManyToOneSpecjTest.testComplexIdClass":                                                                    "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.lazy.CompositeKeyDeleteTest.testRemove":                                                                        "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b2.IdClassGeneratedValueManyToOneTest.testComplexIdClass":                                                                   "6583",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testSerialized":                                                                                             "26725",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testVersioning":                                                                                             "27274",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testClub":                                                                                              "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testIqFilter":                                                                                          "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testNonHumanFilter":                                                                                    "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testPregnantFilter":                                                                                    "5807",
	"org.hibernate.test.annotations.id.IdTest.testLowAllocationSize":                                                                                                                 "6583",
	"org.hibernate.test.annotations.id.IdTest.testTableGenerator":                                                                                                                    "6583",
	"org.hibernate.test.annotations.id.sequences.IdTest.testLowAllocationSize":                                                                                                       "6583",
	"org.hibernate.test.annotations.id.sequences.IdTest.testTableGenerator":                                                                                                          "6583",
	"org.hibernate.test.annotations.identifiercollection.IdentifierCollectionTest.testIdBag":                                                                                         "6583",
	"org.hibernate.test.annotations.lob.LobTest.testBlob":                                                                                                                            "26725",
	"org.hibernate.test.annotations.lob.LobTest.testClob":                                                                                                                            "26725",
	"org.hibernate.test.annotations.lob.LobTest.testSerializableToBlob":                                                                                                              "26725",
	"org.hibernate.test.annotations.lob.MaterializedBlobTest.testSaving":                                                                                                             "26725",
	"org.hibernate.test.annotations.lob.SerializableToBlobTypeTest.testPersist":                                                                                                      "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testBlob":                                                                                                                   "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testClob":                                                                                                                   "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testSerializableToBlob":                                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedByteArray":                                                                                              "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedCharArray":                                                                                              "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveByteArray":                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveCharArray":                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedString":                                                                                                 "26725",
	"org.hibernate.test.annotations.manytoone.referencedcolumnname.ManyToOneReferencedColumnNameTest.testReoverableExceptionInFkOrdering":                                            "24062",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithAliasOneToOneJoin":                                                                       "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinCollection":                                                                 "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinOnetoOne":                                                                   "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testNaturalKeyLookupWithConstraint":                                                                      "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSimpleImmutableNaturalKeyLookup":                                                                     "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSubCriteriaOneToOneJoin":                                                                             "6583",
	"org.hibernate.test.annotations.onetoone.hhh9798.OneToOneJoinTableTest.storeNonUniqueRelationship":                                                                               "5807",
	"org.hibernate.test.annotations.tableperclass.TablePerClassTest.testUnionSubClass":                                                                                               "6583",
	"org.hibernate.test.annotations.xml.hbm.HbmWithIdentityTest.testManyToOneAndInterface":                                                                                           "24062",
	"org.hibernate.test.batch.BatchOptimisticLockingTest.testBatchAndOptimisticLocking":                                                                                              "6583",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromEngineer":                                                                                       "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromPerson":                                                                                         "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testUpdate":                                                                                                   "5807",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyBasicFieldMergeTest.test":                                                                                                      "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyInitializationWithoutInlineDirtyTrackingTest.test":                                                                             "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyLoadingByEnhancerSetterTest.testField":                                                                                         "26725",
	"org.hibernate.test.converter.lob.ConverterAndLobTest.basicTest":                                                                                                                 "26725",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteLeaf":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteNonLeaf":                                                                    "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteRoot":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateLeaf":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateNonLeaf":                                                                    "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateRoot":                                                                       "5807",
	"org.hibernate.test.generated.DefaultGeneratedValueIdentityTest.testGenerationWithIdentityInsert":                                                                                "24062",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCollectionFetchWithDistinctionAndLimit":                                                                                         "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentNullnessChecks":                                                                                                        "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEntityAndOneToOneReturnedByQuery":                                                                                               "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExpressionWithParamInFunction":                                                                                                  "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testMultipleRefsToSameParam":                                                                                                        "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectExpressions":                                                                                                              "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSimpleSelect":                                                                                                                   "pass?",
	"org.hibernate.test.hql.ASTParserLoadingTest.testStandardFunctions":                                                                                                              "31708",
	"org.hibernate.test.hql.ASTParserLoadingTest.testWhere":                                                                                                                          "pass?",
	"org.hibernate.test.hql.BulkManipulationTest.testBooleanHandling":                                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnDiscriminatorSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnJoinedSubclass":                                                                                                         "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnMappedJoin":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteRestrictedOnManyToOne":                                                                                                    "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassAbstractRoot":                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassConcreteSubclass":                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassLeafSubclass":                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteWithMetadataWhereFragments":                                                                                               "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertAcrossMappedJoinFails":                                                                                                    "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertIntoSuperclassPropertiesFails":                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithManyToOne":                                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMismatchedTypes":                                                                                                      "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMultipleNamedParams":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSelectListUsingJoins":                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSubqueriesAndNamedParams":                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testManyToManyBulkDeleteMultiTable":                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsert":                                                                                                                   "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsertWithNamedParam":                                                                                                     "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleNativeSQLInsert":                                                                                                          "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testTempTableGenerationIsolation":                                                                                                   "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnAnimal":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnComponent":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnDiscriminatorSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnMammal":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnManyToOne":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnDiscriminatorSubclass":                                                                                           "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnJoinedSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullUnionSubclass":                                                                                                     "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithSubquery":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithWhereExistsSubquery":                                                                                                  "5807",
	"org.hibernate.test.hql.CriteriaHQLAlignmentTest.testCountReturnValues":                                                                                                          "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesForward":                                                                                        "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesPositioning":                                                                                    "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesReverse":                                                                                        "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesSingleRowResultSet":                                                                             "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClause":                                                                                                                           "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsNonSubqueryWithKey":                                                                                                       "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubquery":                                                                                                                 "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithEqualOperator":                                                                                                "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKey":                                                                                                          "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKeyAndOtherJoinReference":                                                                                     "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseFailsWithFetch":                                                                                                             "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlDeleteOnJoinedSubclass":                                                                                "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlUpdateOnJoinedSubclass":                                                                                "5807",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testAllParams":                                                                                                          "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testDistinctId":                                                                                                         "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testRollingBack":                                                                                                        "6583",
	"org.hibernate.test.id.NonUniqueIdTest.testLoadEntityWithNonUniqueId":                                                                                                            "12123",
	"org.hibernate.test.idgen.enhanced.forcedtable.BasicForcedTableSequenceTest.testNormalBoundary":                                                                                  "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.HiLoForcedTableSequenceTest.testNormalBoundary":                                                                                   "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.PooledForcedTableSequenceTest.testNormalBoundary":                                                                                 "6583",
	"org.hibernate.test.idgen.enhanced.table.BasicTableTest.testNormalBoundary":                                                                                                      "6583",
	"org.hibernate.test.idgen.enhanced.table.HiLoTableTest.testNormalBoundary":                                                                                                       "6583",
	"org.hibernate.test.idgen.enhanced.table.PooledTableTest.testNormalBoundary":                                                                                                     "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsParallelGeneration":                                                              "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsSerialGeneration":                                                                "6583",
	"org.hibernate.test.idgen.identity.hhh10429.IdentityGeneratorExtendsTest.testIdentifierGeneratorExtendsIdentityGenerator":                                                        "24062",
	"org.hibernate.test.idgen.identity.joinedSubClass.JoinedSubclassHierarchyWithIdentityGenerationTest.shouldPersistDebtorAccountWhenParentServiceAgreementPersisted":               "5807",
	"org.hibernate.test.idgen.namescope.IdGeneratorNamesLocalScopeTest.testNoSequenceGenratorNameClash":                                                                              "6583",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                         "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithUpdate": "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                   "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":          "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithCascadeOnPersist.testInsertOrderingAvoidingForeignKeyConstraintViolation":                                                   "6583",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchOrdering":                                                                                   "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchingAmongstSubClasses":                                                                       "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableMultiLevelInheritance.testBatchingAmongstSubClasses":                                                             "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchOrdering":                                                                                 "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchingAmongstSubClasses":                                                                     "5807",
	"org.hibernate.test.interfaceproxy.InterfaceProxyTest.testInterfaceProxies":                                                                                                      "26725",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSubclass":                                                                                                "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSuperclass":                                                                                              "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testGetSuperclass":                                                                                                   "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySubclass":                                                                                                   "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySuperclass":                                                                                                 "5807",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testLockingJoinedSubclass":                                                                                                 "6583",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrBiggerThenBatch":                                                   "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrEqualWithBatch":                                                    "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrLessThenBatch":                                                     "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeEqJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeGtJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeLtJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.jpa.compliance.tck2_2.TableGeneratorVisibilityTest.testGeneratorIsVisible":                                                                                   "6583",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testStrictUnversioned":                                                                         "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testUnstrictUnversioned":                                                                       "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testVersioned":                                                                                 "5807",
	"org.hibernate.test.jpa.lock.JPALockTest.testLockModeTypeRead":                                                                                                                   "6583",
	"org.hibernate.test.jpa.lock.JPALockTest.testLockModeTypeWrite":                                                                                                                  "6583",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutLock":                                                                                                             "6583",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutRefresh":                                                                                                          "6583",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleNonVersionedInstanceFoundOnLock":                                                                                        "31671",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleVersionedInstanceFoundOnLock":                                                                                           "6583",
	"org.hibernate.test.legacy.ABCProxyTest.testSubclassing":                                                                                                                         "6583",
	"org.hibernate.test.legacy.FooBarTest.testQuery":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testVersioning":                                                                                                                            "6583",
	"org.hibernate.test.legacy.FumTest.testCompositeID":                                                                                                                              "6583",
	"org.hibernate.test.legacy.FumTest.testDeleteOwner":                                                                                                                              "6583",
	"org.hibernate.test.legacy.IJ2Test.testUnionSubclass":                                                                                                                            "6583",
	"org.hibernate.test.legacy.IJTest.testFormulaDiscriminator":                                                                                                                      "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCategories":                                                                                                                      "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace":                                                                                                               "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace2":                                                                                                              "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplaceOnUpdate":                                                                                                       "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testExample":                                                                                                                         "32610",
	"org.hibernate.test.legacy.MasterDetailTest.testSelfManyToOne":                                                                                                                   "32610",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTable":                                                                                                                        "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableCollections":                                                                                                             "32610",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableGeneratedId":                                                                                                             "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableManyToOne":                                                                                                               "32610",
	"org.hibernate.test.legacy.ParentChildTest.testComplexCriteria":                                                                                                                  "6583",
	"org.hibernate.test.legacy.ParentChildTest.testLoadAfterNonExists":                                                                                                               "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testLocking":                                                                                                                          "6583",
	"org.hibernate.test.legacy.SQLFunctionsTest.testBlobClob":                                                                                                                        "26725",
	"org.hibernate.test.loadplans.process.inheritance.Test.basicTest":                                                                                                                "5807",
	"org.hibernate.test.lob.BlobLocatorTest.testBoundedBlobLocatorAccess":                                                                                                            "26725",
	"org.hibernate.test.lob.ClobLocatorTest.testBoundedClobLocatorAccess":                                                                                                            "26725",
	"org.hibernate.test.lob.LobStringTest.testHqlQuery":                                                                                                                              "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectClobPropertyInNativeQuery":                                                                                                       "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectStringLobAnnotatedInNativeQuery":                                                                                                 "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingLobPropertyInNativeQuery":                                                                                                         "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingStringLobAnnotatedPropertyInNativeQuery":                                                                                          "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testBoundedLongByteArrayAccess":                                                                                                     "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testSaving":                                                                                                                         "26725",
	"org.hibernate.test.lob.MaterializedClobTest.testBoundedLongStringAccess":                                                                                                        "26725",
	"org.hibernate.test.lob.PostgreSqlLobStringTest.testBadClobDataSavedAsStringworksAfterUpdate":                                                                                    "26725",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteria":                                                                                                                     "6583",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteriaAliasSpecific":                                                                                                        "6583",
	"org.hibernate.test.locking.LockModeTest.testLoading":                                                                                                                            "6583",
	"org.hibernate.test.locking.LockModeTest.testQuery":                                                                                                                              "6583",
	"org.hibernate.test.locking.LockModeTest.testQueryUsingLockOptions":                                                                                                              "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshLockedEntity":                                                                                                                "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitHigherLevelLockMode":                                                                                             "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitLowerLevelLockMode":                                                                                              "6583",
	"org.hibernate.test.locking.PessimisticReadSkipLockedTest.testPostgreSQLSkipLocked":                                                                                              "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testNoWait":                                                                                                          "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testSkipLocked":                                                                                                      "6583",
	"org.hibernate.test.locking.PessimisticWriteSkipLockedTest.testPostgreSQLSkipLocked":                                                                                             "6583",
	"org.hibernate.test.locking.UpgradeSkipLockedTest.testPostgreSQLSkipLocked":                                                                                                      "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testCriteria":                                                                                                            "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testHql":                                                                                                                 "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testNativeSql":                                                                                                           "6583",
	"org.hibernate.test.mixed.MixedTest.testMixedInheritance":                                                                                                                        "26725",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                   "6583",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                      "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdNullValueOnPersist":                                                                                    "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdQuerySupportingNullValues":                                                                             "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testUniqueAssociation":                                                                                              "6583",
	"org.hibernate.test.onetomany.OneToManyTest.testOneToManyLinkTable":                                                                                                              "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.breakReferences":                                                                                        "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities":                                                                                         "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities_entrySet":                                                                                "5807",
	"org.hibernate.test.ops.genericApi.BasicGetLoadAccessTest.testIt":                                                                                                                "6583",
	"org.hibernate.test.ops.genericApi.ProxiedGetLoadAccessTest.testIt":                                                                                                              "6583",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAll":                                                                                                            "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAllDelete":                                                                                                      "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirty":                                                                                                          "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirtyDelete":                                                                                                    "Expected Failure",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBC":                                                                                                "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBCByName":                                                                                          "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernate":                                                                           "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernateWithoutEnablePassingNulls":                                                  "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureOutParameter":                                                                                     "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureRefCursor":                                                                                        "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testSysRefCursorAsOutParameter":                                                                                      "17511",
	"org.hibernate.test.proxy.ProxyTest.testLockUninitializedProxy":                                                                                                                  "6583",
	"org.hibernate.test.proxy.ProxyTest.testRefreshLockInitializedProxy":                                                                                                             "6583",
	"org.hibernate.test.quote.QuoteTest.testUnionSubclassEntityQuoting":                                                                                                              "5807",
	"org.hibernate.test.quote.TableGeneratorQuotingTest.testTableGeneratorQuoting":                                                                                                   "16769",
	"org.hibernate.test.schemaupdate.PostgreSQLMultipleSchemaSequenceTest.test":                                                                                                      "26443",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[0]":                                                                                              "24062",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[1]":                                                                                              "24062",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithFunctionIndexTest.testUpdateSchema":                                                                                             "9682",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithViewsTest.testUpdateSchema":                                                                                                     "24897",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testImprovedSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":      "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":              "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaUpdateDoesNotFailResolvingCrossSchemaForeignKey":                          "26443",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[0]":                                                                                                "16769",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[1]":                                                                                                "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[0]":                                                                                                    "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[1]":                                                                                                    "16769",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingGroupedSchemaValidator":                                                                                  "10028",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingIndividuallySchemaValidator":                                                                             "10028",
	"org.hibernate.test.schemavalidation.matchingtablenames.TableNamesWithUnderscoreTest.testSchemaValidationDoesNotFailDueToAMoreThanOneTableFound":                                 "16769",
	"org.hibernate.test.subclassfilter.JoinedSubclassFilterTest.testFiltersWithJoinedSubclass":                                                                                       "5807",
	"org.hibernate.test.subselect.SubselectTest.testCustomColumnReadAndWrite":                                                                                                        "31673",
	"org.hibernate.test.subselect.SubselectTest.testEntitySubselect":                                                                                                                 "31673",
	"org.hibernate.test.timestamp.JdbcTimestampUTCTimeZoneTest.testTimeZone":                                                                                                         "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithDefaultUTCTimeZoneTest.testTimeZone":                                                                                              "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithoutUTCTimeZoneTest.testTimeZone":                                                                                                  "3781",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedDirtyQueries":                                                                                                                 "Expected Failure",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedQueries":                                                                                                                      "pass?",
	"org.hibernate.test.tool.schema.SchemaToolTransactionHandlingTest.testValidateInExistingJtaTransaction":                                                                          "16769",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testBulkOperationsWithDifferentConnections":                                                                                  "5807",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                            "12123",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                               "12123",
	"org.hibernate.test.where.annotations.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                             "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromNonUniqueAssociationTable":                                                            "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromUniqueAssociationTable":                                                               "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                    "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                                       "12123",
	"org.hibernate.test.where.hbm.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                     "unknown",
}

var hibernateBlackList2_1 = blacklist{
	"org.hibernate.id.QuotedIdentifierTest.testDirectIdPropertyAccess":                                                                                                               "24062",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testDateTimeFunctions":                                                                                                         "31708",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigDecimals":                                                                                               "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigIntegers":                                                                                               "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfDoubles":                                                                                                   "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfFloats":                                                                                                    "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfIntegers":                                                                                                  "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfLongs":                                                                                                     "5807",
	"org.hibernate.jpa.test.criteria.basic.CastTest.testCastToString":                                                                                                                "5807",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testQuotientConversion":                                                                                                     "26732",
	"org.hibernate.jpa.test.emops.RemoveTest.testUpdatedAndRemove":                                                                                                                   "6583",
	"org.hibernate.jpa.test.exception.ExceptionTest.testConstraintViolationException":                                                                                                "26738",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                           "6583",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                          "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                            "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                           "6583",
	"org.hibernate.jpa.test.lob.BlobTest.testBlobSerialization":                                                                                                                      "26725",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticLock":                                                                                                              "6583",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticWriteLockNoWait":                                                                                                   "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithPessimisticWriteLockTimeoutException":                                                                                          "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithTimeoutHint":                                                                                                                   "6583",
	"org.hibernate.jpa.test.lock.LockTest.testLockWriteOnUnversioned":                                                                                                                "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockSkipLocked":                                                                                               "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockWithoutNoWait":                                                                                            "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testEntityLockModeStateAfterQueryLocking":                                                                                          "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementOverall":                                                                                             "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementSpecific":                                                                                            "6583",
	"org.hibernate.jpa.test.lock.StatementIsClosedAfterALockExceptionTest.testStatementIsClosed":                                                                                     "6583",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuery":                                                                                 "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuerySpecifyingType":                                                                   "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testMultipleConstructorResultNativeQuery":                                                                         "unknown",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testAlwaysTransactionalOperations":                                                                                   "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testMergeWhenExtended":                                                                                               "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackClearPC":                                                                                                 "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackExceptionOnOptimisticLockException":                                                                      "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testSetRollbackOnlyAndFlush":                                                                                         "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionAndContains":                                                                                          "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionCommitDoesNotFlush":                                                                                   "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionalOperationsWhenExtended":                                                                             "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.IdMapManyToOneSpecjTest.testComplexIdClass":                                                                    "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.lazy.CompositeKeyDeleteTest.testRemove":                                                                        "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b2.IdClassGeneratedValueManyToOneTest.testComplexIdClass":                                                                   "6583",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testSerialized":                                                                                             "26725",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testVersioning":                                                                                             "27274",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testClub":                                                                                              "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testIqFilter":                                                                                          "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testNonHumanFilter":                                                                                    "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testPregnantFilter":                                                                                    "5807",
	"org.hibernate.test.annotations.id.IdTest.testLowAllocationSize":                                                                                                                 "6583",
	"org.hibernate.test.annotations.id.IdTest.testTableGenerator":                                                                                                                    "6583",
	"org.hibernate.test.annotations.id.sequences.IdTest.testLowAllocationSize":                                                                                                       "6583",
	"org.hibernate.test.annotations.id.sequences.IdTest.testTableGenerator":                                                                                                          "6583",
	"org.hibernate.test.annotations.identifiercollection.IdentifierCollectionTest.testIdBag":                                                                                         "6583",
	"org.hibernate.test.annotations.lob.LobTest.testBlob":                                                                                                                            "26725",
	"org.hibernate.test.annotations.lob.LobTest.testClob":                                                                                                                            "26725",
	"org.hibernate.test.annotations.lob.LobTest.testSerializableToBlob":                                                                                                              "26725",
	"org.hibernate.test.annotations.lob.MaterializedBlobTest.testSaving":                                                                                                             "26725",
	"org.hibernate.test.annotations.lob.SerializableToBlobTypeTest.testPersist":                                                                                                      "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testBlob":                                                                                                                   "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testClob":                                                                                                                   "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testSerializableToBlob":                                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedByteArray":                                                                                              "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedCharArray":                                                                                              "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveByteArray":                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveCharArray":                                                                                     "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedString":                                                                                                 "26725",
	"org.hibernate.test.annotations.manytoone.referencedcolumnname.ManyToOneReferencedColumnNameTest.testReoverableExceptionInFkOrdering":                                            "24062",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithAliasOneToOneJoin":                                                                       "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinCollection":                                                                 "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinOnetoOne":                                                                   "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testNaturalKeyLookupWithConstraint":                                                                      "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSimpleImmutableNaturalKeyLookup":                                                                     "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSubCriteriaOneToOneJoin":                                                                             "6583",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testCascadeDeleteWithUnidirectionalAssociation":                                                                          "26738",
	"org.hibernate.test.annotations.onetoone.hhh9798.OneToOneJoinTableTest.storeNonUniqueRelationship":                                                                               "5807",
	"org.hibernate.test.annotations.tableperclass.TablePerClassTest.testUnionSubClass":                                                                                               "6583",
	"org.hibernate.test.annotations.xml.hbm.HbmWithIdentityTest.testManyToOneAndInterface":                                                                                           "24062",
	"org.hibernate.test.batch.BatchOptimisticLockingTest.testBatchAndOptimisticLocking":                                                                                              "6583",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromEngineer":                                                                                       "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromPerson":                                                                                         "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testUpdate":                                                                                                   "5807",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyBasicFieldMergeTest.test":                                                                                                      "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyInitializationWithoutInlineDirtyTrackingTest.test":                                                                             "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyLoadingByEnhancerSetterTest.testField":                                                                                         "26725",
	"org.hibernate.test.converter.lob.ConverterAndLobTest.basicTest":                                                                                                                 "26725",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteLeaf":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteNonLeaf":                                                                    "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteRoot":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateLeaf":                                                                       "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateNonLeaf":                                                                    "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateRoot":                                                                       "5807",
	"org.hibernate.test.generated.DefaultGeneratedValueIdentityTest.testGenerationWithIdentityInsert":                                                                                "24062",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCollectionFetchWithDistinctionAndLimit":                                                                                         "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentNullnessChecks":                                                                                                        "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEntityAndOneToOneReturnedByQuery":                                                                                               "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExpressionWithParamInFunction":                                                                                                  "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAQLMapValueQualifier":                                                                                                         "3288",
	"org.hibernate.test.hql.ASTParserLoadingTest.testMultipleRefsToSameParam":                                                                                                        "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectExpressions":                                                                                                              "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSimpleSelect":                                                                                                                   "pass?",
	"org.hibernate.test.hql.ASTParserLoadingTest.testStandardFunctions":                                                                                                              "31708",
	"org.hibernate.test.hql.ASTParserLoadingTest.testWhere":                                                                                                                          "pass?",
	"org.hibernate.test.hql.BulkManipulationTest.testBooleanHandling":                                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnDiscriminatorSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnJoinedSubclass":                                                                                                         "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnMappedJoin":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteRestrictedOnManyToOne":                                                                                                    "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassAbstractRoot":                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassConcreteSubclass":                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassLeafSubclass":                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteWithMetadataWhereFragments":                                                                                               "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteWithSubquery":                                                                                                             "31670",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertAcrossMappedJoinFails":                                                                                                    "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertIntoSuperclassPropertiesFails":                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithManyToOne":                                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMismatchedTypes":                                                                                                      "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMultipleNamedParams":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSelectListUsingJoins":                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSubqueriesAndNamedParams":                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testManyToManyBulkDeleteMultiTable":                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsert":                                                                                                                   "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsertWithNamedParam":                                                                                                     "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleNativeSQLInsert":                                                                                                          "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testTempTableGenerationIsolation":                                                                                                   "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnAnimal":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnComponent":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnDiscriminatorSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnMammal":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnManyToOne":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnDiscriminatorSubclass":                                                                                           "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnJoinedSubclass":                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullUnionSubclass":                                                                                                     "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithSubquery":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithWhereExistsSubquery":                                                                                                  "5807",
	"org.hibernate.test.hql.CriteriaHQLAlignmentTest.testCountReturnValues":                                                                                                          "5807",
	"org.hibernate.test.hql.DeleteQuerySubqueryReferencingTargetPropertyTest.testSubQueryReferencingTargetProperty":                                                                  "31670",
	"org.hibernate.test.hql.DeleteWithSubqueryTest.testDeleteMemberOf":                                                                                                               "31670",
	"org.hibernate.test.hql.InsertWithSubSelectTest.testInsert":                                                                                                                      "31721",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesForward":                                                                                        "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesPositioning":                                                                                    "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesReverse":                                                                                        "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesSingleRowResultSet":                                                                             "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClause":                                                                                                                           "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsNonSubqueryWithKey":                                                                                                       "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubquery":                                                                                                                 "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithEqualOperator":                                                                                                "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKey":                                                                                                          "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKeyAndOtherJoinReference":                                                                                     "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseFailsWithFetch":                                                                                                             "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlDeleteOnJoinedSubclass":                                                                                "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlUpdateOnJoinedSubclass":                                                                                "5807",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testAllParams":                                                                                                          "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testDistinctId":                                                                                                         "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testRollingBack":                                                                                                        "6583",
	"org.hibernate.test.id.NonUniqueIdTest.testLoadEntityWithNonUniqueId":                                                                                                            "12123",
	"org.hibernate.test.idgen.enhanced.forcedtable.BasicForcedTableSequenceTest.testNormalBoundary":                                                                                  "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.HiLoForcedTableSequenceTest.testNormalBoundary":                                                                                   "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.PooledForcedTableSequenceTest.testNormalBoundary":                                                                                 "6583",
	"org.hibernate.test.idgen.enhanced.table.BasicTableTest.testNormalBoundary":                                                                                                      "6583",
	"org.hibernate.test.idgen.enhanced.table.HiLoTableTest.testNormalBoundary":                                                                                                       "6583",
	"org.hibernate.test.idgen.enhanced.table.PooledTableTest.testNormalBoundary":                                                                                                     "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsParallelGeneration":                                                              "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsSerialGeneration":                                                                "6583",
	"org.hibernate.test.idgen.identity.hhh10429.IdentityGeneratorExtendsTest.testIdentifierGeneratorExtendsIdentityGenerator":                                                        "24062",
	"org.hibernate.test.idgen.identity.joinedSubClass.JoinedSubclassHierarchyWithIdentityGenerationTest.shouldPersistDebtorAccountWhenParentServiceAgreementPersisted":               "5807",
	"org.hibernate.test.idgen.namescope.IdGeneratorNamesLocalScopeTest.testNoSequenceGenratorNameClash":                                                                              "6583",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                         "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithUpdate": "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                   "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":          "5807",
	"org.hibernate.test.inheritance.discriminator.SingleTableInheritancePersistTest.testPolymorphicAssociation":                                                                      "31589",
	"org.hibernate.test.inheritance.discriminator.TablePerClassInheritancePersistTest.testPolymorphicAssociation":                                                                    "pass?",
	"org.hibernate.test.insertordering.InsertOrderingWithCascadeOnPersist.testInsertOrderingAvoidingForeignKeyConstraintViolation":                                                   "6583",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchOrdering":                                                                                   "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchingAmongstSubClasses":                                                                       "26738",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableMultiLevelInheritance.testBatchingAmongstSubClasses":                                                             "26738",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchOrdering":                                                                                 "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchingAmongstSubClasses":                                                                     "5807",
	"org.hibernate.test.interfaceproxy.InterfaceProxyTest.testInterfaceProxies":                                                                                                      "26725",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSubclass":                                                                                                "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSuperclass":                                                                                              "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testGetSuperclass":                                                                                                   "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySubclass":                                                                                                   "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySuperclass":                                                                                                 "5807",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testLockingJoinedSubclass":                                                                                                 "6583",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrBiggerThenBatch":                                                   "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrEqualWithBatch":                                                    "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrLessThenBatch":                                                     "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeEqJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeGtJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeLtJdbcBatchSize":                                                             "24062",
	"org.hibernate.test.jpa.compliance.tck2_2.TableGeneratorVisibilityTest.testGeneratorIsVisible":                                                                                   "6583",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testStrictUnversioned":                                                                         "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testUnstrictUnversioned":                                                                       "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testVersioned":                                                                                 "5807",
	"org.hibernate.test.jpa.lock.JPALockTest.testLockModeTypeRead":                                                                                                                   "6583",
	"org.hibernate.test.jpa.lock.JPALockTest.testLockModeTypeWrite":                                                                                                                  "6583",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutLock":                                                                                                             "6583",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutRefresh":                                                                                                          "6583",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleNonVersionedInstanceFoundOnLock":                                                                                        "31671",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleVersionedInstanceFoundOnLock":                                                                                           "6583",
	"org.hibernate.test.legacy.ABCProxyTest.testSubclassing":                                                                                                                         "6583",
	"org.hibernate.test.legacy.FooBarTest.testOnCascadeDelete":                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testQuery":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testVersioning":                                                                                                                            "6583",
	"org.hibernate.test.legacy.FumTest.testCompositeID":                                                                                                                              "6583",
	"org.hibernate.test.legacy.FumTest.testDeleteOwner":                                                                                                                              "6583",
	"org.hibernate.test.legacy.IJ2Test.testUnionSubclass":                                                                                                                            "6583",
	"org.hibernate.test.legacy.IJTest.testFormulaDiscriminator":                                                                                                                      "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCategories":                                                                                                                      "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace":                                                                                                               "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace2":                                                                                                              "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplaceOnUpdate":                                                                                                       "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTable":                                                                                                                        "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableGeneratedId":                                                                                                             "6583",
	"org.hibernate.test.legacy.ParentChildTest.testComplexCriteria":                                                                                                                  "6583",
	"org.hibernate.test.legacy.ParentChildTest.testLoadAfterNonExists":                                                                                                               "26738",
	"org.hibernate.test.legacy.ParentChildTest.testLocking":                                                                                                                          "6583",
	"org.hibernate.test.legacy.SQLFunctionsTest.testBlobClob":                                                                                                                        "26725",
	"org.hibernate.test.loadplans.process.inheritance.Test.basicTest":                                                                                                                "5807",
	"org.hibernate.test.lob.BlobLocatorTest.testBoundedBlobLocatorAccess":                                                                                                            "26725",
	"org.hibernate.test.lob.ClobLocatorTest.testBoundedClobLocatorAccess":                                                                                                            "26725",
	"org.hibernate.test.lob.LobStringTest.testHqlQuery":                                                                                                                              "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectClobPropertyInNativeQuery":                                                                                                       "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectStringLobAnnotatedInNativeQuery":                                                                                                 "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingLobPropertyInNativeQuery":                                                                                                         "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingStringLobAnnotatedPropertyInNativeQuery":                                                                                          "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testBoundedLongByteArrayAccess":                                                                                                     "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testSaving":                                                                                                                         "26725",
	"org.hibernate.test.lob.MaterializedClobTest.testBoundedLongStringAccess":                                                                                                        "26725",
	"org.hibernate.test.lob.PostgreSqlLobStringTest.testBadClobDataSavedAsStringworksAfterUpdate":                                                                                    "26725",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteria":                                                                                                                     "6583",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteriaAliasSpecific":                                                                                                        "6583",
	"org.hibernate.test.locking.LockModeTest.testLoading":                                                                                                                            "6583",
	"org.hibernate.test.locking.LockModeTest.testQuery":                                                                                                                              "6583",
	"org.hibernate.test.locking.LockModeTest.testQueryUsingLockOptions":                                                                                                              "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshLockedEntity":                                                                                                                "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitHigherLevelLockMode":                                                                                             "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitLowerLevelLockMode":                                                                                              "6583",
	"org.hibernate.test.locking.PessimisticReadSkipLockedTest.testPostgreSQLSkipLocked":                                                                                              "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testNoWait":                                                                                                          "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testSkipLocked":                                                                                                      "6583",
	"org.hibernate.test.locking.PessimisticWriteSkipLockedTest.testPostgreSQLSkipLocked":                                                                                             "6583",
	"org.hibernate.test.locking.UpgradeSkipLockedTest.testPostgreSQLSkipLocked":                                                                                                      "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testCriteria":                                                                                                            "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testHql":                                                                                                                 "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testNativeSql":                                                                                                           "6583",
	"org.hibernate.test.manytomany.ManyToManyBidirectionalTest.testRemoveMappedBySide":                                                                                               "26738",
	"org.hibernate.test.mixed.MixedTest.testMixedInheritance":                                                                                                                        "26725",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                   "6583",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                      "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdNullValueOnPersist":                                                                                    "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdQuerySupportingNullValues":                                                                             "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testUniqueAssociation":                                                                                              "6583",
	"org.hibernate.test.ondelete.OnDeleteTest.testJoinedSubclass":                                                                                                                    "26738",
	"org.hibernate.test.onetomany.OneToManyTest.testOneToManyLinkTable":                                                                                                              "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.breakReferences":                                                                                        "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities":                                                                                         "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities_entrySet":                                                                                "5807",
	"org.hibernate.test.ops.genericApi.BasicGetLoadAccessTest.testIt":                                                                                                                "6583",
	"org.hibernate.test.ops.genericApi.ProxiedGetLoadAccessTest.testIt":                                                                                                              "6583",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAll":                                                                                                            "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAllDelete":                                                                                                      "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirty":                                                                                                          "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirtyDelete":                                                                                                    "Expected Failure",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBC":                                                                                                "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBCByName":                                                                                          "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernate":                                                                           "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernateWithoutEnablePassingNulls":                                                  "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureOutParameter":                                                                                     "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureRefCursor":                                                                                        "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testSysRefCursorAsOutParameter":                                                                                      "17511",
	"org.hibernate.test.proxy.ProxyTest.testLockUninitializedProxy":                                                                                                                  "6583",
	"org.hibernate.test.proxy.ProxyTest.testRefreshLockInitializedProxy":                                                                                                             "6583",
	"org.hibernate.test.quote.QuoteTest.testUnionSubclassEntityQuoting":                                                                                                              "5807",
	"org.hibernate.test.quote.TableGeneratorQuotingTest.testTableGeneratorQuoting":                                                                                                   "16769",
	"org.hibernate.test.schemaupdate.PostgreSQLMultipleSchemaSequenceTest.test":                                                                                                      "26443",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[0]":                                                                                              "26738",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[1]":                                                                                              "26738",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithFunctionIndexTest.testUpdateSchema":                                                                                             "9682",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithViewsTest.testUpdateSchema":                                                                                                     "24897",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testImprovedSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":      "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":              "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaUpdateDoesNotFailResolvingCrossSchemaForeignKey":                          "26443",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[0]":                                                                                                "16769",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[1]":                                                                                                "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[0]":                                                                                                    "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[1]":                                                                                                    "16769",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingGroupedSchemaValidator":                                                                                  "10028",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingIndividuallySchemaValidator":                                                                             "10028",
	"org.hibernate.test.schemavalidation.matchingtablenames.TableNamesWithUnderscoreTest.testSchemaValidationDoesNotFailDueToAMoreThanOneTableFound":                                 "16769",
	"org.hibernate.test.subclassfilter.JoinedSubclassFilterTest.testFiltersWithJoinedSubclass":                                                                                       "5807",
	"org.hibernate.test.subselect.SubselectTest.testCustomColumnReadAndWrite":                                                                                                        "31673",
	"org.hibernate.test.subselect.SubselectTest.testEntitySubselect":                                                                                                                 "31673",
	"org.hibernate.test.timestamp.JdbcTimestampUTCTimeZoneTest.testTimeZone":                                                                                                         "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithDefaultUTCTimeZoneTest.testTimeZone":                                                                                              "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithoutUTCTimeZoneTest.testTimeZone":                                                                                                  "3781",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedDirtyQueries":                                                                                                                 "Expected Failure",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedQueries":                                                                                                                      "pass?",
	"org.hibernate.test.tool.schema.SchemaToolTransactionHandlingTest.testValidateInExistingJtaTransaction":                                                                          "16769",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testBulkOperationsWithDifferentConnections":                                                                                  "5807",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                            "12123",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                               "12123",
	"org.hibernate.test.where.annotations.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                             "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromNonUniqueAssociationTable":                                                            "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromUniqueAssociationTable":                                                               "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                    "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                                       "12123",
	"org.hibernate.test.where.hbm.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                     "unknown",
}

var hibernateBlackList2_0 = blacklist{
	"org.hibernate.engine.spi.ExtraStateTest.shouldMaintainExtraStateWhenUsingIdentityIdGenerationStrategy":                                                                                      "unknown",
	"org.hibernate.event.EmbeddableCallbackTest.test":                                                                                                                                            "unknown",
	"org.hibernate.id.CreateDeleteTest.createAndDeleteAnEntityInTheSameTransactionTest":                                                                                                          "unknown",
	"org.hibernate.id.FlushIdGenTest.testPersistBeforeTransaction":                                                                                                                               "unknown",
	"org.hibernate.id.IdClassManyToOneCascadeTest.testPersistCascadesToManyToOne":                                                                                                                "unknown",
	"org.hibernate.id.QuotedIdentifierTest.testDirectIdPropertyAccess":                                                                                                                           "24062",
	"org.hibernate.id.SequenceHiLoGeneratorNoIncrementTest.testHiLoAlgorithm":                                                                                                                    "unknown",
	"org.hibernate.id.SequenceHiLoGeneratorTest.testHiLoAlgorithm":                                                                                                                               "unknown",
	"org.hibernate.id.uuid.GeneratedValueTest.testGeneratedUuidId":                                                                                                                               "unknown",
	"org.hibernate.jpa.test.CompositeIdRowValueTest.testTupleAfterSubQuery":                                                                                                                      "unknown",
	"org.hibernate.jpa.test.CountEntityWithCompositeIdTest.shouldCount":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testClear":                                                                                                                                         "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testConfiguration":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testContains":                                                                                                                                      "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testEntityManager":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testEntityNotFoundException":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testGet":                                                                                                                                           "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testPersistNoneGenerator":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.EntityManagerTest.testSetProperty":                                                                                                                                   "unknown",
	"org.hibernate.jpa.test.association.AssociationTest.testBidirOneToOne":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.association.AssociationTest.testMergeAndBidirOneToOne":                                                                                                               "unknown",
	"org.hibernate.jpa.test.beanvalidation.BeanValidationTest.testBeanValidationIntegrationOnCommit":                                                                                             "unknown",
	"org.hibernate.jpa.test.beanvalidation.BeanValidationTest.testBeanValidationIntegrationOnFlush":                                                                                              "unknown",
	"org.hibernate.jpa.test.cacheable.api.JpaCacheApiUsageTest.testEviction":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.callbacks.CallbackAndDirtyTest.testDirtyButNotDirty":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testCallBackListenersHierarchy":                                                                                                              "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testCallbackMethod":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testEntityListener":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testIdNullSetByPrePersist":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testPostPersist":                                                                                                                             "unknown",
	"org.hibernate.jpa.test.callbacks.CallbacksTest.testPrePersistOnCascade":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.callbacks.PreUpdateBytecodeEnhancementTest.testPreUpdateModifications":                                                                                               "unknown",
	"org.hibernate.jpa.test.callbacks.PreUpdateCustomEntityDirtinessStrategyTest.testPreUpdateModifications":                                                                                     "unknown",
	"org.hibernate.jpa.test.callbacks.PreUpdateDirtyCheckingInterceptorTest.testPreUpdateModifications":                                                                                          "unknown",
	"org.hibernate.jpa.test.callbacks.xml.EntityListenerViaXmlTest.testUsage":                                                                                                                    "unknown",
	"org.hibernate.jpa.test.cascade.CascadeTest.testCascade":                                                                                                                                     "unknown",
	"org.hibernate.jpa.test.cascade.CascadeTest.testNoCascadeAndMerge":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.cascade.DeleteOrphanTest.testDeleteOrphan":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.cascade.DetachAndContainsTest.testDetach":                                                                                                                            "unknown",
	"org.hibernate.jpa.test.cascade.FetchTest.testCascadeAndFetchCollection":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.cascade.FetchTest.testCascadeAndFetchEntity":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.cascade.FetchTest.testTwoLevelDeepPersist":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.cascade.FetchTest.testTwoLevelDeepPersistOnManyToOne":                                                                                                                "unknown",
	"org.hibernate.jpa.test.cascade.FetchTest2.testProxyTransientStuff":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.cascade.MergeTest.testMergeDetachedEntityWithNewOneToManyElements":                                                                                                   "unknown",
	"org.hibernate.jpa.test.cascade.MergeTest.testMergeLoadedEntityWithNewOneToManyElements":                                                                                                     "unknown",
	"org.hibernate.jpa.test.cascade.MergeWithTransientNonCascadedAssociationTest.testMergeWithTransientNonCascadedAssociation":                                                                   "unknown",
	"org.hibernate.jpa.test.cascade.multicircle.MultiCircleJpaCascadeTest.testMerge":                                                                                                             "unknown",
	"org.hibernate.jpa.test.cascade.multicircle.MultiCircleJpaCascadeTest.testPersist":                                                                                                           "unknown",
	"org.hibernate.jpa.test.cascade.multicircle.MultiCircleJpaCascadeTest.testPersistNoCascadeToTransient":                                                                                       "unknown",
	"org.hibernate.jpa.test.cascade.multicircle.MultiCircleJpaCascadeTest.testPersistThenUpdateNoCascadeToTransient":                                                                             "unknown",
	"org.hibernate.jpa.test.cascade.multilevel.MultiLevelCascadeCollectionEmbeddableTest.testHibernateDeleteEntityInitializeCollections":                                                         "unknown",
	"org.hibernate.jpa.test.cascade.multilevel.MultiLevelCascadeCollectionIdClassTest.testHibernateDeleteEntityInitializeCollections":                                                            "unknown",
	"org.hibernate.jpa.test.cascade.multilevel.MultiLevelCascadeRegularIdBasedParentChildAssociationTest.testHibernateDeleteEntityWithoutInitializingCollections":                                "unknown",
	"org.hibernate.jpa.test.cascade.multilevel.MultiLevelCascadeTest.test":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.cdi.BasicCdiTest.testIt":                                                                                                                                             "unknown",
	"org.hibernate.jpa.test.collection.PostLoadTest.testAccessAssociatedSetInPostLoad":                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testDeprecation":                                                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testJustSimpleRootCriteria":                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testSerialization":                                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testSimpleFetchCriteria":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testSimpleJoinCriteria":                                                                                                               "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testTrim":                                                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaCompilingTest.testTrimAChar":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaQueryTypeQueryAdapterTest.testSetParameterPassingTypeNotFails":                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.CriteriaQueryTypeQueryAdapterTest.testSetParameterTypeInstantNotFails":                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.ElementCollectionConverterTest.testCriteriaQueryWithElementCollectionUsingConverter":                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.EntitySuperclassCollectionTest.testPerson":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.ManipulationCriteriaTest.basicMultipleAssignments":                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.ManipulationCriteriaTest.basicTest":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.ManipulationCriteriaTest.testDeleteWithUnCorrelatedSubquery":                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.OnKeywordTest.basicTest":                                                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.ParameterTest.testNonPrimitiveArrayParameterBinding":                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.ParameterTest.testParameterInParameterList":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.ParameterTest.testParameterInParameterList2":                                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.ParameterTest.testPrimitiveArrayParameterBinding":                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testConstructor":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testDateTimeFunctions":                                                                                                                     "31708",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testEqualityComparisonEntityConversion":                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testEqualityComparisonLiteralConversion":                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testFunctionDialectFunctions":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testJoinUsingNegatedPredicate":                                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.QueryBuilderTest.testTypeConversion":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.SuperclassCollectionTest.testOtherPerson":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.SuperclassCollectionTest.testOtherSubclass":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.SuperclassCollectionTest.testPerson":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testJoinMethodOnATreatedJoin":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testJoinOnTreatedJoin":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testJoinOnTreatedRoot":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testJoinOnTreatedRootWithJoin":                                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testTreatJoin":                                                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testTreatJoin2":                                                                                                                               "unknown",
	"org.hibernate.jpa.test.criteria.TreatJoinTest.testTreatedJoinInWhereClause":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.basicTest":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.basicTest2":                                                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.testSelectSubclassPropertyFromDowncast":                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.testTreatWithRestrictionOnAbstractClass":                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.treatPathClassTest":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.treatPathClassTestHqlControl":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.treatRoot":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.TreatKeywordTest.treatRootReturnSuperclass":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.TreatListJoinTest.testTreatJoin":                                                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigDecimals":                                                                                                           "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfBigIntegers":                                                                                                           "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfDoubles":                                                                                                               "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfFloats":                                                                                                                "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfIntegers":                                                                                                              "5807",
	"org.hibernate.jpa.test.criteria.basic.AggregationResultTest.testSumOfLongs":                                                                                                                 "5807",
	"org.hibernate.jpa.test.criteria.basic.BasicCriteriaUsageTest.testDateCompositeCustomType":                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.basic.BasicCriteriaUsageTest.testDoubleNegation":                                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.basic.BasicCriteriaUsageTest.testFunctionCriteria":                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.BasicCriteriaUsageTest.testTrivialCompilation":                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.basic.CastTest.testCastToString":                                                                                                                            "5807",
	"org.hibernate.jpa.test.criteria.basic.ConcatTest.testConcat":                                                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.basic.ConcatTest.testSelectCaseWithConcat":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testDiff":                                                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testDiffWithQuotient":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyConjunction":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyConjunctionIsFalse":                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyConjunctionIsTrue":                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyDisjunction":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyDisjunctionIsFalse":                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testEmptyDisjunctionIsTrue":                                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testInExplicitTupleList":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testInExplicitTupleListVarargs":                                                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testInExpressionVarargs":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testJoinedElementCollectionValuesInTupleList":                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testParameterReuse":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testQuotientAndMultiply":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.basic.ExpressionsTest.testSumWithQuotient":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.basic.ListIndexTest.testListIndex":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testByteArray":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testCharArray":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testComplicatedNotAnd":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testComplicatedNotOr":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testEmptyConjunction":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testEmptyDisjunction":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testExplicitBuilderBooleanHandling":                                                                                                     "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testNotMultipleOr":                                                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testQuotientConversion":                                                                                                                 "26732",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testSimpleNot":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.basic.PredicateTest.testSimpleNot2":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentCriteriaTest.testEmbeddableInPath":                                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentCriteriaTest.testOneToManyJoinFetchedInEmbeddable":                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentCriteriaTest.testParameterizedFunctions":                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testEqualExpressionForThePropertyOfTheElementCollectionPropertyOfAComponent":                                          "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testEqualityForThePropertyOfAComponent":                                                                               "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testInExpressionForAComponent":                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testInExpressionForTheManyToOnePropertyOfAComponent":                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testSizeExpressionForTheElementCollectionPropertyOfAComponent":                                                        "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testSizeExpressionForTheElementCollectionPropertyOfASubComponent":                                                     "unknown",
	"org.hibernate.jpa.test.criteria.components.ComponentInWhereClauseTest.testSizeExpressionForTheOneToManyPropertyOfAComponent":                                                                "unknown",
	"org.hibernate.jpa.test.criteria.components.EntitySuperclassComponentWithCollectionTest.testSizeExpressionForTheElementCollectionPropertyOfAComponent":                                       "unknown",
	"org.hibernate.jpa.test.criteria.components.EntitySuperclassComponentWithCollectionTest.testSizeExpressionForTheElementCollectionPropertyOfASubComponent":                                    "unknown",
	"org.hibernate.jpa.test.criteria.components.EntitySuperclassComponentWithCollectionTest.testSizeExpressionForTheOneToManyPropertyOfAComponent":                                               "unknown",
	"org.hibernate.jpa.test.criteria.components.MappedSuperclassComponentWithCollectionTest.testSizeExpressionForTheElementCollectionPropertyOfAComponent":                                       "unknown",
	"org.hibernate.jpa.test.criteria.components.MappedSuperclassComponentWithCollectionTest.testSizeExpressionForTheElementCollectionPropertyOfASubComponent":                                    "unknown",
	"org.hibernate.jpa.test.criteria.components.MappedSuperclassComponentWithCollectionTest.testSizeExpressionForTheOneToManyPropertyOfAComponent":                                               "unknown",
	"org.hibernate.jpa.test.criteria.components.joins.ComponentJoinTest.getResultWithMetamodelDerivedPath":                                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.components.joins.ComponentJoinTest.getResultWithStringPropertyDerivedPath":                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.enumcollection.EnumIsMemberTest.testQueryEnumCollection":                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.fetchscroll.CriteriaToScrollableResultsFetchTest.testNoScroll":                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.fetchscroll.CriteriaToScrollableResultsFetchTest.testWithScroll":                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.idclass.IdClassPredicateTest.testCountIdClassAttributes":                                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.idclass.IdClassPredicateTest.testDeclaredIdClassAttributes":                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.idclass.IdClassPredicateTest.testSupertypeIdClassAttributes":                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.limitExpression.LimitExpressionTest.testAnEmptyListIsReturnedWhenSetMaxResultsToZero":                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.mapjoin.MapJoinTest.allEntities":                                                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.mapjoin.MapJoinTestWithEmbeddable.testSelectingKeyOfMapJoin":                                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.mapjoin.MapJoinTestWithEmbeddable.testSelectingValueOfMapJoin":                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.CriteriaLiteralInSelectExpressionTest.testBooleanLiteral":                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.CriteriaLiteralInSelectExpressionTest.testNullLiteral":                                                                                           "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.CriteriaLiteralInSelectExpressionTest.testNullLiteralFirst":                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.CriteriaLiteralInSelectExpressionTest.testStringLiteral":                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.CriteriaLiteralInSelectExpressionTest.testStringLiteral2":                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.nulliteral.NullLiteralExpression.testNullLiteralExpressionInCriteriaUpdate":                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.paths.AbstractPathImplTest.testGetNonExistingAttributeViaName":                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.paths.AbstractPathImplTest.testIllegalDereference":                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.paths.AbstractPathImplTest.testTypeExpression":                                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.paths.FetchAndJoinTest.testImplicitJoinFromExplicitCollectionJoin":                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.paths.ImplicitJoinTest.testImplicitJoinFromExplicitCollectionJoin":                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testCollectionIsEmptyCriteria":                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testCollectionIsEmptyHql":                                                                                              "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testCollectionSizeCriteria":                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testCollectionSizeHql":                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testElementMapSizeCriteria":                                                                                            "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testElementMapSizeHql":                                                                                                 "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testEntityMapIsEmptyCriteria":                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testEntityMapIsEmptyHql":                                                                                               "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testEntityMapSizeCriteria":                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.paths.PluralAttributeExpressionsTest.testEntityMapSizeHql":                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.GroupBySelectCaseTest.selectCaseInGroupByAndSelectExpression":                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.GroupBySelectCaseTest.selectCaseInOrderByAndSelectExpression":                                                                                    "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.SelectCaseTest.selectCaseWithCastedTypeValuesShouldWork":                                                                                         "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.SelectCaseTest.selectCaseWithValuesShouldWork":                                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.SelectCaseTest.simpleSelectCaseWithCastedTypeValuesShouldWork":                                                                                   "unknown",
	"org.hibernate.jpa.test.criteria.selectcase.SelectCaseTest.simpleSelectCaseWithValuesShouldWork":                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.simplecase.BasicSimpleCaseTest.testCaseIntegerResult":                                                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.simplecase.BasicSimpleCaseTest.testCaseLiteralResult":                                                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.simplecase.BasicSimpleCaseTest.testCaseLiteralResult2":                                                                                                      "unknown",
	"org.hibernate.jpa.test.criteria.simplecase.BasicSimpleCaseTest.testCaseStringResult":                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.subquery.CorrelatedSubqueryTest.testBasicCorrelation":                                                                                                       "unknown",
	"org.hibernate.jpa.test.criteria.subquery.CorrelatedSubqueryTest.testCorrelationExplicitSelectionCorrelation":                                                                                "unknown",
	"org.hibernate.jpa.test.criteria.subquery.CorrelatedSubqueryTest.testRestrictedCorrelation":                                                                                                  "unknown",
	"org.hibernate.jpa.test.criteria.subquery.CorrelatedSubqueryTest.testRestrictedCorrelationNoExplicitSelection":                                                                               "unknown",
	"org.hibernate.jpa.test.criteria.subquery.UncorrelatedSubqueryTest.testEqualAll":                                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.subquery.UncorrelatedSubqueryTest.testLessThan":                                                                                                             "unknown",
	"org.hibernate.jpa.test.criteria.tuple.TupleCriteriaTest.testArray":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.tuple.TupleCriteriaTest.testInvalidTupleIndexAccess":                                                                                                        "unknown",
	"org.hibernate.jpa.test.criteria.tuple.TupleCriteriaTest.testTuple":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.criteria.tuple.TupleCriteriaTest.testVariousTupleAccessMethods":                                                                                                      "unknown",
	"org.hibernate.jpa.test.ejb3configuration.EntityManagerFactorySerializationTest.testSerialization":                                                                                           "unknown",
	"org.hibernate.jpa.test.ejb3configuration.EntityManagerSerializationTest.testSerialization":                                                                                                  "unknown",
	"org.hibernate.jpa.test.ejb3configuration.InterceptorTest.testOnLoadCallInInterceptor":                                                                                                       "unknown",
	"org.hibernate.jpa.test.emops.FlushModeTest.testCreateEMFlushMode":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.emops.FlushTest.testFlushOnDetached":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesAllowedLoggedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                   "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                       "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesAllowedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                         "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesAllowedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                             "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesCustomTest.testMergeMultipleEntityCopiesAllowed":                                                                                      "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesCustomTest.testMergeMultipleEntityCopiesAllowedAndDisallowed":                                                                         "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesDisallowedByDefaultTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                             "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesDisallowedByDefaultTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                 "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesDisallowedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                      "unknown",
	"org.hibernate.jpa.test.emops.MergeMultipleEntityCopiesDisallowedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                          "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testConcurrentMerge":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testMergeManyToMany":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testMergeManyToManyWithDeference":                                                                                                                    "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testMergeUnidirectionalOneToMany":                                                                                                                    "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testMergeWithIndexColumn":                                                                                                                            "unknown",
	"org.hibernate.jpa.test.emops.MergeTest.testRemoveAndMerge":                                                                                                                                  "unknown",
	"org.hibernate.jpa.test.emops.RefreshTest.testRefreshNonManaged":                                                                                                                             "unknown",
	"org.hibernate.jpa.test.emops.RemoveTest.testRemove":                                                                                                                                         "unknown",
	"org.hibernate.jpa.test.emops.RemoveTest.testRemoveAndFind":                                                                                                                                  "unknown",
	"org.hibernate.jpa.test.emops.RemoveTest.testUpdatedAndRemove":                                                                                                                               "6583",
	"org.hibernate.jpa.test.emops.cascade.CascadePersistTest.testLazyCollectionsStayLazyOnPersist":                                                                                               "unknown",
	"org.hibernate.jpa.test.enhancement.TestLazyPropertyOnPreUpdate.testNoUpdate":                                                                                                                "unknown",
	"org.hibernate.jpa.test.enhancement.TestLazyPropertyOnPreUpdate.testPreUpdate":                                                                                                               "unknown",
	"org.hibernate.jpa.test.enhancement.TestLazyPropertyOnPreUpdate.testPreUpdateOverride":                                                                                                       "unknown",
	"org.hibernate.jpa.test.exception.ExceptionTest.testConstraintViolationException":                                                                                                            "26738",
	"org.hibernate.jpa.test.exception.ExceptionTest.testEntityNotFoundException":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.exception.ExceptionTest.testOptimisticLockingException":                                                                                                              "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphNativeQueryTest.testNativeQueryFetchGraph":                                                                                                         "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphNativeQueryTest.testNativeQueryLoadGraph":                                                                                                          "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphNativeQueryTest.testQuery":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.attributeNodeInheritanceTest":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.inheritanceTest":                                                                                                                              "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.loadCollection":                                                                                                                               "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.loadInverseCollection":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.loadIsMemeberQueriedCollection":                                                                                                               "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphTest.loadMultipleAssociations":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphUsingFetchGraphTest.fetchAttributeNodeFromSubgraph":                                                                                                "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphUsingFetchGraphTest.fetchSubGraphFromSubgraph":                                                                                                     "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphUsingFetchGraphTest.fetchUsingHql":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphWithFetchAnnotationTest.testWithEntityGraph":                                                                                                       "unknown",
	"org.hibernate.jpa.test.graphs.EntityGraphWithFetchAnnotationTest.testWithoutEntityGraph":                                                                                                    "unknown",
	"org.hibernate.jpa.test.graphs.inherited.InheritedEntityGraphTest.collectionAttributeNodeInheritanceTest":                                                                                    "unknown",
	"org.hibernate.jpa.test.graphs.inherited.InheritedEntityGraphTest.collectionAttributeSubgraphInheritanceTest":                                                                                "unknown",
	"org.hibernate.jpa.test.graphs.inherited.InheritedEntityGraphTest.singleAttributeNodeInheritanceTest":                                                                                        "unknown",
	"org.hibernate.jpa.test.graphs.inherited.InheritedEntityGraphTest.singleAttributeSubgraphInheritanceTest":                                                                                    "unknown",
	"org.hibernate.jpa.test.graphs.mapped_by_id.LoadGraphFindByIdTest.findByPrimaryKeyWithQuery":                                                                                                 "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testCollectionSizeLoadedWithGraph":                                                                                         "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testEntityGraphWithCollectionSubquery":                                                                                     "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testEntityGraphWithExplicitFetch":                                                                                          "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testEntityGraphWithExplicitFetchAndRestriction":                                                                            "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testLoadGraph":                                                                                                             "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testLoadGraphOrderByWithImplicitJoin":                                                                                      "unknown",
	"org.hibernate.jpa.test.graphs.queryhint.QueryHintEntityGraphTest.testLoadGraphWithRestriction":                                                                                              "unknown",
	"org.hibernate.jpa.test.hibernateFilters.EntityManagerWithFilteredSessionTest.testTypedQueryCreation":                                                                                        "unknown",
	"org.hibernate.jpa.test.hibernateFilters.ProxyPreservingFiltersOutsideInitialSessionTest.testChangeFilterBeforeInitializeInSameSession":                                                      "unknown",
	"org.hibernate.jpa.test.hibernateFilters.ProxyPreservingFiltersOutsideInitialSessionTest.testMergeNoFilterThenInitializeTempSession":                                                         "unknown",
	"org.hibernate.jpa.test.hibernateFilters.ProxyPreservingFiltersOutsideInitialSessionTest.testSaveOrUpdateNoFilterThenInitializeTempSession":                                                  "unknown",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                                       "6583",
	"org.hibernate.jpa.test.indetifier.AssignedInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                                      "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheFirstGeneratedIdIsEqualToTableGeneratorInitialValuePlusOne":                                        "6583",
	"org.hibernate.jpa.test.indetifier.DefaultInitialValueTableGeneratorConfiguredTest.testTheGeneratedIdValuesAreCorrect":                                                                       "6583",
	"org.hibernate.jpa.test.inheritance.InheritanceTest.testFind":                                                                                                                                "unknown",
	"org.hibernate.jpa.test.lob.BlobTest.testBlobSerialization":                                                                                                                                  "26725",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticLock":                                                                                                                          "6583",
	"org.hibernate.jpa.test.lock.LockTest.testContendedPessimisticWriteLockNoWait":                                                                                                               "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithPessimisticWriteLockTimeoutException":                                                                                                      "6583",
	"org.hibernate.jpa.test.lock.LockTest.testFindWithTimeoutHint":                                                                                                                               "6583",
	"org.hibernate.jpa.test.lock.LockTest.testLockOptimistic":                                                                                                                                    "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockOptimisticForceIncrement":                                                                                                                      "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockOptimisticForceIncrementDifferentEm":                                                                                                           "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockPessimisticForceIncrement":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockRead":                                                                                                                                          "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockWrite":                                                                                                                                         "unknown",
	"org.hibernate.jpa.test.lock.LockTest.testLockWriteOnUnversioned":                                                                                                                            "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockSkipLocked":                                                                                                           "6583",
	"org.hibernate.jpa.test.lock.LockTest.testUpdateWithPessimisticReadLockWithoutNoWait":                                                                                                        "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testEntityLockModeStateAfterQueryLocking":                                                                                                      "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testNoVersionCheckAfterRemove":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testOptimisticForcedIncrementOverall":                                                                                                          "unknown",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testOptimisticForcedIncrementSpecific":                                                                                                         "unknown",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testOptimisticOverall":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testOptimisticSpecific":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementOverall":                                                                                                         "6583",
	"org.hibernate.jpa.test.lock.QueryLockingTest.testPessimisticForcedIncrementSpecific":                                                                                                        "6583",
	"org.hibernate.jpa.test.lock.StatementIsClosedAfterALockExceptionTest.testStatementIsClosed":                                                                                                 "6583",
	"org.hibernate.jpa.test.lock.UpgradeLockTest.testUpgradeReadLockToOptimisticForceIncrement":                                                                                                  "unknown",
	"org.hibernate.jpa.test.mapping.DefaultCascadeTest.testCascadePersist":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.mapping.UnidirectionalOneToManyIndexColumnTest.testRemovingAChild":                                                                                                   "unknown",
	"org.hibernate.jpa.test.mapping.UnidirectionalOneToManyOrderColumnTest.testAddingAnElement":                                                                                                  "unknown",
	"org.hibernate.jpa.test.mapping.UnidirectionalOneToManyOrderColumnTest.testRemovingAnElement":                                                                                                "unknown",
	"org.hibernate.jpa.test.mapping.UnidirectionalOneToManyOrderColumnTest.testRemovingAndAddingAnElement":                                                                                       "unknown",
	"org.hibernate.jpa.test.mapping.UnidirectionalOneToManyOrderColumnTest.testRemovingOneAndAddingTwoElements":                                                                                  "unknown",
	"org.hibernate.jpa.test.ops.ContainsTest.testLifecycle":                                                                                                                                      "unknown",
	"org.hibernate.jpa.test.ops.FindTest.testSubclassWrongId":                                                                                                                                    "unknown",
	"org.hibernate.jpa.test.ops.GetLoadJpaComplianceDifferentSessionsTest.testReattachEntityToSessionWithJpaComplianceProxy":                                                                     "unknown",
	"org.hibernate.jpa.test.ops.GetLoadJpaComplianceTest.testLoadIdNotFound_FieldBasedAccess":                                                                                                    "unknown",
	"org.hibernate.jpa.test.ops.GetLoadJpaComplianceTest.testLoadIdNotFound_PropertyBasedAccess":                                                                                                 "unknown",
	"org.hibernate.jpa.test.ops.GetLoadJpaComplianceTest.testReferenceIdNotFound_FieldBasedAccess":                                                                                               "unknown",
	"org.hibernate.jpa.test.ops.GetLoadJpaComplianceTest.testReferenceIdNotFound_PropertyBasedAccess":                                                                                            "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testGet":                                                                                                                                             "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testLoad":                                                                                                                                            "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testLoadGetId":                                                                                                                                       "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testLoadIdNotFound_FieldBasedAccess":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testLoadIdNotFound_PropertyBasedAccess":                                                                                                              "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testReferenceIdNotFound_FieldBasedAccess":                                                                                                            "unknown",
	"org.hibernate.jpa.test.ops.GetLoadTest.testReferenceIdNotFound_PropertyBasedAccess":                                                                                                         "unknown",
	"org.hibernate.jpa.test.ops.MergeNewTest.testMergeAfterRemove":                                                                                                                               "unknown",
	"org.hibernate.jpa.test.ops.MergeNewTest.testMergeNew":                                                                                                                                       "unknown",
	"org.hibernate.jpa.test.ops.MergeTest.testMergeTree":                                                                                                                                         "unknown",
	"org.hibernate.jpa.test.ops.PersistTest.testBasic":                                                                                                                                           "unknown",
	"org.hibernate.jpa.test.ops.PersistTest.testCreateException":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.ops.PersistTest.testCreateExceptionWithGeneratedId":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.ops.PersistTest.testCreateTree":                                                                                                                                      "unknown",
	"org.hibernate.jpa.test.ops.PersistTest.testCreateTreeWithGeneratedId":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.orphan.onetoone.OneToOneOrphanTest.testFlushTransientOneToOneNoCascade":                                                                                              "unknown",
	"org.hibernate.jpa.test.orphan.onetoone.multilevelcascade.DeleteMultiLevelOrphansTest.testDirectAndNestedAssociationsOrphanedWhileManaged":                                                   "unknown",
	"org.hibernate.jpa.test.orphan.onetoone.multilevelcascade.DeleteMultiLevelOrphansTest.testDirectAssociationOrphanedWhileManaged":                                                             "unknown",
	"org.hibernate.jpa.test.orphan.onetoone.multilevelcascade.DeleteMultiLevelOrphansTest.testReplacedDirectAssociationWhileManaged":                                                             "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testConfiguration":                                                                                                               "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testDefaultPar":                                                                                                                  "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testDefaultParForPersistence_1_0":                                                                                                "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testExcludeHbmPar":                                                                                                               "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testExplodedPar":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testExtendedEntityManager":                                                                                                       "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testExternalJar":                                                                                                                 "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testListenersDefaultPar":                                                                                                         "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testORMFileOnMainAndExplicitJars":                                                                                                "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testOverriddenPar":                                                                                                               "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testRelativeJarReferences":                                                                                                       "unknown",
	"org.hibernate.jpa.test.packaging.PackagedEntityManagerTest.testSpacePar":                                                                                                                    "unknown",
	"org.hibernate.jpa.test.query.AddNamedQueryTest.basicTest":                                                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.CachedQueryTest.testCacheableQuery":                                                                                                                            "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuery":                                                                                             "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testConstructorResultNativeQuerySpecifyingType":                                                                               "unknown",
	"org.hibernate.jpa.test.query.ConstructorResultNativeQueryTest.testMultipleConstructorResultNativeQuery":                                                                                     "unknown",
	"org.hibernate.jpa.test.query.DateTimeParameterTest.testBindingCalendarAsDate":                                                                                                               "unknown",
	"org.hibernate.jpa.test.query.LimitExpressionTest.testAnEmptyListIsReturnedWhenSetMaxResultsToZero":                                                                                          "unknown",
	"org.hibernate.jpa.test.query.NamedQueryCommentTest.testSelectNamedNativeQueryWithSqlComment":                                                                                                "unknown",
	"org.hibernate.jpa.test.query.NamedQueryCommentTest.testSelectNamedQueryWithSqlComment":                                                                                                      "unknown",
	"org.hibernate.jpa.test.query.NamedQueryCommentTest.testUpdateNamedNativeQueryWithSqlComment":                                                                                                "unknown",
	"org.hibernate.jpa.test.query.NamedQueryCommentTest.testUpdateNamedQueryWithSqlComment":                                                                                                      "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNamedQueriesOrdinalParametersAreOneBased":                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNamedQueryOrdinalParametersConflict":                                                                                                        "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNamedQueryOrdinalParametersConflict2":                                                                                                       "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNativeNamedQueriesOrdinalParametersAreOneBased":                                                                                             "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNativeNamedQueriesOrdinalParametersConflict":                                                                                                "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNativeNamedQueriesOrdinalParametersConflict2":                                                                                               "unknown",
	"org.hibernate.jpa.test.query.NamedQueryTest.testNativeQueriesFromNamedQueriesDoNotShareQuerySpaces":                                                                                         "unknown",
	"org.hibernate.jpa.test.query.NativeQueryOrdinalParametersTest.testConflictWithSessionNativeQuery":                                                                                           "unknown",
	"org.hibernate.jpa.test.query.NativeQueryOrdinalParametersTest.testNativeQueryIndexedOrdinalParameter":                                                                                       "unknown",
	"org.hibernate.jpa.test.query.NativeQueryOrdinalParametersTest.testNativeQueryOrdinalParameter":                                                                                              "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testAggregationReturnType":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testDistinct":                                                                                                                                        "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testEscapeCharacter":                                                                                                                                 "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testExplicitPositionalParameter":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testGetSingleResultWithNoResultException":                                                                                                            "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testIsNull":                                                                                                                                          "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testMultipleParameterLists":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNamedParameterWithUserError":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNativeQueryByEntity":                                                                                                                             "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNativeQueryByResultSet":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNativeQueryWithPositionalParameter":                                                                                                              "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNativeQuestionMarkParameter":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullNamedParameter":                                                                                                                              "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullNamedParameterParameter":                                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullNamedParameterParameterIncompatible":                                                                                                         "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullPositionalParameter":                                                                                                                         "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullPositionalParameterParameter":                                                                                                                "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testNullPositionalParameterParameterIncompatible":                                                                                                    "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testPagedQuery":                                                                                                                                      "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testParameterCollectionAndPositional":                                                                                                                "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testParameterCollectionParenthesesAndPositional":                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testParameterCollectionSingletonParenthesesAndPositional":                                                                                            "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testParameterList":                                                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testParameterListInExistingParens":                                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testPositionalParameterForms":                                                                                                                        "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testPositionalParameterWithUserError":                                                                                                                "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testTypeExpression":                                                                                                                                  "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testTypedNamedNativeQuery":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testTypedScalarQueries":                                                                                                                              "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testUnavailableNamedQuery":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.query.QueryTest.testUpdateQuery":                                                                                                                                     "unknown",
	"org.hibernate.jpa.test.query.QueryWithInParamListAndNamedEntityGraphTest.testEntityGraph":                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.QueryWithInParamListAndNamedEntityGraphTest.testEntityGraphAndInClause":                                                                                        "unknown",
	"org.hibernate.jpa.test.query.QueryWithInParamListAndNamedEntityGraphTest.testInClause":                                                                                                      "unknown",
	"org.hibernate.jpa.test.query.QueryWithLiteralsInSelectExpressionTest.testSelectLiterals":                                                                                                    "unknown",
	"org.hibernate.jpa.test.query.QueryWithLiteralsInSelectExpressionTest.testSelectNonNullLiteralsCastToBoolean":                                                                                "unknown",
	"org.hibernate.jpa.test.query.QueryWithLiteralsInSelectExpressionTest.testSelectNullLiterals":                                                                                                "unknown",
	"org.hibernate.jpa.test.query.TupleQueryTest.testGetAlias":                                                                                                                                   "unknown",
	"org.hibernate.jpa.test.query.TupleQueryTest.testGetAliasReturnNullIfNoAliasExist":                                                                                                           "unknown",
	"org.hibernate.jpa.test.query.TypedValueParametersTest.testJpa":                                                                                                                              "unknown",
	"org.hibernate.jpa.test.query.TypedValueParametersTest.testNative":                                                                                                                           "unknown",
	"org.hibernate.jpa.test.refresh.RefreshDetachedInstanceWhenIsAllowedTest.testRefreshDetachedInstance":                                                                                        "unknown",
	"org.hibernate.jpa.test.refresh.RefreshDetachedInstanceWhenIsAllowedTest.testUnwrappedSessionRefreshDetachedInstance":                                                                        "unknown",
	"org.hibernate.jpa.test.refresh.RefreshDetachedInstanceWhenIsNotAllowedTest.testRefreshDetachedInstance":                                                                                     "unknown",
	"org.hibernate.jpa.test.refresh.RefreshDetachedInstanceWhenIsNotAllowedTest.testUnwrappedSessionRefreshDetachedInstance":                                                                     "unknown",
	"org.hibernate.jpa.test.schemagen.SchemaCreateDropTest.testQueryWithoutTransaction":                                                                                                          "unknown",
	"org.hibernate.jpa.test.temporal.TemporalTypeTest.testTemporalType":                                                                                                                          "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testCommitReleasesLogicalConnection":                                                                         "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testMergeThenCloseWithAnActiveTransaction":                                                                   "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testMergeWithDeletionOrphanRemovalThenCloseWithAnActiveTransaction":                                          "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testPersistThenCloseWithAnActiveTransaction":                                                                 "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testRemoveThenCloseWithAnActiveTransaction":                                                                  "unknown",
	"org.hibernate.jpa.test.transaction.CloseEntityManagerWithActiveTransactionTest.testUpdateThenCloseWithAnActiveTransaction":                                                                  "unknown",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testAlwaysTransactionalOperations":                                                                                               "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testMergeWhenExtended":                                                                                                           "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackClearPC":                                                                                                             "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testRollbackExceptionOnOptimisticLockException":                                                                                  "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testSetRollbackOnlyAndFlush":                                                                                                     "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionAndContains":                                                                                                      "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionCommitDoesNotFlush":                                                                                               "6583",
	"org.hibernate.jpa.test.transaction.FlushAndTransactionTest.testTransactionalOperationsWhenExtended":                                                                                         "6583",
	"org.hibernate.jpa.test.transaction.JtaReusingEntityTransactionTest.entityTransactionShouldBeReusableTest":                                                                                   "unknown",
	"org.hibernate.jpa.test.transaction.TransactionRollbackTest.checkRollBackTransactionIsExecutedOnceWhenACommitFails":                                                                          "unknown",
	"org.hibernate.jpa.test.xml.JpaEntityNameTest.testUsingSimpleHbmInJpa":                                                                                                                       "unknown",
	"org.hibernate.jpa.test.xml.XmlAttributeOverrideTest.testAttributeOverriding":                                                                                                                "unknown",
	"org.hibernate.jpa.test.xml.XmlAttributeOverrideTest.testDefaultEventListener":                                                                                                               "unknown",
	"org.hibernate.jpa.test.xml.sequences.XmlAttributeOverrideTest.testAttributeOverriding":                                                                                                      "unknown",
	"org.hibernate.jpa.userguide.util.GetIdentifierTest.testEmbeddedId":                                                                                                                          "unknown",
	"org.hibernate.jpa.userguide.util.GetIdentifierTest.testIdClass":                                                                                                                             "unknown",
	"org.hibernate.jpa.userguide.util.GetIdentifierTest.testProxyObject":                                                                                                                         "unknown",
	"org.hibernate.jpa.userguide.util.GetIdentifierTest.testSimpleId":                                                                                                                            "unknown",
	"org.hibernate.jpa.userguide.util.IsLoadedTest.testIsLoadedOnPrivateSuperclassProperty":                                                                                                      "unknown",
	"org.hibernate.jpa.version.JpaSpecVersionValueUpdatingTest.testVersionNotIncrementedOnModificationOfNonOwningCollectionCascaded":                                                             "unknown",
	"org.hibernate.jpa.version.JpaSpecVersionValueUpdatingTest.testVersionNotIncrementedOnModificationOfNonOwningCollectionNonCascaded":                                                          "unknown",
	"org.hibernate.persister.entity.CustomSqlSchemaResolvingTest.testSchemaNotReplacedInCustomSQL":                                                                                               "unknown",
	"org.hibernate.property.DirectPropertyAccessorTest.testDirectIdPropertyAccess":                                                                                                               "unknown",
	"org.hibernate.property.GetAndIsVariantGetterWithTransientAnnotationTest.testBothGetterAndIsVariantAreIgnoredWhenMarkedTransient":                                                            "unknown",
	"org.hibernate.property.GetAndIsVariantGetterWithTransientAnnotationTest.testGetAndIsVariantCanHaveDifferentReturnValueWhenOneHasATransientAnnotation":                                       "unknown",
	"org.hibernate.query.InClauseParameterPaddingTest.testInClauseParameterPadding":                                                                                                              "unknown",
	"org.hibernate.query.NativeQueryWithParenthesesTest.testParseParentheses":                                                                                                                    "unknown",
	"org.hibernate.query.criteria.internal.expression.SearchedCaseExpressionTest.testEqualClause":                                                                                                "unknown",
	"org.hibernate.query.hhh12076.AnnotationMappingJoinClassTest.testClassExpressionInOnClause":                                                                                                  "unknown",
	"org.hibernate.query.hhh12076.AnnotationMappingJoinClassTest.testClassExpressionInWhereClause":                                                                                               "unknown",
	"org.hibernate.query.hhh12225.HQLTypeTest.test":                                                                                                                                              "unknown",
	"org.hibernate.secure.JaccIntegratorTest.testAllow":                                                                                                                                          "unknown",
	"org.hibernate.serialization.EntityProxySerializationTest.testInitializedProxySerializationIfTargetInPersistenceContext":                                                                     "unknown",
	"org.hibernate.serialization.EntityProxySerializationTest.testProxyInitializationWithoutTX":                                                                                                  "unknown",
	"org.hibernate.serialization.EntityProxySerializationTest.testProxyInitializationWithoutTXAfterDeserialization":                                                                              "unknown",
	"org.hibernate.serialization.EntityProxySerializationTest.testUninitializedProxySerializationIfTargetInPersistenceContext":                                                                   "unknown",
	"org.hibernate.serialization.MapProxySerializationTest.testInitializedProxySerializationIfTargetInPersistenceContext":                                                                        "unknown",
	"org.hibernate.serialization.MapProxySerializationTest.testProxyInitializationWithoutTX":                                                                                                     "unknown",
	"org.hibernate.serialization.MapProxySerializationTest.testProxyInitializationWithoutTXAfterDeserialization":                                                                                 "unknown",
	"org.hibernate.serialization.MapProxySerializationTest.testUninitializedProxySerializationIfTargetInPersistenceContext":                                                                      "unknown",
	"org.hibernate.session.AssociateEntityWithTwoSessionsTest.test":                                                                                                                              "unknown",
	"org.hibernate.sharedSession.SessionWithSharedConnectionTest.testChildSessionCallsAfterTransactionAction":                                                                                    "unknown",
	"org.hibernate.sharedSession.SessionWithSharedConnectionTest.testSharedTransactionContextFlushBeforeCompletion":                                                                              "unknown",
	"org.hibernate.sharedSession.SessionWithSharedConnectionTest.testSharedTransactionContextSessionClosing":                                                                                     "unknown",
	"org.hibernate.test.abstractembeddedcomponents.cid.AbstractCompositeIdTest.testEmbeddedCompositeIdentifierOnAbstractClass":                                                                   "unknown",
	"org.hibernate.test.abstractembeddedcomponents.propertyref.AbstractComponentPropertyRefTest.testPropertiesRefCascades":                                                                       "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testAnnReferencesHbm":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testDeclarativeMix":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testHbmWithSubclassExtends":                                                                                                                "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testIgnoringHbm":                                                                                                                           "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testPrecedenceAnnotation":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.ConfigurationTest.testPrecedenceHbm":                                                                                                                         "unknown",
	"org.hibernate.test.annotations.CreationTimestampTest.generatesCurrentTimestamp":                                                                                                             "unknown",
	"org.hibernate.test.annotations.EntityTest.testColumn":                                                                                                                                       "unknown",
	"org.hibernate.test.annotations.EntityTest.testColumnUnique":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.EntityTest.testEntityName":                                                                                                                                   "unknown",
	"org.hibernate.test.annotations.EntityTest.testFieldAccess":                                                                                                                                  "unknown",
	"org.hibernate.test.annotations.EntityTest.testLoad":                                                                                                                                         "unknown",
	"org.hibernate.test.annotations.EntityTest.testNonGetter":                                                                                                                                    "unknown",
	"org.hibernate.test.annotations.EntityTest.testTemporalType":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.EntityTest.testUniqueConstraint":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.EntityTest.testVersion":                                                                                                                                      "unknown",
	"org.hibernate.test.annotations.JoinedSubclassTest.testCompositePk":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.JoinedSubclassTest.testDeclaredValues":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.JoinedSubclassTest.testDefaultValues":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.replaceParentClassElementCollection":                                                                                          "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.replaceSubClassElementCollection":                                                                                             "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.updateDetachedEntity":                                                                                                         "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.updateParentClassOneToOneAssociation":                                                                                         "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.updateParentClassProperty":                                                                                                    "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.updateSubClassOnrToOneAssociation":                                                                                            "unknown",
	"org.hibernate.test.annotations.UpdateTimeStampInheritanceTest.updateSubClassProperty":                                                                                                       "unknown",
	"org.hibernate.test.annotations.UpdateTimestampTest.generatesCurrentTimestamp":                                                                                                               "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testFieldsOverriding":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testNonOverridenSubclass":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testOverridenSubclass":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testPropertyOverriding":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testSuperclassNonOverriding":                                                                                                               "unknown",
	"org.hibernate.test.annotations.access.AccessTest.testSuperclassOverriding":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testDefaultConfigurationModeIsInherited":                                                                                               "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testEmbeddableExplicitAccessStrategy":                                                                                                  "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testEmbeddableUsesAccessStrategyOfContainingClass":                                                                                     "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testFieldsOverriding":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testNonOverridenSubclass":                                                                                                              "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testOverridenSubclass":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testPropertyOverriding":                                                                                                                "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testSuperclassNonOverriding":                                                                                                           "unknown",
	"org.hibernate.test.annotations.access.jpa.AccessTest.testSuperclassOverriding":                                                                                                              "unknown",
	"org.hibernate.test.annotations.any.AnyTest.testDefaultAnyAssociation":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.any.AnyTest.testManyToAnyWithMap":                                                                                                                            "unknown",
	"org.hibernate.test.annotations.any.AnyTest.testMetaDataUseWithManyToAny":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.array.ArrayTest.testOneToMany":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.beanvalidation.BeanValidationAutoTest.testListeners":                                                                                                         "unknown",
	"org.hibernate.test.annotations.beanvalidation.BeanValidationDisabledTest.testListeners":                                                                                                     "unknown",
	"org.hibernate.test.annotations.beanvalidation.BeanValidationGroupsTest.testListeners":                                                                                                       "unknown",
	"org.hibernate.test.annotations.beanvalidation.BeanValidationProvidedFactoryTest.testListeners":                                                                                              "unknown",
	"org.hibernate.test.annotations.beanvalidation.DDLWithoutCallbackTest.testMinAndMaxChecksGetApplied":                                                                                         "unknown",
	"org.hibernate.test.annotations.beanvalidation.DDLWithoutCallbackTest.testRangeChecksGetApplied":                                                                                             "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testAssocInEmbeddedNotValidated":                                                                             "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testCollectionAssocNotValidated":                                                                             "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testEmbedded":                                                                                                "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testEmbeddedCollection":                                                                                      "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testNonLazyAssocFieldWithConstraintsFailureExpected":                                                         "unknown",
	"org.hibernate.test.annotations.beanvalidation.HibernateTraversableResolverTest.testToOneAssocNotValidated":                                                                                  "unknown",
	"org.hibernate.test.annotations.beanvalidation.MergeNotNullCollectionTest.testOneToManyNotNullCollection":                                                                                    "unknown",
	"org.hibernate.test.annotations.beanvalidation.MergeNotNullCollectionTest.testOneToManyNullCollection":                                                                                       "unknown",
	"org.hibernate.test.annotations.bytecode.ProxyBreakingTest.testProxiedBridgeMethod":                                                                                                          "unknown",
	"org.hibernate.test.annotations.cascade.CascadeTest.testDetach":                                                                                                                              "unknown",
	"org.hibernate.test.annotations.cascade.CascadeTest.testMerge":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.cascade.CascadeTest.testPersist":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.cascade.CascadeTest.testRemove":                                                                                                                              "unknown",
	"org.hibernate.test.annotations.cascade.CascadeToEmbeddedManyToOneTest.testPersistCascadeToEmbedded":                                                                                         "unknown",
	"org.hibernate.test.annotations.cascade.CascadeToEmbeddedManyToOneTest.testPersistCascadeToSetOfEmbedded":                                                                                    "unknown",
	"org.hibernate.test.annotations.cascade.NonNullableCircularDependencyCascadeTest.testIdClassInSuperclass":                                                                                    "unknown",
	"org.hibernate.test.annotations.cascade.circle.identity.CascadeCircleIdentityIdTest.testCascade":                                                                                             "unknown",
	"org.hibernate.test.annotations.cascade.circle.sequence.CascadeCircleSequenceIdTest.testCascade":                                                                                             "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.jpa.identity.MultiCircleJpaCascadeIdentityTest.testMerge":                                                                                "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.jpa.identity.MultiCircleJpaCascadeIdentityTest.testPersist":                                                                              "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.jpa.sequence.MultiCircleJpaCascadeSequenceTest.testMerge":                                                                                "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.jpa.sequence.MultiCircleJpaCascadeSequenceTest.testPersist":                                                                              "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.identity.MultiCircleNonJpaCascadeIdentityTest.testMerge":                                                                          "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.identity.MultiCircleNonJpaCascadeIdentityTest.testPersist":                                                                        "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.identity.MultiCircleNonJpaCascadeIdentityTest.testSave":                                                                           "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.identity.MultiCircleNonJpaCascadeIdentityTest.testSaveOrUpdate":                                                                   "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.sequence.MultiCircleNonJpaCascadeSequenceTest.testMerge":                                                                          "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.sequence.MultiCircleNonJpaCascadeSequenceTest.testPersist":                                                                        "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.sequence.MultiCircleNonJpaCascadeSequenceTest.testSave":                                                                           "unknown",
	"org.hibernate.test.annotations.cascade.multicircle.nonjpa.sequence.MultiCircleNonJpaCascadeSequenceTest.testSaveOrUpdate":                                                                   "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testGetWithDetachedEntityInCompositeIDWithManagedCopy":                                                                                   "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testGetWithUpdatedDetachedEntityInCompositeID":                                                                                           "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositeId":                                                                                                              "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositeIdClass":                                                                                                         "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositeIdClassInPC":                                                                                                     "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositePk":                                                                                                              "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositePkAndSubclass":                                                                                                   "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testManyToOneInCompositePkInPC":                                                                                                          "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testOneToOneInCompositePk":                                                                                                               "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testQueryInAndComposite":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testQueryInAndCompositeWithHQL":                                                                                                          "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testSecondaryTableWithCompositeId":                                                                                                       "unknown",
	"org.hibernate.test.annotations.cid.CompositeIdTest.testSecondaryTableWithIdClass":                                                                                                           "unknown",
	"org.hibernate.test.annotations.cid.keymanytoone.EagerKeyManyToOneTest.testLoadEntityWithEagerFetchingToKeyManyToOneReferenceBackToSelf":                                                     "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testAttributedJoin":                                                                              "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testCompositeElement":                                                                            "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testFetchEagerAndFilter":                                                                         "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testLazyCollectionofElements":                                                                    "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testMapKeyType":                                                                                  "unknown",
	"org.hibernate.test.annotations.collectionelement.CustomImprovedNamingCollectionElementTest.testSimpleElement":                                                                               "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testAttributedJoin":                                                                                     "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testCompositeElement":                                                                                   "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testFetchEagerAndFilter":                                                                                "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testLazyCollectionofElements":                                                                           "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testMapKeyType":                                                                                         "unknown",
	"org.hibernate.test.annotations.collectionelement.DefaultNamingCollectionElementTest.testSimpleElement":                                                                                      "unknown",
	"org.hibernate.test.annotations.collectionelement.EmbeddableCollectionElementWithLazyManyToOneTest.testLazyBoth":                                                                             "unknown",
	"org.hibernate.test.annotations.collectionelement.EmbeddableCollectionElementWithLazyManyToOneTest.testLazyManyToOneInCollectionElementEmbeddable":                                           "unknown",
	"org.hibernate.test.annotations.collectionelement.EmbeddableCollectionElementWithLazyManyToOneTest.testLazyManyToOneInEmbeddable":                                                            "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testAttributedJoin":                                                                                    "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testCompositeElement":                                                                                  "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testFetchEagerAndFilter":                                                                               "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testLazyCollectionofElements":                                                                          "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testMapKeyType":                                                                                        "unknown",
	"org.hibernate.test.annotations.collectionelement.ImprovedNamingCollectionElementTest.testSimpleElement":                                                                                     "unknown",
	"org.hibernate.test.annotations.collectionelement.OrderByTest.testOrderByName":                                                                                                               "unknown",
	"org.hibernate.test.annotations.collectionelement.OrderByTest.testOrderByWithDottedNotation":                                                                                                 "unknown",
	"org.hibernate.test.annotations.collectionelement.QueryTest.testMemberOfSyntax":                                                                                                              "unknown",
	"org.hibernate.test.annotations.collectionelement.embeddables.withcustomenumdef.TestBasicOps.testEmbeddableWithNullables":                                                                    "unknown",
	"org.hibernate.test.annotations.collectionelement.embeddables.withcustomenumdef.TestBasicOps.testLoadAndStore":                                                                               "unknown",
	"org.hibernate.test.annotations.collectionelement.indexedCollection.IndexedCollectionOfElementsTest.testIndexedCollectionOfElements":                                                         "unknown",
	"org.hibernate.test.annotations.collectionelement.ordered.ElementCollectionSortingTest.testSortingElementCollectionSyntax":                                                                   "unknown",
	"org.hibernate.test.annotations.collectionelement.ordered.ElementCollectionSortingTest.testSortingEmbeddableCollectionOfPrimitives":                                                          "unknown",
	"org.hibernate.test.annotations.collectionelement.recreate.RecreateCollectionTest.testUpdateCollectionOfElements":                                                                            "unknown",
	"org.hibernate.test.annotations.dataTypes.BasicOperationsTest.testCreateAndDelete":                                                                                                           "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeDerivedIdentityTest.testBidirectonalKeyManyToOneId":                                                                 "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeDerivedIdentityTest.testCreateProject":                                                                              "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testBindTransientEntityWithCopiedKeyManyToOne":                                           "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testBindTransientEntityWithDetachedKeyManyToOne":                                         "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testBindTransientEntityWithPersistentKeyManyToOne":                                       "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testBindTransientEntityWithTransientKeyManyToOne":                                        "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testMergeDetachedIdManyToOne":                                                            "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.CompositeIdDerivedIdWithIdClassTest.testMergeTransientIdManyToOne":                                                           "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.OneToOneWithDerivedIdentityTest.testCase":                                                                                    "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.OneToOneWithDerivedIdentityTest.testInsertFooAndBarWithDerivedIdPC":                                                          "unknown",
	"org.hibernate.test.annotations.derivedidentities.bidirectional.OneToOneWithDerivedIdentityTest.testSelectWithDerivedId":                                                                     "unknown",
	"org.hibernate.test.annotations.derivedidentities.e1.a.DerivedIdentitySimpleParentIdClassDepTest.testManyToOne":                                                                              "unknown",
	"org.hibernate.test.annotations.derivedidentities.e1.a.DerivedIdentitySimpleParentIdClassDepTest.testQueryNewEntityInPC":                                                                     "unknown",
	"org.hibernate.test.annotations.derivedidentities.e1.b.DerivedIdentitySimpleParentEmbeddedIdDepTest.testManyToOne":                                                                           "unknown",
	"org.hibernate.test.annotations.derivedidentities.e1.b.DerivedIdentitySimpleParentEmbeddedIdDepTest.testOneToOne":                                                                            "unknown",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.IdMapManyToOneSpecjTest.testComplexIdClass":                                                                                "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b.specjmapid.lazy.CompositeKeyDeleteTest.testRemove":                                                                                    "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.b2.IdClassGeneratedValueManyToOneTest.testComplexIdClass":                                                                               "6583",
	"org.hibernate.test.annotations.derivedidentities.e1.c.DerivedIdentitySimpleParentEmbeddedDepTest.testManyToOne":                                                                             "unknown",
	"org.hibernate.test.annotations.derivedidentities.e2.a.DerivedIdentityIdClassParentIdClassDepTest.testManytoOne":                                                                             "unknown",
	"org.hibernate.test.annotations.derivedidentities.e2.b.DerivedIdentityIdClassParentEmbeddedIdDepTest.testManyToOne":                                                                          "unknown",
	"org.hibernate.test.annotations.derivedidentities.e3.a.DerivedIdentityEmbeddedIdParentIdClassTest.testManyToOne":                                                                             "unknown",
	"org.hibernate.test.annotations.derivedidentities.e3.b.DerivedIdentityEmbeddedIdParentEmbeddedIdDepTest.testManyToOne":                                                                       "unknown",
	"org.hibernate.test.annotations.derivedidentities.e3.b2.DerivedIdentityEmbeddedIdParentEmbeddedIdGrandparentEmbeddedIdDepTest.testManyToOne":                                                 "unknown",
	"org.hibernate.test.annotations.derivedidentities.e3.b3.DerivedIdentityEmbeddedIdParentEmbeddedIdGrandparentEmbeddedIdColumnOverridesDepTest.testManyToOne":                                  "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.a.DerivedIdentitySimpleParentSimpleDepTest.testManyToOneExplicitJoinColumn":                                                             "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.a.DerivedIdentitySimpleParentSimpleDepTest.testOneToOneExplicitJoinColumn":                                                              "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.a.DerivedIdentitySimpleParentSimpleDepTest.testSimplePkValueLoading":                                                                    "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.b.DerivedIdentitySimpleParentSimpleDepMapsIdTest.testExplicitlyAssignedDependentIdAttributeValue":                                       "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.b.DerivedIdentitySimpleParentSimpleDepMapsIdTest.testManyToOneExplicitJoinColumn":                                                       "unknown",
	"org.hibernate.test.annotations.derivedidentities.e4.b.DerivedIdentitySimpleParentSimpleDepMapsIdTest.testOneToOneExplicitJoinColumn":                                                        "unknown",
	"org.hibernate.test.annotations.derivedidentities.e5.a.DerivedIdentityIdClassParentSameIdTypeIdClassDepTest.testOneToOneExplicitJoinColumn":                                                  "unknown",
	"org.hibernate.test.annotations.derivedidentities.e5.a.DerivedIdentityIdClassParentSameIdTypeIdClassDepTest.testTckLikeBehavior":                                                             "unknown",
	"org.hibernate.test.annotations.derivedidentities.e5.b.DerivedIdentityIdClassParentSameIdTypeEmbeddedIdDepTest.testOneToOneExplicitJoinColumn":                                               "unknown",
	"org.hibernate.test.annotations.derivedidentities.e5.c.ForeignGeneratorViaMapsIdTest.testForeignGenerator":                                                                                   "unknown",
	"org.hibernate.test.annotations.derivedidentities.e6.a.DerivedIdentityEmbeddedIdParentSameIdTypeIdClassDepTest.testOneToOneExplicitJoinColumn":                                               "unknown",
	"org.hibernate.test.annotations.derivedidentities.e6.b.DerivedIdentityEmbeddedIdParentSameIdTypeEmbeddedIdDepTest.testOneToOneExplicitJoinColumn":                                            "unknown",
	"org.hibernate.test.annotations.embeddables.attributeOverrides.AttributeOverrideEnhancedUserTypeTest.testIt":                                                                                 "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testCompositeId":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testDefaultCollectionTable":                                                                                                            "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testDottedProperty":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddableInsideEmbeddable":                                                                                                        "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddedAndMultipleManyToOne":                                                                                                      "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddedAndOneToMany":                                                                                                              "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddedAndOneToManyHql":                                                                                                           "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddedInSecondaryTable":                                                                                                          "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testEmbeddedSuperclass":                                                                                                                "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testManyToOneInsideComponent":                                                                                                          "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testParent":                                                                                                                            "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testQueryWithEmbeddedIsNull":                                                                                                           "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testQueryWithEmbeddedWithNullUsingSubAttributes":                                                                                       "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testSimple":                                                                                                                            "unknown",
	"org.hibernate.test.annotations.embedded.EmbeddedTest.testTransientMergeComponentParent":                                                                                                     "unknown",
	"org.hibernate.test.annotations.embedded.many2one.EmbeddableWithMany2OneTest.testBasicOps":                                                                                                   "unknown",
	"org.hibernate.test.annotations.embedded.many2one.EmbeddableWithMany2OneTest.testJoinAcrossEmbedded":                                                                                         "unknown",
	"org.hibernate.test.annotations.engine.collection.UnidirCollectionWithMultipleOwnerTest.testUnidirCollectionWithMultipleOwner":                                                               "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testCache":                                                                                                              "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testCascadedDeleteOfChildEntitiesBug2":                                                                                  "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testCascadedDeleteOfChildOneToOne":                                                                                      "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testCompositeType":                                                                                                      "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testEntity":                                                                                                             "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testFilter":                                                                                                             "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testFilterOnCollection":                                                                                                 "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testFormula":                                                                                                            "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testImportTypeDefinitions":                                                                                              "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testInheritFiltersFromMappedSuperclass":                                                                                 "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testNonLazy":                                                                                                            "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testParameterizedType":                                                                                                  "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testPolymorphism":                                                                                                       "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testSerialized":                                                                                                         "26725",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testSetSimpleValueTypeNameInSecondPass":                                                                                 "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testType":                                                                                                               "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testTypeDefNameAndDefaultForTypeAttributes":                                                                             "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testTypeDefWithoutNameAndDefaultForTypeAttributes":                                                                      "unknown",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testVersioning":                                                                                                         "27274",
	"org.hibernate.test.annotations.entity.BasicHibernateAnnotationsTest.testWhereClause":                                                                                                        "unknown",
	"org.hibernate.test.annotations.entity.Java5FeaturesTest.testEnums":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.entity.Java5FeaturesTest.testInterface":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.entity.PropertyDefaultMappingsTest.testSerializableObject":                                                                                                   "unknown",
	"org.hibernate.test.annotations.entity.PropertyDefaultMappingsTest.testTransientField":                                                                                                       "unknown",
	"org.hibernate.test.annotations.entitynonentity.EntityNonEntityTest.testMix":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.enumerated.EnumeratedTypeTest.testFormula":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.enumerated.EnumeratedTypeTest.testSet":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.enumerated.EnumeratedTypeTest.testTrimmedEnumChar":                                                                                                           "unknown",
	"org.hibernate.test.annotations.enumerated.EnumeratedTypeTest.testTypeCriteria":                                                                                                              "unknown",
	"org.hibernate.test.annotations.enumerated.EnumeratedTypeTest.testTypeQuery":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.enumerated.custom_mapkey.MapKeyCustomEnumTypeTest.testCriteria":                                                                                              "unknown",
	"org.hibernate.test.annotations.enumerated.custom_mapkey.MapKeyCustomEnumTypeTest.testQuery":                                                                                                 "unknown",
	"org.hibernate.test.annotations.enumerated.mapkey.MapKeyEnumeratedTest.testMapKeyEnumerated":                                                                                                 "unknown",
	"org.hibernate.test.annotations.fetch.FetchingTest.testExtraLazy":                                                                                                                            "unknown",
	"org.hibernate.test.annotations.fetch.FetchingTest.testHibernateFetchingLazy":                                                                                                                "unknown",
	"org.hibernate.test.annotations.fetch.FetchingTest.testLazy":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.fetch.FetchingTest.testOneToManyFetchEager":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.fetchprofile.MoreFetchProfileTest.testFetchWithTwoOverrides":                                                                                                 "unknown",
	"org.hibernate.test.annotations.filter.secondarytable.SecondaryTableTest.testFilter":                                                                                                         "unknown",
	"org.hibernate.test.annotations.filter.subclass.MappedSuperclass.FilterInheritanceTest.testSelectFromHuman":                                                                                  "unknown",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testClub":                                                                                                          "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testIqFilter":                                                                                                      "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testNonHumanFilter":                                                                                                "5807",
	"org.hibernate.test.annotations.filter.subclass.joined.JoinedSubClassTest.testPregnantFilter":                                                                                                "5807",
	"org.hibernate.test.annotations.filter.subclass.singletable.SingleTableTest.testIqFilter":                                                                                                    "unknown",
	"org.hibernate.test.annotations.filter.subclass.singletable.SingleTableTest.testNonHumanFilter":                                                                                              "unknown",
	"org.hibernate.test.annotations.filter.subclass.singletable.SingleTableTest.testPregnantFilter":                                                                                              "unknown",
	"org.hibernate.test.annotations.filter.subclass.tableperclass.TablePerClassTest.testIqFilter":                                                                                                "unknown",
	"org.hibernate.test.annotations.filter.subclass.tableperclass.TablePerClassTest.testNonHumanFilter":                                                                                          "unknown",
	"org.hibernate.test.annotations.filter.subclass.tableperclass.TablePerClassTest.testPregnantFilter":                                                                                          "unknown",
	"org.hibernate.test.annotations.formula.FormulaWithPartitionByTest.testFormulaAnnotationWithPartitionBy":                                                                                     "unknown",
	"org.hibernate.test.annotations.generics.EmbeddedGenericsTest.testWorksWithGenericEmbedded":                                                                                                  "unknown",
	"org.hibernate.test.annotations.generics.GenericsTest.testManyToOneGenerics":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.generics.UnresolvedTypeTest.testAcceptsUnresolvedPropertyTypesIfATargetEntityIsExplicitlySet":                                                                "unknown",
	"org.hibernate.test.annotations.generics.UnresolvedTypeTest.testAcceptsUnresolvedPropertyTypesIfATypeExplicitlySet":                                                                          "unknown",
	"org.hibernate.test.annotations.id.EnumIdTest.testEnumAsId":                                                                                                                                  "unknown",
	"org.hibernate.test.annotations.id.IdClassTest.testIdClassInSuperclass":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.id.IdTest.testClassLevelGenerator":                                                                                                                           "unknown",
	"org.hibernate.test.annotations.id.IdTest.testDefaultSequence":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.id.IdTest.testGenericGenerator":                                                                                                                              "unknown",
	"org.hibernate.test.annotations.id.IdTest.testGenericGenerators":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.id.IdTest.testIdClass":                                                                                                                                       "unknown",
	"org.hibernate.test.annotations.id.IdTest.testIdInEmbeddableSuperclass":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.id.IdTest.testLowAllocationSize":                                                                                                                             "6583",
	"org.hibernate.test.annotations.id.IdTest.testMethodLevelGenerator":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.id.IdTest.testNoGenerator":                                                                                                                                   "unknown",
	"org.hibernate.test.annotations.id.IdTest.testParameterizedAuto":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.id.IdTest.testSequenceGenerator":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.id.IdTest.testTableGenerator":                                                                                                                                "6583",
	"org.hibernate.test.annotations.id.sequences.EnumIdTest.testEnumAsId":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.id.sequences.IdClassTest.testIdClassInSuperclass":                                                                                                            "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testClassLevelGenerator":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testDefaultSequence":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testGenericGenerator":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testGenericGenerators":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testIdClass":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testIdInEmbeddableSuperclass":                                                                                                            "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testLowAllocationSize":                                                                                                                   "6583",
	"org.hibernate.test.annotations.id.sequences.IdTest.testMethodLevelGenerator":                                                                                                                "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testParameterizedAuto":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testSequenceGenerator":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testSequencePerEntity":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.id.sequences.IdTest.testTableGenerator":                                                                                                                      "6583",
	"org.hibernate.test.annotations.idclass.IdClassCompositePKTest.testEntityMappningPropertiesAreNotIgnored":                                                                                    "unknown",
	"org.hibernate.test.annotations.idclass.IdClassMappedSuperclassTest.testIdClassWithMappedSuperclassAndId":                                                                                    "unknown",
	"org.hibernate.test.annotations.idclassgeneratedvalue.IdClassGeneratedValueTest.testBaseLine":                                                                                                "unknown",
	"org.hibernate.test.annotations.idclassgeneratedvalue.IdClassGeneratedValueTest.testMultipleGeneratedValue":                                                                                  "unknown",
	"org.hibernate.test.annotations.idclassgeneratedvalue.IdClassGeneratedValueTest.testSingleGeneratedValue":                                                                                    "unknown",
	"org.hibernate.test.annotations.identifiercollection.IdentifierCollectionTest.testIdBag":                                                                                                     "6583",
	"org.hibernate.test.annotations.idmanytoone.IdManyToOneTest.testCriteriaRestrictionOnIdManyToOne":                                                                                            "unknown",
	"org.hibernate.test.annotations.idmanytoone.IdManyToOneTest.testIdClassManyToOne":                                                                                                            "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableEntityUpdateQueryHandlingModeExceptionTest.testBulkUpdate":                                                                                "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableEntityUpdateQueryHandlingModeWarningTest.testBulkUpdate":                                                                                  "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableTest.testChangeImmutableAttribute":                                                                                                        "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableTest.testImmutableAttribute":                                                                                                              "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableTest.testImmutableCollection":                                                                                                             "unknown",
	"org.hibernate.test.annotations.immutable.ImmutableTest.testImmutableEntity":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testDefaultMapKey":                                                                                                           "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testEmbeddableWithEntityKey":                                                                                                 "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testEntityKeyElementTarget":                                                                                                  "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testEnumKeyType":                                                                                                             "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testFkList":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testJoinedTableList":                                                                                                         "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKey":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyAndIdClass":                                                                                                        "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyEmbeddableWithEntityKey":                                                                                           "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyEntityEntity":                                                                                                      "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyLoad":                                                                                                              "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyOnManyToMany":                                                                                                      "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyOnManyToManyOnId":                                                                                                  "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testMapKeyToEntity":                                                                                                          "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testRealMap":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testSortedMap":                                                                                                               "unknown",
	"org.hibernate.test.annotations.indexcoll.IndexedCollectionTest.testTemporalKeyMap":                                                                                                          "unknown",
	"org.hibernate.test.annotations.indexcoll.MapKeyTest.testMapKeyOnEmbeddedId":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.indexcoll.eager.EagerIndexedCollectionTest.testEntityKeyElementTarget":                                                                                       "unknown",
	"org.hibernate.test.annotations.indexcoll.eager.EagerIndexedCollectionTest.testEnumKeyType":                                                                                                  "unknown",
	"org.hibernate.test.annotations.indexcoll.eager.EagerIndexedCollectionTest.testRealMap":                                                                                                      "unknown",
	"org.hibernate.test.annotations.indexcoll.eager.EagerIndexedCollectionTest.testTemporalKeyMap":                                                                                               "unknown",
	"org.hibernate.test.annotations.inheritance.SubclassTest.test2ndLevelSubClass":                                                                                                               "unknown",
	"org.hibernate.test.annotations.inheritance.SubclassTest.testEmbeddedSuperclass":                                                                                                             "unknown",
	"org.hibernate.test.annotations.inheritance.SubclassTest.testFormula":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.inheritance.SubclassTest.testPolymorphism":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassAndSecondaryTable.testSecondaryTableAndJoined":                                                                              "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testDefault":                                                                                                           "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testInheritance":                                                                                                       "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testJoinedAbstractClass":                                                                                               "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testManyToOneOnAbstract":                                                                                               "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testManyToOneWithJoinTable":                                                                                            "unknown",
	"org.hibernate.test.annotations.inheritance.joined.JoinedSubclassTest.testSecondaryTables":                                                                                                   "unknown",
	"org.hibernate.test.annotations.inheritance.mixed.SubclassTest.testDefault":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.inheritance.union.SubclassTest.testDefault":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testCompositePK":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testCustomSQL":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testDefaultValue":                                                                                                                              "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testExplicitValue":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testFetchModeOnSecondaryTable":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testManyToOne":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testMappedSuperclassAndSecondaryTable":                                                                                                         "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testReferenceColumnWithBacktics":                                                                                                               "unknown",
	"org.hibernate.test.annotations.join.JoinTest.testUniqueConstaintOnSecondaryTable":                                                                                                           "unknown",
	"org.hibernate.test.annotations.join.namingstrategy.NamingStrategyJoinTest.testJoinToSecondaryTable":                                                                                         "unknown",
	"org.hibernate.test.annotations.loader.LoaderTest.testBasic":                                                                                                                                 "unknown",
	"org.hibernate.test.annotations.loader.LoaderTest.testGetNotExisting":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.lob.LobTest.testBinary":                                                                                                                                      "unknown",
	"org.hibernate.test.annotations.lob.LobTest.testBlob":                                                                                                                                        "26725",
	"org.hibernate.test.annotations.lob.LobTest.testClob":                                                                                                                                        "26725",
	"org.hibernate.test.annotations.lob.LobTest.testSerializableToBlob":                                                                                                                          "26725",
	"org.hibernate.test.annotations.lob.MaterializedBlobTest.testSaving":                                                                                                                         "26725",
	"org.hibernate.test.annotations.lob.SerializableToBlobTypeTest.testPersist":                                                                                                                  "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testBinary":                                                                                                                             "unknown",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testBlob":                                                                                                                               "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testClob":                                                                                                                               "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testSerializableToBlob":                                                                                                                 "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedByteArray":                                                                                                          "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedCharArray":                                                                                                          "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveByteArray":                                                                                                 "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedPrimitiveCharArray":                                                                                                 "26725",
	"org.hibernate.test.annotations.lob.VersionedLobTest.testVersionUnchangedString":                                                                                                             "26725",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testAssociationTableAndOrderBy":                                                                                      "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testAssociationTableAndOrderByWithSet":                                                                               "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testBasic":                                                                                                           "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testCanUseCriteriaQuery":                                                                                             "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testCompositePk":                                                                                                     "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testDefault":                                                                                                         "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testDefaultCompositePk":                                                                                              "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testJoinedSubclassManyToMany":                                                                                        "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testJoinedSubclassManyToManyWithNonPkReference":                                                                      "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testManyToManyEmbeddableBiDirectionalDotNotationInMappedBy":                                                          "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testMappedBy":                                                                                                        "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testOneToManyEmbeddableBiDirectionalDotNotationInMappedBy":                                                           "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testOrderByContractor":                                                                                               "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testOrderByEmployee":                                                                                                 "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testReferencedColumnNameToSuperclass":                                                                                "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testRemoveInBetween":                                                                                                 "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyMaxFetchDepth0Test.testSelf":                                                                                                            "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testAssociationTableAndOrderBy":                                                                                                    "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testAssociationTableAndOrderByWithSet":                                                                                             "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testBasic":                                                                                                                         "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testCanUseCriteriaQuery":                                                                                                           "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testCompositePk":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testDefault":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testDefaultCompositePk":                                                                                                            "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testJoinedSubclassManyToMany":                                                                                                      "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testJoinedSubclassManyToManyWithNonPkReference":                                                                                    "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testManyToManyEmbeddableBiDirectionalDotNotationInMappedBy":                                                                        "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testMappedBy":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testOneToManyEmbeddableBiDirectionalDotNotationInMappedBy":                                                                         "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testOrderByContractor":                                                                                                             "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testOrderByEmployee":                                                                                                               "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testReferencedColumnNameToSuperclass":                                                                                              "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testRemoveInBetween":                                                                                                               "unknown",
	"org.hibernate.test.annotations.manytomany.ManyToManyTest.testSelf":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneJoinTest.testManyToOneJoinTable":                                                                                                          "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneJoinTest.testOneToOneJoinTable":                                                                                                           "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testCascade":                                                                                                                         "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testCompositeFK":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testCreate":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testDefaultMetadata":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testEager":                                                                                                                           "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testFetch":                                                                                                                           "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testFormulaOnOtherSide":                                                                                                              "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testImplicitCompositeFk":                                                                                                             "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testManyToOneNonPk":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testManyToOneNonPkSecondaryTable":                                                                                                    "unknown",
	"org.hibernate.test.annotations.manytoone.ManyToOneTest.testTwoManyToOneNonPk":                                                                                                               "unknown",
	"org.hibernate.test.annotations.manytoone.referencedcolumnname.ManyToOneReferencedColumnNameTest.testReoverableExceptionInFkOrdering":                                                        "24062",
	"org.hibernate.test.annotations.manytoonewithformula.ManyToOneWithFormulaTest.testManyToOneFromNonPk":                                                                                        "unknown",
	"org.hibernate.test.annotations.manytoonewithformula.ManyToOneWithFormulaTest.testManyToOneFromNonPkToNonPk":                                                                                 "unknown",
	"org.hibernate.test.annotations.manytoonewithformula.ManyToOneWithFormulaTest.testManyToOneFromPk":                                                                                           "unknown",
	"org.hibernate.test.annotations.manytoonewithformula.ManyToOneWithFormulaTest.testManyToOneToPkWithOnlyFormula":                                                                              "unknown",
	"org.hibernate.test.annotations.manytoonewithformula.ManyToOneWithFormulaTest.testReferencedColumnNameBelongsToEmbeddedIdOfReferencedEntity":                                                 "unknown",
	"org.hibernate.test.annotations.mappedsuperclass.intermediate.IntermediateMappedSuperclassTest.testGetOnIntermediateMappedSuperclass":                                                        "unknown",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithAliasOneToOneJoin":                                                                                   "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinCollection":                                                                             "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testCriteriaWithFetchModeJoinOnetoOne":                                                                               "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testNaturalKeyLookupWithConstraint":                                                                                  "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSimpleImmutableNaturalKeyLookup":                                                                                 "6583",
	"org.hibernate.test.annotations.naturalid.ImmutableNaturalKeyLookupTest.testSubCriteriaOneToOneJoin":                                                                                         "6583",
	"org.hibernate.test.annotations.naturalid.NaturalIdOnSingleManyToOneTest.testManyToOneNaturalIdCached":                                                                                       "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdOnSingleManyToOneTest.testMappingProperties":                                                                                              "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdTest.testMappingProperties":                                                                                                               "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdTest.testNaturalIdCached":                                                                                                                 "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdTest.testNaturalIdLoaderCached":                                                                                                           "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdTest.testNaturalIdLoaderNotCached":                                                                                                        "unknown",
	"org.hibernate.test.annotations.naturalid.NaturalIdTest.testNaturalIdUncached":                                                                                                               "unknown",
	"org.hibernate.test.annotations.naturalid.cid.EmbeddedAndNaturalIdTest.testByNaturalId":                                                                                                      "unknown",
	"org.hibernate.test.annotations.naturalid.cid.EmbeddedAndNaturalIdTest.testNaturalIdCriteria":                                                                                                "unknown",
	"org.hibernate.test.annotations.naturalid.cid.EmbeddedAndNaturalIdTest.testSave":                                                                                                             "unknown",
	"org.hibernate.test.annotations.notfound.NotFoundLogicalOneToOneTest.testLogicalOneToOne":                                                                                                    "unknown",
	"org.hibernate.test.annotations.notfound.NotFoundOneToOneNonInsertableNonUpdateableTest.testOneToOne":                                                                                        "unknown",
	"org.hibernate.test.annotations.notfound.NotFoundTest.testManyToOne":                                                                                                                         "unknown",
	"org.hibernate.test.annotations.notfound.OneToOneNotFoundTest.testOneToOne":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testCascadeDelete":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testCascadeDeleteOrphan":                                                                                                             "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testCascadeDeleteWithUnidirectionalAssociation":                                                                                      "26738",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testColumnDefinitionPropagation":                                                                                                     "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testFetching":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testJoinColumns":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testListWithBagSemanticAndOrderBy":                                                                                                   "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testOrderByOnSuperclassProperty":                                                                                                     "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testSimpleOneToManyCollection":                                                                                                       "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testSimpleOneToManySet":                                                                                                              "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testUnidirectionalDefault":                                                                                                           "unknown",
	"org.hibernate.test.annotations.onetomany.OneToManyTest.testUnidirectionalExplicit":                                                                                                          "unknown",
	"org.hibernate.test.annotations.onetomany.OrderByTest.testInverseIndex":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.onetomany.OrderByTest.testInverseIndexCascaded":                                                                                                              "unknown",
	"org.hibernate.test.annotations.onetomany.OrderByTest.testOrderByNoElement":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.onetomany.OrderByTest.testOrderByOnIdClassProperties":                                                                                                        "unknown",
	"org.hibernate.test.annotations.onetomany.OrderByTest.testOrderByOneToManyWithJoinTable":                                                                                                     "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneJoinTableNonOptionalTest.testSavingEntitiesWithANullOneToOneAssociationValue":                                                               "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneJoinTableOptionalTest.testSavingEntitiesWithANullOneToOneAssociationValue":                                                                  "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneMapsIdJoinColumnTest.testLifecycle":                                                                                                         "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneMapsIdTest.testLifecycle":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testBidirectionalFkOneToOne":                                                                                                           "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testBidirectionalTrueOneToOne":                                                                                                         "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testCompositePk":                                                                                                                       "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testDefaultOneToOne":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testEagerFetching":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testForeignGenerator":                                                                                                                  "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testHqlQuery":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testOneToOneWithExplicitFk":                                                                                                            "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testOneToOneWithExplicitSecondaryTableFk":                                                                                              "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testPkOneToOneSelectStatementDoesNotGenerateExtraJoin":                                                                                 "unknown",
	"org.hibernate.test.annotations.onetoone.OneToOneTest.testUnidirectionalTrueOneToOne":                                                                                                        "unknown",
	"org.hibernate.test.annotations.onetoone.OptionalOneToOneMappedByTest.testBidirAssignedId":                                                                                                   "unknown",
	"org.hibernate.test.annotations.onetoone.OptionalOneToOneMappedByTest.testBidirDefaultIdGenerator":                                                                                           "unknown",
	"org.hibernate.test.annotations.onetoone.OptionalOneToOneMappedByTest.testBidirQueryEntityProperty":                                                                                          "unknown",
	"org.hibernate.test.annotations.onetoone.OptionalOneToOnePKJCTest.testNotFoundBidirAssignedId":                                                                                               "unknown",
	"org.hibernate.test.annotations.onetoone.OptionalOneToOnePKJCTest.testNotFoundBidirDefaultIdGenerator":                                                                                       "unknown",
	"org.hibernate.test.annotations.onetoone.hhh4851.HHH4851Test.testHHH4851":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.onetoone.hhh9798.OneToOneJoinTableTest.storeNonUniqueRelationship":                                                                                           "5807",
	"org.hibernate.test.annotations.override.AssociationOverrideTest.testDottedNotation":                                                                                                         "unknown",
	"org.hibernate.test.annotations.override.inheritance.MappedSuperclassAttributeOverrideTest.test":                                                                                             "unknown",
	"org.hibernate.test.annotations.polymorphism.PolymorphismTest.testNonPkInheritedFk":                                                                                                          "unknown",
	"org.hibernate.test.annotations.polymorphism.PolymorphismTest.testPolymorphism":                                                                                                              "unknown",
	"org.hibernate.test.annotations.query.NamedQueryTest.testNamedQueriesOrdinalParametersAreOneBased":                                                                                           "unknown",
	"org.hibernate.test.annotations.query.NamedQueryTest.testNativeNamedQueriesOrdinalParametersAreOneBased":                                                                                     "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testClassQueries":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testCollectionSQLOverriding":                                                                                                           "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testDiscriminator":                                                                                                                     "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testEntitySQLOverriding":                                                                                                               "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testImplicitNativeQuery":                                                                                                               "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testImportQueryFromMappedSuperclass":                                                                                                   "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testNativeQueryAndCompositePKAndComponents":                                                                                            "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testNativeQueryWithFormulaAttribute":                                                                                                   "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testNativeQueryWithNullParameterTyped":                                                                                                 "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testPackageQueries":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testQueryWithNullParameterTyped":                                                                                                       "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testSQLQuery":                                                                                                                          "unknown",
	"org.hibernate.test.annotations.query.QueryAndSQLTest.testSQLQueryWithManyToOne":                                                                                                             "unknown",
	"org.hibernate.test.annotations.quote.resultsetmappings.ExplicitSqlResultSetMappingTest.testCompleteScalarAutoDiscovery":                                                                     "unknown",
	"org.hibernate.test.annotations.quote.resultsetmappings.ExplicitSqlResultSetMappingTest.testPartialScalarAutoDiscovery":                                                                      "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testManyToMany":                                                                                                "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testManyToOne":                                                                                                 "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testManyToOneInsideComponentReferencedColumn":                                                                  "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testManyToOneReferenceManyToOne":                                                                               "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testOneToMany":                                                                                                 "unknown",
	"org.hibernate.test.annotations.referencedcolumnname.ReferencedColumnNameTest.testUnidirectionalOneToMany":                                                                                   "unknown",
	"org.hibernate.test.annotations.reflection.ElementCollectionConverterTest.testConverterIsAppliedToElementCollection":                                                                         "unknown",
	"org.hibernate.test.annotations.selectbeforeupdate.UpdateDetachedTest.testUpdateDetachedChanged":                                                                                             "unknown",
	"org.hibernate.test.annotations.selectbeforeupdate.UpdateDetachedTest.testUpdateDetachedChangedAndUnchanged":                                                                                 "unknown",
	"org.hibernate.test.annotations.selectbeforeupdate.UpdateDetachedTest.testUpdateDetachedUnchanged":                                                                                           "unknown",
	"org.hibernate.test.annotations.selectbeforeupdate.UpdateDetachedTest.testUpdateDetachedUnchangedAndChanged":                                                                                 "unknown",
	"org.hibernate.test.annotations.strategy.StrategyTest.testComponentSafeStrategy":                                                                                                             "unknown",
	"org.hibernate.test.annotations.subselect.SubselectTest.testSubselectWithSynchronize":                                                                                                        "unknown",
	"org.hibernate.test.annotations.tableperclass.TablePerClassTest.testConstraintsOnSuperclassProperties":                                                                                       "unknown",
	"org.hibernate.test.annotations.tableperclass.TablePerClassTest.testUnionSubClass":                                                                                                           "6583",
	"org.hibernate.test.annotations.target.TargetTest.testTargetOnEmbedded":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.target.TargetTest.testTargetOnMapKey":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.target.TargetTest.testTargetOnMapKeyManyToMany":                                                                                                              "unknown",
	"org.hibernate.test.annotations.tuplizer.TuplizerTest.testEntityTuplizer":                                                                                                                    "unknown",
	"org.hibernate.test.annotations.tuplizer.bytebuddysubclass.TuplizerInstantiatesByteBuddySubclassTest.hhh11655Test":                                                                           "unknown",
	"org.hibernate.test.annotations.type.TypeTest.testIdWithMulticolumns":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.type.dynamicparameterized.DynamicParameterizedTypeTest.testParameterValues":                                                                                  "unknown",
	"org.hibernate.test.annotations.uniqueconstraint.UniqueConstraintTest.testUniquenessConstraintWithSuperclassProperty":                                                                        "unknown",
	"org.hibernate.test.annotations.uniqueconstraint.UniqueConstraintThrowsConstraintViolationExceptionTest.testUniqueConstraintWithEmptyColumnName":                                             "unknown",
	"org.hibernate.test.annotations.various.GeneratedTest.testGenerated":                                                                                                                         "unknown",
	"org.hibernate.test.annotations.various.IndexTest.testIndexAndJoined":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.various.IndexTest.testIndexManyToOne":                                                                                                                        "unknown",
	"org.hibernate.test.annotations.various.OptimisticLockAnnotationTest.testOptimisticLockExcludeOnNameProperty":                                                                                "unknown",
	"org.hibernate.test.annotations.xml.ejb3.Ejb3XmlTest.testMapXMLSupport":                                                                                                                      "unknown",
	"org.hibernate.test.annotations.xml.ejb3.Ejb3XmlTest.testXMLEntityHandled":                                                                                                                   "unknown",
	"org.hibernate.test.annotations.xml.ejb3.Ejb3XmlTest.testXmlDefaultOverriding":                                                                                                               "unknown",
	"org.hibernate.test.annotations.xml.ejb3.OrmVersion1SupportedTest.testOrm1Support":                                                                                                           "unknown",
	"org.hibernate.test.annotations.xml.hbm.HbmTest.testManyToMany":                                                                                                                              "unknown",
	"org.hibernate.test.annotations.xml.hbm.HbmTest.testManyToOne":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.xml.hbm.HbmTest.testOneToMany":                                                                                                                               "unknown",
	"org.hibernate.test.annotations.xml.hbm.HbmWithIdentityTest.testManyToOneAndInterface":                                                                                                       "24062",
	"org.hibernate.test.any.AnyTypeTest.testFlushProcessing":                                                                                                                                     "unknown",
	"org.hibernate.test.array.ArrayTest.testArrayJoinFetch":                                                                                                                                      "unknown",
	"org.hibernate.test.batch.BatchOptimisticLockingTest.testBatchAndOptimisticLocking":                                                                                                          "6583",
	"org.hibernate.test.batch.BatchTest.testBatchInsertUpdate":                                                                                                                                   "unknown",
	"org.hibernate.test.batch.BatchTest.testBatchInsertUpdateSizeEqJdbcBatchSize":                                                                                                                "unknown",
	"org.hibernate.test.batch.BatchTest.testBatchInsertUpdateSizeGtJdbcBatchSize":                                                                                                                "unknown",
	"org.hibernate.test.batch.BatchTest.testBatchInsertUpdateSizeLtJdbcBatchSize":                                                                                                                "unknown",
	"org.hibernate.test.batch.BatchingInheritanceDeleteTest.testDelete":                                                                                                                          "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDefaultStyleTest.testMostNotFoundFromQuery":                                                                                           "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDefaultStyleTest.testNotFoundFromGet":                                                                                                 "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDefaultStyleTest.testSeveralNotFoundFromQuery":                                                                                        "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDynamicStyleTest.testMostNotFoundFromQuery":                                                                                           "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDynamicStyleTest.testNotFoundFromGet":                                                                                                 "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnoreDynamicStyleTest.testSeveralNotFoundFromQuery":                                                                                        "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnorePaddedStyleTest.testMostNotFoundFromQuery":                                                                                            "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnorePaddedStyleTest.testNotFoundFromGet":                                                                                                  "unknown",
	"org.hibernate.test.batchfetch.BatchFetchNotFoundIgnorePaddedStyleTest.testSeveralNotFoundFromQuery":                                                                                         "unknown",
	"org.hibernate.test.batchfetch.BatchFetchTest.testBatchFetch":                                                                                                                                "unknown",
	"org.hibernate.test.batchfetch.BatchFetchTest.testBatchFetch2":                                                                                                                               "unknown",
	"org.hibernate.test.batchfetch.BatchingEntityLoaderInitializationWithNoLockModeTest.testJoin":                                                                                                "unknown",
	"org.hibernate.test.batchfetch.DynamicBatchFetchTest.testDynamicBatchFetch":                                                                                                                  "unknown",
	"org.hibernate.test.batchfetch.PaddedBatchFetchTestCase.paddedBatchFetchTest":                                                                                                                "unknown",
	"org.hibernate.test.bidi.AuctionTest2.testLazy":                                                                                                                                              "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkCompositeIdTest.testDeleteFromEngineer":                                                                                                          "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkCompositeIdTest.testDeleteFromPerson":                                                                                                            "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkCompositeIdTest.testUpdate":                                                                                                                      "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkIdTest.testDeleteFromEngineer":                                                                                                                   "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkIdTest.testDeleteFromPerson":                                                                                                                     "unknown",
	"org.hibernate.test.bulkid.CteValuesListBulkIdTest.testUpdate":                                                                                                                               "unknown",
	"org.hibernate.test.bulkid.GlobalQuotedIdentifiersBulkIdTest.testBulkUpdate":                                                                                                                 "unknown",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromEngineer":                                                                                                   "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testDeleteFromPerson":                                                                                                     "5807",
	"org.hibernate.test.bulkid.GlobalTemporaryTableBulkCompositeIdTest.testUpdate":                                                                                                               "5807",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkCompositeIdTest.testDeleteFromEngineer":                                                                                                      "unknown",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkCompositeIdTest.testDeleteFromPerson":                                                                                                        "unknown",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkCompositeIdTest.testUpdate":                                                                                                                  "unknown",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkIdTest.testDeleteFromEngineer":                                                                                                               "unknown",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkIdTest.testDeleteFromPerson":                                                                                                                 "unknown",
	"org.hibernate.test.bulkid.InlineIdsInClauseBulkIdTest.testUpdate":                                                                                                                           "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkCompositeIdTest.testDeleteFromEngineer":                                                                                                      "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkCompositeIdTest.testDeleteFromPerson":                                                                                                        "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkCompositeIdTest.testUpdate":                                                                                                                  "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkIdTest.testDeleteFromEngineer":                                                                                                               "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkIdTest.testDeleteFromPerson":                                                                                                                 "unknown",
	"org.hibernate.test.bulkid.InlineIdsOrClauseBulkIdTest.testUpdate":                                                                                                                           "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkCompositeIdTest.testDeleteFromEngineer":                                                                                           "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkCompositeIdTest.testDeleteFromPerson":                                                                                             "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkCompositeIdTest.testUpdate":                                                                                                       "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkIdTest.testDeleteFromEngineer":                                                                                                    "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkIdTest.testDeleteFromPerson":                                                                                                      "unknown",
	"org.hibernate.test.bulkid.InlineIdsSubSelectValuesListBulkIdTest.testUpdate":                                                                                                                "unknown",
	"org.hibernate.test.bytecode.InvocationTargetExceptionTest.testProxiedInvocationException":                                                                                                   "unknown",
	"org.hibernate.test.bytecode.enhancement.access.MixedAccessTest.test":                                                                                                                        "unknown",
	"org.hibernate.test.bytecode.enhancement.basic.BasicSessionTest.test":                                                                                                                        "unknown",
	"org.hibernate.test.bytecode.enhancement.cascade.CascadeDeleteTest.test":                                                                                                                     "unknown",
	"org.hibernate.test.bytecode.enhancement.cascade.CascadeDetachedTest.test":                                                                                                                   "unknown",
	"org.hibernate.test.bytecode.enhancement.cascade.CascadeWithFkConstraintTest.test":                                                                                                           "unknown",
	"org.hibernate.test.bytecode.enhancement.detached.DetachedGetIdentifierTest.test":                                                                                                            "unknown",
	"org.hibernate.test.bytecode.enhancement.dirty.DirtyTrackingCollectionTest.test":                                                                                                             "unknown",
	"org.hibernate.test.bytecode.enhancement.dirty.DirtyTrackingNonUpdateableTest.test":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.eviction.EvictionTest.test":                                                                                                                         "unknown",
	"org.hibernate.test.bytecode.enhancement.join.HHH3949Test.test1":                                                                                                                             "unknown",
	"org.hibernate.test.bytecode.enhancement.join.HHH3949Test.test2":                                                                                                                             "unknown",
	"org.hibernate.test.bytecode.enhancement.join.HHH3949Test.test3":                                                                                                                             "unknown",
	"org.hibernate.test.bytecode.enhancement.join.HHH3949Test.test4":                                                                                                                             "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.HHH_10708.UnexpectedDeleteTest1.test":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.HHH_10708.UnexpectedDeleteTest2.test":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.HHH_10708.UnexpectedDeleteTest3.test":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyBasicFieldMergeTest.test":                                                                                                                  "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyBasicFieldNotInitializedTest.test":                                                                                                         "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionDeletedTest.test":                                                                                                                "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionDetachTest.testDetach":                                                                                                           "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionDetachTest.testDetachProxy":                                                                                                      "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionDetachTest.testRefresh":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionLoadingTest.testNoTransaction":                                                                                                   "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyCollectionLoadingTest.testTransaction":                                                                                                     "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyInitializationWithoutInlineDirtyTrackingTest.test":                                                                                         "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyLoadingByEnhancerSetterTest.testField":                                                                                                     "26725",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyLoadingIntegrationTest.test":                                                                                                               "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyLoadingTest.test":                                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.LazyProxyOnEnhancedEntityTest.test":                                                                                                            "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.basic.LazyBasicFieldAccessTest.test":                                                                                                           "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.basic.LazyBasicPropertyAccessTest.execute":                                                                                                     "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.cache.LazyInCacheTest.test":                                                                                                                    "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.cache.UninitializedAssociationsInCacheTest.attributeLoadingFromCache":                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.cache.UninitializedLazyBasicCacheTest.test":                                                                                                    "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.group.LazyGroupMappedByTest.test":                                                                                                              "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.group.LazyGroupTest.testAccess":                                                                                                                "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.group.LazyGroupTest.testUpdate":                                                                                                                "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.group.SimpleLazyGroupUpdateTest.test":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.notfound.LazyNotFoundManyToOneNonUpdatableNonInsertableTest.test":                                                                              "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.notfound.LazyNotFoundOneToOneNonUpdatableNonInsertableTest.test":                                                                               "unknown",
	"org.hibernate.test.bytecode.enhancement.lazy.notfound.LazyNotFoundOneToOneTest.test":                                                                                                        "unknown",
	"org.hibernate.test.bytecode.enhancement.lazyCache.InitFromCacheTest.execute":                                                                                                                "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.CompositeMergeTest.test":                                                                                                                      "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeDetachedCascadedCollectionInEmbeddableTest.testMergeDetached":                                                                            "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeDetachedNonCascadedCollectionInEmbeddableTest.testMergeDetached":                                                                         "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeEnhancedDetachedOrphanRemovalTest.testMergeDetachedNonEmptyCollection":                                                                   "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeEnhancedDetachedOrphanRemovalTest.testMergeDetachedOrphanRemoval":                                                                        "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeEnhancedEntityTest.testMerge":                                                                                                            "unknown",
	"org.hibernate.test.bytecode.enhancement.merge.MergeEnhancedEntityTest.testRefresh":                                                                                                          "unknown",
	"org.hibernate.test.bytecode.enhancement.ondemandload.OnDemandLoadTest.testClearedSession":                                                                                                   "unknown",
	"org.hibernate.test.bytecode.enhancement.ondemandload.OnDemandLoadTest.testClosedSession":                                                                                                    "unknown",
	"org.hibernate.test.bytecode.enhancement.otherentityentrycontext.OtherEntityEntryContextTest.test":                                                                                           "unknown",
	"org.hibernate.test.bytecode.enhancement.pk.EmbeddedPKTest.test":                                                                                                                             "unknown",
	"org.hibernate.test.cache.ByteCodeEnhancedImmutableReferenceCacheTest.testAssociatedWithMultiplePersistenceContexts":                                                                         "unknown",
	"org.hibernate.test.cache.ByteCodeEnhancedImmutableReferenceCacheTest.testUseOfDirectReferencesInCache":                                                                                      "unknown",
	"org.hibernate.test.cache.CacheAnnotationTests.testCacheReadConcurrencyStrategyNone":                                                                                                         "unknown",
	"org.hibernate.test.cache.CacheAnnotationTests.testCacheWriteConcurrencyStrategyNone":                                                                                                        "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCachedValueAfterEviction":                                                                                                          "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionInsert":                                                                                                     "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionInsertWithEntityOutOfContext":                                                                               "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionRemove":                                                                                                     "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionRemoveWithEntityOutOfContext":                                                                               "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionUpdate":                                                                                                     "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionUpdateWhenChildIsSetToNull":                                                                                 "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testCollectionCacheEvictionUpdateWithEntityOutOfContext":                                                                               "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionTest.testUpdateWithNullRelation":                                                                                                            "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionWithoutMappedByTest.testCollectionCacheEvictionInsert":                                                                                      "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionWithoutMappedByTest.testCollectionCacheEvictionRemove":                                                                                      "unknown",
	"org.hibernate.test.cache.CollectionCacheEvictionWithoutMappedByTest.testCollectionCacheEvictionUpdate":                                                                                      "unknown",
	"org.hibernate.test.cache.EntityUpdateCacheModeIgnoreTest.testCachModeIgnore":                                                                                                                "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsert":                                                                                                                                       "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertThenUpdate":                                                                                                                             "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertThenUpdateThenRollback":                                                                                                                 "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertWithClear":                                                                                                                              "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertWithClearThenRollback":                                                                                                                  "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertWithRefresh":                                                                                                                            "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertWithRefreshThenRollback":                                                                                                                "unknown",
	"org.hibernate.test.cache.InsertedDataTest.testInsertWithRollback":                                                                                                                           "unknown",
	"org.hibernate.test.cache.LockModeTest.testDefaultLockModeOnCollectionInitialization":                                                                                                        "unknown",
	"org.hibernate.test.cache.LockModeTest.testDefaultLockModeOnEntityLoad":                                                                                                                      "unknown",
	"org.hibernate.test.cache.LockModeTest.testReadLockModeOnEntityLoad":                                                                                                                         "unknown",
	"org.hibernate.test.cache.ManyToManyCollectionCacheEvictionTest.testManyToManyPersist":                                                                                                       "unknown",
	"org.hibernate.test.cache.ReferenceCacheTest.testUseOfDirectReferencesInCache":                                                                                                               "unknown",
	"org.hibernate.test.cache.RegionNameTest.testLegacyStatsSpi":                                                                                                                                 "unknown",
	"org.hibernate.test.cache.SharedRegionTest.test":                                                                                                                                             "unknown",
	"org.hibernate.test.cache.polymorphism.PolymorphicCacheTest.testPolymorphismAndCache":                                                                                                        "unknown",
	"org.hibernate.test.cascade.BidirectionalOneToManyCascadeTest.testSaveChildWithParent":                                                                                                       "unknown",
	"org.hibernate.test.cascade.BidirectionalOneToManyCascadeTest.testSaveOrphanDeleteChildWithParent":                                                                                           "unknown",
	"org.hibernate.test.cascade.BidirectionalOneToManyCascadeTest.testSaveParentWithChild":                                                                                                       "unknown",
	"org.hibernate.test.cascade.BidirectionalOneToManyCascadeTest.testSaveParentWithOrphanDeleteChild":                                                                                           "unknown",
	"org.hibernate.test.cascade.CascadeTestWithAssignedParentIdTest.testSaveChildWithParent":                                                                                                     "unknown",
	"org.hibernate.test.cascade.MergeTest.testMergeDetachedEntityWithNewOneToManyElements":                                                                                                       "unknown",
	"org.hibernate.test.cascade.MergeTest.testMergeEntityWithNewOneToManyElements":                                                                                                               "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathGetAndModify":                                                                                                                  "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathMergeModifiedDetached":                                                                                                         "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathMergeModifiedDetachedIntoProxy":                                                                                                "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathMergeNonCascadedTransientEntityInCollection":                                                                                   "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathMergeNonCascadedTransientEntityInManyToOne":                                                                                    "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathMergeNonCascadedTransientEntityInOneToOne":                                                                                     "unknown",
	"org.hibernate.test.cascade.MultiPathCascadeTest.testMultiPathUpdateModifiedDetached":                                                                                                        "unknown",
	"org.hibernate.test.cascade.RefreshTest.testRefreshCascade":                                                                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.CascadeManagedAndTransientTest.testAttachedChildInMerge":                                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.CascadeMergeToChildBeforeParentTest.testMerge":                                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.CascadeMergeToChildBeforeParentTest.testMergeData3Nodes":                                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.CascadeMergeToChildBeforeParentTest.testMergeTransientChildBeforeTransientParent":                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeData3Nodes":                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeDeliveryNode":                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeEntityWithNonNullableEntityNull":                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeEntityWithNonNullablePropSetToNull":                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeEntityWithNonNullableTransientEntity":                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergePickupNode":                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeRoute":                                                                                     "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeTour":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testMergeTransport":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveData3Nodes":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveDeliveryNode":                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveEntityWithNonNullableEntityNull":                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveEntityWithNonNullablePropSetToNull":                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveEntityWithNonNullableTransientEntity":                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSavePickupNode":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveTour":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveTransport":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateData3Nodes":                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateDeliveryNode":                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdatePickupNode":                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateRoute":                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateTour":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullFalseDelayedInsertTest.testSaveUpdateTransport":                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeData3Nodes":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeDeliveryNode":                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeEntityWithNonNullableEntityNull":                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeEntityWithNonNullablePropSetToNull":                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeEntityWithNonNullableTransientEntity":                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergePickupNode":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeRoute":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeTour":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testMergeTransport":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveData3Nodes":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveDeliveryNode":                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveEntityWithNonNullableEntityNull":                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveEntityWithNonNullablePropSetToNull":                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveEntityWithNonNullableTransientEntity":                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSavePickupNode":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveTour":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveTransport":                                                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateData3Nodes":                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateDeliveryNode":                                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdatePickupNode":                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateRoute":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateTour":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullTrueDelayedInsertTest.testSaveUpdateTransport":                                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeData3Nodes":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeDeliveryNode":                                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeEntityWithNonNullableEntityNull":                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeEntityWithNonNullablePropSetToNull":                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeEntityWithNonNullableTransientEntity":                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergePickupNode":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeRoute":                                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeTour":                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testMergeTransport":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveData3Nodes":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveDeliveryNode":                                                                                     "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveEntityWithNonNullableEntityNull":                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveEntityWithNonNullablePropSetToNull":                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveEntityWithNonNullableTransientEntity":                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSavePickupNode":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveTour":                                                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveTransport":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateData3Nodes":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateDeliveryNode":                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdatePickupNode":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateRoute":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateTour":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityFalseTest.testSaveUpdateTransport":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeData3Nodes":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeDeliveryNode":                                                                                     "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeEntityWithNonNullableEntityNull":                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeEntityWithNonNullablePropSetToNull":                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeEntityWithNonNullableTransientEntity":                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergePickupNode":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeRoute":                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeTour":                                                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testMergeTransport":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveData3Nodes":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveDeliveryNode":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveEntityWithNonNullableEntityNull":                                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveEntityWithNonNullablePropSetToNull":                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveEntityWithNonNullableTransientEntity":                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSavePickupNode":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveTour":                                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveTransport":                                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateData3Nodes":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateDeliveryNode":                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdatePickupNode":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateRoute":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateTour":                                                                                        "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeCheckNullibilityTrueTest.testSaveUpdateTransport":                                                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeData3Nodes":                                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeDeliveryNode":                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeEntityWithNonNullableEntityNull":                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeEntityWithNonNullablePropSetToNull":                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeEntityWithNonNullableTransientEntity":                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergePickupNode":                                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeRoute":                                                                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeTour":                                                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testMergeTransport":                                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveData3Nodes":                                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveDeliveryNode":                                                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveEntityWithNonNullableEntityNull":                                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveEntityWithNonNullablePropSetToNull":                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveEntityWithNonNullableTransientEntity":                                                                     "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSavePickupNode":                                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveTour":                                                                                                     "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveTransport":                                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateData3Nodes":                                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateDeliveryNode":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdatePickupNode":                                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateRoute":                                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateTour":                                                                                               "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeDelayedInsertTest.testSaveUpdateTransport":                                                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeData3Nodes":                                                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeDeliveryNode":                                                                                                         "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeEntityWithNonNullableEntityNull":                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeEntityWithNonNullablePropSetToNull":                                                                                   "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeEntityWithNonNullableTransientEntity":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergePickupNode":                                                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeRoute":                                                                                                                "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeTour":                                                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testMergeTransport":                                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveData3Nodes":                                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveDeliveryNode":                                                                                                          "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveEntityWithNonNullableEntityNull":                                                                                       "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveEntityWithNonNullablePropSetToNull":                                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveEntityWithNonNullableTransientEntity":                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSavePickupNode":                                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveTour":                                                                                                                  "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveTransport":                                                                                                             "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateData3Nodes":                                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateDeliveryNode":                                                                                                    "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateEntityWithNonNullableEntityNull":                                                                                 "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateEntityWithNonNullablePropSetToNull":                                                                              "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateEntityWithNonNullableTransientEntity":                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdatePickupNode":                                                                                                      "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateRoute":                                                                                                           "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateTour":                                                                                                            "unknown",
	"org.hibernate.test.cascade.circle.MultiPathCircleCascadeTest.testSaveUpdateTransport":                                                                                                       "unknown",
	"org.hibernate.test.cdi.converters.delayed.DelayedCdiHostedConverterTest.testIt":                                                                                                             "unknown",
	"org.hibernate.test.cdi.converters.legacy.ConversionAutoApplyTest.testConverterIsNotIncorrectlyApplied":                                                                                      "unknown",
	"org.hibernate.test.cdi.converters.legacy.ConvertBetweenTest.testBetweenLiteral":                                                                                                             "unknown",
	"org.hibernate.test.cdi.converters.legacy.ConvertBetweenTest.testBetweenParameters":                                                                                                          "unknown",
	"org.hibernate.test.cdi.converters.legacy.QueryTest.testCriteriaBetween":                                                                                                                     "unknown",
	"org.hibernate.test.cdi.converters.legacy.QueryTest.testJpqlLiteralBetween":                                                                                                                  "unknown",
	"org.hibernate.test.cdi.converters.legacy.QueryTest.testJpqlParametrizedBetween":                                                                                                             "unknown",
	"org.hibernate.test.cdi.converters.standard.CdiHostedConverterTest.testIt":                                                                                                                   "unknown",
	"org.hibernate.test.cdi.events.delayed.DelayedCdiSupportTest.testIt":                                                                                                                         "unknown",
	"org.hibernate.test.cdi.events.extended.ValidExtendedCdiSupportTest.testIt":                                                                                                                  "unknown",
	"org.hibernate.test.cdi.events.nocdi.ValidNoCdiSupportTest.testIt":                                                                                                                           "unknown",
	"org.hibernate.test.cdi.events.standard.StandardCdiSupportTest.testIt":                                                                                                                       "unknown",
	"org.hibernate.test.cid.CompositeIdTest.testCompositeIds":                                                                                                                                    "unknown",
	"org.hibernate.test.cid.CompositeIdTest.testDistinctCountOfEntityWithCompositeId":                                                                                                            "unknown",
	"org.hibernate.test.cid.CompositeIdTest.testMultipleCollectionFetch":                                                                                                                         "unknown",
	"org.hibernate.test.cid.CompositeIdTest.testNonLazyFetch":                                                                                                                                    "unknown",
	"org.hibernate.test.cid.CompositeIdTest.testQuery":                                                                                                                                           "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testCompositeIdSimple":                                                                                                                  "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testDelete":                                                                                                                             "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testDetachedId":                                                                                                                         "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testDetachedProperty":                                                                                                                   "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testEvict":                                                                                                                              "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testGeneratedIdsWithChildren":                                                                                                           "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testLoad":                                                                                                                               "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testMerge":                                                                                                                              "unknown",
	"org.hibernate.test.cid.CompositeIdWithGeneratorTest.testSaveOrUpdate":                                                                                                                       "unknown",
	"org.hibernate.test.classloader.ProxyInterfaceClassLoaderTest.testProxyClassLoader":                                                                                                          "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDelete":                                                                                              "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteAfterLock":                                                                                     "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteAfterPersist":                                                                                  "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteAfterPersistAndFlush":                                                                          "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteOnDelete":                                                                                      "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteOnMerge":                                                                                       "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteOnSaveOrUpdate":                                                                                "unknown",
	"org.hibernate.test.collection.backref.map.compkey.BackrefCompositeMapKeyTest.testOrphanDeleteOnSaveOrUpdateAfterSerialization":                                                              "unknown",
	"org.hibernate.test.collection.bag.BagDuplicatesTest.HHH10385Test":                                                                                                                           "unknown",
	"org.hibernate.test.collection.bag.BagElementNullBasicTest.addNullValue":                                                                                                                     "unknown",
	"org.hibernate.test.collection.bag.BagElementNullBasicTest.testPersistNullValue":                                                                                                             "unknown",
	"org.hibernate.test.collection.bag.BagElementNullBasicTest.testUpdateNonNullValueToNull":                                                                                                     "unknown",
	"org.hibernate.test.collection.bag.BagElementNullBasicTest.testUpdateNonNullValueToNullWithExtraValue":                                                                                       "unknown",
	"org.hibernate.test.collection.bag.MultipleBagFetchHqlTest.testMultipleBagFetchHql":                                                                                                          "unknown",
	"org.hibernate.test.collection.bag.PersistentBagContainsTest.testContains":                                                                                                                   "unknown",
	"org.hibernate.test.collection.bag.PersistentBagTest.testMergePersistentEntityWithNewOneToManyElements":                                                                                      "unknown",
	"org.hibernate.test.collection.bag.PersistentBagTest.testWriteMethodDirtying":                                                                                                                "unknown",
	"org.hibernate.test.collection.basic.JoinFetchElementCollectionTest.testJoinFetchElementCollection":                                                                                          "unknown",
	"org.hibernate.test.collection.custom.basic.UserCollectionTypeAnnotationsVariantTest.testBasicOperation":                                                                                     "unknown",
	"org.hibernate.test.collection.custom.basic.UserCollectionTypeHbmVariantTest.testBasicOperation":                                                                                             "unknown",
	"org.hibernate.test.collection.custom.parameterized.ParameterizedUserCollectionTypeAnnotationsVariantTest.testBasicOperation":                                                                "unknown",
	"org.hibernate.test.collection.custom.parameterized.ParameterizedUserCollectionTypeHbmVariantTest.testBasicOperation":                                                                        "unknown",
	"org.hibernate.test.collection.delayedOperation.BagDelayedOperationTest.testSimpleAddDetached":                                                                                               "unknown",
	"org.hibernate.test.collection.delayedOperation.BagDelayedOperationTest.testSimpleAddManaged":                                                                                                "unknown",
	"org.hibernate.test.collection.delayedOperation.BagDelayedOperationTest.testSimpleAddTransient":                                                                                              "unknown",
	"org.hibernate.test.collection.delayedOperation.ListAddTest.addQuestionToDetachedQuizz":                                                                                                      "unknown",
	"org.hibernate.test.collection.delayedOperation.ListAddTest.addQuestionWithIndexAndInitializeTheListShouldAddQuestionAtSpecifiedIndex":                                                       "unknown",
	"org.hibernate.test.collection.delayedOperation.ListAddTest.addQuestionWithIndexShouldAddQuestionAtSpecifiedIndex":                                                                           "unknown",
	"org.hibernate.test.collection.delayedOperation.ListAddTest.testAddQuestionAfterSessionIsClosed":                                                                                             "unknown",
	"org.hibernate.test.collection.delayedOperation.ListDelayedOperationTest.testSimpleAddDetached":                                                                                              "unknown",
	"org.hibernate.test.collection.delayedOperation.ListDelayedOperationTest.testSimpleAddManaged":                                                                                               "unknown",
	"org.hibernate.test.collection.delayedOperation.ListDelayedOperationTest.testSimpleAddTransient":                                                                                             "unknown",
	"org.hibernate.test.collection.delayedOperation.ListDelayedOperationTest.testSimpleRemoveDetached":                                                                                           "unknown",
	"org.hibernate.test.collection.delayedOperation.SetDelayedOperationTest.testSimpleAddDetached":                                                                                               "unknown",
	"org.hibernate.test.collection.delayedOperation.SetDelayedOperationTest.testSimpleAddManaged":                                                                                                "unknown",
	"org.hibernate.test.collection.delayedOperation.SetDelayedOperationTest.testSimpleAddTransient":                                                                                              "unknown",
	"org.hibernate.test.collection.delayedOperation.SetDelayedOperationTest.testSimpleRemoveDetached":                                                                                            "unknown",
	"org.hibernate.test.collection.delayedOperation.SetDelayedOperationTest.testSimpleRemoveManaged":                                                                                             "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedCascadeDereferencedCollectionTest.testGetAndNullifyCollection":                                                                        "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedCascadeDereferencedCollectionTest.testGetAndReplaceCollection":                                                                        "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedCascadeDereferencedCollectionTest.testMergeNullCollection":                                                                            "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedCascadeDereferencedCollectionTest.testSaveOrUpdateNullCollection":                                                                     "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedNoCascadeDereferencedCollectionTest.testGetAndNullifyCollection":                                                                      "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedNoCascadeDereferencedCollectionTest.testGetAndReplaceCollection":                                                                      "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedNoCascadeDereferencedCollectionTest.testMergeNullCollection":                                                                          "unknown",
	"org.hibernate.test.collection.dereferenced.UnversionedNoCascadeDereferencedCollectionTest.testSaveOrUpdateNullCollection":                                                                   "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedCascadeDereferencedCollectionTest.testGetAndNullifyCollection":                                                                          "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedCascadeDereferencedCollectionTest.testGetAndReplaceCollection":                                                                          "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedCascadeDereferencedCollectionTest.testMergeNullCollection":                                                                              "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedCascadeDereferencedCollectionTest.testSaveOrUpdateNullCollection":                                                                       "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedNoCascadeDereferencedCollectionTest.testGetAndNullifyCollection":                                                                        "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedNoCascadeDereferencedCollectionTest.testGetAndReplaceCollection":                                                                        "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedNoCascadeDereferencedCollectionTest.testMergeNullCollection":                                                                            "unknown",
	"org.hibernate.test.collection.dereferenced.VersionedNoCascadeDereferencedCollectionTest.testSaveOrUpdateNullCollection":                                                                     "unknown",
	"org.hibernate.test.collection.idbag.IdBagElementNullBasicTest.addNullValue":                                                                                                                 "unknown",
	"org.hibernate.test.collection.idbag.IdBagElementNullBasicTest.testPersistNullValue":                                                                                                         "unknown",
	"org.hibernate.test.collection.idbag.IdBagElementNullBasicTest.testUpdateNonNullValueToNull":                                                                                                 "unknown",
	"org.hibernate.test.collection.idbag.IdBagElementNullBasicTest.testUpdateNonNullValueToNullWithExtraValue":                                                                                   "unknown",
	"org.hibernate.test.collection.idbag.PersistentIdBagTest.testWriteMethodDirtying":                                                                                                            "unknown",
	"org.hibernate.test.collection.lazynocascade.LazyAssociationNoCascadeTest.testNoCascadeCache":                                                                                                "unknown",
	"org.hibernate.test.collection.list.ListElementNullBasicTest.addNullValue":                                                                                                                   "unknown",
	"org.hibernate.test.collection.list.ListElementNullBasicTest.testPersistNullValue":                                                                                                           "unknown",
	"org.hibernate.test.collection.list.ListElementNullBasicTest.testUpdateNonNullValueToNull":                                                                                                   "unknown",
	"org.hibernate.test.collection.list.ListElementNullBasicTest.testUpdateNonNullValueToNullToNonNull":                                                                                          "unknown",
	"org.hibernate.test.collection.list.ListIndexReferenceFromListElementTest.testIt":                                                                                                            "unknown",
	"org.hibernate.test.collection.list.PersistentListTest.testInverseListIndex":                                                                                                                 "unknown",
	"org.hibernate.test.collection.list.PersistentListTest.testInverseListIndex2":                                                                                                                "unknown",
	"org.hibernate.test.collection.list.PersistentListTest.testWriteMethodDirtying":                                                                                                              "unknown",
	"org.hibernate.test.collection.map.MapElementNullBasicTest.addNullValue":                                                                                                                     "unknown",
	"org.hibernate.test.collection.map.MapElementNullBasicTest.testPersistNullValue":                                                                                                             "unknown",
	"org.hibernate.test.collection.map.MapElementNullBasicTest.testUpdateNonNullValueToNull":                                                                                                     "unknown",
	"org.hibernate.test.collection.map.MapElementNullBasicTest.testUpdateNonNullValueToNullToNonNull":                                                                                            "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testClearMap":                                                                                                                           "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testJoinFetchElementCollectionWithParentSelect":                                                                                         "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testMapKeyColumnInEmbeddableElement":                                                                                                    "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testMapKeyColumnNonInsertableNonUpdatableBidirOneToMany":                                                                                "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testMapKeyColumnNonInsertableNonUpdatableUnidirOneToMany":                                                                               "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testPutAgainstUninitializedMap":                                                                                                         "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testRemoveAgainstUninitializedMap":                                                                                                      "unknown",
	"org.hibernate.test.collection.map.PersistentMapTest.testWriteMethodDirtying":                                                                                                                "unknown",
	"org.hibernate.test.collection.map.hhh7557.EntityMapTest.testInsertIntoMap":                                                                                                                  "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testCopyInitializedCollectionReferenceAfterGet":                                                                    "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testCopyInitializedCollectionReferenceToNewEntityCollectionRoleAfterGet":                                           "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testCopyPersistentCollectionReferenceAfterFlush":                                                                   "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testCopyPersistentCollectionReferenceBeforeFlush":                                                                  "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testCopyUninitializedCollectionReferenceAfterGet":                                                                  "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testDeleteCommitCopyToNewOwnerInNewSession":                                                                        "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testDeleteCommitCopyToNewOwnerNewCollectionRoleInNewSession":                                                       "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testDeleteCopyToNewOwnerInNewSessionBeforeFlush":                                                                   "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testDeleteCopyToNewOwnerNewCollectionRoleInNewSessionBeforeFlush":                                                  "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testSaveOrUpdateOwnerWithCollectionInNewSessionAfterFlush":                                                         "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testSaveOrUpdateOwnerWithCollectionInNewSessionBeforeFlush":                                                        "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testSaveOrUpdateOwnerWithInitializedCollectionInNewSession":                                                        "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionTest.testSaveOrUpdateOwnerWithUninitializedCollectionInNewSession":                                                      "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionWarningTest.testSetCurrentSessionOverwritesNonConnectedSesssion":                                                        "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionWarningTest.testSetCurrentSessionOverwritesNonConnectedSesssionFlushed":                                                 "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionWarningTest.testUnsetSessionCannotOverwriteConnectedSesssion":                                                           "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionWarningTest.testUnsetSessionCannotOverwriteConnectedSesssionFlushed":                                                    "unknown",
	"org.hibernate.test.collection.multisession.MultipleSessionCollectionWarningTest.testUnsetSessionCannotOverwriteNonConnectedSesssion":                                                        "unknown",
	"org.hibernate.test.collection.ordered.joinedInheritence.OrderCollectionOfJoinedHierarchyTest.testOrdering":                                                                                  "unknown",
	"org.hibernate.test.collection.ordered.joinedInheritence.OrderCollectionOfJoinedHierarchyTest.testQuerySyntaxCheck":                                                                          "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testCollectionInheritance":                                                                                                            "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testExtraLazy":                                                                                                                        "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testFetch":                                                                                                                            "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testMerge":                                                                                                                            "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testUpdateOrder":                                                                                                                      "unknown",
	"org.hibernate.test.collection.original.CollectionTest.testValueMap":                                                                                                                         "unknown",
	"org.hibernate.test.collection.set.PersistentSetNonLazyTest.testCollectionMerging":                                                                                                           "unknown",
	"org.hibernate.test.collection.set.PersistentSetNonLazyTest.testCollectiondirtyChecking":                                                                                                     "unknown",
	"org.hibernate.test.collection.set.PersistentSetNonLazyTest.testCompositeElementWriteMethodDirtying":                                                                                         "unknown",
	"org.hibernate.test.collection.set.PersistentSetNonLazyTest.testLoadChildCheckParentContainsChildNoCache":                                                                                    "unknown",
	"org.hibernate.test.collection.set.PersistentSetNonLazyTest.testWriteMethodDirtying":                                                                                                         "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testCollectionMerging":                                                                                                                  "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testCollectiondirtyChecking":                                                                                                            "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testCompositeElementWriteMethodDirtying":                                                                                                "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testLoadChildCheckParentContainsChildCache":                                                                                             "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testLoadChildCheckParentContainsChildNoCache":                                                                                           "unknown",
	"org.hibernate.test.collection.set.PersistentSetTest.testWriteMethodDirtying":                                                                                                                "unknown",
	"org.hibernate.test.collection.set.SetElementNullBasicTest.addNullValue":                                                                                                                     "unknown",
	"org.hibernate.test.collection.set.SetElementNullBasicTest.testPersistNullValue":                                                                                                             "unknown",
	"org.hibernate.test.collection.set.SetElementNullBasicTest.testUpdateNonNullValueToNull":                                                                                                     "unknown",
	"org.hibernate.test.collectionalias.CollectionAliasTest.test":                                                                                                                                "unknown",
	"org.hibernate.test.collectionalias.ReorderedMappingsCollectionAliasTest.test":                                                                                                               "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testComponent":                                                                                                                             "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testComponentFormulaQuery":                                                                                                                 "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testComponentQueries":                                                                                                                      "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testComponentStateChangeAndDirtiness":                                                                                                      "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testCustomColumnReadAndWrite":                                                                                                              "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testMergeComponent":                                                                                                                        "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testNamedQuery":                                                                                                                            "unknown",
	"org.hibernate.test.component.basic.ComponentTest.testUpdateFalse":                                                                                                                           "unknown",
	"org.hibernate.test.component.basic2.ComponentJoinsTest.testComponentJoins":                                                                                                                  "unknown",
	"org.hibernate.test.component.basic2.ComponentJoinsTest.testComponentJoinsHHH7849":                                                                                                           "unknown",
	"org.hibernate.test.component.cascading.collection.CascadeToComponentCollectionTest.testMerging":                                                                                             "unknown",
	"org.hibernate.test.component.cascading.collection.CascadeToComponentCollectionTest.testMergingOriginallyNullComponent":                                                                      "unknown",
	"org.hibernate.test.component.cascading.toone.CascadeToComponentAssociationTest.testMerging":                                                                                                 "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyEagerTest.testGetEntityWithEmptyCollection":                                                                                   "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyEagerTest.testQueryEntityJoinFetchEmptyCollection":                                                                            "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyEagerTest.testQueryEntityLeftJoinFetchEmptyCollection":                                                                        "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyEagerTest.testQueryEntityWithEmptyCollection":                                                                                 "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyLazyTest.testGetEntityWithEmptyCollection":                                                                                    "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyLazyTest.testQueryEntityJoinFetchEmptyCollection":                                                                             "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyLazyTest.testQueryEntityLeftJoinFetchEmptyCollection":                                                                         "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeCollectionKeyLazyTest.testQueryEntityWithEmptyCollection":                                                                                  "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeEquivalentToNullTest.testNoPrimitiveNoParent":                                                                                              "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeEquivalentToNullTest.testParent":                                                                                                           "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyCachedTest.testGetEntityWithNullManyToOne":                                                                                     "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyCachedTest.testQueryEntityAndNullManyToOne":                                                                                    "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyCachedTest.testQueryEntityJoinFetchNullManyToOne":                                                                              "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyCachedTest.testQueryEntityLeftJoinFetchNullManyToOne":                                                                          "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyCachedTest.testQueryEntityWithNullManyToOne":                                                                                   "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyTest.testGetEntityWithNullManyToOne":                                                                                           "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyTest.testQueryEntityAndNullManyToOne":                                                                                          "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyTest.testQueryEntityJoinFetchNullManyToOne":                                                                                    "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyTest.testQueryEntityLeftJoinFetchNullManyToOne":                                                                                "unknown",
	"org.hibernate.test.component.empty.EmptyCompositeManyToOneKeyTest.testQueryEntityWithNullManyToOne":                                                                                         "unknown",
	"org.hibernate.test.component.empty.EmptyCompositesDirtynessTest.testCompositesEmpty":                                                                                                        "unknown",
	"org.hibernate.test.component.empty.EmptyInitializedCompositesDirtynessTest.testInitializedCompositesEmpty":                                                                                  "unknown",
	"org.hibernate.test.component.empty.EmptyInitializedCompositesTest.testCompositesEmpty":                                                                                                      "unknown",
	"org.hibernate.test.component.empty.SelectBeforeUpdateEmbeddedTest.testSelectBeforeUpdateUsingEmptyComposites":                                                                               "unknown",
	"org.hibernate.test.component.empty.SelectBeforeUpdateEmbeddedTest.testSelectBeforeUpdateUsingNullComposites":                                                                                "unknown",
	"org.hibernate.test.component.proxy.ComponentBasicProxyTest.testBasicProxyingWithProtectedMethodCalledInConstructor":                                                                         "unknown",
	"org.hibernate.test.compositeelement.CompositeElementTest.testCustomColumnReadAndWrite":                                                                                                      "unknown",
	"org.hibernate.test.compositeelement.CompositeElementTest.testHandSQL":                                                                                                                       "unknown",
	"org.hibernate.test.compositeusertype.CompositeUserTypeTest.testRemovalWithNullableFields":                                                                                                   "unknown",
	"org.hibernate.test.compositeusertype.nested.NestedCompositeUserTypeTest.testIt":                                                                                                             "unknown",
	"org.hibernate.test.converter.AttributeConverterTest.testBasicTimestampUsage":                                                                                                                "unknown",
	"org.hibernate.test.converter.AttributeConverterTest.testBasicUsage":                                                                                                                         "unknown",
	"org.hibernate.test.converter.AttributeConverterTest.testEnumConverter":                                                                                                                      "unknown",
	"org.hibernate.test.converter.BasicJodaTimeConversionTest.testSimpleConvertUsage":                                                                                                            "unknown",
	"org.hibernate.test.converter.ConverterTest.testJPQLUpperAttributeValueBindParameterType":                                                                                                    "unknown",
	"org.hibernate.test.converter.ConverterTest.testJPQLUpperDbValueBindParameter":                                                                                                               "unknown",
	"org.hibernate.test.converter.DirtyCheckingTest.dirtyCheckAgainstMutatedNameInstance":                                                                                                        "unknown",
	"org.hibernate.test.converter.DirtyCheckingTest.dirtyCheckAgainstNewNameInstance":                                                                                                            "unknown",
	"org.hibernate.test.converter.DirtyCheckingTest.dirtyCheckAgainstNewNumberInstance":                                                                                                          "unknown",
	"org.hibernate.test.converter.ExplicitDateConvertersTest.testSimpleConvertUsage":                                                                                                             "unknown",
	"org.hibernate.test.converter.ExplicitEnumConvertersTest.testSimpleConvertUsage":                                                                                                             "unknown",
	"org.hibernate.test.converter.ExplicitJavaTypeDescriptorTest.testIt":                                                                                                                         "unknown",
	"org.hibernate.test.converter.ExplicitlyNamedConverterClassesTest.testNonAutoAppliedConvertIsNotApplied":                                                                                     "unknown",
	"org.hibernate.test.converter.LongToDateConversionTest.testSetParameter":                                                                                                                     "unknown",
	"org.hibernate.test.converter.NullHandlingTests.testNullReplacementOnBinding":                                                                                                                "unknown",
	"org.hibernate.test.converter.NullHandlingTests.testNullReplacementOnExtraction":                                                                                                             "unknown",
	"org.hibernate.test.converter.QueryTest.testJpqlBooleanLiteral":                                                                                                                              "unknown",
	"org.hibernate.test.converter.QueryTest.testJpqlFloatLiteral":                                                                                                                                "unknown",
	"org.hibernate.test.converter.SimpleConvertAnnotationTest.testSimpleConvertUsage":                                                                                                            "unknown",
	"org.hibernate.test.converter.SimpleConvertsAnnotationTest.testSimpleConvertsUsage":                                                                                                          "unknown",
	"org.hibernate.test.converter.XmlWithExplicitConvertAnnotationsTest.testSimpleConvertUsage":                                                                                                  "unknown",
	"org.hibernate.test.converter.caching.BasicStructuredCachingOfConvertedValueTest.basicCacheStructureTest":                                                                                    "unknown",
	"org.hibernate.test.converter.caching.BasicUnstructuredCachingOfConvertedValueTest.basicCacheStructureTest":                                                                                  "unknown",
	"org.hibernate.test.converter.elementCollection.CollectionCompositeElementConversionTest.testElementCollectionConversion":                                                                    "unknown",
	"org.hibernate.test.converter.elementCollection.CollectionElementConversionTest.testElementCollectionConversion":                                                                             "unknown",
	"org.hibernate.test.converter.elementCollection.CollectionElementExplicitConversionTest.testElementCollectionConversion":                                                                     "unknown",
	"org.hibernate.test.converter.elementCollection.ElementCollectionTests.testSimpleConvertUsage":                                                                                               "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testEnumOrdinal":                                                                                                                      "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testEnumString":                                                                                                                       "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testIntegerWrapper":                                                                                                                   "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testIntegerWrapperThrowsException":                                                                                                    "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testNumberImplicit":                                                                                                                   "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testNumberImplicitOverrided":                                                                                                          "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testSameTypeConverter":                                                                                                                "unknown",
	"org.hibernate.test.converter.literal.QueryLiteralTest.testStringWrapper":                                                                                                                    "unknown",
	"org.hibernate.test.converter.lob.ConverterAndLobTest.basicTest":                                                                                                                             "26725",
	"org.hibernate.test.converter.map.MapElementConversionTest.testElementCollectionConversion":                                                                                                  "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumDefaultType":                                                                                                          "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumExplicitOrdinalType":                                                                                                  "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumExplicitStringType":                                                                                                   "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumExplicitType":                                                                                                         "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumImplicitOverrideOrdinalType":                                                                                          "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumImplicitOverrideStringType":                                                                                           "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumImplicitOverridedType":                                                                                                "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testEnumImplicitType":                                                                                                         "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testExplicitType":                                                                                                             "unknown",
	"org.hibernate.test.converter.map.MapKeyAttributeConverterTest.testImplicitType":                                                                                                             "unknown",
	"org.hibernate.test.converter.map.MapKeyConversionTest.testElementCollectionConversion":                                                                                                      "unknown",
	"org.hibernate.test.criteria.ComplexJoinAliasTest.testCriteriaThroughCompositeId":                                                                                                            "unknown",
	"org.hibernate.test.criteria.CriteriaAliasFetchTest.testFetchWithAlias":                                                                                                                      "unknown",
	"org.hibernate.test.criteria.CriteriaAliasFetchTest.testFixForHHH7842DoesNotBreakOldBehavior":                                                                                                "unknown",
	"org.hibernate.test.criteria.CriteriaLockingTest.testSetLockModeDifferentFromNONELogAWarnMessageWhenTheDialectUseFollowOnLockingIsTrue":                                                      "unknown",
	"org.hibernate.test.criteria.CriteriaLockingTest.testSetLockModeNONEDoNotLogAWarnMessageWhenTheDialectUseFollowOnLockingIsTrue":                                                              "unknown",
	"org.hibernate.test.criteria.CriteriaOrderByTest.testCriteriaOrderBy":                                                                                                                        "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testAliasJoinCriterion":                                                                                                                       "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testClassProperty":                                                                                                                            "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testClassProperty2":                                                                                                                           "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testCriteriaCollectionOfComponent":                                                                                                            "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testCriteriaCollectionOfValue":                                                                                                                "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testCriteriaFetchMode":                                                                                                                        "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testDetachedCriteria":                                                                                                                         "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testDetachedCriteriaTimeout":                                                                                                                  "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testDistinctProjectionsOfComponents":                                                                                                          "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testEscapeCharacter":                                                                                                                          "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testGroupByComponent":                                                                                                                         "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testMultiplePropertiesSubquery":                                                                                                               "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testNestedCorrelatedSubquery":                                                                                                                 "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testNotNot":                                                                                                                                   "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testNullCriteria":                                                                                                                             "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedComponent":                                                                                                                       "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedCompositeId":                                                                                                                     "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedCompositeIdWithAlias":                                                                                                            "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedEmbeddedCompositeId":                                                                                                             "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedId":                                                                                                                              "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedListIncludesComponent":                                                                                                           "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectedListIncludesEmbeddedCompositeId":                                                                                                 "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectionCache":                                                                                                                          "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjections":                                                                                                                              "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testProjectionsUsingProperty":                                                                                                                 "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testRestrictionOnSubclassCollection":                                                                                                          "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testScrollCriteria":                                                                                                                           "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testSubcriteriaJoinTypes":                                                                                                                     "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testSubselect":                                                                                                                                "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testSubselectWithComponent":                                                                                                                   "unknown",
	"org.hibernate.test.criteria.CriteriaQueryTest.testVarargJunctionSyntax":                                                                                                                     "unknown",
	"org.hibernate.test.criteria.InTest.testIn":                                                                                                                                                  "unknown",
	"org.hibernate.test.criteria.LikeTest.testLike":                                                                                                                                              "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testAliasWithNonNullRestrictions":                                                                                                         "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testAliasWithNullOrNonNullRestrictions":                                                                                                   "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testMultipleRootCriteriaRestrictionsOnCollections":                                                                                        "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testMultipleSubCriteriaRestrictionsOnCollections":                                                                                         "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testNonNullSubcriteriaRestrictionsOnRootCriteria":                                                                                         "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testRootAndSubCriteriaRestrictionsOnCollections":                                                                                          "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubCriteriaRestrictionsOnCollectionsNestedInManyToOne":                                                                                "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubcriteriaWithClauseAliasToEntityMap":                                                                                                "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubcriteriaWithNonNullRestrictions":                                                                                                   "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubcriteriaWithNonNullRestrictionsAliasToEntityMap":                                                                                   "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubcriteriaWithNullOrNonNullRestrictions":                                                                                             "unknown",
	"org.hibernate.test.criteria.OuterJoinCriteriaTest.testSubcriteriaWithNullOrNonNullRestrictionsAliasToEntityMap":                                                                             "unknown",
	"org.hibernate.test.criteria.SessionCreateQueryFromCriteriaTest.testScrollMethod":                                                                                                            "unknown",
	"org.hibernate.test.criteria.SessionCreateQueryFromCriteriaTest.testStreamMethod":                                                                                                            "unknown",
	"org.hibernate.test.criteria.SessionCreateQueryFromCriteriaTest.testUniqueResult":                                                                                                            "unknown",
	"org.hibernate.test.criteria.limitexpression.LimitExpressionTest.testAnEmptyListIsReturnedWhenSetMaxResultsToZero":                                                                           "unknown",
	"org.hibernate.test.criteria.limitexpression.LimitExpressionTest.testWithFetchJoin":                                                                                                          "unknown",
	"org.hibernate.test.criteria.many_to_many.CriteriaManyToManyTest.testJoinTable":                                                                                                              "unknown",
	"org.hibernate.test.criteria.many_to_many.CriteriaManyToManyTest.testMappedBy":                                                                                                               "unknown",
	"org.hibernate.test.criteria.mapsid.MapsIdOneToOneSelectTest.findByParentId":                                                                                                                 "unknown",
	"org.hibernate.test.criteria.mapsid.MapsIdOneToOneSelectTest.selectByParent":                                                                                                                 "unknown",
	"org.hibernate.test.cuk.CompositePropertyRefTest.testOneToOnePropertyRef":                                                                                                                    "unknown",
	"org.hibernate.test.cut.CompositeUserTypeTest.testCompositeUserType":                                                                                                                         "unknown",
	"org.hibernate.test.cut.CompositeUserTypeTest.testCustomColumnReadAndWrite":                                                                                                                  "unknown",
	"org.hibernate.test.cut.CompositeUserTypeTest.testNotEqualOperator":                                                                                                                          "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testCascadeAllDeleteOrphanFromClearedPersistentAssnToTransientEntity":                                                          "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testCascadeAllFromClearedPersistentAssnToTransientEntity":                                                                      "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testTransientEntityDeleteCascadingToCircularity":                                                                               "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testTransientEntityDeletionCascadingToDetachedAssociation":                                                                     "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testTransientEntityDeletionCascadingToPersistentAssociation":                                                                   "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testTransientEntityDeletionCascadingToTransientAssociation":                                                                    "unknown",
	"org.hibernate.test.deletetransient.DeleteTransientEntityTest.testTransientEntityDeletionNoCascades":                                                                                         "unknown",
	"org.hibernate.test.dirtiness.CustomDirtinessStrategyTest.testCustomStrategyWithFlushInterceptor":                                                                                            "unknown",
	"org.hibernate.test.dirtiness.CustomDirtinessStrategyTest.testOnlyCustomStrategy":                                                                                                            "unknown",
	"org.hibernate.test.dirtiness.CustomDirtinessStrategyTest.testOnlyCustomStrategyConsultedOnNonDirty":                                                                                         "unknown",
	"org.hibernate.test.discriminator.DiscriminatorTest.testAccessAsIncorrectSubclass":                                                                                                           "unknown",
	"org.hibernate.test.discriminator.DiscriminatorTest.testDiscriminatorSubclass":                                                                                                               "unknown",
	"org.hibernate.test.discriminator.DiscriminatorTest.testLoadSuperclassProxyEvictPolymorphicAccess":                                                                                           "unknown",
	"org.hibernate.test.discriminator.DiscriminatorTest.testLoadSuperclassProxyPolymorphicAccess":                                                                                                "unknown",
	"org.hibernate.test.discriminator.DiscriminatorTest.testQuerySubclassAttribute":                                                                                                              "unknown",
	"org.hibernate.test.discriminator.SimpleInheritanceTest.testAccessAsIncorrectSubclass":                                                                                                       "unknown",
	"org.hibernate.test.discriminator.SimpleInheritanceTest.testDiscriminatorSubclass":                                                                                                           "unknown",
	"org.hibernate.test.discriminator.SimpleInheritanceTest.testLoadSuperclassProxyEvictPolymorphicAccess":                                                                                       "unknown",
	"org.hibernate.test.discriminator.SimpleInheritanceTest.testLoadSuperclassProxyPolymorphicAccess":                                                                                            "unknown",
	"org.hibernate.test.discriminator.SimpleInheritanceTest.testQuerySubclassAttribute":                                                                                                          "unknown",
	"org.hibernate.test.discriminator.joined.JoinedSubclassInheritanceTest.testConfiguredDiscriminatorValue":                                                                                     "unknown",
	"org.hibernate.test.distinct.SelectDistinctHqlTest.test":                                                                                                                                     "unknown",
	"org.hibernate.test.distinct.SelectDistinctHqlTest.testDistinctPassThroughFalse":                                                                                                             "unknown",
	"org.hibernate.test.distinct.SelectDistinctHqlTest.testDistinctPassThroughTrue":                                                                                                              "unknown",
	"org.hibernate.test.dynamicentity.interceptor.InterceptorDynamicEntityTest.testIt":                                                                                                           "unknown",
	"org.hibernate.test.dynamicentity.tuplizer.TuplizerDynamicEntityTest.testIt":                                                                                                                 "unknown",
	"org.hibernate.test.dynamicentity.tuplizer2.ImprovedTuplizerDynamicEntityTest.testIt":                                                                                                        "unknown",
	"org.hibernate.test.dynamicmap.DynamicMapTest.bootstrappingTest":                                                                                                                             "unknown",
	"org.hibernate.test.ecid.EmbeddedCompositeIdTest.testMerge":                                                                                                                                  "unknown",
	"org.hibernate.test.ecid.EmbeddedCompositeIdTest.testMerging":                                                                                                                                "unknown",
	"org.hibernate.test.ecid.EmbeddedCompositeIdTest.testPolymorphism":                                                                                                                           "unknown",
	"org.hibernate.test.entitymode.map.basic.DynamicClassTest.testLazyDynamicClass":                                                                                                              "unknown",
	"org.hibernate.test.entitymode.map.subclass.SubclassDynamicMapTest.testConcreateSubclassDeterminationOnEmptyDynamicMap":                                                                      "unknown",
	"org.hibernate.test.entityname.EntityNameFromSubClassTest.testEntityName":                                                                                                                    "unknown",
	"org.hibernate.test.enums.EnumExplicitTypeTest.hbmEnumWithExplicitTypeTest":                                                                                                                  "unknown",
	"org.hibernate.test.enums.EnumTypeTest.hbmEnumTypeTest":                                                                                                                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testDeleteParentAndChild":                                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testDeleteParentButNotChild":                                    "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testDeleteParentWithNoChildren":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testDeleteParentWithNullChildren":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testMoveAllChildrenToDifferentParent":                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testMoveChildToDifferentParent":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testMoveCollectionToDifferentParent":                            "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testSaveParentEmptyChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testSaveParentOneChild":                                         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentNoneToOneChild":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentNullToOneChild":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneChildToNoneByClear":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneToTwoChildren":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentOneToTwoSameChildren":                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManyBagToSetCollectionEventTest.testUpdateParentTwoChildrenToOne":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testDeleteParentAndChild":                                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testDeleteParentButNotChild":                                    "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testDeleteParentWithNoChildren":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testDeleteParentWithNullChildren":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testMoveAllChildrenToDifferentParent":                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testMoveChildToDifferentParent":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testMoveCollectionToDifferentParent":                            "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testSaveParentEmptyChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testSaveParentOneChild":                                         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentNoneToOneChild":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentNullToOneChild":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneChildToNoneByClear":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneToTwoChildren":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentOneToTwoSameChildren":                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.manytomany.BidirectionalManyToManySetToSetCollectionEventTest.testUpdateParentTwoChildrenToOne":                               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testDeleteParentAndChild":                                              "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testDeleteParentButNotChild":                                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testDeleteParentWithNoChildren":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testDeleteParentWithNullChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testMoveAllChildrenToDifferentParent":                                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testMoveChildToDifferentParent":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testMoveCollectionToDifferentParent":                                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testSaveParentEmptyChildren":                                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testSaveParentOneChild":                                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentNoneToOneChild":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentNullToOneChild":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByClear":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneToTwoChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneToTwoSameChildren":                                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagCollectionEventTest.testUpdateParentTwoChildrenToOne":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testDeleteParentAndChild":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testDeleteParentButNotChild":                                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testDeleteParentWithNoChildren":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testDeleteParentWithNullChildren":                              "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testMoveAllChildrenToDifferentParent":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testMoveChildToDifferentParent":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testMoveCollectionToDifferentParent":                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent": "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testSaveParentEmptyChildren":                                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testSaveParentOneChild":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentNoneToOneChild":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentNullToOneChild":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":               "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneChildToNoneByClear":                         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneToTwoChildren":                              "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentOneToTwoSameChildren":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManyBagSubclassCollectionEventTest.testUpdateParentTwoChildrenToOne":                              "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testDeleteParentAndChild":                                              "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testDeleteParentButNotChild":                                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testDeleteParentWithNoChildren":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testDeleteParentWithNullChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testMoveAllChildrenToDifferentParent":                                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testMoveChildToDifferentParent":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testMoveCollectionToDifferentParent":                                   "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":         "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testSaveParentEmptyChildren":                                           "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testSaveParentOneChild":                                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentNoneToOneChild":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentNullToOneChild":                                        "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                          "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                       "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildToNoneByClear":                                 "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                                "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneToTwoChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentOneToTwoSameChildren":                                  "unknown",
	"org.hibernate.test.event.collection.association.bidirectional.onetomany.BidirectionalOneToManySetCollectionEventTest.testUpdateParentTwoChildrenToOne":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testDeleteParentAndChild":                                          "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testDeleteParentButNotChild":                                       "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testDeleteParentWithNoChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testDeleteParentWithNullChildren":                                  "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testMoveAllChildrenToDifferentParent":                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testMoveChildToDifferentParent":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testMoveCollectionToDifferentParent":                               "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":     "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testSaveParentEmptyChildren":                                       "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testSaveParentOneChild":                                            "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentNoneToOneChild":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentNullToOneChild":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                   "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                   "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByClear":                             "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                            "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneToTwoChildren":                                  "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentOneToTwoSameChildren":                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.manytomany.UnidirectionalManyToManyBagCollectionEventTest.testUpdateParentTwoChildrenToOne":                                  "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testDeleteParentAndChild":                                            "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testDeleteParentButNotChild":                                         "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testDeleteParentWithNoChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testDeleteParentWithNullChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testMoveAllChildrenToDifferentParent":                                "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testMoveChildToDifferentParent":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testMoveCollectionToDifferentParent":                                 "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":       "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testSaveParentEmptyChildren":                                         "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testSaveParentOneChild":                                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentNoneToOneChild":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                        "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentNullToOneChild":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                        "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                     "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                     "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByClear":                               "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneToTwoChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentOneToTwoSameChildren":                                "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManyBagCollectionEventTest.testUpdateParentTwoChildrenToOne":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testDeleteParentAndChild":                                            "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testDeleteParentButNotChild":                                         "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testDeleteParentWithNoChildren":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testDeleteParentWithNullChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testMoveAllChildrenToDifferentParent":                                "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testMoveChildToDifferentParent":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testMoveCollectionToDifferentParent":                                 "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":       "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testSaveParentEmptyChildren":                                         "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testSaveParentOneChild":                                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentNoneToOneChild":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                        "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentNullToOneChild":                                      "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                        "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                     "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                     "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildToNoneByClear":                               "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                              "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneToTwoChildren":                                    "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentOneToTwoSameChildren":                                "unknown",
	"org.hibernate.test.event.collection.association.unidirectional.onetomany.UnidirectionalOneToManySetCollectionEventTest.testUpdateParentTwoChildrenToOne":                                    "unknown",
	"org.hibernate.test.event.collection.detached.BadMergeHandlingTest.testMergeAndHold":                                                                                                         "unknown",
	"org.hibernate.test.event.collection.detached.DetachedMultipleCollectionChangeTest.testMergeMultipleCollectionChangeEvents":                                                                  "unknown",
	"org.hibernate.test.event.collection.detached.MergeCollectionEventTest.testCollectionEventHandlingOnMerge":                                                                                   "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testDeleteParentAndChild":                                                                                           "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testDeleteParentWithNoChildren":                                                                                     "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testDeleteParentWithNullChildren":                                                                                   "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testMoveAllChildrenToDifferentParent":                                                                               "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testMoveChildToDifferentParent":                                                                                     "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testMoveCollectionToDifferentParent":                                                                                "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testMoveCollectionToDifferentParentFlushMoveToDifferentParent":                                                      "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testSaveParentEmptyChildren":                                                                                        "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testSaveParentOneChild":                                                                                             "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentNoneToOneChild":                                                                                     "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentNoneToOneChildDiffCollection":                                                                       "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentNullToOneChild":                                                                                     "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentNullToOneChildDiffCollection":                                                                       "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneChildDiffCollectionDiffChild":                                                                    "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneChildDiffCollectionSameChild":                                                                    "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneChildToNoneByClear":                                                                              "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneChildToNoneByRemove":                                                                             "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneToTwoChildren":                                                                                   "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentOneToTwoSameChildren":                                                                               "unknown",
	"org.hibernate.test.event.collection.values.ValuesBagCollectionEventTest.testUpdateParentTwoChildrenToOne":                                                                                   "unknown",
	"org.hibernate.test.event.entity.MergeListPreAndPostPersistTest.testAllPropertiesCopied":                                                                                                     "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitDeleteListenerRollback":                                                                                                "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitDeleteListenerSuccess":                                                                                                 "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitInsertListenerRollback":                                                                                                "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitInsertListenerSuccess":                                                                                                 "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitUpdateListenerRollback":                                                                                                "unknown",
	"org.hibernate.test.events.LegacyPostCommitListenerTest.testPostCommitUpdateListenerSuccess":                                                                                                 "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitDeleteListenerRollback":                                                                                                      "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitDeleteListenerSuccess":                                                                                                       "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitInsertListenerRollback":                                                                                                      "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitInsertListenerSuccess":                                                                                                       "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitUpdateListenerRollback":                                                                                                      "unknown",
	"org.hibernate.test.events.PostCommitListenerTest.testPostCommitUpdateListenerSuccess":                                                                                                       "unknown",
	"org.hibernate.test.eviction.EvictionTest.testEvictingDetachedEntity":                                                                                                                        "unknown",
	"org.hibernate.test.eviction.EvictionTest.testNormalUsage":                                                                                                                                   "unknown",
	"org.hibernate.test.exception.SQLExceptionConversionTest.testIntegrityViolation":                                                                                                             "unknown",
	"org.hibernate.test.exception.SQLExceptionConversionTest.testNotNullConstraint":                                                                                                              "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetAdd":                                                                                                                 "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetContains":                                                                                                            "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetIsEmpty":                                                                                                             "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetIterator":                                                                                                            "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetRemove":                                                                                                              "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetSize":                                                                                                                "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetToArray":                                                                                                             "unknown",
	"org.hibernate.test.extralazy.ExtraLazyCollectionConsistencyTest.testSetToArrayTyped":                                                                                                        "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testGet":                                                                                                                                         "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testIndexFormulaMap":                                                                                                                             "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testMap":                                                                                                                                         "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testOrphanDelete":                                                                                                                                "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testRemoveClear":                                                                                                                                 "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testSQLQuery":                                                                                                                                    "unknown",
	"org.hibernate.test.extralazy.ExtraLazyTest.testWhereClauseOnBidirectionalCollection":                                                                                                        "unknown",
	"org.hibernate.test.fetchprofiles.CollectionLoadedInTwoPhaseLoadTest.testIfEverythingIsLoaded":                                                                                               "unknown",
	"org.hibernate.test.fetchprofiles.EntityLoadedInTwoPhaseLoadTest.testIfAllRelationsAreInitialized":                                                                                           "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testCriteriaManyToOneFetchProfile":                                                                                               "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testHQL":                                                                                                                         "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testLoadComponentDerefFetchProfile":                                                                                              "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testLoadDeepFetchProfile":                                                                                                        "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testLoadManyToOneFetchProfile":                                                                                                   "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testLoadOneToManyFetchProfile":                                                                                                   "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testNormalCriteria":                                                                                                              "unknown",
	"org.hibernate.test.fetchprofiles.join.JoinFetchProfileTest.testNormalLoading":                                                                                                               "unknown",
	"org.hibernate.test.fetchprofiles.join.selfReferencing.JoinSelfReferentialFetchProfileTest.testEnablingJoinFetchProfileAgainstSelfReferentialAssociation":                                    "unknown",
	"org.hibernate.test.fileimport.SingleLineImportFileTest.testImportFile":                                                                                                                      "unknown",
	"org.hibernate.test.filter.CriteriaQueryWithAppliedFilterTest.testRestrictionsOnComponentTypes":                                                                                              "unknown",
	"org.hibernate.test.filter.CriteriaQueryWithAppliedFilterTest.testSubquery":                                                                                                                  "unknown",
	"org.hibernate.test.filter.CriteriaQueryWithAppliedFilterTest.testSubqueryWithRestrictionsOnComponentTypes":                                                                                  "unknown",
	"org.hibernate.test.filter.CriteriaQueryWithAppliedFilterTest.testSubqueryWithRestrictionsOnComponentTypes2":                                                                                 "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testCombinedClassAndCollectionFiltersEnabled":                                                                                                   "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testCriteriaControl":                                                                                                                            "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testCriteriaQueryFilters":                                                                                                                       "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testCriteriaSubqueryWithFilters":                                                                                                                "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testFilterApplicationOnHqlQueryWithImplicitSubqueryContainingNamedParameter":                                                                    "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testFilterApplicationOnHqlQueryWithImplicitSubqueryContainingPositionalParameter":                                                               "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testFiltersOnMultiTableHqlDelete":                                                                                                               "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testFiltersOnSimpleHqlDelete":                                                                                                                   "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testFiltersWithCustomerReadAndWrite":                                                                                                            "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testGetFilters":                                                                                                                                 "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testHQLSubqueryWithFilters":                                                                                                                     "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testHqlFilters":                                                                                                                                 "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testHqlQueryWithColons":                                                                                                                         "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testInStyleFilterParameter":                                                                                                                     "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyBase":                                                                                                                             "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyBaseThruCriteria":                                                                                                                 "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyFilterOnCriteria":                                                                                                                 "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyFilterOnLoad":                                                                                                                     "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyFilterOnQuery":                                                                                                                    "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testManyToManyOnCollectionLoadAfterHQL":                                                                                                         "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testOneToManyFilters":                                                                                                                           "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testSecondLevelCachedCollectionsFiltering":                                                                                                      "unknown",
	"org.hibernate.test.filter.DynamicFilterTest.testSqlSyntaxOfFiltersWithUnions":                                                                                                               "unknown",
	"org.hibernate.test.filter.FilterDotNameTest.testCollectionFilterNameWithDots":                                                                                                               "unknown",
	"org.hibernate.test.filter.FilterDotNameTest.testCollectionFilterNameWithoutDots":                                                                                                            "unknown",
	"org.hibernate.test.filter.FilterDotNameTest.testEntityFilterNameWithDots":                                                                                                                   "unknown",
	"org.hibernate.test.filter.FilterDotNameTest.testEntityFilterNameWithoutDots":                                                                                                                "unknown",
	"org.hibernate.test.filter.FilterWitSubSelectFetchModeTest.testFiltersAreApplied":                                                                                                            "unknown",
	"org.hibernate.test.filter.ManyToManyWithDynamicFilterTest.testManyToManyCollectionWithActiveFilterOnJoin":                                                                                   "unknown",
	"org.hibernate.test.filter.ManyToManyWithDynamicFilterTest.testManyToManyCollectionWithNoFilterOnJoin":                                                                                       "unknown",
	"org.hibernate.test.filter.OneToManyWithDynamicFilterTest.testForIssue":                                                                                                                      "unknown",
	"org.hibernate.test.filter.hql.BasicFilteredBulkManipulationTest.testBasicFilteredHqlDelete":                                                                                                 "unknown",
	"org.hibernate.test.filter.hql.BasicFilteredBulkManipulationTest.testBasicFilteredHqlUpdate":                                                                                                 "unknown",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteLeaf":                                                                                   "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteNonLeaf":                                                                                "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlDeleteRoot":                                                                                   "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateLeaf":                                                                                   "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateNonLeaf":                                                                                "5807",
	"org.hibernate.test.filter.hql.JoinedFilteredBulkManipulationTest.testFilteredJoinedSubclassHqlUpdateRoot":                                                                                   "5807",
	"org.hibernate.test.flush.NativeCriteriaSyncTest.test":                                                                                                                                       "unknown",
	"org.hibernate.test.flush.NonTransactionalDataAccessTest.testFlushAllowingOutOfTransactionUpdateOperations":                                                                                  "unknown",
	"org.hibernate.test.flush.NonTransactionalDataAccessTest.testFlushDisallowingOutOfTransactionUpdateOperations":                                                                               "unknown",
	"org.hibernate.test.flush.NonTransactionalDataAccessTest.testFlushOutOfTransaction":                                                                                                          "unknown",
	"org.hibernate.test.flush.TestAutoFlushBeforeQueryExecution.testAutoflushIsNotRequiredWithUnrelatedCollectionChange":                                                                         "unknown",
	"org.hibernate.test.flush.TestAutoFlushBeforeQueryExecution.testAutoflushIsRequired":                                                                                                         "unknown",
	"org.hibernate.test.flush.TestClearBatchFetchQueueAfterFlush.testClearBatchFetchQueueAfterFlush":                                                                                             "unknown",
	"org.hibernate.test.flush.TestCollectionInitializingDuringFlush.testInitializationDuringFlush":                                                                                               "unknown",
	"org.hibernate.test.foreignkeys.disabled.DisabledForeignKeyTest.expandedTests":                                                                                                               "unknown",
	"org.hibernate.test.foreignkeys.disabled.InheritanceManyToManyForeignKeyTest.testForeignKeyNameUnicity":                                                                                      "unknown",
	"org.hibernate.test.formulajoin.FormulaJoinTest.testFormulaJoin":                                                                                                                             "unknown",
	"org.hibernate.test.generated.DefaultGeneratedValueIdentityTest.testGenerationWithIdentityInsert":                                                                                            "24062",
	"org.hibernate.test.generated.DefaultGeneratedValueTest.testGeneration":                                                                                                                      "unknown",
	"org.hibernate.test.generated.DefaultGeneratedValueTest.testUpdateTimestampGeneration":                                                                                                       "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testIdentityColumnGeneratedIds":                                                                                         "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testPersistOutsideTransaction":                                                                                          "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testPersistOutsideTransactionCascadedFromManyToOne":                                                                     "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testPersistOutsideTransactionCascadedToInverseCollection":                                                               "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testPersistOutsideTransactionCascadedToManyToOne":                                                                       "unknown",
	"org.hibernate.test.generatedkeys.identity.IdentityGeneratedKeysTest.testPersistOutsideTransactionCascadedToNonInverseCollection":                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderByAggregatedArgAliasRef":                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderByEntityWithFetchJoinedCollection":                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderByNoSelectAliasRef":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderByOnJoinedSubclassPropertyWhoseColumnIsNotInDrivingTable":                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderBySelectAliasRef":                                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderBySelectNewArgAliasRef":                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingOrderByTest.testOrderBySelectNewMapArgAliasRef":                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testAggregatedJoinAlias":                                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testAggregation":                                                                                                                                "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testAliases":                                                                                                                                    "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testAnyMappingReference":                                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testArithmetic":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCachedJoinedAndJoinFetchedManyToOne":                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCachedJoinedAndJoinFetchedOneToMany":                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCast":                                                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCastInSelect":                                                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCoalesce":                                                                                                                                   "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCollectionFetchWithDistinctionAndLimit":                                                                                                     "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCollectionFetching":                                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCollectionJoinsInSubselect":                                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentJoins":                                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentNullnessChecks":                                                                                                                    "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentOrderBy":                                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentParameterBinding":                                                                                                                  "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testComponentQueries":                                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testConcatenation":                                                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testCrazyIdFieldNames":                                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testDecimalLiterals":                                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testDynamicInstantiationQueries":                                                                                                                "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testDynamicInstantiationWithToOneQueries":                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEJBQLFunctions":                                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEntityAndOneToOneReturnedByQuery":                                                                                                           "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEntityFetching":                                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testEntityPropertySelect":                                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExpandListParameter":                                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExplicitEntityCasting":                                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExplicitToOneInnerJoin":                                                                                                                     "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExplicitToOneInnerJoinAndImplicitToOne":                                                                                                     "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExplicitToOneOuterJoin":                                                                                                                     "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExpressionWithParamInFunction":                                                                                                              "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testExtract":                                                                                                                                    "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testFetchInSubqueryFails":                                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testFilterWithCustomColumnReadAndWrite":                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testFromOnly":                                                                                                                                   "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testHavingWithCustomColumnReadAndWrite":                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testIllegalMixedTransformerQueries":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testImplicitJoin":                                                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testImplicitJoinsInDifferentClauses":                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testImplicitPolymorphism":                                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testImplicitSelectEntityAssociationInShallowQuery":                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testIndexParams":                                                                                                                                "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testInitProxy":                                                                                                                                  "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testIntegerLiterals":                                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testInvalidCollectionDereferencesFail":                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testInvalidFetchSemantics":                                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAPositionalParameterList":                                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAQLMapEntryQualifier":                                                                                                                     "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAQLMapKeyQualifier":                                                                                                                       "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAQLMapValueQualifier":                                                                                                                     "3288",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJPAQLQualifiedIdentificationVariablesControl":                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJdkEnumStyleEnumConstant":                                                                                                                   "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJoinFetchCollectionOfValues":                                                                                                                "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJoinFetchedCollectionOfJoinedSubclass":                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJoinedCollectionOfJoinedSubclass":                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJoinedCollectionOfJoinedSubclassProjection":                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testJpaTypeOperator":                                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testLazyAssociationInComponent":                                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testLoadSnapshotWithCustomColumnReadAndWrite":                                                                                                   "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testManyToManyFilter":                                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testMultipleBagFetchesFail":                                                                                                                     "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testMultipleRefsToSameParam":                                                                                                                    "5807",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNakedComponentPropertyRef":                                                                                                                  "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNakedEntityAssociationReference":                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNakedImplicitJoins":                                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNakedMapIndex":                                                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNakedPropertyRef":                                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNestedCollectionFetch":                                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNestedComponentIsNull":                                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testNumericExpressionReturnTypes":                                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testOneToManyFilter":                                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testOrderByExtraParenthesis":                                                                                                                    "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testOrderedWithCustomColumnReadAndWrite":                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testOrdinalParameters":                                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testPaginationWithPolymorphicQuery":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testParameterMixing":                                                                                                                            "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testProjectionQueries":                                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testQueryMetadataRetrievalWithFetching":                                                                                                         "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testResultTransformerEntityQueries":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testResultTransformerScalarQueries":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testRowValueConstructorSyntaxInInList":                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseCase":                                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseCaseWithCountDistinct":                                                                                                          "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseCaseWithSum":                                                                                                                    "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseDistinctImplicitJoinOrderByJoinedProperty":                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseImplicitJoin":                                                                                                                   "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseImplicitJoinOrderByJoinedProperty":                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseImplicitJoinWithIterate":                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectClauseSubselect":                                                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSelectExpressions":                                                                                                                          "26732",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSimpleSelect":                                                                                                                               "pass?",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSimpleSelectWithLimitAndOffset":                                                                                                             "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSpecialClassPropertyReference":                                                                                                              "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSpecialClassPropertyReferenceFQN":                                                                                                           "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testStandardFunctions":                                                                                                                          "31708",
	"org.hibernate.test.hql.ASTParserLoadingTest.testStr":                                                                                                                                        "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSubSelectAsArithmeticOperand":                                                                                                               "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSubclassOrSuperclassPropertyReferenceInJoinedSubclass":                                                                                      "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testSuperclassPropertyReferenceAfterCollectionIndexedAccess":                                                                                    "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testUnaryMinus":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.ASTParserLoadingTest.testWhere":                                                                                                                                      "pass?",
	"org.hibernate.test.hql.BulkManipulationTest.testBooleanHandling":                                                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnDiscriminatorSubclass":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnJoinedSubclass":                                                                                                                     "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteOnMappedJoin":                                                                                                                         "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteRestrictedOnManyToOne":                                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteSyntaxWithCompositeId":                                                                                                                "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassAbstractRoot":                                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassConcreteSubclass":                                                                                                        "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteUnionSubclassLeafSubclass":                                                                                                            "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteWithMetadataWhereFragments":                                                                                                           "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testDeleteWithSubquery":                                                                                                                         "31670",
	"org.hibernate.test.hql.BulkManipulationTest.testIncrementCounterVersion":                                                                                                                    "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testIncrementTimestampVersion":                                                                                                                  "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertAcrossMappedJoinFails":                                                                                                                "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertIntoSuperclassPropertiesFails":                                                                                                        "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithAssignedCompositeId":                                                                                                              "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithGeneratedId":                                                                                                                      "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithGeneratedTimestampVersion":                                                                                                        "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithGeneratedVersionAndId":                                                                                                            "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithManyToOne":                                                                                                                        "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMismatchedTypes":                                                                                                                  "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithMultipleNamedParams":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSelectListUsingJoins":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testInsertWithSubqueriesAndNamedParams":                                                                                                         "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testManyToManyBulkDelete":                                                                                                                       "unknown",
	"org.hibernate.test.hql.BulkManipulationTest.testManyToManyBulkDeleteMultiTable":                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsert":                                                                                                                               "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleInsertWithNamedParam":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testSimpleNativeSQLInsert":                                                                                                                      "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testTempTableGenerationIsolation":                                                                                                               "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnAnimal":                                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnComponent":                                                                                                                          "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnDiscriminatorSubclass":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnMammal":                                                                                                                             "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateOnManyToOne":                                                                                                                          "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnDiscriminatorSubclass":                                                                                                       "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullOnJoinedSubclass":                                                                                                              "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateSetNullUnionSubclass":                                                                                                                 "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithSubquery":                                                                                                                         "5807",
	"org.hibernate.test.hql.BulkManipulationTest.testUpdateWithWhereExistsSubquery":                                                                                                              "5807",
	"org.hibernate.test.hql.CaseStatementTest.testSearchedCaseStatementFixture":                                                                                                                  "unknown",
	"org.hibernate.test.hql.CaseStatementTest.testSearchedCaseStatementWithAllParamResults":                                                                                                      "unknown",
	"org.hibernate.test.hql.CaseStatementTest.testSearchedCaseStatementWithParamResult":                                                                                                          "unknown",
	"org.hibernate.test.hql.CaseStatementTest.testSimpleCaseStatementFixture":                                                                                                                    "unknown",
	"org.hibernate.test.hql.CaseStatementTest.testSimpleCaseStatementWithParamAllResults":                                                                                                        "unknown",
	"org.hibernate.test.hql.CaseStatementTest.testSimpleCaseStatementWithParamResult":                                                                                                            "unknown",
	"org.hibernate.test.hql.CastFunctionTest.testFloatCasting":                                                                                                                                   "unknown",
	"org.hibernate.test.hql.CastFunctionTest.testIntegerCasting":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.CastFunctionTest.testLongCasting":                                                                                                                                    "unknown",
	"org.hibernate.test.hql.CastFunctionTest.testStringCasting":                                                                                                                                  "unknown",
	"org.hibernate.test.hql.CastNullSelectExpressionTest.testSelectCastNull":                                                                                                                     "unknown",
	"org.hibernate.test.hql.CastNullSelectExpressionTest.testSelectNewCastNull":                                                                                                                  "unknown",
	"org.hibernate.test.hql.ClassicTranslatorTest.testQueries":                                                                                                                                   "unknown",
	"org.hibernate.test.hql.CoalesceTest.HHH_10463_NullInCoalesce_PostgreSQL_Workaround":                                                                                                         "unknown",
	"org.hibernate.test.hql.CoalesceTest.HHH_10463_TestCoalesce":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testLeftJoinMapAndUseKeyExpression":                                                                                              "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapEntryExpressionInSelect":                                                                                                  "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapKeyExpressionDereferenceInSelect":                                                                                         "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapKeyExpressionInSelect":                                                                                                    "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapKeyExpressionInWhere":                                                                                                     "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapValueExpressionInSelect":                                                                                                  "unknown",
	"org.hibernate.test.hql.CollectionMapWithComponentValueTest.testMapValueExpressionInWhere":                                                                                                   "unknown",
	"org.hibernate.test.hql.CriteriaHQLAlignmentTest.testCountReturnValues":                                                                                                                      "5807",
	"org.hibernate.test.hql.CriteriaHQLAlignmentTest.testCriteriaAggregationReturnType":                                                                                                          "unknown",
	"org.hibernate.test.hql.DeleteQuerySubqueryReferencingTargetPropertyTest.testSubQueryReferencingTargetProperty":                                                                              "31670",
	"org.hibernate.test.hql.DeleteWithSubqueryTest.testDeleteMemberOf":                                                                                                                           "31670",
	"org.hibernate.test.hql.DeleteWithSubqueryTest.testDeleteMultipleWhereIns":                                                                                                                   "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testInnerEntityJoins":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testJoinOnEntityJoinNode":                                                                                                                             "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testLeftOuterEntityJoins":                                                                                                                             "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testLeftOuterEntityJoinsWithImplicitInnerJoinInSelectClause":                                                                                          "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testNoImpliedJoinGeneratedForEqualityComparison":                                                                                                      "unknown",
	"org.hibernate.test.hql.EntityJoinTest.testRightOuterEntityJoins":                                                                                                                            "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testInnerEntityJoins":                                                                                                             "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testInnerEntityJoinsWithVariable":                                                                                                 "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testInnerEntityJoinsWithVariableSingleQuoted":                                                                                     "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testJoinOnEntityJoinNode":                                                                                                         "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testLeftOuterEntityJoins":                                                                                                         "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testLeftOuterEntityJoinsWithImplicitInnerJoinInSelectClause":                                                                      "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testNoImpliedJoinGeneratedForEqualityComparison":                                                                                  "unknown",
	"org.hibernate.test.hql.EntityWithUnusualTableNameJoinTest.testRightOuterEntityJoins":                                                                                                        "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetMultiColumnSameNameAsArgFunctionCriteria":                                                                                            "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetMultiColumnSameNameAsArgFunctionHQL":                                                                                                 "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetMultiColumnSameNameAsNoArgFunctionCriteria":                                                                                          "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetMultiColumnSameNameAsNoArgFunctionHQL":                                                                                               "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetOneColumnSameNameAsArgFunctionCriteria":                                                                                              "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testGetOneColumnSameNameAsArgFunctionHQL":                                                                                                   "unknown",
	"org.hibernate.test.hql.FunctionNameAsColumnTest.testNoArgFcnAndColumnSameNameAsNoArgFunctionHQL":                                                                                            "unknown",
	"org.hibernate.test.hql.HibernateFirstResultMaxResultsTest.testPagination":                                                                                                                   "unknown",
	"org.hibernate.test.hql.InsertWithSubSelectTest.testInsert":                                                                                                                                  "31721",
	"org.hibernate.test.hql.InsertWithSubSelectTest.testSelect":                                                                                                                                  "unknown",
	"org.hibernate.test.hql.JoinOnClauseTest.testOnClauseUsesSuperclassAttribute":                                                                                                                "unknown",
	"org.hibernate.test.hql.MapFunctionExpressionsTest.testMapEntryExpressionInSelect":                                                                                                           "unknown",
	"org.hibernate.test.hql.MapFunctionExpressionsTest.testMapKeyExpressionInSelect":                                                                                                             "unknown",
	"org.hibernate.test.hql.MapFunctionExpressionsTest.testMapKeyExpressionInWhere":                                                                                                              "unknown",
	"org.hibernate.test.hql.MapFunctionExpressionsTest.testMapValueExpressionInSelect":                                                                                                           "unknown",
	"org.hibernate.test.hql.ParameterCollectionTest.testReusingQueryWithNewParameterValues":                                                                                                      "unknown",
	"org.hibernate.test.hql.ParameterTest.testClassAsParameter":                                                                                                                                  "unknown",
	"org.hibernate.test.hql.ParameterTest.testObjectAsParameter":                                                                                                                                 "unknown",
	"org.hibernate.test.hql.ParameterTest.testSetPropertiesMapNotContainingAllTheParameters":                                                                                                     "unknown",
	"org.hibernate.test.hql.ParameterTest.testSetPropertiesMapWithNullValues":                                                                                                                    "unknown",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesEmptyResultSet":                                                                                             "unknown",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesForward":                                                                                                    "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesPositioning":                                                                                                "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesReverse":                                                                                                    "5807",
	"org.hibernate.test.hql.ScrollableCollectionFetchingTest.testScrollingJoinFetchesSingleRowResultSet":                                                                                         "5807",
	"org.hibernate.test.hql.ScrollableTest.testScrollableResults":                                                                                                                                "unknown",
	"org.hibernate.test.hql.ScrollableTest.testScrollableResults2":                                                                                                                               "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewListEmbeddedIdValue":                                                                                                            "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewListEntity":                                                                                                                     "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewMapEmbeddedIdValue":                                                                                                             "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewMapEntity":                                                                                                                      "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewObjectEmbeddedIdValue":                                                                                                          "unknown",
	"org.hibernate.test.hql.SelectNewEmbeddedIdTest.testSelectNewObjectEntity":                                                                                                                   "unknown",
	"org.hibernate.test.hql.SizeAttributeReferenceTest.controlGroup":                                                                                                                             "unknown",
	"org.hibernate.test.hql.SizeAttributeReferenceTest.testSizeAttributeReference":                                                                                                               "unknown",
	"org.hibernate.test.hql.SubQueryTest.testCorrelatedJoin":                                                                                                                                     "unknown",
	"org.hibernate.test.hql.SubQueryTest.testSubQueryAsCaseElseResultExpression":                                                                                                                 "unknown",
	"org.hibernate.test.hql.SubQueryTest.testSubQueryAsSearchedCaseExpression":                                                                                                                   "unknown",
	"org.hibernate.test.hql.SubQueryTest.testSubQueryAsSearchedCaseResultExpression":                                                                                                             "unknown",
	"org.hibernate.test.hql.SubQueryTest.testSubQueryAsSimpleCaseTestExpression":                                                                                                                 "unknown",
	"org.hibernate.test.hql.SubQueryTest.testSubQueryAsSimpleCaseWhenExpression":                                                                                                                 "unknown",
	"org.hibernate.test.hql.TreatKeywordTest.memberOfTreatTest":                                                                                                                                  "unknown",
	"org.hibernate.test.hql.UpdateEntitiesWithPackageNamesStartingWithKeywordsTest.testUpdateEntityWithPackageNameStartingWithFrom":                                                              "unknown",
	"org.hibernate.test.hql.UpdateEntitiesWithPackageNamesStartingWithKeywordsTest.testUpdateEntityWithPackageNameStartingWithIn":                                                                "unknown",
	"org.hibernate.test.hql.WithClauseJoinRewriteTest.testInheritanceReAliasing":                                                                                                                 "unknown",
	"org.hibernate.test.hql.WithClauseTest.testWithClause":                                                                                                                                       "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsNonSubqueryWithKey":                                                                                                                   "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubquery":                                                                                                                             "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithEqualOperator":                                                                                                            "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKey":                                                                                                                      "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseAsSubqueryWithKeyAndOtherJoinReference":                                                                                                 "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithClauseFailsWithFetch":                                                                                                                         "5807",
	"org.hibernate.test.hql.WithClauseTest.testWithJoinRHS":                                                                                                                                      "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToManyFetchAndJoinTest.testExplicitJoinAfterFetchJoins":                                                                                                 "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToManyFetchAndJoinTest.testExplicitJoinBeforeFetchJoins":                                                                                                "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToManyFetchAndJoinTest.testExplicitJoinBetweenFetchJoins":                                                                                               "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToOneFetchAndJoinTest.testExplicitJoinAfterFetchJoins":                                                                                                  "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToOneFetchAndJoinTest.testExplicitJoinBeforeFetchJoins":                                                                                                 "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToOneFetchAndJoinTest.testExplicitJoinBetweenFetchJoins":                                                                                                "unknown",
	"org.hibernate.test.hql.fetchAndJoin.ToOneFetchAndJoinTest.testFetchJoinsWithImplicitJoinInRestriction":                                                                                      "unknown",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlDeleteOnJoinedSubclass":                                                                                            "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassBulkManipTest.testHqlUpdateOnJoinedSubclass":                                                                                            "5807",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassSubQueryTest.testSubQueryConstraintPropertyInEntityTable":                                                                               "unknown",
	"org.hibernate.test.hql.joinedSubclass.JoinedSubclassSubQueryTest.testSubQueryConstraintPropertyInSuperclassTable":                                                                           "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScroll":                                                                                                                  "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScrollFirstResult":                                                                                                       "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScrollFirstResultInTransaction":                                                                                          "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScrollLast":                                                                                                              "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScrollSecondResult":                                                                                                      "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testIncompleteScrollSecondResultInTransaction":                                                                                         "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testListOrderChildrenDesc":                                                                                                             "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testNoScroll":                                                                                                                          "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScroll":                                                                                                                            "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScrollOrderChildrenDesc":                                                                                                           "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScrollOrderParentAsc":                                                                                                              "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScrollOrderParentAscChildrenAsc":                                                                                                   "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScrollOrderParentAscChildrenDesc":                                                                                                  "unknown",
	"org.hibernate.test.hqlfetchscroll.HQLScrollFetchTest.testScrollOrderParentDesc":                                                                                                             "unknown",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testAllParams":                                                                                                                      "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testDistinctId":                                                                                                                     "6583",
	"org.hibernate.test.id.MultipleHiLoPerTableGeneratorTest.testRollingBack":                                                                                                                    "6583",
	"org.hibernate.test.id.NonUniqueIdTest.testLoadEntityWithNonUniqueId":                                                                                                                        "12123",
	"org.hibernate.test.id.PooledHiLoSequenceIdentifierTest.testSequenceIdentifierGenerator":                                                                                                     "unknown",
	"org.hibernate.test.id.SequenceGeneratorTest.testStartOfSequence":                                                                                                                            "unknown",
	"org.hibernate.test.id.UseIdentifierRollbackTest.testSimpleRollback":                                                                                                                         "unknown",
	"org.hibernate.test.id.array.ByteArrayIdTest.testMultipleDeletions":                                                                                                                          "unknown",
	"org.hibernate.test.id.array.ByteArrayIdTest.testMultipleUpdates":                                                                                                                            "unknown",
	"org.hibernate.test.id.array.CharacterArrayIdTest.testMultipleDeletions":                                                                                                                     "unknown",
	"org.hibernate.test.id.array.CharacterArrayIdTest.testMultipleUpdates":                                                                                                                       "unknown",
	"org.hibernate.test.id.array.PrimitiveByteArrayIdTest.testMultipleDeletions":                                                                                                                 "unknown",
	"org.hibernate.test.id.array.PrimitiveByteArrayIdTest.testMultipleUpdates":                                                                                                                   "unknown",
	"org.hibernate.test.id.array.PrimitiveCharacterArrayIdTest.testMultipleDeletions":                                                                                                            "unknown",
	"org.hibernate.test.id.array.PrimitiveCharacterArrayIdTest.testMultipleUpdates":                                                                                                              "unknown",
	"org.hibernate.test.id.sequence.NegativeValueSequenceTest.testNegativeOneAllocationSizeNoopOptimizer":                                                                                        "unknown",
	"org.hibernate.test.id.sequence.NegativeValueSequenceTest.testNegativeTwoAllocationSizeNoopOptimizer":                                                                                        "unknown",
	"org.hibernate.test.id.sequence.NegativeValueSequenceTest.testPositiveOneAllocationSizeNoopOptimizer":                                                                                        "unknown",
	"org.hibernate.test.id.sequence.NegativeValueSequenceTest.testPositiveTwoAllocationSizeNoopOptimizer":                                                                                        "unknown",
	"org.hibernate.test.id.sequence.NegativeValueSequenceTest.testPositiveTwoAllocationSizePooledOptimizer":                                                                                      "unknown",
	"org.hibernate.test.id.sequence.OptimizerTest.testGenerationPastBound":                                                                                                                       "unknown",
	"org.hibernate.test.id.usertype.UserTypeComparableIdTest.testUserTypeId":                                                                                                                     "unknown",
	"org.hibernate.test.id.usertype.UserTypeNonComparableIdTest.testUserTypeId":                                                                                                                  "unknown",
	"org.hibernate.test.id.uuid.sqlrep.sqlbinary.UUIDBinaryTest.testUsage":                                                                                                                       "unknown",
	"org.hibernate.test.id.uuid.sqlrep.sqlchar.UUIDCharTest.testUsage":                                                                                                                           "unknown",
	"org.hibernate.test.id.uuid.strategy.CustomStrategyTest.testUsage":                                                                                                                           "unknown",
	"org.hibernate.test.idbag.IdBagTest.testJoin":                                                                                                                                                "unknown",
	"org.hibernate.test.idbag.IdBagTest.testUpdateIdBag":                                                                                                                                         "unknown",
	"org.hibernate.test.idclass.IdClassTest.testIdClass":                                                                                                                                         "unknown",
	"org.hibernate.test.idgen.biginteger.increment.BigIntegerIncrementGeneratorTest.testBasics":                                                                                                  "unknown",
	"org.hibernate.test.idgen.biginteger.sequence.BigIntegerSequenceGeneratorTest.testBasics":                                                                                                    "unknown",
	"org.hibernate.test.idgen.biginteger.sequence.BigIntegerSequenceGeneratorZeroScaleTest.testBasics":                                                                                           "unknown",
	"org.hibernate.test.idgen.enhanced.forcedtable.BasicForcedTableSequenceTest.testNormalBoundary":                                                                                              "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.HiLoForcedTableSequenceTest.testNormalBoundary":                                                                                               "6583",
	"org.hibernate.test.idgen.enhanced.forcedtable.PooledForcedTableSequenceTest.testNormalBoundary":                                                                                             "6583",
	"org.hibernate.test.idgen.enhanced.sequence.BasicSequenceTest.testNormalBoundary":                                                                                                            "unknown",
	"org.hibernate.test.idgen.enhanced.sequence.BasicSequenceTest.testSequencePerEntity":                                                                                                         "unknown",
	"org.hibernate.test.idgen.enhanced.sequence.HiLoSequenceTest.testNormalBoundary":                                                                                                             "unknown",
	"org.hibernate.test.idgen.enhanced.sequence.PooledSequenceTest.testNormalBoundary":                                                                                                           "unknown",
	"org.hibernate.test.idgen.enhanced.table.BasicTableTest.testNormalBoundary":                                                                                                                  "6583",
	"org.hibernate.test.idgen.enhanced.table.HiLoTableTest.testNormalBoundary":                                                                                                                   "6583",
	"org.hibernate.test.idgen.enhanced.table.PooledTableTest.testNormalBoundary":                                                                                                                 "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsParallelGeneration":                                                                          "6583",
	"org.hibernate.test.idgen.enhanced.table.concurrent.HiloOptimizerConcurrencyTest.testTwoSessionsSerialGeneration":                                                                            "6583",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorJtaTest.addRelationExplicitFlush":                                                                                                          "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorJtaTest.addRelationImplicitFlush":                                                                                                          "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorJtaTest.addRelationImplicitFlushCloseEntityManager":                                                                                        "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorJtaTest.addRelationImplicitFlushOneTx":                                                                                                     "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorResourceLocalTest.addRelationExplicitFlush":                                                                                                "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorResourceLocalTest.addRelationImplicitFlush":                                                                                                "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorResourceLocalTest.addRelationImplicitFlushCloseEntityManager":                                                                              "unknown",
	"org.hibernate.test.idgen.foreign.ForeignGeneratorResourceLocalTest.addRelationImplicitFlushOneTx":                                                                                           "unknown",
	"org.hibernate.test.idgen.identity.hhh10429.IdentityGeneratorExtendsTest.testIdentifierGeneratorExtendsIdentityGenerator":                                                                    "24062",
	"org.hibernate.test.idgen.identity.joinedSubClass.JoinedSubclassHierarchyWithIdentityGenerationTest.shouldPersistDebtorAccountWhenParentServiceAgreementPersisted":                           "5807",
	"org.hibernate.test.idgen.namescope.IdGeneratorNamesLocalScopeTest.testNoSequenceGenratorNameClash":                                                                                          "6583",
	"org.hibernate.test.idprops.IdentifierPropertyReferencesTest.testCriteriaIdPropertyReferences":                                                                                               "unknown",
	"org.hibernate.test.idprops.IdentifierPropertyReferencesTest.testHqlIdPropertyReferences":                                                                                                    "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testChangeImmutableEntityProxyToModifiable":                                                                                                      "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testChangeImmutableEntityToModifiable":                                                                                                           "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testDeleteDetachedImmutable":                                                                                                                     "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testDeleteDetachedModifiedImmutable":                                                                                                             "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testGetAndDeleteManagedImmutable":                                                                                                                "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutable":                                                                                                                                   "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableChildEntityWithMerge":                                                                                                               "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableChildEntityWithUpdate":                                                                                                              "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableCollectionWithMerge":                                                                                                                "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableCollectionWithUpdate":                                                                                                               "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableEntityAddImmutableToInverseMutableCollection":                                                                                       "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableEntityRemoveImmutableFromInverseMutableCollection":                                                                                  "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableEntityRemoveImmutableFromInverseMutableCollectionByDelete":                                                                          "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableEntityRemoveImmutableFromInverseMutableCollectionByDeref":                                                                           "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableParentEntityWithMerge":                                                                                                              "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testImmutableParentEntityWithUpdate":                                                                                                             "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testNewEntityViaImmutableEntityWithImmutableCollectionUsingMerge":                                                                                "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testNewEntityViaImmutableEntityWithImmutableCollectionUsingSaveOrUpdate":                                                                         "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testPersistAndUpdateImmutable":                                                                                                                   "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testPersistImmutable":                                                                                                                            "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testPersistUpdateImmutableInSameTransaction":                                                                                                     "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testRefreshImmutable":                                                                                                                            "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testSaveImmutable":                                                                                                                               "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testSaveOrUpdateImmutable":                                                                                                                       "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testUnmodifiedImmutableParentEntityWithMerge":                                                                                                    "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testUpdateAndDeleteManagedImmutable":                                                                                                             "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testUpdatedEntityViaImmutableEntityWithImmutableCollectionUsingMerge":                                                                            "unknown",
	"org.hibernate.test.immutable.ImmutableTest.testUpdatedEntityViaImmutableEntityWithImmutableCollectionUsingSaveOrUpdate":                                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testAddExistingManyToManyElementToPersistentEntity":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testAddNewManyToManyElementToPersistentEntity":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionMergeWithExistingElement":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionUpdateWithExistingElement":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionMergeWithNewElement":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfExisting":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfNew":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionUpdateWithNewElement":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testDeleteManyToManyElement":                                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithMerge":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithUpdate":                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testMoveManyToManyElementToExistingEntityCollection":                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testMoveManyToManyElementToNewEntityCollection":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testRemoveManyToManyElementByDelete":                                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingMerge":                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingMergeBothSides":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingUpdate":                                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingUpdateBothSides":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseManyToManyTest.testUpdateProperty":                                                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testAddExistingOneToManyElementToPersistentEntity":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testAddNewOneToManyElementToPersistentEntity":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testDeleteOneToManyElement":                                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testDeleteOneToManyOrphan":                                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testMoveOneToManyElementToExistingEntityCollection":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testMoveOneToManyElementToNewEntityCollection":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithMerge":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                                     "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementByDelete":                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementUsingMerge":                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementUsingUpdate":                                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingMerge":                                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingUpdate":                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyJoinTest.testUpdateProperty":                                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testAddExistingOneToManyElementToPersistentEntity":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testAddNewOneToManyElementToPersistentEntity":                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testDeleteOneToManyElement":                                                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testDeleteOneToManyOrphan":                                                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testMoveOneToManyElementToExistingEntityCollection":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testMoveOneToManyElementToNewEntityCollection":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithMerge":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithUpdate":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testRemoveOneToManyElementByDelete":                                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testRemoveOneToManyElementUsingMerge":                                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testRemoveOneToManyElementUsingUpdate":                                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testRemoveOneToManyOrphanUsingMerge":                                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testRemoveOneToManyOrphanUsingUpdate":                                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.EntityWithInverseOneToManyTest.testUpdateProperty":                                                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testAddExistingManyToManyElementToPersistentEntity":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testAddNewManyToManyElementToPersistentEntity":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionMergeWithExistingElement":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionUpdateWithExistingElement":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionMergeWithNewElement":                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfExisting":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfNew":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionUpdateWithNewElement":                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testDeleteManyToManyElement":                                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithMerge":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithUpdate":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testMoveManyToManyElementToExistingEntityCollection":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testMoveManyToManyElementToNewEntityCollection":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testRemoveManyToManyElementByDelete":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingMerge":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingMergeBothSides":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingUpdate":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testRemoveManyToManyElementUsingUpdateBothSides":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseManyToManyTest.testUpdateProperty":                                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testAddNewOneToManyElementToPersistentEntity":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testDeleteOneToManyElement":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testDeleteOneToManyOrphan":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testMoveOneToManyElementToExistingEntityCollection":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testMoveOneToManyElementToNewEntityCollection":                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithMerge":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithUpdate":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testRemoveOneToManyElementByDelete":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testRemoveOneToManyOrphanUsingMerge":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testRemoveOneToManyOrphanUsingUpdate":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyFailureExpectedTest.testUpdateProperty":                                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testAddNewOneToManyElementToPersistentEntity":                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testDeleteOneToManyElement":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testDeleteOneToManyOrphan":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testMoveOneToManyElementToExistingEntityCollection":             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testMoveOneToManyElementToNewEntityCollection":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithMerge":              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testOneToManyCollectionOptimisticLockingWithUpdate":             "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testRemoveOneToManyElementByDelete":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testRemoveOneToManyOrphanUsingMerge":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testRemoveOneToManyOrphanUsingUpdate":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinFailureExpectedTest.testUpdateProperty":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testAddExistingOneToManyElementToPersistentEntity":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testAddNewOneToManyElementToPersistentEntity":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testDeleteOneToManyElement":                                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testDeleteOneToManyOrphan":                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testMoveOneToManyElementToExistingEntityCollection":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testMoveOneToManyElementToNewEntityCollection":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithMerge":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                            "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementByDelete":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementUsingMerge":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testRemoveOneToManyElementUsingUpdate":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingMerge":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingUpdate":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyJoinTest.testUpdateProperty":                                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testAddExistingOneToManyElementToPersistentEntity":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testAddNewOneToManyElementToPersistentEntity":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testDeleteOneToManyElement":                                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testDeleteOneToManyOrphan":                                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testMoveOneToManyElementToExistingEntityCollection":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testMoveOneToManyElementToNewEntityCollection":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithMerge":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithUpdate":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testRemoveOneToManyElementByDelete":                                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testRemoveOneToManyElementUsingMerge":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testRemoveOneToManyElementUsingUpdate":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testRemoveOneToManyOrphanUsingMerge":                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testRemoveOneToManyOrphanUsingUpdate":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.inverse.VersionedEntityWithInverseOneToManyTest.testUpdateProperty":                                                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testAddExistingManyToManyElementToPersistentEntity":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testAddNewManyToManyElementToPersistentEntity":                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionMergeWithExistingElement":                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionUpdateWithExistingElement":                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionMergeWithNewElement":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfExisting":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfNew":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionUpdateWithNewElement":                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testDeleteManyToManyElement":                                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithMerge":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithUpdate":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testMoveManyToManyElementToExistingEntityCollection":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testMoveManyToManyElementToNewEntityCollection":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testRemoveManyToManyElementByDelete":                                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingMerge":                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingMergeBothSides":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingUpdate":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingUpdateBothSides":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyTest.testUpdateProperty":                                                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testAddExistingManyToManyElementToPersistentEntity":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testAddNewManyToManyElementToPersistentEntity":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithEmptyManyToManyCollectionMergeWithExistingElement":               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithEmptyManyToManyCollectionUpdateWithExistingElement":              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithNonEmptyManyToManyCollectionMergeWithNewElement":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithNonEmptyManyToManyCollectionOfExisting":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithNonEmptyManyToManyCollectionOfNew":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testCreateWithNonEmptyManyToManyCollectionUpdateWithNewElement":                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testDeleteManyToManyElement":                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testManyToManyCollectionOptimisticLockingWithMerge":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testManyToManyCollectionOptimisticLockingWithUpdate":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testMoveManyToManyElementToExistingEntityCollection":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testMoveManyToManyElementToNewEntityCollection":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testRemoveManyToManyElementByDelete":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testRemoveManyToManyElementUsingMerge":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testRemoveManyToManyElementUsingMergeBothSides":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testRemoveManyToManyElementUsingUpdate":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testRemoveManyToManyElementUsingUpdateBothSides":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseManyToManyUnidirTest.testUpdateProperty":                                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testAddExistingOneToManyElementToPersistentEntity":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testAddNewOneToManyElementToPersistentEntity":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testDeleteOneToManyElement":                                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testDeleteOneToManyOrphan":                                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testMoveOneToManyElementToExistingEntityCollection":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testMoveOneToManyElementToNewEntityCollection":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithMerge":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                               "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementByDelete":                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementUsingMerge":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementUsingUpdate":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingMerge":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingUpdate":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyJoinTest.testUpdateProperty":                                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testAddExistingOneToManyElementToPersistentEntity":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testAddNewOneToManyElementToPersistentEntity":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testDeleteOneToManyElement":                                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testDeleteOneToManyOrphan":                                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testMoveOneToManyElementToExistingEntityCollection":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testMoveOneToManyElementToNewEntityCollection":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithMerge":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithUpdate":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testRemoveOneToManyElementByDelete":                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testRemoveOneToManyElementUsingMerge":                                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testRemoveOneToManyElementUsingUpdate":                                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testRemoveOneToManyOrphanUsingMerge":                                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testRemoveOneToManyOrphanUsingUpdate":                                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyTest.testUpdateProperty":                                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testAddExistingOneToManyElementToPersistentEntity":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testAddNewOneToManyElementToPersistentEntity":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                                 "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testDeleteOneToManyElement":                                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testDeleteOneToManyOrphan":                                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testMoveOneToManyElementToExistingEntityCollection":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testMoveOneToManyElementToNewEntityCollection":                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testOneToManyCollectionOptimisticLockingWithMerge":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testOneToManyCollectionOptimisticLockingWithUpdate":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testRemoveOneToManyElementByDelete":                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testRemoveOneToManyElementUsingMerge":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testRemoveOneToManyElementUsingUpdate":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testRemoveOneToManyOrphanUsingMerge":                                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testRemoveOneToManyOrphanUsingUpdate":                                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.EntityWithNonInverseOneToManyUnidirTest.testUpdateProperty":                                                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testAddExistingManyToManyElementToPersistentEntity":                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testAddNewManyToManyElementToPersistentEntity":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionMergeWithExistingElement":            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithEmptyManyToManyCollectionUpdateWithExistingElement":           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionMergeWithNewElement":              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfExisting":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionOfNew":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testCreateWithNonEmptyManyToManyCollectionUpdateWithNewElement":             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testDeleteManyToManyElement":                                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithMerge":                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testManyToManyCollectionOptimisticLockingWithUpdate":                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testMoveManyToManyElementToExistingEntityCollection":                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testMoveManyToManyElementToNewEntityCollection":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testRemoveManyToManyElementByDelete":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingMerge":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingMergeBothSides":                             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingUpdate":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testRemoveManyToManyElementUsingUpdateBothSides":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseManyToManyTest.testUpdateProperty":                                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testAddExistingOneToManyElementToPersistentEntity":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testAddNewOneToManyElementToPersistentEntity":                            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":            "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testDeleteOneToManyElement":                                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testDeleteOneToManyOrphan":                                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testMoveOneToManyElementToExistingEntityCollection":                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testMoveOneToManyElementToNewEntityCollection":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithMerge":                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testOneToManyCollectionOptimisticLockingWithUpdate":                      "5807",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementByDelete":                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementUsingMerge":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyElementUsingUpdate":                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingMerge":                                     "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testRemoveOneToManyOrphanUsingUpdate":                                    "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyJoinTest.testUpdateProperty":                                                      "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testAddExistingOneToManyElementToPersistentEntity":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testAddNewOneToManyElementToPersistentEntity":                                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionMergeWithExistingElement":              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithEmptyOneToManyCollectionUpdateWithExistingElement":             "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionMergeWithNewElement":                "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfExisting":                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionOfNew":                              "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testCreateWithNonEmptyOneToManyCollectionUpdateWithNewElement":               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testDeleteOneToManyElement":                                                  "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testDeleteOneToManyOrphan":                                                   "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testMoveOneToManyElementToExistingEntityCollection":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testMoveOneToManyElementToNewEntityCollection":                               "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithMerge":                           "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testOneToManyCollectionOptimisticLockingWithUpdate":                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testRemoveOneToManyElementByDelete":                                          "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testRemoveOneToManyElementUsingMerge":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testRemoveOneToManyElementUsingUpdate":                                       "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testRemoveOneToManyOrphanUsingMerge":                                         "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testRemoveOneToManyOrphanUsingUpdate":                                        "unknown",
	"org.hibernate.test.immutable.entitywithmutablecollection.noninverse.VersionedEntityWithNonInverseOneToManyTest.testUpdateProperty":                                                          "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceEagerTest.joinFindEntity":                                                                                                     "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceEagerTest.joinFindParenEntity":                                                                                                "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceEagerTest.joinUnrelatedCollectionOnBaseType":                                                                                  "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceEagerTest.selectBaseType":                                                                                                     "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceTest.selectWhereTypeEqual":                                                                                                    "unknown",
	"org.hibernate.test.inheritance.discriminator.JoinedInheritanceTest.simpleSelectTest":                                                                                                        "unknown",
	"org.hibernate.test.inheritance.discriminator.MappedSuperclassExtendsEntityTest.testQueryingSingle":                                                                                          "unknown",
	"org.hibernate.test.inheritance.discriminator.MultiInheritanceImplicitDowncastTest.testQueryingMultiple":                                                                                     "unknown",
	"org.hibernate.test.inheritance.discriminator.MultiInheritanceImplicitDowncastTest.testQueryingSingle":                                                                                       "unknown",
	"org.hibernate.test.inheritance.discriminator.MultiSingleTableLoadTest.testEagerLoadMultipleHoldersWithDifferentSubtypes":                                                                    "unknown",
	"org.hibernate.test.inheritance.discriminator.MultiSingleTableLoadTest.testFetchJoinLoadMultipleHoldersWithDifferentSubtypes":                                                                "unknown",
	"org.hibernate.test.inheritance.discriminator.PersistChildEntitiesWithDiscriminatorTest.doIt":                                                                                                "unknown",
	"org.hibernate.test.inheritance.discriminator.SingleTableInheritancePersistTest.testPolymorphicAssociation":                                                                                  "31589",
	"org.hibernate.test.inheritance.discriminator.SingleTableRelationsTest.testJoinFetch":                                                                                                        "unknown",
	"org.hibernate.test.inheritance.discriminator.SingleTableRelationsTest.testLazyInitialization":                                                                                               "unknown",
	"org.hibernate.test.inheritance.discriminator.TablePerClassInheritancePersistTest.testPolymorphicAssociation":                                                                                "pass?",
	"org.hibernate.test.inheritance.discriminator.embeddable.JoinedSubclassWithEmbeddableTest.testSelectFromEmbeddedField":                                                                       "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.JoinedSubclassWithEmbeddableTest.testSelectFromParent":                                                                              "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.JoinedSubclassWithEmbeddableTest.testSelectFromSubclass":                                                                            "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.SingleTableWithEmbeddableTest.testSelectFromEmbeddedField":                                                                          "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.SingleTableWithEmbeddableTest.testSelectFromParent":                                                                                 "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.SingleTableWithEmbeddableTest.testSelectFromSubclass":                                                                               "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.TablePerClassWithEmbeddableTest.testSelectFromEmbeddedField":                                                                        "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.TablePerClassWithEmbeddableTest.testSelectFromParent":                                                                               "unknown",
	"org.hibernate.test.inheritance.discriminator.embeddable.TablePerClassWithEmbeddableTest.testSelectFromSubclass":                                                                             "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassTest.shouldNotRetrieveSubSubSubEntityWithCriteria":                                                                "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassTest.shouldNotRetrieveSubSubSubEntityWithHQL":                                                                     "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassTest.shouldRetrieveSubEntity":                                                                                     "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassTest.shouldRetrieveSubSubEntityWithCriteria":                                                                      "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassTest.shouldRetrieveSubSubEntityWithHQL":                                                                           "unknown",
	"org.hibernate.test.inheritance.discriminator.joinedsubclass.JoinedSubclassWithRootInterfaceTest.testIt":                                                                                     "unknown",
	"org.hibernate.test.inheritance.relationship.JoinedInheritanceWithOneToManyTest.test":                                                                                                        "unknown",
	"org.hibernate.test.insertordering.ElementCollectionTest.testBatchOrdering":                                                                                                                  "unknown",
	"org.hibernate.test.insertordering.InsertOrderingDuplicateTest.testBatching":                                                                                                                 "unknown",
	"org.hibernate.test.insertordering.InsertOrderingHasParentTest.testInsert":                                                                                                                   "unknown",
	"org.hibernate.test.insertordering.InsertOrderingTest.testBatchOrdering":                                                                                                                     "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBaseClassReferencingSubclass.testBatching":                                                                                              "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalManyToMany.testBatching":                                                                                                   "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalMapsIdOneToOne.testBatching":                                                                                               "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToMany.testBatching":                                                                                                    "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToManyFlushProblem.testBatchingWithFlush":                                                                               "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToManyFlushProblem.testBatchingWithFlush2":                                                                              "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToOne.testBatching":                                                                                                     "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToOneFlushProblem.testInsertSortingWithFlushPersistLeftBeforeRight":                                                     "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithBidirectionalOneToOneFlushProblem.testInsertSortingWithFlushPersistRightBeforeLeft":                                                     "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithCascadeOnPersist.testInsertOrderingAvoidingForeignKeyConstraintViolation":                                                               "6583",
	"org.hibernate.test.insertordering.InsertOrderingWithCompositeTypeAssociation.testOrderedInsertSupport":                                                                                      "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchOrdering":                                                                                               "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableInheritance.testBatchingAmongstSubClasses":                                                                                   "26738",
	"org.hibernate.test.insertordering.InsertOrderingWithJoinedTableMultiLevelInheritance.testBatchingAmongstSubClasses":                                                                         "26738",
	"org.hibernate.test.insertordering.InsertOrderingWithManyToOne.testBatching":                                                                                                                 "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithMultipleManyToOne.testBatching":                                                                                                         "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithSecondaryTable.testInheritanceWithSecondaryTable":                                                                                       "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithSingleTableInheritance.testBatchOrdering":                                                                                               "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithSingleTableInheritance.testBatchingAmongstSubClasses":                                                                                   "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchOrdering":                                                                                             "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithTablePerClassInheritance.testBatchingAmongstSubClasses":                                                                                 "5807",
	"org.hibernate.test.insertordering.InsertOrderingWithUnidirectionalOneToOne.testBatching":                                                                                                    "unknown",
	"org.hibernate.test.insertordering.InsertOrderingWithUnidirectionalOneToOneJoinColumn.testBatchingWithEmbeddableId":                                                                          "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testCollectionIntercept":                                                                                                                     "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testComponentInterceptor":                                                                                                                    "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testInitiateIntercept":                                                                                                                       "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testPrepareStatementIntercept":                                                                                                               "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testPropertyIntercept":                                                                                                                       "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testPropertyIntercept2":                                                                                                                      "unknown",
	"org.hibernate.test.interceptor.InterceptorTest.testStatefulIntercept":                                                                                                                       "unknown",
	"org.hibernate.test.interfaceproxy.InterfaceProxyTest.testInterfaceProxies":                                                                                                                  "26725",
	"org.hibernate.test.iterate.IterateTest.testIterate":                                                                                                                                         "unknown",
	"org.hibernate.test.iterate.IterateTest.testScroll":                                                                                                                                          "unknown",
	"org.hibernate.test.jdbc.GeneralWorkTest.testGeneralReturningUsage":                                                                                                                          "unknown",
	"org.hibernate.test.jdbc.GeneralWorkTest.testGeneralUsage":                                                                                                                                   "unknown",
	"org.hibernate.test.jdbc.internal.BasicConnectionTest.testBasicJdbcUsage":                                                                                                                    "unknown",
	"org.hibernate.test.jdbc.internal.BatchingTest.testBatchingUsage":                                                                                                                            "unknown",
	"org.hibernate.test.jdbc.internal.BatchingTest.testNonBatchingUsage":                                                                                                                         "unknown",
	"org.hibernate.test.jdbc.internal.BatchingTest.testSessionBatchingUsage":                                                                                                                     "unknown",
	"org.hibernate.test.jdbc.internal.SessionJdbcBatchTest.testSessionFactorySetting":                                                                                                            "unknown",
	"org.hibernate.test.jdbc.internal.SessionJdbcBatchTest.testSessionSettingOverridesSessionFactorySetting":                                                                                     "unknown",
	"org.hibernate.test.join.JoinTest.testCustomColumnReadAndWrite":                                                                                                                              "unknown",
	"org.hibernate.test.join.JoinTest.testSequentialSelects":                                                                                                                                     "unknown",
	"org.hibernate.test.join.JoinTest.testSequentialSelectsOptionalData":                                                                                                                         "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testMergeNonNullOptionalJoinToDiffNonNullDetached":                                                                                                 "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testMergeNonNullOptionalJoinToNullDetached":                                                                                                        "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testMergeNullOptionalJoinToNonNullDetached":                                                                                                        "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNonNullOptionalJoinToDiffNonNull":                                                                                                        "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNonNullOptionalJoinToDiffNonNullDetached":                                                                                                "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNonNullOptionalJoinToNull":                                                                                                               "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNonNullOptionalJoinToNullDetached":                                                                                                       "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNullOptionalJoinToNonNull":                                                                                                               "unknown",
	"org.hibernate.test.join.OptionalJoinTest.testUpdateNullOptionalJoinToNonNullDetached":                                                                                                       "unknown",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSubclass":                                                                                                            "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testCriteriaSuperclass":                                                                                                          "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testGetSuperclass":                                                                                                               "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySubclass":                                                                                                               "5807",
	"org.hibernate.test.join.SubclassesWithSamePropertyNameTest.testQuerySuperclass":                                                                                                             "5807",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testAccessAsIncorrectSubclass":                                                                                                         "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testCustomColumnReadAndWrite":                                                                                                          "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testJoinedSubclass":                                                                                                                    "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testLockingJoinedSubclass":                                                                                                             "6583",
	"org.hibernate.test.joinedsubclass.JoinedSubclassTest.testQuerySubclassAttribute":                                                                                                            "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassWithEmbeddableTest.testEmbeddedFieldIsNotNull":                                                                                              "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassWithExplicitDiscriminatorTest.basicUsageTest":                                                                                               "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassWithIgnoredExplicitDiscriminatorTest.basicUsageTest":                                                                                        "unknown",
	"org.hibernate.test.joinedsubclass.JoinedSubclassWithImplicitDiscriminatorTest.basicUsageTest":                                                                                               "unknown",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrBiggerThenBatch":                                                               "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrEqualWithBatch":                                                                "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrLessThenBatch":                                                                 "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeEqJdbcBatchSize":                                                                         "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeGtJdbcBatchSize":                                                                         "24062",
	"org.hibernate.test.joinedsubclassbatch.IdentityJoinedSubclassBatchingTest.testBatchInsertUpdateSizeLtJdbcBatchSize":                                                                         "24062",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrBiggerThenBatch":                                                                       "unknown",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrEqualWithBatch":                                                                        "unknown",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.doBatchInsertUpdateJoinedSubclassNrLessThenBatch":                                                                         "unknown",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.testBatchInsertUpdateSizeEqJdbcBatchSize":                                                                                 "unknown",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.testBatchInsertUpdateSizeGtJdbcBatchSize":                                                                                 "unknown",
	"org.hibernate.test.joinedsubclassbatch.JoinedSubclassBatchingTest.testBatchInsertUpdateSizeLtJdbcBatchSize":                                                                                 "unknown",
	"org.hibernate.test.joinfetch.JoinFetchTest.testCollectionFilter":                                                                                                                            "unknown",
	"org.hibernate.test.joinfetch.JoinFetchTest.testJoinFetch":                                                                                                                                   "unknown",
	"org.hibernate.test.joinfetch.JoinFetchTest.testJoinFetchManyToMany":                                                                                                                         "unknown",
	"org.hibernate.test.joinfetch.JoinFetchTest.testProjection":                                                                                                                                  "unknown",
	"org.hibernate.test.jpa.cascade.CascadeTest.testManyToOneGeneratedIdsOnSave":                                                                                                                 "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.QueryExecutionTest.testCollectionFetch":                                                                                                            "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.TableGeneratorVisibilityTest.testGeneratorIsVisible":                                                                                               "6583",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testStrictUnversioned":                                                                                     "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testUnstrictUnversioned":                                                                                   "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.CachingWithSecondaryTablesTests.testVersioned":                                                                                             "5807",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.InheritedCacheableTest.testMapping":                                                                                                        "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.InheritedCacheableTest.testOnlySubclassIsCached":                                                                                           "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.SubclassOnlyCachingTests.testMapping":                                                                                                      "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.caching.SubclassOnlyCachingTests.testOnlySubclassIsCached":                                                                                         "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnBiDiOneToManyFKTest.testReferenceToAlreadyMappedColumn":                                                                   "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnBiDiOneToManyFKTest.testReferenceToNonMappedColumn":                                                                       "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnElementCollectionTest.testReferenceToAlreadyMappedColumn":                                                                 "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnElementCollectionTest.testReferenceToNonMappedColumn":                                                                     "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnManyToManyTest.testReferenceToAlreadyMappedColumn":                                                                        "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnManyToManyTest.testReferenceToNonMappedColumn":                                                                            "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnOneToManyFKTest.testReferenceToAlreadyMappedColumn":                                                                       "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnOneToManyFKTest.testReferenceToNonMappedColumn":                                                                           "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnOneToManyJoinTableTest.testReferenceToAlreadyMappedColumn":                                                                "unknown",
	"org.hibernate.test.jpa.compliance.tck2_2.mapkeycolumn.MapKeyColumnOneToManyJoinTableTest.testReferenceToNonMappedColumn":                                                                    "unknown",
	"org.hibernate.test.jpa.convert.JdbcSqlTypeDescriptorConverterTest.testJava8TimeObjectsUsingJdbcSqlTypeDescriptors":                                                                          "unknown",
	"org.hibernate.test.jpa.fetch.FetchingTest.testHibernateFetchingLazy":                                                                                                                        "unknown",
	"org.hibernate.test.jpa.fetch.FetchingTest.testLazy":                                                                                                                                         "unknown",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutFind":                                                                                                                         "unknown",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutLock":                                                                                                                         "6583",
	"org.hibernate.test.jpa.lock.LockExceptionTests.testLockTimeoutRefresh":                                                                                                                      "6583",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleNonVersionedInstanceFoundInQueryResult":                                                                                             "unknown",
	"org.hibernate.test.jpa.lock.RepeatableReadTest.testStaleVersionedInstanceFoundInQueryResult":                                                                                                "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testNaturalIdCache":                                                                                                                 "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testNaturalIdCheck":                                                                                                                 "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testNaturalIdDeleteUsingCache":                                                                                                      "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testNaturalIdLoadAccessCache":                                                                                                       "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testNaturalIdRecreateUsingCache":                                                                                                    "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testSimpleNaturalIdLoadAccessCache":                                                                                                 "unknown",
	"org.hibernate.test.jpa.naturalid.ImmutableNaturalIdTest.testUpdate":                                                                                                                         "unknown",
	"org.hibernate.test.jpa.naturalid.MutableNaturalIdTest.testInLineSynchWithIdentityColumn":                                                                                                    "unknown",
	"org.hibernate.test.jpa.naturalid.MutableNaturalIdTest.testSimpleNaturalIdLoadAccessCacheWithUpdate":                                                                                         "unknown",
	"org.hibernate.test.jpa.orphan.one2one.OneToOneEagerNonOptionalOrphanRemovalTest.testOneToOneLazyNonOptionalOrphanRemoval":                                                                   "unknown",
	"org.hibernate.test.jpa.orphan.one2one.OneToOneEagerOrphanRemovalTest.testOneToOneEagerOrphanRemoval":                                                                                        "unknown",
	"org.hibernate.test.jpa.orphan.one2one.OneToOneLazyNonOptionalOrphanRemovalTest.testOneToOneLazyNonOptionalOrphanRemoval":                                                                    "unknown",
	"org.hibernate.test.jpa.orphan.one2one.OneToOneLazyOrphanRemovalTest.testOneToOneLazyOrphanRemoval":                                                                                          "unknown",
	"org.hibernate.test.jpa.proxy.JPAProxyTest.testEjb3ProxyUsage":                                                                                                                               "unknown",
	"org.hibernate.test.jpa.proxy.JPAProxyTest.testGetSemantics":                                                                                                                                 "unknown",
	"org.hibernate.test.jpa.ql.FunctionKeywordTest.basicFixture":                                                                                                                                 "unknown",
	"org.hibernate.test.jpa.ql.FunctionKeywordTest.basicTest":                                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testMixedTckUsage":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testUsageInDistinct":                                                                                                                  "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testUsageInJpaInCollectionSyntax":                                                                                                     "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testUsageInPath":                                                                                                                      "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testUsageInSelect":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.IdentificationVariablesTest.testUsageInSelectObject":                                                                                                              "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testAliasNameSameAsUnqualifiedEntityName":                                                                                                     "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testGeneratedSubquery":                                                                                                                        "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testIdentifierCaseSensitive":                                                                                                                  "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testOrderByAlias":                                                                                                                             "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testParameterCollectionParenthesesAndPositional":                                                                                              "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testReusedNamedCollectionParam":                                                                                                               "unknown",
	"org.hibernate.test.jpa.ql.JPAQLComplianceTest.testReusedPositionalCollectionParam":                                                                                                          "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testDeleteMultipleValues":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testDeleteSingleValue":                                                                                                                       "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testInsertMultipleValues":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testInsertSingleValue":                                                                                                                       "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testMultipleSelect":                                                                                                                          "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testSingleSelect":                                                                                                                            "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testUpdateMultipleValues":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.ql.NamedNativeQueryTest.testUpdateSingleValue":                                                                                                                       "unknown",
	"org.hibernate.test.jpa.ql.NativeQueryTest.testJpaStylePositionalParametersInNativeSql":                                                                                                      "unknown",
	"org.hibernate.test.jpa.ql.OnKeywordTest.basicTest":                                                                                                                                          "unknown",
	"org.hibernate.test.jpa.ql.TreatKeywordTest.testBasicUsageInJoin":                                                                                                                            "unknown",
	"org.hibernate.test.jpa.ql.TreatKeywordTest.testFilteringDiscriminatorSubclasses":                                                                                                            "unknown",
	"org.hibernate.test.jpa.ql.TreatKeywordTest.testFilteringJoinedSubclasses":                                                                                                                   "unknown",
	"org.hibernate.test.jpa.ql.TreatKeywordTest.testTreatWithRestrictionOnAbstractClass":                                                                                                         "unknown",
	"org.hibernate.test.jpa.removed.RemovedEntityTest.testRemoveChildThenFlushWithCascadePersist":                                                                                                "unknown",
	"org.hibernate.test.jpa.removed.RemovedEntityTest.testRemoveThenContains":                                                                                                                    "unknown",
	"org.hibernate.test.jpa.removed.RemovedEntityTest.testRemoveThenGet":                                                                                                                         "unknown",
	"org.hibernate.test.jpa.removed.RemovedEntityTest.testRemoveThenSave":                                                                                                                        "unknown",
	"org.hibernate.test.jpa.removed.RemovedEntityTest.testRemoveThenSaveWithCascades":                                                                                                            "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerCollectionLazyKeyManyToOneTest.testLoadingStrategies":                                                                                  "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerCollectionLazyKeyManyToOneTest.testQueryingOnMany2One":                                                                                 "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerCollectionLazyKeyManyToOneTest.testSaveCascadedToKeyManyToOne":                                                                         "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerKeyManyToOneTest.testLoadEntityWithEagerFetchingToKeyManyToOneReferenceBackToSelf":                                                     "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerKeyManyToOneTest.testLoadingStrategies":                                                                                                "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.EagerKeyManyToOneTest.testSaveCascadedToKeyManyToOne":                                                                                       "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.LazyKeyManyToOneTest.testLoadingStrategies":                                                                                                 "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.LazyKeyManyToOneTest.testQueryingOnMany2One":                                                                                                "unknown",
	"org.hibernate.test.keymanytoone.bidir.component.LazyKeyManyToOneTest.testSaveCascadedToKeyManyToOne":                                                                                        "unknown",
	"org.hibernate.test.keymanytoone.bidir.embedded.KeyManyToOneTest.testCriteriaRestrictionOnKeyManyToOne":                                                                                      "unknown",
	"org.hibernate.test.keymanytoone.bidir.embedded.KeyManyToOneTest.testLoadingStrategies":                                                                                                      "unknown",
	"org.hibernate.test.keymanytoone.bidir.embedded.KeyManyToOneTest.testQueryingOnMany2One":                                                                                                     "unknown",
	"org.hibernate.test.keymanytoone.bidir.embedded.KeyManyToOneTest.testSaveCascadedToKeyManyToOne":                                                                                             "unknown",
	"org.hibernate.test.keymanytoone.bidir.ondelete.KeyManyToOneCascadeDeleteTest.testEmbeddedCascadeRemoval":                                                                                    "unknown",
	"org.hibernate.test.keymanytoone.unidir.ondelete.KeyManyToOneCascadeDeleteTest.testComponentCascadeRemoval":                                                                                  "unknown",
	"org.hibernate.test.lazydetachedpersist.LazyPersistWithDetachedAssociationTest.testDetachedAssociationOnPersisting":                                                                          "unknown",
	"org.hibernate.test.lazyload.JtaLazyLoadingTest.testLazyCollectionLoadingAfterEndTransaction":                                                                                                "unknown",
	"org.hibernate.test.lazyload.LazyLoadingLoggingTest.classMethod":                                                                                                                             "unknown",
	"org.hibernate.test.lazyload.LazyLoadingTest.testGetIdManyToOne":                                                                                                                             "unknown",
	"org.hibernate.test.lazyload.LazyLoadingTest.testGetIdOneToOne":                                                                                                                              "unknown",
	"org.hibernate.test.lazyload.LazyLoadingTest.testLazyCollectionLoadingAfterEndTransaction":                                                                                                   "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testDiscriminatorFiltering":                                                                                                                          "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testNarrow":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testOneToOne":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testSharedColumn":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testSubclassMap":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.ABCProxyTest.testSubclassing":                                                                                                                                     "6583",
	"org.hibernate.test.legacy.ABCTest.testFormulaAssociation":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.ABCTest.testGetSave":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.ABCTest.testSubclassing":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.ComponentNotNullTest.testComponentNotNull":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.ComponentNotNullTest.testCompositeElement":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.CustomSQLTest.testJoinedSubclass":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.loadFoo":                                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testAfterDelete":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testAny":                                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testArraysOfTimes":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FooBarTest.testAutoFlush":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testAutoFlushCollections":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testAutosaveChildren":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FooBarTest.testBagMultipleElements":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testBagOneToMany":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testBatchLoad":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCache":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCachedCollection":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCallback":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCascadeDeleteDetached":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCascadeSave":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCollectionCache":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCollectionOfSelf":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCollectionWhere":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCollectionsInSelect":                                                                                                                               "31777",
	"org.hibernate.test.legacy.FooBarTest.testComplicatedQuery":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FooBarTest.testComponentParent":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testComponents":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCreate":                                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCreateUpdate":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCriteriaCollection":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FooBarTest.testCustom":                                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testDeleteRecursive":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testDeleteTransient":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testDeleteUpdatedTransient":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testDereferenceLazyCollection":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testDyna":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testEmbeddedCompositeID":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testEmptyCollection":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFetchInitializedCollection":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFetchInitializedCollectionDupe":                                                                                                                    "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFetchList":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFind":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFindByCriteria":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.FooBarTest.testFindLoad":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.FooBarTest.testForCertain":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testForceOuterJoin":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.FooBarTest.testForeignKeys":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testIdBag":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testIterators":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testJoin":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLateCollectionAdd":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLazyCollections":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLazyCollectionsTouchedDuringPreCommit":                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLimit":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testListRemove":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLoad":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testLoadAfterDelete":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testManyToManyBag":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FooBarTest.testManyToOne":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testMoveLazyCollection":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FooBarTest.testMultiColumnQueries":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FooBarTest.testNamedParams":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testNewFlushing":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testNewSessionLifecycle":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testNoForeignKeyViolations":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testNonlazyCollection":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testObjectType":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testOnCascadeDelete":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testOneToOneGenerator":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testOrderBy":                                                                                                                                           "unknown",
	"org.hibernate.test.legacy.FooBarTest.testOrphanDelete":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testPSCache":                                                                                                                                           "unknown",
	"org.hibernate.test.legacy.FooBarTest.testParameterCheck":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.FooBarTest.testPersistCollections":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FooBarTest.testPersistentLifecycle":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testPolymorphism":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testPropertyRef":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testProxyArray":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testQuery":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testQueryCollectionOfValues":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.FooBarTest.testQueryLockMode":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FooBarTest.testReachability":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testRecursiveLoad":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FooBarTest.testRefresh":                                                                                                                                           "unknown",
	"org.hibernate.test.legacy.FooBarTest.testRefreshProxy":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FooBarTest.testRemoveContains":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.FooBarTest.testRemoveFromIdbag":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FooBarTest.testReuseDeletedCollection":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSaveAddDelete":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSaveDelete":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSaveFlush":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSaveOrUpdateCopyAny":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSerializableType":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FooBarTest.testSortables":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FooBarTest.testTransientOrphanDelete":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FooBarTest.testUpdate":                                                                                                                                            "unknown",
	"org.hibernate.test.legacy.FooBarTest.testUpdateCollections":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.FooBarTest.testUpdateFromTransient":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FooBarTest.testUpdateOrder":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.FooBarTest.testVersionedCollections":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testVersioning":                                                                                                                                        "6583",
	"org.hibernate.test.legacy.FooBarTest.testVeto":                                                                                                                                              "unknown",
	"org.hibernate.test.legacy.FooBarTest.testWierdSession":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FumTest.testBeanResultTransformer":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FumTest.testCompositeID":                                                                                                                                          "6583",
	"org.hibernate.test.legacy.FumTest.testCompositeIDCollections":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.FumTest.testCompositeIDOneToOne":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.FumTest.testCompositeIDQuery":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.FumTest.testCompositeIDs":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FumTest.testCompositeKeyPathExpressions":                                                                                                                          "unknown",
	"org.hibernate.test.legacy.FumTest.testCriteria":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.FumTest.testCriteriaCollection":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.FumTest.testDeleteOwner":                                                                                                                                          "6583",
	"org.hibernate.test.legacy.FumTest.testKeyManyToOne":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.FumTest.testListIdentifiers":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.FumTest.testQuery":                                                                                                                                                "unknown",
	"org.hibernate.test.legacy.FumTest.testUnflushedSessionSerialization":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.IJ2Test.testUnionSubclass":                                                                                                                                        "6583",
	"org.hibernate.test.legacy.IJTest.testFormulaDiscriminator":                                                                                                                                  "6583",
	"org.hibernate.test.legacy.LimitExpressionTest.testLimitZero":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.MapTest.testMap":                                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.MapTest.testMapOneToOne":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.MapTest.testOneToOnePropertyRef":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testCachedCollectionRefresh":                                                                                                                     "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testCascading":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testCategories":                                                                                                                                  "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionQuery":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionRefresh":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace":                                                                                                                           "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplace2":                                                                                                                          "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCollectionReplaceOnUpdate":                                                                                                                   "6583",
	"org.hibernate.test.legacy.MasterDetailTest.testCopy":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testExample":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testIncomingOutgoing":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testInterface":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testMasterDetail":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testMixNativeAssigned":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testMultiLevelCascade":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testNamedQuery":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testNoUpdateManyToOne":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testNonLazyBidirectional":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testNotNullDiscriminator":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testOuterJoin":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testPolymorphicCriteria":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testQueuedBagAdds":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testSelfManyToOne":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.MasterDetailTest.testUpdateLazyCollections":                                                                                                                       "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testCollection":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testCollectionOnly":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testCollectionPointer":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testConstraints":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testCriteria":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testFetchOneToMany":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testJoins":                                                                                                                                         "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTable":                                                                                                                                    "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableCollections":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableGeneratedId":                                                                                                                         "6583",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableManyToOne":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testMultiTableNativeId":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testNarrow":                                                                                                                                        "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testOneToOne":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testQueries":                                                                                                                                       "unknown",
	"org.hibernate.test.legacy.MultiTableTest.testSubclassCollection":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.OneToOneCacheTest.testOneToOneCache":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testArrayCriteria":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testArrayHQL":                                                                                                                                     "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testBag":                                                                                                                                          "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testCascadeCompositeElements":                                                                                                                     "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testCircularCascade":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testClassWhere":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testClassWhereManyToMany":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testCollectionQuery":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testComplexCriteria":                                                                                                                              "6583",
	"org.hibernate.test.legacy.ParentChildTest.testContainer":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testDeleteEmpty":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testLazyManyToOneCriteria":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testLazyManyToOneGet":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testLazyManyToOneHQL":                                                                                                                             "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testLoadAfterNonExists":                                                                                                                           "26738",
	"org.hibernate.test.legacy.ParentChildTest.testLocking":                                                                                                                                      "6583",
	"org.hibernate.test.legacy.ParentChildTest.testManyToMany":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testObjectType":                                                                                                                                   "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testParentChild":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testParentNullChild":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testQueryOneToOne":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.ParentChildTest.testReplicate":                                                                                                                                    "unknown",
	"org.hibernate.test.legacy.QueryByExampleTest.testExcludingQBE":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.QueryByExampleTest.testJunctionNotExpressionQBE":                                                                                                                  "unknown",
	"org.hibernate.test.legacy.QueryByExampleTest.testSimpleQBE":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testBlobClob":                                                                                                                                    "26725",
	"org.hibernate.test.legacy.SQLFunctionsTest.testBroken":                                                                                                                                      "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testCachedQuery":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testCachedQueryOnInsert":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testCachedQueryRegion":                                                                                                                           "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testDialectSQLFunctions":                                                                                                                         "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testNothinToUpdate":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testSQLFunctions":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testSetProperties":                                                                                                                               "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testSetPropertiesMap":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.SQLFunctionsTest.testSqlFunctionAsAlias":                                                                                                                          "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testComponentNoStar":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testComponentStar":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testCompositeIdId":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testDoubleAliasing":                                                                                                                                 "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testEmbeddedCompositeProperties":                                                                                                                    "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLAssociatedObjects":                                                                                                                     "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLDiscriminatedDiffSession":                                                                                                              "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLDiscriminatedSameSession":                                                                                                              "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLMultipleObject":                                                                                                                        "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLParameters":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLProperties":                                                                                                                            "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLSimpleByDiffSessions":                                                                                                                  "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindBySQLStar":                                                                                                                                  "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testFindSimpleBySQL":                                                                                                                                "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testPropertyResultSQL":                                                                                                                              "unknown",
	"org.hibernate.test.legacy.SQLLoaderTest.testTS":                                                                                                                                             "unknown",
	"org.hibernate.test.legacy.StatisticsTest.testSessionStats":                                                                                                                                  "unknown",
	"org.hibernate.test.loadplans.process.EncapsulatedCompositeAttributeResultSetProcessorTest.testNestedCompositeElementCollectionProcessing":                                                   "unknown",
	"org.hibernate.test.loadplans.process.EncapsulatedCompositeAttributeResultSetProcessorTest.testSimpleNestedCompositeAttributeProcessing":                                                     "unknown",
	"org.hibernate.test.loadplans.process.EncapsulatedCompositeIdResultSetProcessorTest.testCompositeIdWithKeyManyToOne":                                                                         "unknown",
	"org.hibernate.test.loadplans.process.EncapsulatedCompositeIdResultSetProcessorTest.testSimpleCompositeId":                                                                                   "unknown",
	"org.hibernate.test.loadplans.process.EntityAssociationResultSetProcessorTest.testManyToOneEntityProcessing":                                                                                 "unknown",
	"org.hibernate.test.loadplans.process.EntityAssociationResultSetProcessorTest.testNestedManyToOneEntityProcessing":                                                                           "unknown",
	"org.hibernate.test.loadplans.process.EntityWithNonLazyCollectionResultSetProcessorTest.testEntityWithSet":                                                                                   "unknown",
	"org.hibernate.test.loadplans.process.EntityWithNonLazyOneToManyListResultSetProcessorTest.testEntityWithList":                                                                               "unknown",
	"org.hibernate.test.loadplans.process.EntityWithNonLazyOneToManySetResultSetProcessorTest.testEntityWithSet":                                                                                 "unknown",
	"org.hibernate.test.loadplans.process.NonEncapsulatedCompositeIdResultSetProcessorTest.testCompositeIdWithKeyManyToOne":                                                                      "unknown",
	"org.hibernate.test.loadplans.process.SimpleResultSetProcessorTest.testSimpleEntityProcessing":                                                                                               "unknown",
	"org.hibernate.test.loadplans.process.inheritance.Test.basicTest":                                                                                                                            "5807",
	"org.hibernate.test.lob.BlobLocatorTest.testBoundedBlobLocatorAccess":                                                                                                                        "26725",
	"org.hibernate.test.lob.ClobLocatorTest.testBoundedClobLocatorAccess":                                                                                                                        "26725",
	"org.hibernate.test.lob.ImageTest.testBoundedLongByteArrayAccess":                                                                                                                            "unknown",
	"org.hibernate.test.lob.ImageTest.testSaving":                                                                                                                                                "unknown",
	"org.hibernate.test.lob.LobStringTest.testHqlQuery":                                                                                                                                          "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectClobPropertyInNativeQuery":                                                                                                                   "26725",
	"org.hibernate.test.lob.LobStringTest.testSelectStringLobAnnotatedInNativeQuery":                                                                                                             "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingLobPropertyInNativeQuery":                                                                                                                     "26725",
	"org.hibernate.test.lob.LobStringTest.testUsingStringLobAnnotatedPropertyInNativeQuery":                                                                                                      "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testBoundedLongByteArrayAccess":                                                                                                                 "26725",
	"org.hibernate.test.lob.MaterializedBlobTest.testSaving":                                                                                                                                     "26725",
	"org.hibernate.test.lob.MaterializedClobTest.testBoundedLongStringAccess":                                                                                                                    "26725",
	"org.hibernate.test.lob.PostgreSqlLobStringTest.testBadClobDataSavedAsStringFails":                                                                                                           "unknown",
	"org.hibernate.test.lob.PostgreSqlLobStringTest.testBadClobDataSavedAsStringworksAfterUpdate":                                                                                                "26725",
	"org.hibernate.test.lob.SerializableTypeTest.testNewSerializableType":                                                                                                                        "unknown",
	"org.hibernate.test.lob.TextTest.testBoundedLongStringAccess":                                                                                                                                "unknown",
	"org.hibernate.test.locale.LocaleTest.testMetadataWithLocale":                                                                                                                                "unknown",
	"org.hibernate.test.locking.JoinedInheritanceOptimisticForceIncrementTest.testForceIncrement":                                                                                                "unknown",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteria":                                                                                                                                 "6583",
	"org.hibernate.test.locking.LockModeTest.testLegacyCriteriaAliasSpecific":                                                                                                                    "6583",
	"org.hibernate.test.locking.LockModeTest.testLoading":                                                                                                                                        "6583",
	"org.hibernate.test.locking.LockModeTest.testQuery":                                                                                                                                          "6583",
	"org.hibernate.test.locking.LockModeTest.testQueryLockModeNoneWithAlias":                                                                                                                     "unknown",
	"org.hibernate.test.locking.LockModeTest.testQueryLockModePessimisticWriteWithAlias":                                                                                                         "unknown",
	"org.hibernate.test.locking.LockModeTest.testQueryUsingLockOptions":                                                                                                                          "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshAfterUpdate":                                                                                                                             "unknown",
	"org.hibernate.test.locking.LockModeTest.testRefreshLockedEntity":                                                                                                                            "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitHigherLevelLockMode":                                                                                                         "6583",
	"org.hibernate.test.locking.LockModeTest.testRefreshWithExplicitLowerLevelLockMode":                                                                                                          "6583",
	"org.hibernate.test.locking.PessimisticReadSkipLockedTest.testPostgreSQLSkipLocked":                                                                                                          "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testNoWait":                                                                                                                      "6583",
	"org.hibernate.test.locking.PessimisticWriteLockTimeoutTest.testSkipLocked":                                                                                                                  "6583",
	"org.hibernate.test.locking.PessimisticWriteSkipLockedTest.testPostgreSQLSkipLocked":                                                                                                         "6583",
	"org.hibernate.test.locking.UpgradeSkipLockedTest.testPostgreSQLSkipLocked":                                                                                                                  "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testCriteria":                                                                                                                        "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testHql":                                                                                                                             "6583",
	"org.hibernate.test.locking.paging.PagingAndLockingTest.testNativeSql":                                                                                                                       "6583",
	"org.hibernate.test.locking.warning.LockNoneWarmingTest.testQuerySetLockModeNONEDoNotLogAWarnMessageWhenTheDialectUseFollowOnLockingIsTrue":                                                  "unknown",
	"org.hibernate.test.manytomany.ManyToManyBidirectionalTest.testRemoveOwnerSide":                                                                                                              "unknown",
	"org.hibernate.test.manytomany.ManyToManyTest.testManyToManyWithFormula":                                                                                                                     "unknown",
	"org.hibernate.test.manytomany.ManyToManyWhereTest.testManyToManyWithWhereConditional":                                                                                                       "unknown",
	"org.hibernate.test.manytomany.batchload.BatchedManyToManyTest.testLoadingNonInverseSide":                                                                                                    "unknown",
	"org.hibernate.test.manytomany.ordered.OrderedManyToManyTest.testManyToManyOrdering":                                                                                                         "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testDeleteDetached":                                                                     "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testRemoveAndAddEqualCollection":                                                        "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testRemoveAndAddEqualElement":                                                           "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testRemoveAndAddEqualElementNonKeyModified":                                             "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testRemoveAndAddSameElement":                                                            "unknown",
	"org.hibernate.test.manytomanyassociationclass.compositeid.ManyToManyAssociationClassCompositeIdTest.testRemoveAndAddSameElementNonKeyModified":                                              "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testDeleteDetached":                                                             "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testRemoveAndAddEqualCollection":                                                "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testRemoveAndAddEqualElement":                                                   "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testRemoveAndAddEqualElementNonKeyModified":                                     "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testRemoveAndAddSameElement":                                                    "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.assigned.ManyToManyAssociationClassAssignedIdTest.testRemoveAndAddSameElementNonKeyModified":                                      "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testDeleteDetached":                                                           "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testRemoveAndAddEqualCollection":                                              "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testRemoveAndAddEqualElement":                                                 "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testRemoveAndAddEqualElementNonKeyModified":                                   "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testRemoveAndAddSameElement":                                                  "unknown",
	"org.hibernate.test.manytomanyassociationclass.surrogateid.generated.ManyToManyAssociationClassGeneratedIdTest.testRemoveAndAddSameElementNonKeyModified":                                    "unknown",
	"org.hibernate.test.map.MapIndexFormulaTest.testIndexFormulaMap":                                                                                                                             "unknown",
	"org.hibernate.test.map.MapIndexFormulaTest.testIndexFunctionOnManyToManyMap":                                                                                                                "unknown",
	"org.hibernate.test.map.MapIndexFormulaTest.testSQLQuery":                                                                                                                                    "unknown",
	"org.hibernate.test.mapcompelem.MapCompositeElementTest.testMapCompositeElementWithFormula":                                                                                                  "unknown",
	"org.hibernate.test.mapcompelem.MapCompositeElementTest.testQueryMapCompositeElement":                                                                                                        "unknown",
	"org.hibernate.test.mapelemformula.MapElementFormulaTest.testManyToManyFormula":                                                                                                              "unknown",
	"org.hibernate.test.mapping.AliasTest.testUnderscoreInColumnName":                                                                                                                            "unknown",
	"org.hibernate.test.mapping.MappingReorderedAliasTest.testUnderscoreInColumnName":                                                                                                            "unknown",
	"org.hibernate.test.mapping.cascade.PersistOnLazyCollectionTest.testPersistOnAlreadyPersistentEntityWithUninitializedLazyCollection":                                                         "unknown",
	"org.hibernate.test.mapping.cascade.PersistOnLazyCollectionTest.testPersistOnNewEntityRelatedToAlreadyPersistentEntityWithUninitializedLazyCollection":                                       "unknown",
	"org.hibernate.test.mixed.MixedTest.testMixedInheritance":                                                                                                                                    "26725",
	"org.hibernate.test.nationalized.StringNationalizedTest.testSaveEntityWithNationalizedProperty":                                                                                              "unknown",
	"org.hibernate.test.naturalid.cid.CompositeIdAndNaturalIdTest.testByNaturalId":                                                                                                               "unknown",
	"org.hibernate.test.naturalid.cid.CompositeIdAndNaturalIdTest.testNaturalIdCriteria":                                                                                                         "unknown",
	"org.hibernate.test.naturalid.cid.CompositeIdAndNaturalIdTest.testSave":                                                                                                                      "unknown",
	"org.hibernate.test.naturalid.composite.EmbeddedNaturalIdTest.test":                                                                                                                          "unknown",
	"org.hibernate.test.naturalid.immutable.IdentifierGeneratorWithNaturalIdCacheTest.testNaturalIdCacheEntry":                                                                                   "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableEntityNaturalIdTest.testNaturalIdCheck":                                                                                                     "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableEntityNaturalIdTest.testSaveParentWithDetachedChildren":                                                                                     "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableNaturalIdTest.testNaturalIdCache":                                                                                                           "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableNaturalIdTest.testNaturalIdCheck":                                                                                                           "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableNaturalIdTest.testNaturalIdDeleteUsingCache":                                                                                                "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableNaturalIdTest.testNaturalIdRecreateUsingCache":                                                                                              "unknown",
	"org.hibernate.test.naturalid.immutable.ImmutableNaturalIdTest.testUpdate":                                                                                                                   "unknown",
	"org.hibernate.test.naturalid.immutableentity.ImmutableEntityNaturalIdTest.testImmutableNaturalIdLifecycle":                                                                                  "unknown",
	"org.hibernate.test.naturalid.immutableentity.ImmutableEntityNaturalIdTest.testImmutableNaturalIdLifecycle2":                                                                                 "unknown",
	"org.hibernate.test.naturalid.inheritance.InheritedNaturalIdTest.testIt":                                                                                                                     "unknown",
	"org.hibernate.test.naturalid.inheritance.InheritedNaturalIdTest.testSubclassDeleteNaturalId":                                                                                                "unknown",
	"org.hibernate.test.naturalid.inheritance.InheritedNaturalIdTest.testSubclassModifieablNaturalId":                                                                                            "unknown",
	"org.hibernate.test.naturalid.inheritance.cache.InheritedNaturalIdCacheTest.testLoadingInheritedEntitiesByNaturalId":                                                                         "unknown",
	"org.hibernate.test.naturalid.inheritance.cache.InheritedNaturalIdNoCacheTest.testLoadExtendedByNormal":                                                                                      "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testCacheSynchronizationOnMutation":                                                                                               "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testClear":                                                                                                                        "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testEviction":                                                                                                                     "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testNaturalIdCache":                                                                                                               "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testNaturalIdDeleteUsingCache":                                                                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testNaturalIdRecreateUsingCache":                                                                                                  "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testNonexistentNaturalIdCache":                                                                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testQuerying":                                                                                                                     "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testReattachmentNaturalIdCheck":                                                                                                   "unknown",
	"org.hibernate.test.naturalid.mutable.MutableNaturalIdTest.testReattachmentUnmodifiedNaturalIdCheck":                                                                                         "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testNaturalIdChangeAfterResolveEntityFrom2LCache":                                                  "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testNaturalIdChangedWhileAttached":                                                                 "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testNaturalIdChangedWhileDetached":                                                                 "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testNaturalIdRecachingWhenNeeded":                                                                  "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                               "6583",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdNonStrictReadWriteTest.testbySimpleNaturalIdResolveEntityFrom2LCacheSubClass":                                             "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testChangedNaturalIdCachedAfterTransactionSuccess":                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testChangedNaturalIdNotCachedAfterTransactionFailure":                                                 "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testInsertUpdateEntity_NaturalIdCachedAfterTransactionSuccess":                                        "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testInsertedNaturalIdCachedAfterTransactionSuccess":                                                   "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testInsertedNaturalIdNotCachedAfterTransactionFailure":                                                "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testNaturalIdCacheStatisticsReset":                                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testNaturalIdChangeAfterResolveEntityFrom2LCache":                                                     "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testNaturalIdChangedWhileAttached":                                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testNaturalIdChangedWhileDetached":                                                                    "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testNaturalIdRecachingWhenNeeded":                                                                     "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testReattachementUnmodifiedInstance":                                                                  "6583",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testToMapConversion":                                                                                  "unknown",
	"org.hibernate.test.naturalid.mutable.cached.CachedMutableNaturalIdStrictReadWriteTest.testbySimpleNaturalIdResolveEntityFrom2LCacheSubClass":                                                "unknown",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdNullValueOnPersist":                                                                                                "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testNaturalIdQuerySupportingNullValues":                                                                                         "6583",
	"org.hibernate.test.naturalid.nullable.NullableNaturalIdTest.testUniqueAssociation":                                                                                                          "6583",
	"org.hibernate.test.nonpkassociation.NonPkManyToOneAssociationHbmTest.testHqlWithFetch":                                                                                                      "unknown",
	"org.hibernate.test.nonpkassociation.NonPkManyToOneAssociationTest.testHqlWithFetch":                                                                                                         "unknown",
	"org.hibernate.test.ondelete.OnDeleteTest.testJoinedSubclass":                                                                                                                                "26738",
	"org.hibernate.test.ondelete.toone.ToOneOnDeleteTest.testManyToOne":                                                                                                                          "unknown",
	"org.hibernate.test.ondelete.toone.hbm.ToOneOnDeleteHbmTest.testManyToOne":                                                                                                                   "unknown",
	"org.hibernate.test.ondemandload.LazyLoadingTest.testLazyCollectionLoadingWithClearedSession":                                                                                                "unknown",
	"org.hibernate.test.ondemandload.LazyLoadingTest.testLazyCollectionLoadingWithClosedSession":                                                                                                 "unknown",
	"org.hibernate.test.ondemandload.LazyLoadingTest.testLazyEntityLoadingWithClosedSession":                                                                                                     "unknown",
	"org.hibernate.test.ondemandload.cache.CacheLazyLoadNoTransTest.hibernateInitialize":                                                                                                         "unknown",
	"org.hibernate.test.ondemandload.cache.CacheLazyLoadNoTransTest.testManyToMany":                                                                                                              "unknown",
	"org.hibernate.test.ondemandload.cache.CacheLazyLoadNoTransTest.testOneToMany":                                                                                                               "unknown",
	"org.hibernate.test.onetomany.OneToManyDuplicateInsertionTest.testDuplicateInsertion":                                                                                                        "unknown",
	"org.hibernate.test.onetomany.OneToManyDuplicateInsertionTest.testDuplicateInsertionWithCascadeAndMerge":                                                                                     "unknown",
	"org.hibernate.test.onetomany.OneToManyTest.testManyToManySize":                                                                                                                              "unknown",
	"org.hibernate.test.onetomany.OneToManyTest.testOneToManyLinkTable":                                                                                                                          "5807",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyCacheTest.testOneToManyMoveElement":                                                                                             "unknown",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyCacheTest.testOneToManyMoveElementWithDirtySimpleProperty":                                                                      "unknown",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyCacheTest.testOneToManyReplaceList":                                                                                             "unknown",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyNoCacheTest.testOneToManyMoveElement":                                                                                           "unknown",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyNoCacheTest.testOneToManyMoveElementWithDirtySimpleProperty":                                                                    "unknown",
	"org.hibernate.test.onetomany.RecursiveBidirectionalOneToManyNoCacheTest.testOneToManyReplaceList":                                                                                           "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyCacheTest.testOneToManyMoveElement":                                                                                    "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyCacheTest.testOneToManyMoveElementWithDirtySimpleProperty":                                                             "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyCacheTest.testOneToManyReplaceList":                                                                                    "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyNoCacheTest.testOneToManyMoveElement":                                                                                  "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyNoCacheTest.testOneToManyMoveElementWithDirtySimpleProperty":                                                           "unknown",
	"org.hibernate.test.onetomany.RecursiveVersionedBidirectionalOneToManyNoCacheTest.testOneToManyReplaceList":                                                                                  "unknown",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.breakReferences":                                                                                                    "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities":                                                                                                     "5807",
	"org.hibernate.test.onetomany.inheritance.joined.MappedSuperclassMapTest.lookupEntities_entrySet":                                                                                            "5807",
	"org.hibernate.test.onetomany.inheritance.perclass.MappedSuperclassMapTest.breakReferences":                                                                                                  "unknown",
	"org.hibernate.test.onetomany.inheritance.perclass.MappedSuperclassMapTest.lookupEntities":                                                                                                   "unknown",
	"org.hibernate.test.onetomany.inheritance.perclass.MappedSuperclassMapTest.lookupEntities_entrySet":                                                                                          "unknown",
	"org.hibernate.test.onetomany.inheritance.single.MappedSuperclassMapTest.breakReferences":                                                                                                    "unknown",
	"org.hibernate.test.onetomany.inheritance.single.MappedSuperclassMapTest.lookupEntities":                                                                                                     "unknown",
	"org.hibernate.test.onetomany.inheritance.single.MappedSuperclassMapTest.lookupEntities_entrySet":                                                                                            "unknown",
	"org.hibernate.test.onetoone.bidirectional.BiDirectionalOneToOneFetchTest.testEagerFetchBidirectionalOneToOneWithDirectFetching":                                                             "unknown",
	"org.hibernate.test.onetoone.bidirectional.BiDirectionalOneToOneFetchTest.testFetchBidirectionalOneToOneWithCircularJoinFetch":                                                               "unknown",
	"org.hibernate.test.onetoone.bidirectional.BiDirectionalOneToOneFetchTest.testFetchBidirectionalOneToOneWithOneJoinFetch":                                                                    "unknown",
	"org.hibernate.test.onetoone.formula.OneToOneFormulaTest.testOneToOneEmbeddedCompositeKey":                                                                                                   "unknown",
	"org.hibernate.test.onetoone.formula.OneToOneFormulaTest.testOneToOneFormula":                                                                                                                "unknown",
	"org.hibernate.test.onetoone.formula.OneToOneFormulaTest.testQuery":                                                                                                                          "unknown",
	"org.hibernate.test.onetoone.joined.JoinedSubclassOneToOneTest.testOneToOneOnSubclass":                                                                                                       "unknown",
	"org.hibernate.test.onetoone.link.OneToOneLinkTest.testOneToOneViaAssociationTable":                                                                                                          "unknown",
	"org.hibernate.test.onetoone.nopojo.DynamicMapOneToOneTest.testOneToOneOnSubclass":                                                                                                           "unknown",
	"org.hibernate.test.onetoone.optional.BidirectionalOptionalOneToOneTest.test":                                                                                                                "unknown",
	"org.hibernate.test.onetoone.optional.OptionalOneToOneTest.testOptionalOneToOneRetrieval":                                                                                                    "unknown",
	"org.hibernate.test.onetoone.singletable.DiscrimSubclassOneToOneTest.testOneToOneOnSubclass":                                                                                                 "unknown",
	"org.hibernate.test.ops.CreateTest.testBasic":                                                                                                                                                "unknown",
	"org.hibernate.test.ops.CreateTest.testCreateException":                                                                                                                                      "unknown",
	"org.hibernate.test.ops.CreateTest.testCreateExceptionWithGeneratedId":                                                                                                                       "unknown",
	"org.hibernate.test.ops.CreateTest.testCreateTree":                                                                                                                                           "unknown",
	"org.hibernate.test.ops.CreateTest.testCreateTreeWithGeneratedId":                                                                                                                            "unknown",
	"org.hibernate.test.ops.CreateTest.testNoUpdatesOnCreateVersionedWithCollection":                                                                                                             "unknown",
	"org.hibernate.test.ops.DeleteTest.testDeleteVersionedWithCollectionNoUpdate":                                                                                                                "unknown",
	"org.hibernate.test.ops.DeleteTest.testNoUpdateOnDelete":                                                                                                                                     "unknown",
	"org.hibernate.test.ops.DeleteTest.testNoUpdateOnDeleteWithCollection":                                                                                                                       "unknown",
	"org.hibernate.test.ops.GetLoadTest.testGetAfterDelete":                                                                                                                                      "unknown",
	"org.hibernate.test.ops.GetLoadTest.testGetLoad":                                                                                                                                             "unknown",
	"org.hibernate.test.ops.MergeManagedAndCopiesAllowedTest.testIt":                                                                                                                             "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testCascadeFromDetachedToGT2DirtyRepresentations":                                                                         "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                         "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testCascadeFromTransientToNonDirtyRepresentations":                                                                        "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testNestedDiffBasicProperty":                                                                                              "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testNestedEntityNewerThanTopLevel":                                                                                        "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testNestedManyToOneChangedToNewEntity":                                                                                    "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testNestedManyToOneChangedToNull":                                                                                         "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testNestedValueCollectionWithChangedElements":                                                                             "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelEntityNewerThanNested":                                                                                        "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelManyToOneChangedToNewEntity":                                                                                  "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                             "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelUnidirOneToManyBackrefWithRemovedElement":                                                                     "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelUnidirOneToManyNoBackrefWithNewElement":                                                                       "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopLevelUnidirOneToManyNoBackrefWithRemovedElement":                                                                   "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedLoggedTest.testTopValueCollectionWithChangedElements":                                                                                "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedOrphanDeleteTest.testTopLevelUnidirOneToManyBackrefWithRemovedElement":                                                               "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedOrphanDeleteTest.testTopLevelUnidirOneToManyNoBackrefWithNewElement":                                                                 "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedOrphanDeleteTest.testTopLevelUnidirOneToManyNoBackrefWithRemovedElement":                                                             "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testCascadeFromDetachedToGT2DirtyRepresentations":                                                                               "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                               "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testCascadeFromTransientToNonDirtyRepresentations":                                                                              "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testNestedDiffBasicProperty":                                                                                                    "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testNestedEntityNewerThanTopLevel":                                                                                              "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testNestedManyToOneChangedToNewEntity":                                                                                          "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testNestedManyToOneChangedToNull":                                                                                               "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testNestedValueCollectionWithChangedElements":                                                                                   "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelEntityNewerThanNested":                                                                                              "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelManyToOneChangedToNewEntity":                                                                                        "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                                   "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelUnidirOneToManyBackrefWithRemovedElement":                                                                           "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelUnidirOneToManyNoBackrefWithNewElement":                                                                             "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopLevelUnidirOneToManyNoBackrefWithRemovedElement":                                                                         "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesAllowedTest.testTopValueCollectionWithChangedElements":                                                                                      "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesCustomTest.testMergeMultipleEntityCopiesAllowed":                                                                                            "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesCustomTest.testMergeMultipleEntityCopiesAllowedAndDisallowed":                                                                               "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesDisallowedByDefaultTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                   "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesDisallowedByDefaultTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                       "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesDisallowedTest.testCascadeFromDetachedToNonDirtyRepresentations":                                                                            "unknown",
	"org.hibernate.test.ops.MergeMultipleEntityCopiesDisallowedTest.testTopLevelManyToOneManagedNestedIsDetached":                                                                                "unknown",
	"org.hibernate.test.ops.MergeTest.testDeleteAndMerge":                                                                                                                                        "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeBidiForeignKeyOneToOne":                                                                                                                           "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeBidiPrimayKeyOneToOne":                                                                                                                            "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeDeepTree":                                                                                                                                         "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeDeepTreeWithGeneratedId":                                                                                                                          "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeManaged":                                                                                                                                          "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeManagedInitializedCollection":                                                                                                                     "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeManagedUninitializedCollection":                                                                                                                   "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeManyToManyWithCollectionDeference":                                                                                                                "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeStaleVersionFails":                                                                                                                                "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeTree":                                                                                                                                             "unknown",
	"org.hibernate.test.ops.MergeTest.testMergeTreeWithGeneratedId":                                                                                                                              "unknown",
	"org.hibernate.test.ops.MergeTest.testNoExtraUpdatesOnMerge":                                                                                                                                 "unknown",
	"org.hibernate.test.ops.MergeTest.testNoExtraUpdatesOnMergeVersioned":                                                                                                                        "unknown",
	"org.hibernate.test.ops.MergeTest.testNoExtraUpdatesOnMergeVersionedWithCollection":                                                                                                          "unknown",
	"org.hibernate.test.ops.MergeTest.testNoExtraUpdatesOnMergeWithCollection":                                                                                                                   "unknown",
	"org.hibernate.test.ops.MergeTest.testNoExtraUpdatesOnPersistentMergeVersionedWithCollection":                                                                                                "unknown",
	"org.hibernate.test.ops.MergeTest.testPersistThenMergeInSameTxnWithTimestamp":                                                                                                                "unknown",
	"org.hibernate.test.ops.MergeTest.testPersistThenMergeInSameTxnWithVersion":                                                                                                                  "unknown",
	"org.hibernate.test.ops.MergeTest.testRecursiveMergeTransient":                                                                                                                               "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testEvictThenSaveOrUpdate":                                                                                                                          "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateDeepTree":                                                                                                                           "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateDeepTreeWithGeneratedId":                                                                                                            "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateGot":                                                                                                                                "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateGotWithMutableProp":                                                                                                                 "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateManaged":                                                                                                                            "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateTree":                                                                                                                               "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSaveOrUpdateTreeWithGeneratedId":                                                                                                                "unknown",
	"org.hibernate.test.ops.SaveOrUpdateTest.testSavePersistentEntityWithUpdate":                                                                                                                 "unknown",
	"org.hibernate.test.ops.SimpleOpsTest.testBasicOperations":                                                                                                                                   "unknown",
	"org.hibernate.test.ops.genericApi.BasicGetLoadAccessTest.testIt":                                                                                                                            "6583",
	"org.hibernate.test.ops.genericApi.BasicGetLoadAccessTest.testNullLoadResult":                                                                                                                "unknown",
	"org.hibernate.test.ops.genericApi.BasicGetLoadAccessTest.testNullQueryResult":                                                                                                               "unknown",
	"org.hibernate.test.ops.genericApi.ProxiedGetLoadAccessTest.testIt":                                                                                                                          "6583",
	"org.hibernate.test.ops.multiLoad.MultiLoadSubSelectCollectionTest.testSubselect":                                                                                                            "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testBasicMultiLoad":                                                                                                                          "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testBasicMultiLoadWithManagedAndChecking":                                                                                                    "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testBasicMultiLoadWithManagedAndNoChecking":                                                                                                  "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testDuplicatedRequestedIds":                                                                                                                  "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testMultiLoadClearsBatchFetchQueue":                                                                                                          "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testMultiLoadWithCacheModeIgnore":                                                                                                            "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testNonExistentIdRequest":                                                                                                                    "unknown",
	"org.hibernate.test.ops.multiLoad.MultiLoadTest.testUnflushedDeleteAndThenMultiLoad":                                                                                                         "unknown",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAll":                                                                                                                        "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockAllDelete":                                                                                                                  "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirty":                                                                                                                      "Expected Failure",
	"org.hibernate.test.optlock.OptimisticLockTest.testOptimisticLockDirtyDelete":                                                                                                                "Expected Failure",
	"org.hibernate.test.ordered.OrderByTest.testOrderBy":                                                                                                                                         "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDelete":                                                                                                                                      "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteAfterLock":                                                                                                                             "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteAfterPersist":                                                                                                                          "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteAfterPersistAndFlush":                                                                                                                  "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnAddElementMergeRemoveElementMerge":                                                                                                   "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnDelete":                                                                                                                              "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnMerge":                                                                                                                               "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnMergeRemoveElementMerge":                                                                                                             "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnSaveOrUpdate":                                                                                                                        "unknown",
	"org.hibernate.test.orphan.OrphanTest.testOrphanDeleteOnSaveOrUpdateAfterSerialization":                                                                                                      "unknown",
	"org.hibernate.test.orphan.PropertyRefTest.testDeleteParentWithBidirOrphanDeleteCollectionBasedOnPropertyRef":                                                                                "unknown",
	"org.hibernate.test.orphan.manytomany.ManyToManyOrphanTest.testManyToManyWithCascadeDeleteOrphan":                                                                                            "unknown",
	"org.hibernate.test.orphan.one2one.OneToOneEagerNonOptionalOrphanRemovalTest.testOneToOneLazyNonOptionalOrphanRemoval":                                                                       "unknown",
	"org.hibernate.test.orphan.one2one.OneToOneEagerOrphanRemovalTest.testOneToOneEagerOrphanRemoval":                                                                                            "unknown",
	"org.hibernate.test.orphan.one2one.OneToOneLazyNonOptionalOrphanRemovalTest.testOneToOneLazyNonOptionalOrphanRemoval":                                                                        "unknown",
	"org.hibernate.test.orphan.one2one.OneToOneLazyOrphanRemovalTest.testOneToOneLazyOrphanRemoval":                                                                                              "unknown",
	"org.hibernate.test.orphan.one2one.OneToOneProxyOrphanRemovalTest.testUnproxyOneToOneWithCascade":                                                                                            "unknown",
	"org.hibernate.test.orphan.one2one.fk.bidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                                      "unknown",
	"org.hibernate.test.orphan.one2one.fk.bidirectional.DeleteOneToOneOrphansTest.testReplacedWhileManaged":                                                                                      "unknown",
	"org.hibernate.test.orphan.one2one.fk.bidirectional.multilevelcascade.DeleteMultiLevelOrphansTest.testDirectAndNestedAssociationsOrphanedWhileManaged":                                       "unknown",
	"org.hibernate.test.orphan.one2one.fk.bidirectional.multilevelcascade.DeleteMultiLevelOrphansTest.testDirectAssociationOrphanedWhileManaged":                                                 "unknown",
	"org.hibernate.test.orphan.one2one.fk.bidirectional.multilevelcascade.DeleteMultiLevelOrphansTest.testReplacedDirectAssociationWhileManaged":                                                 "unknown",
	"org.hibernate.test.orphan.one2one.fk.composite.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                                          "unknown",
	"org.hibernate.test.orphan.one2one.fk.composite.DeleteOneToOneOrphansTest.testReplacedWhileManaged":                                                                                          "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.bidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                             "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.bidirectional.DeleteOneToOneOrphansTest.testReplacedWhileManaged":                                                                             "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.bidirectional.multilevelcascade.DeleteMultiLevelOrphansTest.testOrphanedWhileManaged":                                                         "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.bidirectional.multilevelcascade.DeleteMultiLevelOrphansTest.testReplacedWhileManaged":                                                         "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.unidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileDetached":                                                                           "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.unidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                            "unknown",
	"org.hibernate.test.orphan.one2one.fk.reversed.unidirectional.DeleteOneToOneOrphansTest.testReplacedWhileManaged":                                                                            "unknown",
	"org.hibernate.test.orphan.one2one.pk.bidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                                      "unknown",
	"org.hibernate.test.orphan.one2one.pk.unidirectional.DeleteOneToOneOrphansTest.testOrphanedWhileManaged":                                                                                     "unknown",
	"org.hibernate.test.orphan.onetomany.DeleteOneToManyOrphansTest.testOrphanedWhileManaged":                                                                                                    "unknown",
	"org.hibernate.test.orphan.onetomany.DeleteOneToManyOrphansTest.testOrphanedWhileManagedMergeOwner":                                                                                          "unknown",
	"org.hibernate.test.orphan.onetomany.DeleteOneToManyOrphansTest.testReplacedWhileManaged":                                                                                                    "unknown",
	"org.hibernate.test.pagination.DistinctSelectTest.testDistinctSelectWithJoin":                                                                                                                "unknown",
	"org.hibernate.test.pagination.PaginationTest.testLimit":                                                                                                                                     "unknown",
	"org.hibernate.test.pagination.PaginationTest.testLimitOffset":                                                                                                                               "unknown",
	"org.hibernate.test.pagination.PaginationTest.testLimitWithExpreesionAndFetchJoin":                                                                                                           "unknown",
	"org.hibernate.test.pagination.PaginationTest.testOffset":                                                                                                                                    "unknown",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBC":                                                                                                            "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testFunctionWithJDBCByName":                                                                                                      "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernate":                                                                                       "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureNullParameterHibernateWithoutEnablePassingNulls":                                                              "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureOutParameter":                                                                                                 "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testStoredProcedureRefCursor":                                                                                                    "17511",
	"org.hibernate.test.procedure.PostgreSQLStoredProcedureTest.testSysRefCursorAsOutParameter":                                                                                                  "17511",
	"org.hibernate.test.propertyref.basic.BasicPropertiesTest.testProperties":                                                                                                                    "unknown",
	"org.hibernate.test.propertyref.basic.PropertyRefTest.testJoinFetchPropertyRef":                                                                                                              "unknown",
	"org.hibernate.test.propertyref.basic.PropertyRefTest.testManyToManyPropertyRef":                                                                                                             "unknown",
	"org.hibernate.test.propertyref.basic.PropertyRefTest.testNonLazyBagKeyPropertyRef":                                                                                                          "unknown",
	"org.hibernate.test.propertyref.basic.PropertyRefTest.testOneToOnePropertyRef":                                                                                                               "unknown",
	"org.hibernate.test.propertyref.cachedcollections.CachedPropertyRefCollectionTest.testRetrievalOfCachedCollectionWithPropertyRefKey":                                                         "unknown",
	"org.hibernate.test.propertyref.component.complete.CompleteComponentPropertyRefTest.testComponentPropertyRef":                                                                                "unknown",
	"org.hibernate.test.propertyref.component.partial.PartialComponentPropertyRefTest.testComponentPropertyRef":                                                                                  "unknown",
	"org.hibernate.test.propertyref.inheritence.discrim.SubclassPropertyRefTest.testOneToOnePropertyRef":                                                                                         "unknown",
	"org.hibernate.test.propertyref.inheritence.joined.JoinedSubclassPropertyRefTest.testPropertyRefToJoinedSubclass":                                                                            "unknown",
	"org.hibernate.test.propertyref.inheritence.union.UnionSubclassPropertyRefTest.testOneToOnePropertyRef":                                                                                      "unknown",
	"org.hibernate.test.proxy.HibernateUnproxyTest.testInitializedProxyCanBeUnproxied":                                                                                                           "unknown",
	"org.hibernate.test.proxy.HibernateUnproxyTest.testNotHibernateProxyShouldThrowException":                                                                                                    "unknown",
	"org.hibernate.test.proxy.HibernateUnproxyTest.testNotInitializedProxyCanBeUnproxiedWithInitialization":                                                                                      "unknown",
	"org.hibernate.test.proxy.HibernateUnproxyTest.testProxyEquality":                                                                                                                            "unknown",
	"org.hibernate.test.proxy.ProxyReferenceEqualityTest.testProxyFromQuery":                                                                                                                     "unknown",
	"org.hibernate.test.proxy.ProxyTest.testFinalizeFiltered":                                                                                                                                    "unknown",
	"org.hibernate.test.proxy.ProxyTest.testFullyLoadedPCSerialization":                                                                                                                          "unknown",
	"org.hibernate.test.proxy.ProxyTest.testInitializedProxySerializationAfterSessionClosed":                                                                                                     "unknown",
	"org.hibernate.test.proxy.ProxyTest.testLockUninitializedProxy":                                                                                                                              "6583",
	"org.hibernate.test.proxy.ProxyTest.testProxy":                                                                                                                                               "unknown",
	"org.hibernate.test.proxy.ProxyTest.testProxyEviction":                                                                                                                                       "unknown",
	"org.hibernate.test.proxy.ProxyTest.testProxyException":                                                                                                                                      "unknown",
	"org.hibernate.test.proxy.ProxyTest.testProxySerialization":                                                                                                                                  "unknown",
	"org.hibernate.test.proxy.ProxyTest.testProxySerializationAfterSessionClosed":                                                                                                                "unknown",
	"org.hibernate.test.proxy.ProxyTest.testRefreshLockInitializedProxy":                                                                                                                         "6583",
	"org.hibernate.test.proxy.ProxyTest.testSubsequentNonExistentProxyAccess":                                                                                                                    "unknown",
	"org.hibernate.test.proxy.narrow.ProxyNarrowingTest.testNarrowedProxyIsInitializedIfOriginalProxyIsInitialized":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                            "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapNoProjectionMultiAndNullList":                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapNoProjectionNullAndNonNullAliasList":                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                       "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithLazyAssnList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testListElementsProjectionList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMapElementsList":                                                                                            "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMapEntryList":                                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMapKeyList":                                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMapValueList":                                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionUnique":                                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectNewList":                                                                                         "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectNewListList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectNewMapUsingAliasesWithFetchJoinList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneEntityProjectionList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneEntityProjectionUnique":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneNonEntityProjectionList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneNonEntityProjectionUnique":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneSelectNewAliasesList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testSelectNewEntityConstructorList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheIgnoreResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                            "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapNoProjectionMultiAndNullList":                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapNoProjectionNullAndNonNullAliasList":                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                       "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithLazyAssnList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testListElementsProjectionList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMapElementsList":                                                                                            "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMapEntryList":                                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMapKeyList":                                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMapValueList":                                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiEntityProjectionList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiEntityProjectionUnique":                                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectNewList":                                                                                         "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectNewListList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectNewMapUsingAliasesWithFetchJoinList":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneEntityProjectionList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneEntityProjectionUnique":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneNonEntityProjectionList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneNonEntityProjectionUnique":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneSelectNewAliasesList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testSelectNewEntityConstructorList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCacheNormalResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapNoProjectionMultiAndNullList":                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapNoProjectionNullAndNonNullAliasList":                                                           "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                         "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                          "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                         "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithLazyAssnList":                                                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testListElementsProjectionList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMapElementsList":                                                                                               "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMapEntryList":                                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMapKeyList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMapValueList":                                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiEntityProjectionList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiEntityProjectionUnique":                                                                                   "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                       "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                      "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectNewList":                                                                                            "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectNewListList":                                                                                        "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                             "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectNewMapUsingAliasesWithFetchJoinList":                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                                 "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                              "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneEntityProjectionList":                                                                                       "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneEntityProjectionUnique":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneNonEntityProjectionList":                                                                                    "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneNonEntityProjectionUnique":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneSelectNewAliasesList":                                                                                       "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                     "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testSelectNewEntityConstructorList":                                                                                "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                                  "unknown",
	"org.hibernate.test.querycache.CriteriaQueryCachePutResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                                 "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                            "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithLazyAssnList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testListElementsProjectionList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMapElementsList":                                                                                                 "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMapEntryList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMapKeyList":                                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMapValueList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiEntityProjectionUnique":                                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectNewList":                                                                                              "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectNewListList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneEntityProjectionList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneEntityProjectionUnique":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneNonEntityProjectionList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneNonEntityProjectionUnique":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneSelectNewAliasesList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testSelectNewEntityConstructorList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheIgnoreResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                                 "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                            "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithLazyAssnList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testListElementsProjectionList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMapElementsList":                                                                                                 "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMapEntryList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMapKeyList":                                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMapValueList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiEntityProjectionList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiEntityProjectionUnique":                                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectNewList":                                                                                              "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectNewListList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneEntityProjectionList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneEntityProjectionUnique":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneNonEntityProjectionList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneNonEntityProjectionUnique":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneSelectNewAliasesList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testSelectNewEntityConstructorList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCacheNormalResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToBeanDtoLiteralArgList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToBeanDtoMultiArgList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToBeanDtoOneArgList":                                                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToBeanDtoWithNullAliasList":                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToEntityMapMultiAggregatedPropProjectionSingleResult":                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToEntityMapMultiProjectionList":                                                                                "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToEntityMapMultiProjectionWithNullAliasList":                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testAliasToEntityMapOneProjectionList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithAliasedJoinFetchedLazyOneToManySingleElementListHql":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneList":                                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyManyToOneUsingProjectionList":                                                              "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinFetchedLazyOneToManyMultiAndNullElementList":                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyManyToOneListCriteria":                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyManyToOneListHql":                                                                               "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListCriteria":                                                              "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManyMultiAndNullListHql":                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListCriteria":                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithJoinedLazyOneToManySingleElementListHql":                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithLazyAssnList":                                                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithLazyAssnUnique":                                                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithNonLazyManyToOneList":                                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithNonLazyOneToManyUnique":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithSelectFetchedLazyOneToManySingleElementListCriteria":                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testEntityWithUnaliasedJoinFetchedLazyOneToManySingleElementList":                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinListCriteria":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinListHql":                                                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithAliasedJoinedAndOwnerProjectedListHql":                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithOwnerAndAliasedJoinedProjectedListHql":                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithOwnerAndPropProjectedList":                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testJoinWithFetchJoinWithPropAndOwnerProjectedList":                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testListElementsProjectionList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMapElementsList":                                                                                                    "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMapEntryList":                                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMapKeyList":                                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMapValueList":                                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiAggregatedPropProjectionSingleResult":                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiEntityProjectionAliasedList":                                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiEntityProjectionList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiEntityProjectionUnique":                                                                                        "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiProjectionListThenApplyAliasToBean":                                                                            "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectAliasToEntityMapUsingAliasesWithFetchJoinList":                                                           "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectNewList":                                                                                                 "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectNewListList":                                                                                             "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectNewMapUsingAliasesList":                                                                                  "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectNewWithLiteralList":                                                                                      "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testMultiSelectUsingImplicitJoinWithFetchJoinListHql":                                                                   "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneEntityProjectionList":                                                                                            "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneEntityProjectionUnique":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneNonEntityProjectionList":                                                                                         "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneNonEntityProjectionUnique":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneSelectNewAliasesList":                                                                                            "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testOneSelectNewNoAliasesList":                                                                                          "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testSelectNewEntityConstructorList":                                                                                     "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testSelectNewMapUsingAliasesList":                                                                                       "unknown",
	"org.hibernate.test.querycache.HqlQueryCachePutResultTransformerTest.testSingleAggregatedPropProjectionSingleResult":                                                                         "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testDelayedLoad":                                                                                                                               "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testGetByCompositeId":                                                                                                                          "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testHitCacheInSameSession":                                                                                                                     "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testInvalidationFromBulkHQL":                                                                                                                   "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testProjectionCache":                                                                                                                           "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testQueryCacheFetch":                                                                                                                           "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testQueryCacheInvalidation":                                                                                                                    "unknown",
	"org.hibernate.test.querycache.QueryCacheTest.testScalarSQLQuery":                                                                                                                            "unknown",
	"org.hibernate.test.querycache.StructuredQueryCacheTest.testEmbeddedIdInOneToMany":                                                                                                           "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateNativeQuerySetTimeout":                                                                                                           "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateNativeQuerySetTimeoutHint":                                                                                                       "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateQuerySetTimeout":                                                                                                                 "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateQuerySetTimeoutHint":                                                                                                             "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateSQLQuerySetTimeout":                                                                                                              "unknown",
	"org.hibernate.test.querytimeout.QueryTimeOutTest.testCreateSQLQuerySetTimeoutHint":                                                                                                          "unknown",
	"org.hibernate.test.quote.QuoteGlobalTest.testQuoteManytoMany":                                                                                                                               "unknown",
	"org.hibernate.test.quote.QuoteTest.testDoubleQuoteJoinColumn":                                                                                                                               "unknown",
	"org.hibernate.test.quote.QuoteTest.testQuoteManytoMany":                                                                                                                                     "unknown",
	"org.hibernate.test.quote.QuoteTest.testUnionSubclassEntityQuoting":                                                                                                                          "5807",
	"org.hibernate.test.quote.TableGeneratorQuotingTest.testTableGeneratorQuoting":                                                                                                               "16769",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testDetachedCriteria":                                                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableCriteriaReturnsExistingReadOnlyEntity":                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableCriteriaReturnsExistingReadOnlyProxyInit":                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableCriteriaReturnsExistingReadOnlyProxyNotInit":                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableSessionDefaultCriteria":                                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableSessionModifiableCriteria":                                                                                              "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testModifiableSessionReadOnlyCriteria":                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlyCriteriaReturnsExistingModifiableProxyInit":                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlyCriteriaReturnsExistingModifiableProxyNotInit":                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlyCriteriaReturnsModifiableExistingEntity":                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlySessionDefaultCriteria":                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlySessionModifiableCriteria":                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testReadOnlySessionReadOnlyCriteria":                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testScrollCriteria":                                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testSubselect":                                                                                                                        "unknown",
	"org.hibernate.test.readonly.ReadOnlyCriteriaQueryTest.testTwoAliasesCache":                                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testDetachedIsReadOnlyAfterEvictViaLazyInitializer":                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testDetachedIsReadOnlyAfterEvictViaSession":                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testDetachedSetReadOnlyAfterEvictViaLazyInitializer":                                                                                          "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testDetachedSetReadOnlyAfterEvictViaSession":                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testIsReadOnlyAfterSessionClosed":                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testIsReadOnlyAfterSessionClosedViaLazyInitializer":                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testModifiableViaLazyInitializerAfterInit":                                                                                                    "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testModifiableViaLazyInitializerBeforeInit":                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testModifiableViaSessionBeforeInit":                                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testModifiableViaSessionBeforeInitByModifiableQuery":                                                                                          "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testModifiableViaSessionBeforeInitByReadOnlyQuery":                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyChangedEvictedUpdate":                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyDelete":                                                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyEntityMergeDetachedProxyWithChange":                                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyInitToModifiableModifiedIsUpdated":                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyModifiedUpdate":                                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyProxyInitMergeDetachedEntityWithChange":                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyProxyInitMergeDetachedProxyWithChange":                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyProxyMergeDetachedEntityWithChange":                                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyProxyMergeDetachedProxyWithChange":                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyRefresh":                                                                                                                          "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyRefreshDeleted":                                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyRefreshDetached":                                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyToModifiableInitWhenModifiedIsUpdated":                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaLazyInitializerAfterInit":                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaLazyInitializerBeforeInit":                                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaLazyInitializerDoesNotInit":                                                                                                    "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaLazyInitializerNoChangeAfterInit":                                                                                              "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaSessionBeforeInit":                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaSessionBeforeInitByModifiableQuery":                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaSessionBeforeInitByReadOnlyQuery":                                                                                              "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaSessionDoesNotInit":                                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testReadOnlyViaSessionNoChangeAfterInit":                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetClosedSessionInLazyInitializer":                                                                                                        "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetModifiableBetweenTwoTransactionsSameSession":                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetReadOnlyAfterSessionClosed":                                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetReadOnlyAfterSessionClosedViaLazyInitializer":                                                                                          "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetReadOnlyBetweenTwoTransactionsSameSession":                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyProxyTest.testSetReadOnlyInTwoTransactionsSameSession":                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultModifiableWithFilterCollectionEntities":                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultModifiableWithQueryForEntity":                                                                                         "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultModifiableWithReadOnlyFilterCollectionEntities":                                                                       "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultModifiableWithReadOnlyQueryForCollectionEntities":                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultModifiableWithReadOnlyQueryForEntity":                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultReadOnlyWithFilterCollectionEntities":                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultReadOnlyWithModifiableFilterCollectionEntities":                                                                       "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultReadOnlyWithModifiableQueryForEntity":                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testDefaultReadOnlyWithQueryForEntity":                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingModifiableAfterSetSessionReadOnly":                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingReadOnlyAfterSetSessionModifiable":                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingReadOnlyAfterSetSessionModifiableExisting":                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingReadOnlyAfterSetSessionModifiableExistingEntityReadOnly":                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingReadOnlyAfterSetSessionModifiableExistingProxyReadOnly":                                                              "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionLazyNonLazyTest.testExistingReadOnlyAfterSetSessionModifiableProxyExisting":                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testMergeWithReadOnlyEntity":                                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testMergeWithReadOnlyProxy":                                                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testModifiableModeWithExistingReadOnlyEntity":                                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testModifiableSessionDefaultQueryReadOnlySessionIterate":                                                                                    "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testModifiableSessionDefaultQueryReadOnlySessionScroll":                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testModifiableSessionReadOnlyQueryIterate":                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testModifiableSessionReadOnlyQueryScroll":                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testQueryModifiableIterate":                                                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testQueryModifiableScroll":                                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testQueryReadOnlyIterate":                                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testQueryReadOnlyScroll":                                                                                                                    "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyDelete":                                                                                                                         "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyGetModifyAndDelete":                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyModeWithExistingModifiableEntity":                                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyOnProxies":                                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyOnTextType":                                                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyProxyRefresh":                                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyProxyRefreshDetached":                                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyRefresh":                                                                                                                        "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlyRefreshDetached":                                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlySessionDefaultQueryIterate":                                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlySessionDefaultQueryScroll":                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlySessionModifiableQueryIterate":                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlySessionTest.testReadOnlySessionModifiableQueryScroll":                                                                                                   "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testMergeWithReadOnlyEntity":                                                                                                                       "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testModifiableModeWithExistingReadOnlyEntity":                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyDelete":                                                                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyGetModifyAndDelete":                                                                                                                    "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyMode":                                                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyModeAutoFlushOnQuery":                                                                                                                  "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyModeWithExistingModifiableEntity":                                                                                                      "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyOnProxies":                                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyOnTextType":                                                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyRefresh":                                                                                                                               "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testReadOnlyRefreshDetached":                                                                                                                       "unknown",
	"org.hibernate.test.readonly.ReadOnlyTest.testSaveReadOnlyModifyInSaveTransaction":                                                                                                           "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testAddNewChildToReadOnlyParent":                                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testAddNewParentToReadOnlyChild":                                                                                                     "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testGetChildMakeReadOnlyThenMergeDetachedChildWithNewParent":                                                                         "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testGetParentMakeReadOnlyThenMergeDetachedParentWithNewChildC":                                                                       "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testMergeDetachedChildWithNewParentCommitWithReadOnlyChild":                                                                          "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testMergeDetachedParentWithNewChildCommitWithReadOnlyParent":                                                                         "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testMergeUnchangedDetachedParentChildren":                                                                                            "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testSetReadOnlyTrueAndFalse":                                                                                                         "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testUpdateChildWithNewParentCommitWithReadOnlyChild":                                                                                 "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testUpdateParentWithNewChildCommitWithReadOnlyParent":                                                                                "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testUpdateSetModifiable":                                                                                                             "unknown",
	"org.hibernate.test.readonly.ReadOnlyVersionedNodesTest.testUpdateSetReadOnlyTwice":                                                                                                          "unknown",
	"org.hibernate.test.reattachment.CollectionReattachmentTest.testUpdateOwnerAfterClear":                                                                                                       "unknown",
	"org.hibernate.test.reattachment.CollectionReattachmentTest.testUpdateOwnerAfterEvict":                                                                                                       "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testIterateWithClearBottomOfLoop":                                                                                                     "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testIterateWithClearTopOfLoop":                                                                                                        "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testIterateWithEvictBottomOfLoop":                                                                                                     "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testIterateWithEvictTopOfLoop":                                                                                                        "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testRemoveAndReattachProxyEntity":                                                                                                     "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testUpdateAfterClear":                                                                                                                 "unknown",
	"org.hibernate.test.reattachment.ProxyReattachmentTest.testUpdateAfterEvict":                                                                                                                 "unknown",
	"org.hibernate.test.refresh.RefreshDetachedInstanceWhenIsAllowedTest.testRefreshDetachedInstance":                                                                                            "unknown",
	"org.hibernate.test.refresh.RefreshDetachedInstanceWhenIsNotAllowedTest.testRefreshDetachedInstance":                                                                                         "unknown",
	"org.hibernate.test.refresh.RefreshUsingEntityNameTest.testRefreshUsingEntityName":                                                                                                           "unknown",
	"org.hibernate.test.resource.transaction.jdbc.autocommit.PostgreSQLSkipAutoCommitTest.test":                                                                                                  "unknown",
	"org.hibernate.test.resulttransformer.ResultTransformerTest.testResultTransformerIsAppliedToScrollableResults":                                                                               "unknown",
	"org.hibernate.test.schematools.TestExtraPhysicalTableTypes.testAddOneExtraPhysicalTableType":                                                                                                "unknown",
	"org.hibernate.test.schematools.TestExtraPhysicalTableTypes.testAddingMultipleExtraPhysicalTableTypes":                                                                                       "unknown",
	"org.hibernate.test.schematools.TestExtraPhysicalTableTypes.testExtraPhysicalTableTypesPropertyEmptyStringValue":                                                                             "unknown",
	"org.hibernate.test.schematools.TestExtraPhysicalTableTypes.testNoExtraPhysicalTabeTypesProperty":                                                                                            "unknown",
	"org.hibernate.test.schemaupdate.ColumnNamesTest.testSchemaUpdateWithQuotedColumnNames[0]":                                                                                                   "unknown",
	"org.hibernate.test.schemaupdate.ColumnNamesTest.testSchemaUpdateWithQuotedColumnNames[1]":                                                                                                   "unknown",
	"org.hibernate.test.schemaupdate.CommentGenerationTest.testSchemaUpdateScriptGeneration":                                                                                                     "unknown",
	"org.hibernate.test.schemaupdate.MigrationTest.testIndexCreationViaSchemaUpdate":                                                                                                             "31761",
	"org.hibernate.test.schemaupdate.MigrationTest.testSameTableNameDifferentExplicitSchemas":                                                                                                    "unknown",
	"org.hibernate.test.schemaupdate.MigrationTest.testSimpleColumnAddition":                                                                                                                     "unknown",
	"org.hibernate.test.schemaupdate.PostgreSQLMultipleSchemaSequenceTest.test":                                                                                                                  "26443",
	"org.hibernate.test.schemaupdate.QuotedTableNameSchemaUpdateTest.testSchemaUpdateWithQuotedTableName":                                                                                        "unknown",
	"org.hibernate.test.schemaupdate.QuotedTableNameWithForeignKeysSchemaUpdateTest.testGeneratingUpdateScript":                                                                                  "unknown",
	"org.hibernate.test.schemaupdate.QuotedTableNameWithForeignKeysSchemaUpdateTest.testUpdateExistingSchema":                                                                                    "unknown",
	"org.hibernate.test.schemaupdate.SchemaExportTest.testBothType":                                                                                                                              "unknown",
	"org.hibernate.test.schemaupdate.SchemaExportTest.testCreateAndDrop":                                                                                                                         "unknown",
	"org.hibernate.test.schemaupdate.SchemaExportTest.testCreateAndDropOnlyType":                                                                                                                 "unknown",
	"org.hibernate.test.schemaupdate.SchemaExportWithIndexAndDefaultSchema.shouldCreateIndex":                                                                                                    "unknown",
	"org.hibernate.test.schemaupdate.SchemaMigrationTargetScriptCreationTest.classMethod":                                                                                                        "unknown",
	"org.hibernate.test.schemaupdate.SchemaMigratorHaltOnErrorTest.testHaltOnError":                                                                                                              "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateDelimiterTest.testSchemaUpdateApplyDelimiterToGeneratedSQL":                                                                                     "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateGeneratingOnlyScriptFileTest.testSchemaUpdateScriptGeneration":                                                                                  "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateHaltOnErrorTest.testHaltOnError":                                                                                                                "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateProceedOnErrorTest.testHaltOnError":                                                                                                             "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateTableBackedSequenceTest.testCreateTableOnUpdate":                                                                                                "unknown",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[0]":                                                                                                          "26738",
	"org.hibernate.test.schemaupdate.SchemaUpdateTest.testSchemaUpdateAndValidation[1]":                                                                                                          "26738",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithFunctionIndexTest.testUpdateSchema":                                                                                                         "9682",
	"org.hibernate.test.schemaupdate.SchemaUpdateWithViewsTest.testUpdateSchema":                                                                                                                 "24897",
	"org.hibernate.test.schemaupdate.TestFkUpdating.testUpdate":                                                                                                                                  "unknown",
	"org.hibernate.test.schemaupdate.foreignkeys.SchemaUpdateWithKeywordAutoQuotingEnabledTest.testUpdate":                                                                                       "unknown",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testImprovedSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":                  "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaMigrationForeignKeysAreGeneratedAfterAllTheTablesAreCreated":                          "26443",
	"org.hibernate.test.schemaupdate.foreignkeys.crossschema.CrossSchemaForeignKeyGenerationTest.testSchemaUpdateDoesNotFailResolvingCrossSchemaForeignKey":                                      "26443",
	"org.hibernate.test.schemaupdate.idbag.IdBagSequenceTest.testIdBagSequenceGeneratorIsCreated":                                                                                                "unknown",
	"org.hibernate.test.schemaupdate.inheritance.ForeignKeyNameTest.testJoinedSubclassForeignKeyNameIsNotAutoGeneratedWhenProvided":                                                              "unknown",
	"org.hibernate.test.schemaupdate.inheritance.ForeignKeyNameTest.testSubclassForeignKeyNameIsNotAutoGeneratedWhenProvided":                                                                    "unknown",
	"org.hibernate.test.schemaupdate.inheritance.hhh_x.InheritanceSchemaUpdateTest.testBidirectionalOneToManyReferencingRootEntity":                                                              "unknown",
	"org.hibernate.test.schemaupdate.uniqueconstraint.UniqueConstraintDropTest.testUniqueConstraintIsGenerated":                                                                                  "unknown",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[0]":                                                                                                            "16769",
	"org.hibernate.test.schemavalidation.LongVarcharValidationTest.testValidation[1]":                                                                                                            "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[0]":                                                                                                                "16769",
	"org.hibernate.test.schemavalidation.NumericValidationTest.testValidation[1]":                                                                                                                "16769",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingGroupedSchemaValidator":                                                                                              "10028",
	"org.hibernate.test.schemavalidation.ViewValidationTest.testSynonymUsingIndividuallySchemaValidator":                                                                                         "10028",
	"org.hibernate.test.schemavalidation.matchingtablenames.TableNamesWithUnderscoreTest.testSchemaValidationDoesNotFailDueToAMoreThanOneTableFound":                                             "16769",
	"org.hibernate.test.sorted.SortTest.testOrderBy":                                                                                                                                             "unknown",
	"org.hibernate.test.sorted.SortTest.testSortNatural":                                                                                                                                         "unknown",
	"org.hibernate.test.sql.NativeQueryScrollableResults.testSetParameters":                                                                                                                      "unknown",
	"org.hibernate.test.sql.autodiscovery.AutoDiscoveryTest.testAutoDiscoveryWithDuplicateColumnLabels":                                                                                          "unknown",
	"org.hibernate.test.sql.autodiscovery.AutoDiscoveryTest.testDialectGetColumnAliasExtractor":                                                                                                  "unknown",
	"org.hibernate.test.sql.autodiscovery.AutoDiscoveryTest.testSqlQueryAutoDiscovery":                                                                                                           "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testAddJoinForManyToMany":                                                                                                            "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testAutoDetectAliasing":                                                                                                              "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testExplicitReturnAPI":                                                                                                               "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testImageTypeInSQLQuery":                                                                                                             "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testManualSynchronization":                                                                                                           "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testMappedAliasStrategy":                                                                                                             "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testMixAndMatchEntityScalar":                                                                                                         "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testRegisteredNamedSQLQueryWithScalar":                                                                                               "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testResultSetMappingDefinition":                                                                                                      "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testSQLQueryInterface":                                                                                                               "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testScalarValues":                                                                                                                    "unknown",
	"org.hibernate.test.sql.hand.query.NativeSQLQueriesTest.testTextTypeInSQLQuery":                                                                                                              "unknown",
	"org.hibernate.test.sql.hand.quotedidentifiers.NativeSqlAndQuotedIdentifiersTest.testBasicEntityMapping":                                                                                     "unknown",
	"org.hibernate.test.sql.hand.quotedidentifiers.NativeSqlAndQuotedIdentifiersTest.testCompleteScalarDiscovery":                                                                                "unknown",
	"org.hibernate.test.sql.hand.quotedidentifiers.NativeSqlAndQuotedIdentifiersTest.testExpandedEntityMapping":                                                                                  "unknown",
	"org.hibernate.test.sql.hand.quotedidentifiers.NativeSqlAndQuotedIdentifiersTest.testPartialScalarDiscovery":                                                                                 "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testSelectAnotherEntityWithNoSyncSpaces":                                                                                   "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testSelectCachedEntityWithNoSyncSpaces":                                                                                    "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testUpdateAnotherEntityWithNoSyncSpaces":                                                                                   "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testUpdateAnotherEntityWithSyncSpaces":                                                                                     "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testUpdateCachedEntityWithNoSyncSpaces":                                                                                    "unknown",
	"org.hibernate.test.sql.syncSpace.NativeQuerySyncSpaceCachingTest.testUpdateCachedEntityWithSyncSpaces":                                                                                      "unknown",
	"org.hibernate.test.stateless.StatelessSessionNativeQueryInsertTest.testInsertInStatelessSession":                                                                                            "unknown",
	"org.hibernate.test.stateless.StatelessSessionQueryTest.testCriteria":                                                                                                                        "unknown",
	"org.hibernate.test.stateless.StatelessSessionQueryTest.testCriteriaWithSelectFetchMode":                                                                                                     "unknown",
	"org.hibernate.test.stateless.StatelessSessionQueryTest.testHQL":                                                                                                                             "unknown",
	"org.hibernate.test.stateless.StatelessSessionTest.testCreateUpdateReadDelete":                                                                                                               "unknown",
	"org.hibernate.test.stateless.StatelessSessionTest.testHqlBulk":                                                                                                                              "unknown",
	"org.hibernate.test.stateless.StatelessSessionTest.testInitId":                                                                                                                               "unknown",
	"org.hibernate.test.stateless.StatelessSessionTest.testRefresh":                                                                                                                              "unknown",
	"org.hibernate.test.stateless.fetching.StatelessSessionFetchingTest.testDynamicFetch":                                                                                                        "unknown",
	"org.hibernate.test.stateless.insert.StatelessSessionInsertTest.testInsertWithForeignKey":                                                                                                    "unknown",
	"org.hibernate.test.stats.SessionStatsTest.testSessionStatistics":                                                                                                                            "unknown",
	"org.hibernate.test.stats.StatsTest.testQueryStatGathering":                                                                                                                                  "unknown",
	"org.hibernate.test.stream.basic.BasicStreamTest.basicStreamTest":                                                                                                                            "unknown",
	"org.hibernate.test.stream.basic.BasicStreamTest.testQueryStream":                                                                                                                            "unknown",
	"org.hibernate.test.stream.basic.BasicStreamTest.testTupleStream":                                                                                                                            "unknown",
	"org.hibernate.test.stream.basic.JpaStreamTest.testQueryStream":                                                                                                                              "unknown",
	"org.hibernate.test.subclassfilter.DiscrimSubclassFilterTest.testFiltersWithSubclass":                                                                                                        "unknown",
	"org.hibernate.test.subclassfilter.JoinedSubclassFilterTest.testFiltersWithJoinedSubclass":                                                                                                   "5807",
	"org.hibernate.test.subclassfilter.UnionSubclassFilterTest.testFiltersWithUnionSubclass":                                                                                                     "unknown",
	"org.hibernate.test.subselect.CompositeIdTypeBindingTest.testCompositeTypeBinding":                                                                                                           "unknown",
	"org.hibernate.test.subselect.SetSubselectTest.testSubselect":                                                                                                                                "unknown",
	"org.hibernate.test.subselect.SubselectTest.testCustomColumnReadAndWrite":                                                                                                                    "31673",
	"org.hibernate.test.subselect.SubselectTest.testEntitySubselect":                                                                                                                             "31673",
	"org.hibernate.test.subselect.join.SubselectInJoinedTableTest.testSubselectInJoinedTable":                                                                                                    "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchCollectionFromBatchTest.testMultiSubselectFetchSamePersisterQueryList":                                                                      "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchCollectionFromBatchTest.testSubselectFetchFromEntityBatch":                                                                                  "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchCollectionFromBatchTest.testSubselectFetchFromQueryList":                                                                                    "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testManyToManyCriteriaJoin":                                                                                                            "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testSubselectFetchCriteria":                                                                                                            "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testSubselectFetchHql":                                                                                                                 "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testSubselectFetchNamedParam":                                                                                                          "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testSubselectFetchPosParam":                                                                                                            "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchTest.testSubselectFetchWithLimit":                                                                                                           "unknown",
	"org.hibernate.test.subselectfetch.SubselectFetchWithFormulaTest.checkSubselectWithFormula":                                                                                                  "unknown",
	"org.hibernate.test.temporal.TimePropertyTest.testTimeAsDate":                                                                                                                                "unknown",
	"org.hibernate.test.temporal.TimestampPropertyTest.testTime":                                                                                                                                 "unknown",
	"org.hibernate.test.temporal.TimestampPropertyTest.testTimeGeneratedByColumnDefault":                                                                                                         "unknown",
	"org.hibernate.test.ternary.TernaryTest.testIndexRelatedFunctions":                                                                                                                           "unknown",
	"org.hibernate.test.ternary.TernaryTest.testTernary":                                                                                                                                         "unknown",
	"org.hibernate.test.timestamp.JdbcTimeCustomTimeZoneTest.testTimeZone":                                                                                                                       "unknown",
	"org.hibernate.test.timestamp.JdbcTimeDefaultTimeZoneTest.testTimeZone":                                                                                                                      "unknown",
	"org.hibernate.test.timestamp.JdbcTimestampCustomSessionLevelTimeZoneTest.testTimeZone":                                                                                                      "unknown",
	"org.hibernate.test.timestamp.JdbcTimestampCustomTimeZoneTest.testTimeZone":                                                                                                                  "unknown",
	"org.hibernate.test.timestamp.JdbcTimestampDefaultTimeZoneTest.testTimeZone":                                                                                                                 "unknown",
	"org.hibernate.test.timestamp.JdbcTimestampUTCTimeZoneTest.testTimeZone":                                                                                                                     "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithDefaultUTCTimeZoneTest.testTimeZone":                                                                                                          "3781",
	"org.hibernate.test.timestamp.JdbcTimestampWithoutUTCTimeZoneTest.testTimeZone":                                                                                                              "3781",
	"org.hibernate.test.timestamp.TimestampTest.testComponent":                                                                                                                                   "unknown",
	"org.hibernate.test.timestamp.TimestampTest.testUpdateFalse":                                                                                                                                 "unknown",
	"org.hibernate.test.tm.AfterCompletionTest.testAfterCompletionCallbackExecutedAfterTransactionTimeout":                                                                                       "unknown",
	"org.hibernate.test.tm.BeforeCompletionFailureTest.testUniqueConstraintViolationDuringManagedFlush":                                                                                          "unknown",
	"org.hibernate.test.tm.CMTTest.testCMT":                                                                                                                                                      "unknown",
	"org.hibernate.test.tm.CMTTest.testConcurrent":                                                                                                                                               "unknown",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedDirtyQueries":                                                                                                                             "Expected Failure",
	"org.hibernate.test.tm.CMTTest.testConcurrentCachedQueries":                                                                                                                                  "pass?",
	"org.hibernate.test.tm.CMTTest.testCurrentSessionWithIterate":                                                                                                                                "unknown",
	"org.hibernate.test.tm.CMTTest.testCurrentSessionWithScroll":                                                                                                                                 "unknown",
	"org.hibernate.test.tool.schema.SchemaToolTransactionHandlingTest.testDropCreateDropInExistingJtaTransaction":                                                                                "unknown",
	"org.hibernate.test.tool.schema.SchemaToolTransactionHandlingTest.testValidateInExistingJtaTransaction":                                                                                      "16769",
	"org.hibernate.test.type.BinaryTypeTest.testByteArrayStringRepresentation":                                                                                                                   "unknown",
	"org.hibernate.test.type.Java8DateTimeTests.basicTests":                                                                                                                                      "unknown",
	"org.hibernate.test.type.LocalDateTest.testLocalDate":                                                                                                                                        "unknown",
	"org.hibernate.test.type.LongListTypeContributorTest.testParameterRegisterredCollection":                                                                                                     "unknown",
	"org.hibernate.test.type.OffsetDateTimeTest.testOffsetDateTimeWithHoursZoneOffset":                                                                                                           "unknown",
	"org.hibernate.test.type.OffsetDateTimeTest.testOffsetDateTimeWithUTCZoneOffset":                                                                                                             "unknown",
	"org.hibernate.test.type.OffsetDateTimeTest.testRetrievingEntityByOffsetDateTime":                                                                                                            "unknown",
	"org.hibernate.test.type.TimeAndTimestampTest.test":                                                                                                                                          "unknown",
	"org.hibernate.test.type.ZonedDateTimeTest.testRetrievingEntityByZoneDateTime":                                                                                                               "unknown",
	"org.hibernate.test.type.ZonedDateTimeTest.testZoneDateTimeWithHoursZoneOffset":                                                                                                              "unknown",
	"org.hibernate.test.type.ZonedDateTimeTest.testZoneDateTimeWithUTCZoneOffset":                                                                                                                "unknown",
	"org.hibernate.test.type.contributor.ArrayTypeContributorTest.test":                                                                                                                          "unknown",
	"org.hibernate.test.typedescriptor.ByteTest.testByteDataPersistenceAndRetrieval":                                                                                                             "unknown",
	"org.hibernate.test.typedescriptor.CharInNativeQueryTest.testNativeQuery":                                                                                                                    "unknown",
	"org.hibernate.test.typedmanytoone.TypedManyToOneTest.testCreateQuery":                                                                                                                       "unknown",
	"org.hibernate.test.typedmanytoone.TypedManyToOneTest.testCreateQueryNull":                                                                                                                   "unknown",
	"org.hibernate.test.typedonetoone.TypedOneToOneTest.testCreateQuery":                                                                                                                         "unknown",
	"org.hibernate.test.typedonetoone.TypedOneToOneTest.testCreateQueryNull":                                                                                                                     "unknown",
	"org.hibernate.test.typeoverride.TypeOverrideTest.testInsert":                                                                                                                                "unknown",
	"org.hibernate.test.typeoverride.TypeOverrideTest.testRegisteredFunction":                                                                                                                    "unknown",
	"org.hibernate.test.typeparameters.TypeParameterTest.testLoading":                                                                                                                            "unknown",
	"org.hibernate.test.typeparameters.TypeParameterTest.testSave":                                                                                                                               "unknown",
	"org.hibernate.test.unconstrained.UnconstrainedTest.testUnconstrained":                                                                                                                       "unknown",
	"org.hibernate.test.unconstrained.UnconstrainedTest.testUnconstrainedNoCache":                                                                                                                "unknown",
	"org.hibernate.test.unconstrained.UnconstrainedTest.testUnconstrainedOuterJoinFetch":                                                                                                         "unknown",
	"org.hibernate.test.unidir.BackrefTest.testBackRef":                                                                                                                                          "unknown",
	"org.hibernate.test.unidir.BackrefTest.testBackRefToProxiedEntityOnMerge":                                                                                                                    "unknown",
	"org.hibernate.test.unidir.UnidirectionalOneToManyNonPkJoinColumnTest.test":                                                                                                                  "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testBulkOperationsWithDifferentConnections":                                                                                              "5807",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testNestedUnionedSubclasses":                                                                                                             "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testUnionSubclass":                                                                                                                       "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testUnionSubclassCollection":                                                                                                             "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testUnionSubclassFetchMode":                                                                                                              "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testUnionSubclassManyToOne":                                                                                                              "unknown",
	"org.hibernate.test.unionsubclass.UnionSubclassTest.testUnionSubclassOneToMany":                                                                                                              "unknown",
	"org.hibernate.test.unionsubclass.alias.SellCarTest.testSellCar":                                                                                                                             "unknown",
	"org.hibernate.test.unionsubclass2.UnionSubclassTest.testCustomColumnReadAndWrite":                                                                                                           "unknown",
	"org.hibernate.test.unionsubclass2.UnionSubclassTest.testQuerySubclassAttribute":                                                                                                             "unknown",
	"org.hibernate.test.unionsubclass2.UnionSubclassTest.testUnionSubclass":                                                                                                                      "unknown",
	"org.hibernate.test.unionsubclass3.UnionSubclassTest.testUnionSubclassClassResults":                                                                                                          "unknown",
	"org.hibernate.test.unionsubclass3.UnionSubclassTest.typeOfLeafTPC":                                                                                                                          "unknown",
	"org.hibernate.test.uniquekey.NaturalIdCachingTest.test":                                                                                                                                     "unknown",
	"org.hibernate.test.version.InstantVersionTest.testInstantUsageAsVersion":                                                                                                                    "unknown",
	"org.hibernate.test.version.LocalDateTimeVersionTest.testInstantUsageAsVersion":                                                                                                              "unknown",
	"org.hibernate.test.version.OffsetDateTimeVersionTest.testInstantUsageAsVersion":                                                                                                             "unknown",
	"org.hibernate.test.version.VersionTest.testCollectionNoVersion":                                                                                                                             "unknown",
	"org.hibernate.test.version.VersionTest.testCollectionVersion":                                                                                                                               "unknown",
	"org.hibernate.test.version.VersionTest.testVersionShortCircuitFlush":                                                                                                                        "unknown",
	"org.hibernate.test.version.ZonedDateTimeVersionTest.testInstantUsageAsVersion":                                                                                                              "unknown",
	"org.hibernate.test.version.db.DbVersionTest.testCollectionNoVersion":                                                                                                                        "unknown",
	"org.hibernate.test.version.db.DbVersionTest.testCollectionVersion":                                                                                                                          "unknown",
	"org.hibernate.test.version.mappedsuperclass.HbmMappingMappedSuperclassWithVersionTest.testMetamodelContainsHbmVersion":                                                                      "unknown",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                        "12123",
	"org.hibernate.test.where.annotations.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                                           "12123",
	"org.hibernate.test.where.annotations.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                         "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                        "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdNotFoundWhereTest.testInitializeFromUniqueAssociationTable":                                                                           "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                                "12123",
	"org.hibernate.test.where.hbm.LazyManyToManyNonUniqueIdWhereTest.testInitializeFromUniqueAssociationTable":                                                                                   "12123",
	"org.hibernate.test.where.hbm.LazyOneToManyNonUniqueIdWhereTest.testInitializeFromNonUniqueAssociationTable":                                                                                 "unknown",
	"org.hibernate.test.where.hbm.WhereTest.testCriteria":                                                                                                                                        "unknown",
	"org.hibernate.test.where.hbm.WhereTest.testHql":                                                                                                                                             "unknown",
	"org.hibernate.test.where.hbm.WhereTest.testHqlWithFetch":                                                                                                                                    "unknown",
	"org.hibernate.test.where.hbm.WhereTest.testNativeQuery":                                                                                                                                     "unknown",
}
