//
// Syd: rock-solid application kernel
// src/kernel/net/connect.rs: connect(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::OwnedFd;

use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::socket::{AddressFamily, SockaddrLike, SockaddrStorage},
};

use crate::{
    cookie::safe_connect,
    fs::{get_nonblock, has_recv_timeout},
    hook::UNotifyEventRequest,
    kernel::net::handle_safe_bind,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn handle_connect(
    fd: OwnedFd,
    addr: &SockaddrStorage,
    request: &UNotifyEventRequest,
    allow_safe_bind: bool,
) -> Result<ScmpNotifResp, Errno> {
    // SAFETY: Record blocking call so it can get invalidated.
    let req = request.scmpreq;
    let is_blocking = if !get_nonblock(&fd)? {
        let ignore_restart = has_recv_timeout(&fd)?;

        // Record the blocking call.
        request.cache.add_sys_block(req, ignore_restart)?;

        true
    } else {
        false
    };

    let result = if let Some(addr) = addr.as_sockaddr_in() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_sockaddr_in6() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_alg_addr() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_link_addr() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_netlink_addr() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_vsock_addr() {
        safe_connect(&fd, addr)
    } else if let Some(addr) = addr.as_unix_addr() {
        safe_connect(&fd, addr)
    } else {
        safe_connect(&fd, addr)
    }
    .map(|_| request.return_syscall(0));

    // Remove invalidation record unless interrupted.
    if is_blocking {
        request
            .cache
            .del_sys_block(req.id, matches!(result, Err(Errno::EINTR)))?;
    }

    if allow_safe_bind
        && result.is_ok()
        && matches!(
            addr.family(),
            Some(AddressFamily::Inet | AddressFamily::Inet6)
        )
    {
        // Handle allow_safe_bind.
        // Ignore errors as connect has already succeeded.
        let _ = handle_safe_bind(request, &fd);
    } else if addr.family() == Some(AddressFamily::Unix) {
        // Handle SO_PASSCRED inode tracking.
        // Ignore errors as connect has already succeeded.
        let _ = request.add_unix(&fd, request.scmpreq.pid(), None);
    }

    result
}
