//go:build fips

package fips

import (
	"testing"

	"github.com/stretchr/testify/require"
	"golang.org/x/crypto/ssh"
)

func TestSupportedAlgorithms_FipsEnabled(t *testing.T) {
	algorithms := SupportedAlgorithms()

	t.Run("ContainsFIPSCompliantAlgorithms", func(t *testing.T) {
		// Ciphers - should contain all AES algorithms
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES128GCM, "Should contain AES-128-GCM")
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES256GCM, "Should contain AES-256-GCM")
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES128CTR, "Should contain AES-128-CTR")
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES192CTR, "Should contain AES-192-CTR")
		require.Contains(t, algorithms.Ciphers, ssh.CipherAES256CTR, "Should contain AES-256-CTR")

		// Key Exchanges - should contain ECDH P-curves and DH
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP256, "Should contain ECDH P-256")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP384, "Should contain ECDH P-384")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP521, "Should contain ECDH P-521")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeDH14SHA256, "Should contain DH Group 14 SHA-256")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeDH16SHA512, "Should contain DH Group 16 SHA-512")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeDHGEXSHA256, "Should contain DH Group Exchange SHA-256")

		// MACs - should contain HMAC with SHA-256 and SHA-512
		require.Contains(t, algorithms.MACs, ssh.HMACSHA256ETM, "Should contain HMAC-SHA256-ETM")
		require.Contains(t, algorithms.MACs, ssh.HMACSHA512ETM, "Should contain HMAC-SHA512-ETM")
		require.Contains(t, algorithms.MACs, ssh.HMACSHA256, "Should contain HMAC-SHA256")
		require.Contains(t, algorithms.MACs, ssh.HMACSHA512, "Should contain HMAC-SHA512")

		// Public Key Auth - should contain RSA, ECDSA, and Ed25519
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoED25519, "Should contain Ed25519")
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoRSASHA256, "Should contain RSA-SHA256")
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoRSASHA512, "Should contain RSA-SHA512")
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoECDSA256, "Should contain ECDSA P-256")
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoECDSA384, "Should contain ECDSA P-384")
		require.Contains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoECDSA521, "Should contain ECDSA P-521")
	})

	t.Run("ExcludesNonFIPSAlgorithms", func(t *testing.T) {
		// Ciphers - should not contain ChaCha20
		require.NotContains(t, algorithms.Ciphers, ssh.CipherChaCha20Poly1305, "Should not contain ChaCha20-Poly1305")

		// Key Exchanges - should not contain X25519/Curve25519
		require.NotContains(t, algorithms.KeyExchanges, ssh.KeyExchangeCurve25519, "Should not contain Curve25519")
		require.NotContains(t, algorithms.KeyExchanges, ssh.KeyExchangeMLKEM768X25519, "Should not contain MLKEM768X25519")
		require.NotContains(t, algorithms.KeyExchanges, "curve25519-sha256@libssh.org", "Should not contain Curve25519 LibSSH alias")

		// MACs - should not contain HMAC-SHA1
		require.NotContains(t, algorithms.MACs, ssh.HMACSHA1, "Should not contain HMAC-SHA1")

		// Public Key Auth - should not contain Security Key variants
		require.NotContains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoSKED25519, "Should not contain Security Key Ed25519")
		require.NotContains(t, algorithms.PublicKeyAuths, ssh.KeyAlgoSKECDSA256, "Should not contain Security Key ECDSA")
	})

	t.Run("HostKeysAreEmpty", func(t *testing.T) {
		// HostKeys should be empty because they're determined by actual server host keys
		require.Empty(t, algorithms.HostKeys, "HostKeys should be empty - determined by server's actual host keys")
	})
}

func TestDefaultAlgorithms_FipsEnabled(t *testing.T) {
	algorithms := DefaultAlgorithms()

	t.Run("FiltersNonFIPSKeyExchanges", func(t *testing.T) {
		// Verify that non-FIPS key exchange algorithms are filtered out
		require.NotContains(t, algorithms.KeyExchanges, ssh.KeyExchangeCurve25519,
			"Should not contain Curve25519-SHA256 key exchange")
		require.NotContains(t, algorithms.KeyExchanges, "curve25519-sha256@libssh.org",
			"Should not contain Curve25519-LibSSH key exchange")
		require.NotContains(t, algorithms.KeyExchanges, ssh.KeyExchangeMLKEM768X25519,
			"Should not contain MLKEM768X25519 key exchange")
	})

	t.Run("ContainsFIPSCompliantKeyExchanges", func(t *testing.T) {
		// Verify that FIPS-compliant key exchanges are still present
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP256,
			"Should contain ECDH P-256")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP384,
			"Should contain ECDH P-384")
		require.Contains(t, algorithms.KeyExchanges, ssh.KeyExchangeECDHP521,
			"Should contain ECDH P-521")
		require.Greater(t, len(algorithms.KeyExchanges), 0,
			"Should have at least one key exchange algorithm")
	})

	t.Run("ContainsDefaults", func(t *testing.T) {
		expectedConfig := &ssh.ServerConfig{}
		expectedConfig.SetDefaults()

		require.Equal(t, expectedConfig.Ciphers, algorithms.Ciphers)
		require.Equal(t, expectedConfig.MACs, algorithms.MACs)
		require.Nil(t, algorithms.HostKeys, "HostKeys should be nil")
		require.Equal(t, expectedConfig.PublicKeyAuthAlgorithms, algorithms.PublicKeyAuths)
		// PublicKeyAuths is normally not set until handshake time
		require.Empty(t, algorithms.PublicKeyAuths)
	})
}
