//go:build !fips

package fips

import (
	"reflect"
	"testing"

	"github.com/stretchr/testify/require"
	"golang.org/x/crypto/ssh"
)

func TestSupportedAlgorithms_FipsDisabled(t *testing.T) {
	// This test runs when FIPS build tag is NOT present
	// Should always return ssh.SupportedAlgorithms()
	algorithms := SupportedAlgorithms()
	expected := ssh.SupportedAlgorithms()

	require.True(t, reflect.DeepEqual(expected, algorithms),
		"SupportedAlgorithms() should return ssh.SupportedAlgorithms() when FIPS is not enabled")
}

func TestSupportedAlgorithmsReturnsValidAlgorithms_NonFips(t *testing.T) {
	algorithms := SupportedAlgorithms()

	// Test that all returned algorithms are valid (non-empty strings)
	require.NotEmpty(t, algorithms.Ciphers, "Should have cipher algorithms")
	require.NotEmpty(t, algorithms.MACs, "Should have MAC algorithms")
	require.NotEmpty(t, algorithms.KeyExchanges, "Should have key exchange algorithms")
	require.NotEmpty(t, algorithms.HostKeys, "Should have host key algorithms")
	require.NotEmpty(t, algorithms.PublicKeyAuths, "Should have public key auth algorithms")

	for _, cipher := range algorithms.Ciphers {
		require.NotEmpty(t, cipher, "Cipher algorithm should not be empty")
	}
	for _, mac := range algorithms.MACs {
		require.NotEmpty(t, mac, "MAC algorithm should not be empty")
	}
	for _, kex := range algorithms.KeyExchanges {
		require.NotEmpty(t, kex, "KeyExchange algorithm should not be empty")
	}
	for _, hostKey := range algorithms.HostKeys {
		require.NotEmpty(t, hostKey, "HostKey algorithm should not be empty")
	}
	for _, pubKey := range algorithms.PublicKeyAuths {
		require.NotEmpty(t, pubKey, "PublicKeyAuth algorithm should not be empty")
	}
}

func TestDefaultAlgorithms(t *testing.T) {
	algs := DefaultAlgorithms()

	expectedConfig := &ssh.ServerConfig{}
	expectedConfig.SetDefaults()

	require.Equal(t, expectedConfig.Ciphers, algs.Ciphers)
	require.Equal(t, expectedConfig.MACs, algs.MACs)
	require.Equal(t, expectedConfig.KeyExchanges, algs.KeyExchanges)
	require.Equal(t, expectedConfig.PublicKeyAuthAlgorithms, algs.PublicKeyAuths)
	require.Nil(t, algs.HostKeys, "HostKeys should be nil")
}
