//go:build fips

package fips

import (
	"slices"

	"golang.org/x/crypto/ssh"
)

// Non-FIPS algorithms to filter out from Go's supported algorithms
var (
	// Key exchange algorithms that are not FIPS compliant
	nonFIPSKeyExchanges = []string{
		ssh.KeyExchangeCurve25519,      // X25519 - explicitly disallowed in FIPS 140-3
		ssh.KeyExchangeMLKEM768X25519,  // Contains X25519
		"curve25519-sha256@libssh.org", // X25519 alias
	}

	// Ciphers that are not FIPS compliant
	nonFIPSCiphers = []string{
		ssh.CipherChaCha20Poly1305, // Not FIPS approved
	}

	// MACs that are not FIPS compliant
	nonFIPSMACs = []string{
		ssh.HMACSHA1,
	}

	// Public key auth algorithms that are not FIPS compliant
	nonFIPSPubKeyAuth = []string{
		ssh.KeyAlgoSKED25519,  // Security Key variant
		ssh.KeyAlgoSKECDSA256, // Security Key variant
	}
)

// filterFIPSCompliant removes non-FIPS algorithms from a slice
func filterFIPSCompliant(algorithms []string, nonCompliant []string) []string {
	return slices.DeleteFunc(algorithms, func(algo string) bool {
		return slices.Contains(nonCompliant, algo)
	})
}

// SupportedAlgorithms returns FIPS-compliant algorithms currently implemented by Go's SSH package
func SupportedAlgorithms() ssh.Algorithms {
	goSupported := ssh.SupportedAlgorithms()

	return ssh.Algorithms{
		Ciphers:      filterFIPSCompliant(goSupported.Ciphers, nonFIPSCiphers),
		MACs:         filterFIPSCompliant(goSupported.MACs, nonFIPSMACs),
		KeyExchanges: filterFIPSCompliant(goSupported.KeyExchanges, nonFIPSKeyExchanges),
		// Leave HostKeys empty because the algorithms are derived from the host keys
		HostKeys:       nil,
		PublicKeyAuths: filterFIPSCompliant(goSupported.PublicKeyAuths, nonFIPSPubKeyAuth),
	}
}

// DefaultAlgorithms returns algorithms currently implemented by this package.
// For FIPS, ED25519 key exchanges are not supported and will result in a panic.
// The algorithms listed here are in preference order.
func DefaultAlgorithms() ssh.Algorithms {
	config := &ssh.ServerConfig{}
	config.SetDefaults()

	return ssh.Algorithms{
		Ciphers:      config.Ciphers,
		MACs:         config.MACs,
		KeyExchanges: filterFIPSCompliant(config.KeyExchanges, nonFIPSKeyExchanges),
		// Leave HostKeys empty because the algorithms are derived from the host keys
		HostKeys:       nil,
		PublicKeyAuths: config.PublicKeyAuthAlgorithms,
	}
}
