import require$$0 from 'tty';
import { createFilter, normalizePath } from '@rollup/pluginutils';

function getDefaultExportFromCjs (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
}

var picocolors = {exports: {}};

let tty = require$$0;

let isColorSupported =
	!("NO_COLOR" in process.env || process.argv.includes("--no-color")) &&
	("FORCE_COLOR" in process.env ||
		process.argv.includes("--color") ||
		process.platform === "win32" ||
		(tty.isatty(1) && process.env.TERM !== "dumb") ||
		"CI" in process.env);

let formatter =
	(open, close, replace = open) =>
	input => {
		let string = "" + input;
		let index = string.indexOf(close, open.length);
		return ~index
			? open + replaceClose(string, close, replace, index) + close
			: open + string + close
	};

let replaceClose = (string, close, replace, index) => {
	let start = string.substring(0, index) + replace;
	let end = string.substring(index + close.length);
	let nextIndex = end.indexOf(close);
	return ~nextIndex ? start + replaceClose(end, close, replace, nextIndex) : start + end
};

let createColors = (enabled = isColorSupported) => ({
	isColorSupported: enabled,
	reset: enabled ? s => `\x1b[0m${s}\x1b[0m` : String,
	bold: enabled ? formatter("\x1b[1m", "\x1b[22m", "\x1b[22m\x1b[1m") : String,
	dim: enabled ? formatter("\x1b[2m", "\x1b[22m", "\x1b[22m\x1b[2m") : String,
	italic: enabled ? formatter("\x1b[3m", "\x1b[23m") : String,
	underline: enabled ? formatter("\x1b[4m", "\x1b[24m") : String,
	inverse: enabled ? formatter("\x1b[7m", "\x1b[27m") : String,
	hidden: enabled ? formatter("\x1b[8m", "\x1b[28m") : String,
	strikethrough: enabled ? formatter("\x1b[9m", "\x1b[29m") : String,
	black: enabled ? formatter("\x1b[30m", "\x1b[39m") : String,
	red: enabled ? formatter("\x1b[31m", "\x1b[39m") : String,
	green: enabled ? formatter("\x1b[32m", "\x1b[39m") : String,
	yellow: enabled ? formatter("\x1b[33m", "\x1b[39m") : String,
	blue: enabled ? formatter("\x1b[34m", "\x1b[39m") : String,
	magenta: enabled ? formatter("\x1b[35m", "\x1b[39m") : String,
	cyan: enabled ? formatter("\x1b[36m", "\x1b[39m") : String,
	white: enabled ? formatter("\x1b[37m", "\x1b[39m") : String,
	gray: enabled ? formatter("\x1b[90m", "\x1b[39m") : String,
	bgBlack: enabled ? formatter("\x1b[40m", "\x1b[49m") : String,
	bgRed: enabled ? formatter("\x1b[41m", "\x1b[49m") : String,
	bgGreen: enabled ? formatter("\x1b[42m", "\x1b[49m") : String,
	bgYellow: enabled ? formatter("\x1b[43m", "\x1b[49m") : String,
	bgBlue: enabled ? formatter("\x1b[44m", "\x1b[49m") : String,
	bgMagenta: enabled ? formatter("\x1b[45m", "\x1b[49m") : String,
	bgCyan: enabled ? formatter("\x1b[46m", "\x1b[49m") : String,
	bgWhite: enabled ? formatter("\x1b[47m", "\x1b[49m") : String,
});

picocolors.exports = createColors();
picocolors.exports.createColors = createColors;

var picocolorsExports = picocolors.exports;
const pico = /*@__PURE__*/getDefaultExportFromCjs(picocolorsExports);

const ESLINT_SEVERITY = {
  ERROR: 2,
  WARNING: 1
};
const CWD = process.cwd();
const PLUGIN_NAME = "vite:eslint2";
const COLOR_MAPPING = {
  error: "red",
  warning: "yellow",
  plugin: "magenta"
};

function interopDefault(m) {
  return m.default || m;
}
const getOptions = ({
  test,
  dev,
  build,
  cache,
  cacheLocation,
  include,
  exclude,
  eslintPath,
  formatter,
  lintInWorker,
  lintOnStart,
  lintDirtyOnly,
  chokidar,
  emitError,
  emitErrorAsWarning,
  emitWarning,
  emitWarningAsError,
  ...eslintConstructorOptions
}) => ({
  test: test ?? false,
  dev: dev ?? true,
  build: build ?? false,
  cache: cache ?? true,
  cacheLocation: cacheLocation ?? ".eslintcache",
  include: include ?? ["src/**/*.{js,jsx,ts,tsx,vue,svelte}"],
  exclude: exclude ?? ["node_modules", "virtual:"],
  eslintPath: eslintPath ?? "eslint",
  formatter: formatter ?? "stylish",
  lintInWorker: lintInWorker ?? false,
  lintOnStart: lintOnStart ?? false,
  lintDirtyOnly: lintDirtyOnly ?? true,
  chokidar: chokidar ?? false,
  emitError: emitError ?? true,
  emitErrorAsWarning: emitErrorAsWarning ?? false,
  emitWarning: emitWarning ?? true,
  emitWarningAsError: emitWarningAsError ?? false,
  ...eslintConstructorOptions
});
const getFilter = (options) => createFilter(options.include, options.exclude);
const getESLintConstructorOptions = (options) => ({
  ...Object.fromEntries(
    Object.entries(options).filter(
      ([key]) => ![
        "test",
        "dev",
        "build",
        "include",
        "exclude",
        "eslintPath",
        "formatter",
        "lintInWorker",
        "lintOnStart",
        "lintDirtyOnly",
        "chokidar",
        "emitError",
        "emitErrorAsWarning",
        "emitWarning",
        "emitWarningAsError"
      ].includes(key)
    )
  ),
  errorOnUnmatchedPattern: false
});
const initializeESLint = async (options) => {
  const { eslintPath, formatter } = options;
  try {
    const _module = await import(eslintPath);
    const module = interopDefault(_module);
    const ESLintClass = module.loadESLint ? await module.loadESLint() : module.ESLint || module.FlatESLint || module.LegacyESLint;
    const eslintInstance = new ESLintClass(getESLintConstructorOptions(options));
    const loadedFormatter = await eslintInstance.loadFormatter(formatter);
    const outputFixes = ESLintClass.outputFixes.bind(ESLintClass);
    return {
      eslintInstance,
      formatter: loadedFormatter,
      outputFixes
    };
  } catch (error) {
    throw new Error(
      `Failed to initialize ESLint. Have you installed and configured correctly? ${error}`
    );
  }
};
const isVirtualModule = (id) => id.startsWith("virtual:") || id[0] === "\0" || !id.includes("/");
const getFilePath = (id) => normalizePath(id).split("?")[0];
const shouldIgnoreModule = async (id, filter, eslintInstance) => {
  if (isVirtualModule(id))
    return true;
  if (!filter(id))
    return true;
  const filePath = getFilePath(id);
  if ([".vue", ".svelte"].some((extname) => filePath.endsWith(extname)) && id.includes("?") && id.includes("type=style")) {
    return true;
  }
  if (eslintInstance)
    return await eslintInstance.isPathIgnored(filePath);
  return false;
};
const removeESLintErrorResults = (results) => results.map((result) => {
  const filteredMessages = result.messages.filter(
    (message) => message.severity !== ESLINT_SEVERITY.ERROR
  );
  const filteredSuppressedMessages = result.suppressedMessages.filter(
    (message) => message.severity !== ESLINT_SEVERITY.ERROR
  );
  return {
    ...result,
    messages: filteredMessages,
    suppressedMessages: filteredSuppressedMessages,
    errorCount: 0,
    fatalErrorCount: 0,
    fixableErrorCount: 0
  };
});
const removeESLintWarningResults = (results) => results.map((result) => {
  const filteredMessages = result.messages.filter(
    (message) => message.severity !== ESLINT_SEVERITY.WARNING
  );
  const filteredSuppressedMessages = result.suppressedMessages.filter(
    (message) => message.severity !== ESLINT_SEVERITY.WARNING
  );
  return {
    ...result,
    messages: filteredMessages,
    suppressedMessages: filteredSuppressedMessages,
    warningCount: 0,
    fixableWarningCount: 0
  };
});
const filterESLintLintResults = (results) => results.filter((result) => result.errorCount > 0 || result.warningCount > 0);
const colorize = (text, textType) => pico[COLOR_MAPPING[textType]](text);
const log = (text, textType, context) => {
  console.log("");
  if (context) {
    if (textType === "error")
      context.error(text);
    else if (textType === "warning")
      context.warn(text);
  } else {
    const t = colorize(`${text}  Plugin: ${colorize(PLUGIN_NAME, "plugin")}\r
`, textType);
    console.log(t);
  }
};
const lintFiles = async ({ files, eslintInstance, formatter, outputFixes, options }, context) => await eslintInstance.lintFiles(files).then(async (lintResults) => {
  if (!lintResults || lintResults.length === 0)
    return;
  if (options.fix)
    outputFixes(lintResults);
  let results = [...lintResults];
  if (!options.emitError)
    results = removeESLintErrorResults(results);
  if (!options.emitWarning)
    results = removeESLintWarningResults(results);
  results = filterESLintLintResults(results);
  if (results.length === 0)
    return;
  const formattedText = await formatter.format(results);
  let formattedTextType;
  if (results.some((result) => result.errorCount > 0)) {
    formattedTextType = options.emitErrorAsWarning ? "warning" : "error";
  } else {
    formattedTextType = options.emitWarningAsError ? "error" : "warning";
  }
  return log(formattedText, formattedTextType, context);
});

export { CWD as C, PLUGIN_NAME as P, getFilter as a, getFilePath as b, getOptions as g, initializeESLint as i, lintFiles as l, shouldIgnoreModule as s };
