Object.defineProperty(exports, '__esModule', { value: true });

const core = require('@sentry/core');
const types = require('../types.js');

/**
 * Checks if a given value is a valid measurement value.
 */
function isMeasurementValue(value) {
  return typeof value === 'number' && isFinite(value);
}

/**
 * Helper function to start child on transactions. This function will make sure that the transaction will
 * use the start timestamp of the created child span if it is earlier than the transactions actual
 * start timestamp.
 */
function startAndEndSpan(
  parentSpan,
  startTimeInSeconds,
  endTime,
  { ...ctx },
) {
  const parentStartTime = core.spanToJSON(parentSpan).start_timestamp;
  if (parentStartTime && parentStartTime > startTimeInSeconds) {
    // We can only do this for SentrySpans...
    if (typeof (parentSpan ).updateStartTime === 'function') {
      (parentSpan ).updateStartTime(startTimeInSeconds);
    }
  }

  // The return value only exists for tests
  return core.withActiveSpan(parentSpan, () => {
    const span = core.startInactiveSpan({
      startTime: startTimeInSeconds,
      ...ctx,
    });

    if (span) {
      span.end(endTime);
    }

    return span;
  });
}

/** Get the browser performance API. */
function getBrowserPerformanceAPI() {
  // @ts-expect-error we want to make sure all of these are available, even if TS is sure they are
  return types.WINDOW && types.WINDOW.addEventListener && types.WINDOW.performance;
}

/**
 * Converts from milliseconds to seconds
 * @param time time in ms
 */
function msToSec(time) {
  return time / 1000;
}

exports.getBrowserPerformanceAPI = getBrowserPerformanceAPI;
exports.isMeasurementValue = isMeasurementValue;
exports.msToSec = msToSec;
exports.startAndEndSpan = startAndEndSpan;
//# sourceMappingURL=utils.js.map
