#include "ecocode.h"
#include <utility>
#include <iostream>

#include <QString>
#include <QMap>
#include <QDebug>

namespace chess {

/*
 * the key of the qmap below is the
 * zobrist hash of the position reache after
 * playing moves according to the ECO classification
 * The zobrist hash is calculated here with
 * POLYGLOT RANDOM ARRAY values
 * The data structure below is autogenerated
 * using the geneco utility available @
 * https://github.com/asdfjkl
 * geneco converts a source list of eco codes (a text
 * file taken from the arasan chess engine (licensed under MIT, here
 * re-licensed under GPL2 and converts it to the data structure below)
 */
static const QMap<quint64, EcoInfo> ECOINFOS {
    {Q_UINT64_C(0x8da7a73e5fdd72dc), {"A00", ""}},
    {Q_UINT64_C(0xe89f55cd52014748), {"A00", ""}},
    {Q_UINT64_C(0xeb16d4aa32ce7438), {"A00", "Van't Kruijs Opening"}},
    {Q_UINT64_C(0xaec84ba6d432356e), {"A00", ""}},
    {Q_UINT64_C(0x2df2e8f47b022952), {"A00", ""}},
    {Q_UINT64_C(0x30e16e80c6e1f31a), {"A00", "Sargossa Opening"}},
    {Q_UINT64_C(0xe57697cefee41181), {"A00", "Grob's Attack"}},
    {Q_UINT64_C(0xeccee3b4b02790b8), {"A00", "Sokolsky Opening"}},
    {Q_UINT64_C(0xbdac8ebad4737e45), {"A00", "Dunst Opening"}},
    {Q_UINT64_C(0x41dd464bfbc3d352), {"A00", "Amar (Paris) Opening"}},
    {Q_UINT64_C(0x6add2d59f64624a4), {"A00", "King's Fianchetto"}},
    {Q_UINT64_C(0xca9df64ccc1c97f1), {"A01", "Nimzowitsch-Larsen Attack"}},
    {Q_UINT64_C(0xbb2deb150f00c115), {"A02", "Bird's Opening"}},
    {Q_UINT64_C(0x3e47c90193d48f33), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0xc0614bf2a044a68b), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0x297377a5293f3927), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0x7ed3e7d37b3af638), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0x2ade9a5d0f5d69ca), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0xb7473ec9a1006415), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0x7ed3e7d37b3af638), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0xa4aab28596007f17), {"A03", "Bird's Opening"}},
    {Q_UINT64_C(0x3155e35f9ce53923), {"A03", "Bird's Opening, From Gambit"}},
    {Q_UINT64_C(0x9d5f7aee7e779da1), {"A04", "Reti Opening"}},
    {Q_UINT64_C(0xcd37113f2b7f0d75), {"A04", "Lisitsin Gambit"}},
    {Q_UINT64_C(0x468f9553d1ce19d6), {"A04", "Reti Opening"}},
    {Q_UINT64_C(0xc6b14e1bd38ddc37), {"A05", "Reti Opening"}},
    {Q_UINT64_C(0x183558fae2a3d387), {"A06", "Reti Opening"}},
    {Q_UINT64_C(0xdc3255b209236e8d), {"A06", "Tennison Gambit"}},
    {Q_UINT64_C(0x34d3e3bb02740bbf), {"A07", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x834216f668cadc10), {"A07", "Reti Opening"}},
    {Q_UINT64_C(0xdacdf433e2639250), {"A07", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x48b719dc95a341eb), {"A07", "Reti Opening, Barcza System"}},
    {Q_UINT64_C(0xe12d3bb897d3527), {"A07", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x2b899c7a5874afa1), {"A07", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x67521168b0ded8bb), {"A08", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x67fa97ec36020050), {"A08", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0xbbfe0d9a302be646), {"A08", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0xeefe3c72753e484), {"A08", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x3cf56e39c16d9b7d), {"A08", "Reti, King's Indian Attack"}},
    {Q_UINT64_C(0x9416c7dea1ac7880), {"A09", "Reti Opening, Classical Variation"}},
    {Q_UINT64_C(0xca18093c559e579b), {"A10", "English Opening"}},
    {Q_UINT64_C(0x5862e4d3225e8420), {"A11", "English Opening"}},
    {Q_UINT64_C(0x3ad7dbf263edfac6), {"A11", "English Opening"}},
    {Q_UINT64_C(0x28872f441c3dd322), {"A11", "English Opening"}},
    {Q_UINT64_C(0x1692bfca8492040f), {"A11", "English Opening"}},
    {Q_UINT64_C(0x5b04bfa318c932d3), {"A11", "English Opening"}},
    {Q_UINT64_C(0x7d4fb6e585097082), {"A11", "English Opening"}},
    {Q_UINT64_C(0xd9cfb5277a0b180c), {"A11", "English Opening"}},
    {Q_UINT64_C(0xc33ec22b4d34e148), {"A11", "English Opening"}},
    {Q_UINT64_C(0x8aca4a650ce1c056), {"A12", "English Opening"}},
    {Q_UINT64_C(0xbc6ffb0d4dbcc7c9), {"A13", "English Opening"}},
    {Q_UINT64_C(0x20999926b94ea905), {"A13", "Reti Opening, Neo-Catalan"}},
    {Q_UINT64_C(0x980423fceaa64a8d), {"A13", "English Opening"}},
    {Q_UINT64_C(0x8cc85013200de61c), {"A13", "English Opening"}},
    {Q_UINT64_C(0x4a2091fd0f16de84), {"A14", "English Opening"}},
    {Q_UINT64_C(0x91f63dc9f864160d), {"A15", "English Opening"}},
    {Q_UINT64_C(0x6a61256b3a8694d4), {"A16", "English Opening"}},
    {Q_UINT64_C(0x1c16d75a22a40486), {"A17", "English Opening"}},
    {Q_UINT64_C(0xe3352aa01b5ad99), {"A17", "English Opening"}},
    {Q_UINT64_C(0x595bf119ec249a80), {"A17", "English Opening"}},
    {Q_UINT64_C(0xb582cd00c63a7816), {"A17", "English Opening"}},
    {Q_UINT64_C(0xb3c192b1c5c87844), {"A17", "English Opening"}},
    {Q_UINT64_C(0x54bb18d66c532e3c), {"A17", "English Opening"}},
    {Q_UINT64_C(0xd2bb20bccbc90e02), {"A17", "English Opening"}},
    {Q_UINT64_C(0x49bca0fb1fe1c23a), {"A17", "English Opening"}},
    {Q_UINT64_C(0xd811da12c924b98c), {"A18", "English, Flohr-Mikenas System"}},
    {Q_UINT64_C(0x3e600bbc3663b2f1), {"A19", "English Opening"}},
    {Q_UINT64_C(0x40600176c67bafad), {"A20", "English Opening"}},
    {Q_UINT64_C(0xfefc57d8516ca42e), {"A20", "English Opening"}},
    {Q_UINT64_C(0x8298adbfc6bbee7a), {"A20", "English Opening"}},
    {Q_UINT64_C(0xbbf719d404992d74), {"A21", "English Opening"}},
    {Q_UINT64_C(0x72b670d24f6ee556), {"A21", "English, Lukin Variation"}},
    {Q_UINT64_C(0x824d8fb5d6e16419), {"A21", "English Opening"}},
    {Q_UINT64_C(0x71a7f6cb8a8f22ea), {"A21", "English Opening"}},
    {Q_UINT64_C(0xe0192d21a9636ce2), {"A22", "English Opening"}},
    {Q_UINT64_C(0xccff966049b4b4da), {"A22", "English, Carls' Bremen System"}},
    {Q_UINT64_C(0x5beff6602437cf8), {"A22", "English Bremen, Smyslov System"}},
    {Q_UINT64_C(0x5e857b8f3e746761), {"A23", ""}},
    {Q_UINT64_C(0x254bc53b323b2691), {"A24", ""}},
    {Q_UINT64_C(0x101ac13615088877), {"A25", "English Opening"}},
    {Q_UINT64_C(0x57b849026771e5a1), {"A25", "English Opening"}},
    {Q_UINT64_C(0xea3720e18a4dabca), {"A25", "Engish Opening"}},
    {Q_UINT64_C(0x472bdbd384c7efb3), {"A25", "English Opening"}},
    {Q_UINT64_C(0x471a6253ae12236e), {"A26", "English Opening"}},
    {Q_UINT64_C(0x831d6f1b45929e64), {"A26", "English Opening"}},
    {Q_UINT64_C(0xa9f63869d6301827), {"A26", "English Opening"}},
    {Q_UINT64_C(0x5838d6b87ecbffd2), {"A26", "English Opening"}},
    {Q_UINT64_C(0xcb7e2dc07deee94a), {"A27", "English Opening"}},
    {Q_UINT64_C(0x90901935d014a8dc), {"A28", "English, Four Knights"}},
    {Q_UINT64_C(0x45e5409cad6ad43e), {"A28", "English, Four Knights"}},
    {Q_UINT64_C(0xbc76a27430c370e4), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x67ca235f5ac8c57), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x34b3cd92e49484b4), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x9a170e47a0043f60), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x9596f46f1b807d75), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0xf0ae069c165c48e1), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x9b1c43e2ad768db2), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0xd7feb1ede2354d7), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x4db546c1574a647c), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0xc70d56287c9d651), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0xddb6c2130a8ac18c), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x319ada0d8a492f4a), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x4fefa139f6baf45a), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0xd13f4d3679a8c3d4), {"A29", "English, Four Knights"}},
    {Q_UINT64_C(0x2c69d892aad95ce6), {"A30", "English, Symmetrical"}},
    {Q_UINT64_C(0xeb6cf2c0719113ee), {"A30", "English, Hedgehog"}},
    {Q_UINT64_C(0x668c1853ab510165), {"A30", "English, Hedgehog"}},
    {Q_UINT64_C(0x9557b7122b60106), {"A30", "English, Hedgehog"}},
    {Q_UINT64_C(0x4bdaf7e68895fe5f), {"A30", "English, Hedgehog"}},
    {Q_UINT64_C(0x69d62f3b7e0c510f), {"A31", "English, Symmetrical"}},
    {Q_UINT64_C(0xfab110b103580fe4), {"A31", "English, Symmetrical"}},
    {Q_UINT64_C(0xef80db43405b3bda), {"A31", "English, Symmetrical"}},
    {Q_UINT64_C(0x644ab25b61815fef), {"A31", "English, Symmetrical"}},
    {Q_UINT64_C(0x272b3a620aeb3ab5), {"A32", "English, Symmetrical"}},
    {Q_UINT64_C(0x18b7daad0a6d597c), {"A32", "English, Symmetrical"}},
    {Q_UINT64_C(0x7751fa22d9981d6f), {"A33", "English, Symmetrical"}},
    {Q_UINT64_C(0xd7fec030683bde3f), {"A34", "English, Symmetrical"}},
    {Q_UINT64_C(0x43dadba72cd8b9f3), {"A34", "English, Symmetrical"}},
    {Q_UINT64_C(0x9336b945c62e5c68), {"A34", "English, Rubinstein Variation"}},
    {Q_UINT64_C(0xdd537a01a703680b), {"A34", "English, Rubinstein Variation"}},
    {Q_UINT64_C(0x1426bae9e09a4b6f), {"A34", "English, Symmetrical"}},
    {Q_UINT64_C(0x7aaca874ef87f5c0), {"A34", "English, Symmetrical"}},
    {Q_UINT64_C(0xd18062eb38534d68), {"A34", "English, Symmetrical"}},
    {Q_UINT64_C(0x7c1318d279aa7b3c), {"A35", "English, Symmetrical"}},
    {Q_UINT64_C(0xdd27da778b57d204), {"A35", "English, Symmetrical"}},
    {Q_UINT64_C(0xfc99c0d1bcb65b97), {"A35", "English, Symmetrical"}},
    {Q_UINT64_C(0x27c3705280cb4361), {"A35", "English, Symmetrical"}},
    {Q_UINT64_C(0x55155984c0b5cd89), {"A35", "English, Symmetrical"}},
    {Q_UINT64_C(0x50f5a393997da304), {"A36", "English, Symmetrical"}},
    {Q_UINT64_C(0xbc0dee035458229c), {"A36", "English, Symmetrical"}},
    {Q_UINT64_C(0x676902f53cbe43a1), {"A37", "English, Symmetrical"}},
    {Q_UINT64_C(0x5ed39494eec60acc), {"A37", "English, Symmetrical"}},
    {Q_UINT64_C(0xed110abfaf5bbb97), {"A37", "English, Botvinnik System"}},
    {Q_UINT64_C(0xad2bcd366821eb3a), {"A37", "English, Botvinnik System"}},
    {Q_UINT64_C(0x7a0f5f0b39b3492e), {"A37", "English, Botvinnik System"}},
    {Q_UINT64_C(0x3c87360091440237), {"A38", "English, Symmetrical"}},
    {Q_UINT64_C(0x6436edc5989996d1), {"A38", "English, Symmetrical"}},
    {Q_UINT64_C(0x74f7d28f52a7f2aa), {"A38", "English, Symmetrical"}},
    {Q_UINT64_C(0x8d6f8f2929271d8e), {"A39", "English, Symmetrical"}},
    {Q_UINT64_C(0xb5e5684145e2e666), {"A39", "English, Symmetrical"}},
    {Q_UINT64_C(0x2b9496112ef27fa3), {"A39", "English, Symmetrical"}},
    {Q_UINT64_C(0x9696169426829005), {"A39", "English, Symmetrical"}},
    {Q_UINT64_C(0x93da085f902883d6), {"A39", "English, Symmetrical"}},
    {Q_UINT64_C(0x830eb9b20758d1de), {"A40", "Queen Pawn"}},
    {Q_UINT64_C(0x976b1f894bd29e8), {"A40", "Englund Gambit"}},
    {Q_UINT64_C(0x2c60fdea125bdb4f), {"A40", "Owen's Defense"}},
    {Q_UINT64_C(0xaf37786bb8bef472), {"A40", ""}},
    {Q_UINT64_C(0x76b5e13adec0afdd), {"A40", ""}},
    {Q_UINT64_C(0x33a979d902a48df2), {"A40", ""}},
    {Q_UINT64_C(0xb01bbda1178222a9), {"A40", ""}},
    {Q_UINT64_C(0xbab42fd3d52098b3), {"A41", ""}},
    {Q_UINT64_C(0x6bdd3ec38e290a24), {"A41", "Modern Defense"}},
    {Q_UINT64_C(0xa0f15a6a3eb4a890), {"A41", "Modern Defense"}},
    {Q_UINT64_C(0x5e89b1ff3454baef), {"A41", "Modern Defense"}},
    {Q_UINT64_C(0x500fcbf03f855586), {"A42", "Modern Defense"}},
    {Q_UINT64_C(0xfdf9789311cc4ab6), {"A42", "Modern Defense"}},
    {Q_UINT64_C(0x657f681cf81fdaa3), {"A43", "Old Benoni"}},
    {Q_UINT64_C(0xf9a3ce6f8f020aa0), {"A43", "Indian Game: Pseudo-Benko"}},
    {Q_UINT64_C(0x3965884b0d8f0a67), {"A43", "Old Benoni"}},
    {Q_UINT64_C(0x9a3e66dbf801f641), {"A44", "Old Benoni"}},
    {Q_UINT64_C(0xd8e08d47aaa29048), {"A45", ""}},
    {Q_UINT64_C(0x7bad8c9142d77d55), {"A45", "Gibbins-Weidenhagen Gambit"}},
    {Q_UINT64_C(0x1ebada711a284b80), {"A45", "Trompowsky Attack"}},
    {Q_UINT64_C(0x38461b1c244f175), {"A46", "Queen Pawn"}},
    {Q_UINT64_C(0xaea7750143068b44), {"A46", "Queen Pawn, London System"}},
    {Q_UINT64_C(0xb3a9c4b66aecbaef), {"A46", "Torre Attack"}},
    {Q_UINT64_C(0x68ed28a13cc746ee), {"A47", "Queen Pawn"}},
    {Q_UINT64_C(0xea3032eab9cb633e), {"A48", "Queen Pawn"}},
    {Q_UINT64_C(0x3164d46b20ab895d), {"A48", "Queen Pawn, London System"}},
    {Q_UINT64_C(0x693d282128e26553), {"A48", "Queen Pawn, London System"}},
    {Q_UINT64_C(0xf7140d073a31a327), {"A48", "Queen Pawn, London System"}},
    {Q_UINT64_C(0x2c6a65dc0941b8f6), {"A48", "Torre Attack"}},
    {Q_UINT64_C(0xc6d689ab591cbb06), {"A49", "Queen Pawn"}},
    {Q_UINT64_C(0x54c31263e9ad3b4f), {"A50", ""}},
    {Q_UINT64_C(0xdebb1a297a48c379), {"A51", "Budapest Defense"}},
    {Q_UINT64_C(0x782ea9f4c034f9f3), {"A51", "Budapest, Fajarowicz Gambit"}},
    {Q_UINT64_C(0xdf05be0d7f37e565), {"A52", "Budapest Defense"}},
    {Q_UINT64_C(0x6d7984023bd57222), {"A53", "Old Indian Defense"}},
    {Q_UINT64_C(0xc7f2781c023469f0), {"A54", "Old Indian Defense"}},
    {Q_UINT64_C(0xf4110341e29c843c), {"A54", "Old Indian Defense"}},
    {Q_UINT64_C(0xbe26a4a799fd4ea2), {"A54", "Old Indian Defense"}},
    {Q_UINT64_C(0x30160e09091c3936), {"A55", "Old Indian Defense"}},
    {Q_UINT64_C(0xb2b2c3cd16ea3032), {"A56", "Benoni"}},
    {Q_UINT64_C(0x4df3cd0a16f41cd0), {"A56", "Czech Benoni"}},
    {Q_UINT64_C(0x68beea8b14611b52), {"A57", "Benko Gambit"}},
    {Q_UINT64_C(0xb3da067d7c877a6f), {"A57", "Benko Gambit"}},
    {Q_UINT64_C(0x39a691df7150ed3e), {"A58", "Benko Gambit"}},
    {Q_UINT64_C(0xc002f16f8edafdf3), {"A58", "Benko Gambit"}},
    {Q_UINT64_C(0xd638410ff1c509cb), {"A59", "Benko Gambit"}},
    {Q_UINT64_C(0x5d1ddcf2d8c23ddd), {"A59", "Benko Gambit"}},
    {Q_UINT64_C(0x7f679fac4a904392), {"A59", "Benko Gambit"}},
    {Q_UINT64_C(0x36c50db24b63974a), {"A59", "Benko Gambit"}},
    {Q_UINT64_C(0x7ecd72af97ee363d), {"A60", "Benoni"}},
    {Q_UINT64_C(0x311e1f254ff87ee), {"A60", "Benoni"}},
    {Q_UINT64_C(0xbf0e01c82c76da29), {"A61", "Benoni"}},
    {Q_UINT64_C(0x9c0c49f5af52f829), {"A62", "Benoni"}},
    {Q_UINT64_C(0x96a4a4427a0b11c2), {"A62", "Benoni"}},
    {Q_UINT64_C(0xd19a499c3309cef5), {"A63", "Benoni"}},
    {Q_UINT64_C(0x308ec3323481dbe5), {"A63", "Benoni"}},
    {Q_UINT64_C(0x68a4c683b9bae3e7), {"A63", "Benoni"}},
    {Q_UINT64_C(0x280fff72e1e6a7ee), {"A63", "Benoni"}},
    {Q_UINT64_C(0xa9d79a24497738ef), {"A64", "Benoni"}},
    {Q_UINT64_C(0x49d9b32dd49f9455), {"A65", "Benoni"}},
    {Q_UINT64_C(0x5d7b9d7bb6813b97), {"A66", "Benoni"}},
    {Q_UINT64_C(0xde197162285263be), {"A67", "Benoni, Mikenas Variation"}},
    {Q_UINT64_C(0x3c0ba18cf07978eb), {"A68", "Benoni"}},
    {Q_UINT64_C(0x53dae535c32cd7ca), {"A69", "Benoni"}},
    {Q_UINT64_C(0x7b090c80c7f66723), {"A70", "Benoni"}},
    {Q_UINT64_C(0x29ee013f2b4b8c90), {"A70", "Benoni"}},
    {Q_UINT64_C(0xb835b2cf3d94e62a), {"A71", "Benoni"}},
    {Q_UINT64_C(0xb29d5f78e8cd0fc1), {"A72", "Benoni"}},
    {Q_UINT64_C(0xcce8244c943ed4d1), {"A73", "Benoni"}},
    {Q_UINT64_C(0x1e1fd5bf731e2c0d), {"A74", "Benoni"}},
    {Q_UINT64_C(0x770561615908f729), {"A75", "Benoni"}},
    {Q_UINT64_C(0xac5187e0c0681d4a), {"A75", "Benoni"}},
    {Q_UINT64_C(0x201f32e5888bf2bf), {"A75", "Benoni"}},
    {Q_UINT64_C(0xd0b9e30ca64e3153), {"A76", "Benoni"}},
    {Q_UINT64_C(0xd9b78eb64f30fd9), {"A77", ""}},
    {Q_UINT64_C(0x431bd37be20c9365), {"A78", ""}},
    {Q_UINT64_C(0x8f891be411c231fd), {"A79", ""}},
    {Q_UINT64_C(0x1761df2bb9d0fc00), {"A80", "Dutch Defense"}},
    {Q_UINT64_C(0xb42cdefd51a5111d), {"A80", "Dutch Defense, Krejcik Gambit"}},
    {Q_UINT64_C(0x3b87646a59072438), {"A81", "Dutch Defense"}},
    {Q_UINT64_C(0xd366d2635250410a), {"A82", "Dutch Defense, Staunton Gambit"}},
    {Q_UINT64_C(0x77560104c9136646), {"A83", ""}},
    {Q_UINT64_C(0x9b42400ffadf5707), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0xed35b23ee2fdc755), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0x36515ec88a1ba668), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0x6dbf6a3d27e1e7fe), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0x5dcaebf509ce43e2), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0xfd226030721bda8), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0xd96172237cd07122), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0x58a2506072c33a80), {"A84", "Dutch Defense"}},
    {Q_UINT64_C(0x3b3b6c5895c79448), {"A85", "Dutch Defense"}},
    {Q_UINT64_C(0x147aacf159b62430), {"A85", "Dutch Defense"}},
    {Q_UINT64_C(0x2dc03a908bce6d5d), {"A85", "Dutch Defense"}},
    {Q_UINT64_C(0xdd4033f6ccd26349), {"A85", "Dutch Defense"}},
    {Q_UINT64_C(0xe6b2b9927cc6e707), {"A85", "Dutch Defense"}},
    {Q_UINT64_C(0xec4acfbbb7f2cea9), {"A86", "Dutch Defense"}},
    {Q_UINT64_C(0x5fe9ce0cc7d5ce2), {"A86", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0xdbd66edd12312e0c), {"A87", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0x96b16e3f69e3559a), {"A87", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0x1b5184acb3234711), {"A87", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0x96a7fa8f9823f2fe), {"A87", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0xff5c9b72dcc104f8), {"A88", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0xc6cbae7fba907240), {"A89", "Dutch Defense, Leningrad"}},
    {Q_UINT64_C(0x2fcd1ef7e23d8682), {"A90", "Dutch Defense"}},
    {Q_UINT64_C(0x71c3d015160fa999), {"A90", "Dutch Defense"}},
    {Q_UINT64_C(0xd59bbd97c91578bd), {"A90", "Dutch Defense"}},
    {Q_UINT64_C(0xabeec6a3b5e6a3ad), {"A90", "Dutch Defense"}},
    {Q_UINT64_C(0x8f9a1ec29271eaf3), {"A90", "Dutch Defense"}},
    {Q_UINT64_C(0x3b016d1828962a13), {"A91", "Dutch Defense"}},
    {Q_UINT64_C(0xe06581ee40704b2e), {"A91", "Dutch Defense"}},
    {Q_UINT64_C(0xeacd6c599529a2c5), {"A92", "Dutch Defense"}},
    {Q_UINT64_C(0x11d23579750e37f3), {"A92", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0x9d74552daf835c9e), {"A93", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0x7780171855e84f98), {"A93", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0xad1d698452931e4c), {"A94", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0x783fc034c02c6691), {"A95", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0xa36b26b5594c8cf2), {"A95", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0xf499a0601aa10dfc), {"A95", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0xc926b229d2fa5af5), {"A95", "Dutch Defense, Stonewall"}},
    {Q_UINT64_C(0xad02810c3ba230b8), {"A96", "Dutch Defense, Classical"}},
    {Q_UINT64_C(0x569599aef940b261), {"A96", "Dutch Defense, Classical"}},
    {Q_UINT64_C(0xad1415bcca6297dc), {"A97", "Dutch Defense, Classical"}},
    {Q_UINT64_C(0x1c0d67a1d8b4abb8), {"A98", "Dutch Defense, Classical"}},
    {Q_UINT64_C(0x21b275e810effcb1), {"A99", "Dutch Defense, Classical"}},
    {Q_UINT64_C(0x823c9b50fd114196), {"B00", "King Pawn"}},
    {Q_UINT64_C(0x29d143b2ec80e495), {"B00", "Nimzowitsch Defense"}},
    {Q_UINT64_C(0x756b94461c50fb0), {"B01", "Scandinavian Defense"}},
    {Q_UINT64_C(0xa145bdee0e3a0300), {"B01", "Scandinavian Defense"}},
    {Q_UINT64_C(0xe4fa4a47daef0ee9), {"B01", "Scandinavian Defense"}},
    {Q_UINT64_C(0xcd5a4325af512e90), {"B01", "Scandinavian, Pytel-Wade Variation"}},
    {Q_UINT64_C(0x653ef537229f749c), {"B01", "Scandinavian Gambit"}},
    {Q_UINT64_C(0x8133eae94d7d3775), {"B01", "Scandinavian, Islandic Gambit"}},
    {Q_UINT64_C(0x21bcd21bb87c3c3e), {"B01", "Scandinavian, Marshall Variation"}},
    {Q_UINT64_C(0xd9d2afa550eb0000), {"B02", "Alekhine's Defense"}},
    {Q_UINT64_C(0xd89ab24c1b5dca3e), {"B03", "Alekhine's Defense, Four Pawns' Attack"}},
    {Q_UINT64_C(0xb6919cd2ef647e28), {"B02", "Alekhine's Defense, Two Pawns' Attack"}},
    {Q_UINT64_C(0x64534b53efb678a8), {"B03", "Alekhine's Defense, Exchange Variation"}},
    {Q_UINT64_C(0xb98e490b8bfe7e93), {"B03", "Alekhine's Defense"}},
    {Q_UINT64_C(0x5b50339c316056c3), {"B04", "Alekhine's Defense"}},
    {Q_UINT64_C(0x175c70ec11a504c1), {"B04", "Alekhine's Defense"}},
    {Q_UINT64_C(0x5983f9ae76e55829), {"B05", "Alekhine's Defense"}},
    {Q_UINT64_C(0xfc3be9852daf9860), {"B05", "Alekhine's Defense"}},
    {Q_UINT64_C(0x6b88c80b869ed3dd), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0x970771c0452fb7f2), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0xe596c2b5aef2b7ec), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0x748b822146c9fbc), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0x1132f4cc486cc8d), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0x5da8d28a308baf1a), {"B06", "Modern Defense"}},
    {Q_UINT64_C(0x255d166e935a642f), {"B07", "Pirc Defense"}},
    {Q_UINT64_C(0x4cb0e3232678354d), {"B07", "Pirc Defense, Czech Variation"}},
    {Q_UINT64_C(0x178da9c380339759), {"B07", "Pirc Defense"}},
    {Q_UINT64_C(0xc82ff9c4ca68a9b0), {"B07", "Pirc Defense"}},
    {Q_UINT64_C(0xd1eb6cedab045375), {"B07", "Pirc Defense"}},
    {Q_UINT64_C(0xd8fe6581385b43), {"B07", "Pirc Defense"}},
    {Q_UINT64_C(0x5ca68cd7b996de2a), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0x258ee2996dea7d62), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0xc981744266d975c9), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0x5928539f569ca898), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0xe4bcf53e06931944), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0x890c685ef6395eae), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0xe2c516b29baa8b60), {"B08", "Pirc Defense, Classical Variation"}},
    {Q_UINT64_C(0xca68209a33a64934), {"B09", "Pirc Defense, Austrian Attack"}},
    {Q_UINT64_C(0x7ad41d2cc8e1829e), {"B09", "Pirc Defense, Austrian Attack"}},
    {Q_UINT64_C(0x104676bf8ad1922d), {"B10", "Caro-Kann Defense"}},
    {Q_UINT64_C(0x1864c1b3203b6d3b), {"B10", "Caro-Kann Defense"}},
    {Q_UINT64_C(0x555a0aebecb1c617), {"B10", "Caro-Kann Defense"}},
    {Q_UINT64_C(0xb70c6acdfb843105), {"B11", "Caro-Kann, Two Knights' Variation"}},
    {Q_UINT64_C(0x8c7935f098f45fbe), {"B11", "Caro-Kann"}},
    {Q_UINT64_C(0xd57359159b18bf6f), {"B12", "Caro-Kann Defense"}},
    {Q_UINT64_C(0x31606dfc03d2d72d), {"B12", "Caro-Kann, Advance Variation"}},
    {Q_UINT64_C(0x1f6a1ab5c4f32fcf), {"B12", "Caro-Kann, Advance Variation"}},
    {Q_UINT64_C(0x69b0e22d14832049), {"B12", "Caro-Kann, Advance Variation"}},
    {Q_UINT64_C(0x6cea994cc5d6656c), {"B12", "Caro-Kann, Advance Variation"}},
    {Q_UINT64_C(0x3ea89d30a7061e38), {"B12", "Caro-Kann, Fantasy Variation"}},
    {Q_UINT64_C(0x4b6de057dc9bef5), {"B13", "Caro-Kann, Exchange Variation"}},
    {Q_UINT64_C(0xff21c6a7bf2b3c2c), {"B13", "Caro-Kann, Exchange Variation"}},
    {Q_UINT64_C(0x889541213ec615f2), {"B13", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0x8301b594404f73be), {"B13", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0x5ab693506f2c1c69), {"B13", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0xcb7272d072071528), {"B13", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0x85ff73b1211a27d2), {"B14", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0x9133005eebb18b43), {"B14", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0x4cbe1ab76aedeff0), {"B14", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0xc78e51c874817fd6), {"B14", "Caro-Kann, Panov-Botvinnik Attack"}},
    {Q_UINT64_C(0xab8e63a3c52e7390), {"B15", "Caro-Kann Defense"}},
    {Q_UINT64_C(0x423a30f8bea1e1db), {"B15", "Caro-Kann, Gurgandize System"}},
    {Q_UINT64_C(0x3ecf958145a4756c), {"B16", "Caro-Kann, Bronstein-Larsen Variation"}},
    {Q_UINT64_C(0xfa25b904ac89bf9b), {"B17", "Caro-Kann Defense"}},
    {Q_UINT64_C(0x4adfabd7fbc45bda), {"B18", "Caro-Kann, Classical Variation"}},
    {Q_UINT64_C(0x41ae97c7b9f29a70), {"B19", "Caro-Kann, Classical Variation"}},
    {Q_UINT64_C(0xa96792f6450d3061), {"B19", "Caro-Kann, Classical Variation"}},
    {Q_UINT64_C(0x644d4afe02564aeb), {"B20", "Sicilian Defense"}},
    {Q_UINT64_C(0x88bc27f9be1b01e9), {"B20", "Smith-Morra Gambit"}},
    {Q_UINT64_C(0xfe456da2d682d73c), {"B20", "Sicilian Defense"}},
    {Q_UINT64_C(0x91e102f4ddfa6b28), {"B20", "Sicilian Defense"}},
    {Q_UINT64_C(0xe73bfa6c0d8a64ae), {"B20", "Sicilian Defense"}},
    {Q_UINT64_C(0x7912df4a1f59a2da), {"B20", "Sicilian Defense"}},
    {Q_UINT64_C(0x995b37f31bc77762), {"B21", "Sicilian Defense, Grand Prix Attack"}},
    {Q_UINT64_C(0xceb83f52a4e026cf), {"B21", "Sicilian Defense, Flank Gambit"}},
    {Q_UINT64_C(0x1297b266d226456d), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0x3c749e6087d33496), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0xe7107296ef3555ab), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0x779e1183cd51eee7), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0xaa043b50c4e2dbb6), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0x4c9ad53b904c6a78), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0xbfc07d49f4839d42), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0x867aeb2826fbd42f), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0xd9969616269c2f33), {"B22", "Sicilian Defense, Alapin Variation"}},
    {Q_UINT64_C(0x9fda525cc0b4c832), {"B23", "Sicilian Defense, Closed"}},
    {Q_UINT64_C(0x18d131ff31f2b509), {"B24", "Sicilan Defense, Closed"}},
    {Q_UINT64_C(0x633729db6a3fc628), {"B25", "Sicilan Defense, Closed"}},
    {Q_UINT64_C(0x73088d98c4105a63), {"B25", "Sicilan Defense, Closed"}},
    {Q_UINT64_C(0x79a0602f1149b388), {"B25", "Sicilan Defense, Closed"}},
    {Q_UINT64_C(0x6a761a561f18e9f1), {"B25", "Sicilian Defense"}},
    {Q_UINT64_C(0x6a761a561f18e9f1), {"B25", "Sicilian Defense"}},
    {Q_UINT64_C(0xe281adfd46169d3e), {"B25", "Sicilian Defense"}},
    {Q_UINT64_C(0x1a266c088b859769), {"B25", "Sicilian Defense"}},
    {Q_UINT64_C(0x103f9237eb05dda6), {"B26", "Sicilan Defense, Closed"}},
    {Q_UINT64_C(0xbf29a6086ab02bd6), {"B27", "Sicilan Defense"}},
    {Q_UINT64_C(0x6dde57fb8d90d30a), {"B28", "Sicilian Defense, O'Kelly Variation"}},
    {Q_UINT64_C(0xe4c792fdc74a6a40), {"B29", "Sicilian Defense"}},
    {Q_UINT64_C(0x14c47eea7b218ed5), {"B30", "Sicilian Defense"}},
    {Q_UINT64_C(0x271aaf451eb51d56), {"B30", "Sicilian Defense, Rossolimo Variation"}},
    {Q_UINT64_C(0x8763831271adde19), {"B30", "Sicilian Defense"}},
    {Q_UINT64_C(0xbe310b46dd97e2f1), {"B30", "Sicilian, Portsmouth Gambit"}},
    {Q_UINT64_C(0xceaefc1e653a8f1d), {"B31", "Sicilian Defense, Rossolimo Variation"}},
    {Q_UINT64_C(0xd1f151406ae8a397), {"B32", "Sicilian Defense"}},
    {Q_UINT64_C(0x6303be6295c8a049), {"B32", "Sicilian Defense, neo-Sveshinikov Variation"}},
    {Q_UINT64_C(0xc37a9235fad06306), {"B33", "Sicilian Defense, Sveshnikov Variation"}},
    {Q_UINT64_C(0x8efd6428a37e4dea), {"B33", "Sicilian Defense, Sveshnikov Variation"}},
    {Q_UINT64_C(0x1b0748fc7e923329), {"B33", " Sicilian Defense, Sveshnikov Variation"}},
    {Q_UINT64_C(0xcfe5737da2ca34), {"B34", "Sicilian Defense, Accelerated Dragon"}},
    {Q_UINT64_C(0xb1fb5640840ff9), {"B35", "Sicilian, Accelerated Dragon"}},
    {Q_UINT64_C(0xa7b715399505f566), {"B35", "Sicilian, Accelerated Dragon"}},
    {Q_UINT64_C(0x8cec7a573ead6133), {"B36", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0xa8217fccd4fd6c53), {"B37", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x696c7a6834b4bdf3), {"B37", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0xfa8f15fcdc928f17), {"B37", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x4f58fc0d44d0b117), {"B38", "Sicilian Defense, Marcozy Bind"}},
    {Q_UINT64_C(0xced463e9fe52dc6e), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x14b6c8f8e92af081), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0xef21d05a2bc87258), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x9609be14ffb4d110), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x3307ad138b65935c), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0xfe9db58b0c961d62), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x87f548365388c75c), {"B38", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0x72028b3c2682e334), {"B39", "Sicilian Defense, Maroczy Bind"}},
    {Q_UINT64_C(0xc95e54397292bb84), {"B40", "Sicilian Defense"}},
    {Q_UINT64_C(0xe6166d08e8be95f2), {"B41", "Sicilian Defense, Paulsen Variation"}},
    {Q_UINT64_C(0xb31bccaad13e5781), {"B42", "Sicilian Defense, Paulsen Variation"}},
    {Q_UINT64_C(0x1d8175aa2a5c172b), {"B43", "Sicilian Defense, Paulsen Variation"}},
    {Q_UINT64_C(0x4e93eb297f979a59), {"B43", "Sicilian Defense, Paulsen Variation"}},
    {Q_UINT64_C(0x9f0c44191e0fc82d), {"B44", "Sicilian Defense"}},
    {Q_UINT64_C(0x6a66ae741f9ff27b), {"B44", "Sicilian Defense, Szen"}},
    {Q_UINT64_C(0xfafac33779d0a6da), {"B44", "Sicilian Defense, Szen"}},
    {Q_UINT64_C(0x649b5cbbdced4af4), {"B45", "Sicilian Defense"}},
    {Q_UINT64_C(0x3f75684e71170b62), {"B45", "Sicilian Defense, Four Knights Variation"}},
    {Q_UINT64_C(0xb66cad483bcdb228), {"B46", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x9a8a1609db1a6a10), {"B46", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x6284639ab0a605f5), {"B47", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0xc3f3119056a3b78a), {"B47", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x118cd876319c1e7b), {"B48", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0xcd98bcd242c66542), {"B48", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x89044ffd352e9a7c), {"B48", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0xb0fbaa7cd799ac04), {"B49", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0xeb159e897a63ed92), {"B49", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x5c218cbb4d67fea0), {"B49", "Sicilian Defense, Taimanov Variation"}},
    {Q_UINT64_C(0x86933069b8c862bb), {"B50", "Sicilian Defense"}},
    {Q_UINT64_C(0xb54de1c6dd5cf138), {"B51", "Sicilian Defense, Rossolimo"}},
    {Q_UINT64_C(0x2b947ea49046ba1e), {"B51", "Sicilian Defense, Rossolimo"}},
    {Q_UINT64_C(0x578c1f9407062109), {"B52", "Sicilian Defense, Rossolimo"}},
    {Q_UINT64_C(0xbcfb5540fbafd9cb), {"B52", "Sicilian Defense, Rossolimo"}},
    {Q_UINT64_C(0x610fb232e064c902), {"B53", "Sicilian Defense, Chekhover Variation"}},
    {Q_UINT64_C(0x1bfd452d4e3b2fb3), {"B53", "Sicilian Defense, Chekhover Variation"}},
    {Q_UINT64_C(0x7b2cf8abc5c4b411), {"B54", "Sicilian Defense"}},
    {Q_UINT64_C(0x4e732a6fc8f41af6), {"B55", "Sicilian Defense"}},
    {Q_UINT64_C(0xf7d5f38a3e857143), {"B55", "Sicilian Defense"}},
    {Q_UINT64_C(0xdb55d4fcaadc775e), {"B56", "Sicilian Defense"}},
    {Q_UINT64_C(0x130bb836930e04fb), {"B57", ""}},
    {Q_UINT64_C(0x3388fe7ba6898fe), {"B58", "Sicilian, Boleslavsky Variation"}},
    {Q_UINT64_C(0x1d31bf80389aa602), {"B59", "Sicilian, Boleslavsky Variation"}},
    {Q_UINT64_C(0xb6e25b280bc70995), {"B60", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x7c3beaf510240604), {"B61", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xc095a91913e599c7), {"B62", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xe88de696d25c4667), {"B63", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x72822a2fd9661b28), {"B63", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xf0aeb597cb9816ca), {"B63", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x5ab505c08f9bca4f), {"B63", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x33d240e0871d310c), {"B64", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xb1fedf5895e33cee), {"B65", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x3a7a1765357cbebb), {"B66", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xdd24d3506f1dbd79), {"B66", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xe096ac14bc046112), {"B67", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xd988ef25ea690d4e), {"B67", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x1d80d119a5955c9b), {"B67", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x94ca2f66f3ef00a), {"B68", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xb2105b3eed55ab18), {"B69", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0xb696c31ab1685be9), {"B69", "Sicilian Defense, Richter-Rauzer Attack"}},
    {Q_UINT64_C(0x32e187a7d153e515), {"B70", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0xcff7faaac8c2d89c), {"B71", "Sicilian Defense, Levenfish Variation"}},
    {Q_UINT64_C(0x41e93c4b5069fe9b), {"B72", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x574d21d2c7690181), {"B73", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0xc994f44803272ea0), {"B74", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x603a1b9152036821), {"B74", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x9fe4e7cc0be4da40), {"B75", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x954c0a7bdebd33ab), {"B76", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x16b99d160e954908), {"B76", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x2e94d8355db74690), {"B76", "Sicilian Defense, Dragon Variation"}},
    {Q_UINT64_C(0x750a293e26d69fae), {"B77", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0x97cbd76cfc8c4f9f), {"B77", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0x11904b91b1aefc49), {"B77", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0xafe6924fafae4007), {"B78", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0xc318132dcb7e1319), {"B78", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0xde468d56cc5eb26e), {"B78", "Sicilian Dragon, Soltis Variation"}},
    {Q_UINT64_C(0x34185ad44aec7eb4), {"B78", "Sicilian Dragon, Soltis Variation"}},
    {Q_UINT64_C(0x374539aff990f76d), {"B78", "Sicilian, Chinese Dragon"}},
    {Q_UINT64_C(0x7695071406dc6188), {"B79", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0x14ca2fb997f8e127), {"B79", "Sicilian Dragon, Yugoslav Attack"}},
    {Q_UINT64_C(0xad2226cdb2fee70c), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x81c49d8c52293f34), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x626c8ae3742eeb2f), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x2a70af792e3a89b2), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xbaded5f226e78963), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x39ade62bb41dcb34), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x1993d424ce92647e), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x3263d85507972960), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xa125e12007f6bb0f), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x1004dbb7e211cfee), {"B80", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xe6f271b5a8b0a11), {"B81", "Sicilian Defense, Keres Attack"}},
    {Q_UINT64_C(0xaf906d043c91e943), {"B81", "Sicilian Defense, Keres Attack"}},
    {Q_UINT64_C(0x346b0b567f083b3f), {"B81", "Sicilian Defense, Keres Attack"}},
    {Q_UINT64_C(0x50345bc0ab6fda85), {"B82", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x8c08e41f1f248965), {"B82", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x263f3012dd129e22), {"B82", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xdea2a534b3dbadaf), {"B83", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x754f7dd6a24a08ac), {"B83", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xc5554c754fb5573), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x743f10d24443c162), {"B84", " Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x66ec5c1ce2a322f2), {"B84", " Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x9bfa2111fb321f7b), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x9152cca62e6bf690), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x3d6a392b176a7ae6), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x1976d35f2736ad0a), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x81863cb91461f766), {"B84", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xe25a774aaf51ed1e), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xbd75b9aed4e3e98e), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x74ab4e94e7344cb8), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x227ed144d3539dd3), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x2461ee65bf18d2d2), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x15caddcf5ffe709d), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0x99b1a8f6d8e951f), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xdc142af56c29d5ab), {"B85", "Sicilian Defense, Scheveningen Variation"}},
    {Q_UINT64_C(0xce9192e59abd31aa), {"B86", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0x1c6663167d9dc976), {"B86", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0xb56619216d1624af), {"B87", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0xcb13621511e5ffbf), {"B87", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0x657c4a078b2c94a9), {"B88", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0xb982f1b0a5b681e), {"B88", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0xdc42c518019fb9bb), {"B88", "Sicilian Defense, Sozin"}},
    {Q_UINT64_C(0x1674f1eb0a168f27), {"B89", "Sicilian Defense, Velimirovic Attack"}},
    {Q_UINT64_C(0xc4830018ed3677fb), {"B89", "Sicilian Defense, Velimirovic Attack"}},
    {Q_UINT64_C(0x9d9ea039441f339a), {"B89", "Sicilian Defense, Velimirovic Attack"}},
    {Q_UINT64_C(0x9a2250f4dfc8f82), {"B90", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xaf906d043c91e943), {"B90", "Sicilian Najdorf, Perenyi Attack"}},
    {Q_UINT64_C(0x626c8ae3742eeb2f), {"B90", "Sicilian Najdorf, English Attack"}},
    {Q_UINT64_C(0x28299e94b133c076), {"B90", "Sicilian Najdorf, English Attack"}},
    {Q_UINT64_C(0xf5556e616ead3d1d), {"B90", "Sicilian Najdorf, English Attack"}},
    {Q_UINT64_C(0xf0d296a95f236c3a), {"B90", "Sicilian Najdorf, English Attack"}},
    {Q_UINT64_C(0x25449e4ead2b57ba), {"B91", "Sicilian Defense Najdorf, Zagreb Var."}},
    {Q_UINT64_C(0x7a22a6f64cd9c521), {"B92", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x439bbfa907a49b5b), {"B92", "Najdorf Variation"}},
    {Q_UINT64_C(0xf4b45802546db20b), {"B93", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xcff87239fd76544a), {"B94", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xb98f8008e554c418), {"B95", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x4499fd05fcc5f991), {"B96", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x249d9710676844ac), {"B96", "Sicilian Defense, Polugaevsky Variation"}},
    {Q_UINT64_C(0x777a86581c6d145d), {"B96", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xe763365b3e6200d8), {"B96", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xcb47c24a5c5b0de7), {"B96", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x83660352368e2f4b), {"B97", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x38ec8a1702994133), {"B97", "Sicilian Najdorf, Poisioned Pawn Variation"}},
    {Q_UINT64_C(0x78619c1b10fd84a2), {"B97", "Sicilian Najdorf, Poisioned Pawn Variation"}},
    {Q_UINT64_C(0x50558eea366e5500), {"B98", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xea77f5d91a1303bb), {"B98", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xe7a6f486c5d2aeee), {"B99", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x700cc5dd428d5c2f), {"B99", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x5c5e54daa09d8fc3), {"B99", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x3fbe2c4c1616d730), {"B99", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0x44ebf5502da743f3), {"B99", "Sicilian Defense, Najdorf Variation"}},
    {Q_UINT64_C(0xf44b6961e533d1c4), {"C00", "French Defense"}},
    {Q_UINT64_C(0x5aefaab4a1a36a10), {"C00", "French, King's Indian Attack"}},
    {Q_UINT64_C(0x624de643ab778a40), {"C00", "French, Chigorin Variation"}},
    {Q_UINT64_C(0x51d2bf21d3e37c06), {"C00", "French Defense, Two Knights Variation"}},
    {Q_UINT64_C(0x87b8157c4ceebfe2), {"C00", "French Defense, Wing Gambit"}},
    {Q_UINT64_C(0xacdfb756a8e4306d), {"C01", "French Defense, Exchange Variation"}},
    {Q_UINT64_C(0x77bb5ba0c0025150), {"C01", "French Defense, Exchange Variation"}},
    {Q_UINT64_C(0xa11d1f363b54ba5d), {"C01", "French Defense, Winawer Exchange"}},
    {Q_UINT64_C(0xd56d72226c3094c4), {"C02", "French Defense, Advance Variation"}},
    {Q_UINT64_C(0x45c65b144307903f), {"C02", "French Defense, Advance Variation"}},
    {Q_UINT64_C(0xa7bd30f5c9bb40a8), {"C02", "French Defense, Milner-Barry Gambit"}},
    {Q_UINT64_C(0xed7adcbdf2d31a90), {"C03", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x304dfcc73332b015), {"C03", "French Tarrasch, Guimard Variation"}},
    {Q_UINT64_C(0x29d436a205b4363), {"C03", "French Tarrasch, Guimard Variation"}},
    {Q_UINT64_C(0xc61ddc5c265e9f38), {"C04", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xb694e8485f295b06), {"C05", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xa7ed29d3c0aa7f02), {"C06", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x8f49327c5390dab0), {"C06", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x9b854193993b7621), {"C06", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x544546aad9f61ae2), {"C06", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x80653c971ecf0584), {"C06", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xb0b0d130d9411ed), {"C07", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x29810168252880c), {"C07", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x641ba3deed591343), {"C07", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x13c0de9acd5e9320), {"C08", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x1a53c39f42980ac1), {"C08", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x42552964b78cf039), {"C08", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xb790fd47c47850b), {"C08", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x6349ea8eb429571e), {"C09", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xf4cf56a30ea715b9), {"C09", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xc52cd58876b246bc), {"C09", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0xa4983489a736e563), {"C09", "French Defense, Tarrasch Variation"}},
    {Q_UINT64_C(0x4f837c7daacc3079), {"C10", "French Defense"}},
    {Q_UINT64_C(0x2dcbdd8723c311be), {"C10", "French Defense, Rubinstein Variation"}},
    {Q_UINT64_C(0x6f885037e86a9082), {"C10", "French, Fort Knox Variation"}},
    {Q_UINT64_C(0xec80d7c915f45ff7), {"C10", "French Defense, Rubinstein"}},
    {Q_UINT64_C(0x4ff17ebadb1fa49f), {"C10", "French Defense"}},
    {Q_UINT64_C(0xe9850370a6e17acd), {"C10", "French Defense"}},
    {Q_UINT64_C(0x146d4888073671ef), {"C11", "French Defense"}},
    {Q_UINT64_C(0x75145e750328578b), {"C11", "French Defense, Steinitz Variation"}},
    {Q_UINT64_C(0xb146608b3945343e), {"C11", "French Defense, Steinitz Variation"}},
    {Q_UINT64_C(0x4b1041ba6564b189), {"C11", "French Defense, Classical, Burn Var."}},
    {Q_UINT64_C(0x2ad1e51d4ebeeff9), {"C11", "French Defense, Classical, Burn Var."}},
    {Q_UINT64_C(0x1b7676b8fc4b6205), {"C12", "French Defense, MacCutcheon Variation"}},
    {Q_UINT64_C(0x1abd80b77e451a77), {"C13", "French Defense, Burn Variation"}},
    {Q_UINT64_C(0xff4d2de59a671b0), {"C13", "French Defense"}},
    {Q_UINT64_C(0xc608d90464a00d6f), {"C13", "French Defense"}},
    {Q_UINT64_C(0xc6fb6c517d1706b6), {"C13", "French Defense, Classical Variation"}},
    {Q_UINT64_C(0x9758f830e3a3c6ed), {"C13", "French Defense, Classical Variation"}},
    {Q_UINT64_C(0xc266e5237ffa5059), {"C14", "French Defense, Alekhine-Chatard Attack"}},
    {Q_UINT64_C(0x2d401dd965fe75de), {"C14", ""}},
    {Q_UINT64_C(0x86c2157be13bf85b), {"C15", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0xaf4f4fce735a935f), {"C15", "French, Winawer, Fingerslip Variation"}},
    {Q_UINT64_C(0x9266feb4c910f5a5), {"C15", "French, Winawer, Alekhine Gambit"}},
    {Q_UINT64_C(0x4d5e245da877761b), {"C15", "French, Winkelmann-Reimer Gambit"}},
    {Q_UINT64_C(0x81894c5271a09470), {"C15", "French, Winkelmann-Reimer Gambit"}},
    {Q_UINT64_C(0xa4507984b302190b), {"C15", "French, Winkelmann-Reimer Gambit"}},
    {Q_UINT64_C(0xe7bb0386e525de3f), {"C16", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x1cad2281a62d542), {"C17", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x10e69c2078695a4d), {"C17", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0xe97aedccb2ba6610), {"C18", "French, Winawer, Armenian Variation"}},
    {Q_UINT64_C(0xa0baf239e3cae648), {"C18", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0xcb4f3c2a8ea437f8), {"C18", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x6339af69cf92777a), {"C18", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x4591a27230a50cc8), {"C19", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x6f8c86ab03be56c3), {"C19", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x543eeda27096a77d), {"C19", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0xeae667af080ac38d), {"C19", "French Defense, Winawer Variation"}},
    {Q_UINT64_C(0x844931a6ef4b9a0), {"C20", ""}},
    {Q_UINT64_C(0x560fb632bb4680f7), {"C21", "Center Game"}},
    {Q_UINT64_C(0x20d54eaa6b368f71), {"C21", "Danish Gambit"}},
    {Q_UINT64_C(0x805579149e46cac9), {"C22", ""}},
    {Q_UINT64_C(0x6bf7273246b76f06), {"C23", "Bishop's Opening"}},
    {Q_UINT64_C(0x301913c7eb4d2e90), {"C24", "Bishop's Opening"}},
    {Q_UINT64_C(0xb536da19561976fa), {"C24", "Urusov Gambit"}},
    {Q_UINT64_C(0x4f3ecf1bd616b32e), {"C24", "Urusov Gambit"}},
    {Q_UINT64_C(0xf3d38bb8ac163b79), {"C25", "Vienna Game"}},
    {Q_UINT64_C(0xd2b02748784217d2), {"C25", "Vienna Game, Pierce Gambit"}},
    {Q_UINT64_C(0x8b2694cbc2b443ee), {"C25", "Vienna Game, Hamppe-Allgaier Gambit"}},
    {Q_UINT64_C(0x89973b6384108965), {"C25", "Vienna Game, Steinitz Gambit"}},
    {Q_UINT64_C(0xa83dbf4d01ec7aef), {"C26", "Vienna Game"}},
    {Q_UINT64_C(0x7639453c2f780b5a), {"C27", "Vienna Game"}},
    {Q_UINT64_C(0xad5da9ca479e6a67), {"C27", "Boden-Kieseritsky Gambit"}},
    {Q_UINT64_C(0xc8378ceba70dfd1e), {"C27", "Vienna Game"}},
    {Q_UINT64_C(0xf18d1a8a7575b473), {"C27", "Vienna Game"}},
    {Q_UINT64_C(0x6063d387383e094a), {"C28", "Vienna Game"}},
    {Q_UINT64_C(0x9d75ae8a21af34c3), {"C28", "Vienna Game"}},
    {Q_UINT64_C(0xd041e05484a90940), {"C29", "Vienna Gambit"}},
    {Q_UINT64_C(0xf552ee1777658429), {"C30", "King's Gambit"}},
    {Q_UINT64_C(0x584faa4cbd576eaa), {"C30", "King's Gambit Declined"}},
    {Q_UINT64_C(0xaebcdae2da9fc5bf), {"C30", "King's Gambit Declined"}},
    {Q_UINT64_C(0x7038cc03ebb1ca0f), {"C31", "Falkbeer Counter Gambit"}},
    {Q_UINT64_C(0x8dab0c00eab1adbd), {"C32", "Falkbeer Counter Gambit"}},
    {Q_UINT64_C(0x1cd8d48946aa83fd), {"C33", "King's Gambit Accepted"}},
    {Q_UINT64_C(0x7f6b60a16ee9555b), {"C33", "King's Gambit Accepted"}},
    {Q_UINT64_C(0xc7bc387f2e4ce2c0), {"C34", "King's Gambit Accepted"}},
    {Q_UINT64_C(0xd3704b90e4e74e51), {"C35", "King's Gambit, Cunningham Variation"}},
    {Q_UINT64_C(0x42d61a6bb298ace6), {"C36", "King's Gambit, Abbazia Defense"}},
    {Q_UINT64_C(0x47ffc8a2baf81d4a), {"C37", "King's Gambit"}},
    {Q_UINT64_C(0x82cae708ab313008), {"C37", "King's Gambit, Rosentreter"}},
    {Q_UINT64_C(0x5edf6b3f6fc79093), {"C37", "King's Gambit, Muzio"}},
    {Q_UINT64_C(0x81194027fbd5e9f6), {"C37", "King's Gambit, Lolli"}},
    {Q_UINT64_C(0xe59f3fa102fd66c1), {"C37", "Ghulam Kassim Gambit"}},
    {Q_UINT64_C(0xdb3d08a9d1d6c95a), {"C37", "King's Gambit, MacDonnell"}},
    {Q_UINT64_C(0x94f0413c69fc0046), {"C38", "King's Gambit, Hanstein"}},
    {Q_UINT64_C(0xdb5c548b11c76434), {"C39", "King's Gambit"}},
    {Q_UINT64_C(0x8968d9df39841cff), {"C39", "King's Gambit, Allgaier"}},
    {Q_UINT64_C(0x282e7c5cbc5812d), {"C39", "King's Gambit, Kieseritsky"}},
    {Q_UINT64_C(0x596cd330663fc0bb), {"C39", "KGA Kieseritsky, Berlin Defense"}},
    {Q_UINT64_C(0xb23eda7330824a87), {"C39", "KGA Kieseritsky, Paulsen Variation"}},
    {Q_UINT64_C(0x997d38b50af92f47), {"C39", "KGA Kieseritsky, Salvio Variation"}},
    {Q_UINT64_C(0x424759f7ab2e1009), {"C39", "KGA Kieseritsky, Rice Gambit"}},
    {Q_UINT64_C(0xd3207fec0612d89d), {"C40", "King Pawn"}},
    {Q_UINT64_C(0x1f7579a53cf9cfe2), {"C40", "Damiano Defense"}},
    {Q_UINT64_C(0x474f1975b89af543), {"C40", "Latvian Gambit"}},
    {Q_UINT64_C(0x564a5df89ac696bb), {"C40", "Elephant Gambit"}},
    {Q_UINT64_C(0x48dfa09cc72e76f7), {"C40", "Gunderam Defense"}},
    {Q_UINT64_C(0xea9ae98dd46a91f0), {"C41", "Philidor's Defense"}},
    {Q_UINT64_C(0x8fd6ea70aabb7ffd), {"C41", "Philidor's Defense"}},
    {Q_UINT64_C(0x67f0ce40b55a5557), {"C41", "Philidor, Antoshin Variation"}},
    {Q_UINT64_C(0xcb4a56eaa8fbe806), {"C41", "Philidor, Hanham Variation"}},
    {Q_UINT64_C(0xbbc0a0be7b2b916c), {"C41", "Philidor Countergambit"}},
    {Q_UINT64_C(0x88ce4b19abe8990b), {"C42", "Petroff Defense"}},
    {Q_UINT64_C(0x735953bb690a1bd2), {"C42", "Petroff Defense"}},
    {Q_UINT64_C(0x368dbfe20ff63c3d), {"C42", "Petroff Defense, Classical Variation"}},
    {Q_UINT64_C(0xfbd062917a039c38), {"C42", "Petroff Defense, Cochrane Gambit"}},
    {Q_UINT64_C(0x4dfb64b3ba21b449), {"C43", "Petroff Defense, Steinitz Variation"}},
    {Q_UINT64_C(0x78cda70e17837d9e), {"C44", "King Pawn"}},
    {Q_UINT64_C(0xe175f96c7f37218), {"C44", "Ponziani Opening"}},
    {Q_UINT64_C(0xd66964db5313c64a), {"C44", "King Pawn"}},
    {Q_UINT64_C(0xbdf888a4064a50dc), {"C44", "Scotch Opening"}},
    {Q_UINT64_C(0x505c7abe12414b4f), {"C44", "Scotch Opening, Goering Gambit"}},
    {Q_UINT64_C(0xbb24e4bbfbb0ad9), {"C44", "Scotch Opening, Goering Gambit"}},
    {Q_UINT64_C(0x4112df7bd300d6e7), {"C45", "Scotch Opening"}},
    {Q_UINT64_C(0x835abfacd561ff47), {"C46", "Three Knights' Game"}},
    {Q_UINT64_C(0xd8b48b59789bbed1), {"C47", "Four Knights' Game"}},
    {Q_UINT64_C(0x1328ba7f31d73091), {"C47", "Four Knights' Game"}},
    {Q_UINT64_C(0x2a922c1ee3af79fc), {"C47", "Four Knights' Game"}},
    {Q_UINT64_C(0x1d81a4f369529393), {"C47", "Scotch Four Knights' Game"}},
    {Q_UINT64_C(0xe16bf32cbc1815a8), {"C47", "Scotch Four Knights' Game"}},
    {Q_UINT64_C(0x46ea005db768f8d3), {"C47", "Belgrade Gambit"}},
    {Q_UINT64_C(0xb39add5c411359b7), {"C47", "Four Knights' Game"}},
    {Q_UINT64_C(0xeb6a5af61d0f2d52), {"C48", "Four Knights' Game"}},
    {Q_UINT64_C(0x3c173617a34b6df3), {"C48", "Four Knights' Game, Rubinstein Variation"}},
    {Q_UINT64_C(0x222b33f056f8e570), {"C49", "Four Knights' Game"}},
    {Q_UINT64_C(0x1b7e13263fc0ab38), {"C50", ""}},
    {Q_UINT64_C(0xfb260c9f56b07a9), {"C50", "Hungarian Defense"}},
    {Q_UINT64_C(0xb663577df5f241bb), {"C50", "Giuoco Piano"}},
    {Q_UINT64_C(0x1c9622d153442d9f), {"C51", "Evans Gambit"}},
    {Q_UINT64_C(0x768c3ac4890df01b), {"C52", "Evans Gambit"}},
    {Q_UINT64_C(0x976dca2683dd88cc), {"C52", "Evans Gambit, Compromised Defense"}},
    {Q_UINT64_C(0xc0b9afe525824e3d), {"C53", "Giuoco Piano"}},
    {Q_UINT64_C(0x9b579b1088780fab), {"C54", "Giuoco Piano"}},
    {Q_UINT64_C(0x35f358c5cce8b47f), {"C54", "Giuoco Piano"}},
    {Q_UINT64_C(0xc49cea41e90fd12), {"C54", "Giuoco Piano"}},
    {Q_UINT64_C(0x78945827b73acfe9), {"C54", "Giuoco Piano"}},
    {Q_UINT64_C(0x69aa6b7439623ada), {"C54", "Giuoco Piano, Moeller Attack"}},
    {Q_UINT64_C(0x409027d3923aeaae), {"C55", "Two Knights' Defense"}},
    {Q_UINT64_C(0x835212bee81e96ad), {"C55", "Boden Gambit"}},
    {Q_UINT64_C(0xee34e406d6aa517a), {"C55", "Two Knights' Defense"}},
    {Q_UINT64_C(0x1edb02fb4788d3f9), {"C55", "Two Knights' Defense"}},
    {Q_UINT64_C(0xcdb33d94f149e26a), {"C55", "Two Knights' Defence"}},
    {Q_UINT64_C(0x729c587ce06a28d6), {"C55", "Two Knights' Defense"}},
    {Q_UINT64_C(0xdd1937963dacaffa), {"C56", "Two Knights' Defense"}},
    {Q_UINT64_C(0x1642c6063bf6120a), {"C57", "Two Knights' Defense"}},
    {Q_UINT64_C(0xbb5f825df1c4f889), {"C57", "Two Knights' Defense, Traxler Gambit"}},
    {Q_UINT64_C(0xd4f3844ae6e0a317), {"C57", "Two Knights' Defense, Ulvestad Variation"}},
    {Q_UINT64_C(0x638a82bec3095e8b), {"C57", "Two Knights' Defense, Fritz Variation"}},
    {Q_UINT64_C(0x5414e11a976477cc), {"C57", "Two Knights' Defense, Fried Liver Attack"}},
    {Q_UINT64_C(0x276dd2be9896e616), {"C57", "Two Knights' Defense, Pincus Variation"}},
    {Q_UINT64_C(0xe1e0c4aad071bc20), {"C58", "Two Knights' Defense"}},
    {Q_UINT64_C(0x30dc3abcf7a80833), {"C58", "Two Knights' Defense, Colman var."}},
    {Q_UINT64_C(0x26d4332e603fe217), {"C59", ""}},
    {Q_UINT64_C(0x4b1376a17217ee1d), {"C60", "Ruy Lopez"}},
    {Q_UINT64_C(0xa2a725fa09987c56), {"C60", "Ruy Lopez, Smyslov Defense"}},
    {Q_UINT64_C(0x755cca1cc99e65a0), {"C60", "Ruy Lopez, Cozio Defense"}},
    {Q_UINT64_C(0x9c6e1a40cc53aebc), {"C61", "Ruy Lopez, Bird's Defense"}},
    {Q_UINT64_C(0x72a9e0c0a06fa770), {"C62", "Ruy Lopez, Old Steinitz Defense"}},
    {Q_UINT64_C(0xdf7c1038cc9fc3c3), {"C63", "Ruy Lopez, Schliemann Defense"}},
    {Q_UINT64_C(0xe60e32fab825049e), {"C64", "Ruy Lopez, Classical Defense"}},
    {Q_UINT64_C(0x4bbacfbd6dd26c6), {"C64", "Ruy Lopez, Classical, Cordel Gambit"}},
    {Q_UINT64_C(0x10fd4254dfedaf8b), {"C65", "Ruy Lopez, Berlin Defense"}},
    {Q_UINT64_C(0x5732af0171663df6), {"C66", "Ruy Lopez, Berlin Defense"}},
    {Q_UINT64_C(0xd33f7739a5c9d388), {"C67", "Ruy Lopez, Berlin Defense"}},
    {Q_UINT64_C(0xc8cbbdb43361c6fa), {"C68", "Ruy Lopez, Exchange Variation"}},
    {Q_UINT64_C(0x35028f47dea98b8a), {"C68", "Ruy Lopez, Exchange Var., Bronstein"}},
    {Q_UINT64_C(0x4c3be29eec5b90ea), {"C69", "Ruy Lopez, Exchange Variation"}},
    {Q_UINT64_C(0xa8e7c91161416feb), {"C70", "Ruy Lopez"}},
    {Q_UINT64_C(0x915d5f70b3392686), {"C71", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0xef282444cfcafd96), {"C72", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0x8c24cc14dccc3f6), {"C73", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0xe787a7e863492900), {"C74", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0x73e8c171ddc104de), {"C74", "Ruy Lopez, Siesta Variation"}},
    {Q_UINT64_C(0x54659bab913f931), {"C75", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0x29c7254bd3554638), {"C76", "Ruy Lopez, Modern Steinitz Defense"}},
    {Q_UINT64_C(0xf309fde4ccbb2e7d), {"C77", "Ruy Lopez"}},
    {Q_UINT64_C(0x5dad3e31882b95a9), {"C77", "Ruy Lopez"}},
    {Q_UINT64_C(0xfc83b9681b1a6591), {"C77", "Ruy Lopez"}},
    {Q_UINT64_C(0xfb790393f1ac4109), {"C77", "Ruy Lopez"}},
    {Q_UINT64_C(0x8d7c86d0b048f56d), {"C78", "Ruy Lopez"}},
    {Q_UINT64_C(0xed78ecc52be54850), {"C78", "Ruy Lopez"}},
    {Q_UINT64_C(0xd54b90dc4170e29), {"C78", "Ruy Lopez, Archangelsk Variation"}},
    {Q_UINT64_C(0x10d9b5d27c774b31), {"C78", "Ruy Lopez, Archangelsk Variation"}},
    {Q_UINT64_C(0xb4c610b16230bc00), {"C79", "Ruy Lopez"}},
    {Q_UINT64_C(0x30cbc889b69f527e), {"C80", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xf85db8a83a0554ce), {"C80", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x1e0f50da334bf239), {"C81", "Ruy Lopez, Open, Howell Attack"}},
    {Q_UINT64_C(0xfed32760ad7fa63b), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x53ce633b674d4cb8), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xaa0db59fdb0e488), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x1a30d2c6f63a7e29), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xf0c7fc282d56d881), {"C82", "Ruy Lopez, Open, Dilworth Variation"}},
    {Q_UINT64_C(0xefca1293c91d2b8), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xea09b4416f683fa5), {"C82", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xea1f548f67d40aaa), {"C83", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x9917ef63e6ee1124), {"C83", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0xb371ecedfd29a29a), {"C83", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x30bb25677d90335d), {"C83", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x422cb50b50b908b4), {"C83", "Ruy Lopez, Open"}},
    {Q_UINT64_C(0x99b0f53f7ae359fc), {"C84", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x5c85da956b2a74be), {"C84", "Ruy Lopez, Center Attack"}},
    {Q_UINT64_C(0x363cd24edd72033f), {"C84", "Ruy Lopez, Center Attack"}},
    {Q_UINT64_C(0xc4a741032d6086c6), {"C84", "Ruy Lopez, Center Attack"}},
    {Q_UINT64_C(0xa682c6eaab4f46cf), {"C84", "Ruy Lopez, Center Attack"}},
    {Q_UINT64_C(0x379a85671ec6ef9b), {"C85", "Ruy Lopez, Delayed Exchange"}},
    {Q_UINT64_C(0xfb67a1d34a70278), {"C86", "Ruy Lopez, Worrall Attack"}},
    {Q_UINT64_C(0x24be6dd52f33bb85), {"C87", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x265611607b7408ab), {"C87", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x2dbc8cedfb46a637), {"C88", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x2714615a2e1f4fdc), {"C88", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x1d89fa1e7746c8b6), {"C88", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x4cdd1fb6438c9a12), {"C88", "Ruy Lopez, Anti-Marshall"}},
    {Q_UINT64_C(0xd4a4bbd662bb0e7c), {"C89", "Ruy Lopez, Marshall Gambit"}},
    {Q_UINT64_C(0x1191947c7372233e), {"C89", "Ruy Lopez, Marshall Gambit"}},
    {Q_UINT64_C(0x68740fa32c170937), {"C90", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0xc6d0cc766887b2e3), {"C90", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x93c7e683c5bb10e9), {"C90", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0xad4120093dde2475), {"C91", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x8087d21deeb4c0c5), {"C92", "Ruy Lopez, Closed"}},
    {Q_UINT64_C(0x30179a999ce66f5e), {"C92", "Ruy Lopez, Zaitsev Variation"}},
    {Q_UINT64_C(0x1270997b796089d0), {"C93", "Ruy Lopez, Smyslov System"}},
    {Q_UINT64_C(0x86b6ba1c0a8f45b1), {"C93", "Ruy Lopez, Smyslov System"}},
    {Q_UINT64_C(0x2b6a0affff2565c6), {"C94", "Ruy Lopez, Breyer Variation"}},
    {Q_UINT64_C(0xee5f2555eeec4884), {"C95", "Ruy Lopez, Breyer Variation"}},
    {Q_UINT64_C(0xd590f8e8438862cf), {"C96", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0x25a99e6fdada2343), {"C96", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0x61af0cb03b51a101), {"C96", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0xf25f4a581f4a7d), {"C97", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0xc8bcddd6fde4047), {"C98", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0xaa348851dbbe87ad), {"C99", "Ruy Lopez, Chigorin Variation"}},
    {Q_UINT64_C(0x6649ba69b8c9ff8), {"D00", "Queen Pawn"}},
    {Q_UINT64_C(0xdd307d2702ec759b), {"D00", "Queen Pawn, Mason Variation"}},
    {Q_UINT64_C(0x7d795b079c723d8a), {"D00", "Queen Pawn"}},
    {Q_UINT64_C(0xc26396ee700c22f2), {"D00", "Blackmar-Diemer Gambit"}},
    {Q_UINT64_C(0x621abab91f14e1bd), {"D00", "Blackmar-Diemer Gambit"}},
    {Q_UINT64_C(0xaa9025b0d6ed51ed), {"D00", "Blackmar-Diemer Gambit"}},
    {Q_UINT64_C(0xc9bd95bd49732ab6), {"D00", "Blackmar-Diemer Gambit"}},
    {Q_UINT64_C(0xa8fb02780de81d30), {"D00", "BDG: Ryder Gambit"}},
    {Q_UINT64_C(0x621abab91f14e1bd), {"D00", "BDG: Huebsch Gambit"}},
    {Q_UINT64_C(0xe862b2f38cf1198b), {"D00", "BDG: Lemberger Counter-Gambit"}},
    {Q_UINT64_C(0xbfca678c5151bae4), {"D00", "BDG: Euwe Defense"}},
    {Q_UINT64_C(0x5bc778523eb3f90d), {"D00", "BDG: Ziegler Defense"}},
    {Q_UINT64_C(0x2009c6e632fcb8fd), {"D00", "BDG: Bogoljubow Defense"}},
    {Q_UINT64_C(0xcb6e5f8f0ef6245c), {"D00", "BDG: Teichmann Defense"}},
    {Q_UINT64_C(0xff46a1c1162bc3ec), {"D00", "Morris Gambit"}},
    {Q_UINT64_C(0xa61db7f1f4945cb7), {"D01", "Veresov Opening"}},
    {Q_UINT64_C(0x6047e0c7441e877f), {"D01", "Veresov Opening"}},
    {Q_UINT64_C(0x86363169bb598c02), {"D01", "Veresov Opening"}},
    {Q_UINT64_C(0xdd007750f36afec5), {"D02", "Queen Pawn"}},
    {Q_UINT64_C(0x5dbaa524c7f05530), {"D02", "Queen Pawn, London System"}},
    {Q_UINT64_C(0x16e636381ce8d6e9), {"D02", "Queen Pawn, London System"}},
    {Q_UINT64_C(0x599c656cdf0c60e0), {"D02", "Queen Pawn, London System"}},
    {Q_UINT64_C(0x86e015a7fb8d4dc6), {"D02", "Queen Pawn, London System"}},
    {Q_UINT64_C(0xd8e2cb360dea06b9), {"D02", "Queen Pawn"}},
    {Q_UINT64_C(0x40b41493ee1a649b), {"D03", "Queen Pawn"}},
    {Q_UINT64_C(0x36c3e6a2f638f4c9), {"D03", "Torre Attack"}},
    {Q_UINT64_C(0x9beea410d2677c6d), {"D03", "Torre Attack"}},
    {Q_UINT64_C(0x7d9f75be2d207710), {"D03", "Torre Attack"}},
    {Q_UINT64_C(0x2bc301177acf37f7), {"D04", "Queen Pawn"}},
    {Q_UINT64_C(0x5db4f32662eda7a5), {"D05", "Colle System"}},
    {Q_UINT64_C(0x8b952845b6d65d6), {"D05", "Colle System"}},
    {Q_UINT64_C(0x3b5020fcc94810b8), {"D05", "Colle System"}},
    {Q_UINT64_C(0xa35e834338715877), {"D05", "Colle System"}},
    {Q_UINT64_C(0x8a470482d88334ff), {"D06", "Queen's Gambit"}},
    {Q_UINT64_C(0x95e6d0c6f663d72), {"D06", "QGD, Keres Defense"}},
    {Q_UINT64_C(0x5fda6b69dd404ec4), {"D06", "QGD, Keres Defense"}},
    {Q_UINT64_C(0x21aadc60c91291fc), {"D07", "QGD, Chigorin Defense"}},
    {Q_UINT64_C(0x3f0cc84b66ccc9), {"D08", "Albin Counter Gambit"}},
    {Q_UINT64_C(0x22a016b116cc4d58), {"D09", ""}},
    {Q_UINT64_C(0x183de96daf43e744), {"D10", "Slav Defense"}},
    {Q_UINT64_C(0xe3aaf1cf6da1659d), {"D10", "Slav Defense"}},
    {Q_UINT64_C(0x69d2f985fe449dab), {"D10", "Slav Winawer Counter Gambit"}},
    {Q_UINT64_C(0xeb2ada825145bc85), {"D10", "Slav Defense"}},
    {Q_UINT64_C(0xc359059bc7a58679), {"D11", "Slav Defense"}},
    {Q_UINT64_C(0x98b7316e6a5fc7ef), {"D11", "Slav Defense"}},
    {Q_UINT64_C(0xb6831a52f9e546c6), {"D12", "Slav Defense"}},
    {Q_UINT64_C(0xe5ed05ea68fbde48), {"D13", "Slav Defense, Exchange Variation"}},
    {Q_UINT64_C(0x1e7a1d48aa195c91), {"D13", "Slav Defense, Exchange Variation"}},
    {Q_UINT64_C(0xbda08ae5467e3da6), {"D14", "Slav Defense, Exchange Variation"}},
    {Q_UINT64_C(0xa16a4759ec8f5b03), {"D14", "Slav Defense, Exchange Variation"}},
    {Q_UINT64_C(0x632029cca8bd4536), {"D15", "Slav Defense"}},
    {Q_UINT64_C(0x75211db6110316df), {"D15", "Slav Defense, Geller Gambit"}},
    {Q_UINT64_C(0x9fe3f303dc273cc3), {"D15", "Slav Defense"}},
    {Q_UINT64_C(0xdaef6e1297107e1b), {"D16", "Slav Defense"}},
    {Q_UINT64_C(0x59f6079c20f57796), {"D17", "Slav Defense"}},
    {Q_UINT64_C(0xf4db452e04aaff32), {"D18", "Slav Defense"}},
    {Q_UINT64_C(0x4d5774db19058a8c), {"D19", "Slav Defense"}},
    {Q_UINT64_C(0x7fddaaed90b1d125), {"D19", "Slav Defense"}},
    {Q_UINT64_C(0x58413db08abdda1c), {"D20", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x8325d146e25bbb21), {"D21", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xf5522377fa792b73), {"D21", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xfcff62072124cb59), {"D22", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xd8cbe5b34fa1fab7), {"D23", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xa6f58fc8f9f08cc), {"D23", "QGA, Mannheim Variation"}},
    {Q_UINT64_C(0x878fb26f555f1a47), {"D23", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x86dd170ce921a32f), {"D23", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x235cfd118d43786e), {"D24", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x552b0f209561e83c), {"D24", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x75e6a7016bfe7213), {"D25", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x47c5f4ae29eadf61), {"D25", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x391553073dce241), {"D26", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x458f93aef7f5486e), {"D26", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x4b726c94419b853c), {"D27", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xad03bd3abedc8e41), {"D27", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x6407ad1835fadea5), {"D27", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xc6cac3d6d34f5b8f), {"D27", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xde75def661d2a8a9), {"D27", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x3b053218f098d5c5), {"D28", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x95696b2dd5d1f924), {"D28", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xb7b984db8464f22), {"D28", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x3856ba3538da038c), {"D28", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0x229500ab92419787), {"D29", "Queen's Gambit Accepted"}},
    {Q_UINT64_C(0xfc30f6b3c0a1a4ad), {"D30", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0xef7892f50e99a952), {"D30", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x8f7cf8e09534146f), {"D30", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x7a7ee1102432674), {"D31", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x51da0eb69e0348c5), {"D31", "Slav Defense, Marshall Gambit"}},
    {Q_UINT64_C(0x9cbfd63a4f5b7a11), {"D31", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x15fa33eadf3e1d7d), {"D31", ""}},
    {Q_UINT64_C(0xe1d63fbffd042d09), {"D32", "Tarrasch Defense"}},
    {Q_UINT64_C(0xb0e21e47223e50d2), {"D32", "Tarrasch Defense, von Henning-Schara Gambit"}},
    {Q_UINT64_C(0xd0e87db6456ff9df), {"D32", "Tarrasch Defense"}},
    {Q_UINT64_C(0xb8c91402d8998e2), {"D32", "Tarrasch Defense"}},
    {Q_UINT64_C(0xbf37f77066b64c89), {"D32", "Tarrasch Defense"}},
    {Q_UINT64_C(0x8c87f2e3dccfe5d9), {"D33", "Tarrasch Defense"}},
    {Q_UINT64_C(0x76559684f673d0a7), {"D34", "Tarrasch Defense, Prague Variation"}},
    {Q_UINT64_C(0x6a78272fe06f3e6e), {"D34", "Tarrasch Defense, Prague Variation"}},
    {Q_UINT64_C(0xc4d25731ef533994), {"D34", "Tarrasch Defense, Prague Variation"}},
    {Q_UINT64_C(0x5c49dae4afb967e2), {"D35", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0xb613741b7f34d209), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xab2dcfdba75868fc), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x39572234d098bb47), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xd8a59aafa3936e36), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xccde0428713c061d), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x8cdb26bfd7a374ef), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x506703248c9f56fa), {"D35", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x884e5029c24e8723), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xbbad2b7422e66aef), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x60c9c7824a000bd2), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xd928f6dfa4f42fe7), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x5741bb8bf5928ca9), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x868dbaca482d047f), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0xac7726d87a28dd40), {"D36", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x872d3612c75f06df), {"D37", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x421d4ecb41cda9c6), {"D37", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x387f9fde22befec9), {"D37", "QGD, Exchange Variation"}},
    {Q_UINT64_C(0x73a11080fec097ec), {"D38", "QGD, Ragozin Variation"}},
    {Q_UINT64_C(0x4e6c5f148ca8cefd), {"D38", "QGD, Ragozin Variation"}},
    {Q_UINT64_C(0x883608223c221535), {"D38", "QGD, Ragozin Variation"}},
    {Q_UINT64_C(0xbbd5737fdc8af8f9), {"D38", "QGD, Ragozin Variation"}},
    {Q_UINT64_C(0x5a3031106e1cfbd6), {"D39", "QGD, Vienna Variation"}},
    {Q_UINT64_C(0x9e373c58859c46dc), {"D39", "QGD, Vienna Variation"}},
    {Q_UINT64_C(0x615ce7bc38180da2), {"D40", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x2b5f3a26169696e3), {"D40", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x3ff2f1aa61239319), {"D40", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x6f767c719ad619bf), {"D41", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0xf098063c253304e3), {"D41", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x956b1c717167c6a9), {"D41", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x6675aec6111982c1), {"D41", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0xf6589c6c10fd2944), {"D42", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0xa3553dce297deb37), {"D42", "QGD, Semi-Tarrasch Variation"}},
    {Q_UINT64_C(0x1557dbfdb09fd564), {"D43", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xd30d8ccb00150eac), {"D43", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xac818b61a660457b), {"D43", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x15f55f80229fc9fc), {"D43", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x10bb5f9522be04f), {"D44", "Semi-Slav Defense, Botvinnik Gambit"}},
    {Q_UINT64_C(0x6ac2cb153fb83581), {"D44", "Semi-Slav Defense, Botvinnik Gambit"}},
    {Q_UINT64_C(0x3556b4ea8d77b76f), {"D44", "Semi-Slav Defense, Botvinnik Gambit"}},
    {Q_UINT64_C(0xc09e7fc4272ad92), {"D44", "Semi-Slav Defense, Botvinnik Gambit"}},
    {Q_UINT64_C(0xb87a994f94c05dc0), {"D45", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x3a80900f66be8c68), {"D45", "Semi-Slav Defense, Anti-Meran"}},
    {Q_UINT64_C(0xd055dd92d813beed), {"D45", "Semi-Slav Defense, Anti-Meran"}},
    {Q_UINT64_C(0xdb941aa161369978), {"D45", "Semi-Slav Defense, Anti-Meran"}},
    {Q_UINT64_C(0xd8aa9a6b7bec4325), {"D45", "Semi-Slav Defense, Anti-Meran"}},
    {Q_UINT64_C(0x662accf5a6fd060), {"D45", "Semi-Slav Defense, Anti-Meran"}},
    {Q_UINT64_C(0xde9443b04de8727f), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xff2e4da42ce3fa92), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xcbd55c2f80249f3f), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xbf08caac298eadc4), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x24b4c7f5448340ec), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0xa640f5c6b61e52a2), {"D46", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x45284ee920e59f57), {"D47", "Semi-Slav Defense"}},
    {Q_UINT64_C(0x252c24fcbb48226a), {"D47", "Semi-Slav Defense, Meran Variation"}},
    {Q_UINT64_C(0x5bd45f5877fe1c2d), {"D47", "Semi-Slav, Meran Variation"}},
    {Q_UINT64_C(0x39b3e62fe28c4b6a), {"D48", "Semi-Slav Defense, Meran Variation"}},
    {Q_UINT64_C(0x5660da565e180bd5), {"D48", "Semi-Slav Defense, Meran Variation"}},
    {Q_UINT64_C(0xfa85fdcdb945771b), {"D49", "Semi-Slav Defense, Meran Variation"}},
    {Q_UINT64_C(0x9a138dd21f33bc2a), {"D50", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0xa9f0f68fff9b51e6), {"D51", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x9db4c312e219fac7), {"D51", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x4dc3b524c4e26bc4), {"D52", "QGD, Cambridge Springs Variation"}},
    {Q_UINT64_C(0x8edffe3dd59810bb), {"D53", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x55bb12cbbd7e7186), {"D53", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x56c19e75a78c252a), {"D54", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0xf23ebdce4c7810c9), {"D55", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x38770ad4966b3d5), {"D55", "Queen's Gambit Declined"}},
    {Q_UINT64_C(0x60c9f6a8dbac59dc), {"D55", "QGD, Tartakover Variation"}},
    {Q_UINT64_C(0x3a469e6b28127b47), {"D56", "QGD, Tartakover Variation"}},
    {Q_UINT64_C(0x4ed2f49a5bf6bd), {"D56", "QGD, Lasker Defense"}},
    {Q_UINT64_C(0xeec36cc8c45017df), {"D57", "QGD, Lasker Defense"}},
    {Q_UINT64_C(0x512fd77bd691ccdc), {"D58", "QGD, Tartakover Variation"}},
    {Q_UINT64_C(0xc0eb36fbcbbac59d), {"D59", "QGD, Tartakover Variation"}},
    {Q_UINT64_C(0xc1ddc693acd0fd05), {"D60", "QGD, Orthodox Defense"}},
    {Q_UINT64_C(0x70c4b48ebe06c161), {"D61", "QGD, Orthodox, Rubinstein Variation"}},
    {Q_UINT64_C(0xb8bc972bda6caaef), {"D61", "QGD, Orthodox, Rubinstein Variation"}},
    {Q_UINT64_C(0x9479672a12b6d77e), {"D61", "QGD, Orthodox, Rubinstein Variation"}},
    {Q_UINT64_C(0xe95647f0a25d9d70), {"D62", "QGD, Orthodox"}},
    {Q_UINT64_C(0xbe4609de2fc2a9db), {"D63", "QGD, Orthodox"}},
    {Q_UINT64_C(0xd52f40ced1411e40), {"D63", "QGD, Orthodox"}},
    {Q_UINT64_C(0x92ae51d9572290db), {"D63", "QGD, Orthodox"}},
    {Q_UINT64_C(0x9d25962c4ad44604), {"D64", "QGD, Orthodox"}},
    {Q_UINT64_C(0x41f8fc120f3aaac5), {"D65", "QGD, Orthodox"}},
    {Q_UINT64_C(0x793145936182b813), {"D66", "QGD, Orthodox"}},
    {Q_UINT64_C(0x269d9733c24d208e), {"D67", "QGD, Orthodox"}},
    {Q_UINT64_C(0x4b71fde35131775f), {"D67", "QGD, Orthodox"}},
    {Q_UINT64_C(0x8f46bbe076558abb), {"D68", "QGD, Orthodox"}},
    {Q_UINT64_C(0xb967933177d7a46b), {"D69", "QGD, Orthodox"}},
    {Q_UINT64_C(0xbd7741389222a904), {"D70", ""}},
    {Q_UINT64_C(0x14fbd86dee213f1a), {"D71", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x337116ffd40c34d8), {"D71", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x8073272645a02588), {"D71", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x794b4eae9e4efe32), {"D72", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xc07bc7e7e534a41f), {"D73", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x2fca5d5384ec764e), {"D74", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xc9bb8cfd7bab7d33), {"D75", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x651d25c9ca91d00), {"D76", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x562b121c4fda3ada), {"D76", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xe75797ee59f557fc), {"D76", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xbe0ebcd399c77f0f), {"D77", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x2c74513cee07acb4), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x168df284db3c0452), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xa0d23168348ac7d9), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x9e3a1f80ef8ddcf2), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x3cc3bf40b792f61b), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xed6d5db82b9dd9d7), {"D78", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x512e65b8eca3b513), {"D79", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0x1f650e46deec2ee), {"D79", "Gruenfeld Defense, Fianchetto Variation"}},
    {Q_UINT64_C(0xc38a7b8ecc1465fb), {"D80", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x5d02cb87c9ebe33), {"D80", "Gruenfeld Defense"}},
    {Q_UINT64_C(0xf973d836f92fcd1d), {"D81", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x18de9d0f55748f98), {"D82", "Gruenfeld Defense, 4. Bf4"}},
    {Q_UINT64_C(0xfe70fbc5f35257d), {"D83", "Gruenfeld Gambit"}},
    {Q_UINT64_C(0xa08f71815daca1db), {"D84", "Gruenfeld Gambit"}},
    {Q_UINT64_C(0x524e9a0ed13f6cba), {"D85", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0x32f2da2bc4f28d7c), {"D85", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0x8a2582f584573ae7), {"D86", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0x725855238662d58f), {"D87", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0xaabd362d16c96b02), {"D87", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0x276022607366d5f2), {"D88", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0xd07d17b687f107bf), {"D89", "Gruenfeld, Exchange Variation"}},
    {Q_UINT64_C(0x18ee9778a4f204c6), {"D90", "Gruenfeld Defense"}},
    {Q_UINT64_C(0xa852aace5fb5cf6c), {"D90", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x7af617819f8b3d17), {"D90", "Gruenfeld Defense, Flohr Variation"}},
    {Q_UINT64_C(0x6e08fdf8ef3f14a4), {"D91", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x73064c4fc6d5250f), {"D92", "Gruenfeld Defense"}},
    {Q_UINT64_C(0xd483e34a37d34440), {"D93", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x35798d5c4136dd58), {"D93", "Gruenfeld Defense"}},
    {Q_UINT64_C(0xf7e07cb852c5ab9f), {"D93", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x57fe87c7bea47c8), {"D94", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x970505930c2a9473), {"D94", "Slav Defense, Schlecter Variation"}},
    {Q_UINT64_C(0x905810e9a4a5ec2b), {"D94", "Slav Defense, Schlecter Variation"}},
    {Q_UINT64_C(0x352ea6739b8806c5), {"D95", "Gruenfeld Defense"}},
    {Q_UINT64_C(0x92ab09766a8e678a), {"D96", "Gruenfeld Defense, Classical Variation"}},
    {Q_UINT64_C(0xa11f4cadad02beaa), {"D97", "Gruenfeld Defense, Classical Variation"}},
    {Q_UINT64_C(0x3365a142dac26d11), {"D97", "Gruenfeld Defense, Boleslavsky System"}},
    {Q_UINT64_C(0xef9fe73d2bfd2216), {"D97", "Gruenfeld Defense, Prins System"}},
    {Q_UINT64_C(0xa3cc869fea87b040), {"D98", "Gruenfeld Defense, Smyslov System"}},
    {Q_UINT64_C(0x814ca588eccd530e), {"D99", "Gruenfeld Defense, Smyslov System"}},
    {Q_UINT64_C(0xe525b1311587325), {"E00", "Catalan"}},
    {Q_UINT64_C(0x3ec85a7ac061e57a), {"E01", "Catalan"}},
    {Q_UINT64_C(0x255f846c07366317), {"E01", "Catalan, Closed"}},
    {Q_UINT64_C(0x629744fc128bbabc), {"E01", "Catalan"}},
    {Q_UINT64_C(0xb58f47b9f8e76bef), {"E01", "Catalan"}},
    {Q_UINT64_C(0x9dbff602bf9a0423), {"E01", "Catalan"}},
    {Q_UINT64_C(0xd7a35fea8d422688), {"E01", "Catalan"}},
    {Q_UINT64_C(0xecce6348925f0b99), {"E02", "Catalan, Open Varation"}},
    {Q_UINT64_C(0x3e6ade075261f9e2), {"E02", "Catalan, Open Variation"}},
    {Q_UINT64_C(0xd6ed49acda2f7513), {"E02", "Catalan, Open Variation"}},
    {Q_UINT64_C(0xcac927bbeca72d91), {"E03", "Catalan, Open Variation"}},
    {Q_UINT64_C(0x37aa8fbefab96aa4), {"E04", "Catalan, Open Variation"}},
    {Q_UINT64_C(0x13245421712b8d9a), {"E04", "Catalan, Open Variation"}},
    {Q_UINT64_C(0x2366fc513012c635), {"E05", "Catalan, Open Variation"}},
    {Q_UINT64_C(0x57bb6ad299b8f4ce), {"E05", "Catalan, Open Variation"}},
    {Q_UINT64_C(0xe6a218cf8b6ec8aa), {"E05", "Catalan, Open Variation"}},
    {Q_UINT64_C(0x2a0429950aca49eb), {"E06", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0x4ad15ac9a5a3fd59), {"E06", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0x5f3076759743e4), {"E06", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xa2a690b62d88690d), {"E06", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xb65e28bd2b2ef7e1), {"E07", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0x7475aa039f8cb85), {"E08", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xf4c29d61f12db3fe), {"E08", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xe25d03ad53fdbc8e), {"E08", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xcc530f2dd4c5b00e), {"E09", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0x24f09e887676f2e9), {"E09", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xcc530f2dd4c5b00e), {"E09", "Catalan, Closed Variation"}},
    {Q_UINT64_C(0xf9d00ca49969ca20), {"E10", ""}},
    {Q_UINT64_C(0x203bb97c5f6ac6db), {"E10", "Catalan, Closed"}},
    {Q_UINT64_C(0xc5adf44c64a5ea3d), {"E10", "Blumenfeld Counter Gambit"}},
    {Q_UINT64_C(0x309165a2d29e0202), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x9f068ad623636596), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0xff6deeb9bfb8627b), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x991dfcf7ae9d941c), {"E11", " "}},
    {Q_UINT64_C(0xb090a6423cfcff18), {"E11", " "}},
    {Q_UINT64_C(0x203bb97c5f6ac6db), {"E11", " "}},
    {Q_UINT64_C(0xa4adb691b44df6bf), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x115d95ecf9a02ec6), {"E11", " "}},
    {Q_UINT64_C(0xb018a0b93d68622e), {"E11", " "}},
    {Q_UINT64_C(0xfdf08bceb64137c6), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x1e9561bc7718cd16), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x994d6c847544672f), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x75a6e6f2b71768b0), {"E11", "Bogo-Indian Defense"}},
    {Q_UINT64_C(0x92b945b467ea7dbb), {"E12", "Queen's Indian Defense, Petrosian"}},
    {Q_UINT64_C(0xd9be1592d75a50f9), {"E12", "Queen's Indian Defense, Petrosian"}},
    {Q_UINT64_C(0xf558aed3378d88c1), {"E12", "Queen's Indian Defense, Petrosian"}},
    {Q_UINT64_C(0xd6a52ba22c274313), {"E13", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x1edd0807484d289d), {"E13", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x3f94070643b5f51f), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x8f044f8231e75a84), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xafbbd66292bf950d), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x418f13eea9848444), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x420d44ba8bd3489a), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x56c137554178e40b), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xe09232f5c9f993a2), {"E14", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xbe5ffef5873da583), {"E15", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x714fac11cc1ca77d), {"E15", "Queen's Indian, Classical Variation"}},
    {Q_UINT64_C(0xbb3f950cb882d261), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x5d4e44a247c5d91c), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x3e55b71824569c47), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x2b1044686988b273), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xdaa7f0ff5cf6ceae), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x5bf3a6bf61147147), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xbd7e14a8d9699643), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xdde9137e02887dd7), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x853f825e8d8d1224), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x17456fb1fa4dc19f), {"E16", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xaff3e6e372297ef0), {"E17", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x5464fe41b0cbfc29), {"E17", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xdb2e7060db834c0b), {"E17", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x20b968c21961ced2), {"E18", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x1903fea3cb1987bf), {"E18", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x67f58112d9c77725), {"E18", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x1ab1245dab284328), {"E18", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x112d0f4c12e768cd), {"E18", "Queen's Indian Defense"}},
    {Q_UINT64_C(0xaba85640b9fe7f4c), {"E19", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x2910a3204606afc5), {"E19", "Queen's Indian Defense"}},
    {Q_UINT64_C(0x106291f6789ae1e6), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x3c842ab7984d39de), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x19117ef0fec539e), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x391bf0876329a876), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xb39fa58dab5ba75), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xaa7ccf0d1e7df69d), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x6b6c6699c2d466a), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xddd22a9ff4cb2757), {"E20", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xcb067d00107c80db), {"E21", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x3c19c7d57c7a40e3), {"E21", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x89e9e4a83197989a), {"E21", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x2a9b324e4da14900), {"E22", "Nimzo-Indian, Spielmann Variation"}},
    {Q_UINT64_C(0x18e419d2406bb012), {"E23", ""}},
    {Q_UINT64_C(0xb112b1e78132d2ec), {"E24", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x39d28678debf36e0), {"E24", "Nimzo-Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xb2923fa1e0a56cdb), {"E25", ""}},
    {Q_UINT64_C(0xfa4e22fb5a2a5135), {"E26", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xbbba5c50546b3b07), {"E27", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xd50bba61f4a1d476), {"E27", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x16971ee27034b3a3), {"E28", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xe06b60ca762dfae), {"E29", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xd638c6c0c8103a2e), {"E30", "Nimzo-Indian, Leningrad Variation"}},
    {Q_UINT64_C(0x7b83e1f9186385ed), {"E31", ""}},
    {Q_UINT64_C(0xa17be3eb6a4cdd82), {"E32", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x461df8532ee73266), {"E32", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xa963b097bdd7881), {"E33", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x5498f5eb8fef579a), {"E33", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x2411c1fff69893a4), {"E34", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x152f83f64ef34772), {"E35", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xef8df0d9bfd41de4), {"E36", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xe249c99f81c0d1bb), {"E37", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x470a3245950bd6ff), {"E38", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x3241fd22a34e6878), {"E39", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xf9ddcc04ea02e638), {"E39", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xe92511d4cba80945), {"E39", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0x873d74b1e10734e6), {"E39", "Nimzo-Indian, Classical Variation"}},
    {Q_UINT64_C(0xbd4fd3445cc56942), {"E40", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x5b3e02eaa382623f), {"E41", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xa5de7baa8b93054f), {"E41", "Nimzo-Indian, Huebner Variation"}},
    {Q_UINT64_C(0x4f9ae9258ba96fc1), {"E42", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xef6f91133c66561a), {"E42", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xd6269a54a246ded9), {"E43", "Nimzo-Indian Defense, Fischer Var."}},
    {Q_UINT64_C(0x7a73d8a9d79feb8a), {"E43", "Nimzo-Indian Defense, Fischer Var."}},
    {Q_UINT64_C(0x83e868cf2742efc), {"E43", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0x2218eeac298b86aa), {"E43", "Nimzo-Indian Defense"}},
    {Q_UINT64_C(0xc282719b8a6dd327), {"E44", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xd92237fc14cd1d9), {"E45", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xb7e73ef3899c80a9), {"E46", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xe2ea9f51b01c42da), {"E47", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x6780bd452cc80cfc), {"E48", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xdcdb519bc6f9a8ab), {"E48", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xc6f09d54d5603ff6), {"E49", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x6c83d205e17ae194), {"E50", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xe9e9f0117daeafb2), {"E51", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xd78d18a3baadda5a), {"E52", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x19682b13b40caed1), {"E52", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x5a95801dbb6966bc), {"E53", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x24e0fb29c79abdac), {"E53", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x1703807427325060), {"E53", "Nimzo-Indian, Averbakh Variation"}},
    {Q_UINT64_C(0xbf5cf670aa975084), {"E54", "Nimzo-Indian, Gligoric System"}},
    {Q_UINT64_C(0xc5315b9277621f9c), {"E54", "Nimzo-Indian, Gligoric System"}},
    {Q_UINT64_C(0x7d77f94fe5d56a1c), {"E54", "Nimzo-Indian, Gligoric System"}},
    {Q_UINT64_C(0x9dbb69df694945cb), {"E54", "Nimzo-Indian, Karpov Variation"}},
    {Q_UINT64_C(0x8cbf8d2d4a3fbd48), {"E55", "Nimzo-Indian, Gligoric, Bronstein var."}},
    {Q_UINT64_C(0xffc8f82a4259b77a), {"E55", "Nimzo-Indian, Gligoric, Bronstein var."}},
    {Q_UINT64_C(0x8f0d23cbd60b18af), {"E56", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xfc01117613de46d5), {"E56", "Nimzo-Indian, Rubinstein, Larsen Var."}},
    {Q_UINT64_C(0x8033a5818fbe1201), {"E57", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x2e7d03da2fa32ba5), {"E58", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0x28623cfb43e864a4), {"E58", "Nimzo-Indian, Rubinstein, Khasin Var."}},
    {Q_UINT64_C(0xb5c10e8342aec68d), {"E59", "Nimzo-Indian, Rubinstein Variation"}},
    {Q_UINT64_C(0xbd7741389222a904), {"E60", "King's Indian Defense"}},
    {Q_UINT64_C(0x6613adcefac4c839), {"E60", "King's Indian Defense"}},
    {Q_UINT64_C(0xd6af907801830393), {"E60", "King's Indian Defense"}},
    {Q_UINT64_C(0xfa492b39e154dbab), {"E60", "King's Indian Defense"}},
    {Q_UINT64_C(0x46e0599a50c02bdd), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xbc0ab8d898485c3), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xfcc483ec8f5882c2), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xd2d8498da19313ef), {"E61", "King's Indian, Smyslov System"}},
    {Q_UINT64_C(0xc0873247e13ae3cb), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0x8d11322e7d61d517), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xbb7c963b72c34e69), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xfb1f6e9f934ee906), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0xe7270b672755a11b), {"E61", "King's Indian Defense"}},
    {Q_UINT64_C(0x3864a5faf1ce101f), {"E62", "King's Indian, Fianchetto Variation"}},
    {Q_UINT64_C(0x154686df1ebaf236), {"E62", "King's Indian, Fianchetto Variation"}},
    {Q_UINT64_C(0x6b33fdeb62492926), {"E62", "King's Indian, Fianchetto Variation"}},
    {Q_UINT64_C(0xd39241a5b2bf41d6), {"E62", "King's Indian, Fianchetto Variation"}},
    {Q_UINT64_C(0x8321ecc5c486eb3), {"E62", "King's Indian, Fianchetto Variation"}},
    {Q_UINT64_C(0x80533915e338a742), {"E63", "King's Indian, Panno Variation"}},
    {Q_UINT64_C(0x614dba9e963d2af0), {"E64", "King's Indian, Yugoslav Variation"}},
    {Q_UINT64_C(0x1f38c1aaeacef1e0), {"E65", "King's Indian, Yugoslav Variation"}},
    {Q_UINT64_C(0xedd5a1b6279c2be), {"E66", "King's Indian, Yugoslav Variation"}},
    {Q_UINT64_C(0xb4df106d89d2cc41), {"E67", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x185d9aaa1201ef8d), {"E67", "King's Indian, Hungarian System"}},
    {Q_UINT64_C(0x40d2631366c4ef67), {"E67", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xf1cb110e7412d303), {"E67", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xedff21a1429b67c3), {"E67", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xa821beada4672695), {"E67", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x84d56e5b8d44526d), {"E68", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x783f3984580ed456), {"E68", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x56229fa86a64aab1), {"E68", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x16af83b4fa8481d6), {"E68", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xfe5c5e0a38274824), {"E69", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x272fcb51915569ab), {"E69", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xdbc59c8e441fef90), {"E69", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0xa8cd2762c525f41e), {"E69", "King's Indian, Fianchetto"}},
    {Q_UINT64_C(0x325b696440075d7d), {"E70", "King's Indian Defense"}},
    {Q_UINT64_C(0xbe1ff05927f1410), {"E70", "King's Indian Defense"}},
    {Q_UINT64_C(0xe31222bb50dcdde2), {"E71", "King's Indian Defense"}},
    {Q_UINT64_C(0x2707444472a8cc28), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0xc17695ea8defc755), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x2dafa9f3a7f125c3), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x985f8a8eea1cfdba), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0xa2567619ddc2e01), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x1e818caeb5f723ff), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x33b2526cfb8d58b9), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x2716b9a3d3a65547), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0xf5e148503486ad9b), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x7e2e5b20155bf6c7), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x122782c479f9058c), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x8f685105285c73e3), {"E72", "King's Indian Defense"}},
    {Q_UINT64_C(0x78617cfc935a5eb3), {"E73", "King's Indian Defense"}},
    {Q_UINT64_C(0xb493c67df6896c90), {"E73", "King's Indian, Averbach Variation"}},
    {Q_UINT64_C(0x52e217d309ce67ed), {"E74", "King's Indian, Averbach Variation"}},
    {Q_UINT64_C(0x9e9da6b188ca61e2), {"E75", "King's Indian, Averbach Variation"}},
    {Q_UINT64_C(0xf6f782088bee2999), {"E76", "King's Indian, Four Pawns Attack"}},
    {Q_UINT64_C(0x8fdfec465f928ad1), {"E77", "King's Indian, Four Pawns Attack"}},
    {Q_UINT64_C(0xb2cad11ec833e091), {"E78", "King's Indian, Four Pawns Attack"}},
    {Q_UINT64_C(0x52a55578345da5f4), {"E79", "King's Indian, Four Pawns Attack"}},
    {Q_UINT64_C(0x6550193432b5fb61), {"E80", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0x6ff8f483e7ec128a), {"E81", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0x7799067f9855be9f), {"E82", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xb71d978d7747ac07), {"E83", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xc1f569de2fcb31ef), {"E84", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0x96884725f533f132), {"E85", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0x10564105aad82f77), {"E86", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0x2c8004766c15676f), {"E87", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xbefae9991bd5b4d4), {"E88", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xd92ead1f9394b490), {"E89", "King's Indian, Saemisch Variation"}},
    {Q_UINT64_C(0xd08513f3fa99752d), {"E90", "King's Indian Defense"}},
    {Q_UINT64_C(0xda2dfe442fc09cc6), {"E90", "King's Indian Defense"}},
    {Q_UINT64_C(0xa52e3f2a680353c7), {"E90", "King's Indian Defense"}},
    {Q_UINT64_C(0xa9ad7dbd2ee5d665), {"E91", "King's Indian Defense"}},
    {Q_UINT64_C(0x23d575f7bd002e53), {"E92", "King's Indian Defense"}},
    {Q_UINT64_C(0x50ddce1b3c3a35dd), {"E92", "King's Indian, Gligoric Variation"}},
    {Q_UINT64_C(0x39767dbe2ea9ae50), {"E92", "King's Indian, Petrosian Variation"}},
    {Q_UINT64_C(0x807a5f0dafe7e1fe), {"E92", "King's Indian, Petrosian Variation"}},
    {Q_UINT64_C(0x99dd36a42426b80e), {"E92", "King's Indian, Petrosian Variation"}},
    {Q_UINT64_C(0xaa3e4df9c48e55c2), {"E93", "King's Indian, Petrosian Variation"}},
    {Q_UINT64_C(0x5da00ec3c1f3f543), {"E94", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0xfd0b45d9cb7ce31d), {"E94", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0x1320a553470c69ff), {"E94", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0xeaf77b15a6f3b39b), {"E95", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0x8d17a0c2b38a2613), {"E96", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0xf64dd621d0625040), {"E97", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0x48adbd2303c24d0c), {"E97", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0xf68ce0a9d8446c52), {"E98", "King's Indian, Classical Variation"}},
    {Q_UINT64_C(0xdc58a8ad9af3ddb5), {"E99", "King's Indian, Mar del Plata Variation"}},
    {Q_UINT64_C(0x53c3142c49aa22d6), {"E99", "King's Indian, Mar del Plata Variation"}},
    {Q_UINT64_C(0xaf5013411bc9c63b), {"E99", "King's Indian, Mar del Plata Variation"}},
    {Q_UINT64_C(0x2a8cf51776aece8f), {"E99", "King's Indian, Mar del Plata Variation"}}
};


EcoCode::EcoCode()
{
}

EcoInfo EcoCode::classify(Board &b) {
    EcoInfo ret;
    quint64 b_zobrist = b.zobrist();
    if(ECOINFOS.contains(b_zobrist)) {
        EcoInfo e = ECOINFOS[b_zobrist];
        ret.code = e.code;
        ret.info = e.info;
    }
    return ret;
}

}
