// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "modelrunner.h"

#include <iostream>

#include <QDebug>

GLOBAL_USE_NAMESPACE

ModelRunner::ModelRunner(QObject *parent) : QThread(parent)
{

}

ModelRunner::~ModelRunner()
{
    terminate();
    wait(1000);
}

void ModelRunner::postTask(QSharedPointer<ModelTask> task)
{
    if (task.isNull())
        return;

    {
        QMutexLocker lock(&taskMtx);
        taskQueue.enqueue(task);
    }

    taskConditon.notify_one();
}

bool ModelRunner::recvTask(QSharedPointer<ModelTask> task)
{
    while (running) {
        QMutexLocker lock(&resultMtx);
        for (int i = 0; i < (int) resultList.size(); i++) {
            if (resultList[i].data() == task.data()) {
                resultList.erase(resultList.begin() + i);
                return true;
            }
        }
        resultConditon.wait(&resultMtx);
    }

    return false;
}

void ModelRunner::terminate()
{
    {
        QMutexLocker lock(&taskMtx);
        if (!running)
            return;

        running = false;
    }

    taskConditon.notify_all();
}

void ModelRunner::run()
{
    running = true;
    std::cerr << "start model task loop" << std::endl;
    while (running) {
        QMutexLocker lock(&taskMtx);
        if (taskQueue.isEmpty()) {
            taskConditon.wait(&taskMtx);
            continue;
        }

        auto task = taskQueue.dequeue();
        lock.unlock();

        {
            task->doTask();
            QMutexLocker lock(&resultMtx);
            resultList.append(task);
            lock.unlock();
            resultConditon.wakeAll();
        }
    }

    std::cerr << "ending model task loop" << std::endl;
    resultConditon.wakeAll();
}

ModelTask::ModelTask(ModelRunner *r) : runner(r)
{

}

ModelTask::~ModelTask()
{

}
