package deepinid

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"

	"errors"

	"github.com/godbus/dbus"
)

var (
	errTokenNotFound = errors.New("Not found this token")
)

func launchLoginApp() error {
	conn, err := dbus.SessionBus()
	if err != nil {
		return err
	}
	log.Info("Launch login app")

	obj := conn.Object("com.deepin.deepinid.Client", "/com/deepin/deepinid/Client")
	err = obj.Call("com.deepin.deepinid.Client.Show", 0).Err
	if err != nil {
		log.Warning("Failed to launch login app:", err)
	}
	return err
}

type tokenRecvInfo struct {
	ID          int64     `json:"id"`
	UserID      int64     `json:"user_id"`
	Token       string    `json:"token"`
	AccessToken string    `json:"access_token"`
	MachineName string    `json:"machine_name"`
	Username    string    `json:"username"`
	Region      string    `json:"region"`
	Nickname    string    `json:"nickname"`
	MachineID   string    `json:"machine_id"`
	Expiry      time.Time `json:"expiry"`
}

func refreshTokenInfo(token, accessToken string) (*tokenRecvInfo, error) {
	// log.Infof("refresh with: %q %q", token, accessToken)
	return fetchToken(storage.SyncServerAPI()+"/sessions/refresh", token, accessToken)
}

func getTokenInfo(token, accessToken string) (*tokenRecvInfo, error) {
	// log.Infof("gettoken with: %q %q", token, accessToken)
	return fetchToken(storage.SyncServerAPI()+"/sessions/info", token, accessToken)
}

func fetchToken(api, token, accessToken string) (*tokenRecvInfo, error) {
	req, err := http.NewRequest(http.MethodGet, api, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Token", token)
	req.Header.Set("Access-Token", accessToken)
	var cli = new(http.Client)
	resp, err := cli.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Errorf("resp body: %v", string(data))
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		log.Errorf("resp body: %v %v", string(data), req.Header)
		if resp.StatusCode == http.StatusNotFound {
			// reset token info
			return nil, errTokenNotFound
		}
		return nil, fmt.Errorf("failed to get token info")
	}

	// log.Infof("get data: %v", string(data))
	var tokenInfo tokenRecvInfo
	err = json.Unmarshal(data, &tokenInfo)
	if err != nil {
		return nil, err
	}
	return &tokenInfo, nil
}

func doLogout(accessToken string) error {
	req, err := http.NewRequest(http.MethodDelete,
		storage.DeepinIDServerAPI()+"/user/session", nil)
	if err != nil {
		return err
	}
	req.Header.Set("Access-Token", accessToken)
	var cli = new(http.Client)
	resp, err := cli.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}
	if resp.StatusCode != http.StatusOK {
		return fmt.Errorf("%s", string(data))
	}
	return nil
}
