/*
 * Copyright (C) 2019 ~ 2020 Uniontech Software Technology Co., Ltd.
 *
 * Author:     yelei <yelei@uniontech.com>
 * Maintainer: yelei <yelei@uniontech.com>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "phonefilethread.h"
#include "utils.h"

#include <QDesktopServices>
#include <QFile>
#include <QDir>
#include <QMimeDatabase>
#include <QImageReader>
#include <QApplication>
#include <QDirIterator>
#include <QDebug>
#include <QtConcurrent>
#include <QMutex>

#include <iostream>
#include <fstream>
#include <stdio.h>

extern "C" {
#include "libavcodec/avcodec.h"
#include "libavformat/avformat.h"
#include "libavfilter/avfilter.h"
#include "libswscale/swscale.h"
#include "libavutil/frame.h"
#include "libavutil/error.h"
#include "libavutil/imgutils.h"
}

#include "Ios/iphonemountcontrol.h"
#include "GlobalDefine.h"
#include "TrObject.h"

QVector<PhoneFileInfo> PhoneFileThread::m_mapDirFileInfo_photo;
QVector<PhoneFileInfo> PhoneFileThread::m_mapDirFileInfo_video;
QMutex PhoneFileThread::m_mutex_photo;
QMutex PhoneFileThread::m_mutex_video;
extern QMutex g_enterDirMutex; //全局加载文件时读取目录信息用

PhoneFileThread::PhoneFileThread(QObject *parent)
    : QThread(parent)
{
    //    qRegisterMetaType<PhoneFileInfo>("PhoneFileInfo");
    m_isCanRun = false;
    m_bLocked = false;
}

PhoneFileThread::~PhoneFileThread()
{
    stopImmediately();
    quitImmediately();
    quit();
    wait();
    qDebug() << __LINE__ << __FUNCTION__;
}

bool PhoneFileThread::copyFile(const QStringList &listFileNames, const QString &strDesPath, const RunMode &mode,
                               const PhoneFileType &type, const QString &strSysVersion, const QString &strRootPath,
                               const QString strPhoneID)
{
    if (strDesPath.isEmpty())
        return false;

    this->m_mode = mode;
    m_listFileNames = listFileNames;
    m_strDesPath = strDesPath;
    if (type == IMAGE) {
        m_listFilters << getImageFilters();
    } else if (type == VIDEO) {
        m_listFilters << getVideoFilters();
    }
    m_strSysVersion = strSysVersion;
    m_strRootPath = strRootPath;
    m_strPhoneID = strPhoneID;
    return true;
}

bool PhoneFileThread::delFile(QStringList &listFileNames, PhoneFileType type)
{
    if (listFileNames.isEmpty())
        return false;

    this->m_mode = DelFile;
    m_listFileNames = listFileNames;
    if (type == IMAGE) {
        m_listFilters << getImageFilters();
    } else if (type == VIDEO) {
        m_listFilters << getVideoFilters();
    }
    return true;
}

bool PhoneFileThread::readPhoto(const QString &strPath, const QString &strPhoneID, const QSize &sMaxSize,
                                DEVICE_TYPE type, bool bOnlyDir, bool bKeepAspectRatio)
{
    m_listFileNames.clear();
    this->m_mode = ReadPhoto;
    this->m_sMaxSize = sMaxSize;
    this->m_bKeepAspectRatio = bKeepAspectRatio;
    this->m_strDesPath = strPath;
    this->m_strPhoneID = strPhoneID;
    this->m_bIsDir = bOnlyDir;
    this->m_phoneType = type;

    return true;
}

bool PhoneFileThread::readVideo(const QString &strPath, const QString &strPhoneID, const QSize &sMaxSize,
                                DEVICE_TYPE type, bool bOnlyDir, bool bKeepAspectRatio)
{
    m_listFileNames.clear();
    this->m_mode = ReadVideo;
    this->m_sMaxSize = sMaxSize;
    this->m_bKeepAspectRatio = bKeepAspectRatio;
    this->m_strDesPath = strPath;
    this->m_strPhoneID = strPhoneID;
    this->m_bIsDir = bOnlyDir;
    this->m_phoneType = type;

    return true;
}

void PhoneFileThread::quitImmediately()
{
    m_isCanRun = false;
    m_bLocked = false;
    m_mutex_read.unlock();
}

void PhoneFileThread::stopImmediately()
{
    if (this->isRunning())
        emit error(PEERROR_END, ""); // 界面停止spinner
}

void PhoneFileThread::clear()
{
    //    m_mapDirToFileCount.clear();
    //    m_mapDirToFileSize.clear();
    m_mapFileToDir.clear();
    m_listFileNames.clear();
}

void PhoneFileThread::clearCache(const QString &strPhoneID)
{
    clearCache(PhoneFileThread::ReadPhoto, strPhoneID);
    clearCache(PhoneFileThread::ReadVideo, strPhoneID);
}

void PhoneFileThread::clearCache(const PhoneFileThread::RunMode &mode, const QString &strPhoneID)
{
    if (mode == ReadPhoto) {
        QVector<PhoneFileInfo>::iterator itr = m_mapDirFileInfo_photo.begin();
        while (itr != m_mapDirFileInfo_photo.end()) {
            PhoneFileInfo &item = *itr;
            if (item.dirPath.contains(strPhoneID)) {
                itr = m_mapDirFileInfo_photo.erase(itr);
            } else {
                itr++;
            }
        }
    } else if (mode == ReadVideo) {
        QVector<PhoneFileInfo>::iterator itr = m_mapDirFileInfo_video.begin();
        while (itr != m_mapDirFileInfo_video.end()) {
            PhoneFileInfo &item = *itr;
            if (item.dirPath.contains(strPhoneID)) {
                itr = m_mapDirFileInfo_video.erase(itr);
            } else {
                itr++;
            }
        }
    }
}

bool PhoneFileThread::lock()
{
    if (m_bLocked)
        return true;

    m_bLocked = true;
    return true;
}

bool PhoneFileThread::unlock()
{
    if (!m_bLocked)
        return true;

    m_bLocked = false;
    m_mutex_read.unlock();
    return true;
}

void PhoneFileThread::run()
{
    m_isCanRun = true;
    switch (m_mode) {
    case ExportFile:
    case ImportFile:
        _copyFile();
        break;
    case DelFile:
        _delFile();
        break;
    case ReadPhoto:
        _readPhoto();
        break;
    case ReadVideo:
        _readVideo_ffmpeg();
        //        _readVideoIcon_ffmpegthumbnailer();
        break;
    }
}

PhoneFileInfo PhoneFileThread::onVideoFindPixmap(QIcon &icon, QString path, int nTime)
{
    PhoneFileInfo info;
    info.type = VIDEO;
    info.phoneID = m_strPhoneID;
    //    info.icon = QPixmap::fromImage(icon).scaled(m_sMaxSize, Qt::IgnoreAspectRatio);
    info.damaged = icon.isNull();
    //    info.originalIcon = icon.scaled(300, 300);
    info.scaleIcon = resizePicture(icon.pixmap(m_sMaxSize));
    info.time = nTime;
    info.name = path.right(path.length() - path.lastIndexOf('/') - 1);
    info.path = path;
    //    info.scaleIcon = info.originalIcon.scaled(m_sMaxSize);
    if (m_bIsDir) {
        info.bIsDir = true;
        info.dirPath = m_mapFileToDir.value(path);
        info.fileCount = m_mapDirToFileCount.value(info.dirPath);
        info.size = m_mapDirToFileSize.value(info.dirPath);
        m_mapDirFileInfo_video.append(info);
    } else {
        info.bIsDir = false;
    }
    if (m_isCanRun) {
        emit sigFindPicture(info);
    }
    if (info.damaged) {
        qDebug() << __LINE__ << "read video error:time:" << nTime;
    }
    return info;
}

bool PhoneFileThread::_copyFile()
{
    int nIndex = 0;
    int nCount = m_listFileNames.count();
    QDir dir;
    dir.mkpath(m_strDesPath);
    if (!m_strDesPath.endsWith(QDir::separator())) {
        m_strDesPath += QDir::separator();
    }
    int nTotal = 0; //总数
    int nError = 0; //失败数
    foreach (QString strItemFile, m_listFileNames) {
        if (!m_isCanRun) {
            goto result;
        }
        QFileInfo fileInfo;
        fileInfo.setFile(strItemFile);
        // send progress
        emit sigProgress(m_mode, nIndex++, nCount, fileInfo.fileName());

        if (fileInfo.isDir()) {
            QString strNewPath = m_strDesPath + fileInfo.fileName() + QDir::separator();
            dir.mkpath(strNewPath);
            QDirIterator dir_iterator(strItemFile, m_listFilters, QDir::Files | QDir::NoSymLinks,
                                      QDirIterator::Subdirectories);
            while (dir_iterator.hasNext()) {
                if (!m_isCanRun) {
                    goto result;
                }
                dir_iterator.next();
                QFileInfo file_info = dir_iterator.fileInfo();
                if (file_info.isFile()) {
                    //                    bool bRet = QFile::copy(file_info.absoluteFilePath(), strNewPath +
                    //                    file_info.fileName());
                    bool bRet = _copyFile_one_auto(file_info, strNewPath);
                    ++nTotal;
                    if (!bRet) {
                        ++nError;
                        //                        emit error(-1,
                        //                        TrObject::getInstance()->getFileOperationText(copy_failed) +
                        //                        file_info.absoluteFilePath());
                    }
                }
            }
        } else if (fileInfo.isFile()) {
            bool bRet = _copyFile_one_auto(strItemFile, m_strDesPath);
            ++nTotal;
            if (!bRet) {
                ++nError;
                //                emit error(-1, TrObject::getInstance()->getFileOperationText(copy_failed) +
                //                strItemFile);
            }
        }
    }
    // send progress
    if (nCount != 0) {
        emit sigProgress(m_mode, nIndex, nCount, "");
    }
result:
    emit sigResult(m_mode, nTotal, nTotal - nError, nError, !m_isCanRun);
    return true;
}

bool PhoneFileThread::_copyFile_one_auto(const QFileInfo &source, const QString &desPath)
{
    //    if (m_mode == PhoneFileThread::ImportFile) {
    //        导入时，可能会替换，删除缓存
    //        PhoneFileThread::m_imgCacheHash.remove(pszFileCoper);
    //    }

    // 软链接不拷贝
    if (source.isSymLink()) {
        emit error(-1, TrObject::getInstance()->getFileOperationText(copy_failed) + source.fileName());
        return false;
    }

    bool bRet = false;
    if (source.size() / 1024 / 1024 /*MB*/ < 120) {
        if (m_strSysVersion.contains("Android") && !Utils::isLowVersionAndroid(m_strSysVersion)) {
            // 考虑使用adb
            bRet = _copyFile_one_adb(source.absoluteFilePath().toLocal8Bit().data(),
                                     (desPath + source.fileName()).toLocal8Bit().data());
        } else {
            bRet = _copyFile_one_f(source.absoluteFilePath().toLocal8Bit().data(),
                                   (desPath + source.fileName()).toLocal8Bit().data());
        }
    } else {
        bRet = _copyFile_one_iof(source.absoluteFilePath().toLocal8Bit().data(),
                                 (desPath + source.fileName()).toLocal8Bit().data());
    }
    return bRet;
}

bool PhoneFileThread::_copyFile_one_qfile(const QString &source, const QString &destination)
{
    QFile srcFile(source);
    if (!srcFile.open(QFile::ReadOnly)) {
        qDebug() << __LINE__ << source;
        qDebug() << __LINE__ << srcFile.errorString();
        return false;
    }

    QString strDesTemp = destination + "_tmp";
    QFile desFile(strDesTemp);
    if (!desFile.open(QIODevice::ReadWrite | QIODevice::Truncate)) {
        qDebug() << __LINE__ << destination;
        qDebug() << __LINE__ << desFile.errorString();
        return false;
    }

    const qint64 maxSize = 4096;
    qint64 size = -1;

    QScopedArrayPointer<char> buffer(new char[maxSize]);

    bool bRes = true;
    do {
        if (!m_isCanRun) {
            bRes = false;
            qDebug() << __LINE__ << __FUNCTION__;
            break;
        }

        if ((size = srcFile.read(buffer.data(), maxSize)) < 0) {
            bRes = false;
            qDebug() << __LINE__ << __FUNCTION__;
            break;
        }

        if (desFile.write(buffer.data(), size) < 0) {
            bRes = false;
            qDebug() << __LINE__ << __FUNCTION__;
            break;
        }
    } while (size > 0);

    if (bRes) {
        QFile::remove(destination);
        QFile::rename(strDesTemp, destination);
    } else {
        if (QThreadPool::globalInstance()->activeThreadCount() >= QThreadPool::globalInstance()->maxThreadCount()) {
            // 线程池满了
            qDebug() << __LINE__ << QThreadPool::globalInstance()->activeThreadCount()
                     << QThreadPool::globalInstance()->maxThreadCount();
            _delFile_one(strDesTemp);
        } else {
            QtConcurrent::run(&PhoneFileThread::_delFile_one, strDesTemp);
        }
    }

    return bRes;
}

bool PhoneFileThread::_copyFile_one_iof(const QString &source, const QString &destination)
{
    using namespace std;
    QString strDesTemp = destination + "_tmp";
    ifstream in(source.toLocal8Bit(), ios::binary);
    ofstream out(strDesTemp.toLocal8Bit(), ios::binary | ios::trunc);
    if (!in.is_open()) {
        qDebug() << "error open file " << source << endl;
        return false;
    }
    if (!out.is_open()) {
        qDebug() << "error open file " << destination << endl;
        return false;
    }

    bool bRes = true;
    const qint64 maxSize = 2048;
    char buf[maxSize];
    long long totalBytes = 0;
    while (in) {
        if (!m_isCanRun) {
            bRes = false;
            break;
        }

        // read从in流中读取2048字节，放入buf数组中，同时文件指针向后移动2048字节
        //若不足2048字节遇到文件结尾，则以实际提取字节读取。
        in.read(buf, maxSize);
        // gcount()用来提取读取的字节数，write将buf中的内容写入out流。
        out.write(buf, in.gcount());
        out.flush();
        totalBytes += in.gcount();
    }
    in.close();
    out.close();

    if (bRes) {
        _delFile_one(destination);
        QFile::rename(strDesTemp, destination);
    } else {
        //        msleep(100);
        if (QThreadPool::globalInstance()->activeThreadCount() >= QThreadPool::globalInstance()->maxThreadCount()) {
            // 线程池满了
            qDebug() << __LINE__ << QThreadPool::globalInstance()->activeThreadCount()
                     << QThreadPool::globalInstance()->maxThreadCount();
            _delFile_one(strDesTemp);
        } else {
            qDebug() << __LINE__;
            QtConcurrent::run(&PhoneFileThread::_delFile_one, strDesTemp);
        }
    }
    return bRes;
}

bool PhoneFileThread::_copyFile_one_f(const QString &source, const QString &destination)
{
    using namespace std;

    fstream fsCopee(source.toLocal8Bit(), ios::binary | ios::in);
    fstream fsCoper(destination.toLocal8Bit(), ios::binary | ios::out);
    fsCoper << fsCopee.rdbuf();
    return true;
}

bool PhoneFileThread::_copyFile_one_adb(const QString &source, const QString &destination)
{
    qDebug() << __LINE__ << __FUNCTION__;
    if (m_mode == PhoneFileThread::ImportFile) {
        QString strAdbPath = Utils::mountPathToAdbPath(m_strRootPath, destination);
        if (Utils::adbPush(m_strPhoneID, source, strAdbPath) != 0)
            return false;
    } else if (m_mode == PhoneFileThread::ExportFile) {
        QString strAdbPath = Utils::mountPathToAdbPath(m_strRootPath, source);
        if (Utils::adbPull(m_strPhoneID, strAdbPath, destination) != 0)
            return false;
    }
    return true;
}

bool PhoneFileThread::_delFile()
{
    int nIndex = 0;
    int nCount = m_listFileNames.count();
    int nTotal = 0; //总数
    int nError = 0; //失败数
    foreach (QString strItemFile, m_listFileNames) {
        if (!m_isCanRun) {
            goto result;
        }
        QFileInfo fileInfo;
        fileInfo.setFile(strItemFile);
        // send progress
        emit sigProgress(m_mode, nIndex++, nCount, fileInfo.fileName());

        if (fileInfo.isDir()) {
            QDirIterator dir_iterator(strItemFile, m_listFilters, QDir::Files | QDir::NoSymLinks,
                                      QDirIterator::Subdirectories);
            while (dir_iterator.hasNext()) {
                if (!m_isCanRun) {
                    goto result;
                }
                dir_iterator.next();
                QFileInfo file_info = dir_iterator.fileInfo();
                if (file_info.isFile()) {
                    ++nTotal;
                    bool bRet = _delFile_one(file_info.absoluteFilePath().toLocal8Bit().data());
                    if (!bRet) {
                        ++nError;
                        emit error(PFERROR_DELETE_FILE,
                                   TrObject::getInstance()->getFileOperationText(Deletion_failed) + file_info.fileName()
                                       + ". \n" + TrObject::getInstance()->getFileOperationText(delete_failed));
                    }
                }
            }
        } else if (fileInfo.isFile()) {
            ++nTotal;
            bool bRet = _delFile_one(strItemFile.toLocal8Bit().data());
            if (!bRet) {
                ++nError;
                emit error(PFERROR_DELETE_FILE, TrObject::getInstance()->getFileOperationText(Deletion_failed)
                                                    + strItemFile.section('/', -1) + ". \n"
                                                    + TrObject::getInstance()->getFileOperationText(delete_failed));
            }
        }
    }
    // send progress
    if (nCount != 0) {
        emit sigProgress(m_mode, nIndex, nCount, "");
    }
result:
    emit sigResult(m_mode, nTotal, nTotal - nError, nError, !m_isCanRun);
    return true;
}

bool PhoneFileThread::_delFile_one(const QString &strFile)
{
    int nRet = remove(strFile.toLocal8Bit().data());
    qDebug() << __LINE__ << __FUNCTION__ << nRet << strFile << errno;
    if (nRet != 0) {
        return false;
    }
    return true;
}

bool PhoneFileThread::_readPhoto()
{
    QMutexLocker locker(&m_mutex_photo);
    if (processCache())
        return true;
    if (!autoFilterSet()) {
        return false;
    }

    if (m_listFileNames.isEmpty()) {
        emit error(PEERROR_NO_FILE, "");
        return false;
    }
    qDebug() << __LINE__ << __FUNCTION__ << m_listFileNames.count();
    QStringList::iterator it = m_listFileNames.begin();
    for (; it != m_listFileNames.end(); ++it) {
        if (m_bLocked) {
            m_mutex_read.lock();
            m_mutex_read.lock();
        } else {
            m_mutex_read.unlock();
        }

        QString strItemPath = *it;

        //如果是目录获取其中一张
        if (m_bIsDir) {
            QString strDirItemPath;
            if (!setDir(strItemPath, strDirItemPath))
                continue;
            strItemPath = strDirItemPath;
        }

        if (m_isCanRun) {
            _readPhoto_one(strItemPath);
        } else {
            if (m_bIsDir) {
                clearCache(m_mode, m_strPhoneID);
            }
            return false;
        }
    }

    this->setProperty("filelist", m_listFileNames);
    emit error(PEERROR_END, ""); // 界面停止spinner

    return true;
}

bool PhoneFileThread::_readPhoto_one(const QString &path)
{
    QPixmap pixmap;
    PhoneFileInfo info;

    //    if (m_bIsDir) {
    QImageReader reader;
    reader.setFileName(path);
    reader.setFormat(Utils::DetectImageFormat(path).toLatin1());
    reader.setAutoTransform(true);
    QSize size = reader.size();
    //        QSize size = reader.size().scaled(m_sMaxSize, Qt::IgnoreAspectRatio);
    if (reader.canRead()) {
        const qreal ratio = qApp->devicePixelRatio();
        reader.setScaledSize(size * ratio);
        pixmap = QPixmap::fromImage(reader.read());
        pixmap.setDevicePixelRatio(ratio);
    } else {
        pixmap.load(path);
    }
    info.damaged = pixmap.isNull();
    //    } else {
    //        info.damaged = false;
    //    }

    info.type = IMAGE;
    info.phoneID = m_strPhoneID;
    info.scaleIcon = resizePicture(pixmap);
    info.time = 0;
    info.name = path.right(path.length() - path.lastIndexOf('/') - 1);
    info.path = path;
    //    info.scaleIcon = info.originalIcon.scaled(m_sMaxSize);
    if (m_bIsDir) {
        info.bIsDir = true;
        info.dirPath = m_mapFileToDir.value(path);
        info.fileCount = m_mapDirToFileCount.value(info.dirPath);
        info.size = m_mapDirToFileSize.value(info.dirPath);
        m_mapDirFileInfo_photo.append(info);
    } else {
        info.bIsDir = false;
    }
    if (m_isCanRun) {
        emit sigFindPicture(info);
    }
    return true;
}

bool PhoneFileThread::_readVideo_ffmpeg()
{
    QMutexLocker locker(&m_mutex_video);
    if (processCache())
        return true;
    if (!autoFilterSet()) {
        return false;
    }

    if (m_listFileNames.isEmpty()) {
        emit error(PEERROR_NO_FILE, "");
        return false;
    }
    qDebug() << __LINE__ << __FUNCTION__ << m_listFileNames.count();
    QStringList::iterator it = m_listFileNames.begin();
    for (; it != m_listFileNames.end(); ++it) {
        if (m_bLocked) {
            m_mutex_read.lock();
            m_mutex_read.lock();
        } else {
            m_mutex_read.unlock();
        }

        QString strItemPath = *it;

        //如果是目录获取其中一张
        if (m_bIsDir) {
            QString strDirItemPath;
            if (!setDir(strItemPath, strDirItemPath))
                continue;
            strItemPath = strDirItemPath;
        }

        QIcon pixmap;
        int nTime;
        bool bRet = _readVideo_ffmpeg_one(strItemPath.toLocal8Bit(), pixmap, nTime);
        if (!bRet) {
            //            emit error(-1, tr("Read video icon error!"));
        }

        if (m_isCanRun) {
            onVideoFindPixmap(pixmap, strItemPath, nTime);
        } else {
            if (m_bIsDir)
                clearCache(m_mode, m_strPhoneID);
            return false;
        }
    }

    this->setProperty("filelist", m_listFileNames);
    emit error(PEERROR_END, ""); // 界面停止spinner

    return true;
}

bool PhoneFileThread::_readVideo_ffmpeg_one(const char *input, QIcon &pixmap, int &nTime)
{
    // 1.注册所有组件
    //    av_register_all();        wzx 不用也可以
    //封装格式上下文，统领全局的结构体，保存了视频文件封装格式的相关信息
    AVFormatContext *pFormatCtx = nullptr; // avformat_alloc_context();
    //        AVFormatContext *pFormatCtx = NULL;
    // 2.打开输入视频文件
    //    printf("fileName:%s\n", input);
    int nRet = avformat_open_input(&pFormatCtx, input, nullptr, nullptr);
    if (nRet != 0) {
        qDebug("%s", "无法打开输入视频文件");
        char buf[1024] = "";
        av_strerror(nRet, buf, 1024);
        printf("Couldn't open file %s: %d(%s)\n", input, nRet, buf);
        return false;
    }
    // 3.获取视频文件信息
    if (avformat_find_stream_info(pFormatCtx, nullptr) < 0) {
        qDebug("%s", "无法获取视频文件信息");
        if (pFormatCtx != nullptr) {
            avformat_close_input(&pFormatCtx); //关闭
            avformat_free_context(pFormatCtx); //释放内存
        }
        return false;
    }
    //获取视频流的索引位置
    //遍历所有类型的流（音频流、视频流、字幕流），找到视频流
    int v_stream_idx = -1;
    unsigned int i = 0;
    // number of streams
    for (; i < pFormatCtx->nb_streams; i++) {
        //流的类型
        if (pFormatCtx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            v_stream_idx = int(i);
            break;
        }
    }
    if (v_stream_idx == -1) {
        qDebug("%s", "找不到视频流");
        if (pFormatCtx != nullptr) {
            avformat_close_input(&pFormatCtx); //关闭
            avformat_free_context(pFormatCtx); //释放内存
        }
        return false;
    }
    //只有知道视频的编码方式，才能够根据编码方式去找到解码器
    //获取视频流中的编解码上下文
    AVCodecParameters *pCodecPara = pFormatCtx->streams[v_stream_idx]->codecpar;
    // 4.根据编解码上下文中的编码id查找对应的解码
    AVCodec *pCodec = avcodec_find_decoder(pCodecPara->codec_id);
    if (pCodec == nullptr) {
        qDebug("%s", "找不到解码器");
        if (pFormatCtx != nullptr) {
            avformat_close_input(&pFormatCtx); //关闭
            avformat_free_context(pFormatCtx); //释放内存
        }
        return false;
    }

    AVCodecContext *pCodecCtx = avcodec_alloc_context3(pCodec); //需要使用avcodec_free_context释放

    //事实上codecpar包含了大部分解码器相关的信息，这里是直接从AVCodecParameters复制到AVCodecContext
    avcodec_parameters_to_context(pCodecCtx, pCodecPara);

    // 5.打开解码器
    if (avcodec_open2(pCodecCtx, pCodec, nullptr) < 0) {
        qDebug("%s", "解码器无法打开");
        if (pFormatCtx != nullptr) {
            avformat_close_input(&pFormatCtx); //关闭
            avformat_free_context(pFormatCtx); //释放内存
        }
        if (pCodecCtx != nullptr) {
            avcodec_free_context(&pCodecCtx); //释放内存
        }
        return false;
    }
    //输出视频信息
    //            qDebug("视频的文件格式：%s",pFormatCtx->iformat->name);
    //            qDebug("视频时长：%d", (pFormatCtx->duration)/1000000);//s
    nTime = int((pFormatCtx->duration) / 1000000);
    //            qDebug("视频的宽高：%d,%d",pCodecCtx->width,pCodecCtx->height);
    //            qDebug("解码器的名称：%s",pCodec->name);
    //准备读取
    // AVPacket用于存储一帧一帧的压缩数据（H264）
    //缓冲区，开辟空间
    AVPacket *packet = reinterpret_cast<AVPacket *>(av_malloc(sizeof(AVPacket)));
    // AVFrame用于存储解码后的像素数据(YUV)
    //内存分配
    AVFrame *pFrame = av_frame_alloc();
    // YUV420
    AVFrame *pFrameYUV = av_frame_alloc();
    //只有指定了AVFrame的像素格式、画面大小才能真正分配内存
    //缓冲区分配内存
    uint8_t *out_buffer = static_cast<uint8_t *>(av_malloc(
        static_cast<size_t>(av_image_get_buffer_size(AV_PIX_FMT_RGB32, pCodecCtx->width, pCodecCtx->height, 1))));
    av_image_fill_arrays(pFrameYUV->data, pFrameYUV->linesize, out_buffer, AV_PIX_FMT_RGB32, pCodecCtx->width,
                         pCodecCtx->height, 1);

    //用于转码（缩放）的参数，转之前的宽高，转之后的宽高，格式等
    struct SwsContext *sws_ctx =
        sws_getContext(pCodecCtx->width, pCodecCtx->height, pCodecCtx->pix_fmt, pCodecCtx->width, pCodecCtx->height,
                       AV_PIX_FMT_BGRA, SWS_BICUBIC, nullptr, nullptr, nullptr);
    int frame_count = 0; // 帧
    //    6.一帧一帧的读取压缩数据

    while (av_read_frame(pFormatCtx, packet) >= 0) {
        //只要视频压缩数据（根据流的索引位置判断）
        if (packet->stream_index == v_stream_idx) {
            // 7.解码一帧视频压缩数据，得到视频像素数据
            int ret = avcodec_send_packet(pCodecCtx, packet);
            if (ret == 0) {
                ret = avcodec_receive_frame(pCodecCtx, pFrame);
            }
            if (ret < 0) {
                //                qDebug("%s", "解码错误");
                continue;
            }

            //为0说明解码完成，非0正在解码
            // AVFrame转为像素格式YUV420，宽高
            // 2 6输入、输出数据
            // 3 7输入、输出画面一行的数据的大小 AVFrame 转换是一行一行转换的
            // 4 输入数据第一列要转码的位置 从0开始
            // 5 输入画面的高度
            sws_scale(sws_ctx, pFrame->data, pFrame->linesize, 0, pCodecCtx->height, pFrameYUV->data,
                      pFrameYUV->linesize);

            //把这个RGB数据 用QImage加载
            QImage image(static_cast<uchar *>(out_buffer), pCodecCtx->width, pCodecCtx->height, QImage::Format_RGB32);
            image = image.scaled(512, 512);

            QString tempFile = QString("/tmp/phone-master-video.png");
            if (image.save(tempFile)) { //   保存为临时文件， 之后在读取

                pixmap = QIcon(tempFile);

                QFile::remove(tempFile); //   读取之后删除
            } else {
                qDebug() << "save video png failed";
            }
            frame_count++;
        }

        if (frame_count == 1)
            break;
    }

    sws_freeContext(sws_ctx);

    av_free(out_buffer);
    av_frame_free(&pFrameYUV);
    av_frame_free(&pFrame);
    av_free(packet);

    avcodec_free_context(&pCodecCtx);
    avcodec_close(pCodecCtx);

    avformat_close_input(&pFormatCtx);
    avformat_free_context(pFormatCtx);
    return true;
}

bool PhoneFileThread::refreshDirInfo(const RunMode &runMode, const QString &strDesPath, const int &nFileCount, const quint64 &nFileSize)
{
    if (strDesPath.isEmpty()) {
        return false;
    }

    QVector<PhoneFileInfo>::iterator itr_begin;
    QVector<PhoneFileInfo>::iterator itr_end;
    if (runMode == ReadPhoto) {
        itr_begin = m_mapDirFileInfo_photo.begin();
        itr_end = m_mapDirFileInfo_photo.end();
    } else {
        itr_begin = m_mapDirFileInfo_video.begin();
        itr_end = m_mapDirFileInfo_video.end();
    }

    while (itr_begin != itr_end) {
        PhoneFileInfo &item = *itr_begin;
        if (item.dirPath == strDesPath) {
            item.fileCount = nFileCount;
            item.size = nFileSize;
            break;
        }
        ++itr_begin;
    }
    return true;
}

// bool PhoneFileThread::_readVideo_ffmpegthumbnailer()
//{
/*
if(!autoFilterSet()){
    return false;
}

QStringList::iterator it = m_listFileNames.begin();
for (; it != m_listFileNames.end(); ++it) {
    try {
        std::vector<uint8_t> imageData;
        std::string path(it->toStdString());

        int size = 200;
        ffmpegthumbnailer::VideoThumbnailer vt(size, false, true, 20, false);
        vt.generateThumbnail(path, ThumbnailerImageTypeEnum::Png, imageData);
        if (imageData.empty())
            continue;

        QByteArray Ret_bytearray = QByteArray(reinterpret_cast<char *>(imageData.data()), int(imageData.size()));
        QPixmap imageresult;
        imageresult.loadFromData(Ret_bytearray);
        if (imageresult.isNull())
            continue;

        onVideoFindPixmap(imageresult, path.c_str(), 0, bIsEnd);
    } catch (std::exception &e) {
        qDebug() << __LINE__ << e.what();
    }
}
*/
//    return true;
//}

bool PhoneFileThread::setDir(const QString &strDesPath, QString &strDesItemPath)
{
    //    QDir dir;
    //    if (!dir.exists(strDesPath))
    //        return false;
    QStringList filters = getFileFilters();

    QList<QFileInfo> list_fileInfo;
    quint64 nSize = 0;
    //    qDebug() << __LINE__ << strDesPath;
    // 数量大小缓存
    bool bReaded = false;
    if (m_mapDirToFileCount.contains(strDesPath)) {
        bReaded = true;
    }

    QDirIterator dir_iterator(strDesPath, filters, QDir::Files | QDir::NoSymLinks, QDirIterator::Subdirectories);
    while (dir_iterator.hasNext()) {
        dir_iterator.next();
        QFileInfo file_info = dir_iterator.fileInfo();
        if (!file_info.isFile()) {
            continue;
        }
        list_fileInfo.append(file_info);
        nSize += quint64(file_info.size());

        if (list_fileInfo.count() == 1) {
            strDesItemPath = file_info.absoluteFilePath();
            m_mapFileToDir.insert(file_info.absoluteFilePath(), strDesPath);
        }
        if (bReaded) {
            break;
        }
    }
    if (!bReaded) {
        m_mapDirToFileCount.insert(strDesPath, list_fileInfo.count());
        m_mapDirToFileSize.insert(strDesPath, nSize);
    }

    if (list_fileInfo.isEmpty())
        return false;

    return true;
}

bool PhoneFileThread::autoFilterSet()
{
    QMutexLocker locker(&g_enterDirMutex);
    qDebug() << __LINE__ << __FUNCTION__ << m_strPhoneID;
    if (m_bIsDir) {
        QStringList listDirPaths;
        if (!m_strPhoneID.isEmpty()) {
            int nTimes = 10; // 10s //解决有些机器挂载慢问题，尝试10秒
            bool bRet = false;
            for (int i = 0; i < nTimes; ++i) {
                if (!m_isCanRun) {
                    return false;
                }

                bRet = getPhotoPath(m_strPhoneID, listDirPaths);
                if (bRet) {
                    break;
                }

                sleep(1);
            }
            if (!bRet) {
                emit error(PFERROR_NOTFIND_DCIM, TrObject::getInstance()->getMountText(Failed_mount_the_directories));
                return false;
            }
        }

        m_listFileNames = listDirPaths;
        //        foreach (QString itemPath, listDirPaths) {
        //            if (!m_isCanRun) {
        //                return false;
        //            }
        //            setDir(itemPath);
        //        }
        qDebug() << __LINE__ << __FUNCTION__;
    } else {
        setFile(m_strDesPath);
    }
    return true;
}

bool PhoneFileThread::getPhotoPath(QString strPhoneID, QStringList &listPath)
{
    listPath.clear();

    if (m_phoneType == Mount_Ios) {
        QDir dir;
        dir.setPath(IPhoneMountControl::getMountAbsolutePath(strPhoneID));
        if (dir.exists()) {
            if (dir.cd("DCIM")) {
                listPath.append(dir.absolutePath());
                return true;
            }
        }
        return false;
    }

    QDir dir(Utils::mountGvfsPath());
    qDebug() << __LINE__ << __FUNCTION__ << Utils::mountGvfsPath();
    QFileInfoList list = dir.entryInfoList(QDir::Dirs | QDir::NoDotAndDotDot);
    qDebug() << __LINE__ << __FUNCTION__ << list.count();
    for (int i = 0; i < list.size(); ++i) {
        QFileInfo fileInfo = list.at(i);
        QString file_path = fileInfo.fileName();
        if (file_path.contains("_" + strPhoneID)) {
            if (dir.cd(file_path)) {
                QStringList listChildrenDir = dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot);
                if (listChildrenDir.count() >= 1) { //多个sd卡 || 照片模式
                    qDebug() << __LINE__ << QString("挂载文件夹数量[%1]:").arg(strPhoneID) << listChildrenDir.count();

                    if (m_isCanRun == false)
                        break;

                    QString strCurPath = dir.absolutePath();

                    bool bIsPhotoMode = false;
                    if (listChildrenDir.contains("DCIM"))
                        bIsPhotoMode = true;
                    foreach (QString strItemChildPath, listChildrenDir) {
                        if (m_isCanRun == false)
                            break;

                        dir.setPath(strCurPath + QDir::separator() + strItemChildPath);
                        qDebug() << __LINE__ << __FUNCTION__ << dir.absolutePath();
                        if (file_path.contains("iPhone")) { // iphone
                            if (!dir.cd("DCIM")) {
                                //                                return false;
                            }
                            return true;
                        } else { // Android
                            if (bIsPhotoMode)
                                listPath.append(dir.absolutePath());
                            else
                                listPath.append(getAndroidPhotoPath(dir.absolutePath()));
                        }
                    }
                    //                    if (listPath.isEmpty()) { //照片模式 getAndroidPhotoPath判断是否存在
                    //                        qDebug() << __LINE__ << dir.cd("..");
                    //                        listPath.append(getAndroidPhotoPath(dir.absolutePath()));
                    //                        qDebug() << __LINE__ << listPath.count();
                    //                    }
                    return true;
                }
            }
        }
    }
    return false;
}

QStringList PhoneFileThread::getAndroidPhotoPath(QString strBasePath)
{
    if (m_isCanRun == false)
        return QStringList();

    QStringList listPath;
    QDir dir(strBasePath);
    QString strPath;
    //下载
    strPath = dir.absolutePath() + "/Download"; // 不能用arg
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    // QQ
    strPath = dir.absolutePath() + "/tencent/QQ_Images";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    strPath = dir.absolutePath() + "/Tencent/QQ_Images";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    //微信
    strPath = dir.absolutePath() + "/tencent/MicroMsg/WeiXin";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    strPath = dir.absolutePath() + "/Tencent/MicroMsg/WeiXin";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    //                        //淘宝
    //                        strPath = dir.absolutePath() + "/Pictures/taobao";
    //                        if (dir.exists(strPath)) {
    //                            listPath.append(strPath);
    //                        }
    //                        //苏宁
    //                        strPath = dir.absolutePath() + "/Pictures/suning";
    //                        if (dir.exists(strPath)) {
    //                            listPath.append(strPath);
    //                        }
    //微博
    strPath = dir.absolutePath() + "/sina/weibo/storage/photoalbum_save/weibo";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    strPath = dir.absolutePath() + "/sina/weibo/weibo";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    // QQ音乐
    strPath = dir.absolutePath() + "/qqmusic/mv";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    //网易云
    strPath = dir.absolutePath() + "/netease/cloudmusic/MV";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    //酷狗
    strPath = dir.absolutePath() + "/kugou/mv";
    if (dir.exists(strPath)) {
        listPath.append(strPath);
    }
    if (!dir.cd("DCIM"))
        return listPath;
    foreach (QFileInfo itemFileInfo, dir.entryInfoList(QDir::Dirs | QDir::NoDotAndDotDot)) {
        listPath.append(itemFileInfo.absoluteFilePath());
    }
    if (dir.cd("../Pictures")) {
        foreach (QFileInfo itemFileInfo, dir.entryInfoList(QDir::Dirs | QDir::NoDotAndDotDot)) {
            if (m_isCanRun == false)
                break;

            listPath.append(itemFileInfo.absoluteFilePath());
            //            qDebug() << __LINE__ << itemFileInfo.absoluteFilePath();
        }
    }
    return listPath;
}

// PhoneFileType PhoneFileThread::getFileType(QString strFilePath)
//{
//    QMimeDatabase db;
//    QMimeType mime = db.mimeTypeForFile(strFilePath);
//    //    qDebug() << __LINE__ << mime.name();
//    if (mime.name().startsWith("image/"))
//        return IMAGE;
//    else if (mime.name().startsWith("video/")) {
//        return VIDEO;
//    }
//    return UNKNOW;
//}

void PhoneFileThread::setFile(QString strDesPath)
{
    QStringList filters = getFileFilters();

    quint64 size = 0;
    QDirIterator dir_iterator(strDesPath, filters, QDir::Files | QDir::NoSymLinks, QDirIterator::Subdirectories);
    while (dir_iterator.hasNext()) {
        if (!m_isCanRun) {
            return;
        }
        dir_iterator.next();
        QFileInfo file_info = dir_iterator.fileInfo();
        QString absPath = file_info.absoluteFilePath();
        size += quint64(file_info.size());

        m_listFileNames.append(absPath);
    }

    refreshDirInfo(m_mode, strDesPath, m_listFileNames.count(), size);
}

QPixmap PhoneFileThread::resizePicture(const QPixmap &icon)
{
    if (icon.isNull()) {
        //读取失败
        QPixmap pix(m_sMaxSize);
        pix.fill(Qt::transparent); //用透明色填充
        return pix;
    }
    QPixmap newIcon;
    if (m_bKeepAspectRatio) {
        newIcon = icon.scaled(m_sMaxSize, Qt::KeepAspectRatio);
        return newIcon;
    }

    if (icon.width() > icon.height()) {
        newIcon = icon.copy(0, 0, icon.height(), icon.height());
    } else {
        newIcon = icon.copy(0, 0, icon.width(), icon.width());
    }

    newIcon = newIcon.scaled(m_sMaxSize);
    //    newIcon = newIcon.scaled(150, 150);

    return newIcon;
}

QStringList PhoneFileThread::getFileFilters()
{
    /*
    通讯录格式：csv vcf xml
    图片格式：bmp gif jpeg jpg png wbmp webp
    视频格式：3gp avi f4v flv mkv mov mp4 rmvb wmv
    音乐格式：aac amr ape flac m4a mid mp3 ogg wav wma xmp
    电子书格式:chm doc ebk2 epub fb2 jar mobi ndb ndz pdb pdf pmlz rb rtf tcr txt zip
    */
    QStringList filters;
    if (m_mode == ReadPhoto) {
        filters << getImageFilters();
    } else if (m_mode == ReadVideo) {
        filters << getVideoFilters();
    }
    return filters;
}

bool PhoneFileThread::processCache()
{
    if (!m_bIsDir)
        return false;

    QStringList listFileNames;
    int nCount = 0;

    if (m_mode == ReadPhoto) {
        foreach (PhoneFileInfo item, m_mapDirFileInfo_photo) {
            if (m_strPhoneID != item.phoneID)
                continue;
            if (!m_isCanRun) {
                break;
            }
            listFileNames.append(item.path);
            emit sigFindPicture(item);
            ++nCount;
        }
    } else if (m_mode == ReadVideo) {
        foreach (PhoneFileInfo item, m_mapDirFileInfo_video) {
            if (m_strPhoneID != item.phoneID)
                continue;
            if (!m_isCanRun) {
                break;
            }
            listFileNames.append(item.path);
            emit sigFindPicture(item);
            ++nCount;
        }
    }

    if (nCount == 0)
        return false;

    this->setProperty("filelist", listFileNames);
    //缓存处理了 不用外面处理
    emit error(PEERROR_END, ""); // 界面停止spinner
    return true;
}

QStringList PhoneFileThread::getImageFilters()
{
    QStringList filters;
    filters << "*.png"
            << "*.jpg"
            << "*.bmp"
            << "*.jpeg"
            << "*.svg"
            << "*.gif";
    return filters;
}

QStringList PhoneFileThread::getVideoFilters()
{
    QStringList filters;
    filters << "*.mp4"
            << "*.mov"
            << "*.avi"
            << "*.rmvb"
            << "*.rm"
            << "*.flv"
            << "*.mkv"
            << "*.f4v"
            << "*.wmv"
            << "*.3gp";
    return filters;
}

QStringList PhoneFileThread::getListFileNames() const
{
    return m_listFileNames;
}
