/*
 * Copyright (C) 2019 ~ 2020 Deepin Technology Co., Ltd.
 *
 * Author:     Kevin.Guan <allmemory@vip.qq.com>
 *             caowei <zhtqs8@163.com>
 *
 * Maintainer: Kevin.Guan <allmemory@vip.qq.com>
 *             caowei <zhtqs8@163.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "cpreload.h"
#include <QSharedMemory>
#include <QUuid>
#include <QImage>
#include <QPixmap>
#include <QDBusInterface>
#include <QDBusMessage>
#include <QDBusConnection>
#include <QDBusVariant>
#include <tuple>
#include <assert.h>
#include <QDebug>
#include <QFile>
#include <QDir>
#include <QApplication>
#include <QDBusReply>
#include <QFileInfo>
#include <QPointer>
#include <sys/stat.h>

QString getCurrentSpaceImagePath();
QString getWorkSpaceImagePath (int spaceNo);
QString getLockImagePath (uid_t uid);
int getCurrentSpaceNumber();

class ImageKey
{
public:
    int purpose;//0表示锁屏，1-16表示工作空间，它跟filePath、memory是一对一的关系
    QString filePath;//原始设置图片
    QString cacheFilePath;//毛玻璃特效缓存图片
    QString shareKey;
    QSharedPointer<QSharedMemory> memory;

    // purpose初始化给个明确的无效值-1
    ImageKey() : purpose(-1), shareKey(QUuid::createUuid().toString(QUuid::WithoutBraces))
    {

    }
};

cPreLoad::cPreLoad (QObject *parent) : QObject (parent)
{
    init();
}


cPreLoad::~cPreLoad()
{
    for (auto plist : m_resources) {
        auto &list = *plist;
        for (auto &pik : list) {
            if (pik->memory->attach()) {
                pik->memory->detach();
            }
        }
    }

}


/**
 * @brief 获取图片的共享内存的key，同时刷新共享内存
 *
 * @param uid Linux用户id
 * @param purpose 用途，锁屏为1,启动器2,工作区为3-19
 * @return QString 共享内存的key
 */
QString cPreLoad::requestSource (int uid, int purpose)
{
    QString shareKey;
    QSharedPointer<ImageKey> pik;
    if (m_resources.contains (uid)) {
        QList<QSharedPointer<ImageKey>> &list = *m_resources[uid];
        auto fit = std::find_if (list.begin(), list.end(), [purpose] (const QSharedPointer<ImageKey> &ik)->bool{
            return ik->purpose == purpose;
        });
        if (fit != list.end()) {
            //在共享内存区找到了该路径
            //为防止一些极端操作比如资源名相同(检测MD5有损耗)，还是得释放掉共享内存再新建一个
            pik = *fit;
            shareKey = pik->shareKey;
            updaetImagePath (pik, uid);
        } else {
            //没有找到的话需要新建一个
            pik = createResource (purpose, uid);
            shareKey = pik->shareKey;
            updaetImagePath (pik, uid);
            m_resources[uid]->append(pik);
        }
    } else {
        pik = createResource (purpose, uid);
        shareKey = pik->shareKey;
        updaetImagePath (pik, uid);
        m_resources[uid].reset(new QList<QSharedPointer<ImageKey>>());
        m_resources[uid]->append(pik);
    }
    return shareKey;
}

//初始化
void cPreLoad::init()
{
    m_resources.clear();
}

//资源是否存在
bool cPreLoad::isResourceExist (QString _path)
{
    uid_t iUserId = getuid();
    return isResourceExist (_path, iUserId);
}

//资源是否存在
bool cPreLoad::isResourceExist (QString _path, uint32_t uid)
{
    if (m_resources.contains (uid)) {
        auto &list = *m_resources[uid];
        for (auto it : list) {
            if (it->filePath == _path) {
                return true;
            }
        }
    }
    return false;
}

//获取资源路径
QString cPreLoad::getResourcePath (int purpose)
{
    uid_t iUserId = getuid();
    return getResourcePath (purpose, iUserId);
}

QString cPreLoad::getResourcePath (int purpose, uint32_t uid)
{
    QString sourcePath;
    //调用者应先调用isResourceExist，此处只防错误的调用
    assert (m_resources.contains (uid));
    auto list = m_resources[uid];
    QSharedPointer<ImageKey> pik = getResource (purpose, uid);
    if (pik) {
        sourcePath = pik->filePath;
    }
    return sourcePath;
}

QSharedPointer<ImageKey> cPreLoad::getResource (int purpose, uint32_t uid)
{
    QSharedPointer<ImageKey> pret;
    //调用者应先调用isResourceExist，此处只防错误的调用
    assert (m_resources.contains (uid));
    auto &list = *m_resources[uid];
    for (auto ik : list) {
        if (ik->purpose == purpose) {
            pret = ik;
            break;
        }
    }
    return pret;
}

//创建内存共享的资源
QSharedPointer<ImageKey> cPreLoad::createResource (int purpose, uint32_t uid)
{
    QSharedPointer<QList<QSharedPointer<ImageKey>>> plist;
    if (m_resources.contains (uid)) {
        plist = m_resources[uid];
        auto &list = *plist;
        QSharedPointer<ImageKey> pik;
        for (auto ik : list) {
            if (ik->purpose == purpose) {
                pik = ik;
                break;
            }
        }
        if (pik) {
            qInfo() << "dde-preload found user image. user id: " << uid << ", purpose: " << purpose;
            return pik;
        } else {
            qInfo() << "dde-preload could not find specific image memory, user id: " << uid << ", purpose: " << purpose;
            pik.reset (new ImageKey);
            pik->purpose = purpose;
            return pik;
        }
    } else {
        qInfo() << "dde-preload could not find user category, user id: " << uid << ", purpose: " << purpose;
        QSharedPointer<ImageKey> pik (new ImageKey);
        pik->purpose = purpose;
        return pik;
    }

}

//更新图片的共享内存
void cPreLoad::updaetImagePath(QSharedPointer<ImageKey> &imageKey, uint32_t uid)
{
    QString imagePath;
    QDir d = QDir::home();
    if (!d.exists(m_blurPath)) {
        if (d.mkpath(m_blurPath)) {
            qDebug() << "updaetImagePath mkpath:" << m_blurPath;
            chmod(d.absolutePath().toUtf8(), S_IXUSR | S_IRUSR | S_IWUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
        } else {
            qDebug() << "updaetImagePath mkpath failed:" << m_blurPath << ", cd .cache:" << d.cd(".cache");
        }
    }
    d.cd (m_blurPath);
    if (imageKey->purpose == 1) {
        const QString fileName = QString::number (imageKey->purpose) + ".jpg";
        imageKey->cacheFilePath = d.absoluteFilePath (fileName);
        if (d.exists (fileName)) {
            imagePath = imageKey->cacheFilePath;
        }
    } else if (imageKey->purpose == 2) {
        int iSpaceNo = getCurrentSpaceNumber();
        if (iSpaceNo > 0) {
            const QString fileName = "2-" + QString::number (iSpaceNo) + ".jpg";
            imageKey->cacheFilePath = d.absoluteFilePath (fileName);
            if (d.exists (fileName)) {
                imagePath = imageKey->cacheFilePath;
            }
        } else {
            qWarning() << __FILE__ << ", " << Q_FUNC_INFO << ", can not get workspace number.";
        }
    }
    if (imagePath.isEmpty()) {
        imagePath = getImagePath (imageKey->purpose, uid);
    }
    if (imagePath.isEmpty()) {
        qInfo() << __FILE__ << ", " << Q_FUNC_INFO << ", file image not found.";
        imagePath = ":/default.jpg";
    } else {
        QFileInfo fi (imagePath);
        if (!fi.isReadable()) {
            qInfo() << __FILE__ << "dde-preload cannot open: " << imagePath << ", file image path: " << imagePath;
            imagePath = ":/default.jpg";
        }
    }
    if (imagePath.startsWith (":/")) {
        imagePath = d.absoluteFilePath ("executable_default.jpg");
        if (QFile::exists (imagePath)) {
            QFile::remove (imagePath);
        }
        QImage img (imagePath);
        img.save (imagePath);
    }
    imageKey->filePath = imagePath;
    readImage (imageKey, uid);
}

bool makeBlurImage (const QString &inputImagePath, const QString &outputImagePath)
{
    const QString effect = "com.deepin.daemon.ImageEffect";
    QDBusMessage send = QDBusMessage::createMethodCall (effect, "/com/deepin/daemon/ImageEffect", effect, "Get");
    QList<QVariant> args;
    args.append ("");
    args.append (inputImagePath);
    send.setArguments (args);
    QDBusMessage rec = QDBusConnection::systemBus().call (send);
    if (rec.type() == QDBusMessage::ReplyMessage) {
        QDBusReply<QString> hd (rec);
#ifdef QT_DEBUG
        qInfo() << "makeBlurImage image effect: " << hd.value();
#endif
        if (QFile::exists (outputImagePath)) {
            QFile::remove (outputImagePath);
        }
        QImage img (hd.value());
        img.save (outputImagePath);
        chmod (outputImagePath.toUtf8().data(), S_IXUSR | S_IRUSR | S_IWUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
        return true;
    }
    qWarning() << "dde-preload cannot call " << effect << ", detail: " << rec;
    return false;
}

/**
 * @brief 调用dbus提供的图像虚化服务，进行图像虚化
 *
 * @param filePath 图片路径，可能是exe的资源文件
 * @param uid 请求者
 * @return QByteArray 虚化后的图像的二进制数据
 */
QByteArray cPreLoad::blurImage (QSharedPointer<ImageKey> &imageKey)
{
    QFileInfo fi (imageKey->cacheFilePath);
    QDir d = fi.dir();
    QString homePath = d.absolutePath();
    if (d.isReadable()) {
        if (makeBlurImage (imageKey->filePath, imageKey->cacheFilePath)) {
            QFile f (imageKey->cacheFilePath);
            f.open (QFile::ReadOnly);
            return f.readAll();
        } else {
            qWarning() << "dde-preload blur image: " << imageKey->cacheFilePath << " fail.";
            QFile f (imageKey->filePath);
            f.open (QFile::ReadOnly);
            return f.readAll();
        }
    } else {
        qWarning() << "dde-preload get user home: " << homePath << " cannot create file.";
        QFile f (imageKey->filePath);
        f.open (QFile::ReadOnly);
        return f.readAll();
    }
}

/**
 * @brief 读取图像，强制刷新共享内存
 *
 * @param imageKey 共享内存的关联信息
 * @param filePath
 * @param uid
 */
void cPreLoad::readImage (QSharedPointer<ImageKey> &imageKey, uint32_t uid)
{
    QByteArray ba;
    if (imageKey->purpose > 2) {
        QFile imgFile (imageKey->filePath);
        imgFile.open (QIODevice::ReadOnly);
        ba = imgFile.readAll();
    } else {
        if (QFile::exists (imageKey->cacheFilePath)) {
            QFile imgFile (imageKey->cacheFilePath);
            imgFile.open (QIODevice::ReadOnly);
            ba = imgFile.readAll();
        } else {
            ba = blurImage (imageKey);
        }
    }
    auto &memory = imageKey->memory;
    if (memory &&  memory->attach()) {
        memory->detach();
        emit releaseSuccessful (int (uid), imageKey->purpose, imageKey->shareKey);
    }
    memory.reset (new QSharedMemory (imageKey->shareKey));
    // memory.reset(new QSharedMemory());
    // memory->setNativeKey(imageKey->shareKey);
    memory->attach();
    memory->create (ba.size());
    memory->lock();
    memcpy (memory->data(), ba.data(), ba.size());
    memory->unlock();
    emit createSuccessful (int (uid), imageKey->purpose, imageKey->shareKey);
}

QString resolvePathFromMessage (QDBusMessage& ret, const QString dbusName, const QString& dbusMethod)
{
    switch (ret.type()) {
    case QDBusMessage::ErrorMessage:
        qWarning() << "dde-preload call " << dbusName << ", " << dbusMethod <<  " encountered error: " << ret.errorMessage();
        break;
    case QDBusMessage::ReplyMessage: {
        QDBusReply<QDBusVariant> reply1 (ret);
        QDBusReply<QString> reply2 (ret);
        if (reply1.isValid()) {
            QString filePath = reply1.value().variant().toString();
            filePath = filePath.remove ("file://");
            return filePath;
        } else if (reply2.isValid()) {
            QString filePath = reply2.value();
            filePath = filePath.remove ("file://");
            return filePath;
        } else {
            qWarning() << "dde-preload call " << dbusName << ", " << dbusMethod <<  " encountered empty return value. error: " << reply1.error().message();
            return "";
        }
    }
    default:
        qWarning() << "dde-preload call " << dbusName << ", " << dbusMethod <<  " encountered unkown message type. ";
        break;
    }
    return "";
}

QString getLockImagePath (uid_t uid)
{
    QString dbusName;
    QString errorMessage;
    QDBusMessage ret;
    QString dbusMethod;
    QString dbusPath (QString ("/com/deepin/daemon/Accounts/User%1").arg (uid));
    dbusName = "com.deepin.daemon.Accounts";
    dbusMethod = "GreeterBackground"; //'com.deepin.daemon.Accounts.User','GreeterBackground'
    QDBusMessage msg = QDBusMessage::createMethodCall (dbusName, dbusPath, "org.freedesktop.DBus.Properties", "Get");
    QList<QVariant> args;
    args.append (dbusName + ".User");
    args.append (dbusMethod);
    msg.setArguments (args);
    QDBusConnection conn = QDBusConnection::systemBus();
    ret = conn.call (msg/*,QDBus::Block,3*/);
    return resolvePathFromMessage (ret, dbusName, dbusMethod);
}

QString getWorkSpaceImagePath (int spaceNo)
{
    if (spaceNo == -1) {
        qWarning() << __FILE__ << ", " << Q_FUNC_INFO << ", invalid space numboer: " << spaceNo;
        return "";
    }
    QString dbusName;
    QString errorMessage;
    QDBusMessage ret;
    QString dbusMethod;
    dbusName = "com.deepin.wm";
    dbusMethod = "GetWorkspaceBackground";
    QDBusMessage msg = QDBusMessage::createMethodCall (dbusName, "/com/deepin/wm", dbusName, dbusMethod);
    QList<QVariant> args;
    args.append (spaceNo);
    msg.setArguments (args);
    QDBusConnection conn = QDBusConnection::sessionBus();
    ret = conn.call (msg);
    return resolvePathFromMessage (ret, dbusName, dbusMethod);
}

QString getCurrentSpaceImagePath()
{
    QString dbusName;
    QString errorMessage;
    QDBusMessage ret;
    QString dbusMethod;
    dbusName = "com.deepin.wm";
    dbusMethod = "GetCurrentWorkspaceBackground";
    QDBusMessage msg = QDBusMessage::createMethodCall (dbusName, "/com/deepin/wm", dbusName, dbusMethod);
    QDBusConnection conn = QDBusConnection::sessionBus();
    ret = conn.call (msg);
    return resolvePathFromMessage (ret, dbusName, dbusMethod);
}

int getCurrentSpaceNumber()
{
    QString dbusName;
    QString errorMessage;
    QDBusMessage ret;
    QString dbusMethod;
    dbusName = "com.deepin.wm";
    dbusMethod = "GetCurrentWorkspace";
    QDBusMessage msg = QDBusMessage::createMethodCall (dbusName, "/com/deepin/wm", dbusName, dbusMethod);
    QDBusConnection conn = QDBusConnection::sessionBus();
    ret = conn.call (msg);
    if (ret.type() == QDBusMessage::ErrorMessage) {
        qWarning() << __FILE__ << "," << Q_FUNC_INFO << "," << ret;
        return -1;
    }
    QDBusReply<int> reply (ret);
    return reply.value();

}



//从DBus获取用户的图片路径
QString cPreLoad::getImagePath (int purpose, uint32_t uid)
{
    if (purpose == 1) {
        return getLockImagePath (uid);
    } else if (purpose == 2) {
        return getCurrentSpaceImagePath();
    } else {
        return getWorkSpaceImagePath (purpose - 2);
    }
    return "";
}

void cPreLoad::exit()
{
    QApplication::quit();
}
