#
# openslide-python - Python bindings for the OpenSlide library
#
# Copyright (c) 2016-2021 Benjamin Gilbert
#
# This library is free software; you can redistribute it and/or modify it
# under the terms of version 2.1 of the GNU Lesser General Public License
# as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
# License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this library; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#

from functools import wraps
import os
from pathlib import Path

# Handle Windows-specific first-import logic here, so individual modules
# don't have to
if os.name == 'nt':
    # In application code, you probably shouldn't use an environment
    # variable for this, unless you're sure you can trust the contents of the
    # environment.
    _dll_path = os.getenv('OPENSLIDE_PATH')
    if _dll_path is not None:
        if hasattr(os, 'add_dll_directory'):
            # Python >= 3.8
            with os.add_dll_directory(_dll_path):
                import openslide
        else:
            # Python < 3.8
            _orig_path = os.environ.get('PATH', '')
            os.environ['PATH'] = _orig_path + ';' + _dll_path
            import openslide  # noqa: F401  module-imported-but-unused

            os.environ['PATH'] = _orig_path

from openslide import OpenSlideVersionError


def file_path(name):
    return Path(__file__).parent / name


def maybe_supported(f):
    '''Decorator to ignore test failures caused by an OpenSlide version that
    doesn't support the tested functionality.'''

    @wraps(f)
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except OpenSlideVersionError:
            pass

    return wrapper
