/*******************************************************************************
*                                                                              *
*   (C) 1997-2021 by Ernst W. Mayer.                                           *
*                                                                              *
*  This program is free software; you can redistribute it and/or modify it     *
*  under the terms of the GNU General Public License as published by the       *
*  Free Software Foundation; either version 2 of the License, or (at your      *
*  option) any later version.                                                  *
*                                                                              *
*  This program is distributed in the hope that it will be useful, but WITHOUT *
*  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
*  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for   *
*  more details.                                                               *
*                                                                              *
*  You should have received a copy of the GNU General Public License along     *
*  with this program; see the file GPL.txt.  If not, you may view one at       *
*  http://www.fsf.org/licenses/licenses.html, or obtain one by writing to the  *
*  Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA     *
*  02111-1307, USA.                                                            *
*                                                                              *
*******************************************************************************/

/****************************************************************************
 * We now include this header file if it was not included before.
 ****************************************************************************/
#ifndef fac_test_dat64_included
#define fac_test_dat64_included

	#include "types.h"

	struct testFac{
		uint32 p;
		uint64 q;
	};

	/*******************************************/
	/*      Fermat-number test factors:        */
	/*******************************************/

	// Here interpret the above testFac struct as a minimalist [n,k]-pair format,
	// where Fn = 2^2^n+1 is the Fermat number and q = k.2^(n+2)+1 the factor:
	// To check any particular (alleged) factor q of Fn using Pari, use Mod(2,q)^(2^n)+1.

	// Testcases with factors < 2^64 = 18446744073709551616 Must pad each array with a trailing pair of zeros:
	static const struct testFac ffac64[] =
	{
		{ 5,5ull},{ 5,52347ull},
		{ 6,1071ull},{ 6,262814145745ull},
		{ 7,116503103764643ull},
		{ 8,1209889024954ull},
		{ 9,1184ull},
		{10,11131ull},{10,1583748ull},
		{11,39ull},{11,119ull},
		{12,7ull},{12,1588ull},{12,3892ull},{12,11613415ull},{12,76668221077ull},
		{13,82731770ull},{13,81294216221684ull},{13,109958186173776ull},
		{15,9264ull},{15,17753925353ull},
		{16,3150ull},
		{17,59251857ull},
		{18,13ull},
		{19,33629ull},{19,308385ull},
		{21,534689ull},
		{23,5ull},
		{25,193652ull},{25,1522849979ull},{25,16168301139ull},	// k = 2^2.48413, 7^2.131^2.1811, 3.487.11066599
		{26,286330ull},
		{27,282030ull},{27,430816215ull},
		{29,1120049ull},				// k = 7.53.3019
		{30,149041ull},{30,255178ull},	// k = 103.1447, 2.7.11.1657
		{32,1479ull},
		{36,10ull},{36,3759613ull},
		{38,6ull},{38,2653ull},
		{39,21ull},
		{42,86970ull},
		{55,29ull},
		{0,0ull}
	};

	/*******************************************/
	/*      Mersenne-number test factors:      */
	/*******************************************/

	// Exponent/factor pair arrays for self-tests of bitlengths 63-65. Must pad each array with a trailing pair of zeros.
	// To check any particular (alleged) factor q of 2^p-1 using Pari, use Mod(2,q)^p-1.

	// Testcases for factors up to 65 bits are in the form {exponent, factor} pairs where, for 65-bit factors, the q-entry
	// of the struct is the lowest 64 bits of the true factor (i.e. if it's a 65-bit factor, we infer the leading ones bit.)

	/* Testcases with factors < 2^63 = 9223372036854775808: */
	static const struct testFac fac63[] =
	{	// Start with the 2 smallest factors of MM31:
		{2147483647,   295257526626031ull},{2147483647, 87054709261955177ull},{16232017, 7411932855077041073ull},{16232537, 6692065114888767967ull},{16233463, 7281938355280163287ull},
		{16238797, 7160849857727202529ull},{16241383, 8321268976801733153ull},{16247081, 6851345661512506279ull},{16264097, 6651246518968801591ull},{16270871, 8750884836637970639ull},
		{16273021, 8530948234727688607ull},{16275769, 8656222760818941631ull},{16276951, 8420763231476654791ull},{16287319, 7856663174122645703ull},{16291313, 7319751180156635263ull},
		{16292609, 9092939873876576857ull},{16304579, 8962807897733335697ull},{16309159, 7820104075051972591ull},{16310741, 7118960435061194119ull},{16315669, 7962842793467390041ull},
		{16317043, 8806735752647806361ull},{16323829, 8450744492397916153ull},{16336891, 7155399841291768831ull},{16344187, 8776703566703469847ull},{16344221, 8505188423818202153ull},
		{16344301, 7953229808761833919ull},{16357703, 7108223032778653207ull},{16372787, 6619838028511456999ull},{16373947, 7438987258875431921ull},{16377859, 8096713054586341063ull},
		{16384493, 7028583389392946873ull},{16385287, 7046001957550504913ull},{16386353, 7907418222099564433ull},{16393847, 7396873540638046079ull},{16402271, 8752890924846173591ull},
		{16410893, 6781869984316584449ull},{16413157, 8150367549328875839ull},{16419283, 8379034208328962239ull},{16419373, 4590586513826579537ull},{16421459, 1904620098914423441ull},
		{16421773, 1919634559953409777ull},{16422079, 1617297709004215951ull},{16422143,  351667921325618969ull},{16422587,  347957656837497407ull},{16422607, 5079529025476239473ull},
		{16422617,  877523758063210759ull},{16422641, 1904594759403697657ull},{16423129,  639383433125674231ull},{16424033,  876833185557798433ull},{16425109, 6929484154075657241ull},
		{16425967, 4868798027321748281ull},{16426297,  664300343615680241ull},{16427093,  647622110489212231ull},{16430203, 5947021049954262703ull},{16430263,  531416416863478007ull},
		{16430567, 6714238248345211081ull},{16430759, 1040588024141145631ull},{16431221, 7515765841945454129ull},{16431301,  769260698510760751ull},{16432147, 2775802912966282649ull},
		{16432331, 4485830256538297097ull},{16432817, 7811250160082452639ull},{16433741,  373097143060190383ull},{16433777, 7025465442860330327ull},{16433867, 2122188836804569639ull},
		{16433881, 6274909219403483887ull},{16433887, 1178275771257792817ull},{16435187,  404893783099670863ull},{16435207, 5706498478502752097ull},{16436087, 1249687921657073809ull},
		{16436681, 5418270029220506569ull},{16437227,  880527514744739041ull},{16437803, 1050659558835031697ull},{16438349, 1152641923853707873ull},{16438459, 8494499222030730001ull},
		{16438733,  897957128490827369ull},{16438921,  355469347041668513ull},{16438931, 6363980290034318527ull},{16439183,  727804660219798193ull},{16439531,  381922949073827639ull},
		{16440629,  551592949703045713ull},{16441427,  514743839833790687ull},{16441723, 7826810543654605759ull},{16442941,  852026135819507489ull},{16443001,  680825060844814393ull},
		{16443893, 1277077523412980857ull},{16444399,  403946809003560737ull},{16445029, 1313020171506418103ull},{16445449, 4532532117087171361ull},{16445881,  314816963337354217ull},
		{16445999, 4000347492271511999ull},{16447793, 3909966094211431663ull},{16448987, 6702585753220580233ull},{16449113, 1395256032300764801ull},{16449187,  371179628750936801ull},
		{16449593, 1283294140172967047ull},{16449661,  382963805761197047ull},{16449827,  589484546028463361ull},{16450703, 8740277214666380953ull},{16451681, 2707820247743716409ull},
		{17715697, 2405285686624656961ull},{17716921, 7862694219641149649ull},{17725703, 1709828509468925647ull},{17729219, 1818195666940512361ull},{17732269, 4749139467781751281ull},
		{17732531, 8620020444320802169ull},{17741401, 1588232881627535471ull},{17741771, 2800144830835674079ull},{17743751, 2269674038404230343ull},{17747491, 8598509628823480727ull},
		{17748257, 9087140485434854489ull},{17750333, 7251833404705913977ull},{17751163, 3820553770628769199ull},{17752633, 4704959116091028679ull},{17753363, 7075257636385056583ull},
		{17754259, 7297159751574473761ull},{17758189, 4396811225247180479ull},{17760511, 4745090588707286561ull},{17760893, 2853041236643552393ull},{17765563, 1672908406048342823ull},
		{17765659, 4428560580432537343ull},{17767787, 1837671283556578471ull},{17770043, 3092513594545328617ull},{17770777,  779919558137764991ull},{17771933, 1217980041393079889ull},
		{17771989, 3683503882483319137ull},{17772103,  934951397486511551ull},{17773807, 2835009341000295337ull},{17774051, 1916127240891187847ull},{17774917, 6513143384743979257ull},
		{17776513, 1948595367612706247ull},{17777099, 3140841499126593247ull},{17777297, 6654525407953706177ull},{17778391, 2960846207983915487ull},{17779787, 8799635039454374711ull},
		{17780947, 1338327839026813807ull},{17786393, 2479692584449808633ull},{17786479, 3705119304553621399ull},{17788829, 4337975187311263769ull},{17793301, 3363405780011350313ull},
		{17796997,  644515749614290871ull},{17799139, 4850550556337408063ull},{17809373, 8858673253181631769ull},{17811337, 5616750395550952409ull},{17811853, 3227714233149548983ull},
		{17812411, 4684506166881580409ull},{17812589, 8674042470759946327ull},{17814707, 1181105980369035367ull},{17816543, 7023683719607349833ull},{17817647, 5482282061362804081ull},
		{17818763, 8358943986123319271ull},{17820659, 1666842203779044511ull},{17821913, 1622936916022631081ull},{17822183, 3907752947435394377ull},{17826299, 1329300478629304361ull},
		{17828047, 2423187390334065937ull},{17829613, 3369241733425614041ull},{17831083, 1477733878367589473ull},{17833499, 1531563491873924849ull},{17834923, 2190674486659724089ull},
		{17835149, 3271992840407563639ull},{17835347, 3564903515554180057ull},{17838193, 1741480327901108647ull},{17838697, 6672437126818019801ull},{17839741, 5448515813271799129ull},
		{17842001, 4652601937964804191ull},{17845909, 8815568571441254321ull},{17845969, 3553263474276793049ull},{17846581, 2048983536460732087ull},{17847689, 7778712127779173767ull},
		{17848387, 3907383057407439271ull},{17848933, 2464245591905914177ull},{17849053, 2150183260070870471ull},{17849191, 2407856202749932553ull},{17851157, 5888131719304374961ull},
		{17851307, 8439995926524285881ull},{17851741, 3407041535153554681ull},{17852273, 6051572822283142393ull},{17857379,  685402904791012417ull},{17861069, 5197640349369569873ull},
		{17861183, 2817440372348117777ull},{17861309, 2323778877594576001ull},{17861561, 7613389509871645681ull},{17862001, 1274705791716433303ull},{17863873, 3718381908400885441ull},
		{17867039, 2848730709312090247ull},{17868757, 4049709796007485271ull},{17869031, 7240853613770980279ull},{17870807, 3143753649823393489ull},{17870953, 1524830233471897897ull},
		{17875537, 1664465123229227681ull},{17876251, 2403629034158699383ull},{17878099,  712042873447678601ull},{17881309,  903201483228046399ull},{17883809, 8490501007003379807ull},
		{17888639, 2105629043627382127ull},{17890667, 5735679228382833647ull},{17891113, 2876396552211694177ull},{17891147, 3093927243161087143ull},{17891807, 2664231706296909673ull},
		{17892221, 1978238729073140969ull},{17893643, 5419863053469262121ull},{17895089, 1533338567834601023ull},{17895103, 5160136017722307967ull},{17896811, 7897800523391520407ull},
		{17897839, 3374272067429879321ull},{17898479,  749593692866362247ull},{17908691, 1327377814615443967ull},{17909531, 1685626660799023367ull},{17909929, 2057175048146017463ull},
		{17915189, 1502198297363289137ull},{17916007, 2548941981249916039ull},{17918779, 6250235537067221041ull},{17920313,  982796297387492737ull},{17920367,  616895704215165097ull},
		{17920663, 6556922280592103159ull},{17921363, 3344080983855421297ull},{17923667, 1310662441188923177ull},{17926043, 1112470599742175431ull},{17929829, 7790880448537146551ull},
		{17930141, 5488202995825707401ull},{17930449, 6804159664440658391ull},{17931607, 5283700822626454303ull},{17931803, 3700358431737961943ull},{17931853, 8959646386932483937ull},
		{17933089, 3094872653106269879ull},{17933821, 5805307060406953537ull},{17934187, 2146159967654259929ull},{17936447, 7823722696080620041ull},{17937497, 3514677418927413071ull},
		{17938157, 1231292317361854199ull},{17940749, 5059333168351045729ull},{17943293, 5993577327478456249ull},{17943533, 1388115696739083727ull},{17948417, 4732957426757595103ull},
		{17948851, 2394620018782987729ull},{17951159, 2058879430641527057ull},{17951729, 7474004052629873911ull},{17952611, 2516644147971176591ull},{17952917, 1864851701193217279ull},
		{17953073, 1007670704243820809ull},{17954081, 1301079621918711617ull},{17954891,  964648425701599319ull},{17957509, 6840963133271524111ull},{17957683, 3978152863479088207ull},
		{17959793, 4669330482742391657ull},{17960119, 6752560053788375471ull},{17960783, 1678745601392391911ull},{17960977, 1228182543339675761ull},{17961421, 1020658709440637711ull},
		{17961443, 2797636755196101839ull},{17961721, 9063572628875007847ull},{17963123,  815678591888801513ull},{17963287, 1686003604554390599ull},{17964007,  778257809046093137ull},
		{17964613, 3754431528196358599ull},{17964931, 6361663250068077871ull},{17965289, 4428736923932099951ull},{17970331, 8117285082720974767ull},{17970349, 3864521381056789633ull},
		{17970529, 1426577669200287553ull},{17970709, 4518264433954012039ull},{17971237, 5471173681460147033ull},{17971697, 3699375676644287231ull},{17972971, 1359645139272461927ull},
		{17974861, 1691952521705558759ull},{17975981, 5648969327860083959ull},{17977579, 3748493120761985231ull},{17977957,  627232853285495647ull},{17978243, 1736248171119512263ull},
		{17979613, 2620854359698631863ull},{17982973, 5845737077781621887ull},{17984053, 2256835105241380121ull},{17984179, 1036884414685348673ull},{17984509, 8476679719127171777ull},
		{17985281, 1810544315152389041ull},{17986021, 2437445911911384503ull},{17986523, 4599843559465033063ull},{17986949, 1748229558785170831ull},{17987491, 2032912769179543457ull},
		{17987941,  630951207087221641ull},{17990671, 4331370829474908449ull},{17990983, 3525084813671441983ull},{17991541,  815037019285024057ull},{17992063, 2834821057516610377ull},
		{17994631, 3640971086950719167ull},{17994709, 6165252858770522527ull},{17994841, 3699354237817492591ull},{17997229, 2985492437257596839ull},{17997613, 3058804716917440529ull},
		{17998921, 1382684188503730727ull},{17999239, 1500972887730833927ull},{18004897, 4759900927558675889ull},{18005509, 9172509111918008737ull},{18005881, 7736227950495029071ull},
		{18006683, 3063561940694759783ull},{18011219, 2390466914947953551ull},{18011557, 7366817463725456441ull},{18011771, 2226790528673362367ull},{18011809, 1503751987321572641ull},
		{18012307,  220881355348927807ull},{18013507,  940523625874376009ull},{18013889, 1510301576718077839ull},{18014389, 7538190749320752511ull},{18014839,  277820521370888999ull},
		{18015551, 2340902482533374311ull},{18015953,  190079508451330231ull},{18016001, 4348492870379062169ull},{18016049, 3626126767636896889ull},{18016189, 1208034199034769401ull},
		{18016223, 8227372396468321529ull},{18016267, 2470104015463296281ull},{18016837, 1161245128754030233ull},{18017513, 3253408194517706441ull},{18018349, 1101875138253273577ull},
		{18018467,  195863445150291847ull},{18019637,  253771669953795991ull},{18020077, 3094501034482670753ull},{18021217,  178202077941327599ull},{18021617, 1095316961454549121ull},
		{18022343, 1902723646205275703ull},{18022379, 6182940253225886719ull},{18022517,  669237263128794313ull},{18023483, 2819237485165040417ull},{18023543,  664891496674705343ull},
		{18024137, 3423810082502902159ull},{18024623, 1358245988590673033ull},{18024833, 2413382318575227841ull},{18025171, 4953558993717617273ull},{18025493, 4046226592751303561ull},
		{18026167, 7301081045496468719ull},{18026759,  347459848092058087ull},{18027403, 1390530689850382279ull},{18027613,  553667433157303321ull},{18027643,  804282151680440903ull},
		{18028457, 3410205068438600407ull},{18029237,  384382401233265737ull},{18030157, 1814163598947018881ull},{18030811,  218703143967921281ull},{18031129,  336284434526159191ull},
		{18031199, 1773675444104420711ull},{18031807,  449829212253045649ull},{18032213,  809991539314593359ull},{18032737, 5563212097056992351ull},{18033047, 4242124268372681071ull},
		{18033313,  892754292286804439ull},{18033559,  578363464774452089ull},{18033749,  649107982030132321ull},{18034129, 8411740586508222721ull},{18034153,  228575622232707121ull},
		{18034193, 1568046814620032273ull},{18034487, 6563374615048802527ull},{18034867,  585223207509348217ull},{18035107, 2402182266587944111ull},{18035789, 6855462942253804601ull},
		{18035819,  430643152681188121ull},{18036311, 1309864431794423207ull},{18036407,  634094963019467681ull},{18036539, 2764330705944299369ull},{18036751,  969575221888062233ull},
		{18037189, 4391462561032616009ull},{18037231, 6998278073234944297ull},{18038123,  234324071638369567ull},{18038299, 1034209390921824937ull},{18038689,  285195009201221039ull},
		{18038843, 8465335113608243977ull},{18039719,  939317313181647817ull},{18039821, 8483872917922747399ull},{18040051,  550135978585350383ull},{18040597,  369386639326138207ull},
		{18040691, 7437117407042207281ull},{18040751, 3007057566367235767ull},{18040969, 1440061135634909441ull},{18041173, 1801135275958123841ull},{18041251,  212667050862844247ull},
		{18041371, 1607487587321961431ull},{18041449,  331381074274740977ull},{18041531, 8155698232304113193ull},{18042161,  719516832900729697ull},{18042293, 4045697812716293801ull},
		{18042361, 1195388362488578681ull},{18042547, 2264048456405883071ull},{18042809, 1041025085117227487ull},{18042887,  820583244199534793ull},{18043657, 6698885379723509521ull},
		{18044771,  217751151498027799ull},{18045019,  886346438516147753ull},{18045553,  684060296358930769ull},{18045983, 1727756785397843353ull},{18046849,  344383295054587687ull},
		{18046937, 1582723876470302791ull},{18046957, 6265230556365760207ull},{18047851, 2794465755922650769ull},{18048769, 1272884020935274439ull},{18048869, 2687244032076582031ull},
		{18048967, 2882596357369915999ull},{18049021,  151593119343580553ull},{18049079,  146863140298552751ull},{18049279,  302279348097426983ull},{18049607, 1027073160738358417ull},
		{18049729,  170705412915485111ull},{18049783, 2193137766783224489ull},{18050027,  249763397783639767ull},{18050477, 9034549389428481287ull},{18051151, 8270716773764582561ull},
		{18052591,  213534006096423721ull},{18053131,  145637662655615297ull},{18053641, 3574744060404862439ull},{18053681, 3523102587334514273ull},{18053713, 1274525953140893983ull},
		{18053771,  179298408456414497ull},{18054019, 5649369575053961591ull},{18054229,  466796606374323649ull},{18054577, 2331242743843392751ull},{18055153,  964029306347175647ull},
		{18055781, 1131377662856950169ull},{18056021, 3306352774543498007ull},{18056341, 2199830028736195519ull},{18056959,  575696006576981399ull},{18057101, 1032151692346949719ull},
		{18057253,  700043530529500567ull},{18057257,  317329196933100529ull},{18057751, 3201031337245812953ull},{18057761, 1268134802832477257ull},{18058127,  975859961953336007ull},
		{18059207, 3648110332144819729ull},{18059857,  428772495755334047ull},{18060187,  239976875255863241ull},{18060257,  306692189293676009ull},{18061111,  161771569920155527ull},
		{18061223,  993238009496376041ull},{18061597,  531092340787785431ull},{18061721,  397467971850349511ull},{18061889,  170575814893989689ull},{18062197,  459009642330749087ull},
		{18062221, 4762721205492398143ull},{18063697,  155760394292528407ull},{18064261,  869279864055005729ull},{18064609,  902086647190626031ull},{18064663, 4326753552702418993ull},
		{18065473, 1016638870230715063ull},{18065573, 4980322140187853369ull},{18066827, 3041030515267231999ull},{18067013, 2965222136820402343ull},{18068201,  732115975825870919ull},
		{18068377, 2654786510720728759ull},{18068873,  382620704518269719ull},{18069367,  935330914958678617ull},{18069533,  439156261511080703ull},{18069851,  895513289517439447ull},
		{18070309,  169411595004322703ull},{18070981,  468402609294531217ull},{18071509,  619224414211506937ull},{18071783,  269279113553092543ull},{18071891, 2051286244830893879ull},
		{18072773, 5242009072298288161ull},{18072793, 8819126928139257361ull},{18073207,  303650943296322233ull},{18073613, 2204719604073948097ull},{18073843, 2123288934790210753ull},
		{18074057,  170868633716270417ull},{18074129,  231723590213766793ull},{18074531, 5617874574099630319ull},{18074869, 1313575677565859287ull},{18075227, 5966010148617767489ull},
		{18075413, 1072730263421826137ull},{18075469,  157453782618412799ull},{18075829, 3540387457043046913ull},{18075929, 1427015700350286601ull},{18076417,  350296688447252401ull},
		{18076753, 9106402842275874799ull},{18076987, 6513441085781826191ull},{18077041,  954866887065157361ull},{18077077,  156911835368516561ull},{18077509, 2858210469415838191ull},
		{18077539, 7409642155457441729ull},{18077849, 1727714706371758799ull},{18078019,  208075572945254543ull},{18078097,  809063907179527457ull},{18078773,  209054887364316841ull},
		{18079447, 2991325265195645407ull},{18079757,  469481662092691673ull},{18080059,  156437228555447297ull},{18080497,  265091087382571111ull},{18080773,  863973615071231263ull},
		{18081527,  629299790246739247ull},{18081593, 1569037754951646233ull},{18082327,  294136552344507503ull},{18082601, 7772927443359071831ull},{18083011,  764051439321188063ull},
		{18083851, 1093556287922395681ull},{18083941, 5005994961296695481ull},{18084103, 5396859190608684961ull},{18084511,  280218395260026287ull},{18085163, 3958373172235987537ull},
		{18085259, 2421951227499722447ull},{18085597, 1652097288619224871ull},{18085877, 5970055030215365329ull},{18086069, 5194261471992541321ull},{18086473,  175396807623692647ull},
		{18086837,  428007377950999711ull},{18086897, 2086916000630989369ull},{18086941,  507585516102880289ull},{18087053, 6736325484534062567ull},{18087241,  229349757904405031ull},
		{18088823,  348983282328464657ull},{18088997, 3817682373416327543ull},{18089221, 7717318284654310807ull},{18090749, 4400781849969539081ull},{18090829,  254929304698201369ull},
		{18091243, 2178950725735373921ull},{18091793, 5360440248670130159ull},{18091939, 2955370698546160567ull},{18092299,  189845745823406633ull},{18092729,  786834525027893201ull},
		{18093421, 2528677625598381841ull},{18094199, 7475161236393989161ull},{18094523,  500414590814444351ull},{18094829, 1692354721386162271ull},{18095381, 1488542967045098257ull},
		{18095489, 3474495571049451089ull},{18096017, 2381707920189331807ull},{18096829, 4298659432822913113ull},{18097117,  252582943145637983ull},{18097691, 1452486143800069319ull},
		{18097777, 3811727973559918463ull},{18098173, 4646682542258342239ull},{18098467, 2766600574338603679ull},{18098719,  498347184635824031ull},{18099317,  197887977417451367ull},
		{18099593,  692527271711510801ull},{18100351,  986858833532297393ull},{18100487, 5559545560067206831ull},{18101351, 5892698227386194249ull},{18101723,  330663381619973953ull},
		{18102113, 4108304515751346191ull},{18102169,  658093059110448113ull},{18102551,  278135947855081399ull},{18102827, 3614226096429255457ull},{18103049, 5669365061064864839ull},
		{18103699, 3418868997915619559ull},{18103961,  300843536605165847ull},{18104221,  382634038236483721ull},{18104369,  833152711474330081ull},{18105079, 2294178426436943801ull},
		{18105331,  200171358090730927ull},{18105929,  233146044693157391ull},{18106423, 9010404214242243343ull},{18106733,  240725289847112183ull},{18108067,  173634447567858457ull},
		{18108109, 2039429751651682297ull},{18108749, 8907523168165945903ull},{18109099, 1421698886573962441ull},{18109373, 1148928461554787041ull},{18109783,  219645083823801679ull},
		{18110101, 6356333444674501297ull},{18110329,  636409319403626969ull},{18110633, 5137754947683470257ull},{18110737, 6949931723478004673ull},{18112219, 1038596187216405943ull},
		{18113047, 2636979483956432687ull},{18113423, 5711003588895540041ull},{18113437, 7237656685324752551ull},{18113647, 3664424537447090401ull},{18113819,  644487044676701329ull},
		{18114329,  180937307436358751ull},{18114851,  191730898327800319ull},{18115219, 6967968772129631927ull},{18116167, 3284006162755109681ull},{18116473,  295911982481558263ull},
		{18116653,  905674446545077337ull},{18116933,  182298573362242793ull},{18117343,  179042324916860929ull},{18117611,  276403239447018521ull},{18117727,  265355157611625017ull},
		{18118589, 2100547192247488343ull},{18118813,  297889740791115089ull},{18118901, 2006169583836711151ull},{18118907, 1394872940423976479ull},{18119047, 3274845436157029897ull},
		{18119107,  149695768675358137ull},{18120203,  931416251824709527ull},{18120373,  326540082364905463ull},{18120517, 2475413327350678009ull},{18120733,  190253658549352231ull},
		{18121843, 6359751903252262831ull},{18122623, 4096620322163539519ull},{18123043, 4163005457310656519ull},{18123817, 7035605257132928143ull},{18123877,  841608666156857441ull},
		{18124969,  250169084767618031ull},{18125017,  185850489228528127ull},{18125089, 2115940694316175591ull},{18125573,  156569053638942983ull},{18126587,  160497608613885097ull},
		{18127093, 2729552475111634177ull},{18127631,  202972162459180159ull},{18129077,  826185140698051193ull},{18129103, 7756965693917461751ull},{18129487, 5710844170141724833ull},
		{18129953, 1026702811985035831ull},{18130379,  844753854261726401ull},{18132473,  508082638582175071ull},{18132703,  984788573937426871ull},{18133009, 4179248003537482327ull},
		{18134377,  353995670830449383ull},{18134561, 1644404173956643577ull},{18135521,  193091585683392047ull},{18135841, 1592191104201793057ull},{18136043,  164769434262411527ull},
		{18136087,  588583452577383391ull},{18136757,  942155756229059431ull},{18136897,  498022844181805609ull},{18137389, 2124101979953718983ull},{18137543, 1712231773162793983ull},
		{18137857,  306824264431376327ull},{18138451, 7582282194251423767ull},{18138649,  172912609859686919ull},{18138871, 3502704858995300303ull},{18138943,  246436192567308041ull},
		{18139543, 7285407584197933721ull},{18139549, 8194085954986376671ull},{18139601,  151382510178993257ull},{18139711, 2628003804944996063ull},{18139787,  426501575901789887ull},
		{18139909, 1046433785458137217ull},{18140159, 3453473050646226497ull},{18140273, 5713537045599308921ull},{18140971,  483292855669549327ull},{18141997, 3815540980096758137ull},
		{18142309, 8994056844398885153ull},{18144433, 7624512815589709681ull},{18144851,  910777532920154969ull},{18145009,  188541992445693137ull},{18146059,  510420446392176527ull},
		{18146173, 1077844070144357809ull},{18146389, 1542355791176058953ull},{18147403, 2286897497466736321ull},{18147457, 1717929434260216087ull},{18147923, 1863286297415992049ull},
		{18148337, 4530369323750308271ull},{18148799, 3297933472724493719ull},{18149039,  880845054646901471ull},{18149707, 1185286443100915081ull},{18149843,  916379742324037193ull},
		{18150389, 3822485160260893033ull},{18151093,  276821161702590647ull},{18151249, 3813458507193178439ull},{18151433,  504627225601545217ull},{18152203, 2250101091316519247ull},
		{18152227,  177869433900604409ull},{18152297,  624789460689330721ull},{18152833,  495113827785503497ull},{18152839,  885462025654928081ull},{18152861,  745935741309920879ull},
		{18153169, 3605232236719796351ull},{18153271, 3877198473695927321ull},{18154307,  849875488878394039ull},{18154781,  571926881349174353ull},{18154943,  155169642064458841ull},
		{18155341, 2105616473697176423ull},{18155399, 6043531412317122401ull},{18155939, 3895014398342628833ull},{18156067, 2786585146312304017ull},{18156139, 5200014469418315143ull},
		{18156353, 3580508484628476463ull},{18156517,  589464156813894023ull},{18156659,  606575457511111639ull},{18156767,  203157325643180927ull},{18157187,  397214103754615871ull},
		{18157303,  576095881253912743ull},{18157859, 6212058775368239569ull},{18157943,  159009663500900609ull},{18158879, 3046826399782247647ull},{18160739, 1834647890429700191ull},
		{18160949,  593999785117695553ull},{18161167,  515672149352372599ull},{18162563,  325095769929189751ull},{18162769,  803974127577401471ull},{18162913, 1357411268881729807ull},
		{18163111,  184841740940977937ull},{18163459, 2856141027727809919ull},{18163823, 2023000940646103897ull},{18164953, 6200977504829196521ull},{18164999,  209733008010133879ull},
		{18165449,  581632703556685577ull},{18165757, 8551912773780986681ull},{18166429, 5446844587056314327ull},{18166543, 2266665690147190831ull},{18166787,  284384707970416967ull},
		{18167431,  472166489464865399ull},{18167533,  454009591465948559ull},{18167563,  708026462172382921ull},{18168611,  249407869166136743ull},{18169201,  166259287297693223ull},
		{18169861,  443632828183509721ull},{18170197,  551075689667191967ull},{18170377,  495087521203707449ull},{18171443,  158909252898758617ull},{18171731, 1761088143471303481ull},
		{18171877,  193406421160001207ull},{18171949,  959021713174709729ull},{18171971,  212582186997306377ull},{18172283,  306340199670707177ull},{18173159, 2710192223805113503ull},
		{18173341, 2655893467840057759ull},{18173773, 5208297326830612199ull},{18173801, 2566037605552291639ull},{18173803, 7609123229331745631ull},{18174001,  280538571090706153ull},
		{18174461,  373169859737261921ull},{18174613, 3696057688567765441ull},{18175307, 4606467234303035863ull},{18175433, 1671941933015884697ull},{18175603,  358104212205598393ull},
		{18176129,  297748851152334433ull},{18176479, 1727768194505086439ull},{18176971,  580938966948809543ull},{18177721,  288661255622268247ull},{18178481, 1412900492698726673ull},
		{18178747, 1051870922187653719ull},{18179011, 4129359287631769799ull},{18179107, 6950769019465931417ull},{18179251,  209784776455622567ull},{18179339, 1071223273576351913ull},
		{18179743, 5450404147513203847ull},{18179747,  278846268444797423ull},{18179881,  579166105626521671ull},{18179963,  422004445218799519ull},{18180161, 6611758316647537577ull},
		{18180523, 3866032370946584039ull},{18180619, 3511515211683351449ull},{18180677, 2788648282376867167ull},{18180931, 6090731622611663591ull},{18181103,  639713485080233719ull},
		{18181591,  493845656085060953ull},{18181643, 6505947238121659543ull},{18182093,  148609684310517929ull},{18182149, 1421216450235542759ull},{18182257, 1446565977045436033ull},
		{18182933, 6599802010049618921ull},{18183043, 4557063360658169617ull},{18183713, 3243799555344182671ull},{18183853,  395470449031694161ull},{18184009, 5686173517548316087ull},
		{18184043,  373695005994685607ull},{18184417,  231657393871958423ull},{18185021,  500385408994477393ull},{18186167, 2137325682797422129ull},{18186403, 3705306186874921793ull},
		{18186653, 4636504888324534583ull},{18186919, 4487805898789947119ull},{18187153, 6983373698391292919ull},{18187291,  370733891538789481ull},{18187451,  556367250482015719ull},
		{18188207, 4575824555835552737ull},{18188347, 2947808031323356391ull},{18188897,  279655125154038481ull},{18189211,  682770116581096463ull},{18189239,  721353005779623391ull},
		{18189683,  284497605842290153ull},{18190547, 2366428611869035327ull},{18190633,  611552330470076887ull},{18190883, 3401415680513815457ull},{18191449,  335577525522872177ull},
		{18191869,  430855402541381393ull},{18192523, 3295002558186922673ull},{18193271,  719309171397378511ull},{18194773, 2206473614203878367ull},{18195173, 2198996002033853881ull},
		{18195563,  870871242175451887ull},{18195643, 6182575437692950537ull},{18196777,  244421485308650417ull},{18196807,  214838963113848809ull},{18198001, 3797160104673324583ull},
		{18198307, 3261313241635456673ull},{18198407,  908961388000467377ull},{18198611,  147117672508277161ull},{18198757, 1291057922497655201ull},{18198839, 7702578482357557921ull},
		{18199123,  627938298322402391ull},{18199243, 5715394251020166799ull},{18199319, 1751859472378580993ull},{18200447, 2241185363083369897ull},{18200453,  964234051444955183ull},
		{18200729,  226278029714621743ull},{18201653,  453572672024917831ull},{18201751,  177416839093993703ull},{18201977, 7556583206839917529ull},{18204647,  384711185932909679ull},
		{18202727,  324976325860139897ull},{18202763, 3619808293690777481ull},{18203599,  255716891729650409ull},{18204551, 1240325425602719639ull},{18206063, 6154427524543057841ull},
		{18204833, 5064367179117501257ull},{18205427,  171635048851402607ull},{18205567,  498049680871445783ull},{18205849,  420551620709984063ull},{18208871, 4825503957031353559ull},
		{18206707,  194980615976282599ull},{18206767,  598824763799582977ull},{18207751, 5450962935976347551ull},{18208459,  732071086083278041ull},{18210487, 7904260220388114433ull},
		{18209309,  421921108806185609ull},{18209663, 8914936054468409521ull},{18209677,  502857357742033897ull},{18210287, 3828673568316018007ull},{18212209, 8790088711977627727ull},
		{18210749, 2639089913010052231ull},{18210839, 1252897572756897161ull},{18210989, 5449720962043005161ull},{18211819, 1475550183670841617ull},{18215221,  326578760775994607ull},
		{18213947, 1836680918690277743ull},{18213967, 5459789331634736321ull},{18214393,  195882229760802737ull},{18214969,  908643053573127737ull},{18217841,  796311598662086929ull},
		{18216113, 2407520616714053377ull},{18216281, 1012316473329291833ull},{18217097, 7081648448968956167ull},{18217231, 7391482353834144271ull},{18219679, 7151731110031749079ull},
		{18218107, 1024998043469714399ull},{18218329, 4062483484914991183ull},{18218821, 1574151660610168777ull},{18219193,  465003359815917473ull},{18222047,  183179527238201713ull},
		{18220121, 1250742979702200079ull},{18220193, 6884674196094390961ull},{18220427, 7661835938013915577ull},{18221563,  355351420754789057ull},{18223081,  271683140874255079ull},
		{18222431, 1345721149104357527ull},{18222473, 3615247421609496839ull},{18222541, 4684088799094989457ull},{18222679, 8986103483467189951ull},{18226207, 2704015245587660479ull},
		{18223157, 5504989872766853537ull},{18224113,  337598450890815041ull},{18225239, 1774628704161652543ull},{18225947,  170795563856396191ull},{18227941,  378986137162265657ull},
		{18226259, 2649773119432899799ull},{18226573,  911655289398967943ull},{18226661,  445670037826519969ull},{18227501, 7493941152443672183ull},{18230129,  797093455655673119ull},
		{18228671, 9040364929018544423ull},{18229217, 6172674414859643609ull},{18229609,  226169112606204431ull},{18229957, 6222489105657968551ull},{18232183,  944222602441112623ull},
		{18230347,  189233265048197969ull},{18230479,  604328392674667081ull},{18231443,  147016546516653391ull},{18232063, 8986734322535829143ull},{18233221,  445148471262696793ull},
		{18232589, 4142114913805405951ull},{18232657,  178737507169092407ull},{18232861, 7113129966272258737ull},{18232873,  910196727414821809ull},{18235691,  164802005523725623ull},
		{18233599,  219626590199244689ull},{18233653, 3827298988308429593ull},{18234467, 5822485231311541927ull},{18235501,  392569969513211351ull},{18236741,  196115251171071497ull},
		{18235759,  634470290505529649ull},{18235829,  209128182032467463ull},{18236173, 1072782161927740559ull},{18236201,  790424566760147359ull},{18240023, 1145096636605819193ull},
		{18236807,  466544298281538161ull},{18237683,  150980093785031393ull},{18238039, 8257131768449278303ull},{18238439,  344896464741121177ull},{18241291, 6931896967949517449ull},
		{18240689,  575543924177829127ull},{18240707, 2551774584244094753ull},{18240791,  502038045619799519ull},{18240793, 1170545082309195503ull},{18244049, 6069262720224940337ull},
		{18242249, 5828183487410879561ull},{18242837,  329863222762232767ull},{18243541, 1145311999667281321ull},{18243691, 4730620580709702407ull},{18245393, 1756929621995499241ull},
		{18244397,  192953761876994009ull},{18244609,  693347914896424681ull},{18244871,  185263315342358527ull},{18244969,  313851947884446847ull},{18246359, 1724545012534814191ull},
		{18245431, 2427947115805275401ull},{18245483, 2236899860371720217ull},{18245489, 3135301377567517289ull},{18245911, 5185505342138518889ull},{18248129, 1889170052270492231ull},
		{18246493,  661645029293447761ull},{18246617,  282971109500942873ull},{18247231,  270517615508384401ull},{18247321,  159147066856219279ull},{18250627,  291592687904150233ull},
		{18248161, 7709695907107275271ull},{18248557, 3186743502444660737ull},{18248621, 4251433917700860097ull},{18249817,  493224469282714441ull},{18251957,  254280057031881247ull},
		{18250751,  254345153322583249ull},{18250927,  924384987729884311ull},{18251573, 8971192775529465359ull},{18251921,  818573016843409937ull},{18253201, 2107029553327261049ull},
		{18252413, 6276507976472527231ull},{18252439, 1382377176451739687ull},{18252473,  156153666341913271ull},{18252721,  955287304650810977ull},{18255473,  319847903150459953ull},
		{18254153,  967283619169722713ull},{18254179,  219243245853140479ull},{18254261,  426226079762262881ull},{18254713, 5193684555003623513ull},{18257809,  308089528042217839ull},
		{18256003, 5286812608446677431ull},{18256541, 2553532129640010329ull},{18257039,  448060853576070257ull},{18257563,  321458459567713471ull},{18261493,  531522059923681919ull},
		{18259859, 2803350460979415799ull},{18260069,  331082250694020607ull},{18261137,  154050640123958233ull},{18261491, 3702574820038690927ull},{18262639, 1209709784108600393ull},
		{18261953, 2806113676062110233ull},{18262523,  684985867125297457ull},{18262577, 2194723316765298271ull},{18262597,  320968254367629577ull},{18265301, 1346964077869385711ull},
		{18263249,  232092702284631449ull},{18263251,  333277109337857999ull},{18263629, 1713796095354485687ull},{18264707, 1832987592900191017ull},{18267541, 1496402839560725617ull},
		{18265801, 4423192174067168921ull},{18266051, 1547492141360310889ull},{18267031,  567432888774070073ull},{18267047, 5106988365977004847ull},{18268391,  305702301712820839ull},
		{18267679,  174693156677091359ull},{18267827,  622339054125734633ull},{18267869, 1029565792938412967ull},{18268213, 1567498750360311343ull},{18270283, 3267912895027409807ull},
		{18268609,  849946831856870551ull},{18268849, 3589933220064654217ull},{18269851,  187189076554887889ull},{18269941,  684132024494418671ull},{18272459,  185218879178598929ull},
		{18270697, 4883150130560297143ull},{18271153,  847597287242915641ull},{18271373, 2678811554504263327ull},{18271699, 1035077953363428593ull},{18273979,  163029569082413321ull},
		{18272867, 4292910235570102649ull},{18273131,  209741654531247401ull},{18273377, 2209176452708378567ull},{18273919, 2540283736047175687ull},{18275233,  615695688832633991ull},
		{18274357, 5250220205935681729ull},{18274603, 1651356375517532207ull},{18274769,  683353857184539559ull},{18274793,  358189838547272569ull},{18276473, 2368605654265044937ull},
		{18275489,  358780015246475209ull},{18275633,  714810154491785543ull},{18275671,  777591913390936481ull},{18276131,  407054061248703889ull},{18279707, 2260875022023462961ull},
		{18277121,  986369995008748303ull},{18277283,  228176384967746561ull},{18278233, 4556501041742873849ull},{18278863, 2454505365996506623ull},{18280693, 4432908001376646401ull},
		{18279997,  154021947782950943ull},{18280187, 4079581038301390247ull},{18280189,  878473513058879207ull},{18280319,  357247190862453119ull},{18282479,  211056575094586799ull},
		{18280897, 2014965098614430153ull},{18281111,  313018661846486831ull},{18281147, 1046495746119219161ull},{18282149, 1158393037316521873ull},{18284093, 8438246621794815367ull},
		{18282533, 2542230761454948857ull},{18282749,  252921162085242593ull},{18282833, 6228271224778947713ull},{18283879,  202167909800447377ull},{18287299,  699415163018787223ull},
		{18284107, 6555765084751538351ull},{18284437, 4596424749553843079ull},{18284887, 3864332598578495761ull},{18287099, 7075693236187789471ull},{18290131, 3505439438466316217ull},
		{18287447,  791184749579030377ull},{18287623, 7137233656547846431ull},{18288509, 1903735137621720479ull},{18289937,  219531327119618807ull},{18291743, 2873872397818760873ull},
		{18290411,  684228304469438089ull},{18290659,  563712281110395383ull},{18290927, 3424153274995352719ull},{18291583, 3062407520782516471ull},{18294047, 4139587137772912577ull},
		{18291773, 2657423207973350447ull},{18292039,  251983731817092089ull},{18293101, 6998952403324571639ull},{18293657,  751539795410988359ull},{18295987, 6428839976520012353ull},
		{18294259,  602428265212755713ull},{18294721, 7589034719689448759ull},{18295003,  530722364249779991ull},{18295183,  155518178941858441ull},{18296737, 1282191130922981113ull},
		{18297571, 3288951767406279151ull},{18299003, 4522165993393804423ull},{18299447, 1506874563020357233ull},{16219289, 7464112771264190017ull},{16230061, 7074289273913679991ull},
		{0,0ull}
	};

	/* Testcases with factors < 2^64 = 18446744073709551616: */
	static const struct testFac fac64[] =
	{
		{17922587,10500365774503436201ull},{17924587,15641779111884102847ull},{17927311,12915704346519849697ull},{17938027, 9287694977263609817ull},{17941211,10016045939325808271ull},
		{17942927,12928465186953089863ull},{17963653,10943084366908922353ull},{17964227,12431505097071887473ull},{17967659,15718414880628053417ull},{17971007,17130259977555383783ull},
		{17973443,17017539148249318519ull},{17981947,17331582637129315543ull},{17984683,17075071760459211223ull},{17988793,10651829289881962249ull},{17990153,13875348797185757897ull},
		{17990213, 9612328233936311881ull},{17996261,10983089768339907263ull},{18016303,11011193468080521223ull},{18019909,12625693644608762351ull},{18024151,10754411617748801759ull},
		{18024311,12013002667638145399ull},{18025781,12825068851465148441ull},{18032587,17170312062251360359ull},{18033511,12031568180050170481ull},{18037577,10566050601376248193ull},
		{18040663,14859011986541991007ull},{18047423,14989510186280279071ull},{18048313,16214445839716767361ull},{18051923,10598954534434896529ull},{18055627,14057592431325128593ull},
		{18056047,16560715604598816241ull},{18059039,12811182984027587681ull},{18064777,15539690406724276993ull},{18066073, 9985641878418216991ull},{18067711,11449566963261644983ull},
		{18075257,13764994941016683383ull},{18075791,16672552177382226647ull},{18077821,11623423906484739527ull},{18079043,17169614195768327983ull},{18079123,17990551511959203257ull},
		{18080273,12155150476665080887ull},{18090001, 9789648295780602191ull},{18091883,10985008068068673007ull},{18095521,12658180920879996367ull},{18107017,15044832325095723511ull},
		{18107351,14046232068642146993ull},{18119851,11179467653397253391ull},{18120799,11418381227787294353ull},{18121517, 9600311723751200999ull},{18121799,10641782461606146511ull},
		{18122101, 9238369956591864047ull},{18125911,14284214035586927489ull},{18130529,16132196293595357033ull},{18131303,12132533847075902297ull},{18133517,13711702151742952271ull},
		{18136529,13553802287542899791ull},{18138227, 9814223728786685249ull},{18138437,11664104294072220647ull},{18142753,11841941210452963607ull},{18143239,13495717684024316977ull},
		{18144733,11453374630895585671ull},{18146147,12409236261334108927ull},{18146309,13941936454603083961ull},{18151307,16695225546321153647ull},{18157823,15274150651947770057ull},
		{18159851,10670964685324409993ull},{18162883,13984615577478771737ull},{18166397,16478002069230680743ull},{18170359,11555285165538057473ull},{18173191,13017176322370888999ull},
		{18174743,10899353333633834737ull},{18174829,13082852309834271431ull},{18175387, 9469759966159211119ull},{18177011,16562044284339342287ull},{18181343,12857396115642922399ull},
		{18184013, 9771683082091843559ull},{18190691,13254207919455131647ull},{18190903,15437851890296633887ull},{18196709,16287562569554793953ull},{18196993, 9275136914779640273ull},
		{18198841,11025186727945055441ull},{18199583, 9854261882174597857ull},{18200849,15285818081668067519ull},{18202049,11180960576914148191ull},{18211309,14428852738137829687ull},
		{18211993,11752043400810302297ull},{18215051,15463714506600122897ull},{18220127,12907674385704768017ull},{18220999,11394039312173819999ull},{18224827,10366044942826856209ull},
		{18228937,17973037128357378841ull},{18231331,18328139888296813361ull},{18232931,10003907358728522969ull},{18235879,15548404807307836393ull},{18242963,16134040372770842257ull},
		{18245207,17112902589623655359ull},{18253589,13909553524167145703ull},{18255499,11109231768403310369ull},{18257399,10738982321410344871ull},{18258283,15374699987470193449ull},
		{18263647,13762137093959242159ull},{18266777,14732963741405224223ull},{18268189, 9989690110846448329ull},{18269219,16284666399547421921ull},{18275251,13620718892071208681ull},
		{18275263,11262843094267061911ull},{18275651,13846368038170428023ull},{18277607,10504362932070630521ull},{18280159,12772485980427473983ull},{18281927,12631296696930554257ull},
		{18286687,17931410955470787601ull},{18292013,12702828058463407919ull},{18296671,14627091909328461551ull},{33402073,13932143274652230937ull},{17719931,12646230872610358319ull},
		{17719993,17458131644489262929ull},{17734169,11268951794162923127ull},{17742793,16341053115349924417ull},{17771027,12007015401209325023ull},{17773141,15918852021480399527ull},
		{17774641, 9341752468258236431ull},{17775269,10013306797089936121ull},{17811401,16160582285842380959ull},{17812637,15811238096354839487ull},{17815169,12298642243571509687ull},
		{17820757,12597566906366358017ull},{17829943,18335410068788651849ull},{17831813,15899391255100053209ull},{17851397,16860276014915517791ull},{17853761,16072139673628398673ull},
		{17870731,12143859436446628321ull},{17889841,16375574063043584023ull},{17902987,16086636940765296889ull},{17905253,17558635633759926961ull},{17911027,14646809963723851673ull},
		{17913557,17895543665490669193ull},{17915981,16726174377018723457ull},{16100419,14461463666320960511ull},{16104013,13440101789246121791ull},{16105279,10991970915886596511ull},
		{16118143,11337011726859890543ull},{16120229,16582660415820077063ull},{16121711,15998585226804618071ull},{16131469,14271788618292432257ull},{16132931,10760230434599829143ull},
		{16132969,10730782077481732817ull},{16135981,14331950710727986663ull},{16136507,10961703930971884903ull},{16137271,15296117896318004719ull},{16138747,17040351390832159609ull},
		{16140287,14702865045484912391ull},{16140419,11533383163345547927ull},{16141121,12125342584232876329ull},{16141553,15810596930722833329ull},{16142201,16759807228908186473ull},
		{16142437,14680112878087367873ull},{16143041,10214864560585659127ull},{16152289,18111383456669963153ull},{16152901,11554056516029888111ull},{16153601,14102697559575163223ull},
		{16157303,10951036977490542007ull},{16161121,12068660979042357751ull},{16161923,13936597432397301953ull},{16167427,17741531859834789367ull},{16169077,13038897762986995439ull},
		{16172587,12923341570853700409ull},{16173133,12468959361471384367ull},{16173691,12146036035731849143ull},{16181381,17389980150980900167ull},{16181761,13632984658968284321ull},
		{16184771,12579003037333710127ull},{16185919,18017349281491722863ull},{16186981,16300711072026110783ull},{16187707,12837482707778126729ull},{16191941,14435333013673979377ull},
		{16194601,10362937015011578407ull},{16199899,14943059862662596097ull},{16201201,13083808953378894799ull},{16203571,10961371406875908433ull},{16203631,15948970413144688847ull},
		{16203851,11851486348839027743ull},{16213993,13617531092155742599ull},{16100213,16092742349020987151ull},{16100297,10748446445825061761ull},{16098127,13581925937697282097ull},
		{16098883,17001973141862761433ull},{16098931,18338048883043493801ull},{16099169,11445451093632964279ull},{16072711,11466372601071120199ull},{16076597,13435999004029409279ull},
		{16080049,18266526887928755321ull},{16081679,11868117613510431529ull},{16085071,14158058431764629761ull},{16086911,16826194113484414487ull},{16087829,14626742151491329369ull},
		{16091021,14224052887860439639ull},{16064149,13197622355391030409ull},{16065149,13545314040804294319ull},{16067243,14152269016994013583ull},{16068049,16067045133341942353ull},
		{16070969,15370375936060062839ull},{16071529,15059986833339178897ull},{16057751,17402936053714225153ull},{16056451,15488725481110429159ull},{16054043,12672311049187782367ull},
		{16052353,11593765511293358047ull},{16052657,11085431129434680449ull},{16050487,10218100740052053263ull},{16048763,14100978243368072207ull},{16045357,12183869854186459801ull},
		{16035809,11855169834318581681ull},{16035941,17649706665176130473ull},{16027681,16209706991529970753ull},{16026343,10440895312670202409ull},{16026499,14608723870843452887ull},
		{16022003,16542024904771148857ull},{15838057,13724050657344286793ull},{15798119,18033822787942121599ull},{15372727,13620902108656457471ull},{15370987,13141770251775333463ull},
		{16423567, 9593004739201446737ull},{16424743,11207409667480506167ull},{16426331, 9315521980684624207ull},{16427317,12920971770507553729ull},{16427401,16822696323471127217ull},
		{16431991,13793486024804669009ull},{16434511,13533131423401796863ull},{16438259, 9952502816461400807ull},{16438421,13236391347894754153ull},{16441151,18194162210171783167ull},
		{16446379, 9889326862010987471ull},{16421063,15295542430741533281ull},{16411141,14567378334050418241ull},{16408439,10288195248504974393ull},{16386497,10040647230145986479ull},
		{16379327,12344397890070085393ull},{16383739, 9942526727677974137ull},{16384727,12597569013851067049ull},{16369789,10288230960652346249ull},{16364419,10607814077757516457ull},
		{16359293,11028045969272017577ull},{16344319, 9803688218926567111ull},{16339627,10586347451730006521ull},{16323487,12699616391325608273ull},{16307677,10780482862846368239ull},
		{16286441,11747242053545138911ull},{16280959,10537253101166306807ull},{16269433,10520779020741875777ull},{16264579,10000450921799397911ull},{16263427,12056468544232052593ull},
		{16229383,10028147391258765553ull},{16218541,11886145592975848529ull},{16237721,11289182387241165217ull},{18202697, 9523705169798429639ull},{0,0ull},{0,0ull},{0,0ull},{0,0ull}
	};

	/* 65-bit factors. Only store the bottom 64 bits of each factor - 65th bit assumed == 1. */
	static const struct testFac fac65[] =
	{
		{16446217, 2580748063118626543ull},{16441037, 7159845698788165911ull},{16423243,16053932793878088265ull},{16422199,15449225178445266103ull},{12307619,  834960775198036465ull},
		{12891233,  320347219092137433ull},{13091369, 4490626336223492551ull},{13251437, 5252892907907759335ull},{13278611, 4992876320436398817ull},{14191367, 3043526892782259057ull},
		{15372433,  835012824499792921ull},{15378317, 1469384281496492841ull},{15381727, 3296686288473649015ull},{15716923, 1071681628997604121ull},{15792929, 4568661977477836713ull},
		{15829339, 1353718096095573831ull},{15857663, 1877275782840805977ull},{15896423, 2846280113165409217ull},{15389291,18328954321183659201ull},{15424001,14055059218732619601ull},
		{15649649,16271423290019634663ull},{15716923, 1071681628997604121ull},{15792079,14719270918611609657ull},{15792929, 4568661977477836713ull},{15829339, 1353718096095573831ull},
		{15857663, 1877275782840805977ull},{15888107, 9672885011805764433ull},{15896423, 2846280113165409217ull},{15896591,12753248453534549335ull},{15913669,10042894560178858561ull},
		{15919751, 9077968001905706863ull},{15935603,11440996883202164671ull},{15937421, 3477787277761589433ull},{15940453, 3293099948043389463ull},{15990151, 9384163689137796817ull},
		{16020359,15674564692213482777ull},{16020997,10583884050943532673ull},{16021231, 2457557263587737193ull},{16023079, 7172265904034801095ull},{16023647,12254780112882764065ull},
		{16026047, 2015612163892272583ull},{16030051,12636535017081930135ull},{16030981, 4533504886207341415ull},{16032113, 4098116656401037545ull},{16033373, 4283833442114036481ull},
		{16037731, 4669494584567097327ull},{16042031, 8521202797833889425ull},{16042363, 1118497044506213463ull},{16043689, 5061951072668619745ull},{16045877, 7034068819169950713ull},
		{16046413, 7970266257131337337ull},{16047763,17720298914808722185ull},{16049867, 2777504824160429671ull},{16051073,17305172545354635225ull},{16052317, 6141121158983034457ull},
		{16056541,10695684989098787847ull},{16061009, 1456724931068173687ull},{16063109,13596747992403972297ull},{16063357, 8295891058026308023ull},{16070227, 2132028069846281607ull},
		{16074001,  613796669972356527ull},{16078813, 6533829271585121673ull},{16085423,10209359921277170607ull},{16090609, 2161620649992862905ull},{16096469, 1554699406605092793ull},
		{16100053, 8795628605553802671ull},{16100347,13291493148042713257ull},{16103719, 1585997210388541833ull},{16109507,12512063892223428961ull},{16109749,  968325311857141351ull},
		{16113029,18313175825134571025ull},{16123907, 7411731869541221617ull},{16126067, 4236545548179355081ull},{16126601,13228754303985015511ull},{16135051, 5990895667432870231ull},
		{16135411,15090056596912858585ull},{16137103,10560319150004001961ull},{16141709,17519457444182672455ull},{16147013,11445706533097774785ull},{16147393, 2659601528506615945ull},
		{16148129,17675056835266093305ull},{16149409,  204949170576663745ull},{16149787,16731036664601428255ull},{16149803, 7545607839488943193ull},{16150331, 6874010104330448145ull},
		{16155731, 9207449049005699935ull},{16162171, 9369851092374283137ull},{16163359, 4539568180228208935ull},{16164581,18211837706550913441ull},{16165507,15412546995979766593ull},
		{16174409, 1732140531559079415ull},{16179661,10492475743208936665ull},{16180903,16145957810547426145ull},{16184957,15091206671754796143ull},{16186603,15907332645141087985ull},
		{16189127, 5212322745663667383ull},{16194487,10173479279066636175ull},{16195183,15192552222295211305ull},{16197113,13841445967322975943ull},{16197499,17050899974760590215ull},
		{16193159, 1385423280458214967ull},{16200641, 2828285506828385065ull},{16203211,11100172884046605975ull},{16205971, 9273740347421492193ull},{16206349,10694307907604074233ull},
		{16213033, 1614410328985344607ull},{16213187, 1954435271914117233ull},{16215149,16532972262340179657ull},{16215317, 1008121765686796233ull},{17801687,  128917458759517407ull},
		{17807753,  336494230971911497ull},{17716367,  905665796573663601ull},{17751211, 5618955195753508855ull},{17774759, 6014993229899159887ull},{17720089,13529342842575268383ull},
		{17760683,14990248056153767583ull},{17775391, 2350339510098510591ull},{17777197, 9571464417356251447ull},{17779403,18177477733292920041ull},{17779837, 7906895062697711601ull},
		{17789269,10001883951858619657ull},{17789957,18252033438277821393ull},{17820343,15700395625722743455ull},{17820353,15618585221939458585ull},{17822141, 3164553597118799223ull},
		{17829673,11147254248974845591ull},{17830871,11225984642974396225ull},{17835733,10023921494986403217ull},{17838259, 3260207741789404783ull},{17848801, 4875901499668317943ull},
		{17852519,15091062508870596217ull},{17852047,  601058353873004391ull},{17857373, 1701161678010384433ull},{17866621, 5208343099296284167ull},{17866657,13299976954651583865ull},
		{17869171,16676513047471408473ull},{17869783,  621259534838019577ull},{17870123,15084110248119306447ull},{17881901, 2826293090465457897ull},{17883293,13129524775288454065ull},
		{17893621, 5577207661761066993ull},{17898871,13731800934001310335ull},{17903773,15168446379627624537ull},{17903917,  840239117157541023ull},{17907991, 1108904181398829207ull},
		{17914537,14467964519448464401ull},{17924609,11950510395434416473ull},{17933621,14476051576170801127ull},{17933987,12271121888256658441ull},{17943683, 4871696868526692145ull},
		{17945779, 2390989063500577855ull},{17952059, 4526068326859465345ull},{17957671,11917441992759533751ull},{17955761, 7098876507611502201ull},{17960443, 1402663829241032233ull},
		{17968817,14222886584371031521ull},{17971001,15950977244893884273ull},{17972399, 3710008620325908985ull},{17972657, 7132236937769524687ull},{17975669, 7099626784541023537ull},
		{17979631, 4226380440331508463ull},{17979641,14785225336208575737ull},{17984581,  858523932063013273ull},{17987267, 6850516154583906681ull},{17987461,14388480591581056111ull},
		{17988371,12210936384520386513ull},{17994199, 2164882008197615161ull},{18005447,  166370812775332375ull},{18250433,  344598024120039225ull},{18224719,  553997798478189057ull},
		{18154817,   79872243127358095ull},{18150971,  655633122262579545ull},{18200599,   92076886106971825ull},{18284677, 1888915001773035865ull},{18051997, 1150367354656911321ull},
		{18238631, 1017245559490767655ull},{18206621, 1568022926158681023ull},{18100601, 2745906387390898953ull},{18245951, 2231815919872618561ull},{18014323, 2458848702076602991ull},
		{18023917, 2058241897646001985ull},{18110201, 2720723434949125423ull},{18282137, 2035603280846190463ull},{18133991, 2632171659342400255ull},{18123323, 3330516929914639263ull},
		{18006713, 3028904303953805257ull},{18229837, 3685053582537101961ull},{18287713, 4160915327054246911ull},{18053807, 4543935829736639911ull},{18290663, 4062712588739458377ull},
		{18076511, 4182270262765602793ull},{18066679, 4897645337609563137ull},{18072421, 5488276451463637903ull},{18144029, 5575221700433972071ull},{18144589, 5051459111074577343ull},
		{18151787, 5763204143979111753ull},{18035429, 6300514689716925561ull},{18224863, 6179587156579443145ull},{18262147, 6266978406726558697ull},{18287327, 6195008163268138447ull},
		{18210433, 6203752099699856671ull},{18050729, 6425811605110608777ull},{18143687, 6473267153150631561ull},{18174397, 6246898253762108191ull},{18062267, 6855361096834950297ull},
		{18292607, 6407061090330898495ull},{18248981, 7276207334850673095ull},{18207569, 7642254765211602895ull},{18091121, 8151237761253289881ull},{18054097, 8052096042537656761ull},
		{18200087, 8912314933991519977ull},{18055393, 8317264894231219935ull},{18249053, 8407382588101954711ull},{18189331, 8789539481715341863ull},{18043997, 8515722445328005785ull},
		{18011239, 8951373386839770463ull},{18298117, 8636556078317653257ull},{18271801, 8490243542019023553ull},{18202253, 8748336695715872857ull},{18100171, 8782994045353648191ull},
		{18191993, 8974531565168049591ull},{0,0ull}
	};

#endif	/* #ifndef fac_test_dat64_included */
