# arch-tag: 4b63b80e-9add-46e1-941f-50556f7af04a
# Copyright (C) 2004 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test Twisted support
"""

import os
import shutil
from twisted.trial import unittest
from framework_twisted import run_in_thread


def make_test_dir(testcase, subdir):
    path = os.path.join(os.getcwd(), testcase.__class__.__name__)
    if os.path.exists(path): shutil.rmtree(path)
    os.makedirs(path)
    return path

def make_spawn(command):
    from pybaz.backends import knotted, logger
    return knotted.TwistedSpawningStrategy(command, logger.Logger())


class TwistedStatusCmd(unittest.TestCase):

    def test_null_cmd(self):
        """TwistedSpawningStrategy.null_cmd works"""
        run_in_thread(self._null_cmd)

    def _null_cmd(self):
        spawn = make_spawn('touch')
        path = make_test_dir(self, 'null_cmd')
        foo = os.path.join(path, 'foo')
        spawn.status_cmd((foo,), None, (0,))
        self.assert_(os.path.isfile(foo))

    def test_exec_problem(self):
        """TwistedSpawningStrategy.null_cmd raises ExecProblem"""
        run_in_thread(self._exec_problem)

    def _exec_problem(self):
        from pybaz import errors
        spawn = make_spawn('false')
        def thunk(): spawn.status_cmd((), None, (0,))
        # thunk()
        self.assertRaises(errors.ExecProblem, thunk)
        try:
            thunk()
            self.fail('did not raise')
        except errors.ExecProblem, e:
            self.assertEqual(1, e.proc.status)
            self.assertEqual(None, e.proc.signal)

    def test_error(self):
        """TwistedSpawningStrategy.null_cmd records error output"""
        run_in_thread(self._error)

    def _error(self):
        path = make_test_dir(self, 'error')
        name = os.path.join(path, 'printerr')
        script = open(name, 'w')
        print >> script, "echo -n first line >&2"
        print >> script, "sleep 1"
        print >> script, "echo >&2"
        print >> script, "echo second line >&2"
        print >> script, "exit 1"
        script.close()
        from pybaz import errors
        spawn = make_spawn('sh')
        try:
            spawn.status_cmd((name,), None, (0,))
            self.fail('did not raise')
        except errors.ExecProblem, e:
            expected = '\n'.join(["first line", "second line", ""])
            self.assertEqual(expected, e.proc.error)

    def test_status(self):
        """TwistedSpawningStrategy.status_cmd works"""
        run_in_thread(self._status)

    def _status(self):
        from pybaz import errors
        spawn = make_spawn('sh')
        status = spawn.status_cmd(('-c', 'exit 0'), None, expected=(0,1))
        self.assertEqual(0, status)
        status = spawn.status_cmd(('-c', 'exit 1'), None, expected=(0,1))
        self.assertEqual(1, status)
        def thunk(): spawn.status_cmd(('-c', 'exit 2'), None, expected=(0,1))
        self.assertRaises(errors.ExecProblem, thunk)

    def test_chdir(self):
        """TwistedSpawningStrategy.status_cmd does chdir"""
        run_in_thread(self._chdir)

    def _chdir(self):
        spawn = make_spawn('touch')
        path = make_test_dir(self, 'chdir')
        status = spawn.status_cmd(('foo',), path, expected=(0,))
        self.assertEqual(0, status)
        name = os.path.join(path, 'foo')
        self.assert_(os.path.isfile(name))

    # TODO: test signal


class TwistedSequenceCmd(unittest.TestCase):

    def test_sequence(self):
        """TwistedSpawningStrategy.sequence_cmd works"""
        run_in_thread(self._sequence)

    def _sequence(self):
        spawn = make_spawn('sh')
        args = ('-c', "echo aaa ; echo bbb ; exit 1")
        cmd = spawn.sequence_cmd(args, None, (0, 1))
        self.assertEqual(False, cmd.finished)
        thunk = lambda : cmd.status
        self.assertRaises(AttributeError, thunk)
        result = list(cmd)
        self.assertEqual(['aaa', 'bbb'], result)
        self.assertEqual(True, cmd.finished)
        self.assertEqual(1, cmd.status)

    def test_failure(self):
        """TwistedSpawningStrategy.sequence_cmd raises ExecProblem"""
        run_in_thread(self._failure)

    def _failure(self):
        from pybaz import errors
        spawn = make_spawn('sh')
        script = "echo aaa ; echo bbb >&2 ; exit 1 ;"
        cmd = spawn.sequence_cmd(('-c', script), None, (0,))
        result = cmd.next()
        self.assertEqual("aaa", result)
        try:
            cmd.next()
            self.fail("did not raise")
        except errors.ExecProblem, e:
            self.assertEqual(1, e.proc.status)
            self.assertEqual(None, e.proc.signal)
            self.assertEqual("bbb\n", e.proc.error)

    def test_chdir(self):
        """TwistedSpawningStrategy.sequence_cmd does chdir"""
        run_in_thread(self._chdir)

    def _chdir(self):
        spawn = make_spawn('pwd')
        path = make_test_dir(self, 'pwd')
        seq = spawn.sequence_cmd((), path, expected=(0,))
        lines = list(seq)
        self.assertEqual([path], lines)


class TwistedStatusTextCmd(unittest.TestCase):

    def test_status_text(self):
        """TwistedSpawningStrategy.status_text_cmd works"""
        run_in_thread(self._status_text)

    def _status_text(self):
        spawn = make_spawn('sh')
        args = ('-c', "echo aaa ; echo bbb ; exit 1")
        (status, text) = spawn.status_text_cmd(args, None, (1,))
        self.assertEqual(1, status)
        self.assertEqual("aaa\nbbb\n", text)

    def test_failure(self):
        """TwistedSpawningStrategy.status_text_cmd raises ExecProblem"""
        run_in_thread(self._failure)

    def _failure(self):
        from pybaz import errors
        spawn = make_spawn('sh')
        args = ('-c', "echo aaa ; echo bbb >&2 ; exit 2")
        try:
            status, text = spawn.status_text_cmd(args, None, (1,))
            self.fail("did not raise")
        except errors.ExecProblem, e:
            self.assertEqual(2, e.proc.status)
            self.assertEqual(None, e.proc.signal)
            self.assertEqual("bbb\n", e.proc.error)

    def test_chdir(self):
        """TwistedSpawningStrategy.text_cmd does chdir"""
        run_in_thread(self._chdir)

    def _chdir(self):
        spawn = make_spawn('pwd')
        path = make_test_dir(self, 'pwd')
        status, text = spawn.status_text_cmd((), path, expected=(0,))
        self.assertEqual(0, status)
        self.assertEqual(path+'\n', text)
