#!/usr/bin/env python
# arch-tag: fb639c6b-566d-4b7c-be27-cfc406fc2fb8
# Copyright (C) 2004 David Allouche <david@allouche.net>
#                    Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test access to containers and name part properties of namespace classes.
"""

import sys
import pybaz as arch
from pybaz.tests import framework
from pybaz.tests.framework import TestCase, TestParams


class ArchNameProperties(TestCase):

    """Test cases access to container and name part properties

    Check the availability and correctness of the "archive",
    "category", "branch", "version", "nonarch", and "patchlevel" of
    archive objects.
    """

    patchlevel = None # for pychecker

    def setUp(self):
        self.params = TestParams()
        def set(**kw):
            for k, v in kw.items():
                v = v % self.__dict__
                setattr(self, k, v)
                setattr(self, k+'_full', '%s/%s' % (self.params.arch_name, v))
        self.arch_name = self.params.arch_name
        set(category='frob')
        set(branch='%(category)s--devo')
        set(version='%(branch)s--1.0')
        self.patchlevel = 'patch-1'
        set(revision='%(version)s--%(patchlevel)s')
        set(nameless_branch='%(category)s')
        set(nameless_version='%(category)s--1.0')
        set(nameless_revision='%(nameless_version)s--%(patchlevel)s')

    tests = []

    def readonly(self, obj, *attrs):
        for attr in attrs:
            def try_set(): setattr(obj, attr, 'error')
            self.assertRaises(AttributeError, try_set)

    def check_hasnot(self, obj, *attrs):
        for attr in attrs:
            self.assert_(not hasattr(obj, attr))

    def assertInstance(self, obj, cls):
        self.assert_(isinstance(obj, cls))

    def check_names(self, obj, namekind):
        self.readonly(obj, 'nonarch', 'fullname')
        self.assertEqual(obj.nonarch, getattr(self, namekind))
        self.assertEqual(str(obj), obj.fullname)
        self.assertEqual(obj.fullname, getattr(self, namekind+'_full'))

    enclosing = {'category': 'archive', 'branch': 'category',
                 'version': 'branch', 'revision': 'version'}

    def check_init(self, cls, kind, nameless=''):
        def check_names(obj): self.check_names(obj, nameless+kind)
        obj = cls(getattr(self, nameless+kind+'_full'))
        check_names(obj)
        def unsafe_ctor(*args): return cls(arch._builtin._unsafe(args))
        if kind != 'revision':
            obj = unsafe_ctor(self.params.arch_name,
                              getattr(self, nameless+kind))
        else:
            obj = unsafe_ctor(self.params.arch_name,
                              getattr(self, nameless+self.enclosing[kind]),
                              self.patchlevel)
        check_names(obj)
        obj = cls(obj)
        check_names(obj)
        return obj

    def check_badinit(self, cls, *names):
        for name in names:
            func = lambda: cls(getattr(self, name))
            self.assertRaises(arch.errors.NamespaceError, func)

    def check_container_helper(self, obj, prop, strval):
        self.readonly(obj, prop)
        val = getattr(obj, prop)
        typ = getattr(arch, prop.capitalize())
        self.assertInstance(val, typ)
        self.assertEqual(str(val), strval)

    def check_containers(self, obj, kind, nameless=''):
        if kind == 'archive':
            self.check_container_helper(obj, 'archive', self.params.arch_name)
            return
        self.check_containers(obj, self.enclosing[kind], nameless)
        if kind == 'category': namekind=kind
        else: namekind = nameless+kind
        self.check_container_helper(obj, kind, getattr(self, namekind+'_full'))

    def test_category(self):
        """Category has correct constructor and properties."""
        cat = self.check_init(arch.Category, 'category')
        self.check_containers(cat, 'archive')
        self.check_badinit(arch.Category, 'arch_name', 'branch_full')
        self.check_hasnot(cat, 'category', 'branch', 'version', 'patchlevel')

    tests.append('test_category')

    def test_branch(self, nameless=False):
        """Branch has correct constructor and properties."""
        nameless = nameless and 'nameless_' or ''
        brn = self.check_init(arch.Branch, 'branch', nameless)
        self.check_containers(brn, 'category')
        self.check_badinit(arch.Branch, 'arch_name',
                           nameless+'version_full')
        self.check_hasnot(brn, 'branch', 'version', 'patchlevel')

    tests.append('test_branch')

    def test_version(self, nameless=False):
        """Version has correct constructor and properties."""
        nameless = nameless and 'nameless_' or ''
        vsn = self.check_init(arch.Version, 'version', nameless)
        self.check_containers(vsn, 'branch', nameless)
        self.check_badinit(arch.Version, nameless+'branch_full',
                           nameless+'revision_full')
        self.check_hasnot(vsn, 'version', 'patchlevel')

    tests.append('test_version')

    def test_revision(self, nameless=False):
        """Revision has correct constructor and properties."""
        nameless = nameless and 'nameless_' or ''
        rvsn = self.check_init(arch.Revision, 'revision', nameless)
        self.check_containers(rvsn, 'version', nameless)
        self.check_badinit(arch.Revision, nameless+'version_full')
        self.readonly(rvsn, 'patchlevel')
        self.assertEqual(rvsn.patchlevel, self.patchlevel)

    tests.append('test_revision')

    def test_nameless_branch(self):
        """Branch (nameless branch) has correct ctor and properties."""
        self.test_branch(nameless=True)
    tests.append('test_nameless_branch')

    def test_nameless_version(self):
        """Version (nameless branch) has correct ctor and properties."""
        self.test_version(nameless=True)
    tests.append('test_nameless_version')

    def test_nameless_revision(self):
        """Revision (nameless branch) has correct ctor and properties."""
        self.test_revision(nameless=True)
    tests.append('test_nameless_revision')


framework.register(__name__)
