#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2006 Jd, Hap Hazard & Yves Perrenoud
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU General Public License (GPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/gpl.txt
#
import types
import os, sys
import gtk, gtk.glade, gobject

import constants
from ManagedNode import ManagedNode, NodeException
from XenNode import XenNode, DomConfig
from utils import XMConfig, is_host_remote
import urllib, urlparse
import xml.parsers.expat

class DomSettings:
    """Class that handles everything related to the Dom Settings dialog"""

    class DiskEdit:
        """Class that handles everything related to the disk edit area of
        the settings dialog"""

        diskmapping = ["file", "phy"]
        modemapping = ["w", "r"]
        colmap = ["Type", "Device Path", "Guest Device", "Mode"]
        guestdevs = ["xvda", "xvdb", "xvdc"]

        def __init__(self, wtree, filesel, parentwin):
            self.dialog = wtree.get_widget("DiskEditDialog")
            self.addbutton = wtree.get_widget("DiskAddButton")
            self.propbutton = wtree.get_widget("DiskPropButton")
            self.rembutton = wtree.get_widget("DiskRemoveButton")
            self.disktype = wtree.get_widget("DiskTypeCombo")
            self.diskdev = wtree.get_widget("DiskDeviceEntry")
            self.diskdevfsel = wtree.get_widget("DiskDeviceButton")
            self.guestdev = wtree.get_widget("GuestDevCombo")
            self.mode = wtree.get_widget("ModeCombo")
            

            self.diskview = wtree.get_widget("DiskTreeView")
            textrenderer = gtk.CellRendererText()

            self.dialog.set_transient_for(parentwin)

            for dev in self.guestdevs:
                self.guestdev.append_text(dev)

            pos = 0
            for colname in self.colmap:
                column = gtk.TreeViewColumn(colname, textrenderer, text=pos)
                self.diskview.append_column(column)
                pos += 1

            wtree.signal_autoconnect(
                {"on_DiskAddButton_clicked": self.__showSettings,
                 "on_DiskPropButton_clicked": (self.__showSettings, True),
                 "on_DiskRemoveButton_clicked": self.__removeClicked,
                 "on_DiskDeviceButton_clicked": (filesel, self.diskdev,
                                                 "open", self.dialog),
                 })

        def __removeClicked(self, widget):
            """Handles the clicking of the remove button in the disk section
            of the settings dialog"""

            model, iter = self.diskview.get_selection().get_selected()

            if iter:
                model.remove(iter)

        def showtree(self, dom=None):
            """Populates the disk tree view with all disks for this dom"""

            self.diskmodel = gtk.ListStore(*[gobject.TYPE_STRING]*4)
            self.diskview.set_model(self.diskmodel)

            if dom:
                for disk in dom.get_config().getDisks():
                    if disk.type == "file":
                        dtype = disk.type
                    else:
                        dtype = "phy"

                    self.diskmodel.append([dtype, disk.filename,
                                           disk.device, disk.mode])

        def __showSettings(self, widget, edit=False):
            """Show the dialog and optionally populate with the selected
            disk entry"""

            model, iter = self.diskview.get_selection().get_selected()

            if edit and iter:
                for combo, mapping, entry in (
                    (self.disktype, self.diskmapping, "Type"),
                    (self.mode, self.modemapping, "Mode")):
                    combo.set_active(mapping.index(
                        model.get_value(iter, self.colmap.index(entry))))

                self.diskdev.set_text(
                    model.get_value(iter, self.colmap.index("Device Path")))

                guestdev =  model.get_value(iter,
                                            self.colmap.index("Guest Device"))
                if guestdev not in self.guestdevs:
                    self.guestdev.prepend_text(guestdev)
                    self.guestdev.set_active(0)
                else:
                    self.guestdev.set_active(self.guestdevs.index(guestdev))
            else:
                self.diskdev.set_text('')

                # The following three lines are a hack to compensate for the
                # fact that the following call doesn't work as documented
                ## self.guestdev.set_active(-1)
                self.guestdev.prepend_text('')
                self.guestdev.set_active(0)
                self.guestdev.remove_text(0)

                for combo in (self.disktype, self.mode):
                    combo.set_active(0)

            res = self.dialog.run() == gtk.RESPONSE_OK

            if not res:
                self.dialog.hide()
                return

            if edit and iter:
                model.remove(iter)

            model.append(
                [self.diskmapping[self.disktype.get_active()],
                 self.diskdev.get_text(),
                 self.guestdev.get_active_text(),
                 self.modemapping[self.mode.get_active()],
                 ])

            self.dialog.hide()

        def getDiskEntries(self):
            """Returns a list of DiskEntry instances for each row of the
            ListStore"""

            delist = []
            iter = self.diskmodel.get_iter_first()
            while iter:
                diskparms = []
                for row in enumerate(self.colmap):
                    diskparms.append(self.diskmodel.get_value(iter, row[0]))

                #_RESTRUCT
                delist.append(DomConfig.DiskEntry(diskparms))
                iter = self.diskmodel.iter_next(iter)

            return delist

    combomapping = ["destroy", "restart", "preserve", "rename-restart"]

    textfields = ("name", "kernel", "ramdisk", "root", "extra", "bootloader")
    valuefields = ("memory", "vcpus")
    combofields = ("on_shutdown", "on_reboot", "on_crash")

    minmem = 16
    defaultmem = 256
    ignoreTypeChange = False

    # DomSettings constructor
    def __init__(self, wtree, image_store):
        self.dialog = wtree.get_widget("SettingsDialog")

        self.kernel = wtree.get_widget("KernelEntry")
        self.ramdisk = wtree.get_widget("RamDiskEntry")
        self.use_bootloader = wtree.get_widget("UseBootloaderCheckButton")
        self.bootloader = wtree.get_widget("BootloaderEntry")
        self.rootdev = wtree.get_widget("RootDevEntry")
        self.kernargs = wtree.get_widget("KernArgsEntry")
        self.diskframe = wtree.get_widget("DiskFrame")
        self.usablecpu = wtree.get_widget("UsableCPUEntry")
        self.vcpu = wtree.get_widget("VCPUSpin")
        self.power = wtree.get_widget("PowerCombo")
        self.reboot = wtree.get_widget("RebootCombo")
        self.crash = wtree.get_widget("CrashCombo")
        self.networkframe = wtree.get_widget("NetworkFrame")
        self.name = wtree.get_widget("NameEntry")
        self.kernelsel = wtree.get_widget("KernelButton")
        self.ramdisksel = wtree.get_widget("RamDiskButton")
        self.memory = wtree.get_widget("MemoryScale")
        self.inmemory = wtree.get_widget("InMemory")
        self.ondisk = wtree.get_widget("OnDisk")
        self.filename = wtree.get_widget("FileNameEntry")
        self.filenamesel = wtree.get_widget("FileNameButton")
        self.okbutton = wtree.get_widget("settings_okbutton")
        self.image_name = wtree.get_widget("ImageName")
        #_RESTRUCT

        self.image_store = image_store

        self.diskedit = self.DiskEdit(wtree, self.__filesel, self.dialog)

        self.diskmapping = {"name": self.name, "memory": self.memory,
                            "kernel": self.kernel, "ramdisk": self.ramdisk,
                            "root": self.rootdev, "cpus": self.usablecpu,
                            "extra": self.kernargs, "vcpus": self.vcpu,
                            "on_shutdown": self.power,
                            "on_reboot": self.reboot,
                            "on_crash": self.crash,
                            "bootloader": self.bootloader,}

        wtree.signal_autoconnect(
            {"on_InMemory_clicked": self.__typeChange,
             "on_KernelButton_clicked": (self.__filesel, self.kernel),
             "on_RamDiskButton_clicked": (self.__filesel, self.ramdisk),
             "on_FileNameButton_clicked": (self.__filesel, self.filename,
                                           "save"),
             "on_UseBootloaderCheckButton_toggled": self.__boot_toggled,
             "on_ImageName_changed": self.__image_changed,
             })

        # populate the image list
        self.availableImages = self.image_store.list
        self.image_name.append_text("")
        for image in self.availableImages:
            self.image_name.append_text(image)

        if len(self.availableImages) > 0:
            self.image_name.set_active(0)

    def __image_changed(self, widget):
        image = self.image_name.get_active_text()
        if image is None or image is "":
            k = ""
            r = ""
        else:
            (k,r) = self.image_store.getFilenames(image)
            
        self.kernel.set_text(k)
        self.ramdisk.set_text(r)

    def __boot_toggled(self, widget):
        use_boot = widget.get_active()
        self.kernel.set_sensitive(not use_boot)
        self.ramdisk.set_sensitive(not use_boot)
        self.bootloader.set_sensitive(use_boot)
        
    

    def __typeChange(self, widget):
        """Handles the radio button that allows one to select what kind
        of settings (either memory resident or on disk) one wishes to edit"""

        if self.ignoreTypeChange:
            return

        if self.inmemory.get_active():
            self.type = "memory"
            self.__showMem()
        elif self.ondisk.get_active():
            self.type = "disk"
            self.__showDisk()
        else:
            raise TypeError

    def __noOpActive(self, widget, state):
        """Change the state of the settings type widgets without performing
        any actions as a result"""

        self.ignoreTypeChange = True
        widget.set_active(state)
        self.ignoreTypeChange = False

    def __filesel(self, widget, field,
                  fstype="open", parentwin=None):
        """Handles the pressing of the file selection buttons"""

        if not parentwin:
            parentwin = self.dialog

        filename = field.get_text()

        init_folder = os.path.dirname(filename)
        if fstype == "open":
            init_file = None
        else:
            init_file = filename

        result, filename = file_selection(self.managed_node,
                                          "Select file", fstype, init_folder,
                                          init_file, parentwin)

        if result:
            field.set_text(filename)

    def __dom0WidgetsOn(self, state):
        """sets the state of the widgets that are immutable only for dom0"""

        for widget in (self.diskframe, self.usablecpu, self.vcpu, #self.memory,
                       self.networkframe):
            widget.set_sensitive(state)
        

    def __diskWidgetsOn(self, state):
        """sets the state of the widgets that only apply to on disk settings"""

        for widget in (self.kernel, self.ramdisk, self.rootdev, self.kernargs,
                       self.kernelsel, self.ramdisksel,
                       self.name,
                       self.usablecpu,
                       self.power, self.reboot, self.crash,
                       self.filename, self.filenamesel, self.diskframe,
                       self.bootloader, self.use_bootloader):
            widget.set_sensitive(state)

            if self.blank:
                self.image_name.set_sensitive(True)
            else:
                self.image_name.set_sensitive(False)

    def __showMem(self):
        """Displays the current selection's memory resident settings"""

        self.__dom0WidgetsOn(self.dom.id)
        self.__diskWidgetsOn(False)

        if self.dom.get_config() and self.dom.get_config().filename:
            self.filename.set_text(self.dom.get_config().filename)
        else:
            self.filename.set_text("")

        for widget, parm in ((self.name, "name"),
                             (self.kernel, "kernel"),
                             (self.ramdisk, "ramdisk"),
                             (self.usablecpu, "vcpu_avail"),
                             (self.bootloader, "bootloader"),
                             ):
            value = self.dom[parm]
            if value:
                widget.set_text(value)
            else:
                widget.set_text('')

       

        curmem = int(self.dom["memory"])
        freemem = int(self.managed_node["free_memory"])
        self.memory.set_range(self.minmem, curmem+freemem)

        for widget, parm in ((self.memory, "memory"), (self.vcpu, "vcpus")):
            widget.set_value(int(self.dom[parm]))

        for widget, parm in ((self.power, "on_poweroff"),
                             (self.reboot, "on_reboot"),
                             (self.crash, "on_crash")):
            widget.set_active(
                self.combomapping.index(self.dom[parm]))

        for widget in (self.rootdev, self.kernargs):
            widget.set_text('')

        self.diskedit.showtree()

    def __showDisk(self):
        """Displays the current selection's on disk settings"""

        self.__dom0WidgetsOn(self.ondisk.flags() & gtk.SENSITIVE)
        self.__diskWidgetsOn(self.ondisk.flags() & gtk.SENSITIVE)

        # Special handling for memory. (Some weired legacy)
        self.memory.set_sensitive(self.ondisk.flags() & gtk.SENSITIVE)

        #_RESTRUCT
        #if self.dom.isResident:
        #    self.domconfig = self.dom.get_config()
        #else:
        #    self.domconfig = self.dom
        self.domconfig = self.dom.get_config()

        self.filename.set_text(self.domconfig.filename)

        bl = self.domconfig["bootloader"]
        if bl and bl.strip() != '':
            self.use_bootloader.set_active(True)
        else:
            self.use_bootloader.set_active(False)

        if self.ondisk.flags() & gtk.SENSITIVE:
            self.__boot_toggled(self.use_bootloader)
       
        for entry, widget in self.diskmapping.iteritems():
            if entry in self.textfields:
                if entry in self.domconfig:
                    value = self.domconfig[entry]
                else:
                    value = ''

                if not value:
                    value = ''
                
                widget.set_text(value)

            elif entry == "memory":
                self.memory.set_range(self.minmem,
                                      int(self.managed_node["total_memory"]))

                if entry in self.domconfig:
                    value = int(self.domconfig[entry])
                else:
                    value = self.defaultmem

                widget.set_value(value)

            elif entry == "vcpus":
                if entry in self.domconfig:
                    value = int(self.domconfig[entry])
                else:
                    value = 1

                widget.set_value(value)

            elif entry in self.combofields:
                if entry in self.domconfig:
                    value = self.combomapping.index(self.domconfig[entry])
                else:
                    value = -1

                widget.set_active(value)

        self.diskedit.showtree(self.dom)

    def __showBlank(self):
        """Clears all the fields"""

        self.__dom0WidgetsOn(True)
        self.__diskWidgetsOn(True)

        self.filename.set_text('')

        for entry in ("name", "kernel", "ramdisk", "root", "extra"):
            self.diskmapping[entry].set_text('')
            
        self.memory.set_value(self.defaultmem)
        self.vcpu.set_value(1)

        for entry in ("on_shutdown", "on_crash"):
            self.diskmapping[entry].set_active(0)

        self.diskmapping["on_reboot"].set_active(1)

        self.diskedit.showtree()
        self.use_bootloader.set_active(False)
        self.bootloader.set_sensitive(False)

    def show(self, widget, managed_node, dom = None, blank=False, rerun=False):
        """Displays the settings dialog"""

        # Requires
        #  - managed_node to get information on available cpu and mem
        #  - dom : for which the settings are being changed.

        ## Kludge, for now only one dialgbox allowed
        self.managed_node = managed_node
        self.dom = dom

        self.blank = blank
        self. enable_write = False
        self.vcpu.set_range(1.0, float(self.managed_node["nr_cpus"]))

        self.image_name.set_active(0)
        
        if not rerun:
            if not blank and dom and self.managed_node.isResident(dom.name):
                self.type = "memory"
                self.inmemory.set_sensitive(True)
                self.__noOpActive(self.inmemory, True)

                if self.dom.get_config() is not None and \
                   self.dom.get_config().is_xenman_generated() :
                    self.ondisk.set_sensitive(True)
                else:
                    self.ondisk.set_sensitive(False)

                self.__showMem()
            else:
                self.type = "disk"
                self.inmemory.set_sensitive(False)
                self.__noOpActive(self.ondisk, True)
                if blank or dom.get_config().is_xenman_generated():
                    self.ondisk.set_sensitive(True)
                else:
                    self.ondisk.set_sensitive(False)

                if blank:
                    self.__showBlank()
                else:
                    self.__showDisk()

        res = self.dialog.run() == gtk.RESPONSE_OK

        if not res:
            self.dialog.hide()
            return

        try: # prevent exception in call back. Causes segfault.
            if self.type == "memory":
                self.dom.setMem(int(self.memory.get_value()))
                self.dom.setVCPUs(int(self.vcpu.get_value()))
            else:
                ## if disk was disabled return
                if self.blank or self.dom.get_config().is_xenman_generated():
                    pass
                else:
                    self.dialog.hide()
                    return

                errmsgs = []

                filename = self.filename.get_text()

                if not filename:
                    errmsgs.append("You must specify a file name.")

                if not self.name.get_text():
                    errmsgs.append("You must specify a name for the domain.")

                if not self.use_bootloader.get_active():
                    for widget, name in ((self.kernel, "kernel"),
                                         (self.ramdisk, "ram disk")):
                        value = widget.get_text()
                        if not value or not os.path.exists(value):
                            errmsgs.append("You must specify a valid %s file name." \
                                           % name)
                else:
                    value = self.bootloader.get_text()
                    if not value:
                        errmsgs.append("You must specify a valid bootloader.")

                if errmsgs:
                    errmsg = "The following errors were found:\n"
                    for msg in errmsgs:
                        errmsg += " - " + msg + "\n"

                    showmsg(errmsg)
                    self.show(widget, self.managed_node, self.dom, blank, True)
                    return

                if managed_node.node_proxy.file_exists(filename) and \
                       ((not blank and filename != self.domconfig.filename) \
                        or blank) and not \
                        confirmation("Do you wish to overwrite: %s" % filename):
                    self.show(widget, self.managed_node,self.dom, blank, True)
                    return


                if self.blank:
                    self.domconfig = DomConfig(self.managed_node)
                else:
                    self.domconfig = DomConfig(self.managed_node, filename)

                # Consider enhancing the for loop below to test whether config
                # elements have actually changed and only update the DomFile
                # instance entry for those that have changed

                for entry, widget in self.diskmapping.iteritems():
                    if entry in self.textfields:
                        self.domconfig[entry] = widget.get_text()

                    elif entry in self.valuefields:
                        self.domconfig[entry] = int(widget.get_value())

                    elif entry in self.combofields:
                        self.domconfig[entry] = self.combomapping[widget.get_active()]

                self.domconfig["disk"] = [
                    repr(i) for i in self.diskedit.getDiskEntries()
                    ]

                # reset depending on bootloader being used or not.
                if self.use_bootloader.get_active() and \
                       self.bootloader.get_text() and \
                       self.bootloader.get_text().strip() != '':
                    self.domconfig['ramdisk'] = ''
                    self.domconfig['kernel'] = ''
                else:
                    self.domconfig['bootloader']=''

                self.domconfig.save(filename)
                self.managed_node.add_dom_config(filename)
        except xml.parsers.expat.ExpatError, e:
            showmsg("Internal Error : Try again.")
        except Exception ,ex:
            showmsg("Exception : " + str(ex))
            
        self.dialog.hide()


class CreateDialog:
    """ Class that handles events from create dom dialog"""

    # keep track so that signals get connected once. We may need to find
    # better solution for this.
    initialized = False 
    
    def __init__(self, wtree, image_store):
        """ Constructor"""
        self.dialog      = wtree.get_widget("CreateDialog")
        self.domName     = wtree.get_widget("DomName")
        self.domMemory   = wtree.get_widget("DomMemory")
        self.domDiskSize = wtree.get_widget("DomDiskSize")
        self.domDiskPath = wtree.get_widget("DomDiskPath")
        self.useLVM      = wtree.get_widget("UseLVM")
        self.LVGList    = wtree.get_widget("LVGList")
        self.LVGLabel   = wtree.get_widget("LVGLabel")
        self.domImage   = wtree.get_widget("DomImage")

        # the image store.
        self.image_store = image_store
        self.availableImages = self.image_store.list

        
        # given to it in the show method
        self.managed_node = None
        
        
        # setup handlers

        if not CreateDialog.initialized:
            wtree.signal_connect("on_createDialog_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_createDialog_ok_button_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_createDialog_DomDiskDirButton_clicked",
                                 self.on_disk_dir_button_clicked)
            wtree.signal_connect("on_createDialog_UseLVM_toggled",
                                 self.on_useLVM_toggled)
            
            # populate the image list
            for image in self.availableImages:
                self.domImage.append_text(image)

            if len(self.availableImages) > 0:
                self.domImage.set_active(0)

            CreateDialog.initialized = True

        #self.init()


    def init(self):
        self.domName.set_text("")
        self.domMemory.set_text("256")
        self.domDiskSize.set_text("3")
        if (self.managed_node.config.get(XMConfig.PATHS,constants.prop_disks_dir) is not None):
            self.domDiskPath.set_text(self.managed_node.config.get(XMConfig.PATHS,constants.prop_disks_dir))
        else:
            self.domDiskPath.set_text('')

        if self.availableImages is None:
            return
        # set the default
        if self.image_store.default is not None:
            lower_default = self.image_store.default.lower()
            if lower_default in self.availableImages:
                pos = self.availableImages.index(lower_default)
                if pos >= 0:
                    self.domImage.set_active(pos)
                elif len(self.availableImages) > 0:
                    self.domImage.set_active(0)
        elif len(self.availableImages) > 0:
            self.domImage.set_active(0)
        
    
    def show(self, managed_node):
        """ Displays the create Dom dialog"""
        self.managed_node = managed_node

        self.init()

        # if no lvm then disable the checkbox
        #if managed_node.lvm_proxy == None:
        self.useLVM.set_sensitive(managed_node.isLVMEnabled)
            

        self.dialog.show() 

    def on_cancel_button_clicked(self, widget):
       """  Cancel creation of dom """
       self.dialog.hide()

    def on_ok_button_clicked(self, widget):
        """  Create Dom """

        
        #_RESTRUCT
        ## if self.domName.get_text() in domindex.domdict.keys():
##             if domindex[self.domName.get_text()].state != Dom.NOT_STARTED_STATE:
##                 showmsg("Running Domain with the same name exists.")
##                return False

        names = self.managed_node.get_dom_names()
        if self.domName.get_text() in names:
            #dom = self.managed_node.get_doms()[self.domName]
            if self.managed_node.isResident(self.domName):
                showmsg("Running Domain with the same name exists.")
                return False
        #_RESTRUCT

        
        # ask permission to overwrite if the domname
        # has already been quickcreated.
        if os.path.isfile(os.path.join(self.managed_node.config.get(XMConfig.PATHS,constants.prop_xenconf_dir),self.domName.get_text())):
            # domain conf file exists
            if confirmation('"%s" already exists: delete and replace?' % self.domName.get_text()):
                # permission to overwrite granted. cleanup exisiting files.
                cleanupQCDomain(self.managed_node,
                                self.domName.get_text(),
                                self.domDiskPath.get_text())
            else:
                # permission not granted. abort domain creation
                showmsg('Aborting Domain Creation')
                self.dialog.hide()
                return
            
        msg = ''
        try:
            diskpath =  self.domDiskPath.get_text()
            if self.useLVM.get_active():
                diskpath = self.LVGList.child.get_text()

            quickcreateDom(self.managed_node,
                           self.domName.get_text(),
                           self.domMemory.get_text(),
                           diskpath,
                           int(float(self.domDiskSize.get_text())*1024),
                           self.useLVM.get_active(),
                           image_store = self.image_store,
                           image_name  = self.domImage.get_active_text())
        except (IOError, OSError), err:
            msg = 'FAILED: '+str(err)
        except Exception, err:
            msg = 'FAILED: '+str(err)
        else:
            msg = 'Creation Successful'

        showmsg(msg)
        self.dialog.hide()
        
    def on_disk_dir_button_clicked(self, widget):
        (res, dirname) = \
              file_selection(self.managed_node,
                             "Choose location for creating file for Dom Disk",
                             "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.domDiskPath.set_text(dirname)

    def on_useLVM_toggled(self, widget, data=None):
        if widget.get_active():
            self.domDiskPath.set_sensitive(False)
            self.LVGList.set_sensitive(True)
            self.LVGLabel.set_sensitive(True)
            self.populateLVGList()
        else:
            self.LVGList.set_sensitive(False)
            self.LVGLabel.set_sensitive(False)
            self.domDiskPath.set_sensitive(True)

    def populateLVGList(self):
        lvglist_store = gtk.ListStore(gobject.TYPE_STRING)

        textrenderer = gtk.CellRendererText()
        self.LVGList.set_model(lvglist_store)
        self.LVGList.set_text_column(0)
                
        slistmodel = gtk.TreeModelSort(lvglist_store)
        slistmodel.set_sort_column_id(0, gtk.SORT_ASCENDING)
        lvgs = self.managed_node.lvm_proxy.listVolumeGroups()
        for lvg in lvgs:
            self.LVGList.append_text(lvg)
            
        if len(lvgs) >= 1:
            self.LVGList.set_active(0)
        

class InitParamsDialog:
    """ Class that handles events from collecting intial parameters dialog"""
    
    initialized = False
    
    def __init__(self, wtree, managed_node):
        """ Constructor"""
        self.managed_node = managed_node
        self.config = managed_node.config
        self.dialog = wtree.get_widget("initParamsDialog")
        self.defDiskLocation = \
                     wtree.get_widget("initParamsDialog_def_disk_location")
        self.defSnapshotLocation = \
                     wtree.get_widget("initParamsDialog_def_snapshot_location")

        if not InitParamsDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_initParamsDialog_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_initParamsDialog_ok_button_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_initParamsDialog_DiskLocationButton_clicked",
                                 self.on_disk_location_button_clicked)
            wtree.signal_connect("on_initParamsDialog_SnapshotLocationButton_clicked",
                                 self.on_snapshot_location_button_clicked)
            InitParamsDialog.initialized = True
            
        self.init()
        self.show()

    def init(self):
        disk_path = self.config.get(XMConfig.PATHS,constants.prop_disks_dir)
        snapshot_path = self.config.get(XMConfig.PATHS,
                                        constants.prop_snapshots_dir)
        if disk_path is None: disk_path = ""
        if snapshot_path is None: snapshot_path = ""
        
        self.defDiskLocation.set_text(disk_path)
        self.defSnapshotLocation.set_text(snapshot_path)
        
    def show(self):
        """ Displays initialization params dialog"""
        self.dialog.show() 

    def on_cancel_button_clicked(self, widget):
       """  Cancel on initialization params """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  OK creation of dom """
       self.config.set(XMConfig.PATHS,constants.prop_disks_dir,
                              self.defDiskLocation.get_text())
       self.config.set(XMConfig.PATHS,constants.prop_snapshots_dir,
                              self.defSnapshotLocation.get_text())
       self.dialog.hide()

    def on_disk_location_button_clicked(self, widget):
        (res, dirname) = file_selection(self.managed_node,
                                       "Select default location for Dom disks",
                                        "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.defDiskLocation.set_text(dirname)

    def on_snapshot_location_button_clicked(self, widget):
        (res, dirname) = file_selection(self.managed_node,
                                        "Select default location for Dom snapshots",
                                        "select_folder", parentwin=self.dialog)
        
        if res and dirname:
            self.defSnapshotLocation.set_text(dirname)


class AddNodeDialog:
    """ Class that handles adding new managed node to the host"""
    
    initialized = False
    
    def __init__(self, wtree, client_config, left_nav):
        """ Constructor"""
        self.client_config = client_config
        self.left_nav = left_nav
        self.dialog = wtree.get_widget("AddNode")
        self.hostname = wtree.get_widget("add_node_hostname")
        self.creds_helper = CredentialsHelper(wtree)

        #self.protocol = wtree.get_widget("add_node_protocol")
        self.xen_port = wtree.get_widget("add_node_port")
        self.username = wtree.get_widget("add_node_username")
        self.password = wtree.get_widget("add_node_password")

                
        if not AddNodeDialog.initialized :
            # setup handlers
            wtree.signal_connect("on_add_node_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_add_node_okbutton_clicked",
                                 self.on_ok_button_clicked)
            AddNodeDialog.initialized = True
            
        self.init()
        self.show()

    def init(self):
        self.hostname.set_text("")
        #self.protocol.set_active(1)
        self.xen_port.set_text(str("8005"))
        self.username.set_text("")
        self.password.set_text("")

        
    def show(self):
        """ Displays add node dialog"""
        self.dialog.show() 

    def on_cancel_button_clicked(self, widget):
       """  Cancel on add node dialog """
       self.init()
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok  button on add node """
       # validate parameters
       if self.hostname.get_text() == "":
           showmsg("Please enter valid host name")
           return
       if self.xen_port.get_text() == "":
           showmsg("Please enter valid xen port")
           return 
       remote = is_host_remote(self.hostname.get_text())
       if not remote:
           showmsg("Local host can not be added")
           return
       # create a new node
       #selected_protocol_text = self.protocol.get_active_text()
       
       node = XenNode(hostname = self.hostname.get_text(),
                      username = self.username.get_text(),
                      password = self.password.get_text(),
                      isRemote = remote, 
                      protocol = 'ssh_tunnel',
                      tcp_port = self.xen_port.get_text(),
                      helper = self.creds_helper)
                      

       # write the param to the conf file.
       self.client_config.setHostProperty(constants.prop_xen_port,
                                          self.xen_port.get_text(),
                                          self.hostname.get_text())
       self.client_config.setHostProperty(constants.prop_login,
                                          self.username.get_text(),
                                          self.hostname.get_text())
       self.left_nav.add_node(node)
       
       # username and password are not stored

       # clear the passwords from memory.
       self.init()
       self.dialog.hide()


class Cred:
    def __init__(self, username, password):
        self.username = username
        self.password = password

    def get_username(self):
        return self.username
    
    def get_password(self):
        return self.password
    

class CredentialsHelper:

    def __init__(self,wtree):
        self.wtree = wtree
    
    
    def get_credentials(self, hostname, username=None):
        dlg = CredentialsDialog(wtree)
        dlg.show(hostname, username)
        ret = dlg.dialog.run()
        cred = None
        if ret == gtk.RESPONSE_OK:
            cred = Cred(dlg.username.get_text(), dlg.password.get_text())
        dlg.init()
        return cred

class CredentialsDialog:
    """ Class that handles events from collecting credentials"""
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("Credentials")
        self.username = wtree.get_widget("cred_username")
        self.password = wtree.get_widget("cred_password")

        if not CredentialsDialog.initialized:
            # setup handlers
            wtree.signal_connect("on_cred_cancelbutton_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_cred_okbutton_clicked",
                                 self.on_ok_button_clicked)
            CredentialsDialog.initialized = True
        self.init()

    def init(self):
        self.username.set_text("")
        self.password.set_text("")
        
    def show(self, hostname, username = None):
        """ Displays initialization params dialog"""
        if username is not None:
            self.username.set_text(username)
        self.dialog.set_title(self.dialog.get_title() + " for " + hostname)
        self.dialog.show()

    def on_cancel_button_clicked(self, widget):
       """  Cancel on initialization params """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Cancel creation of dom """
       self.dialog.hide()

    

class RemoteFileDialog:
    """ Simple dialog box to take file or path from the user """
    initialized = False
    def __init__(self, wtree):
        """ Constructor"""
        self.dialog = wtree.get_widget("RemoteFileDialog")
        self.filename = wtree.get_widget("remote_file_filename")
        self.file_selector = wtree.get_widget("remote_file_selector")
        self.file_label = wtree.get_widget("remote_file_label")
        self.hostname = wtree.get_widget("remote_file_hostname")
        self.frame_label = wtree.get_widget("remote_file_frame_title")
        if not RemoteFileDialog.initialized:
            # setup handlers
            wtree.signal_connect("on_remote_file_cancel_button_clicked",
                                 self.on_cancel_button_clicked)
            wtree.signal_connect("on_remote_file_okbutton_clicked",
                                 self.on_ok_button_clicked)
            wtree.signal_connect("on_remote_file_selectorButton_clicked",
                                 self.on_file_selector_clicked)
            RemoteFileDialog.initialized = True
            
        self.init()

    def init(self):
        self.filename.set_text("")

    def show(self, managed_node,
             title, fstype, init_folder,
             init_file, parentwin):

        self.managed_node = managed_node
        self.title = title
        self.fstype = fstype
        self.init_folder = init_folder
        self.init_file = init_file
        self.parentwin = parentwin

        self.hostname.set_text(managed_node.hostname)

        if self.fstype == "select_folder":
            self.file_label.set_text("Folder name")
            self.frame_label.set_markup("<b>Specify Folder </b>")
        else:
            self.file_label.set_text("Filename")
            self.frame_label.set_markup("<b>Specify Filename </b>")
            
        self.dialog.set_title(title)

        # choose decent default
        if init_folder is not None and init_file is not None:
            # remove trailing / from path
            if init_folder.rfind("/") == len(init_folder) - 1:
                init_folder = init_folder[0:len(init_folder) -1]
            # remove leading / from filename
            if init_file.find("/") == 0: 
                init_file = init_file[1:]
            self.filename.set_text(init_folder + "/" + init_file)
            
        elif init_folder is not None :
            self.filename.set_text(init_folder)
        elif init_file is not None:
            self.filename.set_text(init_file)

    def on_cancel_button_clicked(self, widget):
       """  Cancel on RemoteFileDialog  """
       self.dialog.hide()
       
    def on_ok_button_clicked(self, widget):
       """  Ok on RemoteFileDialog """
       self.dialog.hide()

    def on_file_selector_clicked(self, widget):
       """  file selector clicked on RemoteFileDialog """
       # force the file selection to local/mounted files.
       (res, selection) = file_selection(self.managed_node,
                                         self.title,
                                         self.fstype,
                                         self.init_folder, self.init_file,
                                         self.parentwin,
                                         force_local = True)
       
       if res and selection:
           self.filename.set_text(selection)

    def run(self):
        return self.dialog.run()

    def get_filename(self):
        return self.filename.get_text()


#############
#    Util functions.. may move to util OR Managed Node
#    Also needs to cut this over to remote api
#



def createVBD(managed_node, filename, size=2048, backup=False):
    """Creates a file Virtual Block Device called 'name'
    at 'location'. 'size' is specified in kB.
    RAISES: IOError, OSError """

    if managed_node.node_proxy.file_exists(filename):
        if backup:
            managed_node.node_proxy.rename(filename,filename+'.bak')
        else:
            managed_node.node_proxy.remove(filename)

    # create the new disk block
    fd = managed_node.node_proxy.open(filename,'w')
    off = size * 1024L**2
    fd.seek(off, 0)
    fd.write('\x00')
    fd.close()


def quickcreateDom(managed_node,
                   name, mem, vbdlocation, vbdsize=2048, uselvm=False,
                   image_store = None, image_name = None):
    """Creates a new dom configuration file using default
    kernel and ramdisk attributes and starts the corresponding
    DomU instance.
    RAISES: IOError, OSError"""

    # check that everything is in order
    kernel, ramdisk = validateConfig(managed_node, managed_node.config, image_store, image_name)

    # create a new domfile object
    dom_filename = os.path.join(managed_node.config.get(XMConfig.PATHS,constants.prop_xenconf_dir),name)
    domfile = DomConfig(managed_node)
 
    # set domfile attributes
    domfile['name'] = name
    domfile['kernel'] = kernel #managed_node.config.get(XMConfig.PATHS,constants.prop_kernel)
    domfile['ramdisk'] = ramdisk #managed_node.config.get(XMConfig.PATHS,constants.prop_ramdisk)
    domfile['memory'] = int(mem)
    domfile['vif'] = ''
    domfile['on_reboot'] = 'destroy'
    domfile['on_crash'] = 'destroy'
    
    if not uselvm :
        # create VBD and set corresponding attribute in domfile
        filename = "%s/%s.disk.xm" % (vbdlocation, name)
        createVBD(managed_node,filename, vbdsize)
        if domfile['disk']:
            domfile['disk'].append('file:%s,hda,w' % filename)
        else:
            domfile['disk'] = ['file:%s,hda,w' % filename]
    else:
        # create a new logical volume in the 'vbdlocation' volume group.
        if(managed_node.lvm_proxy.createLogicalVolume(name+'.disk.xm',vbdsize,  vbdlocation)):
            if domfile['disk']:
                domfile['disk'].append('phy:/dev/%s/%s.disk.xm,hda,w'%(vbdlocation,name))
            else:
                domfile['disk'] = ['phy:/dev/%s/%s.disk.xm,hda,w' %(vbdlocation,name)]
        else:
            raise OSError('could not create logical volume')

    # commit domfile to disk and start.w/ default ramdisk
    # TODO: avoid writing the domfile here and try to do the
    #       first launch w/ kernel & ramdisk passed in
    #       directly.

    domfile.save(dom_filename)
    newdom = managed_node.create_dom(domfile)

    bldr = image_store.getImage(image_name)[2]
    if bldr != '':
        # bootloader specified        _
        domfile['kernel'] = ''
        domfile['ramdisk']= ''
        domfile['bootloader'] = bldr #managed_node.config.get(XMConfig.PATHS,constants.prop_bootloader)
        domfile.write()

def cleanupQCDomain(managed_node, name, vbdlocation=''):
    """ Delete the xen configuration file and associated
    disk devices created during a quickcreate"""
    domfilename = os.path.join(managed_node.config.get(XMConfig.PATHS,constants.prop_xenconf_dir),name)
    if managed_node.node_proxy.file_exists(domfilename):
        managed_node.node_proxy.remove(domfilename)

    if name in managed_node.get_dom_names():
        #_RESTRUCT
        for file in managed_node.get_dom(name).get_config().getDisks():
            if file.type is 'lvm':
                managed_node.lvm_proxy.removeLogicalVolume(file.filename)
                print 'deleting: ' + file.filename
            elif file.type is 'file':
                if managed_node.node_proxy.file_exists(file.filename):
                    print 'deleting: ' + file.filename
                    managed_node.node_proxy.remove(file.filename)
    else:
        print "Couldn't find the domain %s. Skipping deletion" % name


def validateConfig(managed_node,config, image_store, image_name):
    """Checks for existance of all critical paths and
    files specified in the DEFAULT's section the config file.
    Creates directories or fetches kernel/ramdisk from staging
    location if necessary."""

 
    # check xenman cache directory
    if config.get(XMConfig.PATHS,constants.prop_cache_dir):
        if not managed_node.node_proxy.file_exists(config.get(XMConfig.PATHS,constants.prop_cache_dir)):
            try:
                managed_node.node_proxy.mkdir(config.get(XMConfig.PATHS,constants.prop_cache_dir))
            except OSError:
                raise Exception('Cannot create default cache directory')
    else:
        raise Exception("Invalid Configuration File: 'cache_dir' not set")

    # get filenames associated w/ the image.
    kernelfile, ramdiskfile = image_store.getFilenames(image_name)
    
    if managed_node.isRemote:
        # check if the files corresponding to the image are
        # already present in the remote cache location.
        cache_dir = os.path.join(config.get(XMConfig.PATHS,constants.prop_cache_dir),
                                 image_name)
        if managed_node.node_proxy.file_exists(cache_dir+'/vmlinuz.default') and \
            managed_node.node_proxy.file_exists(cache_dir+'/initrd.default'):
            # files already present on remote host.
            pass
        else:
            # files need to be transfered
            kernel_dest = cache_dir+'/vmlinuz.default'
            ramdisk_dest = cache_dir+'/initrd.default'
            if not managed_node.node_proxy.file_exists(cache_dir):
                managed_node.node_proxy.mkdir(cache_dir)
            managed_node.node_proxy.put(kernelfile,kernel_dest)
            managed_node.node_proxy.put(ramdiskfile,ramdisk_dest)
            kernelfile, ramdiskfile = kernel_dest, ramdisk_dest

    return kernelfile, ramdiskfile
            

#   if managed_node.isRemote:
#       # temporary local destination for remote host
#       download_dir = '/tmp'
#   else:
#       # final destination for localhost
#      download_dir = config.get(XMConfig.PATHS,constants.prop_cache_dir)
        
            
    # check default kernel
#    if not config.get(XMConfig.PATHS,constants.prop_kernel):
#        # default kernel not set. get it from the staging area
#        kernel = os.path.join(config.get(XMConfig.PATHS,constants.prop_cache_dir),
#                              'vmlinuz.default')
#        download_file = os.path.join(download_dir,'vmlinuz.default') 
#        try:
#            utils.fetchImage(config.get(XMConfig.PATHS,constants.prop_staging_path_kernel),
#                       download_file)
#            # if d/l'd for a remote host, copy it over
#            if managed_node.isRemote:
#                managed_node.node_proxy.put(download_file,kernel)
#            # set the fetched image as the default kernel
#            config.set(XMConfig.PATHS,constants.prop_kernel,kernel)
#        except:
#            raise Exception('Invalid Staging Area in Configuration: Cannot fetch kernel image')


    # check default ramdisk
#    if not config.get(XMConfig.PATHS,constants.prop_ramdisk):
#        # default ramdisk not set. get it from the staging area
#        ramdisk = os.path.join(config.get(XMConfig.PATHS,constants.prop_cache_dir),
#                               'initrd.img.default')
#        download_file = os.path.join(download_dir,'initrd.img.default') 
#        try:
#            utils.fetchImage(config.get(XMConfig.PATHS,constants.prop_staging_path_ramdisk),
#                       download_file)
#            # if d/l'd for a remote host, copy it over
#            if managed_node.isRemote:
#                managed_node.node_proxy.put(download_file,ramdisk)
#            # set the fetched image as the default ramdisk
#            config.set(XMConfig.PATHS,constants.prop_ramdisk, ramdisk)
#        except:
#            raise Exception('Invalid Staging Area in Configuration: Cannot fetch ramdisk image')
#
#    return True



########## UI Utils #################

def showmsg(msg):
    """Displays the status message passed as an argument in a popup window"""

    wtree.get_widget('StatusMsg').set_text(msg)
    dialog = wtree.get_widget('StatusDialog')
    dialog.run()
    dialog.hide()

def confirmation(msg):
    """Displays a confirmation dialog message and returns a boolean
    representing the result"""

    wtree.get_widget('ConfirmationDialog').set_title("Confirm")
    wtree.get_widget('ConfirmationMsg').set_text(msg)
    dialog = wtree.get_widget('ConfirmationDialog')
    res = dialog.run() == gtk.RESPONSE_YES
    dialog.hide()
    return res

def file_selection(managed_node,
                   title, fstype,
                   init_folder=None, init_file=None,
                   parentwin=None, force_local = False):
    """This function will setup a file selection dialogue and return a tuple
    consisting of a boolean which is true if the user selected a filename,
    followed by the filename. fstype is either "open" or "save" or
    "select_folder" and title is whateveryou want displayed as the
    window's title bar. The init_folder and init_file params would be used
    as initial values for the dialog box"""

    gnome_vfs_enabled = True
    if main_context.has_key("client_config"):
        client_config = main_context["client_config"]
        gnome_vfs_prop_val = client_config.get(XMConfig.CLIENT_CONFIG,
                                              constants.prop_gnome_vfs_enabled)
        if gnome_vfs_prop_val is not None:
             gnome_vfs_enabled = eval(gnome_vfs_prop_val)
             
    if gnome_vfs_enabled:
        back_end = 'gnome-vfs'
    else:
        back_end = None

    if back_end is None and managed_node.is_remote() and not force_local:
        remote_dlg = RemoteFileDialog(wtree)
        remote_dlg.show(managed_node, title, fstype, init_folder, init_file,
                    parentwin)
        ret = remote_dlg.run()
        res = ret == gtk.RESPONSE_OK
        filename = remote_dlg.get_filename()
        return (res, filename)


    if fstype == "save":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_SAVE,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_SAVE,
                                                 gtk.RESPONSE_OK),
                                        backend=back_end)
    elif fstype == "open":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK),
                                        backend = back_end)
    elif fstype == "select_folder":
        filesel = gtk.FileChooserDialog(title,
                                        action=gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                        buttons=(gtk.STOCK_CANCEL,
                                                 gtk.RESPONSE_CANCEL,
                                                 gtk.STOCK_OPEN,
                                                 gtk.RESPONSE_OK),
                                        backend= back_end)

    # enable remote file browsing.    
    filesel.set_local_only(False)

    # if remote node, create a url and set it.
    if managed_node.is_remote() and not force_local:
        hostname = managed_node.hostname
        username = managed_node.username
        if username is None:
            username = os.getlogin() 
        url = "ssh://"+username+"@"+hostname+"/"
        title = filesel.get_title()
        filesel.set_title(title + ":" + username +"@"+ hostname)
        if init_folder is not None and init_folder != "":
            if init_folder[0] == '/':
                url = url + init_folder[1:]
                filesel.set_current_folder_uri(url)
        else:
            filesel.set_current_folder_uri(url)
    else: # local node
        if init_folder is not None and init_folder is not "":
            filesel.set_current_folder(init_folder)
            
    if init_file is not None:
        filesel.set_current_name(init_file)
    
    if parentwin:
        filesel.set_transient_for(parentwin)
        filesel.set_position(gtk.WIN_POS_CENTER_ON_PARENT)

    res = filesel.run() == gtk.RESPONSE_OK

    if res:
        filename = filesel.get_filename()
        if managed_node.is_remote() and not force_local:
            selected_uri = filesel.get_uri()
            selected_uri = urllib.unquote(selected_uri)
            if selected_uri and selected_uri.find(hostname) == -1:
                msg = "File not selected from the managed node.\n"
                msg += "Please use Open Location and use ssh://"+username+"@" +hostname +"/ to access remote files.\n"
                msg += "Open Location available through right click menu."
                showmsg(msg)
                filename = None
            else:
                filename = get_filename_from_uri(selected_uri)
    else:
        filename = None

        
    filesel.destroy()


    return (res, filename)

def get_filename_from_uri(uri):
    if uri == None:
        return None
    # Kludge : as urllib functions require http
    if uri.find("ssh") == 0:
        uri =  uri.replace("ssh", "http", 1)
    (prot, host, path,param,query,fragments ) = urlparse.urlparse(uri)
    if path is not None:
        return path


    
def checkAndSetDefaultPaths(wtree, managed_node):
    """ Check if the default path for snapshot and disk location are set."""
    disk_prop = managed_node.config.get(XMConfig.PATHS,
                                        constants.prop_disks_dir)
    snapshot_prop = managed_node.config.get(XMConfig.PATHS,
                                            constants.prop_snapshots_dir)
    if disk_prop is None or snapshot_prop is None:
        # give user opportunity to fix the properties.
        InitParamsDialog(wtree,managed_node).dialog.run()




### Module initialization

for path in (os.path.dirname(sys.argv[0]),'.', '/usr/share/xenman', '.'):
    filename = path + '/xenman.glade'
    if os.path.exists(filename):
        gladefile = filename
        break
else:
    print "ERROR: Couldn't find glade interface definition file!"
    sys.exit(1)   # bad, but ok for now.



paused_pb   = gtk.gdk.pixbuf_new_from_file(path +\
                                           "/pixmaps/small_pause.png")
resident_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                           "/pixmaps/small_started_state.png")
not_resident_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                               "/pixmaps/small_shutdown.png")

node_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                       "/pixmaps/small_node.png")

pool_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                       "/pixmaps/small_pool.png")

unknown_pb = gtk.gdk.pixbuf_new_from_file(path + \
                                          "/pixmaps/small_unknown_state.png")

wtree = gtk.glade.XML(gladefile)

# KLUDGE : way to share global context
main_context = {}


####
if __name__ == '__main__':
    helper = CredentialsHelperDialog(wtree)
    cred = helper.get_credentials("foobar", "root")
    if cred:
        print cred.get_username(), cred.get_password()
    
