/*
SMS Server Tools
Copyright (C) 2000 Stefan Frings

This program is free software unless you got it under another license directly
from the author. You can redistribute it and/or modify it under the terms of 
the GNU General Public License as published by the Free Software Foundation.
Either version 2 of the License, or (at your option) any later version.

http://www.isis.de/members/~s.frings
mailto:s.frings@mail.isis.de
*/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <time.h>
#include <mm.h>
#include <fcntl.h>
#include "extras.h"
#include "locking.h"
#include "smsd_cfg.h"
#include "stats.h"
#include "version.h"
#include "blacklist.h"
#include "logging.h"

int thread_id;     // main thread=-1, all others=device id.
int logfilehandle; // handle of log file.


void stop_if_file_exists(char* filename)
{
  int datei;
  datei=open(filename,O_RDONLY);
  if (datei>=0)
  {
    close(datei);
    writelogfile(LOG_CRIT,"Fatal error: cannot move or delete %s. Check file permissions.",filename);
    kill(0,SIGTERM);
  }  
}



/* =======================================================================
   Reads an SMS file and splits it into many shorter parts
   ======================================================================= */

int getSMSdata(char* filename, /* Filename */
               char* To, /* destination number */
	       char* From, /* sender name or number */
	       char* Text, /* the selected part of the text */
	       int* Binary, /* 1 is the content is marked as binary */
	       int part, /* this selects the part (beginning with 1) */
	       int size) /* Size of the part. Alls parts must be the same size */
{
#define maxlen 2048
  FILE* File;
  char line[300];
  char* sms;
  char binary[30];
  int character=0;
  int startpos=0;
  int emptyline=0;
  To[0]=0;
  From[0]=0;
  Text[0]=0;
  *Binary=0;  
  File=fopen(filename,"r");
  if (File)
  { 
    sms=calloc(1,maxlen);
    sms[0]=0;
    while (fgets(line,sizeof(line),File))
    {
//      cutspaces(line);
      if (emptyline)
      {	
        if ((character+strlen(line))<maxlen)
        {
	  int i;
	  for (i=0; i<strlen(line); i++)
	    if (line[i]=='\'') 
	      line[i]='\"';
          strcat(sms,line);	
          character+=strlen(line);
        }
      }	
      else
      {
        if ((line[0]==0) || (line[0]=='\n') || (line[0]=='\r'))
        emptyline=1;
        if (strstr(line,"To:")==line) 
        { 
          if (strstr(line+3,"00")==(line+3))
            strcpy(To,line+5);
  	  else if (strchr(line+3,'+')==(line+3))
	    strcpy(To,line+4);
	  else  
            strcpy(To,line+3);
	  cutspaces(To);
        }
        else if (strstr(line,"From:")==line)
        {
          strcpy(From,line+5);
	  cutspaces(From);
        }
        else if (strstr(line,"Binary:")==line)
        {
          strncpy(binary,line+8,30);
	  cutspaces(binary);
	  if ((strcasecmp(binary,"yes")==0) || (strcasecmp(binary,"true")==0) || (strcasecmp(binary,"1")==0))
	    *Binary=1;
	}
      }
    }
    fclose(File);
    if (*Binary==1)
    {
      Text[0]=0;
      free(sms);
      if (part>1)
        return 0;
      else
        return 1;
    }
    else
    {
      if (character>size)
        startpos=(size-3)*(part-1);
      else
        startpos=size*(part-1);	
      if ((startpos>=character) && (character>0))
      {
        free(sms);
        return 0;
      }
      else if (character>size)
      {
        sprintf(Text,"%i ",part);
        strncat(Text,sms+startpos,size-3);
        strcat(Text,"\x00");
      }
      else
        strcpy(Text,sms);
      free(sms);
      return 1;
    }
  }
  else return 0;
}

/* =======================================================================
   Mainspooler (sorts SMS into queues)
   ======================================================================= */

void mainspooler()
{
  char filename[300];
  char to[100];
  char from[100];
  char text[300];
  char directory[300];
  char cmdline[500];
  int queue;
  int binary;
  int i;
  int black;
  writelogfile(LOG_INFO,"Task for main queue has started.");
  while (1)
  {
    if (getfile(d_spool,filename))
    {
      if (getSMSdata(filename,to,from,text,&binary,1,160))
      {
        black=inblacklist(to);
        if ((queue=getqueue(to,directory)>=0) && (black==0))
	{
	  movefile(filename,directory);
	  stop_if_file_exists(filename);
          writelogfile(LOG_INFO,"Moved SMS from %s to %s to queue %s",from,to,queues[queue].name);
	}  
	else
	{
	  if (black)
	    writelogfile(LOG_NOTICE,"Invalid To: field in message file %s (blacklist).",filename);	  
	  else
	    writelogfile(LOG_NOTICE,"Invalid To: field in message file %s (unknown provider).",filename);
	  rejected_counter++;
	  if (eventhandler[0])
	  {
            sprintf(cmdline,"%s %s %s",eventhandler,"FAILED",filename);
            my_system(cmdline);
	  }	      
	  movefile(filename,d_failed);
	  stop_if_file_exists(filename);
	}
      }
      else
        ; /* File ist nicht lesbar, einfach etwas warten und erneut versuchen */      
    }
    else
    {
      for (i=0; i<delaytime; i++)
      {
	print_status();
	checkwritestats();	
        sleep(1);
      }
    }  
  }
}

/* =======================================================================
   Device-Spooler (one for each modem)
   ======================================================================= */

int sendsms(int device, char* from, char* to, char* text, int binary, char* filename)
/* binary can be 0=ascii  or 1=binary */
/* filename is only used in binary mode */
{
  int result;
  char cmdline[500];
  time_t start_time=time(0);
  statistics[device]->status='s';  
  writelogfile(LOG_INFO,"Sending SMS from %s to %s on device %s",from,to,devices[device].name);
  
    if (debug)
    {
      printf("----------\n%s\n----------\n",text);
      result=0;    
    }
    else
    {
      sprintf(cmdline,"%s/putsms -l%i -L%i -n%s -b%i -e%i -d%s",mypath,logfilehandle,loglevel,devices[device].name,devices[device].baudrate,errorsleeptime,devices[device].device);
      // For Windows: 
      // sprintf(cmdline,"%s/putsms.exe -l%i -L%i -n%s -b%i -e%i -d%s",mypath,logfilehandle,loglevel,devices[device].name,devices[device].baudrate,errorsleeptime,devices[device].device);
      
      if (devices[device].smsc[0])
      {
        strcat(cmdline," -s");
	strcat(cmdline,devices[device].smsc);
      }
      if (devices[device].mode[0])
      {
        strcat(cmdline," -m");
	strcat(cmdline,devices[device].mode);
      }
      if (devices[device].pin[0])
      {
        strcat(cmdline," -p");
	strcat(cmdline,devices[device].pin);
      }
      if (devices[device].cs_convert)
        strcat(cmdline," -c");
      if (devices[device].report)
        strcat(cmdline," -r");
      if (devices[device].initstring[0])
      {
        strcat(cmdline," -i");
	strcat(cmdline,devices[device].initstring);
      }	
      if (binary==0)
      {
        strcat(cmdline," -- ");
        strcat(cmdline,to);
        strcat(cmdline," '");
	strcat(cmdline,text);
	strcat(cmdline,"'");
      }
      else
      {
        strcat(cmdline," -F");
	strcat(cmdline,filename);
      }
      writelogfile(LOG_DEBUG,"Calling: %s\n",cmdline);
    
      result=my_system(cmdline);
    }
  
  statistics[device]->usage_s+=time(0)-start_time;
  if (result==0)
  {
    writelogfile(LOG_INFO,"Sending SMS to %s was successful.",to);
    statistics[device]->succeeded_counter++;
    return 1;
  }
  else
  {
    writelogfile(LOG_WARNING,"Sending SMS to %s failed",to);
    statistics[device]->failed_counter++;
    return 0;
  }
}

int receivesms(int device) /* returns 1 if successful */
{
  char cmdline[500];
  char filename[300];
  int result;
  char newname[300];
  char* cp;
  int start_time=time(0);
  statistics[device]->status='r';
  writelogfile(LOG_INFO,"Checking device %s for incoming SMS",devices[device].name);
  sprintf(filename,"/var/tmp/%s.XXXXXX",devices[device].name);
  close(mkstemp(filename));
  sprintf(cmdline,"%s/getsms -l%i -L%i -n%s -b%i -e%i -d%s -s%s -o%s",mypath,logfilehandle,loglevel,devices[device].name,devices[device].baudrate,errorsleeptime,devices[device].device,devices[device].name, filename);
  // For Windows: 
  // sprintf(cmdline,"%s/getsms.exe -l%i -L%i -n%s -b%i -e%i -d%s -s%s -o%s",mypath,logfilehandle,loglevel,devices[device].name,devices[device].baudrate,errorsleeptime,devices[device].device,devices[device].name, filename);  
  if (devices[device].mode[0])
  {
    strcat(cmdline," -m");
    strcat(cmdline,devices[device].mode);
  }
  if (devices[device].pin[0])
  {
    strcat(cmdline," -p");
    strcat(cmdline,devices[device].pin);
  }
  if (devices[device].cs_convert)
    strcat(cmdline," -c");
  if (devices[device].initstring[0])
  {
    strcat(cmdline," -i");
    strcat(cmdline,devices[device].initstring);
  }	
  if (debug)
    strcat(cmdline," -k");
  writelogfile(LOG_DEBUG,"Calling: %s\n",cmdline);
  
  result=my_system(cmdline); 
  statistics[device]->usage_r+=time(0)-start_time;
  if ((result!=0) && (result!=5))
  {
    unlink(filename);
    return 0;
  }
  else 
  {
    movefile(filename,d_incoming);
    stop_if_file_exists(filename);
    statistics[device]->received_counter++;
    if (eventhandler[0] || devices[device].eventhandler[0])
    {
      cp=strrchr(filename,'/');
      if (cp)
	sprintf(newname,"%s%s",d_incoming,cp);
      else
        sprintf(newname,"%s/%s",d_incoming,filename);    
      if (result==5) //Status Report
      {
        if (devices[device].eventhandler[0])
          sprintf(cmdline,"%s %s %s",devices[device].eventhandler,"REPORT",newname);
        else
          sprintf(cmdline,"%s %s %s",eventhandler,"REPORT",newname);   
      }
      else // SMS Received
      {
        if (devices[device].eventhandler[0])
          sprintf(cmdline,"%s %s %s",devices[device].eventhandler,"RECEIVED",newname);
        else
          sprintf(cmdline,"%s %s %s",eventhandler,"RECEIVED",newname);  
      }		
      my_system(cmdline);
    }
    return 1;
  }
}

void devicespooler(int device)
{
  char filename[300];
  char to[100];
  char from[100];
  char text[400];
  char directory[300];
  char lockfilename[300];
  char cmdline[500];
  int q,queue;
  int part;
  int errorcounter;
  int error;
  int workless;
  int max_len;
  int binary;
  errorcounter=0;
  writelogfile(LOG_INFO,"Task for device %s has started.",devices[device].name);

  while (1) /* endless loop */
  {
    do /* send as many SMS as available */
    {	
      workless=1;
      for (q=0; q<PROVIDER; q++)
      {
        if (devices[device].queues[q][0])
        {
          queue=getqueue(devices[device].queues[q],directory);
          if (queue>=0)
  	  {
            if (getfile(directory,filename))
	    {
              if (lockfile(filename))
              {
	        workless=0;
	        part=1;
	        if (strcmp(devices[device].mode,"ascii")==0)
	          max_len=140;
	        else
	          max_len=160;
  	        while (getSMSdata(filename,to,from,text,&binary,part,max_len))
	        {
		  if (!sendsms(device,from,to,text,binary,filename)) 
		  {
//		    movefile(filename,d_failed);
//		    stop_if_file_exists(filename);
		    errorcounter++;
		    error=1;
		    if (errorcounter>=3)
		    {
	  	      writelogfile(LOG_CRIT,"Fatal error: sending failed 3 times. Blocking %i sec.",blocktime);
		      statistics[device]->multiple_failed_counter++;
		      statistics[device]->status='b';
		      sleep(blocktime);
		      errorcounter=0;
		    }
		  }  
		  else
		    error=0;
		  part++;
	        }
		statistics[device]->status='i';
	        if (eventhandler[0] || devices[device].eventhandler[0])
                {
	          if (error==1)
	            strcpy(text,"FAILED");
	 	  else
		    strcpy(text,"SENT");
		  if (devices[device].eventhandler[0])  
  	            sprintf(cmdline,"%s %s %s",devices[device].eventhandler,text,filename);
		  else
  	            sprintf(cmdline,"%s %s %s",eventhandler,text,filename);
                  my_system(cmdline);
                }
  	        if (error==1)
	          movefile(filename,d_failed);
	        else
		{
		  if (d_sent[0])
		    movefile(filename,d_sent);
		  else
	            unlink(filename);
		}
		stop_if_file_exists(filename);
	        unlock(filename);
	      }
	    }     
	  }  
        }
        else
          break;
      }
    }     
    while (workless==0);  

    if (devices[device].incoming) /* receive one SMS if configured to do this */
      if (receivesms(device)==1)
        workless=0;

    if (workless==1) /* wait a little bit if there was no SMS to send or receive */
    {
      statistics[device]->status='i';
      sleep(delaytime);
    }  
  }
}


/* =======================================================================
   Task-Maker (creates threads for the modems and the main queue)
   ======================================================================= */

void taskmaker()
{
  int i;
  thread_id=-1;
  for (i=0; i<DEVICES; i++)
  {
    if (devices[i].name[0])
      if (fork()==0)
      {
      	thread_id=i;
        write_pid("/var/run/smsd.pid");
	devicespooler(i);
        remove_pid("/var/run/smsd.pid");      
      }
  }
  mainspooler();
}

void termination_handler (int signum)
{
  kill(0,SIGTERM);
  if (thread_id==-1)
  {
    savestats();
    MM_destroy();
  }    
  remove_pid("/var/run/smsd.pid");  
  _exit(0);
}

/* =======================================================================
   Main
   ======================================================================= */

int main(int argc,char** argv)
{
  write_pid("/var/run/smsd.pid");
  if (signal(SIGINT,termination_handler)==SIG_IGN)
    signal(SIGINT,SIG_IGN);
  if (signal(SIGHUP,termination_handler)==SIG_IGN)
    signal(SIGHUP,SIG_IGN);
  if (signal(SIGTERM,termination_handler)==SIG_IGN)
    signal(SIGTERM,SIG_IGN);
    
  parsearguments(argc,argv); 
  initcfg();
  readcfg();
  logfilehandle=openlogfile("smsd",logfile,LOG_DAEMON,loglevel);  
  if (debug)
    printcfg();
  initstats();
  loadstats();
  taskmaker();
}
