/*
 * ppm2raw.cc -- Converts PPM streams to Raw DV
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <unistd.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <PPMDVFileInput.h>
#include <Threader.h>
#include <frame.h>
#include <time.h>
#include <pthread.h>

class PPM2Raw : public DVEncoderParams, public Threader
{
	private:
		bool ppm_output;
		bool ppm_preview;
		int pump_size;
		int dropper;

	public: 
		bool aborted;
		GdkInterpType scaler;
	
		PPM2Raw( ) : 
			ppm_output( false ), 
			ppm_preview( false ), 
			pump_size( 25 ), 
			dropper( 1 ), 
			aborted( false ), 
			scaler( GDK_INTERP_HYPER )
		{
		}

		virtual ~PPM2Raw( )
		{
		}

		virtual string LogId( )
		{
			return "PPM2Raw";
		}

		void SetPPMOutput( bool _ppm_output )
		{
			ppm_output = _ppm_output;
		}

		void SetPPMPreview( bool _ppm_preview )
		{
			ppm_preview = _ppm_preview;
		}
	
		void SetPumpSize( int _pump_size )
		{
			pump_size = _pump_size;
		}

		void SetDropper( int _dropper )
		{
			dropper = _dropper;
		}

		void SetScaler( int interp )
		{
			switch( interp % 4 )
			{
				case 0:
					scaler = GDK_INTERP_NEAREST;
					break;
				case 1:
					scaler = GDK_INTERP_TILES;
					break;
				case 2:
					scaler = GDK_INTERP_BILINEAR;
					break;
				case 3:
					scaler = GDK_INTERP_HYPER;
					break;
			}
		}
		
	protected:
		void Thread( )
		{
			bool running = true;
			
			PPMDVFileInput input( *this );
			input.SetPumpSize( pump_size );
			input.SetFile( stdin );
			input.SetPPMOutput( ppm_output );
			input.SetPPMPreview( ppm_preview );
			input.SetDropper( dropper );
			input.scaler = scaler;

			input.ThreadStart( );

			while( running && ThreadIsRunning( ) )
			{
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );
					if ( !ppm_output )
					{
						running = fwrite( frame.data, frame.GetFrameSize( ), 1, stdout ) != 0;
						aborted = !running;
					}
					fflush( stdout );
					input.QueueInputFrame( );
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}

			input.ThreadStop( );
		}
};

static void Usage( )
{
	cerr << "Usage: ppm2raw [ options ]" << endl;
	cerr << "Where: options are" << endl;
	cerr << "       -n        - images should be treated as NTSC" << endl;
	cerr << "       -a        - scale image and maintain aspect ratio" << endl;
	cerr << "       -s        - scale image and ignore aspect ratio" << endl;
	cerr << "       -w        - wide screen output (default: off)" << endl;
	cerr << "       -p number - encoder passes (default: 3)" << endl;
	cerr << "       -q qno    - encoder qno (default: 0)" << endl;
	cerr << "       -A wav    - wav file to use for dubbing (default: silence)" << endl;
	cerr << "       -f freq   - audio frequency (default: 48khz)" << endl;
	cerr << "       -b colour - background colour (default: 000000 [black])" << endl;
	cerr << "       -o        - bypass dv encoding and just pump out scaled ppms" << endl;
	cerr << "       -fbs size - Frame buffer size" << endl;
	cerr << "       -e n      - output every nth frame" << endl;
	cerr << "       -2        - 2 pass encoding (default: off)" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	PPM2Raw output;

	Diagnostics::SetApp( "ppm2raw" );

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	try
	{
		for ( int i = 1; i < argc && !output.aborted; i ++ )
		{
			if ( !strcmp( argv[ i ], "-A" ) )
				output.SetAudioFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-fbs" ) )
				output.SetPumpSize( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-n" ) )
				output.SetPAL( false );
			else if ( !strcmp( argv[ i ], "-s" ) )
				output.SetScale( SCALE_FULL );
			else if ( !strcmp( argv[ i ], "-w" ) )
				output.SetWide( true );
			else if ( !strcmp( argv[ i ], "-a" ) )
				output.SetScale( SCALE_ASPECT_RATIO );
			else if ( !strcmp( argv[ i ], "-p" ) )
				output.SetEncodePasses( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-q" ) )
				output.SetStaticQno( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-f" ) )
				output.SetFrequency( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-e" ) )
				output.SetDropper( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-o" ) )
				output.SetPPMOutput( true );
			else if ( !strcmp( argv[ i ], "-preview" ) )
				output.SetPPMPreview( true );
			else if ( !strcmp( argv[ i ], "-scaler" ) )
				output.SetScaler( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-b" ) )
				output.SetBackgroundColour( argv[ ++ i ] );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-2" ) )
				output.SetTwoPassEncoding( true );
			else
				Usage( );
		}

		output.ThreadExecute( );
	}
	catch ( string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}
	catch ( const char *exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	exit( output.aborted );
}
