/*
 * Copyright (c) 1998, 1999 Phil Thompson (phil@river-bank.demon.co.uk)
 *
 * The SIP lexer.
 */

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "sip.h"
#include "parser.h"


#ifndef FLEX_SCANNER
#error "Only flex is supported at the moment"
#endif


#define	YY_FATAL_ERROR(s)	fatallex(s)

#define	MAX_INCLUDE_DEPTH	10


static struct inputFile {
	int		lineno;		/* The line number. */
	YY_BUFFER_STATE	bs;		/* The flex buffer state handle. */
	char		*name;		/* The file name. */
	parserContext	pc;		/* The parser context. */
} inputFileStack[MAX_INCLUDE_DEPTH];

static int currentFile = -1;		/* Index of the current input file. */

static FILE *openFile(char *);
static void fatallex(char *);
%}

%s code
%x ccomment

%%

^[ \t]*%Include			{return TK_INCLUDE;}
^[ \t]*%Import			{return TK_IMPORT;}
^[ \t]*%ImportWithTimeline	{return TK_IMPORTTL;}
^[ \t]*%Module			{return TK_MODULE;}
^[ \t]*%Timeline		{return TK_TIMELINE;}
^[ \t]*%SecondaryVersion	{return TK_SECVERSION;}
^[ \t]*%ExposeFunction		{return TK_EXPOSE;}
^[ \t]*%MappedType		{return TK_MAPPEDTYPE;}
^[ \t]*%If			{return TK_IF;}
<INITIAL>^[ \t]*%End		{return TK_END;}
class				{return TK_CLASS;}
struct				{return TK_STRUCT;}
public				{return TK_PUBLIC;}
protected			{return TK_PROTECTED;}
private				{return TK_PRIVATE;}
signals				{return TK_SIGNALS;}
slots				{return TK_SLOTS;}
PyMethods			{return TK_PYMETHODS;}
PyNumberMethods			{return TK_PYNUMMETHODS;}
PySequenceMethods		{return TK_PYSEQMETHODS;}
PyMappingMethods		{return TK_PYMAPMETHODS;}
char				{return TK_CHAR;}
bool				{return TK_BOOL;}
short				{return TK_SHORT;}
int				{return TK_INT;}
long				{return TK_LONG;}
float				{return TK_FLOAT;}
double				{return TK_DOUBLE;}
void				{return TK_VOID;}
virtual				{return TK_VIRTUAL;}
enum				{return TK_ENUM;}
unsigned			{return TK_UNSIGNED;}
const				{return TK_CONST;}
static				{return TK_STATIC;}
typedef				{return TK_TYPEDEF;}
namespace			{return TK_NAMESPACE;}
::				{return TK_SCOPE;}
SIP_SIGNAL			{return TK_SIPSIGNAL;}
SIP_RXOBJ_CON			{return TK_SIPRXCON;}
SIP_RXOBJ_DIS			{return TK_SIPRXDIS;}
SIP_SLOT_CON			{return TK_SIPSLOTCON;}
SIP_SLOT_DIS			{return TK_SIPSLOTDIS;}
SIP_ARRAY			{return TK_SIPARRAY;}
SIP_UARRAY			{return TK_SIPUARRAY;}
SIP_ARRAY_SIZE			{return TK_SIPARRAYSIZE;}
SIP_ARRAY_USIZE			{return TK_SIPARRAYUSIZE;}


[ \t] {				/* Ignore whitespace. */
	;
}

\n {				/* Maintain the line number. */
	++inputFileStack[currentFile].lineno;
}

\/\/.* {			/* Ignore C++ style comments. */
	;
}


-?[0-9]+ {			/* A signed decimal number. */
	yylval.number = strtol(yytext,NULL,0);
	return TK_NUMBER;
}


-?(([0-9]+)|([0-9]*\.[0-9]+)([eE][-+]?[0-9]+)?) {/* A floating point number. */
	yylval.real = strtod(yytext,NULL);
	return TK_REAL;
}


0x[0-9a-fA-F]+ {		/* An unsigned hexadecimal number. */
	yylval.number = strtol(yytext,NULL,16);
	return TK_NUMBER;
}


__[a-z]+__ {			/* The name of an internal Python method. */
	yylval.text = sipStrdup(yytext);
	return TK_PYMETHODNAME;
}                                                                               


[_A-Za-z][_A-Za-z0-9]* {	/* An identifier name. */
	yylval.text = sipStrdup(yytext);
	return TK_NAME;
}


[._A-Za-z][._A-Za-z0-9\-]* {	/* A filename. */
	yylval.text = sipStrdup(yytext);
	return TK_FILENAME;
}


\"[^"\n]*["\n] {		/* A double-quoted string. */
	char *dp, *sp;

	/* Copy the string without the quotes. */

	yylval.text = sipMalloc(strlen(yytext) + 1);

	dp = yylval.text;
	sp = yytext;

	while (*sp != '\0')
	{
		if (*sp != '"')
			*dp++ = *sp;

		++sp;
	}

	*dp = '\0';

	return TK_STRING;
}


\'[^'\n]*['\n] {		/* A single-quoted character. */
	if (strlen(yytext) != 3)
		fatallex("Exactly one character expected between single quotes");

	yylval.qchar = yytext[1];

	return TK_QCHAR;
}


\/\* {				/* Ignore C-style comments. */
	BEGIN ccomment;
}
<ccomment>\n {
	++inputFileStack[currentFile].lineno;
}
<ccomment>\*\/ {
	BEGIN INITIAL;
}
<ccomment>. {
	;
}


^%Copying {			/* The software license. */
	BEGIN code;
	return TK_COPYING;
}

^%ConvertFromTypeCode {		/* The start of a from-type code block. */
	BEGIN code;
	return TK_FROMTYPE;
}

^%CanConvertToTypeCode {	/* The start of a can-to-type code block. */
	BEGIN code;
	return TK_CANTOTYPE;
}

^%ConvertToTypeCode {		/* The start of a to-type code block. */
	BEGIN code;
	return TK_TOTYPE;
}

^%ConvertToSubClassCode {	/* The start of a to-sub-class code block. */
	BEGIN code;
	return TK_TOSUBCLASS;
}

^%VersionCode {			/* The start of a version code block. */
	BEGIN code;
	return TK_VERSIONCODE;
}

^%HeaderCode {			/* The start of a C++ header code block. */
	BEGIN code;
	return TK_HEADERCODE;
}

^%ExportedHeaderCode {		/* The start of an exported C++ header code block. */
	BEGIN code;
	return TK_EXPHEADERCODE;
}

^%C\+\+Code {			/* The start of a C++ code block. */
	BEGIN code;
	return TK_CPPCODE;
}

^%MemberCode {			/* The start of a C++ member code block. */
	BEGIN code;
	return TK_MEMBERCODE;
}

^%VirtualCode {			/* The start of a C++ virtual code block. */
	BEGIN code;
	return TK_VIRTUALCODE;
}

^%PrePythonCode {		/* The start of a pre-Python code block. */
	BEGIN code;
	return TK_PREPYCODE;
}

^%PythonCode {			/* The start of a Python code block. */
	BEGIN code;
	return TK_PYCODE;
}

^%Doc {				/* The start of a documentation block. */
	BEGIN code;
	return TK_DOC;
}

^%ExportedDoc {			/* The start of an exported documentation block. */
	BEGIN code;
	return TK_EXPORTEDDOC;
}

^%Makefile {			/* The start of a Makefile code block. */
	BEGIN code;
	return TK_MAKEFILE;
}

^%VariableCode {		/* The start of an access code block. */
	BEGIN code;
	return TK_VARCODE;
}

<code>^%End {			/* The end of a code block. */
	BEGIN INITIAL;
	return TK_END;
}

<code>^[^%]* {			/* The contents of a code block. */
	char *cp;
	struct inputFile *ifp;

	ifp = &inputFileStack[currentFile];

	yylval.codeb = sipMalloc(sizeof (codeBlock));

	yylval.codeb -> frag = sipStrdup(yytext);
	yylval.codeb -> linenr = ifp -> lineno;
	yylval.codeb -> filename = sipStrdup(ifp -> name);
	yylval.codeb -> next = NULL;

	/* Bump the line count. */

	for (cp = yytext; *cp != '\0'; ++cp)
		if (*cp == '\n')
			ifp -> lineno++;

	return TK_CODEBLOCK;
}

. {				/* Any else is returned as is. */
	return yytext[0];
}

%%

/*
 * Hook into EOF handling.  Return 0 if there is more to process.
 */

int yywrap()
{
	struct inputFile *ifp;

	ifp = &inputFileStack[currentFile--];

	/* Tell the parser if this is the end of a file. */

	parserEOF(ifp -> name,&ifp -> pc);

	/* Tidy up this file. */

	fclose(yyin);
	free(ifp -> name);

	/* See if this was the original file. */

	if (currentFile < 0)
		return 1;

	yy_delete_buffer(YY_CURRENT_BUFFER);
	yy_switch_to_buffer(ifp -> bs);

	return 0;
}


/*
 * Set up an input file to be read by the lexer, opening it if necessary.
 */

void setInputFile(FILE *fp,char *name,parserContext *pc)
{
	char *fullname;

	if (++currentFile >= MAX_INCLUDE_DEPTH)
		fatal("Too many nested %%Include, %%Import or %%ImportWithTimeline statements\n");

	if (fp != NULL || (fp = openFile(name)) != NULL)
		fullname = sipStrdup(name);
	else
	{
		stringList *sl;

		fullname = NULL;

		for (sl = includeDirList; sl != NULL; sl = sl -> next)
		{
			if (fullname != NULL)
				free(fullname);

			fullname = concat(sl -> s,"/",name,NULL);

			if ((fp = openFile(fullname)) != NULL)
				break;
		}

		if (fp == NULL)
			fatal("Unable to find file \"%s\"\n",name);
	}

	yyin = fp;

	inputFileStack[currentFile].lineno = 1;
	inputFileStack[currentFile].name = fullname;
	inputFileStack[currentFile].pc = *pc;

	if (currentFile > 0)
	{
		inputFileStack[currentFile].bs = YY_CURRENT_BUFFER;
		yy_switch_to_buffer(yy_create_buffer(yyin,YY_BUF_SIZE));
	}
}


/*
 * Open a file for reading or return NULL if it doesn't exist.  Any other error
 * is fatal.
 */

static FILE *openFile(char *name)
{
	FILE *fp;

	if ((fp = fopen(name,"r")) == NULL && errno != ENOENT)
		fatal("Error in opening file %s\n",name);

	return fp;
}


/*
 * Handle fatal yacc errors.
 */

void yyerror(char *s)
{
	if (currentFile < 0)
		fatal("Unexpected end of input\n");

	fatal("%s:%d: %s\n",
		inputFileStack[currentFile].name,
		inputFileStack[currentFile].lineno,
		s);
}


/*
 * Handle fatal lex errors.
 */

static void fatallex(char *s)
{
	fatal("%s:%d: Lexical analyser error: %s\n",
		inputFileStack[currentFile].name,
		inputFileStack[currentFile].lineno,
		s);
}
