//===============================================================
//
//		ExprParser		-- Written by Giovanni Dicanio
//
//		A C++ class to parse mathematical expressions.
//
//		(v. 0.3)
//
//
//		Copyright (c) by Giovanni Dicanio
//
//===============================================================



//---------------------------------------------------------------
// File:			exprparser.cpp
// Description:		Class main implementation file
// Author:			Giovanni Dicanio
// First date:		20th January 2000
// Last update:		1st February 2000
//---------------------------------------------------------------



//---------------------------------------------------------------
//						INCLUDE SECTION
//---------------------------------------------------------------

#include "exprparser.h"	// class header
#include <string.h>			// standard C strings managment
#include <stdlib.h>			// memory managment, atof(), etc.
#include <ctype.h>			// isdigit(), isalpha(), etc.
#include <assert.h>			// for debugging purposes



//---------------------------------------------------------------
//				STATIC DATA MEMBERS INITALISATION SECTION
//---------------------------------------------------------------


// This static character is pointed by m_currChar to specify
// that the parsing process must be interrupted, beacuse an
// error occurred (see setError() implementation for more info).
char ExprParser::stopParsingChar = '\0';


// Maximum number of variables that can be stored in a parser object.
int ExprParser::maxNumberOfVariables = EXPRPARSER_MAX_VARIABLES;

// Maximum length of a variable identifier string.
int ExprParser::maxVarIdentifierLength = EXPRPARSER_MAX_VARIABLE_IDENTIFIER_CHARS;

// Set 'pi' constant value.
const ExprParser::real ExprParser::pi = 3.1415926535897932384626433832795;

// Set 'e' constant value.
const ExprParser::real ExprParser::e = 2.71828182845904523536028747135266;



//---------------------------------------------------------------
//				MEMBER FUNCTIONS IMPLEMENTATION SECTION
//---------------------------------------------------------------

ExprParser::ExprParser()
{
	initParser(NULL);
}


ExprParser::ExprParser(const char * expr)
{
	initParser(expr);
}


ExprParser::ExprParser(const ExprParser & ep)
{
	initParser(NULL);
	copyFromParser(ep);
}


ExprParser::~ExprParser()
{
	destroyParser();
}


ExprParser & ExprParser::setExpression(const char * expr)
{
	// Copy new expression string in the parser object.
	stringDelete(&m_expr);
	m_expr = stringCreateFrom(expr);
	
	// Expression has been changed => we need reparsing!
	m_needParsing = true;

	return *this;
}


ExprParser::real ExprParser::getValue()
{
	if (!m_needParsing)
		return m_value;
	else 
	{
		doParsing();
		return m_value;		
	}
}


ExprParser & ExprParser::operator=(const ExprParser & ep)
{
	if (&ep != this)
	{
		destroyParser();		// free current data
		copyFromParser(ep);		// store new data
	}

	return *this;
}


bool ExprParser::copyFromParser(const ExprParser & ep)
{
	// Return true on OK; false on error.

	// NOTE:
	// Current parser data is trashed out!
	// Call destroyParser() before calling this function, if required!

	// Copy parser data.
	m_expr = stringCreateFrom(ep.m_expr);
	if (m_expr == NULL)
		return false;	// error: string copy failed!

	// Copy status member variables.
	m_value = ep.m_value;
	m_error = ep.m_error;
	m_needParsing = ep.m_needParsing;
	m_angleMode = ep.m_angleMode;

	// Copy variables from source parser.
	return copyAllVariablesFromParser(ep);
}


bool ExprParser::initParser(const char * expr)
{
	// Init current parser object.
	// Call destroyParser() before calling this function, if you
	// need to cleanup parser previous allocated resources.

	m_expr = stringCreateFrom(expr);
	if (m_expr == NULL)
		return false;		// error: memory allocation failed!

	m_error = OK;			// no error occurred
	m_needParsing = true;	// expression not yet parsed!

	m_freeVariableSlot = 0;	// no varible stored in the array

	m_angleMode = EXPRPARSER_DEFAULT_ANGLE_UNIT; // set default angle unit

	return true;			// initialisation has been successful
}


bool  ExprParser::destroyParser()
{
	// Free dynamics resources allocate by the parser.

	stringDelete(&m_expr);	// destroy expression string memory

	freeVariables();		// free variables

	return true;			// destruction has been successful
}


char * ExprParser::stringCreateFrom(const char * source)
{
	// If source is NULL, allocate a '\0' string.
	// If source is != NULL, allocate memory and copy source into it.

	if (source) 
	{
		// Get the number of characters in source string.
		int len = strlen(source);

		// Try allocating string memory.
		char * s = new char[len+1];	// +1 for '\0'
		if (!s)
			return NULL;	// error allocating string memory

		// Copy characters from source string.
		strcpy(s, source);

		// Return this new string.
		return s;

	}
	else 
	{
		// Create a '\0' string.

		char * s = new char[1];
		
		if (!s)
			return NULL;	// error allocating memory!
		
		*s = '\0';			// set to empty string
		
		return s;
	}
}


void ExprParser::stringDelete(char **ptrString)
{
	assert(ptrString);	// ptrString must be a valid pointer!

	if (*ptrString) 
	{
		free(*ptrString);	// free memory
		*ptrString = NULL;	// avoid dangling pointers
	}
}


bool ExprParser::doParsing()
{
	m_needParsing = true;

	if (!startParsing())			// startup the parsing process
		return false;

	m_value = doExpression(); // start parsing the expression

	if (m_error == OK) 
	{
		m_needParsing = false;
		return true;			// parsing has been successful
	}
	else
		return false;		// parsing error (see
}


bool ExprParser::startParsing()
{
	// Start with first character in the expression string.
	m_currChar = m_expr;

	// No error occurred.
	m_error = OK;

	// Error position not defined (-1).
	m_errorPos = -1;

	
	// Before doing parsing, check parenthese balancing.
	if (!parenthesesBalanced()) 
	{
		setError(PARENTHESES_NOT_BALANCED);
		return false;
	}
	else
		return true;
}


// Expression ::= <MulDiv> '+' <MulDiv>
// Expression ::= <MulDiv> '-' <MulDiv>
ExprParser::real ExprParser::doExpression()
{
	bool stop;
	real result;

	skipBlanks();
	result = doMulDiv();

	stop = false;
	while (!stop)
	{
		if (*m_currChar == '+')
		{	
			goAhead();
			skipBlanks();
			result += doMulDiv();
			stop = false;
		
		} 
		else if (*m_currChar == '-') 
		{
		
			goAhead();
			skipBlanks();
			result -= doMulDiv();
			stop = false;
		
		}
		else
			stop = true;
	}

	return result;
}


// MulDiv ::= <NumberFunction> '*' <NumberFunction>
// MulDiv ::= <NumberFunction> '/' <NumberFunction>
ExprParser::real ExprParser::doMulDiv()
{
	bool stop;
	real result;

	result = doNumberFunction();

	stop = false;
	while (!stop)
	{
		if (*m_currChar == '*')
		{
			goAhead();
			skipBlanks();
			result *= doNumberFunction();
			stop = false;
		}
		else if (*m_currChar == '/') 
		{
			goAhead();
			skipBlanks();
			
			real denominator = doNumberFunction();
			if (denominator == 0.0)			// better check would be: fabs(denominator) < eps
				setError(DIVIDE_BY_ZERO);
			else
				result /= denominator;

			stop = false;
		}
		else
			stop = 1;

	}

	return result;
}


bool ExprParser::setError(errorCode err)
{
	// Get the error position in the expression string.
	m_errorPos = (int)(m_currChar - m_expr);
	
	// Set error status.
	m_error = err;
	
	// Stop parsing!
	m_currChar = &stopParsingChar;

	// Always return false, as an error occurred.
	return false;
}


ExprParser::real ExprParser::doNumberFunction()
{
	real result;

	result = 0;

	if (*m_currChar == '(') 
	{
		goAhead();
		skipBlanks();
		result = doExpression();

		if (*m_currChar != ')')
		{
			setError(PARENTHESES_NOT_BALANCED);

			// NOTE:
			// This error should be discovered during the startup process
			// of parsing (there is a function called parenthesesBalanced()
			// to do this!).
			return 0;	/* beware... */
		}

		goAhead();
		skipBlanks();
	
	}
	else if (isdigit(*m_currChar))
		result = doNumber();		
	else if (isalpha(*m_currChar))
		result = doFunctionVariable();

	return result;
}


ExprParser::real ExprParser::doNumber()
{
	real result;
	bool stopParsingNumber;	
	const int bufChars = 80;
	char buf[bufChars+1];
	int i;

	stopParsingNumber = false;

	for (i = 0; isdigit(*m_currChar); i++)
	{
		buf[i] = *m_currChar;
		goAhead();
	}

	// We encountered a non digit char. 
	// If this char is '.' or 'E' or 'e', we must go on reading;
	// else, it means we've done our job.
	if (*m_currChar == '.')
	{
		// continue reading
		buf[i] = *m_currChar;
		++m_currChar;
		++i;

		for( ; isdigit(*m_currChar); i++)
		{
			buf[i] = *m_currChar;
			goAhead();
		}
	}
	
	if (*m_currChar == 'E' || *m_currChar == 'e')
	{
		// continue reading
		buf[i] = *m_currChar;
		++m_currChar;
		++i;

		for( ; isdigit(*m_currChar); i++)
		{
			buf[i] = *m_currChar;
			goAhead();
		}
	}
	else
		stopParsingNumber = true;

	if ( !(stopParsingNumber) && (*m_currChar == '+' || *m_currChar == '-'))
	{
		// continue reading
		buf[i] = *m_currChar;
		++m_currChar;
		++i;

		for( ; isdigit(*m_currChar); i++)
		{
			buf[i] = *m_currChar;
			goAhead();
		}
	}
	
	skipBlanks();
	buf[i] = '\0';
	result = atof(buf);

	return result;
}


// It returns a static string!
char * ExprParser::getIdentifier()
{
	static variableNameType identifier;

	// m_currChar must point to the first identifier character.
	// Identifiers must start with a letter of '_'.
	// Other characters can be a-z, A-Z and digits 0-9.

	if ((*m_currChar != '_') && (!isalpha(*m_currChar)))
	{
		// Error: bad identifier
		setError(BAD_IDENTIFIER);
		identifier[0] = '\0';		// empty string
		return identifier;
	}

	int i;
	for (i = 0; i < maxVarIdentifierLength; i++)
	{
		if ((isalpha(*m_currChar)) || (*m_currChar == '_') || (isdigit(*m_currChar)))
		{
			// This is a valid character.
			// Store it into the identifier string.
			identifier[i] = *m_currChar;

			++m_currChar;		// go to next character
		}
		else
			break;	// end of identifier
	}

	identifier[i] = '\0';	// don't forget to set the End of String character!

	return identifier;
}


ExprParser::real ExprParser::doFunctionVariable()
{
	// Do both functions and variables stuff!
	// (variables and functions are processed in similar ways,
	// so I put all the code in this function.)

	char * identifier;
	real result;

	// Get the identifier (maybe a function or variable identifier).
	identifier = getIdentifier();
	if (identifier == NULL)
	{
		// Error!
		setError(BAD_IDENTIFIER);
		return 0;					// dummy value
	}


	// Check for the functions we implement; if there is no match,
	// it means that the identifier may be a variable identifier.
	if (strcmp(identifier, "sin") == 0)
		result = doFunctionSin();
	else if (strcmp(identifier, "cos") == 0)
		result = doFunctionCos();
	else if (strcmp(identifier, "tan") == 0)
		result = doFunctionTan();
	else if (strcmp(identifier, "cot") == 0)
		result = doFunctionCot();
	else if (strcmp(identifier, "asin") == 0)
		result = doFunctionASin();
	else if (strcmp(identifier, "acos") == 0)
		result = doFunctionACos();
	else if (strcmp(identifier, "atan") == 0)
		result = doFunctionATan();
	else if (strcmp(identifier, "sqrt") == 0)
		result = doFunctionSqrt();
	else if (strcmp(identifier, "sqr") == 0)
		result = doFunctionSqr();
	else if (strcmp(identifier, "pow") == 0)
		result = doFunctionPow();
	else if (strcmp(identifier, "exp") == 0)
		result = doFunctionExp();
	else if (strcmp(identifier, "ln") == 0)
		result = doFunctionLn();
	// ...
	// ... add more functions here ...
	// ...
	
	// some useful constants ('pi' and 'e')
	else if (strcmp(identifier, "pi") == 0)
		result = pi;
	else if (strcmp(identifier, "e") == 0)
		result = e;
	
	else
		getVariable(identifier, result);
		// If there will be a variable error (e.g.: variable 
		// identifier unknown), the getVariable() function
		// will call setError(), stopping the parsing process.

	//
	// NOTE:
	// 
	// The way I process functions in the above "if...else if..."
	// statement is very trivial! I absolutely need to find
	// a better solution (e.g.: use hash tables!).

	return result;
}


void ExprParser::skipBlanks()
{
	while (*m_currChar)
		if ((*m_currChar != ' ') && (*m_currChar != '\t'))
			break;
		else
			m_currChar++;
}


bool ExprParser::parenthesesBalanced()
{
	int numOpen = 0;
	int numClose = 0;
	
	char * ch = m_expr;
	assert(ch != NULL);

	while (*ch != '\0')
	{
		if (*ch == '(')
			numOpen++;
		else if (*ch == ')')
			numClose++;

		ch++;
	}
	
	if (numOpen == numClose)
		return true;
	else
		return false;
}


bool ExprParser::copyAllVariablesFromParser(const ExprParser & ep)
{
	// Do a simple and fast raw copy.
	memcpy(m_variables, ep.m_variables, sizeof(Variable) * maxNumberOfVariables);

	// Don't forget to copy the free slot pointer, too!
	m_freeVariableSlot = ep.m_freeVariableSlot;

	// Copy has been successful.
	return true;
}


bool ExprParser::getVariable(const char * name, real & value)
{
	if (name == NULL)
		return setError(UNKNOWN_VARIABLE);

	if (*name == '\0')
		return setError(UNKNOWN_VARIABLE);

	// Scan the variable array to find the variable
	int i;
	for (i = 0; i < m_freeVariableSlot; i++)
		if (strcmp(m_variables[i].identifier, name) == 0) 
		{
			value = m_variables[i].value;
			return true;
		}

	// error: variable not found
	return setError(UNKNOWN_VARIABLE);
}


bool ExprParser::setVariable(const char * name, real value)
{
	if (name == NULL)
		return setError(UNKNOWN_VARIABLE);

	if (*name == '\0')
		return setError(UNKNOWN_VARIABLE);

	int i;
	for (i = 0; i < m_freeVariableSlot; i++)
		if (strcmp(m_variables[i].identifier, name) == 0)
		{
			m_variables[i].value = value;
			return true;
		}

	// The variable has not been found into the array;
	// try to add the new variable.
	
	// Is there any room in the variable array?
	if (m_freeVariableSlot >= maxNumberOfVariables)
		return setError(VARIABLE_MEMORY_LEAK);

	// OK, store the new variable.
	strcpy(m_variables[m_freeVariableSlot].identifier, name);
	m_variables[m_freeVariableSlot].value = value;
	++m_freeVariableSlot;

	// All right.
	return true;
}


ExprParser::real ExprParser::convertAngleInRadians(real angle) const
{
	// This function converts angles in radians.
	// It is useful for trig functions.
	// In fact, standard C math library trig functions assume
	// that angles are in radians; but this parser object can
	// manage also angles expressed in other units (e.g.: 
	// the more human readable degrees ^_^).
	// So, trig functions implementations will always call
	// this function to be sure that their input parameter
	// angle is in radians.

	real radAngle;	// angle in radians

	if (m_angleMode == DEGREES)
		radAngle = angle * (pi / 180.0);	// convert degrees to radians
	else	// m_angleMode == RADIANS
		radAngle = angle;					// angle already in radians

	return radAngle;			// return angle expressed in radians
}


ExprParser::real ExprParser::convertAngleInCurrMode(real angleRad) const
{
	// This function convert an angle expressed in radians to
	// the same angle expressed in the current angle unit.
	// 
	// This function is useful for return values of inverse
	// trig functions (e.g.: asin, acosi, atan, ...).
	// In fact, these functions implementations in the standard
	// C math library return angles expressed in radians.
	// So we need to convert these angles in the current angle mode.

	real angle;			// angle expressed in current angle mode

	if (m_angleMode == DEGREES)
		angle = angleRad * (180.0 / pi); // get angle in degrees
	else // m_angleMode == RADIANS
		angle = angleRad;				// angle already in radians

	return angle;
}


//---------------------------------------------------------------
// END OF FILE: exprparser.cpp
//---------------------------------------------------------------
