# File: nntpbox.py
# Purpose: nntp news class

import utils

from gtk import *
import GtkExtra

import string
import socket
import time
import nntplib
import rfc822
import threading
from pyneheaders import *

import msgfilter
from superbox import *
import boxformats
import boxformats.configbox
from ptk.big_edit_box import *
import pynei18n

def str_nntp_error(e):
	"""
	Python 1.5 nntplib errors are strings.
	Python 2.0 nntplib errors are class objects.
	:-(
	"""
	try:
		# Python 2.x?
		return str(e.args)
	except AttributeError:
		# Python 1.5 :-(
		return str(e)

def nntp_time(gmtime):
	"""
	Convert gmtime something nntp likes. Returns tuple:
	("yymmdd", "hhmmss").
	"""
	yymmdd = time.strftime("%y%m%d", gmtime)
	hhmmss = time.strftime("%H%M%S", gmtime)

	return (yymmdd, hhmmss)

class newsgroup(superbox):
	"""
	A subscribed newsgroup.
	"""
	def get_flags(self):
		return BOX_ISDELETEABLE | BOX_MAKE_NEW_MSG | \
		       BOX_REPLY_SINGLE | BOX_REPLY_GROUP | BOX_REPLY_FORWARD | \
		       BOX_UPDATE_METHOD

	def __init__(self, groupname, uid, parent_nntpbox):
		"""
		Create a new subscribed group.
		"""
		superbox.__init__(self, groupname, uid)
		self.messages = []
		self.filters = []
		self.opts = self.opts | OPT_THREAD
		# set last checked as 'expire_after' days ago
		t = time.gmtime( time.time() - 60*60*24 * parent_nntpbox.expire_after)
		yymmdd, hhmmss = nntp_time(t)
		self.last_checked = (yymmdd, hhmmss)	# Last downloaded news. nntp format
		self.io = parent_nntpbox.io
		self.expire_after = parent_nntpbox.expire_after

	def kill_thread(self, msg_id):
		"""
		Kill this message and all referencing it.
		"""
		# delete actual message
		self.messages.remove(msg_id)
		self.io.delete_article(msg_id)
		
		# cache message headers
		msgs = []
		for i in self.messages:
			m = self.io.load_header(i)
			if m != None:
				msgs.append(m)
		# nuke ones referencing this.
		for i in msgs:
			if i[HEAD_REFERENCES] == None:
				continue
			elif string.find(i[HEAD_REFERENCES], msg_id) != -1:
				self.messages.remove(i[HEAD_MESSAGE_ID])
				self.io.delete_article(i[HEAD_MESSAGE_ID])

	def clean_4_save(self):
		"""
		Remove stuff we don't want saved to disk
		"""
		if self.__dict__.has_key("locked"):
			del self.locked
		if self.__dict__.has_key("changed"):
			del self.changed
		del self.io

	def remote_get_header(self, NNTP_obj, id):
		"""
		Return headers of message 'id' from NNTP server.
		"""
		try:
			lines = NNTP_obj.head(id)[3]
		except nntplib.error_temp:
			# article not on server
			return None
		except nntplib.error_perm:
			raise "NNTP_cockup_error"
		head = string.join(lines, "\n")
		# i've seen nulls in mesages. bad.
		head = string.replace(head, "\000", "")
		return head

	def remote_get_body(self, NNTP_obj, id):
		"""
		Return body of message 'id' from NNTP server.
		"""
		try:
			lines = NNTP_obj.body(id)[3]
		except nntplib.error_temp:
			# article not on server
			return None
		body = string.join(lines, "\n")
		# evil nulls in string. crashes GTK+
		body = string.replace(body, "\000", "")
		return body

	def remote_delete(self, NNTP_obj, id):
		pass

	def get_rest_of_msg(self, user, parent_nntpbox, msg_ids):
		"""
		Collect the rest of a message whose header but not
		body has already been downloaded.
		"""
		try:
			con = parent_nntpbox.get_connection()
		except (nntplib.error_perm, socket.error, EOFError):
			# Doh.. some day tell the user we failed.
			return
		for msg_id in msg_ids:
			msg = self.io.load_article(msg_id)
			if not (msg.opts & MSG_NO_BODY):
				continue
			msg.body = msg.body + self.remote_get_body(con, msg_id)
			msg.opts = msg.opts & (~MSG_NO_BODY)
			msg.parseheaders(user)
			self.io.save_article(msg)
		con.quit()
		self.changed = 1
		threads_enter()
		user.update()
		threads_leave()

	def update(self, servers, pager, progressbar, parent_user, parent_nntpbox):
		"""
		Download new articles.
		"""
		# download marked headers if desired
		if self.opts & OPT_HEADERS_ONLY:
			self.collect_bodies_of_marked(parent_user, pager, progressbar)
			# restore this or we fail later trying to remove it
			self.locked = 1
			threads_enter()
			progressbar.set_sensitive(TRUE)
			threads_leave()
			
		usermsg = _("Checking %s newsgroup") % self.name
		threads_enter()
		pager.msg_print(usermsg)
		progressbar.set_percentage(0.0)
		progressbar.set_format_string(usermsg)
		threads_leave()
		# Get nice nntp format last check times
		yymmdd, hhmmss = self.last_checked
		while 1:
			new_time = parent_nntpbox.get_server_time(servers[0])
			if new_time == None:
				# connection timeout. get new one
				servers[0] = parent_nntpbox.get_connection()
				continue
			else:
				break
		# Get list of new articles ids
		usermsg = _("Collecting %s headers...") % self.name
		threads_enter()
		pager.msg_print(usermsg)
		progressbar.set_format_string(usermsg)
		threads_leave()
		#try:
		#	a_list = servers[0].newnews(self.name, yymmdd, hhmmss)[1]
		#except nntplib.error_perm, e:
		# Disabled newnews use for the moment, since it tends to be
		# a POS (on both my ISPs anyway...)
		if 1==1:
			# newnews not available? use group() and xhdr()
			try:
				a_list = servers[0].group(self.name)
			except nntplib.error_temp, e:
				# probably "411 no such group"
				usermsg = _("Error")+": %s" % str_nntp_error(e)
				threads_enter()
				pager.msg_print(usermsg)
				progressbar.set_format_string(usermsg)
				threads_leave()
				return
			# start and message numbers
			s_msgs = a_list[2]
			e_msgs = a_list[3]
			# get message ids.
			msg_ids = {}
			for i in servers[0].xhdr("message-id", s_msgs+"-"+e_msgs)[1]:
				msg_ids[i[0]] = i[1]
			# get date headers
			try:
				_a_list = []
				for i in servers[0].xhdr("date", s_msgs+"-"+e_msgs)[1]:
					id = i[0]
					date = rfc822.parsedate(i[1])
					if date == None:
						# failed to parse.. just assume
						_a_list.append( msg_ids[id] )
						continue
					elif date[0] < 1000:
						# 2 digit years...
						year = 2000 + date[0]
						date = (year,) + date[1:]
					# add one to day...
					date = date[0:2] + ( date[2]+1, ) + date[3:]
					if (yymmdd, hhmmss) < nntp_time(date):
						# more recent than last checked
						_a_list.append( msg_ids[id] )
			except nntplib.error_perm:
				# this server doesn't support jack shit. just
				# collect all available messages we don't have
				# and to hell with proper expiry, etc. :-(
				_a_list = []
				for i in msg_ids.keys():
					_a_list.append( msg_ids[i] )
			# eliminate messages we already have
			a_list = []
			for i in _a_list:
				if not (i in self.messages):
					a_list.append(i)
			del _a_list
			del msg_ids

		usermsg = _("%d new articles on %s") % (len(a_list), self.name)
		threads_enter()
		pager.msg_print(usermsg)
		progressbar.set_format_string(usermsg)
		threads_leave()

		def _collect_thread(servers, server_num, a_list, total_num, progressbar, pager, self, parent_nntpbox, msgfilter, parent_user):
			# make sure the connection has not died
			if len(a_list) != 0:
				try:
					# this is a silly way of doing it :-/
					servers[server_num].stat(a_list[0])
				except nntplib.error_temp:
					# no such article, or somesuch. ignore
					pass
				except (nntplib.error_perm, AttributeError):
					servers[server_num] = parent_nntpbox.get_connection()
			server = servers[server_num]
			# download the articles
			while 1:
				try:
					x = a_list.pop()
				except IndexError:
					return
				# if the article isn't already present (unlikely)
				if not x in self.messages:
					# if the article isn't already cached download it
					if not parent_nntpbox.io.has_article(x):
						status_msg = msgfilter.filter_collect(parent_user, self, server, x)
						if status_msg != "":
							threads_enter()
							pager.msg_print(status_msg)
							threads_leave()
					# otherwise add reference because the article is already downloaded
					else:
						self.messages.append(x)
				# update progress bar
				num_left = len(a_list)
				threads_enter()
				progressbar.set_percentage((total_num-num_left)/float(total_num))
				progressbar.set_format_string("%d/%d of " % ((total_num-num_left), total_num) + self.name)
				threads_leave()
		# nothing to collect :-/
		if len(a_list) == 0:
			self.last_checked = (new_time[1], new_time[2])
			return
		# start collection threads
		threads = []
		total_num = len(a_list)
		for i in range(1, len(servers)):
			t = threading.Thread(target=_collect_thread, args=(servers, i, a_list, total_num, progressbar, pager, self, parent_nntpbox, msgfilter, parent_user))
			t.start()
			threads.append(t)
		_collect_thread(servers, 0, a_list, total_num, progressbar, pager, self, parent_nntpbox, msgfilter, parent_user)
		# wait till threads are finished executing
		while 1:
			finished = 1
			for i in threads:
				if i.isAlive():
					finished = 0
			if finished:
				break
			time.sleep(1)
		# finish up
		self.last_checked = (new_time[1], new_time[2])

		threads_enter()
		self.changed = 1
		parent_user.update()
		threads_leave()
		
	def get_menu(self):
		"""
		This box type specific options.
		"""
		if self.opts & OPT_HEADERS_ONLY:
			return [ _("Download Marked") ]
		else:
			return []

	def menu_chosen(self, _a, choice_n_user):
		"""
		Activate the chosen menu option.
		"""
		choice, user = choice_n_user
		if (choice == _("Download Marked")) and (not self.__dict__.has_key("locked")):
			pager = utils.ProgressWindow(_("Collecting messages..."), hidetext=_("Show Logs"))
			progressbar = pager.get_progress_bar()
			threading.Thread(target=self.collect_bodies_of_marked, args=(user, pager, progressbar)).start()
		return

	def collect_bodies_of_marked(self, user, pager, progressbar):
		"""
		Download bodies of marked messages (those with only headers
		downloaded)
		"""
		self.locked = 1
		
		usermsg = _("Searching for marked messages in %s...") % self.name
		threads_enter()
		progressbar.set_percentage(0.0)
		progressbar.set_format_string(usermsg)
		pager.msg_print(usermsg)
		threads_leave()
		
		no_bdy_msgs = []

		for i in self.messages:
			head = self.io.load_header(i)
			if (head[HEAD_OPTS] & MSG_NO_BODY) and (head[HEAD_OPTS] & MSG_ISMARKED):
				msg = self.io.load_article(i)
				no_bdy_msgs.append(msg)

		if len(no_bdy_msgs) == 0:
			usermsg = _("No marked messages to collect")
			threads_enter()
			pager.msg_print(usermsg)
			progressbar.set_format_string(usermsg)
			progressbar.set_sensitive(FALSE)
			threads_leave()
			del self.locked
			return

		usermsg = _("%d marked messages to collect in %s") % (len(no_bdy_msgs), self.name)
		threads_enter()
		pager.msg_print(usermsg)
		progressbar.set_format_string(usermsg)
		threads_leave()
		
		try:
			con = user.parent_of(self).get_connection()
		except (nntplib.error_perm, nntplib.error_temp, socket.error, EOFError), e:
			usermsg = _("Error")+": "+str_nntp_error(e)
			threads_enter()
			pager.msg_print(usermsg)
			progressbar.set_format_string(usermsg)
			progressbar.set_sensitive(FALSE)
			threads_leave()
			del self.locked
			return
	
		# just 1 collection thread for the moment :-(
		for i in range(0, len(no_bdy_msgs)):
			msg = no_bdy_msgs[i]
			msg.body = msg.body + self.remote_get_body(con, msg.headers["message-id"])
			msg.opts = msg.opts & (~MSG_NO_BODY)
			msg.opts = msg.opts & (~MSG_ISMARKED)
			msg.parseheaders(user)
			self.io.save_article(msg)
			
			threads_enter()
			progressbar.set_percentage((i+1)/float(len(no_bdy_msgs)))
			progressbar.set_format_string("%d/%d of " % (i+1, len(no_bdy_msgs)) + self.name)
			threads_leave()
		del self.locked

		threads_enter()
		progressbar.set_sensitive(FALSE)
		self.changed = 1
		user.update()
		threads_leave()

	def getstats(self):
		"""
		Returns a tuple. (total msgs, ).
		"""
		return (len(self.messages),)

	def setup(self, parent_user):
		"""
		Configure a newsgroup's collection settings.
		"""
		win = GtkWindow()
		win.set_title(_("%s Setup") % self.name)

		box = GtkVBox()
		win.add(box)
		box.show()

		notebook = GtkNotebook()
		notebook.set_tab_pos(POS_TOP)
		box.pack_start(notebook)
		notebook.show()
	
		# First page: Misc settingsi
		box0 = GtkVBox()
		box0.set_border_width(5)
		label = GtkLabel(_("Settings"))
		notebook.append_page(box0, label)
		box0.show()

		box1 = GtkHBox()
		box0.pack_start(box1)
		box1.show()

		box1_1 = GtkVBox(spacing=5)
		box1_1.set_border_width(5)
		box1.pack_start(box1_1)
		box1_1.show()

		box1_2 = GtkVBox(spacing=5)
		box1_2.set_border_width(5)
		box1.pack_end(box1_2)
		box1_2.show()

		input_boxes = ( ( _("Last Checked (YYMMDD):"), self.last_checked[0] ),
				( _("Last Checked (HHMMSS):"), self.last_checked[1] ),
				( _("Expire after (days):"), str(self.expire_after) ) )
		entry_boxes = []

		for x in range(0, len(input_boxes)):
			label = GtkLabel(input_boxes[x][0])
			label.set_justify(JUSTIFY_LEFT)
			box1_1.pack_start(label)
			label.show()

			entry_box = GtkEntry()
			entry_box.set_text(input_boxes[x][1])
			box1_2.pack_start(entry_box)
			entry_box.show()

			entry_boxes.append(entry_box)

		settings_box0 = big_edit_box( self,
		     ( ("opts", _("Only collect headers"), VAR_TYPE_PACKED_BIT_BOOLEAN, 0, OPT_HEADERS_ONLY), )
		)
		box0.pack_start(settings_box0, expand=FALSE)
		settings_box0.show()

		# 2nd page: Filters
		label = GtkLabel(_("Filters"))
		filterbox = msgfilter.filter_editbox(self, parent_user)
		notebook.append_page(filterbox, label)
		filterbox.show()

		def save_changes(_button, boxes=entry_boxes, self=self, parent_user=parent_user, \
				settings_box0=settings_box0, filterbox=filterbox, win=win):
			# Extract info
			self.last_checked = (boxes[0].get_text(), boxes[1].get_text())
			self.filters = filterbox.filters
			settings_box0.apply_changes()
			try:
				self.expire_after = int(boxes[2].get_text())
			except ValueError:
				pass
			# update folder list
			self.changed = 1
			parent_user.update()
			win.destroy()

		# Seperator between entry boxes and buttons
		separator = GtkHSeparator()
		box.pack_start(separator, expand=FALSE)
		separator.show()

		# Buttons at bottom
		lastbox = GtkHBox(spacing=5)
		lastbox.set_border_width(10)
		box.pack_start(lastbox, expand=FALSE)
		lastbox.show()

		cancel_button = GtkButton(" "+_("Cancel")+" ")
		cancel_button.connect("clicked", win.destroy)
		lastbox.pack_end(cancel_button, expand=FALSE)
		cancel_button.show()

		ok_button = GtkButton(" "+_("Ok")+" ")
		ok_button.connect("clicked", save_changes)
		lastbox.pack_end(ok_button, expand=FALSE)
		ok_button.show()

		win.show()



class nntpbox(superbox):
	"""
	News box.
	"""
	def get_flags(self):
		return BOX_ISDELETEABLE | BOX_UPDATE_METHOD

	def __init__(self, user, name, uid):
		"""
		Create new nntp inbox.
		"""

		# Basic setup
		superbox.__init__(self, name, uid)
		self.server = "news.someisp.com"
		self.port = 119
		self.username = ""
		self.password = ""
		self.realname = "your name"
		self.emailaddr = "email@address.com"
		self.replyto = ""
		self.org = ""
		self.sigfile = "~/.signature"
		self.export_format = (boxformats.BF_FLATFILEBOX, None)
		self.expire_after = 7
		self.collection_threads = 4
		self.contents = []

		# Newsgroup info
		self.groups = []	# List of newsgroups in form returned
					# by server
		self.groups_last_checked = None	# Last updated group list

		self.startup(user)

	def clean_4_save(self):
		"""
		Remove stuff we don't want saved to disk
		"""
		if self.__dict__.has_key("locked"):
			del self.locked
		if self.__dict__.has_key("changed"):
			del self.changed
		del self.io

	def startup(self, user):
		"""
		Get an io module for loading/saving articles
		"""
		i = self.export_format[0]
		# this is a pretty piece of code :-)
		bf_class = boxformats.__dict__[boxformats.__all__[i]]. \
			   __dict__[boxformats.__all__[i]]
		self.io = bf_class(user, self)
		# give kids a reference ;-)
		for x in self.contents:
			x.io = self.io

	def sort_groups(self):
		def _sort(name1, name2):
			#name1 = a[0]
			#name2 = b[0]
			if name1 < name2:
				return -1
			elif name1 > name2:
				return 1
			return 0
		self.groups.sort(_sort)

	def get_server_time(self, server):
		"""
		Get time, date strings from server or just
		use local time if the server doesn't support.
		"""
		try:
			new_time = server.date()
		except (nntplib.error_temp, nntplib.error_perm), e:
			try:
				# python 1.5 and 2.x nntp exceptions are
				# totally incompatible...
				e = e.args[0]
			except AttributeError:
				pass
			if string.split(e)[0] == "503":
				# 503 connection timeout. freak out
				return None
			else:
				# not supported. just do it the lame way
				new_time = (None,) + nntp_time(time.gmtime(time.time()))
		return new_time

	def get_menu(self):
		"""
		This box type specific options.
		"""
		return []

	def menu_chosen(self, _a, choice):
		"""
		Activate the chosen menu option.
		"""
		return

	def getstats(self):
		"""
		Returns a tuple. (subscribed groups).
		"""
		return (len(self.contents),)

	def post_message(self, connection, msg):
		"""
		Post message 'msg' with connection.
		"""
		from StringIO import StringIO
		msg_body = StringIO(msg.body)
		connection.post(msg_body)

	def get_connection(self):
		"""
		Connect to the news server. Return an NNTP object.
		"""
		if self.username == "" and self.password == "":
			ns = nntplib.NNTP(self.server, self.port)
		elif self.password == "":
			ns = nntplib.NNTP(self.server, self.port, self.username)
		else:
			ns = nntplib.NNTP(self.server, self.port, self.username, self.password)
		return ns

	def get_new_group_list(self):
		"""
		Get a new list of newsgroups.
		Can [will] take ages. this should be threaded or something...
		"""
		try:
			ns = self.get_connection()
		except (socket.error, EOFError), e:
			threads_enter()
			utils.info_box("socket.error", _("Error connecting to %s:\n%s") % (self.name, str(e)), _("Ok"))
			threads_leave()
			return
		except nntplib.error_perm, e:
			threads_enter()
			utils.info_box(_("NNTP Error"), _("NNTP Error:")+"\n"+str_nntp_error(e), _("Ok"))
			threads_leave()
			return

		# failed connection
		if ns == None:
			return

		date = self.get_server_time(ns)
		self.groups_last_checked = (date[1], date[2])

		crudelist = ns.list()[1]
		self.groups = []

		for x in crudelist:
			self.groups.append("%s %s" % (x[0], x[3]))

		ns.quit()

		# sort group list
		self.sort_groups()

		return
		
	def subscribe_to(self, groupname, parent_user):
		"""
		Subscribe to a group
		"""
		def sort_groups(a, b):
			if a.name < b.name:
				return -1
			elif a.name > b.name:
				return 1
			else:
				return 0
		# Check it isn't already subscribed to
		for x in self.contents:
			if x.name == groupname:
				break
		ng = newsgroup(groupname, parent_user.get_uid(), self)
		self.contents.append(ng)
		self.contents.sort(sort_groups)

	def setup(self, parent_user):
		"""
		Configure an nntp box.
		"""
		win = GtkWindow()
		win.set_title(self.name+" "+_("Setup"))

		box = GtkVBox()
		win.add(box)
		box.show()

		notebook = GtkNotebook()
		notebook.set_tab_pos(POS_TOP)
		box.pack_start(notebook)
		notebook.show()
	
		# First page: Misc settings
		settings_box = big_edit_box(self,
		      ( ("name", _("Name:"), VAR_TYPE_STRING, 0, 0),
			("server", _("Server:"), VAR_TYPE_STRING, 0, 0),
			("port", _("Port:"), VAR_TYPE_INTEGER, 0, 119),
			("username", _("Username:"), VAR_TYPE_STRING, 0, 0),
			("password", _("Password:"), VAR_TYPE_STRING, MOD_HIDE, 0),
			("realname", _("Real name:"), VAR_TYPE_STRING, 0, 0),
			("emailaddr", _("Email address:"), VAR_TYPE_STRING, 0, 0),
			("replyto", _("Reply to:"), VAR_TYPE_STRING, 0, 0),
			("org", _("Organisation:"), VAR_TYPE_STRING, 0, 0),
			("sigfile", _("Signature file:"), VAR_TYPE_STRING, 0, 0,
				(GtkExtra.file_sel_box, (_("Select Signature File"),))),
			("expire_after", _("Default expiry (days):"), VAR_TYPE_INTEGER, 0, None),
			("collection_threads", _("Download threads:"), VAR_TYPE_INTEGER, 0, 4) )
		)
		label = GtkLabel(_("Settings"))
		notebook.append_page(settings_box, label)
		settings_box.show()

		# Seperator between entry boxes and buttons
		separator = GtkHSeparator()
		box.pack_start(separator, expand=FALSE)
		separator.show()

		# 2nd page: Newsgroup settings
		box2 = GtkHBox()
		box2.set_border_width(10)
		label = GtkLabel(_("Newsgroups"))
		notebook.append_page(box2, label)
		box2.show()

		box2_1 = GtkVBox(spacing=5)
		box2_1.set_border_width(5)
		box2.pack_start(box2_1)
		box2_1.show()
		
		grouplistbox = GtkCList(2, [ _("Available groups"), _("Posting") ] )
		grouplistbox.set_column_width(0,150)
		grouplistbox.set_column_width(1,20)
		swin = GtkScrolledWindow()
		swin.set_policy(POLICY_NEVER, POLICY_AUTOMATIC)
		box2_1.pack_end(swin)
		swin.show()
		swin.add(grouplistbox)
		grouplistbox.show()
		
		searchbox = GtkEntry()
		box2_1.pack_start(searchbox, expand=FALSE)
		searchbox.show()

		def _search(_button, self=self, glist=grouplistbox, sbox=searchbox):
			search = sbox.get_text()

			# Nothing. Unfiltered
			if search == "":
				glist.freeze()
				# Wipe NG list
				glist.clear()
				# add NGs
				for x in self.groups:
					glist.append(string.split(x))
				glist.thaw()
				return
				
			# Otherwise do some filtering
			fgroups = []
			for x in self.groups:
				name, mode = string.split(x)
				if string.find(name, search) != -1:
					fgroups.append((name, mode))
			glist.freeze()
			# Wipe NG list
			glist.clear()
			# add filtered NGs
			for x in fgroups:
				glist.append(x)
			glist.thaw()


		box2_2 = GtkVBox(spacing=5)
		box2_2.set_border_width(5)
		box2.pack_end(box2_2, expand=FALSE)
		box2_2.show()

		search_button = GtkButton(" "+_("Search")+" ")
		search_button.connect("clicked", _search)
		box2_2.pack_start(search_button, expand=FALSE)
		search_button.show()

		# add NGs
		grouplistbox.freeze()
		for x in self.groups:
			grouplistbox.append(string.split(x))
		grouplistbox.thaw()

		def _subscribe_group(_button, self=self, user=parent_user, \
				     searchbox=searchbox, glist=grouplistbox):
			import mainwin
			# Nothing selected
			if glist.selection == []:
				# nothing selected. try subscribing to what is
				# in the search box
				if searchbox.get_text() != "":
					self.subscribe_to(searchbox.get_text(), user)
				else:
					return
			else:
				selected = glist.selection[0]
				self.subscribe_to(glist.get_text(selected, 0), user)
			# Big folderview update
			self.changed = 1
			user.update(mainwin.UPDATE_FOLDERVIEW)

		newlist_button = GtkButton(" "+_("New List")+" ")
		box2_2.pack_end(newlist_button, expand=FALSE)
		newlist_button.show()

		def _get_new_group_list(_button, newlist_button=newlist_button, settings_box=settings_box, self=self, glist=grouplistbox):
			def _new_gl(settings_box=settings_box, newlist_button=newlist_button, self=self, glist=glist):
				# make current changes active briefly...
				# XXX it sucks indexing entry_objects like this... a
				# dictionary would be nicer
				threads_enter()
				newlist_button.set_sensitive(FALSE)
				
				oldname = self.name
				self.name = settings_box.entry_objects[0].get_text()
				oldserver = self.server
				self.server = settings_box.entry_objects[1].get_text()
				try:
					oldport = self.port
					self.port = int(settings_box.entry_objects[2].get_text())
				except ValueError:
					self.port = 119
				oldusername = self.username
				self.username = settings_box.entry_objects[3].get_text()
				oldpassword = self.password
				self.password = settings_box.entry_objects[4].get_text()
				threads_leave()
		
				# get list of groups
				self.get_new_group_list()

				# and restore settings
				self.name = oldname
				self.server = oldserver
				self.port = oldport
				self.username = oldusername
				self.password = oldpassword

				threads_enter()
				glist.freeze()
				# Wipe NG list
				glist.clear()
				# add NGs
				for x in self.groups:
					glist.append(string.split(x))
				glist.thaw()
				newlist_button.set_sensitive(TRUE)
				threads_leave()

			threading.Thread(target=_new_gl).start()

		newlist_button.connect("clicked", _get_new_group_list)

		subscribe_button = GtkButton(" "+_("Subscribe")+" ")
		subscribe_button.connect("clicked", _subscribe_group)
		box2_2.pack_end(subscribe_button, expand=FALSE)
		subscribe_button.show()

		def save_changes(_button, settings_box=settings_box, self=self, \
				parent_user=parent_user, win=win):
			# Extract info
			settings_box.apply_changes()
			# update folder list
			self.changed = 1
			parent_user.update()
			win.destroy()

		# Buttons at bottom
		lastbox = GtkHBox(spacing=5)
		lastbox.set_border_width(10)
		box.pack_start(lastbox, expand=FALSE)
		lastbox.show()

		cancel_button = GtkButton(" "+_("Cancel")+" ")
		cancel_button.connect("clicked", win.destroy)
		lastbox.pack_end(cancel_button, expand=FALSE)
		cancel_button.show()

		ok_button = GtkButton(" "+_("Ok")+" ")
		ok_button.connect("clicked", save_changes)
		lastbox.pack_end(ok_button, expand=FALSE)
		ok_button.show()

		win.show()

	def update(self, parent_user, pager, only_update=None):
		"""
		Download new news.
		'only_update' should be an individual newsgroup instance
		to update. If omitted all are updated.
		"""
		usermsg = self.name+": "+_("Connecting to %s...") % self.server
		threads_enter()
		progressbar = pager.get_progress_bar()
		pager.msg_print(usermsg)
		progressbar.set_format_string(usermsg)
		threads_leave()

		try:
			# list of nntp connections, mostly null at the moment
			ns = [ self.get_connection() ]
			for i in range(1, self.collection_threads):
				ns.append(None)
		except (nntplib.error_perm, nntplib.error_temp, socket.error, EOFError), e:
			usermsg = _("Error connecting to %s: %s") % (self.name, str_nntp_error(e))
			threads_enter()
			pager.msg_print(usermsg)
			progressbar.set_format_string(usermsg)
			progressbar.set_sensitive(FALSE)
			threads_leave()
			return

		# Update single group if desired:
		if only_update != None:
			if not only_update.__dict__.has_key("locked"):
				only_update.locked = 1
				only_update.update(ns, pager, progressbar, parent_user, self)
				del only_update.locked
		# otherwise update everything
		else:
			if self.groups_last_checked == None:
				usermsg = _("Getting %s newsgroup list...") % self.name
				threads_enter()
				pager.msg_print(usermsg)
				progressbar.set_format_string(usermsg)
				threads_leave()
				self.get_new_group_list()
			usermsg = _("Checking %s for new groups...") % self.name
			threads_enter()
			pager.msg_print(usermsg)
			progressbar.set_format_string(usermsg)
			threads_leave()
			# Convert time last checked to something nntp likes
			yymmdd, hhmmss = self.groups_last_checked
			# update last checked
			date = self.get_server_time(ns[0])
			self.groups_last_checked = (date[1], date[2])
			# look for new groups
			try:
				_newgroups = ns[0].newgroups(yymmdd, hhmmss)[1]
				# Convert to a nice list of group names and append to old list
				newgroups = []
				for x in _newgroups:
					newgroups.append( "%s %s" % (string.split(x, " ")[0], string.split(x, " ")[3]) )
				self.groups = self.groups + newgroups

				usermsg = _("%d new %s groups.") % (len(newgroups), self.name)
				threads_enter()
				pager.msg_print(usermsg)
				progressbar.set_format_string(usermsg)
				threads_leave()

				if len(newgroups) != 0:
					# sort group list if there are new groups
					self.sort_groups()
			# erk. that didn't work...
			except nntplib.error_temp, e:
				usermsg = _("Error %s getting %s newsgroup list...") % (str_nntp_error(e), self.name)
				threads_enter()
				pager.msg_print(usermsg)
				threads_leave()

			for x in self.contents:
				if not x.__dict__.has_key("locked"):
					x.locked = 1
					x.update(ns, pager, progressbar, parent_user, self)
					del x.locked

		usermsg = "Finished checking %s news" % self.name
		# update display
		threads_enter()
		pager.msg_print(usermsg)
		progressbar.set_percentage(0.0)
		progressbar.set_format_string(usermsg)
		progressbar.set_sensitive(FALSE)
		self.changed = 1
		parent_user.update()
		threads_leave()

