/*
 * PROLL: Interrupt management module.
 * $Id: sched_4m.c,v 1.8 1999/04/27 05:42:36 zaitcev Exp $
 */
#include <general.h>
#include <romlib.h>
#include <system.h>
#include "phys_jj.h"

/*
 * Interrupt control in sun4m
 * This code is originated from Dave Redman, of Tadpole, in Sun since 1997.
 */

/* These registers are used for sending/receiving irqs from/to
 * different cpu's.
 */
struct sun4m_intreg_percpu {
	unsigned int tbt;        /* Intrs pending for this cpu, by PIL. */
	/* These next two registers are WRITE-ONLY and are only
	 * "on bit" sensitive, "off bits" written have NO affect.
	 */
	unsigned int clear;  /* Clear this cpus irqs here. */
	unsigned int set;    /* Set this cpus irqs here. */
};

/*
 * djhr
 * Actually the clear and set fields in this struct are misleading..
 * according to the SLAVIO manual (and the same applies for the SEC)
 * the clear field clears bits in the mask which will ENABLE that IRQ
 * the set field sets bits in the mask to DISABLE the IRQ.
 *
 * Also the undirected_xx address in the SLAVIO is defined as
 * RESERVED and write only..
 *
 * DAVEM_NOTE: The SLAVIO only specifies behavior on uniprocessor
 *             sun4m machines, for MP the layout makes more sense.
 */
struct sun4m_intreg_master {
	unsigned int tbt;        /* IRQ's that are pending, see sun4m masks. */
	unsigned int irqs;       /* Master IRQ bits. */

	/* Again, like the above, two these registers are WRITE-ONLY. */
	unsigned int clear;      /* Clear master IRQ's by setting bits here. */
	unsigned int set;        /* Set master IRQ's by setting bits here. */

	/* This register is both READ and WRITE. */
	unsigned int undirected_target;  /* Which cpu gets undirected irqs. */
};

#define SUN4M_INT_ENABLE        0x80000000
#define SUN4M_INT_E14           0x00000080
#define SUN4M_INT_E10           0x00080000

#define SUN4M_HARD_INT(x)       (0x000000001 << (x))
#define SUN4M_SOFT_INT(x)       (0x000010000 << (x))

#define SUN4M_INT_MASKALL       0x80000000        /* mask all interrupts */
#define SUN4M_INT_MODULE_ERR    0x40000000        /* module error */
#define SUN4M_INT_M2S_WRITE     0x20000000        /* write buffer error */
#define SUN4M_INT_ECC           0x10000000        /* ecc memory error */
#define SUN4M_INT_FLOPPY        0x00400000        /* floppy disk */
#define SUN4M_INT_MODULE        0x00200000        /* module interrupt */
#define SUN4M_INT_VIDEO         0x00100000        /* onboard video */
#define SUN4M_INT_REALTIME      0x00080000        /* system timer */
#define SUN4M_INT_SCSI          0x00040000        /* onboard scsi */
#define SUN4M_INT_AUDIO         0x00020000        /* audio/isdn */
#define SUN4M_INT_ETHERNET      0x00010000        /* onboard ethernet */
#define SUN4M_INT_SERIAL        0x00008000        /* serial ports */
#define SUN4M_INT_SBUSBITS      0x00003F80        /* sbus int bits */

#define SUN4M_INT_SBUS(x)       (1 << (x+7))
#define SUN4M_INT_VME(x)        (1 << (x))


/*
 * Registers of hardware timer in sun4m.
 */
struct sun4m_timer_percpu {
	volatile unsigned int l14_timer_limit; /* Initial value is 0x009c4000 */
	volatile unsigned int l14_cur_count;
};

struct sun4m_timer_global {
        volatile unsigned int l10_timer_limit;
        volatile unsigned int l10_cur_count;
};

/*
 */
struct handsc {
	void (*ifunc)(void *);
	void *iarg;
};

/*
 */
unsigned long volatile jiffies;		/* ms? ns? */

/*
 */
static struct sun4m_timer_percpu *m14p;
static struct sun4m_timer_global *m10p;
static struct sun4m_intreg_percpu *micp;
static struct sun4m_intreg_master *mimp;

static int set_bolt;			/* Tick counter limit */
static struct handsc hndv[16];

static unsigned int intr_to_mask[16] = {
	0,	0,	0,	0,	0,	0, SUN4M_INT_ETHERNET,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
};

/*
 * Entries from Gero's stuff.
 * chk_timeout() is the main scheduler.
 */
void
set_timeout(int sec)
{
	volatile int clear;

	set_bolt = sec * HZ;
	clear = m10p->l10_timer_limit;
	m10p->l10_timer_limit = (((1000000/HZ) + 1) << 10);
}

int	/* 0 - not expired yet; <>0 - timer expired */
chk_timeout()
{
	int lim = (((1000000/HZ) + 1) << 10);
	unsigned int clear;
	unsigned int intc;
	int n;
	struct handsc *hndp;

	/*
	 * Watch for interrupts.
	 */
	intc = micp->tbt;
	if ((intc & ~(1 << 10)) != 0) {		/* Ignore level 10 timer */
		n = highc(intc, 16);
		hndp = &hndv[n];
		if (hndp->ifunc != 0) {
#if 0
			printk("chk_timer: intr "
			    "percpu.tbt 0x%x master.tbt 0x%x\n",
			    intc, mimp->tbt);
#endif
			(*hndp->ifunc)(hndp->iarg);
		} else {
			printk("chk_timer: unserviced %d "
			    "percpu.tbt 0x%x master.tbt 0x%x\n",
			    n, intc, mimp->tbt);
		}
	}

	/*
	 * Check if historic timeout expired, run timers.
	 */
	clear = m10p->l10_cur_count;
	if ((clear & 0x7FFFFFFF) < lim && (clear & 0x80000000) == 0)
		return 0;
	clear = m10p->l10_timer_limit;
	m10p->l10_timer_limit = (((1000000/HZ) + 1) << 10);
	jiffies++;
	run_timers();
	if (set_bolt > 0) {
		set_bolt--;
		return 0;
	}
	return 1;
}

/*
 * Linux environment emulator.
 */
int request_irq(unsigned int irq, void (*func)(void *), void *arg)
{
	struct handsc *hndp;
	unsigned int mask;

	if (irq < 0 || irq >= 16) {
		printk("request_irq: bad irq %d\n", irq);
		return -1;
	}
	hndp = &hndv[irq];

	if (hndp->ifunc != 0) {
		printk("request_irq: busy %d\n", irq);
		return -1;
	}

	hndp->ifunc = func;
	hndp->iarg = arg;

	mask = intr_to_mask[irq];
	if (mask != 0) {
		mimp->clear = mask;
	}
	return 0;
}

void free_irq(unsigned int irq, void *arg)
{
	struct handsc *hndp;

	if (irq < 0 || irq >= 16) {
		printk("free_irq: bad irq %d\n", irq);
		return;
	}
	hndp = &hndv[irq];

	if (hndp->ifunc == 0) {
		printk("free_irq: already free %d\n", irq);
		return;
	}
	if (hndp->iarg != arg) {
		printk("free_irq: bad arg at %d\n", irq);
		return;
	}

	hndp->ifunc = 0;
	hndp->iarg = 0;
	return;
}

/*
 */
int
sched_init()
{

	m14p = map_io(PHYS_JJ_CLOCK, sizeof(struct sun4m_timer_percpu));
	if (m14p == 0) {
		printk("sched_init: cannot map l14 timer\n");
		return -1;
	}

	m10p = map_io(PHYS_JJ_CLOCK1, sizeof(struct sun4m_timer_global));
	if (m10p == 0) {
		printk("sched_init: cannot map l14 timer\n");
		return -1;
	}

	m10p->l10_timer_limit = (((1000000/HZ) + 1) << 10);
	m14p->l14_timer_limit = 0;

	micp = map_io(PHYS_JJ_INTR0, sizeof(struct sun4m_intreg_percpu));
	if (micp == 0) {
		printk("sched_init: cannot map CPU intr control\n");
		return -1;
	}

	mimp = map_io(PHYS_JJ_INTR_G, sizeof(struct sun4m_intreg_master));
	if (mimp == 0) {
		printk("sched_init: cannot map global intr control\n");
		return -1;
	}

	mimp->set = ~SUN4M_INT_MASKALL;
	micp->clear = ~0x17fff;

	return 0;
}
