/*-----------------------------------------------------------------------
QMBTagger - Qt Based MusicBrainz MP3/OGG/FLAC Tagger
Copyright (C) 2003,2004 Ewen Cheslack-Postava

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------*/

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if HAVE_LIBMUSICBRAINZ
#if HAVE_FLAC

#include "qmbflacfile.h"
#include "qmbfile.h"
#include <stdio.h>
#include <fstream>
#include <qapplication.h>
#include <qstring.h>
#include <musicbrainz/mb_c.h>
#include <FLAC++/metadata.h>

QMBFLACFile::QMBFLACFile()
	: QMBFile(), FLAC::Decoder::File(), offset(0), decode_error(false)
{
	//we do nothing here.  we want to allow checking of return value,
	//so we only allow class to function through calling of Extract()
}

//get the data we want from the vorbis stream
//heavily borrowed code from vorbis doc on decoding vorbis stream
//returns:
//	0  indicates success
//	-1 indicates failed to open
//	-2 indicates invalid info
//	1 indicates success, but metadata extraction failed
int QMBFLACFile::Extract(QString filename) {
	
	int ret = 0;

	//record the full filename so we will be able to use it for output
	fullfilename = filename;

	//set the filename in the metadata
	int lastslash = filename.findRev('/');
	metadata.SetFileName(filename.right(filename.length() - lastslash - 1));

	//set the filename with the FLAC::Decoder::File class
	set_filename(fullfilename.latin1());

	//tell the decoder to call the metadata callback on vorbis comments so we can get the metadata already stored in the file
	set_metadata_respond(FLAC__METADATA_TYPE_VORBIS_COMMENT);
	
	//Start reading the FLAC file
	FLAC__FileDecoderState state = init();
	
	if (state != FLAC__FILE_DECODER_OK) {
		qWarning("QMBFLACFile::Extract(): error opening file: %i (%s).", state, FLAC__FileDecoderStateString[state]);
		finish();
		return -1;
	}
	
	//read through all the metadata so our callback can catch it all
	if(!process_until_end_of_metadata ()) {
		state = get_state();
		qWarning("QMBFLACFile::Extract(): error getting metadata: %i (%s).", state, FLAC__FileDecoderStateString[state]);
		decode_error = true;
		finish();
		return -2;
	}
	
	//if the trmid was read in with the metadata, quit now
	if (!metadata.GetTRMID().isEmpty()) {
		finish();
		return ret;
	}
	
	//set duration information in the metadata, but in milliseconds instead of seconds
	metadata.SetDuration(QString::number(songlength*1000));

	//if the info is invalid, return without doing anything
	if (channels == 0 || sampleRate == 0 || bits_per_sample == 0) {
		qWarning("QMBFLACFile::Extract(): Channels (%d), SampleRate (%ld), or Sample Resolution (%d) not set properly for file (%s).", channels, sampleRate, bits_per_sample, fullfilename.local8Bit().data());
		finish();
		return -2;
	}
	
	//figure out buffer length, we want 30 seconds of data,
	//= channels*30 sec*rate*bits_per_sample/8
	bufferlen = 30*channels*sampleRate*bits_per_sample/8;

	//create the actual buffer, leaving enough room for the max possible overflow
	buffer = new char[bufferlen + max_block_size*channels*bits_per_sample/8];

	//now lets get the data from the FLAC file
	while (!decode_error && offset < (unsigned long) bufferlen) {
		if (!process_single()) {
			state = get_state();
			qWarning("QMBFLACFile::Extract(): error getting raw data: %i (%s).", state, FLAC__FileDecoderStateString[state]);
			decode_error = true;
			finish();
			return -2;
		}
	}
	
	//we're done reading, now clean up
	finish();

	return ret;
}

::FLAC__StreamDecoderWriteStatus QMBFLACFile::write_callback(const ::FLAC__Frame *frame, const FLAC__int32 * const writebuffer[]) {
	if (decode_error)
		return ::FLAC__STREAM_DECODER_WRITE_STATUS_ABORT;
	
	unsigned int num_samples = frame->header.blocksize;
	//unsigned int bytes_per_sample = bits_per_sample/8;
	
	FLAC__int16* offsetbuffer = (FLAC__int16*)(buffer+offset);
	
	//FIXME: This could definitely be better, but I was too lazy to figure out the less loops/math way...
	for(unsigned int sample = 0; sample < num_samples; sample++) {
		for(int channel = 0; channel < channels; channel++) {
			//for(unsigned int byte = 0; byte < bytes_per_sample; byte++) {
				//FIXME: This could definitely be a memcpy (this inner loop)
			*offsetbuffer = (FLAC__int16)(writebuffer[channel][sample]);
			offsetbuffer++;
				//(offsetbuffer+sample*channels*bytes_per_sample)[channel*bytes_per_sample+byte] = (writebuffer[channel])[sample*bytes_per_sample+byte];
			//}
		}
	}

	offset += 2*num_samples*channels;
	
	return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
}

void QMBFLACFile::metadata_callback(const ::FLAC__StreamMetadata *metadata) {

	if (decode_error)
		return;
	
	if (metadata->type == FLAC__METADATA_TYPE_STREAMINFO) {
		sampleRate = metadata->data.stream_info.sample_rate;
		channels = metadata->data.stream_info.channels;
		bits_per_sample = metadata->data.stream_info.bits_per_sample;
		songlength = (double)(metadata->data.stream_info.total_samples) / sampleRate;
		max_block_size = metadata->data.stream_info.max_blocksize;
	}
	else if (metadata->type == FLAC__METADATA_TYPE_VORBIS_COMMENT) {
		for(unsigned int i = 0; i < metadata->data.vorbis_comment.num_comments; i++) {
			QString thiscomment = QString::fromLatin1((const char*)metadata->data.vorbis_comment.comments[i].entry, metadata->data.vorbis_comment.comments[i].length);
			if (thiscomment.left(6).upper() == "TITLE=") {
				this->metadata.SetTrackName(thiscomment.right(thiscomment.length() - 6));
			} else if (thiscomment.left(6).upper() == "ALBUM=") {
				this->metadata.SetAlbumName(thiscomment.right(thiscomment.length() - 6));
			} else if (thiscomment.left(7).upper() == "ARTIST=") {
				this->metadata.SetArtistName(thiscomment.right(thiscomment.length() - 7));
			} else if (thiscomment.left(12).upper() == "TRACKNUMBER=") {
				this->metadata.SetTrackNumber(thiscomment.right(thiscomment.length() - 12));
			} else if (thiscomment.left(20).upper() == "MUSICBRAINZ_TRACKID=") {
				this->metadata.SetTrackID(thiscomment.right(thiscomment.length() - 20));
			} else if (thiscomment.left(21).upper() == "MUSICBRAINZ_ARTISTID=") {
				this->metadata.SetArtistID(thiscomment.right(thiscomment.length() - 21));
			} else if (thiscomment.left(20).upper() == "MUSICBRAINZ_ALBUMID=") {
				this->metadata.SetAlbumID(thiscomment.right(thiscomment.length() - 20));
			} else if (thiscomment.left(18).upper() == "MUSICBRAINZ_TRMID=") {
				this->metadata.SetTRMID(thiscomment.right(thiscomment.length() - 18));
			}
		}
	}
	
	return;
}

void QMBFLACFile::error_callback(::FLAC__StreamDecoderErrorStatus status) {
	qWarning("QMBFLACFile::error_callback: error code %i (%s)", (unsigned int) status, ::FLAC__StreamDecoderErrorStatusString[status]);
	decode_error = true;
	return;
}

#endif /*HAVE_LIBMUSICBRAINZ*/
#endif /* HAVE_FLAC */
