/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  monitor-host.c:  This file contains the top-level monitor code,
 *    accessible from the host space. (kernel independent code)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_HOST_SPACE
#include "monitor.h"

/* Instrumentation of how many hardware interrupts were redirected
 * to the host, while the VM was running.
 */
extern unsigned intRedirCount[];

#if InstrumentMonPageFaults
unsigned monPageFaultCount = 0; // xxx Fix this
#endif


/************************************************************************/
/* Declarations                                                         */
/************************************************************************/

cpuid_info_t cpuid_info;

static int  initIDTSlot(vm_t *vm, unsigned vec, int type);
static void mapMonPages(vm_t *vm, Bit32u *, unsigned, Bit32u *, page_t *,
                          unsigned user, unsigned writable, char *name);
#if ANAL_CHECKS
static void mapBlankPage(vm_t *vm, Bit32u *laddr_p, page_t *pageTable);
#endif

#define RW0 0
#define RW1 1
#define US0 0
#define US1 1

#define IDT_INTERRUPT          0
#define IDT_EXCEPTION_ERROR    1
#define IDT_EXCEPTION_NOERROR  2



/*
 *  We need to set the monitor CS/DS base address so that the module pages,
 *  which are mapped starting at linear address 'laddr' into the guest address 
 *  space, reside at the same offset relative to the monitor CS base as they 
 *  reside relative to the kernel CS base in the host address space.  This way,
 *  we can execute the (non-relocatable) module code within the guest address 
 *  space ...
 */
#define MON_BASE_FROM_LADDR(laddr) \
    ((laddr) - monitor_pages.startOffsetPageAligned)


static selector_t nullSelector = { raw: 0 };

#if InstrumentIO
// xxx Delete this instrumentation stuff
unsigned ioAccessCount[64 * 1024];
#endif


  unsigned
hostModuleInit(void)
{
  /* Kernel independent stuff to do at kernel module load time. */

  if (!getCpuCapabilities()) {
    hostprint("getCpuCapabilities returned error\n");
    return(0); /* Fail. */
    }
  else {
#if 0
    printk(KERN_WARNING "ptype:%u, family:%u, model:%u stepping:%u\n",
      cpuid_info.procSignature.fields.procType,
      cpuid_info.procSignature.fields.family,
      cpuid_info.procSignature.fields.model,
      cpuid_info.procSignature.fields.stepping);
#endif
    }

  /* xxx Should check that host CS.base is page aligned here. */

#if 0
  /* Verify that certain code/data exists on page boundaries. */
  if ( ((Bit32u) __r3hStubsDataStart) & 0x00000fff ) {
    hostprint("__r3hStubsDataStart not on page boundary.\n");
    return(0); /* Fail. */
    }
  if ( ((Bit32u) __r3hStubsCodeStart) & 0x00000fff ) {
    hostprint("__r3hStubsCodeStart not on page boundary.\n");
    return(0); /* Fail. */
    }
  if ( ((Bit32u) __r3hSectionStart) & 0x00000fff ) {
    hostprint("__r3hSectionStart not on page boundary.\n");
    return(0); /* Fail. */
    }
  if ( ((Bit32u) __r3hSectionEnd) & 0x00000fff ) {
    hostprint("__r3hSectionEnd not on page boundary.\n");
    return(0); /* Fail. */
    }
  /* Verify that the ring3 handler pages are in the necessary order.  This
   * order is assumed when mapping them in:
   *   __r3hStubsDataStart
   *   __r3hStubsCodeStart
   *   __r3hSectionStart
   *   __r3hSectionEnd
   */
  if ( (((Bit32u) __r3hStubsCodeStart) - ((Bit32u) __r3hStubsDataStart)) !=
       0x00001000 ) {
    hostprint("r3hStubs Code does not follow Data.\n");
    return(0); /* Fail. */
    }
  if ( (((Bit32u) __r3hSectionStart) - ((Bit32u) __r3hStubsCodeStart)) !=
       0x00001000 ) {
    hostprint("r3h SectionStart does not follow StubsCodeStart.\n");
    return(0); /* Fail. */
    }

// +++ Rename r3hSection{Start,End}

  /* Initialize opcode virtualization maps.  This is done only once,
   * when the module is loaded.  Maps can be based on CPU capabilities.
   * For now, the same maps apply to all VMs.
   */
  dtInitVOpcodeMaps();
#endif

#if 1
  {
  Bit32u cr0;

  asm volatile ( "movl %%cr0, %0" : "=r" (cr0) );
  hostprint("host CR0=0x%x\n", cr0);
  }
#endif

  return(1); /* Pass. */
}

  void
hostDeviceOpenInit(vm_t *vm)
{
  /* Kernel independent stuff to do at device open time. */

  /* Set defaults of tweakable parameters */
  vm->prescanDepth = PrescanDepthDefault;
  vm->prescanRing3 = PrescanRing3On;
  vm->mon_state = MON_STATE_UNINITIALIZED;
}

  int
initMonitor(vm_t *vm, Bit32u kernel_offset, unsigned reason,
            guest_cpu_t *guest_cpu)
{
  /* These variables are used to set up the monitor pagetables */
  unsigned pdi, pti;
  unsigned int i;
  Bit32u nexus_size;
  page_t  *pageTable;
  Bit32u laddr, base;
  int r;

  vm->kernel_offset = kernel_offset;

  /* Start out using pointers in host space. */
  vm->addr = &vm->host.addr;

  vm->system.a20 = 1; /* start with A20 line enabled */
  vm->system.a20AddrMask  = 0xffffffff; /* all address lines contribute */
  vm->system.a20IndexMask = 0x000fffff; /* all address lines contribute */

  /* Initialize nexus */
  mon_memzero(vm->host.addr.nexus, 4096);

  /* Copy transition code (nexus) into code page allocated for this VM. */
  nexus_size = ((Bit32u) &__nexus_end) - ((Bit32u) &__nexus_start);
  if (nexus_size > 4096)
    goto error;
  mon_memcpy(vm->host.addr.nexus, &__nexus_start, nexus_size);


  /* Init the convenience pointers. */

  /* Pointer to host2mon routine inside nexus page */
  vm->host.__host2mon = (void (*)(void)) HOST_NEXUS_OFFSET(vm, __host2mon);

  /* Pointer to guest context on monitor stack */
  vm->host.addr.guest_context = (guest_context_t *)
    ( (Bit32u)vm->host.addr.nexus + PAGESIZE -
      (sizeof(guest_context_t) + sizeof(v86_sregs_t)) );

  /* Zero out various monitor data structures */
  mon_memzero(vm->host.addr.log_buffer, 4096*LOG_BUFF_PAGES);
  mon_memzero(&vm->log_buffer_info,
              sizeof(vm->log_buffer_info));
  mon_memzero(vm->host.addr.page_dir, 4096);
  mon_memzero(vm->host.addr.idt, MON_IDT_PAGES*4096);
  mon_memzero(vm->host.addr.gdt, MON_GDT_PAGES*4096);
  mon_memzero(vm->host.addr.ldt, MON_LDT_PAGES*4096);
  mon_memzero(vm->host.addr.tss, MON_TSS_PAGES*4096);
  mon_memzero(vm->host.addr.idt_stubs, MON_IDT_STUBS_PAGES*4096);


  /*
   *  ================
   *  Nexus Page Table
   *  ================
   *
   *  All structures needed by the monitor inside the guest environment
   *  (code to perform the transition between host<-->guest, fault handler
   *  code, various processor data structures like page directory, GDT,
   *  IDT, TSS etc.) are mapped into a single Page Table.
   *
   *  This allows us to migrate the complete nexus to anywhere in the
   *  guest address space by just updating a single (unused) page directory
   *  entry in the monitor/guest page directory to point to this nexus
   *  page table.
   *
   *  To simplify nexus migration, we try to avoid storing guest linear
   *  addresses to nexus structures as far as possible.  Instead, we use
   *  offsets relative to the monitor code/data segments.  As we update
   *  the base of these segments whenever the monitor migrates, the net
   *  effect is that those *offsets* remain valid across nexus migration. 
   */

  /* Fill in the PDE flags.  The US bit is set to 1 (user access).
   * All of the US bits in the monitor PTEs are set to 0 (system access),
   * except those containing the DT tcode pages/structures which can
   * be accessed at user level.
   */
  vm->host.nexus_pde.fields.base = vm->pages.nexus_page_tbl;
  vm->host.nexus_pde.fields.avail = 0;
  vm->host.nexus_pde.fields.G = 0;      /* not global */
  vm->host.nexus_pde.fields.PS = 0;     /* 4K pages */
  vm->host.nexus_pde.fields.D = 0;      /* (unused in pde) */
  vm->host.nexus_pde.fields.A = 0;      /* not accessed */
  vm->host.nexus_pde.fields.PCD = 0;    /* normal caching */
  vm->host.nexus_pde.fields.PWT = 0;    /* normal write-back */
  vm->host.nexus_pde.fields.US = 1;     /* user access (see above) */
  vm->host.nexus_pde.fields.RW = 1;     /* read or write */
  vm->host.nexus_pde.fields.P = 1;      /* present in memory */

  /* clear Page Table */
  pageTable = vm->host.addr.nexus_page_tbl;
  mon_memzero(pageTable, 4096);

  /* xxx Comment here */
  laddr = 0;
  base = MON_BASE_FROM_LADDR(laddr);

#if 0
  /* Map in the pages comprising the code/data of the monitor.
   * Most pages are mapped in R/W with system (U/S==0) privilege, and
   * are not accessible by the guest at ring3.  Though the ring3 handler
   * code pages for DT acceleration are mapped in as RO by user code
   * (U/S==1), and ring3 handler data page is mapped in as RW by user
   * code.  As this single data page is modified, a private copy is
   * needed by each VM.  So for this data page, an allocated page is
   * mapped in, and the original page loaded/allocated for the host
   * kernel module is not used.
   */
  {
  unsigned pageI=0;  /* Page index. */
  unsigned nPages; /* Number of pages. */

  /* Map in all host kernel module pages, up to and not including
   * the ring3 handler data page (System, RW).
   */
  nPages  = ( ((Bit32u)__r3hStubsDataStart) -
              monitor_pages.startOffsetPageAligned ) >> 12;
  if ( (nPages == 0) || ((nPages+pageI) > monitor_pages.n_pages) ) {
    hostprint("__r3hStubsDataStart OOB(1).\n");
    goto error;
    }
  mapMonPages(vm, &monitor_pages.page[pageI], nPages, &laddr,
              pageTable, US0, RW1, "Monitor code/data pages");
  pageI += nPages;

  /* Map in the ring3 handler data page (User, RW). */
  nPages = 1;
  if ( (nPages == 0) || ((nPages+pageI) > monitor_pages.n_pages) ) {
    hostprint("__r3hStubsDataStart OOB(2).\n");
    goto error;
    }
  //mapMonPages(vm, &monitor_pages.page[pageI], nPages, &laddr,   //)
  vm->guest.addr.r3hData = (r3hData_t *) (laddr - base);
  mapMonPages(vm, &vm->pages.r3hData, nPages, &laddr,
              pageTable, US1, RW1, "Monitor code/data pages");
  pageI += nPages;

  /* Map in the ring3 handler code pages (User, RO). */
  nPages = (((Bit32u)__r3hSectionEnd) - ((Bit32u)__r3hStubsCodeStart)) >> 12;
  if ( (nPages == 0) || ((nPages+pageI) > monitor_pages.n_pages) ) {
    hostprint("ring3 handler code pages OOB.\n");
    goto error;
    }
  mapMonPages(vm, &monitor_pages.page[pageI], nPages, &laddr,
              pageTable, US1, RW0, "Monitor code/data pages");
  pageI += nPages;

  /* Map in the rest of the host kernel module pages, if any (System, RW). */
  if (monitor_pages.n_pages > pageI) {
    nPages = monitor_pages.n_pages - pageI;
    mapMonPages(vm, &monitor_pages.page[pageI], nPages, &laddr,
                pageTable, US0, RW1, "Monitor code/data pages");
    }
  }
#endif
  mapMonPages(vm, monitor_pages.page, monitor_pages.n_pages, &laddr,
              pageTable, US0, RW1, "Monitor code/data pages");


#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif

  vm->guest.addr.nexus = (nexus_t *) (laddr - base);
  mapMonPages(vm, &vm->pages.nexus, 1, &laddr, pageTable, US0, RW1, "Nexus");
  vm->guest.addr.guest_context = (guest_context_t *)
    ( (Bit32u)vm->guest.addr.nexus + PAGESIZE -
      (sizeof(guest_context_t) + sizeof(v86_sregs_t)) );

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->host.addr.nexus->vm = (void *) (laddr - base);
  mapMonPages(vm, vm->pages.vm, BYTES2PAGES(sizeof(*vm)),
              &laddr, pageTable, US0, RW1, "VM structure");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.idt = (gate_t *) (laddr - base);
  mapMonPages(vm, vm->pages.idt, MON_IDT_PAGES, &laddr, pageTable, US0, RW1,
              "IDT");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.gdt = (descriptor_t *) (laddr - base);
  mapMonPages(vm, vm->pages.gdt, MON_GDT_PAGES, &laddr, pageTable, US0, RW1,
              "GDT");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.ldt = (descriptor_t *) (laddr - base);
  mapMonPages(vm, vm->pages.ldt, MON_LDT_PAGES, &laddr, pageTable, US0, RW1,
              "LDT");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.tss = (tss_t *) (laddr - base);
  mapMonPages(vm, vm->pages.tss, MON_TSS_PAGES, &laddr, pageTable, US0, RW1,
              "TSS");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.idt_stubs = (idt_stub_t *) (laddr - base);
  mapMonPages(vm, vm->pages.idt_stubs, MON_IDT_STUBS_PAGES, &laddr,
                  pageTable, US0, RW1, "IDT stubs");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Monitor Page Directory */
  vm->guest.addr.page_dir = (pageEntry_t *) (laddr - base);
  mapMonPages(vm, &vm->pages.page_dir, 1, &laddr, pageTable, US0, RW1,
              "Monitor Page Directory");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Nexus Page Table */
  vm->guest.addr.nexus_page_tbl = (page_t *) (laddr - base);
  mapMonPages(vm, &vm->pages.nexus_page_tbl, 1, &laddr, pageTable, US0, RW1,
              "Nexus Page Table");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Map virtualized guest page tables into monitor. */
  vm->guest.addr.page_tbl = (page_t *) (laddr - base);
  mapMonPages(vm, vm->pages.page_tbl, MON_PAGE_TABLES,
                  &laddr, pageTable, US0, RW1, "Guest Page Tables");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Map of linear addresses of page tables mapped into monitor */
  vm->guest.addr.page_tbl_laddr_map = (unsigned *) (laddr - base);
  mapMonPages(vm, &vm->pages.page_tbl_laddr_map, 1, &laddr, pageTable, US0, RW1,
              "Page Table Laddr Map");


#if 0
  /* DT features. */
#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.dtL2MHash = (dtL2MHash_t *) (laddr - base);
  mapMonPages(vm, vm->pages.dtL2MHash, vm->pages.dtL2MHashNPages,
                &laddr, pageTable, US0, RW1, "dtL2MHash");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.dtG2THash = (dtG2THash_t *) (laddr - base);
  mapMonPages(vm, vm->pages.dtG2THash, vm->pages.dtG2THashNPages,
                &laddr, pageTable, US1, RW1, "dtG2THash");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.dtPageMetaTable = (dtPageMeta_t *) (laddr - base);
  mapMonPages(vm, vm->pages.dtPageMetaTable,
              vm->pages.dtPageMetaTableNPages, &laddr, pageTable, US0, RW1,
              "dtPageMetaTable");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.dtPageMetaTableUsage = (Bit8u *) (laddr - base);
  mapMonPages(vm, vm->pages.dtPageMetaTableUsage,
              vm->pages.dtPageMetaTableUsageNPages, &laddr, pageTable, US0, RW1,
              "dtPageMetaTableUsage");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.tcodeChunk = (tcodeChunk_t *) (laddr - base);
  mapMonPages(vm, vm->pages.tcodeChunk, vm->pages.tcodeChunkNPages,
              &laddr, pageTable, US1, RW1, "tcodeChunk");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  vm->guest.addr.tcodeChunkUsage = (Bit8u *) (laddr - base);
  mapMonPages(vm, vm->pages.tcodeChunkUsage,
              vm->pages.tcodeChunkUsageNPages, &laddr, pageTable, US0, RW1,
              "tcodeChunkUsage");
#endif



#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /*
   *  We need a buffer to implement a debug print facility which
   *  can work in either host or monitor space.  Map the buffer
   *  into monitor/guest space.
   */
  vm->guest.addr.log_buffer = (unsigned char *) (laddr - base);
  mapMonPages(vm, vm->pages.log_buffer, LOG_BUFF_PAGES, &laddr,
              pageTable, US0, RW1, "Log Buffer");

  {
  /* The physical addresses of the following pages are not */
  /* yet established.  Pass dummy info until they are mapped. */
  Bit32u tmp[1];
  tmp[0] = 0;

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Window into the guest's current physical code page */
  vm->guest.addr.code_phy_page = (unsigned char *) (laddr - base);
  mapMonPages(vm, tmp, 1, &laddr, pageTable, US0, RW1, "Code Phy Page");

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif
  /* Temporary window into a guest physical page, for accessing */
  /* guest GDT, IDT, etc info. */
  vm->guest.addr.tmp_phy_page0 = (unsigned char *) (laddr - base);
  mapMonPages(vm, tmp, 1, &laddr, pageTable, US0, RW1, "Tmp Phy Page0");

  vm->guest.addr.tmp_phy_page1 = (unsigned char *) (laddr - base);
  mapMonPages(vm, tmp, 1, &laddr, pageTable, US0, RW1, "Tmp Phy Page1");
  }

#if ANAL_CHECKS
  mapBlankPage(vm, &laddr, pageTable);
#endif

  hostprint("Using %u/1024 PTE slots in 4Meg monitor range.\n",
            (laddr >> 12) & 0x3ff);

  /* Pointer to mon2host routine inside nexus page */
  vm->guest.__mon2host = (void (*)(void)) MON_NEXUS_OFFSET(vm, __mon2host);


  /*
   *  =====================
   *  Transition Page Table
   *  =====================
   *
   *  To aid in the transition between host<-->monitor/guest spaces,
   *  we need to have an address identity map situation for at least
   *  one page; the page containing the transition code.   As we do
   *  not know in advance whether this linear address range is in use
   *  by the guest as well, we set aside a complete additional Page
   *  Table, which contains only a single PTE pointing to the nexus page.
   *
   *  To create the identity map, we simply change the corresponding
   *  monitor page directory entry to point to this transition Page Table.
   *  This happens transparently inside the host<-->guest transition code; 
   *  both the guest/monitor code and the host side code never see this 
   *  transition page table entered into the page directory!
   *
   *  NOTE: We need to ensure that the nexus page table never spans the
   *        same 4Meg linear address space region as this page table!
   *        As we are free to choose the nexus linear address, this is
   *        not a problem.
   */

  /* Get full linear address of nexus code page, as seen in host space. */
  laddr = (Bit32u)vm->host.addr.nexus + kernel_offset;
  pdi = laddr >> 22;
  pti = (laddr >> 12) & 0x3ff;

  /*
   *  We need to be able to access the PDE in the monitor page directory
   *  that corresponds to this linear address from both host and monitor 
   *  address spaces.
   */
  vm->host.addr.nexus->transition_pde_p_host = vm->host.addr.page_dir + pdi;
  vm->host.addr.nexus->transition_pde_p_mon  = (pageEntry_t *)
                        (((Bit32u)vm->guest.addr.page_dir) + (pdi << 2));
  vm->host.addr.nexus->transition_laddr = laddr;

  /* Fill in the PDE flags */
  vm->host.addr.nexus->transition_pde.fields.base = vm->pages.transition_PT;
  vm->host.addr.nexus->transition_pde.fields.avail = 0;
  vm->host.addr.nexus->transition_pde.fields.G = 0;   /* not global */
  vm->host.addr.nexus->transition_pde.fields.PS = 0;  /* 4K pages */
  vm->host.addr.nexus->transition_pde.fields.D = 0;   /* (unused in pde) */
  vm->host.addr.nexus->transition_pde.fields.A = 0;   /* not accessed */
  vm->host.addr.nexus->transition_pde.fields.PCD = 0; /* normal caching */
  vm->host.addr.nexus->transition_pde.fields.PWT = 0; /* normal write-back*/
  vm->host.addr.nexus->transition_pde.fields.US = 0;  /* no user access  */
  vm->host.addr.nexus->transition_pde.fields.RW = 1;  /* read or write */
  vm->host.addr.nexus->transition_pde.fields.P = 1;   /* present in memory*/

  /* Clear Page Table; only one PTE is used. */
  pageTable = vm->host.addr.transition_PT;
  mon_memzero(pageTable, 4096);

  /* Fill in the PTE for identity mapping the code page */
  pageTable->pte[pti].fields.base = vm->pages.nexus;
  pageTable->pte[pti].fields.avail = 0;
  pageTable->pte[pti].fields.G = 0;      /* not global          */
  pageTable->pte[pti].fields.PS = 0;     /* (unused in pte)     */
  pageTable->pte[pti].fields.D = 0;      /* clean               */
  pageTable->pte[pti].fields.A = 0;      /* not accessed        */
  pageTable->pte[pti].fields.PCD = 0;    /* normal caching      */
  pageTable->pte[pti].fields.PWT = 0;    /* normal write-back   */
  pageTable->pte[pti].fields.US = 0;     /* user can not access */
  pageTable->pte[pti].fields.RW = 1;     /* read or write       */
  pageTable->pte[pti].fields.P = 1;      /* present in memory   */


  /* 
   *  Setup the TSS for the monitor/guest environment
   *
   *  We don't need to set the pagedir in the TSS, because we don't 
   *  actually jump to it anyway.  The TSS is just used to set the kernel 
   *  stack and in a later stage, perhaps the I/O permission bitmap.
   */

  /* No task chain */
  vm->host.addr.tss->back = 0;

  /* No debugging or I/O, for now */
  vm->host.addr.tss->trap = 0;
  vm->host.addr.tss->io = sizeof(tss_t);

  /* Monitor stack offset */
  vm->host.addr.tss->esp0 =
    ((Bit32u)vm->guest.addr.nexus) + PAGESIZE - sizeof(v86_sregs_t);


  /*
   * Set up initial monitor code and stack offset
   */

  vm->host.addr.nexus->mon_jmp_info.offset   = MON_NEXUS_OFFSET(vm, __mon_cs);
  vm->host.addr.nexus->mon_stack_info.offset =
      vm->host.addr.tss->esp0 - (sizeof(guest_context_t) + 48);
// xxx 48 above should be calculated from code below which winds
// xxx up monitor stack.


  /*
   *  Setup the IDT for the monitor/guest environment
   */

  r = 0;
  r |= initIDTSlot(vm,  0, IDT_EXCEPTION_NOERROR); /* Divide error        */
  r |= initIDTSlot(vm,  1, IDT_EXCEPTION_NOERROR); /* Debug exceptions    */
  r |= initIDTSlot(vm,  2, IDT_INTERRUPT);         /* NMI                 */
  r |= initIDTSlot(vm,  3, IDT_EXCEPTION_NOERROR); /* Breakpoint          */
  r |= initIDTSlot(vm,  4, IDT_EXCEPTION_NOERROR); /* Overflow            */
  r |= initIDTSlot(vm,  5, IDT_EXCEPTION_NOERROR); /* Bounds check        */
  r |= initIDTSlot(vm,  6, IDT_EXCEPTION_NOERROR); /* Invalid opcode      */
  r |= initIDTSlot(vm,  7, IDT_EXCEPTION_NOERROR); /* FPU not available   */
  r |= initIDTSlot(vm,  8, IDT_EXCEPTION_ERROR);   /* Double fault        */
  r |= initIDTSlot(vm,  9, IDT_EXCEPTION_NOERROR); /* FPU segment overrun */
  r |= initIDTSlot(vm, 10, IDT_EXCEPTION_ERROR);   /* Invalid TSS         */
  r |= initIDTSlot(vm, 11, IDT_EXCEPTION_ERROR);   /* Segment not present */
  r |= initIDTSlot(vm, 12, IDT_EXCEPTION_ERROR);   /* Stack exception     */
  r |= initIDTSlot(vm, 13, IDT_EXCEPTION_ERROR);   /* GP fault            */
  r |= initIDTSlot(vm, 14, IDT_EXCEPTION_ERROR);   /* Page fault          */
  r |= initIDTSlot(vm, 15, IDT_EXCEPTION_NOERROR); /* reserved            */
  r |= initIDTSlot(vm, 16, IDT_EXCEPTION_NOERROR); /* Coprocessor error   */
  r |= initIDTSlot(vm, 17, IDT_EXCEPTION_ERROR);   /* Alignment check     */
  r |= initIDTSlot(vm, 18, IDT_EXCEPTION_NOERROR); /* Machine check       */

  /* Reserved exceptions */
  for (i = 19; i < 32; i++)
      r |= initIDTSlot(vm, i, IDT_EXCEPTION_NOERROR);

  /* Hardware interrupts */
  for (i = 32; i < 256; i++)
      r |= initIDTSlot(vm, i, IDT_INTERRUPT);
  if (r!=0) 
      goto error;


  /*
   *  Setup the initial guest context
   */

  mon_memzero(vm->host.addr.guest_context, sizeof(guest_context_t));

  /* Wind up the monitor stack such that it looks like we just finished */
  /* handling an event (like an interrupt), and are returning back to */
  /* the tail end of the interrupt stub. */
  {
  Bit32u *ptr;

  ptr = (Bit32u *) (((unsigned char *) vm->host.addr.guest_context) - 4);
  *ptr-- = (Bit32u) &__ret_to_guest;
  *ptr-- = 0x02; /* eflags: only reserved bit on */
  *ptr-- = 0; /* eax */
  *ptr-- = 0; /* ecx */
  *ptr-- = 0; /* edx */
  *ptr-- = 0; /* ebx */
  *ptr-- = 0; /* esp dummy */
  *ptr-- = 0; /* ebp */
  *ptr-- = 0; /* esi */
  *ptr-- = 0; /* edi */
  *ptr-- = 0; /* FS; start with null value. */
  *ptr-- = 0; /* GS; start with null value. */
  }

  /* Only support one monitor->user message at a time.  Always 1. */
  vm->mon_msgs.header.msg_type = VMMessageNone;
  vm->mon_msgs.header.msg_len  = 0;
  vm->mon_state = MON_STATE_RUNNABLE;
  vm->mon_request = MON_REQ_NONE;

  /* Initialize DT features. */
  //dtInitialize(vm);

  monprint(vm, "init_monitor OK -----------------\n");
  monprint(vm, "nexus_size = 0x%x\n", nexus_size);

  initTimers(vm);
  ioInit(vm);

  return(1); /* all OK */

error:
  return(0); /* error */
}

  int
setGuestCPU(vm_t *vm, unsigned reason, guest_cpu_t *guest_cpu)
{
  Bit32u eflags, cr0;

  vm->modeChange = ModeChangeEventPaging | ModeChangeRequestPaging;

  eflags = guest_cpu->eflags;
  cr0    = guest_cpu->cr0;

  /* EFLAGS: virtualized guest flags are stored in veflags, */
  /*   the rest are stored in guest_context->eflags. */
  /* Start out by setting the virtualized flags (the ones */
  /* needed for the monitor). */
  vm->guest_cpu.veflags.raw = 0;
  vm->host.addr.guest_context->eflags.raw = FLG_IF | 0x02;
  /* Run RM code in V86 mode. */
  if ( !(cr0 & 0x1) )
    vm->host.addr.guest_context->eflags.raw |= FLG_VM;
  write_eflags(vm, eflags, ~0);


  vm->host.addr.guest_context->eax = guest_cpu->eax;
  vm->host.addr.guest_context->ebx = guest_cpu->ebx;
  vm->host.addr.guest_context->ecx = guest_cpu->ecx;
  vm->host.addr.guest_context->edx = guest_cpu->edx;
  vm->host.addr.guest_context->ebp = guest_cpu->ebp;
  vm->host.addr.guest_context->esi = guest_cpu->esi;
  vm->host.addr.guest_context->edi = guest_cpu->edi;
  vm->host.addr.guest_context->esp = guest_cpu->esp;

  vm->host.addr.guest_context->eip = guest_cpu->eip;

  /* Set CPL */
  if ( !(guest_cpu->cr0 & 0x1) )
    vm->guest_cpu.cpl = 0; /* RM */
  else if (guest_cpu->eflags & FLG_VM)
    vm->guest_cpu.cpl = 3; /* V86 */
  else
    vm->guest_cpu.cpl = guest_cpu->cs.des.dpl;

  vm->guest_cpu.selector[SRegCS] = guest_cpu->cs.sel;
  vm->guest_cpu.desc_cache[SRegCS].valid = guest_cpu->cs.valid;
  vm->guest_cpu.desc_cache[SRegCS].desc = guest_cpu->cs.des;
  vm->guest_cpu.desc_cache[SRegCS].base =
    BaseOfDescriptor(guest_cpu->cs.des);
  vm->guest_cpu.desc_cache[SRegCS].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->cs.des);

  vm->guest_cpu.selector[SRegDS] = guest_cpu->ds.sel;
  vm->guest_cpu.desc_cache[SRegDS].valid = guest_cpu->ds.valid;
  vm->guest_cpu.desc_cache[SRegDS].desc = guest_cpu->ds.des;
  vm->guest_cpu.desc_cache[SRegDS].base =
    BaseOfDescriptor(guest_cpu->ds.des);
  vm->guest_cpu.desc_cache[SRegDS].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->ds.des);

  vm->guest_cpu.selector[SRegES] = guest_cpu->es.sel;
  vm->guest_cpu.desc_cache[SRegES].valid = guest_cpu->es.valid;
  vm->guest_cpu.desc_cache[SRegES].desc = guest_cpu->es.des;
  vm->guest_cpu.desc_cache[SRegES].base =
    BaseOfDescriptor(guest_cpu->es.des);
  vm->guest_cpu.desc_cache[SRegES].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->es.des);

  vm->guest_cpu.selector[SRegFS] = guest_cpu->fs.sel;
  vm->guest_cpu.desc_cache[SRegFS].valid = guest_cpu->fs.valid;
  vm->guest_cpu.desc_cache[SRegFS].desc = guest_cpu->fs.des;
  vm->guest_cpu.desc_cache[SRegFS].base =
    BaseOfDescriptor(guest_cpu->fs.des);
  vm->guest_cpu.desc_cache[SRegFS].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->fs.des);

  vm->guest_cpu.selector[SRegGS] = guest_cpu->gs.sel;
  vm->guest_cpu.desc_cache[SRegGS].valid = guest_cpu->gs.valid;
  vm->guest_cpu.desc_cache[SRegGS].desc = guest_cpu->gs.des;
  vm->guest_cpu.desc_cache[SRegGS].base =
    BaseOfDescriptor(guest_cpu->gs.des);
  vm->guest_cpu.desc_cache[SRegGS].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->gs.des);

  vm->guest_cpu.selector[SRegSS] = guest_cpu->ss.sel;
  vm->guest_cpu.desc_cache[SRegSS].valid = guest_cpu->ss.valid;
  vm->guest_cpu.desc_cache[SRegSS].desc = guest_cpu->ss.des;
  vm->guest_cpu.desc_cache[SRegSS].base =
    BaseOfDescriptor(guest_cpu->ss.des);
  vm->guest_cpu.desc_cache[SRegSS].limit_scaled =
    LimitOfDataDescriptor(guest_cpu->ss.des);

#warning "fix LDTR/TR values passed to setGuestCPU"
  vm->guest_cpu.ldtr_selector.raw = 0;
  vm->guest_cpu.ldtr_cache.valid = 0;
  vm->guest_cpu.tr_selector.raw = 0;
  vm->guest_cpu.tr_cache.valid = 0;

  vm->guest_cpu.gdtr.base = guest_cpu->gdtr.base;
  vm->guest_cpu.gdtr.limit = guest_cpu->gdtr.limit;
  vm->guest_cpu.idtr.base = guest_cpu->idtr.base;
  vm->guest_cpu.idtr.limit = guest_cpu->idtr.limit;

  vm->guest_cpu.dr0 = guest_cpu->dr0;
  vm->guest_cpu.dr1 = guest_cpu->dr1;
  vm->guest_cpu.dr2 = guest_cpu->dr2;
  vm->guest_cpu.dr3 = guest_cpu->dr3;
  vm->guest_cpu.dr6 = guest_cpu->dr6;
  vm->guest_cpu.dr7 = guest_cpu->dr7;

  vm->guest_cpu.tr3 = guest_cpu->tr3;
  vm->guest_cpu.tr4 = guest_cpu->tr4;
  vm->guest_cpu.tr5 = guest_cpu->tr5;
  vm->guest_cpu.tr6 = guest_cpu->tr6;
  vm->guest_cpu.tr7 = guest_cpu->tr7;

  vm->guest_cpu.cr0.raw = guest_cpu->cr0 | 0x32; /* +++ hack for now */
  vm->guest_cpu.cr2     = guest_cpu->cr2;
  vm->guest_cpu.cr3     = guest_cpu->cr3;
  vm->guest_cpu.cr4.raw = guest_cpu->cr4;

  vm->guest_cpu.inhibit_mask = 0;
  vm->guest_cpu.async_event = 0;
  vm->guest_cpu.EXT = 0;
  vm->guest_cpu.debug_trap = 0;
  vm->guest_cpu.errorno = 0;

  vm->guest_cpu.prev_esp = -1;
  vm->guest_cpu.prev_eip = -1;

  /* Mark that all selector/descriptor cache values are currently
   * stored in guest_cpu area.
   */
  vm->segmentUpdated  = 0x3f;

  return(1); /* OK */
}


  unsigned
initGuestPhyMem(vm_t *vm)
{
  unsigned i;
  mon_memzero(vm->page_usage, sizeof(vm->page_usage));
  for (i=0; i<vm->pages.guest_n_pages; i++) {
    /* For now, we start out by preallocating physical pages */
    /* for the guest, though not necessarily mapped into linear */
    /* space. */
    vm->page_usage[i].attr.raw = 0;
    vm->page_usage[i].tsc = 0;
    vm->page_usage[i].attr.fields.allocated = 1;
    }
 
  {
  Bit32u rom_page;
  unsigned npages;
 
  /* Mark BIOS ROM area as ReadOnly */
  rom_page = 0xf0000 >> 12;
  npages = (1 + 0xfffff - 0xf0000) / 4096;
  for (i=0; i<npages; i++)
    vm->page_usage[rom_page + i].attr.fields.RO = 1;

  /* Mark VGA BIOS ROM area as ReadOnly */
  rom_page = 0xc0000 >> 12;
  npages = (1 + 0xc7fff - 0xc0000) / 4096;
  for (i=0; i<npages; i++)
    vm->page_usage[rom_page + i].attr.fields.RO = 1;
  }
 
#if 1
  /* Mark VGA framebuffer area as Memory Mapped IO */
  {
  Bit32u vga_page;
  unsigned npages;
 
  vga_page = 0xa0000 >> 12;
  npages = (1 + 0xbffff - 0xa0000) / 4096;
  for (i=0; i<npages; i++)
    vm->page_usage[vga_page + i].attr.fields.memMapIO = 1;
  }
#endif

  return(0);
}


  int
initIDTSlot(vm_t *vm, unsigned vec, int type)
/*
 *  initIDTSlot():  Initialize a monitor IDT slot.
 */
{
  /* IDT slot stubs */

  idt_stub_t *stub = &vm->host.addr.idt_stubs[vec];
  Bit32u stub_mon = ((Bit32u) vm->guest.addr.idt_stubs) +
                    vec*sizeof(idt_stub_t);

  if (sizeof(idt_stub_t) != IDT_STUB_SIZE)
    return( -1 );

  switch (type) {
    case IDT_INTERRUPT:
      stub->m2.pushla = 0x68;
      stub->m2.dummy  = 0;
      stub->m2.pushlb = 0x68;
      stub->m2.vector = vec;
      stub->m2.jmp    = 0xe9;
      stub->m2.reloc  = ((Bit32u) &__handle_int) -
        (stub_mon + sizeof(idt_method2_t));
      break;

    case IDT_EXCEPTION_ERROR:
      stub->m1.pushl  = 0x68;
      stub->m1.vector = vec;
      stub->m1.jmp    = 0xe9;
      stub->m1.reloc  = ((Bit32u) &__handle_fault) -
        (stub_mon + sizeof(idt_method1_t));
      break;

    case IDT_EXCEPTION_NOERROR:
      stub->m2.pushla = 0x68;
      stub->m2.dummy  = 0;
      stub->m2.pushlb = 0x68;
      stub->m2.vector = vec;
      stub->m2.jmp    = 0xe9;
      stub->m2.reloc  = ((Bit32u) &__handle_fault) -
        (stub_mon + sizeof(idt_method2_t));
      break;

    default:
      return -1;
    }

  /* Set the interrupt gate */
  SET_INT_GATE(vm->host.addr.idt[vec],
               nullSelector, stub_mon, D_PRESENT, D_DPL0, D_D32);
  return 0;
}


/*
 *  Map pages allocated by host, into the linear address space of
 *  the monitor/guest, given the Page Table supplied.
 */

  void
mapMonPages(vm_t *vm, Bit32u *pages, unsigned n, Bit32u *laddr_p,
            page_t *pageTable, unsigned user, unsigned writable, char *name)
{
  unsigned i, pti;


#if 0
hostprint("mapMonPages: '%s' mapped at 0x%x .. 0x%x.\n",
          name,
          (*laddr_p) - MON_BASE_FROM_LADDR(0),
          ((*laddr_p) + (n*4096)) - MON_BASE_FROM_LADDR(0) );
#endif

  pti = (*laddr_p >> 12) & 0x3ff;
  for (i = 0; i < n; i++, pti++) {
    if (pti > 1024)
        break;  /* This should not happen! */

    /* Fill in the PTE flags */
    pageTable->pte[pti].fields.base = pages[i];
    pageTable->pte[pti].fields.avail = 0;
    pageTable->pte[pti].fields.G = 0;         /* not global */
    pageTable->pte[pti].fields.PS = 0;        /* (unused in pte) */
    pageTable->pte[pti].fields.D = 0;         /* clean */
    pageTable->pte[pti].fields.A = 0;         /* not accessed */
    pageTable->pte[pti].fields.PCD = 0;       /* normal caching */
    pageTable->pte[pti].fields.PWT = 0;       /* normal write-back */
    pageTable->pte[pti].fields.US = user;     /* 0=system, 1=user */
    pageTable->pte[pti].fields.RW = writable; /* 0=RO, 1=RW */
    pageTable->pte[pti].fields.P = 1;         /* present in memory */
    }

  /*
   *  Advance linear address pointer, for the next set of pages
   *  to be mapped.
   */
  *laddr_p += 4096 * n;
}

#if ANAL_CHECKS
  void
mapBlankPage(vm_t *vm, Bit32u *laddr_p, page_t *pageTable)
{
  unsigned pti;
 
  pti = (*laddr_p >> 12) & 0x3ff;
  if (pti > 1024)
    return;  /* This should not happen! */
 
  /* Fill in the PTE flags */
  pageTable->pte[pti].fields.base = 0;
  pageTable->pte[pti].fields.avail = 0;
  pageTable->pte[pti].fields.G = 0;      /* not global */
  pageTable->pte[pti].fields.PS = 0;     /* (unused in pte) */
  pageTable->pte[pti].fields.D = 0;      /* clean */
  pageTable->pte[pti].fields.A = 0;      /* not accessed */
  pageTable->pte[pti].fields.PCD = 0;    /* normal caching */
  pageTable->pte[pti].fields.PWT = 0;    /* normal write-back */
  pageTable->pte[pti].fields.US = 0;
  pageTable->pte[pti].fields.RW = 0;
  pageTable->pte[pti].fields.P = 0;
 
  /*
   *  Advance linear address pointer, for the next set of pages
   *  to be mapped.
   */
  *laddr_p += 4096;
}
#endif


  int
runGuestLoop(vm_t *vm)
/*
 *  Main VM loop
 */
{
  /* Sanity check */
  if ( vm->mon_state == MON_STATE_UNINITIALIZED ||
       vm->mon_state == MON_STATE_PANIC ) {
    monprint(vm, "runGuestLoop: mon not in runnable state.\n");
    return 1;
    }

  for (;;) {
    unsigned long eflags;

    /* If print buffer has contents, return to user space to print. */
    if (vm->log_buffer_info.offset) {
      vm->mon_msgs.header.msg_type = VMMessagePrintBuf;
      vm->mon_msgs.header.msg_len  = 0;
      vm->mon_request = MON_REQ_NONE; /* Request satisfied */
      resetPrintBuf(vm); // xxx Fix print mess
      return( 0 );
      }

    vm_save_flags(eflags);
    vm_restore_flags(eflags & ~0x00004300); /* clear NT/IF/TF */
#if ANAL_CHECKS
    if (!(eflags & 0x200)) {
      vm_restore_flags(eflags);
      hostprint("runGuestLoop: EFLAGS.IF==0\n");
      return 1;
      }
#endif

    /* Call assembly routine to effect transition. */
    vm->addr = &((vm_t *)vm->host.addr.nexus->vm)->guest.addr;
    vm->host.__host2mon();
    vm->addr = &vm->host.addr;

    /* First check for an asynchronous event (interrupt redirection) */
    if ( vm->mon_request == MON_REQ_REDIRECT ) {
      vm_restore_flags(eflags & ~0x00000200); /* restore all but IF */
      soft_int(vm->redirect_vector); /* sets IF to 1 */
      intRedirCount[vm->redirect_vector]++;
      vm->mon_request = MON_REQ_NONE; /* Request satisfied */
      }

    /* Event was synchronous; monitor requested a switch back to host. */
    else {
      vm_restore_flags(eflags);

      /* Perform action requested by monitor */
      switch ( vm->mon_request ) {
        case MON_REQ_REMAP_MONITOR:
          {
          if ( mapMonitor(vm, read_eflags(vm), 0) ) {
            vm->mon_request = MON_REQ_NONE; /* Request satisfied */
            break;
            }
          else {
            vm->mon_msgs.header.msg_type = VMMessagePanic;
            vm->mon_msgs.header.msg_len = 0;
            vm->mon_request = MON_REQ_PANIC;
            vm->mon_state   = MON_STATE_PANIC;
            hostprint("mapMonitor failed.\n");
            hostprint("Panic w/ abort_code=%u\n", vm->abort_code);
            return 0;
            }
          }

        case MON_REQ_RESPONSE_PENDING:
#if InstrumentVMMessages
          vm->vmMessageCount[vm->mon_msgs.header.msg_type]++;
#endif
          return 0;

        case MON_REQ_NO_RESPONSE:
#if InstrumentVMMessages
          vm->vmMessageCount[vm->mon_msgs.header.msg_type]++;
#endif
          vm->mon_request = MON_REQ_NONE; /* Request satisfied */
          return 0;

        case MON_REQ_PANIC:
          if (vm->abort_code)
            hostprint("Panic w/ abort_code=%u\n", vm->abort_code);
          return 0;

        case MON_REQ_NOP: /* Do nothing. */
          vm->mon_request = MON_REQ_NONE; /* Request satisfied */
          break;

        default:
          monprint(vm, "default mon_request (%u).\n", vm->mon_request);
          return 1;
        }
      }

    /* Let host decide whether we are allowed another timeslice */
    if ( !host_idle() ) {
      /* We are returning only because the host wants to */
      /* schedule other work.  There are no messages, so clear */
      /* out the last message. */
      ClearMonMessageQ(vm);
      return 0;
      }
    }
}

  void
getCpuResetValues(guest_cpu_t *cpu)
{
  /* Set fields in a guest_cpu_t structure to RESET values */
  mon_memzero(cpu, sizeof(*cpu));

  /* General Registers */
  cpu->eax = 0; /* processor tests passed */
  cpu->ebx = 0;
  cpu->ecx = 0;
  cpu->edx = (cpuid_info.procSignature.fields.family<<8) |
             cpuid_info.procSignature.fields.stepping;
  cpu->ebp = 0;
  cpu->esi = 0;
  cpu->edi = 0;
  cpu->esp = 0;
  cpu->eflags = 0x00000002; /* only reserved bit set */
#if USE_LOADER
  cpu->eip = 0x0000;
#else
  cpu->eip = 0xfff0;
#endif

  /* CS */
  cpu->cs.sel.raw = 0xf000;
  cpu->cs.des.limit_low = 0xffff;
  cpu->cs.des.base_low = 0x0000;
  cpu->cs.des.base_med = 0x0f;
  cpu->cs.des.type = D_CODE | D_WRITE | D_ACCESSED;
  cpu->cs.des.dpl = 0;
  cpu->cs.des.p = 1;
  cpu->cs.des.limit_high = 0;
  cpu->cs.des.avl = 0;
  cpu->cs.des.reserved = 0;
  cpu->cs.des.d_b = 0;
  cpu->cs.des.g = 0;
  cpu->cs.des.base_high = 0x00;
  cpu->cs.valid = 1;

  /* DS */
  cpu->ds.sel.raw = 0;
  cpu->ds.des.limit_low = 0xffff;
  cpu->ds.des.base_low = 0;
  cpu->ds.des.base_med = 0;
  cpu->ds.des.type = D_DATA | D_WRITE | D_ACCESSED;
  cpu->ds.des.dpl = 0;
  cpu->ds.des.p = 1;
  cpu->ds.des.limit_high = 0;
  cpu->ds.des.avl = 0;
  cpu->ds.des.reserved = 0;
  cpu->ds.des.d_b = 0;
  cpu->ds.des.g = 0;
  cpu->ds.des.base_high = 0;
  cpu->ds.valid = 1;

  /* Copy DS to other data segments */
  cpu->ss = cpu->ds;
  cpu->es = cpu->ds;
  cpu->fs = cpu->ds;
  cpu->gs = cpu->ds;

  /* LDTR */
  cpu->ldtr.sel.raw = 0;
  /* +++ check out Table 8-1 */
  cpu->ldtr.valid = 0;

  /* TR */
  cpu->tr.sel.raw = 0;
  /* +++ check out Table 8-1 */
  cpu->tr.valid = 0;

  /* GDTR */
  cpu->gdtr.base = 0;
  /* +++ check out Table 8-1 */
  cpu->gdtr.limit = 0;

  /* IDTR */
  cpu->idtr.base = 0;
  /* +++ check out Table 8-1 */
  cpu->idtr.limit = 0x03ff;

  /* Debug Registers */
  cpu->dr0 = 0;
  cpu->dr1 = 0;
  cpu->dr2 = 0;
  cpu->dr3 = 0;
  if (cpuid_info.procSignature.fields.family <= 4) {
    cpu->dr6 = 0xFFFF1FF0;
    cpu->dr7 = 0x00000000;
    }
  else {
    cpu->dr6 = 0xFFFF0FF0;
    cpu->dr7 = 0x00000400;
    }

  /* Test Registers */
  cpu->tr3 = 0;
  cpu->tr4 = 0;
  cpu->tr5 = 0;
  cpu->tr6 = 0;
  cpu->tr7 = 0;

  /* Control Registers */
  /* +++ CR0 reset: CD/NW unchanged, bit4 set, other cleared */
  cpu->cr0 = 0x60000010;
  cpu->cr1 = 0;
  cpu->cr2 = 0;
  cpu->cr3 = 0;
  cpu->cr4 = 0;

  cpu->inhibit_mask = 0;
}



  int
ioctlSetA20E(vm_t *vm, unsigned long val)
{
  unsigned prev_a20;

  if ( !val && vm->guest_cpu.cr0.fields.pg ) {
    /* Disabling the A20 line with paging on, is not
     * currently handled
     */
    monprint(vm, "SetA20E: val=0, CR0.PG=1 unhandled\n");
    return 0; /* fail */
    }
  val = (val > 0); /* make 0 or 1 */
  prev_a20 = vm->system.a20;

  vm->system.a20 = val;
  vm->system.a20AddrMask  = 0xffefffff | (val << 20);
  vm->system.a20IndexMask = 0x000ffeff | (val << 8);

  if ( prev_a20 != vm->system.a20 ) {
    /* change in A20 line enable status */
    vm->modeChange |= ModeChangeEventPaging | ModeChangeRequestPaging;
    }

  return 1; /* OK */
}


  int
ioctlMessageQ(vm_t *vm, vm_messages_t *user_msgs)
{
    if ( vm->mon_state != MON_STATE_RUNNABLE )
    {
        monprint(vm, "ioctlMessageQ failed: not runnable state\n");
        goto panic;
    }

    switch (user_msgs->header.msg_type) 
    {
    case VMMessageIOInResponse:
    {
        IO_msg_t *user_io_msg = (IO_msg_t *) user_msgs->msg;
        IO_msg_t *mon_io_msg  = (IO_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIOInRequest) ||
             (user_io_msg->port != mon_io_msg->port) ||
             (user_io_msg->len != mon_io_msg->len) ||
             (user_io_msg->op != mon_io_msg->op) ) 
        {
            monprint(vm, "plex86: IOResponse mismatch.\n");
            goto panic;
        }

        mon_io_msg->data = user_io_msg->data;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (runGuestLoop(vm)) 
        {
            monprint(vm, "run_guest failed after IOInResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageIOBatchResponse:
    {
        IOBatch_msg_t *user_io_msg = (IOBatch_msg_t *) user_msgs->msg;
        IOBatch_msg_t *mon_io_msg  = (IOBatch_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIOBatchRequest) ||
             (user_io_msg->port != mon_io_msg->port) ||
             (user_io_msg->len != mon_io_msg->len) ||
             (user_io_msg->op != mon_io_msg->op) ) 
        {
            monprint(vm, "plex86: IOBatchResponse mismatch.\n");
            goto panic;
        }

        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        mon_io_msg->n = user_io_msg->n;
        if (runGuestLoop(vm)) 
        {
            monprint(vm, "run_guest failed after IOInResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageMemMapIOReadResponse:
    {
        memMapIO_msg_t *user_msg = (memMapIO_msg_t *) user_msgs->msg;
        memMapIO_msg_t *mon_msg  = (memMapIO_msg_t *) vm->mon_msgs.msg;
 
        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageMemMapIOReadRequest) ||
             (user_msg->addr != mon_msg->addr) ||
             (user_msg->len != mon_msg->len) ||
             (user_msg->op != mon_msg->op) )
        {
            monprint(vm, "plex86: MemMapIOReadResponse mismatch.\n");
            goto panic;
        }
 
        mon_msg->data = user_msg->data;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (runGuestLoop(vm))
        {
            monprint(vm, "run_guest failed after MemMapIOReadResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageIntResponse:
    {
        INT_msg_t *user_int_msg = (INT_msg_t *) user_msgs->msg;
        INT_msg_t *mon_int_msg  = (INT_msg_t *) vm->mon_msgs.msg;

        /* Make sure user response is valid for monitor request */
        if ( (vm->mon_request != MON_REQ_RESPONSE_PENDING) ||
             (vm->mon_msgs.header.msg_type != VMMessageIntRequest) ||
             (user_int_msg->vector != mon_int_msg->vector) )
        {
            monprint(vm, "plex86: IntResponse mismatch.\n");
            goto panic;
        }

        mon_int_msg->reflect = user_int_msg->reflect;
        vm->mon_request = MON_REQ_NONE; /* request is satisfied */
        if (runGuestLoop(vm)) 
        {
            monprint(vm, "run_guest failed after IntResponse\n");
            goto panic;
        }
        break;
    }

    case VMMessageRunGuestN:
        {
        run_guest_n_t *msg_p;
        if ( (vm->mon_request != MON_REQ_NONE) ||
             (vm->mon_state != MON_STATE_RUNNABLE) )
        {
            monprint(vm, "plex86: RunGuest: not runnable state (%u)\n",
                         vm->mon_state);
            goto panic;
        }

        ClearMonMessageQ(vm);

        msg_p = (run_guest_n_t *) user_msgs->msg;

        switch (msg_p->method) {
          case RunGuestNMethodExecute:
            /* Normal execution; run indefinitely.  As we will run guest
             * code natively (but under SIV control), we don't have
             * enough control to stop execution exactly after a certain
             * number of instructions.  Thus, a count of
             * ICOUNT_INDEFINITE must be passed.
             */
            if (msg_p->icount != ICOUNT_INDEFINITE) {
              monprint(vm, "RunGuestN: MethodExecute w/ icount!=indef\n");
              goto panic;
              }
            vm->executeN = 0;
            vm->executeMethod = msg_p->method;
            break;

          case RunGuestNMethodEmulate:
          case RunGuestNMethodBreakpoint:
            /* Execute guest code for N instructions. */
            if (msg_p->icount == ICOUNT_INDEFINITE) {
              /* We can use these methods to run code indefinitely,
               * but execution will be very slow comparatively.
               * Usually, this is for debugging when you don't know
               * how long before some interesting event occurs.
               */
              vm->executeN = 0;
              }
            else if (msg_p->icount == ICOUNT_CONTINUE) {
              /* Continue executing, using the current icount
               * value.  This is for the case when execution was
               * interrupted to service something (like IO) and
               * control was transferred back to the host temporarily.
               */
              if (msg_p->icount == 0) {
                monprint(vm, "ICOUNT_CONTINUE w/ icount=0\n");
                goto panic;
                }
              if (msg_p->method != vm->executeMethod) {
                monprint(vm, "RunGuestN: continue: different method.\n");
                goto panic;
                }
              /* Use existing vm->executeN value */
              }
            else {
              vm->executeN = msg_p->icount;
              }
            vm->executeMethod = msg_p->method;
            break;

          default:
            monprint(vm, "RunGuestN: unknown method %u\n", msg_p->method);
            goto panic;
          }

        if (runGuestLoop(vm)) 
        {
            monprint(vm, "run_guest failed after RunGuest\n");
            goto panic;
        }
        break;
        }

    default:
        monprint(vm, "runGuestLoop: unknown message type (%u)\n",
                 user_msgs->header.msg_type);
        return 1;
    }


    /* Copy monitor message to user space */
    mon_memcpy(user_msgs, &vm->mon_msgs, 
               vm->mon_msgs.header.msg_len + sizeof(vm->mon_msgs.header));

    /*if (vm->log_buffer_info.event && !vm->log_buffer_info.locked)
     *    host_print_buf(vm); */
    return 0;


 panic:
    vm->mon_request = MON_REQ_PANIC;
    vm->mon_state   = MON_STATE_PANIC;
    vm->mon_msgs.header.msg_type = VMMessagePanic;
    vm->mon_msgs.header.msg_len = 0;

    mon_memcpy(user_msgs, &vm->mon_msgs, 
               vm->mon_msgs.header.msg_len + sizeof(vm->mon_msgs.header));

    /*if (vm->log_buffer_info.event && !vm->log_buffer_info.locked)
     *    host_print_buf(vm); */
    return 0;
}


  void
getGuestCpuState(vm_t *vm, guest_cpu_t *cpu)
{
  /* Get complete guest CPU state */
  guest_context_t *context;

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  cache_sreg(vm, SRegDS);
  cache_sreg(vm, SRegES);
  cache_sreg(vm, SRegFS);
  cache_sreg(vm, SRegGS);

  context = vm->host.addr.guest_context;

  cpu->eax = context->eax;
  cpu->ebx = context->ebx;
  cpu->ecx = context->ecx;
  cpu->edx = context->edx;
  cpu->ebp = context->ebp;
  cpu->esi = context->esi;
  cpu->edi = context->edi;
  cpu->esp = context->esp;
  cpu->eflags = read_eflags(vm);
  cpu->eip = context->eip;
  if (vm->guest_cpu.prev_eip != -1) {
    /* Rewind EIP,ESP back to beginning of instruction so that */
    /* debug disassembly will show correct instruction. */
    cpu->eip = vm->guest_cpu.prev_eip;
    cpu->esp = vm->guest_cpu.prev_esp;
    }

  cpu->cs.sel.raw = vm->guest_cpu.selector[SRegCS].raw;
  cpu->cs.des = vm->guest_cpu.desc_cache[SRegCS].desc;
  cpu->cs.valid = vm->guest_cpu.desc_cache[SRegCS].valid;

  cpu->ss.sel.raw = vm->guest_cpu.selector[SRegSS].raw;
  cpu->ss.des = vm->guest_cpu.desc_cache[SRegSS].desc;
  cpu->ss.valid = vm->guest_cpu.desc_cache[SRegSS].valid;

  cpu->ds.sel.raw = vm->guest_cpu.selector[SRegDS].raw;
  cpu->ds.des = vm->guest_cpu.desc_cache[SRegDS].desc;
  cpu->ds.valid = vm->guest_cpu.desc_cache[SRegDS].valid;

  cpu->es.sel.raw = vm->guest_cpu.selector[SRegES].raw;
  cpu->es.des = vm->guest_cpu.desc_cache[SRegES].desc;
  cpu->es.valid = vm->guest_cpu.desc_cache[SRegES].valid;

  cpu->fs.sel.raw = vm->guest_cpu.selector[SRegFS].raw;
  cpu->fs.des = vm->guest_cpu.desc_cache[SRegFS].desc;
  cpu->fs.valid = vm->guest_cpu.desc_cache[SRegFS].valid;

  cpu->gs.sel.raw = vm->guest_cpu.selector[SRegGS].raw;
  cpu->gs.des = vm->guest_cpu.desc_cache[SRegGS].desc;
  cpu->gs.valid = vm->guest_cpu.desc_cache[SRegGS].valid;

  cpu->ldtr.sel = vm->guest_cpu.ldtr_selector;
  cpu->ldtr.des = vm->guest_cpu.ldtr_cache.desc;
  cpu->ldtr.valid = vm->guest_cpu.ldtr_cache.valid;

  cpu->tr.sel = vm->guest_cpu.tr_selector;
  cpu->tr.des = vm->guest_cpu.tr_cache.desc;
  cpu->tr.valid = vm->guest_cpu.tr_cache.valid;

  cpu->gdtr.base = vm->guest_cpu.gdtr.base;
  cpu->gdtr.limit = vm->guest_cpu.gdtr.limit;

  cpu->idtr.base = vm->guest_cpu.idtr.base;
  cpu->idtr.limit = vm->guest_cpu.idtr.limit;

  cpu->dr0 = vm->guest_cpu.dr0;
  cpu->dr1 = vm->guest_cpu.dr1;
  cpu->dr2 = vm->guest_cpu.dr2;
  cpu->dr3 = vm->guest_cpu.dr3;
  cpu->dr6 = vm->guest_cpu.dr6;
  cpu->dr7 = vm->guest_cpu.dr7;

  cpu->tr3 = vm->guest_cpu.tr3;
  cpu->tr4 = vm->guest_cpu.tr4;
  cpu->tr5 = vm->guest_cpu.tr5;
  cpu->tr6 = vm->guest_cpu.tr6;
  cpu->tr7 = vm->guest_cpu.tr7;

  cpu->cr0 = vm->guest_cpu.cr0.raw;
  cpu->cr1 = vm->guest_cpu.cr1;
  cpu->cr2 = vm->guest_cpu.cr2;
  cpu->cr3 = vm->guest_cpu.cr3;
  cpu->cr4 = vm->guest_cpu.cr4.raw;

  cpu->inhibit_mask = vm->guest_cpu.inhibit_mask;
}


/* */
/* Allocate various pages/memory needed by monitor */
/* */

  int
allocVmPages( vm_t *vm, unsigned nmegs )
{
  vm_pages_t *pg = &vm->pages;
  vm_addr_t  *ad = &vm->host.addr;
  unsigned where;
  //unsigned size;

  /* clear out allocated pages lists */
  mon_memzero(pg, sizeof(*pg));
  mon_memzero(ad, sizeof(*ad));


  /* Guest physical memory pages */
  pg->guest_n_megs  = nmegs;
  pg->guest_n_pages = nmegs * 256;
  pg->guest_n_bytes = nmegs * 1024 * 1024;

  where = 1;
  if ( !(ad->guest = host_alloc(pg->guest_n_megs * 1024 * 1024)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->guest, MON_GUEST_PAGES, 
           ad->guest, pg->guest_n_megs * 1024 * 1024) ) {
    goto error;
    }
  where++;

  /* Monitor page directory */
  if ( !(ad->page_dir = (pageEntry_t *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if (!(pg->page_dir = host_map_page(ad->page_dir))) {
    goto error;
    }
  where++;

  /* Monitor page tables */
  if ( !(ad->page_tbl = host_alloc(4096 * MON_PAGE_TABLES)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->page_tbl, MON_PAGE_TABLES, 
           ad->page_tbl, 4096 * MON_PAGE_TABLES)) {
    goto error;
    }
  where++;

  /* Map of the linear addresses of page tables currently */
  /* mapped into the monitor space. */
  if ( !(ad->page_tbl_laddr_map = (unsigned *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if ( !(pg->page_tbl_laddr_map = host_map_page(ad->page_tbl_laddr_map)) ) {
    goto error;
    }
  where++;

  /* Nexus page table */
  if ( !(ad->nexus_page_tbl = (page_t *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if ( !(pg->nexus_page_tbl = host_map_page(ad->nexus_page_tbl)) ) {
    goto error;
    }
  where++;

  /* Transition page table */
  if ( !(ad->transition_PT = (page_t *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if ( !(pg->transition_PT = host_map_page(ad->transition_PT)) ) {
    goto error;
    }
  where++;

  if ( !(ad->log_buffer = host_alloc(4096 * LOG_BUFF_PAGES)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->log_buffer, LOG_BUFF_PAGES, 
           ad->log_buffer, 4096 * LOG_BUFF_PAGES)) {
    goto error;
    }
  where++;

  /* Nexus page */
  if ( !(ad->nexus = (nexus_t *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if ( !(pg->nexus = host_map_page(ad->nexus)) ) {
    goto error;
    }
  where++;

#if 0
  /* Ring3 Handler data page. */
  if ( !(ad->r3hData = (r3hData_t *)host_alloc_page()) ) {
    goto error;
    }
  where++;
  if ( !(pg->r3hData = host_map_page(ad->r3hData)) ) {
    goto error;
    }
  where++;

  /* DT: L2M hash table. */
  size   = sizeof(dtL2MHash_t);
  pg->dtL2MHashNPages = BytesToPages(size);
  if (pg->dtL2MHashNPages > DTL2MMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->dtL2MHash = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->dtL2MHash, pg->dtL2MHashNPages, ad->dtL2MHash, size)) {
    goto error;
    }
  where++;

  /* DT: G2T hash table. */
  size   = sizeof(dtG2THash_t);
  pg->dtG2THashNPages = BytesToPages(size);
  if (pg->dtG2THashNPages > DTG2TMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->dtG2THash = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->dtG2THash, pg->dtG2THashNPages, ad->dtG2THash, size)) {
    goto error;
    }
  where++;

  /* DT: page meta table. */
  size   = sizeof(dtPageMeta_t) * DTPageMetaTableN;
  pg->dtPageMetaTableNPages = BytesToPages(size);
  if (pg->dtPageMetaTableNPages > DTMetaTableMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->dtPageMetaTable = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->dtPageMetaTable, pg->dtPageMetaTableNPages,
                ad->dtPageMetaTable, size)) {
    goto error;
    }
  where++;

  /* DT: page meta table usage matrix. */
  size   = sizeof(Bit8u) * ((DTPageMetaTableN+7) / 8);
  pg->dtPageMetaTableUsageNPages = BytesToPages(size);
  if (pg->dtPageMetaTableUsageNPages > DTMetaTableUsageMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->dtPageMetaTableUsage = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->dtPageMetaTableUsage, pg->dtPageMetaTableUsageNPages,
                ad->dtPageMetaTableUsage, size)) {
    goto error;
    }
  where++;

  /* DT: tcode chunk pool. */
  size   = SizeOfTcodeBuffer;
  pg->tcodeChunkNPages = BytesToPages(size);
  if (pg->tcodeChunkNPages > DTTcodeChunkMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->tcodeChunk = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->tcodeChunk, pg->tcodeChunkNPages, ad->tcodeChunk, size)) {
    goto error;
    }
  where++;

  /* DT: tcode chunk usage matrix. */
  size   = sizeof(Bit8u) * ((TCodeChunkN+7) / 8);
  pg->tcodeChunkUsageNPages = BytesToPages(size);
  if (pg->tcodeChunkUsageNPages > DTTcodeChunkUsageMaxPages) {
    goto error;
    }
  where++;
  if ( !(ad->tcodeChunkUsage = host_alloc(size)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->tcodeChunkUsage, pg->tcodeChunkUsageNPages,
                ad->tcodeChunkUsage, size)) {
    goto error;
    }
  where++;
#endif


  /* Monitor IDT */
  if ( !(ad->idt = host_alloc(MON_IDT_PAGES*4096)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->idt, MON_IDT_PAGES, ad->idt, MON_IDT_SIZE)) {
    goto error;
    }
  where++;

  /* Monitor GDT */
  if ( !(ad->gdt = host_alloc(MON_GDT_PAGES*4096)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->gdt, MON_GDT_PAGES, ad->gdt, MON_GDT_SIZE)) {
    goto error;
    }
  where++;

  /* Monitor LDT */
  if ( !(ad->ldt = host_alloc(MON_LDT_PAGES*4096)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->ldt, MON_LDT_PAGES, ad->ldt, MON_LDT_SIZE)) {
    goto error;
    }
  where++;

  /* Monitor TSS */
  if ( !(ad->tss = host_alloc(MON_TSS_PAGES*4096)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->tss, MON_TSS_PAGES, ad->tss, MON_TSS_SIZE)) {
    goto error;
    }
  where++;

  /* Monitor IDT stubs */
  if ( !(ad->idt_stubs = host_alloc(MON_IDT_STUBS_PAGES*4096)) ) {
    goto error;
    }
  where++;
  if (!host_map(pg->idt_stubs, MON_IDT_STUBS_PAGES, 
           ad->idt_stubs, MON_IDT_STUBS_SIZE)) {
    goto error;
    }
  where++;

  /* Get the physical pages associated with the vm_t structure. */
  if (!host_map(pg->vm, MAX_VM_STRUCT_PAGES, vm, sizeof(*vm))) {
    goto error;
    }
  where++;

  return 0;

 error:
    unallocVmPages( vm );
    return( where );
}


/* */
/* Unallocate pages/memory used by monitor */
/* */

  void
unallocVmPages( vm_t *vm )
{
    vm_pages_t *pg = &vm->pages;
    vm_addr_t  *ad = &vm->host.addr;

#if InstrumentMonPageFaults
  hostprint("monPageFaultCount = %u\n", monPageFaultCount);
#endif

#if InstrumentVMMessages
    {
    unsigned i;
    for (i=0; i<MaxVMMessages; i++) {
      if ( vm->vmMessageCount[i] ) {
        hostprint("vmMCount[%u] = %u\n", i, vm->vmMessageCount[i]);
        vm->vmMessageCount[i] = 0; /* Reset to 0. */
        }
      }
    }
#endif

#if InstrumentIO
    {
    unsigned i;
    for (i=0; i<(64*1024); i++) {
      if (ioAccessCount[i])
        hostprint("port 0x%x accessed %u times.\n", i, ioAccessCount[i]);
      }
    }
#endif

    /* Guest physical memory pages */
    if (ad->guest) host_free(ad->guest);

    /* Monitor page directory */
    if (ad->page_dir) host_free_page(ad->page_dir);

    /* Monitor page tables */
    if (ad->page_tbl) host_free(ad->page_tbl);

    /* Map of linear addresses of page tables mapped into monitor. */
    if (ad->page_tbl_laddr_map) host_free_page(ad->page_tbl_laddr_map);

    /* Nexus page table */
    if (ad->nexus_page_tbl) host_free_page(ad->nexus_page_tbl);

    /* Transition page table */
    if (ad->transition_PT) host_free_page(ad->transition_PT);

    if (ad->log_buffer) host_free(ad->log_buffer);

    /* Nexus page */
    if (ad->nexus) host_free_page(ad->nexus);

#if 0
    /* DT features */
    if (ad->r3hData) host_free_page(ad->r3hData);
    if (ad->dtL2MHash) host_free(ad->dtL2MHash);
    if (ad->dtG2THash) host_free(ad->dtG2THash);
    if (ad->dtPageMetaTable) host_free(ad->dtPageMetaTable);
    if (ad->dtPageMetaTableUsage) host_free(ad->dtPageMetaTableUsage);
    if (ad->tcodeChunk) host_free(ad->tcodeChunk);
    if (ad->tcodeChunkUsage) host_free(ad->tcodeChunkUsage);
#endif

    /* Monitor IDT */
    if (ad->idt) host_free(ad->idt);

    /* Monitor GDT */
    if (ad->gdt) host_free(ad->gdt);

    /* Monitor LDT */
    if (ad->ldt) host_free(ad->ldt);

    /* Monitor TSS */
    if (ad->tss) host_free(ad->tss);

    /* Monitor IDT stubs */
    if (ad->idt_stubs) host_free(ad->idt_stubs);


    /* clear out allocated pages lists */
    mon_memzero(pg, sizeof(*pg));
    mon_memzero(ad, sizeof(*ad));
}

  unsigned
getCpuCapabilities(void)
{
  Bit32u eax, ebx, ecx, edx;

  /* Get the highest allowed cpuid level */
  asm volatile (
    "xorl %%eax,%%eax\n\t"
    "cpuid"
    : "=a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx)
    :
    : "cc"
    );
  cpuid_info.maxval = eax;
  if (eax < 1)
    return(0); /* not enough capabilities */

  /* Copy vendor string */
  * (Bit32u*) &cpuid_info.vendorID[0] = ebx;
  * (Bit32u*) &cpuid_info.vendorID[4] = edx;
  * (Bit32u*) &cpuid_info.vendorID[8] = ecx;
  cpuid_info.vendorID[12] = 0;

  /* CPUID w/ EAX==1: Processor Signature & Feature Flags */
  asm volatile (
    "movl $1,%%eax\n\t"
    "cpuid"
    : "=a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx)
    :
    : "cc"
    );
  cpuid_info.procSignature.raw = eax;
  cpuid_info.featureFlags.raw = edx;
  /* Plex86 needs TSC */
  if (cpuid_info.featureFlags.fields.tsc==0)
    return(0);

#if 1
#warning "getCpuCapabilites hacked to always return stripped Pentium"
  cpuid_info.procSignature.raw = 0;
  cpuid_info.featureFlags.raw = 0;
  cpuid_info.procSignature.fields.family = 5;
  cpuid_info.procSignature.fields.model = 1;
  cpuid_info.procSignature.fields.stepping = 3;
#else
#warning "getCpuCapabilites returning host CPU values"
#endif

  return(1);
}

