/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  exception.c:  emulation of guest exceptions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */



#include "plex86.h"
#include "monitor.h"


/* Exception classes.  These are used as indexes into the 'is_exception_OK'
 * array below, and are stored in the 'exception' array also
 */
#define ET_BENIGN       0
#define ET_CONTRIBUTORY 1
#define ET_PAGE_FAULT   2
 
#define ET_DOUBLE_FAULT 10
 
 
const unsigned is_exception_OK[3][3] = {
    { 1, 1, 1 }, /* 1st exception is BENIGN */
    { 1, 0, 1 }, /* 1st exception is CONTRIBUTORY */
    { 1, 0, 0 }  /* 1st exception is PAGE_FAULT */
    };


  void
UndefinedOpcode(vm_t *vm)
{
  if (vm->i.b1 != 0x63) {
    /* Windows hits the ARPL command a bunch of times. */
    /* Too much spew... */
    monprint(vm, "#UD: %02x causes exception 6\n",
              (unsigned) vm->i.b1);
    }
  exception(vm, ExceptionUD, 0);
}


  void
interrupt(vm_t *vm, Bit8u vector, unsigned is_INT, unsigned is_error_code,
          Bit16u error_code)
{
  invalidate_prefetch_q();

  /* Discard any traps and inhibits for new context; traps will */
  /* resume upon return. */
  vm->guest_cpu.debug_trap = 0;
  vm->guest_cpu.inhibit_mask = 0;

#if 0
  /* If interrupt is intercepted by user-mode debugger, */
  /* forward it to user space */
  if (BMAP_GET(vm->host_fwd_ints, vector))
  {
    if (!sysReflectInt(vm, vector))
      return;
  }
#endif

/*  unsigned prev_errno; */

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);

/* +++ see if we need this or can recode without it */
vm->guest_cpu.save_cs  = vm->guest_cpu.desc_cache[SRegCS];
vm->guest_cpu.save_ss  = vm->guest_cpu.desc_cache[SRegSS];
vm->guest_cpu.save_eip = G_EIP(vm);
vm->guest_cpu.save_esp = G_ESP(vm);

/*  prev_errno = BX_CPU_THIS_PTR errorno; */

  if(!RealMode(vm)) {
    gate_t             gate_descriptor;
    descriptor_cache_t cs_cache, tss_cache;
    selector_t         cs_selector;
    selector_t         tss_selector;

    selector_t gate_dest_selector;
    Bit32u gate_dest_offset;

    /* interrupt vector must be within IDT table limits, */
    /* else #GP(vector number*8 + 2 + EXT) */
    if ( (vector*8 + 7) > vm->guest_cpu.idtr.limit) {
      monprint(vm, "interrupt(): vector > idtr.limit\n");
      exception(vm, ExceptionGP, vector*8 + 2);
      }

    /* descriptor AR byte must indicate interrupt gate, trap gate, */
    /* or task gate, else #GP(vector*8 + 2 + EXT) */
    access_linear(vm, vm->guest_cpu.idtr.base + vector*8,     8, 0,
      OP_READ, &gate_descriptor);

    if ( gate_descriptor.type & D_S ) {
      monpanic(vm, "interrupt: gate descriptor S=1\n");
      exception(vm, ExceptionGP, vector*8 + 2);
      }

    switch (gate_descriptor.type) {
      case 5: /* task gate */
      case 6: /* 286 interrupt gate */
      case 7: /* 286 trap gate */
      case 14: /* 386 interrupt gate */
      case 15: /* 386 trap gate */
        break;
      default:
        monpanic(vm, "interrupt: gate.type(%u) != {5,6,7,14,15}\n",
          (unsigned) gate_descriptor.type);
        exception(vm, ExceptionGP, vector*8 + 2);
      }

    /* if software interrupt, then gate descripor DPL must be >= CPL, */
    /* else #GP(vector * 8 + 2 + EXT) */
    if (is_INT  &&  (gate_descriptor.dpl < G_CPL(vm))) {
/* ??? */
      monprint(vm, "interrupt: is_INT && (dpl < CPL)\n");
      exception(vm, ExceptionGP, vector*8 + 2);
      }

    /* Gate must be present, else #NP(vector * 8 + 2 + EXT) */
    if (gate_descriptor.p == 0) {
      monprint(vm, "interrupt: p == 0\n");
      exception(vm, ExceptionNP, vector*8 + 2);
      }

    switch (gate_descriptor.type) {
      case 5: /* 286/386 task gate */
        /* examine selector to TSS, given in task gate descriptor */
        tss_selector = gate_descriptor.selector;

        /* must specify global in the local/global bit, */
        /*      else #TS(TSS selector) */
/* +++
 * 486/Pent books say #TSS(selector)
 * PPro+ says #GP(selector)
 */
        if (tss_selector.fields.ti) {
          monpanic(vm, "interrupt: tss_selector.ti=1\n");
          exception(vm, ExceptionTS, tss_selector.raw & 0xfffc);
          }

        /* index must be within GDT limits, else #TS(TSS selector) */
        fetch_raw_descriptor(vm, tss_selector, &tss_cache.desc,
          ExceptionTS);
        descriptor2cache(vm, &tss_cache);

        /* AR byte must specify available TSS, */
        /*   else #TS(TSS selector) */
        if ( (tss_cache.valid==0) ||
             ((tss_cache.desc.type!=9) && (tss_cache.desc.type!=1)) ) {
          monpanic(vm, "exception: TSS selector points to bad TSS\n");
          exception(vm, ExceptionTS, tss_selector.raw & 0xfffc);
          }


        /* TSS must be present, else #NP(TSS selector) */
        /* done in task_switch() */

        /* switch tasks with nesting to TSS */
        task_switch(vm, tss_selector, &tss_cache, TASK_FROM_CALL_OR_INT);

        /* if interrupt was caused by fault with error code */
        /*   stack limits must allow push of 2 more bytes, else #SS(0) */
        /* push error code onto stack */

        /*??? push_16 vs push_32 */
        if ( is_error_code ) {
          if (vm->guest_cpu.desc_cache[SRegCS].desc.d_b)
            push32(vm, error_code);
          else
            push16(vm, error_code);
          }

        /* instruction pointer must be in CS limit, else #GP(0) */
        /*if (EIP > cs_descriptor.u.segment.limit_scaled) {} +++ ??? */
        if (G_EIP(vm) > vm->guest_cpu.desc_cache[SRegCS].limit_scaled) {
          monpanic(vm, "exception(): eIP > CS.limit\n");
          exception(vm, ExceptionGP, 0);
          }
        return;

      case 6: /* 286 interrupt gate */
      case 7: /* 286 trap gate */
      case 14: /* 386 interrupt gate */
      case 15: /* 386 trap gate */
        gate_dest_selector = gate_descriptor.selector;
        gate_dest_offset   = gate_descriptor.offset_low;
        if ( gate_descriptor.type >= 14 ) { /* 386 gate */
          gate_dest_offset  |= (gate_descriptor.offset_high<<16);
          }

        /* examine CS selector and descriptor given in gate descriptor */
        /* selector must be non-null else #GP(EXT) */
        if ( IsNullSelector(gate_dest_selector) ) {
          monpanic(vm, "int_trap_gate: selector null\n");
          exception(vm, ExceptionGP, 0);
          }

        cs_selector = gate_dest_selector;

        /* selector must be within its descriptor table limits */
        /* else #GP(selector+EXT) */
        fetch_raw_descriptor(vm, cs_selector, &cs_cache.desc, ExceptionGP);
        descriptor2cache(vm, &cs_cache);

        /* descriptor AR byte must indicate code seg */
        /* and code segment descriptor DPL<=CPL, else #GP(selector+EXT) */
        if ( !cs_cache.valid ||
             !(cs_cache.desc.type & D_S) ||
             !(cs_cache.desc.type & D_EXECUTE) ||
             (cs_cache.desc.dpl>G_CPL(vm)) ) {
          monpanic(vm, "interrupt: not code segment\n");
          exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
          }

        /* segment must be present, else #NP(selector + EXT) */
        if ( cs_cache.desc.p==0 ) {
          monpanic(vm, "interrupt: segment not present\n");
          exception(vm, ExceptionNP, cs_selector.raw & 0xfffc);
          }

        /* if code segment is non-conforming and DPL < CPL then */
        /* INTERRUPT TO INNER PRIVILEGE: */
        if ( !(cs_cache.desc.type & D_CONFORM) &&
             (cs_cache.desc.dpl<G_CPL(vm)) ) {
          Bit16u old_SS, old_CS;
          Bit32u ESP_for_cpl_x, old_EIP, old_ESP;
          Bit32u change_mask;
          descriptor_cache_t ss_cache;
          selector_t         ss_selector;
          int bytes;

          /* check selector and descriptor for new stack in current TSS */
          get_SS_ESP_from_TSS(vm, cs_cache.desc.dpl,
                              &ss_selector, &ESP_for_cpl_x);

          /* Selector must be non-null else #TS(EXT) */
          if ( IsNullSelector(ss_selector) ) {
            monpanic(vm, "interrupt: SS selector null\n");
            /* TS(ext) */
            exception(vm, ExceptionTS, 0);
            }

          /* selector index must be within its descriptor table limits */
          /* else #TS(SS selector + EXT) */

          /* fetch 2 dwords of descriptor; call handles out of limits checks */
          fetch_raw_descriptor(vm, ss_selector, &ss_cache.desc, ExceptionTS);
          descriptor2cache(vm, &ss_cache);

          /* selector rpl must = dpl of code segment, */
          /* else #TS(SS selector + ext) */
          if (ss_selector.fields.rpl != cs_cache.desc.dpl) {
            monpanic(vm, "interrupt: SS.rpl != CS.dpl\n");
            exception(vm, ExceptionTS, ss_selector.raw & 0xfffc);
            }

          /* stack seg DPL must = DPL of code segment, */
          /* else #TS(SS selector + ext) */
          if (ss_cache.desc.dpl != cs_cache.desc.dpl) {
            monpanic(vm, "interrupt: SS.dpl != CS.dpl\n");
            exception(vm, ExceptionTS, ss_selector.raw & 0xfffc);
            }

          /* descriptor must indicate writable data segment, */
          /* else #TS(SS selector + EXT) */
          if ( !ss_cache.valid ||
               !(ss_cache.desc.type & D_S)  ||
               (ss_cache.desc.type & D_EXECUTE)  ||
               !(ss_cache.desc.type & D_WRITE) ) {
            monpanic(vm, "interrupt: SS not writable data segment\n");
            exception(vm, ExceptionTS, ss_selector.raw & 0xfffc);
            }

          /* seg must be present, else #SS(SS selector + ext) */
          if (ss_cache.desc.p==0) {
            monpanic(vm, "interrupt: SS not present\n");
            exception(vm, ExceptionSS, ss_selector.raw & 0xfffc);
            }

          if (gate_descriptor.type>=14) {
            /* 386 int/trap gate */
            /* new stack must have room for 20|24 bytes, else #SS(0) */
            if ( is_error_code )
              bytes = 24;
            else
              bytes = 20;
            if (V8086Mode(vm))
              bytes += 16;
            }
          else {
            /* new stack must have room for 10|12 bytes, else #SS(0) */
            if ( is_error_code )
              bytes = 12;
            else
              bytes = 10;
            if (V8086Mode(vm)) {
              bytes += 8;
              monpanic(vm, "interrupt: int/trap gate VM\n");
              }
            }

/* 486,Pentium books */
/* new stack must have room for 10/12 bytes, else #SS(0) 486 book */
/* PPro+ */
/* new stack must have room for 10/12 bytes, else #SS(seg selector) */
          if ( !can_push(vm, &ss_cache, ESP_for_cpl_x, bytes) ) {
            monpanic(vm, "interrupt: new stack doesn't have room for %u bytes\n",
               (unsigned) bytes);
            /* SS(???) */
            }

          /* IP must be within CS segment boundaries, else #GP(0) */
          if (gate_dest_offset > cs_cache.limit_scaled) {
            monpanic(vm, "interrupt: gate eIP > CS.limit\n");
            exception(vm, ExceptionGP, 0);
            }

          old_ESP = G_ESP(vm);
          old_SS  = vm->guest_cpu.selector[SRegSS].raw;
          old_EIP = G_EIP(vm);
          old_CS  = vm->guest_cpu.selector[SRegCS].raw;

          /* load new SS:SP values from TSS */
          load_ss_pro(vm, ss_selector, &ss_cache, cs_cache.desc.dpl);

          if (ss_cache.desc.d_b)
            G_ESP(vm) = ESP_for_cpl_x;
          else
            G_SP(vm) = ESP_for_cpl_x; /* leave upper 16bits */

          /* load new CS:IP values from gate */
          /* set CPL to new code segment DPL */
          /* set RPL of CS to CPL */
          load_cs_pro(vm, cs_selector, &cs_cache, cs_cache.desc.dpl);
          G_EIP(vm) = gate_dest_offset;

          if (gate_descriptor.type>=14) { /* 386 int/trap gate */
            if (V8086Mode(vm)) {
              selector_t null_sel;
              null_sel.raw = 0; /* make a static one */
              cache_selector(vm, SRegGS);
              cache_selector(vm, SRegFS);
              cache_selector(vm, SRegDS);
              cache_selector(vm, SRegES);
              push32(vm, vm->guest_cpu.selector[SRegGS].raw);
              push32(vm, vm->guest_cpu.selector[SRegFS].raw);
              push32(vm, vm->guest_cpu.selector[SRegDS].raw);
              push32(vm, vm->guest_cpu.selector[SRegES].raw);
              invalidate_sreg_pro(vm, SRegGS, null_sel);
              invalidate_sreg_pro(vm, SRegFS, null_sel);
              invalidate_sreg_pro(vm, SRegDS, null_sel);
              invalidate_sreg_pro(vm, SRegES, null_sel);
              }
            /* push long pointer to old stack onto new stack */
            push32(vm, old_SS);
            push32(vm, old_ESP);

            /* push EFLAGS */
            push32(vm, read_eflags(vm));

            /* push long pointer to return address onto new stack */
            push32(vm, old_CS);
            push32(vm, old_EIP);

            if ( is_error_code )
              push32(vm, error_code);
            }
          else { /* 286 int/trap gate */
            if (V8086Mode(vm)) {
              monpanic(vm, "286 int/trap gate, VM\n");
              }
            /* push long pointer to old stack onto new stack */
            push16(vm, old_SS);
            push16(vm, old_ESP); /* ignores upper 16bits */

            /* push FLAGS */
            push16(vm, read_eflags(vm));

            /* push return address onto new stack */
            push16(vm, old_CS);
            push16(vm, old_EIP); /* ignores upper 16bits */

            if ( is_error_code )
              push16(vm, error_code);
            }

          /* VM,RF,NT,TF <- 0 */
          change_mask = 0x00034100;
          /* if INTERRUPT GATE set IF to 0 */
          if ( !(gate_descriptor.type & 1) ) /* even is int-gate */
            change_mask |= 0x00000200; /* IF <- 0 */
/*monprint(vm, "interrupt: 0, changemask=0x%x oldeflags=0x%x\n", */
/*  change_mask, read_eflags(vm)); */
          write_eflags(vm, 0, change_mask); /* write 0 to specified bits */
          return;
          }

        if (V8086Mode(vm)) {
          exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
          }

        /* if code segment is conforming OR code segment DPL = CPL then */
        /* INTERRUPT TO SAME PRIVILEGE LEVEL: */
        if ( (cs_cache.desc.type & D_CONFORM) ||
             (cs_cache.desc.dpl==G_CPL(vm)) ) {
          int bytes;
          Bit32u temp_ESP;
          Bit32u change_mask;

          if (vm->guest_cpu.desc_cache[SRegSS].desc.d_b)
            temp_ESP = G_ESP(vm);
          else
            temp_ESP = G_SP(vm);

          /* Current stack limits must allow pushing 6|8 bytes, else #SS(0) */
          if (gate_descriptor.type >= 14) { /* 386 gate */
            if ( is_error_code )
              bytes = 16;
            else
              bytes = 12;
            }
          else { /* 286 gate */
            if ( is_error_code )
              bytes = 8;
            else
              bytes = 6;
            }

          if ( !can_push(vm, &vm->guest_cpu.desc_cache[SRegSS],
                         temp_ESP, bytes) ) {
            monprint(vm, "interrupt: stack doesn't have room\n");
            exception(vm, ExceptionSS, 0);
            }

          /* eIP must be in CS limit else #GP(0) */
          if (gate_dest_offset > cs_cache.limit_scaled) {
            monpanic(vm, "interrupt: gate offset > cs descriptor limit\n");
            exception(vm, ExceptionGP, 0);
            }

          /* push flags onto stack */
          /* push current CS selector onto stack */
          /* push return offset onto stack */
          if (gate_descriptor.type >= 14) { /* 386 gate */
            push32(vm, read_eflags(vm));
            push32(vm, vm->guest_cpu.selector[SRegCS].raw);
            push32(vm, G_EIP(vm));
            if ( is_error_code )
              push32(vm, error_code);
            }
          else { /* 286 gate */
            push16(vm, read_eflags(vm));
            push16(vm, vm->guest_cpu.selector[SRegCS].raw);
            push16(vm, G_IP(vm));
            if ( is_error_code )
              push16(vm, error_code);
            }

          /* load CS:IP from gate */
          /* load CS descriptor */
          /* set the RPL field of CS to CPL */
          load_cs_pro(vm, cs_selector, &cs_cache, G_CPL(vm));
          G_EIP(vm) = gate_dest_offset;

          /* VM,RF,NT,TF <- 0 */
          change_mask = 0x00034100;
          /* if INTERRUPT GATE set IF to 0 */
          if ( !(gate_descriptor.type & 1) ) /* even is int-gate */
            change_mask |= 0x00000200; /* IF <- 0 */
/*monprint(vm, "interrupt: 0, changemask=0x%x oldeflags=0x%x\n", */
/*  change_mask, read_eflags(vm)); */
          write_eflags(vm, 0, change_mask); /* write 0 to specified bits */
          return;
          }

        /* else #GP(CS selector + ext) */
        monprint(vm, "interrupt: bad descriptor type=0x%x, CPL=%u\n",
          cs_cache.desc.type, G_CPL(vm));
        exception(vm, ExceptionGP, cs_selector.raw & 0xfffc);
        break;

      default:
        monpanic(vm, "interrupt: bad descriptor type (%u)!\n",
          gate_descriptor.type);
        break;
      }
    }
  else { /* real mode */
    Bit16u cs_selector, ip;

    if ( (vector*4+3) > vm->guest_cpu.idtr.limit )
      monpanic(vm, "interrupt: RM vector > limit\n");

    push16(vm, read_eflags(vm));
    push16(vm, vm->guest_cpu.selector[SRegCS].raw);
    push16(vm, G_IP(vm));

    access_linear(vm, vm->guest_cpu.idtr.base + 4*vector,
                  2, 0, OP_READ, &ip);
    access_linear(vm, vm->guest_cpu.idtr.base + 4*vector + 2,
                  2, 0, OP_READ, &cs_selector);

    load_seg_reg(vm, SRegCS, cs_selector);
    G_EIP(vm) = ip;

    /* AC,RF,NT,IF,TF <- 0 +++ */
    write_eflags(vm, 0, 0x00054300); /* write 0 to specified bits */
    }
}

  void
exception(vm_t *vm, unsigned vector, Bit32u error_code)
{
  Boolean  push_error;
  Bit8u    exception_type;
  unsigned prev_errno;

  invalidate_prefetch_q();

  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);

  /* If not initial error, restore previous register values from */
  /* previous attempt to handle exception */
  if (vm->guest_cpu.errorno) {
    load_cs_pro(vm, vm->guest_cpu.selector[SRegCS], &vm->guest_cpu.save_cs,
                vm->guest_cpu.save_cs.desc.dpl);
    load_ss_pro(vm, vm->guest_cpu.selector[SRegSS], &vm->guest_cpu.save_ss,
                vm->guest_cpu.save_ss.desc.dpl);
    G_EIP(vm) = vm->guest_cpu.save_eip;
    G_ESP(vm) = vm->guest_cpu.save_esp;
    }

  vm->guest_cpu.errorno++;
  if (vm->guest_cpu.errorno >= 3) {
    monpanic(vm, "exception: 3rd exception with no resolution\n");
    }

  /* careful not to get here with curr_exception[1]==DOUBLE_FAULT */
  /* ...index on DOUBLE_FAULT below, will be out of bounds */

  /* if 1st was a double fault (software INT?), then shutdown */
  if ( (vm->guest_cpu.errorno==2) &&
       (vm->guest_cpu.curr_exception[0]==ET_DOUBLE_FAULT) ) {
    monpanic(vm, "exception: tripple fault encountered\n");
    }

  /* ??? this is not totally correct, should be done depending on
   * vector */
  /* backup IP to value before error occurred */
  G_EIP(vm) = vm->guest_cpu.prev_eip;
  G_ESP(vm) = vm->guest_cpu.prev_esp;

  /* note: fault-class exceptions _except_ #DB set RF in */
  /*       eflags image. */

  switch (vector) {
    case  0: /* DIV by 0 */
      push_error = 0;
      exception_type = ET_CONTRIBUTORY;
      G_SetRF(vm, 1);
      break;
    case  1: /* debug exceptions */
      push_error = 0;
      exception_type = ET_BENIGN;
      break;
    case  2: /* NMI */
      push_error = 0;
      exception_type = ET_BENIGN;
      break;
    case  3: /* breakpoint */
      push_error = 0;
      exception_type = ET_BENIGN;
      break;
    case  4: /* overflow */
      push_error = 0;
      exception_type = ET_BENIGN;
      break;
    case  5: /* bounds check */
      push_error = 0;
      exception_type = ET_BENIGN;
      G_SetRF(vm, 1);
      break;
    case  6: /* invalid opcode */
      push_error = 0;
      exception_type = ET_BENIGN;
      G_SetRF(vm, 1);
      break;
    case  7: /* device not available */
      push_error = 0;
      exception_type = ET_BENIGN;
      G_SetRF(vm, 1);
      break;
    case  8: /* double fault */
      push_error = 1;
      exception_type = ET_DOUBLE_FAULT;
      break;
    case  9: /* coprocessor segment overrun (286,386 only) */
      push_error = 0;
      exception_type = ET_CONTRIBUTORY;
      G_SetRF(vm, 1);
      monpanic(vm, "exception(9): unfinished\n");
      break;
    case 10: /* invalid TSS */
      push_error = 1;
      exception_type = ET_CONTRIBUTORY;
      error_code = (error_code & 0xfffe) | vm->guest_cpu.EXT;
      G_SetRF(vm, 1);
      break;
    case 11: /* segment not present */
      push_error = 1;
      exception_type = ET_CONTRIBUTORY;
      error_code = (error_code & 0xfffe) | vm->guest_cpu.EXT;
      G_SetRF(vm, 1);
      break;
    case 12: /* stack fault */
      push_error = 1;
      exception_type = ET_CONTRIBUTORY;
      error_code = (error_code & 0xfffe) | vm->guest_cpu.EXT;
      G_SetRF(vm, 1);
      break;
    case 13: /* general protection */
      push_error = 1;
      exception_type = ET_CONTRIBUTORY;
      error_code = (error_code & 0xfffe) | vm->guest_cpu.EXT;
      G_SetRF(vm, 1);
      break;
    case 14: /* page fault */
      push_error = 1;
      exception_type = ET_PAGE_FAULT;
      /* ??? special format error returned */
      G_SetRF(vm, 1);
      break;
    case 15: /* reserved */
      monpanic(vm, "exception(15): reserved\n");
      push_error = 0;     /* keep compiler happy for now */
      exception_type = 0; /* keep compiler happy for now */
      break;
    case 16: /* floating-point error */
      push_error = 0;
      exception_type = ET_BENIGN;
      G_SetRF(vm, 1);
      break;
    case 17: /* alignment check */
      monpanic(vm, "exception: alignment-check, vector 17 unimplemented\n");
      push_error = 0;     /* keep compiler happy for now */
      exception_type = 0; /* keep compiler happy for now */
      G_SetRF(vm, 1);
      break;
    case 18: /* machine check */
      monpanic(vm, "exception: machine-check, vector 18 unimplemented\n");
      push_error = 0;     /* keep compiler happy for now */
      exception_type = 0; /* keep compiler happy for now */
      break;
    default:
      monpanic(vm, "exception (%u): bad vector\n", (unsigned) vector);
      push_error = 0;     /* keep compiler happy for now */
      exception_type = 0; /* keep compiler happy for now */
      break;
    }


  if (exception_type != ET_PAGE_FAULT) {
    /* Page faults have different format */
    error_code = (error_code & 0xfffe) | vm->guest_cpu.EXT;
    }
  vm->guest_cpu.EXT = 1;

  /* if we've already had 1st exception, see if 2nd causes a
   * Double Fault instead.  Otherwise, just record 1st exception
   */
  if (vm->guest_cpu.errorno >= 2) {
    if (is_exception_OK[vm->guest_cpu.curr_exception[0]][exception_type])
      vm->guest_cpu.curr_exception[1] = exception_type;
    else
      vm->guest_cpu.curr_exception[1] = ET_DOUBLE_FAULT;
    }
  else {
    vm->guest_cpu.curr_exception[0] = exception_type;
    }

  if (!RealMode(vm)) {
    prev_errno = vm->guest_cpu.errorno;
    interrupt(vm, vector, 0, push_error, error_code);
    vm->guest_cpu.errorno = 0; /* error resolved */
    LongJmp(vm->jmp_buf_env, 1); /* go back to main decode loop */
    }
  else { /* real mode */
    interrupt(vm, vector, 0, 0, 0);
    vm->guest_cpu.errorno = 0; /* error resolved */
    LongJmp(vm->jmp_buf_env, 1); /* go back to main decode loop */
    }
  monpanic(vm, "exception: LongJmp unsuccessful\n");
}
