/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  arith32.c:  32-bit arithmetic instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CMPXCHG_EdGd(vm_t *vm)
{
  Bit32u op2, op1, diff;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subl %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (diff)
    : "g" (op1), "1" (G_EAX(vm))
    : "memory", "cc"
    );
 
  if (diff == 0) {
    /* dest <-- src */
    op2 = ReadReg32(vm, vm->i.nnn);
 
    /* +++ docs say CMPXCHG causes a write cycle regardless */
    if (vm->i.mod == 0xc0) {
      WriteReg32(vm, vm->i.rm, op2);
      }
    else {
      write_RMW_virtual_dword(vm, &op2);
      }
    }
  else {
    /* accumulator <-- dest */
    G_EAX(vm) = op1;
    }

  SetOSZAPC(vm, eflags);
}

  void
ADD_EdId(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op2 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addl %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EdId(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_EdId(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EdId(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subl %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EAXId(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = G_EAX(vm);
  op2 = vm->i.Id;
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  G_EAX(vm) = op1;
  SetOSZAPC(vm, eflags);
}

  void
SBB_EdGd(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_EAXId(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = G_EAX(vm);
  op2 = vm->i.Id;

  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  G_EAX(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EdGd(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_GdEd(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcl %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg32(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EAXId(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = G_EAX(vm);
  op2 = vm->i.Id;
 
  asm volatile (
    "subl %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  G_EAX(vm) = op1;
  SetOSZAPC(vm, eflags);
}

  void
ADD_EAXId(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op1 = G_EAX(vm);
  op2 = vm->i.Id;
 
  asm volatile (
    "addl %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );

  G_EAX(vm) = op1;
  SetOSZAPC(vm, eflags);
}

  void
CMP_GdEd(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "cmpl %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EdGd(vm_t *vm)
{
  Bit32u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subl %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_GdEd(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;

  op1 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc         \n\t"
      "sbbl %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc         \n\t"
      "sbbl %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=r" (op1)
      : "1" (op1), "g" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg32(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
CDQ(vm_t *vm)
{
  if (G_EAX(vm) & 0x80000000) {
    G_EDX(vm) = 0xFFFFFFFF;
    }
  else {
    G_EDX(vm) = 0x00000000;
    }
}


  void
INC_ERX(vm_t *vm)
{
  Bit32u erx;
  unsigned reg;
  Bit32u eflags;

  reg = vm->i.b1 & 0x07;
  erx = ReadReg32(vm, reg);

  asm volatile (
    "incl %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (erx) 
    : "1" (erx)
    : "memory", "cc"
    );

  WriteReg32(vm, reg, erx);
  SetOSZAP(vm, eflags);
}

  void
DEC_ERX(vm_t *vm)
{
  Bit32u op1;
  unsigned reg;
  Bit32u eflags;
 
  reg = vm->i.b1 & 0x07;
  op1 = ReadReg32(vm, reg);

  asm volatile (
    "decl %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  WriteReg32(vm, reg, op1);
  SetOSZAP(vm, eflags);
}

  void
DEC_Ed(vm_t *vm)
{
  Bit32u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "decl %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAP(vm, eflags);
}

  void
INC_Ed(vm_t *vm)
{
  Bit32u op1, eflags;

  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  asm volatile (
    "incl %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1) 
    : "1" (op1)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }

  SetOSZAP(vm, eflags);
}

  void
CMP_EdGd(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  asm volatile (
    "cmpl %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_EdId(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op2 = vm->i.Id;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "cmpl %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_EAXId(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op1 = G_EAX(vm);
  op2 = vm->i.Id;
 
  asm volatile (
    "cmpl %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "r" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_EdGd(vm_t *vm)
{
  Bit32u op2, op1, eflags;
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addl %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_GdEd(vm_t *vm)
{
  Bit32u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "addl %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );
 
  WriteReg32(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_GdEd(vm_t *vm)
{
  Bit32u eflags;
  Bit32u op1, op2;
 
  op1 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "subl %3, %1 \n\t"
    "pushfl      \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1), "g" (op2)
    : "memory", "cc"
    );

 
  WriteReg32(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
CWDE(vm_t *vm)
{
  G_EAX(vm) = (Bit16s) G_AX(vm);
}

  void
NEG_Ed(vm_t *vm)
{
  Bit32u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "negl %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=r" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_dword(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
XADD_EdGd(vm_t *vm)
{
  Bit32u op2, op1, sum;
  Bit32u eflags;
 
  /* XADD dst(r/m8), src(r8)
   * temp <-- src + dst         | sum = op2 + op1
   * src  <-- dst               | op2 = op1
   * dst  <-- tmp               | op1 = sum
   */
 
  op2 = ReadReg32(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg32(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_dword(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
  sum = op1;
 
  asm volatile (
    "addl %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=r" (sum)
    : "1" (sum), "g" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg32(vm, vm->i.nnn, op1);
    WriteReg32(vm, vm->i.rm, sum);
    }
  else {
    write_RMW_virtual_dword(vm, &sum);
    WriteReg32(vm, vm->i.nnn, op1);
    }
 
  SetOSZAPC(vm, eflags);
}
