/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  branch-mon.c: Generate tcode for branch instructions.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"




  unsigned
dtGenJcc(vm_t *vm, Bit8u *p, unsigned remain, instruction_t *i, Bit32u eip)
{
  Bit8u *sequenceEndOffsetPtr, *pOrig, *branchDataOffsetPtr,
        *r3hBranchOffsetPtr, *branchDataPtr;
  Bit8u jumpCaseNegated;
  Bit32u newEIP;

  pOrig = p;

  /* Calculate new guest EIP after relative branch. */
  newEIP = (eip + i->ilen) + (Bit32s) i->Id;
  if (i->os_32 == 0)
    newEIP &= 0x0000ffff;
// xxx CS.limit vs newEIP check here?

  sequenceEndOffsetPtr = 0; /* Keep compiler happy. */
  if ( (i->b1!=0xe9) && (i->b1!=0xeb) ) {
    /* Negate branch condition to jump to tcode for following instruction.
     * This way we can inline code to handle the branch condition next.
     */
    jumpCaseNegated = (i->b1 & 0xe) | ((~i->b1) & 1);
    if (remain < 2)
      return( 0 ); /* Out of space. */
    *p++ = 0x70 | jumpCaseNegated;
    sequenceEndOffsetPtr = p;
    p += 1; /* Leave empty until we patch it. */
    remain -= 2;
    }

  /*   gs; movl $branchData, __r3hBranchDataAddr
   *   jmp __r3hBranchStatic
   * branchData:
   *   newEIP
   *   hashID
   *   tcodeDirectAddr  (patched in)
   */

  if ( remain < 11 )
    return(0); /* Out of space. */
  *p++ = 0x65; /* gs; movl $branchData, __r3hBranchDataAddr */
  *p++ = 0xc7;
  *p++ = 0x05;
  * (Bit32u *) p = (Bit32u) & __r3hBranchDataAddr;
  p += 4;
  branchDataOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 11;

  if ( remain < 5 )
    return(0); /* Out of space. */
  *p++ = 0xe9; /* jmp __r3hBranchStatic */
  r3hBranchOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 5;

  branchDataPtr = p;
  if ( remain < 4 )
    return(0); /* Out of space. */
  * (Bit32u *) p = newEIP;
  p += 4;
  remain -= 4;

#if DT_UseBackpatch
  if ( remain < 8 )
    return(0); /* Out of space. */
  * (Bit32u *) p = HashIDNone; /* Invalid ID to start with. */
  p += 4;
  * (Bit32u *) p = TcodeOffsetNone; /* Invalid direct tcode addr (patched). */
  p += 4;
  remain -= 8;
#endif

  if ( (i->b1!=0xe9) && (i->b1!=0xeb) ) {
    /* Patch offset from negated branch here */
    *sequenceEndOffsetPtr = p - (sequenceEndOffsetPtr + 1);
    }

  /* Branch relocation information. */
  if ((vm->relocN+1) >= MaxRelocEntries)
    return(0); /* Out of entries. */
  vm->reloc[vm->relocN].tcodePtr = r3hBranchOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) __r3hBranchStatic;
  vm->relocN++;
  vm->reloc[vm->relocN].tcodePtr = branchDataOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) branchDataPtr;
  vm->relocN++;

  return( p - pOrig );
}


  unsigned
dtGenJCXZ(vm_t *vm, Bit8u *p, unsigned remain, instruction_t *i, Bit32u eip)
{
  Bit8u *sequenceEndOffsetPtr, *branchCodeOffsetPtr, *pOrig,
        *branchDataOffsetPtr, *r3hBranchOffsetPtr,
        *branchDataPtr;
  Bit32u newEIP;

  pOrig = p;

  /* Calculate new guest EIP after relative branch. */
  newEIP = (eip + i->ilen) + (Bit32s) i->Id;
  if (i->os_32 == 0)
    newEIP &= 0x0000ffff;
// xxx CS.limit vs newEIP check here?

  /* Pseudo-code:
   *
   *   jcxz branchCode
   *   jmp  seqEnd
   * branchCode:
   *   ...
   *   seqEnd
   */

  if (vm->i.as_32 == 0) {
    /* If guest is using 16-bit address size, then we have to use the
     * AddrSize prefix to request 16-bit use of ECX.
     */
    if (remain < 3)
      return( 0 ); /* Out of space. */
    *p++ = 0x67; /* AddrSize prefix. */
    *p++ = 0xe3; /* JCXZ_Jb */
    branchCodeOffsetPtr = p;
    p++; /* Patch. */
    remain -= 3;
    }
  else {
    /* Guest using 32-bit addressing.  Tcode runs in a 32-bit segment,
     * so this is the default.
     */
    if (remain < 2)
      return( 0 ); /* Out of space. */
    *p++ = 0xe3; /* JCXZ_Jb */
    branchCodeOffsetPtr = p;
    p++; /* Patch. */
    remain -= 2;
    }

  if (remain < 2)
    return( 0 ); /* Out of space. */
  *p++ = 0xeb; /* Jmp Jb */
  sequenceEndOffsetPtr = p;
  p++; /* Patch. */
  remain -= 2;

  /* Patch offset to branch code here. */
  *branchCodeOffsetPtr = p - (branchCodeOffsetPtr + 1);

  /*   gs; movl $branchData, __r3hBranchDataAddr
   *   jmp __r3hBranchStatic
   * branchData:
   *   newEIP
   *   hashID
   *   tcodeDirectAddr  (patched in)
   */

  if ( remain < 11 )
    return(0); /* Out of space. */
  *p++ = 0x65; /* gs; movl $branchData, __r3hBranchDataAddr */
  *p++ = 0xc7;
  *p++ = 0x05;
  * (Bit32u *) p = (Bit32u) & __r3hBranchDataAddr;
  p += 4;
  branchDataOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 11;

  if ( remain < 5 )
    return(0); /* Out of space. */
  *p++ = 0xe9; /* jmp __r3hBranchStatic */
  r3hBranchOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 5;

  branchDataPtr = p;
  if ( remain < 4 )
    return(0); /* Out of space. */
  * (Bit32u *) p = newEIP;
  p += 4;
  remain -= 4;

#if DT_UseBackpatch
  if ( remain < 8 )
    return(0); /* Out of space. */
  * (Bit32u *) p = HashIDNone; /* Invalid ID to start with. */
  p += 4;
  * (Bit32u *) p = TcodeOffsetNone; /* Invalid direct tcode addr (patched). */
  p += 4;
  remain -= 8;
#endif

  /* Patch offset from negated branch here */
  *sequenceEndOffsetPtr = p - (sequenceEndOffsetPtr + 1);

  /* Branch relocation information. */
  if ((vm->relocN+1) >= MaxRelocEntries)
    return(0); /* Out of entries. */
  vm->reloc[vm->relocN].tcodePtr = r3hBranchOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) __r3hBranchStatic;
  vm->relocN++;
  vm->reloc[vm->relocN].tcodePtr = branchDataOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) branchDataPtr;
  vm->relocN++;
  return( p - pOrig );
}

  unsigned
dtGenLoopJb(vm_t *vm, Bit8u *p, unsigned remain, instruction_t *i, Bit32u eip)
{
  Bit8u *sequenceEndOffsetPtr, *branchCodeOffsetPtr, *pOrig,
        *branchDataOffsetPtr, *r3hBranchOffsetPtr,
        *branchDataPtr;
  Bit32u newEIP;

  pOrig = p;

  /* Calculate new guest EIP after relative branch. */
  newEIP = (eip + i->ilen) + (Bit32s) i->Id;
  if (i->os_32 == 0)
    newEIP &= 0x0000ffff;
// xxx CS.limit vs newEIP check here?

  /* Pseudo-code:
   *
   *   loop branchCode
   *   jmp  seqEnd
   * branchCode:
   *   ...
   *   seqEnd
   */

  if (vm->i.as_32 == 0) {
    /* If guest is using 16-bit address size, then we have to use the
     * AddrSize prefix to request 16-bit use of ECX.
     */
    if (remain < 3)
      return( 0 ); /* Out of space. */
    *p++ = 0x67; /* AddrSize prefix. */
    *p++ = 0xe2; /* loop_Jb */
    branchCodeOffsetPtr = p;
    p++; /* Patch. */
    remain -= 3;
    }
  else {
    /* Guest using 32-bit addressing.  Tcode runs in a 32-bit segment,
     * so this is the default.
     */
    if (remain < 2)
      return( 0 ); /* Out of space. */
    *p++ = 0xe2; /* loop_Jb */
    branchCodeOffsetPtr = p;
    p++; /* Patch. */
    remain -= 2;
    }

  if (remain < 2)
    return( 0 ); /* Out of space. */
  *p++ = 0xeb; /* Jmp Jb */
  sequenceEndOffsetPtr = p;
  p++; /* Patch. */
  remain -= 2;

  /* Patch offset to branch code here. */
  *branchCodeOffsetPtr = p - (branchCodeOffsetPtr + 1);

  /*   gs; movl $branchData, __r3hBranchDataAddr
   *   jmp __r3hBranchStatic
   * branchData:
   *   newEIP
   *   hashID
   *   tcodeDirectAddr  (patched in)
   */

  if ( remain < 11 )
    return(0); /* Out of space. */
  *p++ = 0x65; /* gs; movl $branchData, __r3hBranchDataAddr */
  *p++ = 0xc7;
  *p++ = 0x05;
  * (Bit32u *) p = (Bit32u) & __r3hBranchDataAddr;
  p += 4;
  branchDataOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 11;

  if ( remain < 5 )
    return(0); /* Out of space. */
  *p++ = 0xe9; /* jmp __r3hBranchStatic */
  r3hBranchOffsetPtr = p;
  p += 4; /* Patch later. */
  remain -= 5;

  branchDataPtr = p;
  if ( remain < 4 )
    return(0); /* Out of space. */
  * (Bit32u *) p = newEIP;
  p += 4;
  remain -= 4;

#if DT_UseBackpatch
  if ( remain < 8 )
    return(0); /* Out of space. */
  * (Bit32u *) p = HashIDNone; /* Invalid ID to start with. */
  p += 4;
  * (Bit32u *) p = TcodeOffsetNone; /* Invalid direct tcode addr (patched). */
  p += 4;
  remain -= 8;
#endif

  /* Patch offset from negated branch here */
  *sequenceEndOffsetPtr = p - (sequenceEndOffsetPtr + 1);

  /* Branch relocation information. */
  if ((vm->relocN+1) >= MaxRelocEntries)
    return(0); /* Out of entries. */
  vm->reloc[vm->relocN].tcodePtr = r3hBranchOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) __r3hBranchStatic;
  vm->relocN++;
  vm->reloc[vm->relocN].tcodePtr = branchDataOffsetPtr;
  vm->reloc[vm->relocN].tcodeOff = (Bit32u) branchDataPtr;
  vm->relocN++;

  return( p - pOrig );
}
