/*
 * main.c:  Main NULL kernel code
 *
 * (C) 1999 Ramon van Handel, The plex86 team
 *
 * HISTORY
 * Date      Author      Rev    Notes
 * 09/01/99  ramon       1.0    First release
 * 14/07/99  ramon       1.1    Adapted for plex86 NULL kernel
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <typewrappers.h>
#include "taskstate.h"
#include "gdt.h"
#include "8254.h"
#include "8259.h"
#include "ioport.h"
#include "decl.h"

/**************************************************************************/

/* Declarations */
extern VOID timerISR(VOID);   /* The assembly stub for the timer   */
extern VOID __syscall(VOID);  /* The assembly stub for system call */

PUBLIC VOID task0(VOID);      /* The first task to run             */
PUBLIC VOID task1(VOID);      /* The second task to run            */
PUBLIC VOID syscall(DATA num, DATA parm1, DATA parm2);

/**************************************************************************/

/* We run two tasks.  Each has its own kernel stack and user stack */
UBYTE kstack[2][1024];  /* The kernel stacks for the demotasks     */
UBYTE ustack[2][1024];  /* The user stacks for the demotasks       */
UDATA esp0[2];          /* The saved kernel stack pointer          */

/**************************************************************************/

PUBLIC VOID kmain(VOID)
/*
 * The NULL kernel startup code
 *
 * INPUT:
 *     none
 *
 * RETURNS:
 *     none
 */
{
    DATA i;

    struct _frame {                /* This structure is the stack frame */
        UDATA ebx, esi, edi, ebp;  /* of the scheduler ISR.  We use it  */
        UDATA edx, ecx, eax;       /* to set up the initial kernel      */
        UWORD16 ds;                /* stacks for processes              */
        UADDR eip, cs;
        UADDR eflags;
        UADDR esp, ss;
    } __attribute__ ((packed)) *frame;

    /* Initialise paging */
    initMem();

    /* Initialise the TSS's:  set up the kernel stack positions */
    tss[0].ss0  = 0x10 | 0x0;  /* The kernel stack segment is has   */
    tss[1].ss0  = 0x10 | 0x0;  /* selector 0x10, using RPL 0x0      */
    tss[0].esp0 = (UADDR) &kstack[0][1024];
    tss[1].esp0 = (UADDR) &kstack[1][1024];

    /* Put the TSS addresses in the GDT                    */
    /* The TSS selectors are 0x28 (task0) and 0x30 (task1) */
    GDT[0x28/8].desc.base_low  = ((UWORD32)&tss[0]) & 0xffff;
    GDT[0x28/8].desc.base_med  = ((UWORD32)&tss[0]>>16)&0xff;
    GDT[0x28/8].desc.base_high = ((UWORD32)&tss[0]>>24)&0xff;
    GDT[0x30/8].desc.base_low  = ((UWORD32)&tss[1]) & 0xffff;
    GDT[0x30/8].desc.base_med  = ((UWORD32)&tss[1]>>16)&0xff;
    GDT[0x30/8].desc.base_high = ((UWORD32)&tss[1]>>24)&0xff;

    /* Set up the initial kernel stack for task0 */
    frame = (struct _frame *) (&kstack[0][1024] - sizeof(*frame));
    frame->eip    = (UADDR) task0; /* The task entry point is task0 */
    frame->cs     = 0x18 | 0x3;  /* Code selector is 0x18, RPL 0x3  */
    frame->eflags = 0x202;       /* Flags:  IF=on; rest is off      */
    frame->esp    = (UADDR) &ustack[0][1024];
    frame->ss     = 0x20 | 0x3;  /* Data selector is 0x20, RPL 0x3  */
    frame->ds     = 0x20 | 0x3;
    esp0[0]       = (UDATA) frame;

    /* Set up the initial kernel stack for task1 */
    frame = (struct _frame *) (&kstack[1][1024] - sizeof(*frame));
    frame->eip    = (UADDR) task1; /* The task entry point is task0 */
    frame->cs     = 0x18 | 0x3;  /* Code selector is 0x18, RPL 0x3  */
    frame->eflags = 0x202;       /* Flags:  IF=on; rest is off      */
    frame->esp    = (UADDR) &ustack[1][1024];
    frame->ss     = 0x20 | 0x3;  /* Data selector is 0x20, RPL 0x3  */

    /* Set 8254 PIT channel 0 to mode 2 (periodic interrupts) */
    outportb(TMR_CTRL, (TMR_SC0 + TMR_both + TMR_MD2));

    /* Set 8254 PIT channel 0 to frequency 100Hz */
    outportb(TMR_CNT0, (UBYTE)   (1193180+50)/100);
    outportb(TMR_CNT0, (UBYTE) (((1193180+50)/100)>>8));

    /* Initialise the interrupts */
    initIntr();

    /* Set up timer ISR in IDT as an interrupt gate */
    setVector(timerISR, M_VEC, (D_INT + D_PRESENT + D_DPL3));

    /* Set up the system call ISR as a trap gate at int 0x80 */
    setVector(__syscall, 0x80, (D_TRAP + D_PRESENT + D_DPL3));

    /* Enable timer IRQ */
    enableIRQ(0);

    /* Clear the screen, so we don't get it all garbled */
    for(i=0; i<80*25; i++)
        ((UWORD16 *)(0xb8000))[i] = 0;

    /* Kickstart the scheduler */
    asm (
        "movl %0,%%esp   \n"
        "movl %1,%%eax   \n"
        "movw %%ax,%%ds  \n"
        "jmp timerISR    \n"
        :
        : "r" (((UADDR)frame)+7*sizeof(UDATA)+1*sizeof(UWORD16)),
          "r" ((UWORD16)0x20|3)
    );
}

/**************************************************************************/

#define SYSSHOWNUM -1

VOID __shownum(DATA nr, DATA pos)
/*
 * Show a hexadecimal number at the specified position
 * on a VGA text console
 *
 * INPUT:
 *     nr:   number to show
 *     pos:  screen position to show it on
 *
 * RETURNS:
 *     none
 */
{
    UBYTE o,i;

    for(i=0;i<8;i++) {
        o = nr & 0xf;
        if(o>9) o += 7;
        o += '0';
        ((UWORD16 *)(0xb8000))[(7-i)+pos] = (o | (((0 << 4) | 0xf) << 8));
        nr >>= 4;
    }
}

VOID syscall(DATA num, DATA parm1, DATA parm2)
/*
 * The system call handler:  demultiplex the system call
 * to the correct call handler.
 *
 * INPUT:
 *     num:    system call number
 *     parm1:  first system call parameter
 *     parm2:  second system call parameter
 *
 * RETURNS:
 *     none
 */
{
    switch(num) {
        case SYSSHOWNUM:
            __shownum(parm1, parm2);
            break;
    };
}

/**************************************************************************/

#define shownum(a,b) \
    asm ( "int $0x80" : : "d" (SYSSHOWNUM), "c" (a), "a" (b) );

PUBLIC VOID task0(VOID)
/*
 * The first demotask
 *
 * INPUT:
 *     none
 *
 * RETURNS:
 *     none
 */
{
    int count;
    for (count=1;;count++) {
        shownum(count, 0);
    }
}

PUBLIC VOID task1(VOID)
/*
 * The second demotask
 *
 * INPUT:
 *     none
 *
 * RETURNS:
 *     none
 */
{
    int count;
    for (count=1;;count++) {
        shownum(count, 10);
    }
}

/**************************************************************************/

PUBLIC UDATA scheduler(UDATA esp)
/*
 * The main scheduler function
 *
 * INPUT:
 *     esp:  The stack pointer for this task
 *
 * RETURNS:
 *     scheduler():  The new stack location to switch to
 */
{
    /* tsssel[] contains the TSS selectors of the two tasks */
    static UWORD16 tsssel[2] = { 0x28, 0x30 };

    /* nextTask usually contains the currently running task */
    static UDATA nextTask = 1;

    /* Store the old stack location */
    esp0[nextTask] = esp;

    /* Pick the next task to run */
    nextTask = 1 - nextTask;   /* What a SOPHISTICATED scheduling */
                               /* alghorithm :)                   */

    /* Load the new task's %tr (so we switch to the right kernel */
    /* stack when an interrupt occurs) and clear the busy bit    */
    GDT[tsssel[nextTask]/8].desc.access = (D_TSS+D_PRESENT) >> 8;
    asm( "ltr %0" : : "r" (tsssel[nextTask]) );

    /* Return to the user, load user stack and registers off the */
    /* new kernel stack in the ISR                               */
    return esp0[nextTask];
}

/**************************************************************************/

asm (
    ".text; .globl timerISR      \n"   /* Assembly stub for scheduler   */
    "timerISR:                   \n"
    "    pushw %ds               \n"   /* Save user data segment        */
    "    pushw %ss               \n"   /* Load kernel data segment from */
    "    popw %ds                \n"   /* stack seg (it's always valid) */
    "    pushl %eax              \n"   /* Push the user registers onto  */
    "    pushl %ecx              \n"   /* the stack                     */
    "    pushl %edx              \n"
    "    pushl %ebp              \n"
    "    pushl %edi              \n"
    "    pushl %esi              \n"
    "    pushl %ebx              \n"
    "    movb $0x20,%al          \n"   /* Send EOI to the master PIC so */
    "    outb %al,$0x20          \n"   /* other IRQs can be triggered   */
    "    movl %esp,%eax          \n"   /* Pass the old task's stack     */
    "    pushl %eax              \n"   /* pointer to the scheduler      */
    "    cld                     \n"   /* GCC likes this                */
    "    call scheduler          \n"   /* Go to the C scheduler code    */
    "    movl %eax,%esp          \n"   /* Switch to the new stack       */
    "    popl %ebx               \n"   /* Restore the registers... off  */
    "    popl %esi               \n"   /* the new stack                 */
    "    popl %edi               \n"
    "    popl %ebp               \n"
    "    popl %edx               \n"
    "    popl %ecx               \n"
    "    popl %eax               \n"
    "    popw %ds                \n"   /* Restore user data segment     */
    "    iret                    \n"   /* Back to the application       */
);

asm (
    ".text; .globl __syscall   \n"  /* Assembly stub for the systemcall */
    "__syscall:                \n"
    "    pushw %ds             \n"  /* Save user data segment           */
    "    pushw %ss             \n"  /* Load kernel data segment from    */
    "    popw %ds              \n"  /* stack seg (it's always valid)    */
    "    pushal                \n"
    "    pushl %eax            \n"  /* Push the registers that are      */
    "    pushl %ecx            \n"  /* clobbered by GCC on the stack    */
    "    pushl %edx            \n"
    "    cld                   \n"  /* GCC likes this                   */
    "    call syscall          \n"  /* Go to the system call handler    */
    "    popl %edx             \n"  /* Restore the registers            */
    "    popl %ecx             \n"
    "    popl %eax             \n"
    "    popal                 \n"
    "    popw %ds              \n"  /* Restore user data segment        */
    "    iret                  \n"  /* Back to the application          */
);

