/*
 * Copyright (c) 2005- Shinji Kashihara.
 * All rights reserved. This program are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies
 * this distribution, and is available at epl-v10.html.
 */
package jp.sourceforge.mergedoc.pleiades.resource;

import junit.framework.TestCase;

/**
 * 翻訳辞書クラスのテスト・クラスです。
 * <p>
 * @author cypher256
 */
public class TranslationDictionaryTest extends TestCase {

	/** ターゲット */
	protected final TranslationDictionary target = new TranslationDictionary();

	/** assertLookup */
	protected void assertLookup(String input, String expected) {

		TranslationResult result = target.lookup(input);
		System.out.println(input + "=" + result.getValue());
		assertEquals(expected, result.getValue());
		assertEquals(true, result.isFound());
	}

	/** assertLookupFalse */
	protected void assertLookupFalse(String input) {

		TranslationResult result = target.lookup(input);
		System.out.println(input + "=" + result.getValue());
		assertEquals(input, result.getValue());
		assertEquals(false, result.isFound());
	}

	/** assertLookupFalse */
	protected void assertLookupFalse(String input, String expected) {

		TranslationResult result = target.lookup(input);
		System.out.println(input + "=" + result.getValue());
		assertEquals(expected, result.getValue());
		assertEquals(false, result.isFound());
	}

	// -------------------------------------------------------------------------

	/** テスト */
	public void testLookupIgnoreMnemonic() {

		TranslationResult result = null;
		String input = null;

		// lookupIgnoreMnemonic ニーモニックなし
		input = "Label";
		result = target.lookupIgnoreMnemonic(input);
		System.out.println(input + "=" + result.getValue());
		assertEquals("ラベル", result.getValue());
		assertEquals(true, result.isFound());

		// lookupIgnoreMnemonic ニーモニックあり
		input = "L&abel";
		result = target.lookupIgnoreMnemonic(input);
		System.out.println(input + "=" + result.getValue());
		assertEquals(input, result.getValue());
		assertEquals(false, result.isFound());
	}

	/** テスト */
	public void testLookup() {

		// ノーマル
		assertLookup("Label", "ラベル");

		// ノーマル false
		assertLookupFalse("... undefined string in dictionary ...");

		// ノーマル、文
		assertLookup("Error reading configuration", "構成の読み取り中にエラーが発生しました");

		// ニーモニック
		assertLookup("L&abel", "ラベル(&A)");

		// ニーモニック、先頭
		assertLookup("&Left", "左(&L)");

		// ニーモニック、文
		assertLookup("Remo&ve Buildfile", "ビルドファイルの除去(&V)");

		// 正規表現
		assertLookup("JavaHL 1.4.5 Win32 Binaries (Optional)", "JavaHL 1.4.5 Win32 バイナリー (オプション)");

		// 末尾 :
		assertLookup("Search results for {0}:", "{0} の検索結果:");

		// 末尾 ...
		assertLookup("&Extension....", "拡張(&E)....");

		// 末尾 >
		assertLookup("&Next >", "次へ(&N) >");

		// 末尾 >>
		assertLookup("&Next >>", "次へ(&N) >>");

		// 先頭 <<
		assertLookup("<< &Previous", "<< 前へ(&P)");

		// a タグ
		assertLookup("&Using {0} - <a>Select other...</a>", "{0} を使用中 - <a>その他の選択(&U)...</a>");

		// 末尾 -->
		assertLookup("&Add -->", "追加(&A) -->");

		// アクセラレーター
		assertLookup("Add &Include@Ctrl+Shift+N", "Include の追加(&I)@Ctrl+Shift+N");

		// アクセラレーターとニーモニック
		assertLookup("Go to &Line...@Ctrl+L", "指定行へジャンプ(&L)...@Ctrl+L");

		// ニーモニックではない &
		assertLookup("A resource named \"&1\" already exists.", "リソース \"&1\" はすでに存在します。");

		// ニーモニックではない &
		assertLookup("K&R [built-in]", "K&R [ビルトイン]");

		// ニーモニックではない &&
		assertLookup("Ignore '&&' in &Java properties files", "Java プロパティー・ファイルの '&&' を無視(&J)");

		// HTML 文字実体参照
		assertLookup("&nbsp;&nbsp;Go&nbsp;&nbsp;", "&nbsp;&nbsp;実行&nbsp;&nbsp;");

		// 先頭空白
		assertLookup(" Label", " ラベル");

		// 先頭タブ
		assertLookup("\tLabel", "\tラベル");

		// 末尾空白
		assertLookup("Label ", "ラベル ");

		// 末尾タブ
		assertLookup("Label\t", "ラベル\t");

		// 末尾 ...
		assertLookup("Label...", "ラベル...");

		// 末尾 (Incubation)
		assertLookup("Label (Incubation)", "ラベル (インキュベーション)");

		// 末尾 (Experimental)
		assertLookup("Label [Experimental]", "ラベル [実験用]");

		// 複数形 (s)
		assertLookup("Delete Bookmark", "ブックマークの削除");
		assertLookup("Delete Bookmark(s)", "ブックマークの削除");

		// ピリオド + 連続する英小文字 (翻訳しない)
		assertLookupFalse(".test");

		// 句点分割
		assertLookup(
				"Add, remove, or edit installed Web browsers.\n"
						+ "The selected Web browser will be used by default when Web pages are opened, although some applications may always use the external browser.",

				"インストール済み Web ブラウザーを追加、除去、または編集します。\n"
						+ "選択された Web ブラウザーは、Web ページが開いているときにデフォルトで使用されますが、一部のアプリケーションでは、常に外部ブラウザーが使用されます。");

		// 句点分割 ()
		assertLookup("{0} must be public. (EJB 1.1: 9.2.3). ", "{0} は public でなければなりません。(EJB 1.1: 9.2.3). ");

		// 句点分割 () 2 つ
		assertLookup("Subversive SVN JDT Ignore Extensions Sources (Optional) (Incubation)",
				"Subversive SVN JDT 無視拡張ソース (オプション) (インキュベーション)");

		// 句点分割 文字を示す -
		assertLookup(
				"Performs a line-by-line comparison of all code lines and reports duplicate code if a sequence of lines differs only in indentation. All import statements in Java code are ignored, any other line - including javadoc, whitespace lines between methods, etc. - is considered (which is why the check is called strict).",
				"すべてのコード行の行単位比較を実行し、行シーケンスのインデントだけが異なる場合、重複したコードを報告します。Java コード内のすべてのインポート・ステートメントは無視され、その他の行 - javadoc、 メソッド間の空白行などを含む - は考慮されます (このチェックが厳密と呼ばれる理由)。");

		// 末尾 。が付いてしまう
		assertLookup("To go to the next frame, or toolbar press Ctrl+Tab", "次のフレームまたはツールバーにジャンプするには、Ctrl+TAB を押してください");

		// ニーモニック
		assertLookup("P&ackage", "パッケージ(&A)");

		// -----------------------------------------------------------------------------------------

		// [#14691] 文末の空白が除去される問題対応確認
		assertLookup("Sets the language module the spell checking engine should use when you enter a log message. ",
				"ログ・メッセージを入力する時にスペル・チェック・エンジンが使用すべき言語モジュールを設定します。 ");
		// [#14691] 文末のセミコロンが除去される問題対応確認
		assertLookup("assertion failed;", "アサーションに失敗しました。");
		// [#14691] 謎の空白
		assertLookup("{n,} - Greedy match >= n times", "{n,} - 正規表現による一致 n 回以上");

		// [#14699] 末尾かっこ
		assertLookup(
				"Type named ''{0}'' is imported (single-type-import) in ''{1}'' (a compilation unit must not import and declare a type with the same name)",
				"''{0}'' という名前の型は ''{1}'' 内にインポート (単一の型のインポート) されています (コンパイル単位は同じ名前で型をインポートおよび宣言できません)");
		// [#14699] (s)
		assertLookup("Archive file ''{0}'' is referenced by the following project(s). Do you still want to delete it?",
				"アーカイブ・ファイル ''{0}'' は次のプロジェクトによって参照されます。削除しますか?");

		// [#14700] 囲み文字
		assertLookup("\"Malformed file transmission received\"", "\"誤った形式のファイル伝送が受信されました\"");
		// [#14700] (s)
		assertLookup(
				"Problems tagging the resources. {0} project(s) successfully tagged and {1} project(s) with errors.",
				"リソースのタグ付けで問題が発生しました。{0} プロジェクトが正常にタグ付けされ、{1} プロジェクトにエラーがありました。");
		assertLookup("You have chosen to ignore {0} resources. Select what to add to the .cvsignore file(s).",
				"{0} 個のリソースを無視するよう選択しました。.cvsignore ファイルに追加するものを選択してください。");

		// [#14796] 分割後の {1}.
		assertLookupFalse("{0}, Function: {1}.");

		// & を含むパラメーター付き URL は翻訳不可
		assertLookupFalse("/aaa/bbb?xxx=1&yyy=2");

		// [#14799]
		assertLookup("{0} does not exist.\nAre you sure you want to create it?", "{0} は存在しません。\nこれを作成しますか?");

		// [#14801]
		assertLookup("Error copying file {0}: {1}", "ファイル {0} のコピー・エラー: {1}");
		assertLookup("Unable to initialize source location - directory does not exist: {0}",
				"ソース・ロケーションを初期化できません。ディレクトリー {0} がありません");

		// [#14802]
		assertLookup("The selected enterprise bean cannot be deleted because it is involved in\n"
				+ "relationships with other enterprise beans.  You must first delete these\n"
				+ "relationships before deleting the bean.",

		"別のエンタープライズ Bean との関係が含まれるため、\n" + "選択したエンタープライズ Bean は削除できません。\n" + "Bean を削除する前に、まず関係を削除する必要があります。");

		// [#14837]
		assertLookupFalse("The resulting string is too long to fit in a DOMString: ''{0}''.");

		// [#14840]
		assertLookup("Minimum VM Size (kB)", "最小 VM サイズ (kB)");
		assertLookup("kB.", "kB。");

		// [#14841]
		assertLookup("&Enable annotation roll-over (on new editors)", "(新しいエディター上で) 注釈のロールオーバーを使用可能にする(&E)");
		assertLookup("Unable to locate JAR/zip in file system as specified by the driver definition: {0}.",
				"ドライバー定義で指定されたファイル・システム内に JAR/zip が見つかりません: {0}。");

		// [#14842]
		assertLookup("'?' is not expected. '(?:' or '(?=' or '(?!' or '(?<' or '(?#' or '(?>'?",
				"'?' をここに書くことはできません。もしかして '(?:' か '(?=' か '(?!' か '(?<' か '(?#' か '(?>' ですか?");

		// [#14843]
		assertLookup("Invalid timeout! Timeout should be positive integer or -1 to disable.",
				"タイムアウト値が無効です! タイムアウト値は、正整数または -1 (使用不可にする場合) でなければなりません。");

		// [#14844]
		assertLookup("none (no sides, default value)", "none (線なし、デフォルト値)");
		assertLookup("groups (between row groups)", "groups (行グループ間)");

		// [#14845]
		assertLookup(
				"It is strongly recommended you restart {0} for the changes to take effect. For some add-ons, it may be possible to apply the changes you have made without restarting. Would you like to restart now?",
				"変更を反映させるために {0} の再始動を強くお勧めします。いくつかのアドオンでは、再始動しなくても変更が反映されている場合もあります。すぐに再始動しますか?");

		// [#14862]
		assertLookup("Bundle symbolic name contains illegal characters. Legal characters are A-Z a-z 0-9 . _ -",
				"バンドルのシンボル名に正しくない文字が含まれています。正しい文字は A-Z a-z 0-9 . _ - です");

		// [#14865]
		assertLookup("Unable to restore source location - class not found: {0}", "ソース・ロケーションを復元できません。{0} クラスが見つかりません");

		// [#14866]
		assertLookup("Error: File not found.", "エラー: ファイルが見つかりません。");

		// [#14867]
		assertLookup("Update the related faces config file. Are you sure?", "関連する faces 構成ファイルを更新します。よろしいですか?");

		// [#14869]
		assertLookup("<details unavailable - not supported by VM>", "<詳細は使用不可 - VM によってサポートされていません>");

		// [#14870]
		assertLookup(
				"Inclusion or exclusion patterns are disabled in project {1}, cannot selectively include or exclude from entry: ''{0}''",
				"包含または除外パターンはプロジェクト {1} 内で使用不可です。エントリー ''{0}'' から選択的に包含または除外できません");

		// [#14883]
		assertLookup("Profile name is empty.", "プロファイル名が空です。");

		// [#14887]
		assertLookup("An error has occurred: {0}. See error log for more details. Do you want to exit?",
				"エラーが発生しました: {0}。詳細については、エラー・ログを参照してください。終了しますか?");

		// [#14888]
		assertLookup(
				"Select a set of UDDI businesses and click <strong>Add to Favorites</strong> to add these to the list of favorites. Click <strong>Add to UDDI Page</strong> to add these to the UDDI Page. Click <strong>Refresh</strong> to refresh the selections.",
				"UDDI ビジネスのセットを選択し、「<strong>お気に入りに追加</strong>」をクリックしてこれらのビジネスをお気に入りのリストに追加します。「<strong>UDDI ページに追加</strong>」をクリックしてこれらのビジネスを UDDI ページに追加します。「<strong>リフレッシュ</strong>」をクリックして選択をリフレッシュします。");

		// [#14889]
		assertLookup("-- Java Generation :: Generating...", "-- Java の生成 :: 生成中...");

		// [#14890]
		assertLookup(
				"Inclusion or exclusion patterns are disabled in project {1}, cannot selectively include or exclude from entry: ''{0}''",
				"包含または除外パターンはプロジェクト {1} 内で使用不可です。エントリー ''{0}'' から選択的に包含または除外できません");

		// [#14891]
		assertLookup("CHKJ2017E: This interface must extend javax.ejb.EJBHome (EJB 2.0: 6.3, 7.10.6).",
				"CHKJ2017E: このインターフェースは javax.ejb.EJBHome を拡張しなければなりません (EJB 2.0: 6.3、7.10.6)。");

		// [#14901]
		assertLookup("Warning: Invalid value for the initParam xpoweredBy. Will use the default value of \"false\"",
				"警告: Invalid value for the initParam xpoweredBy. デフォルト値 \"false\" を使用します");
		assertLookup("Problems encountered during code generation. Select detail entries for more information.",
				"コード生成中に問題が発生しました。Select detail entries for more information.");
		assertLookup(
				"No custom build file found in {0}. If you want to generate a new build file, temporarily disable the ''custom'' property. ",
				"{0} でカスタム・ビルド・ファイルが検出されませんでした。If you want to generate a new build file, temporarily disable the ''custom'' property. ");

		// [#14902]
		assertLookup("Missing directory entry: {0}.", "欠落しているディレクトリー・エントリー: {0}.");
		assertLookup("Missing directory entry: {1}.", "欠落しているディレクトリー・エントリー: {1}.");

		// [#14944]
		assertLookup("IO exception.", "IO 例外。");
		assertLookup("IO exception", "IO 例外");

		// [#14946]
		assertLookup(
				"When selecting {0} compliance, make sure to have a compatible JRE installed and activated (Currently {1}). "
						+ "Configure the <a href=\"1\" >Installed JREs</a> or change the JRE on the <a href=\"2\" >Java build path</a>.",

				"{0} 準拠を選択する場合は、互換性がある JRE がインストール済みでアクティブにされていることを確認してください (現在は {1})。"
						+ "<a href=\"1\" >インストール済み JRE</a> を構成するか、<a href=\"2\" >Java ビルド・パス</a> にある JRE を変更してください。");

		// [#14947]
		assertLookup("Package: Generate Javadoc for all but private classes and members.",
				"Package: private のクラスおよびメンバー以外について Javadoc を生成します。");
		assertLookup("Package: Generate Jsdoc for all but private classes and members.",
				"Package: private のクラスおよびメンバー以外について Jsdoc を生成します。");

		// [#14954]
		assertLookupFalse("visibility scope where Javadoc comments are not checked.");
		assertLookupFalse("A type.");

		// [#14955]
		assertLookup("n/a", "なし");
		assertLookup("N/A", "なし");

		// [#15186]
		assertLookupFalse("C&VS", "CVS(&V)");

		// [#15193]
		assertLookupFalse("Policy on how to pad parentheses. nospace = do not pad (e.g. method(a, b)). space = ensure padding (e.g. method( a, b )).");

		// [#15277]
		assertLookup(
				"Project Validator that validates the Spring Config files (e.g. class and property names, referenced beans etc.)",
				"Spring 構成ファイルを検証するプロジェクト・バリデーター (例 クラスとプロパティーの名前、 参照される bean など)");
	}
}
